/*
 **************************************************************************
 *
 * Utility program to combine the boot rom kernel with the packet driver
 *
 * Module:  pass1.c
 * Purpose: Combine the bootrom kernel and the packet driver
 * Entries: main
 *
 **************************************************************************
 *
 * Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>
#ifdef MSDOS
#include <io.h>
#endif
#include "../../headers/memory.h"
#include "../../headers/version.h"
#include "./utility.h"


#define BLKSIZE		512		/* Size of input buffer */
#define MAXARGS		126		/* Maximum size of pktdrvr cmd line */
#define EXESIG		0x5A4D		/* Signature of DOS EXE files */

#define MAXPKTNUM	10		/* Maximum number of driver programs */
#define PKTHEADER	4		/* Length of packet driver header */
#define COMSIG		0x4B47		/* Signature for packet driver header */
#define SIZEOFS		2		/* Offset to word with size of pktdrv */


char *progname;
char *krnname;
char *pktargs[MAXPKTNUM];
char *pktname[MAXPKTNUM];
char *outname;
int pktnum;
int infile;
int outfile;



/*
 * Read config file
 */
static void readconfig(char *fname)
{
  FILE *cfgfile;
  char buf[1024], *cp;
  char *progname, *progargs;
  int num = 1;

#ifdef MSDOS
  if ((cfgfile = fopen(fname, "rt")) == NULL) {
#else
  if ((cfgfile = fopen(fname, "r")) == NULL) {
#endif
	perror(fname);
	exit(1);
  }
  while (fgets(buf, sizeof(buf) - 1, cfgfile) != NULL && num < MAXPKTNUM) {
	for (cp = buf; *cp && isspace(*cp); cp++) ;
	progname = cp;
	for ( ; *cp && !isspace(*cp) && !iscntrl(*cp); cp++) ;
	*cp++ = '\0';
	for ( ; *cp && isspace(*cp); cp++) ;
	progargs = cp;
	for ( ; *cp && !iscntrl(*cp); cp++) ;
	*cp = '\0';
	pktname[num] = strdup(progname);
	pktargs[num] = strdup(progargs);
	num++;
  }
  (void)fclose(cfgfile);
}



/*
 * Write one block of data to the output file.
 */
int dowrite(unsigned char *buf, int bufsize)
{
  int ofs;

  if ((ofs = write(outfile, (char *)buf, bufsize)) < 0) {
	perror(outname);
	exit(1);
  }
#ifdef DEBUG
  fprintf(stderr, "w:%d<%s>\n", ofs, outname);
#endif
  return ofs;
}



/*
 * Read one block of data either from the kernel image file or the
 * packet driver binary.
 */
int doread(unsigned char *buf, int bufsize, char *name)
{
  int ofs;

  if ((ofs = read(infile, (char *)buf, bufsize)) < 0) {
	perror(name);
	exit(1);
  }
#ifdef DEBUG
  fprintf(stderr, "r:%d<%s>\n", ofs, name);
#endif
  return ofs;
}



void usage(void)
{
  fprintf(stderr, "usage: %s -k<kernel file> -p<number><driver file>\n",
								progname);
  fprintf(stderr, "\t\t-a<number><driver args> [-o<out file>] [-c<config file>] [-v]\n");
  exit(1);
}



void main(int argc, char **argv)
{
  unsigned long argofs[MAXPKTNUM];
  unsigned long pktofs[MAXPKTNUM];
  unsigned long pktlen[MAXPKTNUM];
  unsigned long tabofs;
  unsigned long writecnt;
  unsigned char inbuf[BLKSIZE];
  unsigned int krnlength;
  int i, len, num;
  int verbose = 0;
  char *cp;

  /* Determine our own name */
#ifdef MSDOS
  if ((cp = strrchr(argv[0], '\\')) == NULL)
#else
  if ((cp = strrchr(argv[0], '/')) == NULL)
#endif
	cp = argv[0];
  else
	cp++;
  progname = strdup (cp);

  /* Isolate the different command line options */
  for (i = 1; i < argc; i++) {
	if (*argv[i] != '-' || strlen(argv[i]) < 2)
		usage();
	 switch (argv[i][1]) {
		case 'k':  if (strlen(argv[i]) > 2) {
				cp = &argv[i][2];
		           } else {
				if (++i >= argc) usage();
				cp = argv[i];
		           }
		           krnname = strdup(cp);
		           break;
		case 'p':  if ((len = strlen(argv[i])) < 3) usage();
		           num = argv[i][2] - '0';
		           if (num < 1 || num >= MAXPKTNUM) usage();
		           if (len > 3) {
				cp = &argv[i][3];
		           } else {
				if (++i >= argc) usage();
				cp = argv[i];
		           }
		           pktname[num] = strdup(cp);
		           break;
		case 'o':  if (strlen(argv[i]) > 2) {
				cp = &argv[i][2];
		           } else {
				if (++i >= argc) usage();
				cp = argv[i];
		           }
		           outname = strdup(cp);
		           break;
		case 'c':  if (strlen(argv[i]) > 2) {
				cp = &argv[i][2];
		           } else {
				if (++i >= argc) usage();
				cp = argv[i];
		           }
		           readconfig(cp);
		           break;
		case 'a':  if ((len = strlen(argv[i])) < 3) usage();
		           num = argv[i][2] - '0';
		           if (num < 1 || num >= MAXPKTNUM) usage();
		           if (len > 3) {
				cp = &argv[i][2];
		           } else {
				if (++i >= argc) usage();
				cp = argv[i];
		           }
		           pktargs[num] = strdup(cp);
		           break;
		case 'v':  verbose++;
		           break;
		default:   usage();
	}
  }

  /*
   * Count number of driver programs and check that they all have command
   * lines, and that all other command line options are given.
   */
  for (i = 1; i < MAXPKTNUM; i++)
	if (pktname[i] != NULL) {
		if (pktargs[i] == NULL) usage();
		if (strlen(pktargs[i]) > MAXARGS) {
			fprintf(stderr, "%s: argument for driver %d too long\n",
								i, progname);
			exit(1);
		}
		pktnum++;
	}
  if (krnname == NULL || pktnum == 0)
	usage();

  /* Print debugging information */
  if (verbose) {
	printf("Kernel file name: %s\n", krnname);
	printf("Output file name: %s\n", outname == NULL ? "stdout" : outname);
	for (i = 1; i < MAXPKTNUM; i++) {
		if (pktname[i] != NULL)
			printf("Driver program %d: %s  \"%s\"\n",
						i, pktname[i], pktargs[i]);
	}
	printf("\n");
  }

  /* Open the output file */
  if (outname == NULL) {
	outfile = fileno(stdout);
  } else {
#ifdef MSDOS
	if ((outfile = open(outname, O_CREAT | O_WRONLY | O_TRUNC | O_BINARY, 0666)) < 0) {
#else
	if ((outfile = open(outname, O_CREAT | O_WRONLY | O_TRUNC, 0666)) < 0) {
#endif
		perror(outname);
		exit(1);
	}
  }

  /*
   * Open the input file, read the first block, and check type of file.
   * If the input file is a boot floppy image, copy the first block directly
   * to the output file, because it is the floppy boot sector.
   */
  len = BLKSIZE;
#ifdef MSDOS
  if ((infile = open(krnname, O_RDONLY | O_BINARY)) < 0) {
#else
  if ((infile = open(krnname, O_RDONLY)) < 0) {
#endif
	perror(krnname);
	exit(1);
  }

  /*
   * Read the first block and determine the actual size of the rom kernel
   * without the BSS segment. Also check the version number.
   */
  if (doread(inbuf, len, krnname) != len) {
	fprintf(stderr, "%s: unexpected end of file\n", krnname);
	exit(1);
  }
  if (*((unsigned char *)(&inbuf[(int)(KRNMAJOROFS)])) != MAJOR_VER ||
      *((unsigned char *)(&inbuf[(int)(KRNMINOROFS)])) != MINOR_VER) {
	fprintf(stderr, "%s: invalid kernel version number\n", krnname);
	exit(1);
  }
  krnlength = (unsigned int)ttoh(*((USHORT *)(&inbuf[(int)(KRNSIZEOFS)])) + 2);

  /*
   * Now copy the rom image file into the output file. While writing to
   * the output file, count the number of bytes written, so that the offset
   * to the packet driver can be determined. Also only copy the relevant
   * bytes excluding the BSS segment, which has only zero bytes anyway.
   */
  writecnt = 0;
  krnlength -= len;
  while (TRUE) {
	writecnt += dowrite(inbuf, len);
	if (krnlength <= 0) break;
	if ((len = doread(inbuf, BLKSIZE, krnname)) == 0) break;
	if (len > krnlength)
		len = krnlength;
	krnlength -= len;
  }
  close(infile);
  if (verbose) printf("End position of kernel:         %ld\n", writecnt);

  /*
   * Next write a dummy pointer table to the output file. This table gets
   * later filled with the pointers to all driver arguments and program
   * images.
   */
  tabofs = writecnt;
  len = (pktnum + 1) * 4;
  memset(inbuf, 0, len);
  writecnt += dowrite(inbuf, len);
  if (verbose) printf("End position of pointer table:  %ld\n", writecnt);

  /*
   * Next write all driver argument lines to the output file. The argument
   * strings contain a leading byte which gives the number of characters
   * in the string.
   */
  for (i = 1; i < MAXPKTNUM; i++)
	if (pktname[i] != NULL) {
		argofs[i] = writecnt;
		len = strlen(pktargs[i]);
		inbuf[0] = len & 0xFF;
		strncpy((char *)(&inbuf[1]), pktargs[i], len);
		inbuf[len+1] = 0;
		len += 2;
		writecnt += dowrite(inbuf, len);
	}
  if (verbose) printf("End position of prog arguments: %ld\n", writecnt);

  /*
   * Now copy the packet driver into the output file, but retain some
   * space for the packet driver header.
   */
  for (i = 1; i < MAXPKTNUM; i++)
	if (pktname[i] != NULL) {
#ifdef MSDOS
		if ((infile = open(pktname[i], O_RDONLY | O_BINARY)) < 0) {
#else
		if ((infile = open(pktname[i], O_RDONLY)) < 0) {
#endif
			perror(pktname[i]);
			exit(1);
		}
		len = doread(&inbuf[PKTHEADER], BLKSIZE-PKTHEADER, pktname[i]) +
								PKTHEADER;
		if ((USHORT)ttoh(*((USHORT *)(&inbuf[PKTHEADER]))) == EXESIG) {
			fprintf(stderr, "%s: cannot use EXE type program\n",
								pktname[i]);
			exit(1);
		}
		pktofs[i] = writecnt;
		pktlen[i] = 0;
		while (TRUE) {
			pktlen[i] += dowrite(inbuf, len);
			if ((len = doread(inbuf, BLKSIZE, pktname[i])) == 0)
									break;
		}
		close(infile);
		writecnt += pktlen[i];
		pktlen[i] -= PKTHEADER;
		if (verbose) printf("End position of prog %d:         %ld\n",
								i, writecnt);
	}

  /*
   * Finally write the driver table offset, the driver command line
   * and image pointers and the driver header information into the
   * destination file.
   */
  if (writecnt > 65535L) {
	fprintf(stderr, "Size of output file >64kB\n");
	exit(1);
  }

  if (lseek(outfile, (long)DATAOFS, 0) < 0) {
	perror(outname);
	exit(1);
  }
  *((USHORT *)(&inbuf[(int)(KRNPRGOFS - DATAOFS)])) = (USHORT)htot(tabofs);
  (void)dowrite(inbuf, 2);

  if (lseek(outfile, tabofs, 0) < 0) {
	perror(outname);
	exit(1);
  }
  num = 0;
  for (i = 1; i < MAXPKTNUM; i++)
	if (pktname[i] != NULL) {
		*((USHORT *)(&inbuf[num+0])) = (USHORT)htot(pktofs[i]);
		*((USHORT *)(&inbuf[num+2])) = (USHORT)htot(argofs[i]);
		num += 4;
	}
  (void)dowrite(inbuf, num);

  for (i = 1; i < MAXPKTNUM; i++)
	if (pktname[i] != NULL) {
		if (lseek(outfile, (long)pktofs[i], 0) < 0) {
			perror(outname);
			exit(1);
		}
		*((USHORT *)(&inbuf[0])) = (USHORT)htot(COMSIG);
		*((USHORT *)(&inbuf[SIZEOFS])) =
					(USHORT)htot((pktlen[i] + 15) >> 4);
		(void)dowrite(inbuf, PKTHEADER);
	}

  if (verbose) printf("\n\n");
  close(outfile);
  exit(0);
}

