;**************************************************************************
;*
;* Boot-ROM-Code to load an operating system across a TCP/IP network.
;*
;* Module:  lddisk.asm
;* Purpose: Load a boot sector from a disk
;* Entries: _loaddisk
;*
;**************************************************************************
;*
;* Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
;*
;*  This program is free software; you can redistribute it and/or modify
;*  it under the terms of the GNU General Public License as published by
;*  the Free Software Foundation; either version 2 of the License, or
;*  any later version.
;*
;*  This program is distributed in the hope that it will be useful,
;*  but WITHOUT ANY WARRANTY; without even the implied warranty of
;*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;*  GNU General Public License for more details.
;*
;*  You should have received a copy of the GNU General Public License
;*  along with this program; if not, write to the Free Software
;*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;*


;
;**************************************************************************
;
; Include assembler macros:
;
include ..\..\headers\asm\macros.inc
include ..\..\headers\asm\layout.inc
include ..\..\headers\asm\memory.inc


;
;**************************************************************************
;
; Miscellaneous definitions
;
HD_BASE		equ	80h		; base number for all harddisks
FD_BASE		equ	00h		; base number for all floppy disks

BOOT_SIG	equ	0AA55h		; boot block signature
BOOT_SIG_OFS	equ	01FEh		; offset to boot block signature

PART_OFS	equ	01BEh		; offset to partition table
PART_HEAD_OFS	equ	0001h		; offset to head number in part entry
PART_SECT_OFS	equ	0002h		; offset to sector number in part entry
PART_TYPE_OFS	equ	0004h		; offset to partition type indicator

ENTRY_LEN	equ	0010h		; size of one partition table entry
ENTRY_NUM	equ	4		; maximum number of entries in part tab

EXT_PART_TYPE	equ	5		; type ID for extended partition


;
;**************************************************************************
;
; BSS segment
;
bss_start

diskid		db	?		; BIOS id for disk
partnum		db	?		; partition number

bss_end


;
;**************************************************************************
;
; Start code segment.
;
text_start

	public	_loaddisk		; define entry points


;
;**************************************************************************
;
; Load a disk boot sector into memory
; Input:  1. arg  -  pointer to name of device to load from
;         2. arg  -  segment of buffer for boot block
; Output: returns 0, if not successful, otherwise -1
;
_loaddisk	proc	near

	penter
	push	es
	getarg	bx,0
	call	decodestr		; decode device string
	jc	short load8
	xor	ah,ah
	int	13h			; reset disk system

; First load the master boot record.

	getarg	es,1			; get destination segment into ES
	mov	cx,0001h		; load first sector on track 0
	xor	dh,dh			; head 0
	call	reads1			; load the sector
	jc	short load8

; Next load the first sector of the partition. If we are to load from a
; partition in an extended partition, we have to read the extended partition
; table first.

	mov	bl,partnum
	or	bl,bl			; check if that's all
	jz	short load7
	cmp	bl,5			; check if we have to load from an
	jb	short load6		; extended partition

	mov	bx,PART_OFS		; search for an extended partition
	mov	cx,ENTRY_NUM
load1:	cmp	byte ptr es:[bx + PART_TYPE_OFS],EXT_PART_TYPE
	je	short load2
	add	bx,ENTRY_LEN		; continue with next entry
	loop	short load1
	jmp	short load8

load2:	call	readsect		; read extended partition table
	jc	short load8
	mov	bl,partnum		; compute partition number in extended
	sub	bl,4			; table

; Now really load the first sector of the boot partition

load6:	dec	bl			; partition numbers start with 1
	xor	bh,bh
	shift	shl,bx,4		; compute offset into partition table
	add	bx,PART_OFS
	call	readsect		; read first sector of boot partition
	jc	short load8

load7:	mov	ax,-1			; return without error
	jmp	short load9

load8:	xor	ax,ax			; return with error
load9:	pop	es
	pleave
	ret

_loaddisk	endp


;
;**************************************************************************
;
; Decode device string
; Input:  BX  -  pointer to string
; Output: Carry flag set if error
; Registers changed: AX, BX
;
decodestr	proc	near

; First decode the first two letters. The string has to start either
; with 'fd' or 'hd'. We don't any other devices yet.

	cmp	byte ptr [bx + 1],'d'	; second letter has to be a 'd'
	jne	short decod8
	mov	ah,[bx+2]
	mov	al,HD_BASE
	cmp	byte ptr [bx],'h'	; check for hard disk
	je	short decod1
	mov	al,FD_BASE
	cmp	byte ptr [bx],'f'	; check for floppy disk
	jne	short decod8

; OK, we have to look for a floppy device. After the initial 'fd' a number
; has to follow. The maximum is 4 disk drives, so check that it's in the
; range 0 to 3.

	sub	ah,'0'			; check for correct number
	jb	short decod8
	cmp	ah,3
	ja	short decod8
	add	al,ah			; compute final device number
	xor	ah,ah			; partition number is always zero
	add	bx,3			; let BX point after last character
	jmp	short decod7

; Now decode the harddisk description. The initial 'hd' has to be followed
; by a letter from 'a' to 'd' indicating the drive number, followed by a
; number indicating the partition number. If the number is missing, 0 is
; assumed, which means the whole disk, e.g. the master boot block.

decod1:	sub	ah,'a'			; check for correct letter
	jb	short decod8
	cmp	ah,3
	ja	short decod8
	add	al,ah			; compute final device number
	add	bx,3			; let BX point after last character
	mov	ah,[bx]
	or	ah,ah			; end of string means "zero"
	jz	short decod7
	sub	ah,'0'			; check for correct partition
	jb	short decod8
	cmp	ah,8			; maximum number of 8 partitions
	ja	short decod8
	inc	bx

; Return with the global variables set correctly

decod7:	cmp	byte ptr [bx],0		; string has to be followed by a null
	jne	short decod8
	mov	diskid,al		; set the global variables
	mov	partnum,ah
	clc				; return without error
	jmp	short decod9

decod8:	stc				; return with error
decod9:	ret

decodestr	endp


;
;**************************************************************************
;
; Read first sector of partition. reads1 is an additional subroutine entry
; point!
; Input:  BX  -  pointer to partition table entry
;         ES  -  segment of bootblock buffer
; Output: Carry flag set if error
; Registers changed: AX, BX, CX, DX

readsect	proc	near

	mov	al,es:[bx + PART_TYPE_OFS]
	or	al,al				; check if partition table entry
	jz	short reads8			; is valid

	mov	cx,es:[bx + PART_SECT_OFS]	; get sector, track and head
	mov	dh,es:[bx + PART_HEAD_OFS]	; number from partition table
reads1:	mov	dl,diskid
	xor	bx,bx
	mov	ax,0201h			; actually read the sector
	int	13h
	jc	short reads8
	cmp	word ptr es:[BOOT_SIG_OFS],BOOT_SIG
	je	short reads9
reads8:	stc
reads9:	ret

readsect	endp


;
;**************************************************************************
;
text_end

	end

