/*
  Copyright (C) 1997  Dimitrios P. Bouras

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   For author contact information, look in the README file.
*/

#include <forms.h>
#include <stdio.h>
#include <stdlib.h>
#include <varargs.h>
#include <time.h>
#include <string.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>
#include "common.h"
#include "xisp.h"
#include "pcode.h"
#include "routing.h"
#include "xisp.xpm"
#include "version.h"
#include "help.c"

/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                         Global program storage                          |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* All application objects ("forms") are declared global */

FD_topFrame		*fd_topFrame;		/* The main application form */
FD_accountInfo	*fd_accountInfo;	/* Account information */
FD_dialInfo		*fd_dialInfo;		/* Dialing options */
FD_aboutInfo	*fd_aboutInfo;		/* Application "About" */
FD_exitDialog	*fd_exitDialog;		/* Exit dialog */
FD_ISPInfo		*fd_ISPInfo;		/* ISP selection */
FD_renameISP	*fd_renameISP;		/* Modify ISP description */
FD_helpInfo		*fd_helpInfo;		/* General GUI information */

/* The options from .xisprc go in here; defaults are held in [0] and
   all further option records are saved from [1] and unwards */

xisprc_t xispOptions[MAXNUM_ISP+1] =	/* table of options */
		 { XISPRC_DEFAULT };
int currentRC = 1;						/* sequence number of, and */
xisprc_t *p_xisprc = &xispOptions[1];	/* pointer to current option block */

/* Files */

char *rcfname;		/* file holding program parameters */
char *pipefname;	/* named pipe used for reading dialer output */
char *envfname;		/* temp file for passing the dialing environment */

/* Global variables */

int dialerFD;		/* file descriptor used for reading dialer output */
int dialerON = 0;	/* indicates whether or not a dialer process is active */
int pppdPID = 0;	/* saved process id of initiated pppd process */
int pppdStat = 0;	/* set to 1 when pppd process status has been collected */
int dialerPID = 0;	/* process id returned by dialer via named pipe */
int	connected = 0;	/* flag indicating connection status */
time_t connTime;	/* ammount of time connected */
char connSpeed[8];	/* formatted string of connection speed */
int linkOK = 0;		/* flag indicating link status */
int linkPoll = 0;	/* starts link polling after dialing is done */
char connIP[16];	/* string IP address of PPP interface when up */
int userDiscon = 0;	/* indicates user initiated disconnection request */

/* Timers for periodic events */

FL_OBJECT *btimer;	/* timer triggering browser updates from dialer output */
FL_OBJECT *ctimer;	/* timer measuring connection time */
FL_OBJECT *ltimer;	/* timer used for polling link status */


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                            Utility routines                             |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Print message together with system error message and exit. Note the
   implicit total length of MSGLEN_ERR bytes for the resulting string. */

#define MSGLEN_ERR 128

void doErr(char *msg)
{
	char emsg[MSGLEN_ERR+1];

	if (errno < sys_nerr)
		sprintf(emsg, "xISP: %s: %s\n", msg, sys_errlist[errno]);
	else
		sprintf(emsg, "xISP: %s: error #%d\n", msg, errno);
	fputs(emsg, stderr);
	exit(1);
}


/* Returns pointer to string with current time in 12-hour format */

char *strtime(void)
{
	struct timeval tv;
	time_t time;
	static char tout[16];

	gettimeofday(&tv, NULL);
	time = tv.tv_sec;
	strftime(tout, 16, "%I:%M:%S %p", localtime(&time));
	return tout;
}


/* Prints out-of-memory diagnostic and aborts program */

void outofMem(void)
{
	fprintf(stderr, "xISP: Out of memory!\n");
	exit(-1);
}


/* Assembles the variables holding the xisprc file name, the filename of
   the named pipe used for communications with the dialer process, as
   well as the name of the dialing environment file . */

void initFnames(void)
{
	struct passwd *user;

	user = getpwuid(getuid());
	rcfname = (char *)malloc(strlen(user->pw_dir)+1+strlen(RCFNAME)+1);
	pipefname = (char *)malloc(6+strlen(user->pw_name)+strlen(PIPEFNAME)+1);
	envfname = (char *)malloc(strlen(user->pw_dir)+1+strlen(ENVFNAME)+1);
	if (rcfname != NULL && pipefname != NULL && envfname != NULL) {
		strcpy(rcfname, user->pw_dir); strcat(rcfname, "/");
		strcat(rcfname, RCFNAME);
		strcpy(pipefname, "/tmp/"); strcat(pipefname, PIPEFNAME);
		strcat(pipefname, "."); strcat(pipefname, user->pw_name);
		strcpy(envfname, user->pw_dir); strcat(envfname, "/");
		strcat(envfname, ENVFNAME);
	}
	else outofMem();
}


/* Initializes the xisprc data structure. */

void initXisprc(int rec)
{
	int i;
	xisprc_t *p = &xispOptions[rec];

	p->descr[0] = 0;
	p->autoRedial = 0;
	p->account[0] = 0;
	p->passwd[0] = 0;
	p->maxAttempts = MAXNUM_RETRY;
	p->sleepDelay = MAXSEC_DELAY;
	p->numPhones = 0;
	for (i=0; i<MAXNUM_TELS; i++)
		p->phone[i][0] = 0;
	p->numSlines = 0;
	for (i=0; i<MAXNUM_SLINES; i++)
		p->sline[i][0] = 0;
}


/* Read all xisprc data records from the user file, or create
   a new file if it's the first time this program is run. */

void readAllXisprc(void)
{
	FILE *rcfp;
	int rw, i;

	rcfp = fopen(rcfname, "r");
	if (rcfp != NULL) {
		rw = fread(&xispOptions[1], sizeof(xisprc_t), MAXNUM_ISP, rcfp);
		if (rw != MAXNUM_ISP)
			doErr("readAllXisprc: fread");
		fclose(rcfp);
	}
	else {
		for (i=0; i<(MAXNUM_ISP+1); i++)
			initXisprc(i);
		rcfp = fopen(rcfname, "w");
		if (rcfp == NULL)
			doErr("readAllXisprc: creating: fopen");
		rw = fwrite(&xispOptions[1], sizeof(xisprc_t), MAXNUM_ISP, rcfp);
		if (rw != MAXNUM_ISP)
			doErr("readAllXisprc: creating: fwrite");
		fclose(rcfp);
	}
}

/* Read the specified xisprc data record from the user file */

void readXisprc(int rec)
{
	FILE *rcfp;
	int rw;

	rcfp = fopen(rcfname, "r");
	if (rcfp != NULL) {
		if (fseek(rcfp, (rec-1)*sizeof(xisprc_t), SEEK_SET))
			doErr("readXisprc: fseek");
		rw = fread(&xispOptions[rec], sizeof(xisprc_t), 1, rcfp);
		if (rw != 1)
			doErr("readXisprc: fread");
		fclose(rcfp);
	}
	else
		doErr("readXisprc: fopen");
}


/* Write the specified xisprc data record to the user file */

void writeXisprc(int rec)
{
	FILE *rcfp;
	int rw;

	rcfp = fopen(rcfname, "r+");
	if (rcfp != NULL) {
		if (fseek(rcfp, (rec-1)*sizeof(xisprc_t), SEEK_SET))
			doErr("writeXisprc: fseek");
		rw = fwrite(&xispOptions[rec], sizeof(xisprc_t), 1, rcfp);
		if (rw != 1)
			doErr("writeXisprc: fwrite");
		fclose(rcfp);
	}
	else
		doErr("writeXisprc: fopen");
}


/* Create and/or open a named pipe for reading only */

int namedPipe(char *fname)
{
	struct stat st;
	int create, fd;

	if (access(fname, F_OK) == -1)				/* check to see if it exists */
		create = 1;								/* nope, creation required */
	else {
		stat(fname, &st);						/* yes, get the node status */
		if (!S_ISFIFO(st.st_mode)) {			/* is it a FIFO? */
			unlink(fname);						/* nope, delete it */
			create = 1;							/* indicate creation required */
		}
		else create = 0;						/* it's a FIFO, all OK */
	}
	if (create)									/* was creation requested? */
		if (mknod(pipefname, S_IFIFO+0600, 0))	/* yes create the FIFO node */
			doErr("namedPipe: mknod");
	fd = open(fname, O_RDONLY|O_NDELAY);		/* and open it for writing */
	if (fd < 0)									/* error means no process has */
		doErr("namedPipe: open");				/* opened it for reading */
	return fd;									/* return the descriptor */
}


/* Search a string for multiple characters returning first occurence */

static char *strfstr(char *haystack, char *needles)
{
	char cn, *hp;

	while ((cn=*needles)) {						/* search for all needles */
		hp = strchr(haystack, cn);				/* found in haystack? */
		if (hp != NULL)							/* yes, return pointer to */
			return(hp);							/* location of matched char */
		++needles;								/* nope, get next needle */
	}
	return(NULL);								/* nothing found */
}


/* Write printf style in the browser object. Note the implicit
   total length of MSGLEN_BROWSER bytes for the resulting string.
   A routine to flush the buffer used by bprintf is also provided. */

#define MSGLEN_BROWSER (MAXBUF_CHILD*2)

char bmsg[MSGLEN_BROWSER+1] = {0};	/* the string buffer used by bprintf() */
char btmp[MAXBUF_CHILD+1] = {0};	/* temporary buffer for new strings */
char *where = bmsg;					/* incomplete line continuation pointer */
int needNL = 0;						/* and the continuation indicator */

void bDoBuf(void)
{
	char *nl;

	while ((nl= strfstr(bmsg,"\r\n"))!=NULL) {	/* string contains CR or LF? */
		*nl = 0;								/* yes, mark it as last char */
		if (needNL) {							/* redisplay string if the */
			fl_replace_browser_line(			/* last line input lacked */
				fd_topFrame->lstBrowser,		/* an accepted new line char */
				fl_get_browser_maxline(			/* so replace instead */
					fd_topFrame->lstBrowser),	/* of adding the line */
				bmsg);
			needNL = 0;							/* done with line with no NL */
		}
		else
			fl_addto_browser(fd_topFrame->		/* display string normally */
							 lstBrowser, bmsg);	/* on the browser object */
		strcpy(bmsg, nl+1);						/* move rest to beginning */
	}
}

int bprintf(va_alist) va_dcl
{
	int bw, pending = 0;
	va_list ap;
	char *fmt;
	static int tot = 0;

	va_start(ap);								/* start variable arg list */
	fmt = va_arg(ap, char*);					/* first string is format */
	bw = vsprintf(btmp, fmt, ap);				/* pass the rest to vsprintf */
	va_end(ap);									/* end variable arg list */
	if ((tot+bw) < (MSGLEN_BROWSER-1))			/* do we have space for new? */
		strcat(where, btmp);					/* yup, tack it on the end */
	else {										/* nope, so */
		strcat(where, "\n");					/* end the string here */
		pending = 1;							/* and indicate new pending */
	}
	bDoBuf();									/* process the message buf */
	if (pending) {								/* pending new string? */
		strcpy(bmsg, btmp);						/* yup, copy it in the buffer */
		bDoBuf();								/* process the buffer again */
	}
	tot = strlen(bmsg);							/* total chars so far */
	where = bmsg + tot;							/* pick up from where we left */
	if (tot) {									/* any trailing characters? */
		if (needNL)
			fl_replace_browser_line(			/* last line input lacked */
				fd_topFrame->lstBrowser,		/* an accepted new line char */
				fl_get_browser_maxline(			/* so replace instead */
					fd_topFrame->lstBrowser),	/* of adding the line */
				bmsg);
		else									/* unless it just happened */
			fl_addto_browser(fd_topFrame->		/* in which case add the */ 
							 lstBrowser, bmsg);	/* new line at the end */
		needNL = 1;								/* indicate we need NL */
	}
	return bw;									/* return bytes written */
}

void bflush()
{
	if ( *bmsg ) {								/* if leftover chars exist */
		fl_replace_browser_line(				/* last line input lacked */
			fd_topFrame->lstBrowser,			/* an accepted new line char */
			fl_get_browser_maxline(				/* so replace it */
				fd_topFrame->lstBrowser),
				bmsg);
		*bmsg = 0;								/* indicate nothing here */
		needNL = 0;								/* no incomplete line */
	}
	where = bmsg;								/* and start all over again */
}


/* Executes pppd process with the variable argument list given. The
   variable argument list is terminated by a (char *)0 pointer. It also
   initializes the browser-update timer. */

#define MAXARGS_CHILD 16

void pppd(va_alist) va_dcl
{
	int i;
	char *args[MAXARGS_CHILD+1];
	va_list ap;

	pppdPID = fork();							/* fork to create child */
	if (pppdPID < 0)							/* ret < 0 : fork failed */
		doErr("pppd: fork");
	if (pppdPID) {								/* in parrent process */
		dialerON = 1;							/* indicate dial in progress */
		pppdStat = 0;							/* status not collected yet */
		fl_set_timer(btimer, BU_INTERVAL);		/* start callback timer */
	}
	else {										/* in pppd process */
		va_start(ap);							/* start variable arg list */
		for (									/* parse rest of the args */
			i=0; i < MAXARGS_CHILD &&			/* to a max of MAXARGS_CHILD */
			(args[i]=va_arg(ap, char*)) !=
				(char *)0;
			i++
		);
		va_end(ap);								/* end variable arg list */
		umask(022);								/* ensure readable pid file */
		execv(args[0], args);					/* in /var/run and exec pppd */
		doErr("pppd: execvp");					/* return here means error */
	}
}


/* Create dialing environment file */

int writeISPenv(xisprc_t *p)
{
	FILE *envfp;
	char ppasswd[MAXLEN_PASSWD+1] = {0}, *pc;
	int i, fs = 0;

	envfp = fopen(envfname, "w");				/* create the file */
	if (envfp == NULL)							/* bail out on error */
		doErr("writeISPenv, fopen");
	pdecode(ppasswd, p->passwd);				/* decrypt password */
	fprintf(envfp, "%s\n", p->account);			/* write all needed */
	fprintf(envfp, "%s\n", ppasswd);			/* dialing variables */
	fprintf(envfp, "%d\n", p->maxAttempts);
	fprintf(envfp, "%d\n", p->sleepDelay);
	fprintf(envfp, "%d\n", p->numPhones);
	for (i=0; i<p->numPhones; i++)
		fprintf(envfp, "%s\n", p->phone[i]);
	fprintf(envfp, "%d\n", p->numSlines);
	for (i=0; i<p->numSlines; i++) {			/* make sure there are */
		if ((pc=strstr(p->sline[i], "%s"))		/* only two %s formatting */
			!= NULL && pc[-1] != '%')			/* strings in all script */
		{										/* lines passed on to */
			if (++fs > 2)						/* xispdial, since these */
				pc[0] = pc[1] = ' ';			/* script lines are part */
		}										/* of a printf style format */
		fprintf(envfp, "%s\n", p->sline[i]);	/* if the % is escaped using */
	}											/* '%%', then all is OK */
	fclose(envfp);								/* finally, close the file */
	return fs;									/* return # of "%s"'s */
}


/* Update all connection indicators */

void updateCONN(void)
{
	fl_set_object_label(fd_topFrame->statusText,
		(connected)? "CONN":"DISCON");
	fl_set_object_label(fd_topFrame->speedText,
		connSpeed);
	fl_set_object_label(fd_topFrame->IPText,
		connIP);
}


/* Retrieve PID of spawned pppd process from RUNDIR/ppp? */

int getPppdPID(char *IF)
{
	char fname[32] = RUNDIR, buf[16];
	int fd, br;

	strcat(fname, "/"); strcat(fname, IF);		/* form the PID file */
	strcat(fname, ".pid");
	if ((fd = open(fname, O_RDONLY)) < 0)		/* open it for reading */
		return 0;								/* if error, return 0 */
	br = read(fd, buf, 16);						/* read the PID string */
	close(fd);									/* close the file */
	if (br < 0)									/* if error in reading */
		return 0;								/* return 0 */
	buf[br] = 0;								/* mark end of string */
	return atoi(buf);							/* and return converted */
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |          Routine adjusting possible actions according to state          |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void possibleActions(int state)
{
	switch (state) {

		case DISCONNECTED: {
			fl_activate_object(fd_topFrame->conButton);
			fl_deactivate_object(fd_topFrame->intButton);
			fl_deactivate_object(fd_topFrame->disButton);
			fl_activate_object(fd_topFrame->optMenu);
    		fl_set_object_lcol(fd_topFrame->optMenu,FL_LCOL);
			fl_activate_object(fd_topFrame->hlpMenu);
    		fl_set_object_lcol(fd_topFrame->hlpMenu,FL_LCOL);
			fl_activate_object(fd_topFrame->autoRedial);
			break;
		}

		case DIALING: {
			fl_deactivate_object(fd_topFrame->conButton);
			fl_activate_object(fd_topFrame->intButton);
			fl_deactivate_object(fd_topFrame->disButton);
			fl_deactivate_object(fd_topFrame->optMenu);
    		fl_set_object_lcol(fd_topFrame->optMenu,FL_INACTIVE);
			fl_deactivate_object(fd_topFrame->hlpMenu);
    		fl_set_object_lcol(fd_topFrame->hlpMenu,FL_INACTIVE);
			fl_deactivate_object(fd_topFrame->autoRedial);
			break;
		}

		case CONNECTED: {
			fl_deactivate_object(fd_topFrame->conButton);
			fl_deactivate_object(fd_topFrame->intButton);
			fl_activate_object(fd_topFrame->disButton);
			fl_deactivate_object(fd_topFrame->optMenu);
    		fl_set_object_lcol(fd_topFrame->optMenu,FL_INACTIVE);
			fl_deactivate_object(fd_topFrame->hlpMenu);
    		fl_set_object_lcol(fd_topFrame->hlpMenu,FL_INACTIVE);
			fl_deactivate_object(fd_topFrame->autoRedial);
			break;
		}

		default: break;
	}
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                      Dialer output parsing routines                     |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

/* Note the maximum field length defined by
   MAXLEN_FIELD, used by the generic parser doutWait() */

#define MAXLEN_FIELD 64

char *doutWait(char *buf, char *field, char *ecs)
{
	char *p1, *p2;
	static int next = 0, nc;
	static char rstr[MAXLEN_FIELD];

	if (next) {									/* processing field string? */
		p2 = strfstr(buf, ecs);					/* yup, find ending char */
		if (p2 != NULL) {						/* if found */
			strncpy(&rstr[nc], buf,				/* save it locally */
					(int)(p2-buf));
			rstr[nc+(int)(p2-buf)] = 0;			/* terminate it */
			return rstr;						/* and return its address */
		}
		else {
			bprintf("%s: %s!\n", "doutWait",	/* ec not found: dialer */
		 "invalid output from dialer process");	/* output is screwed up */
			return NULL;
		}
	}

	p1 = strstr(buf, field);					/* wanted field in there? */
	if (p1 != NULL) {							/* yup, parse to ending ec */
		p2 = strfstr(p1, ecs);
		if (p2 != NULL) {						/* if found */
			nc = (int)(p2-p1)-strlen(field);	/* string has so many chars */
			strncpy(rstr, p1+strlen(field),		/* save it locally */
					nc);
			rstr[nc] = 0;						/* terminate it */
			return rstr;						/* and return its address */
		}
		else {									/* oops, string ends */
			strcpy(rstr, p1+strlen(field));		/* copy what's there */
			next = 1;							/* take note for next call */
			nc = strlen(buf)-(int)(p1-buf);
			return NULL;
		}
	}
	return NULL;
}

/* Parse dialer output for its PID number */

int doutPID(char *buf)
{
	char pids[16], *p;

	p = doutWait(buf, "PID=", ".");				/* look for PID=?????. */
	if (p != NULL) {							/* found it? */
		strcpy(pids, p);						/* yup, copy the string */
		dialerPID = atoi(pids);					/* convert and save it */
		possibleActions(DIALING);				/* mark possible actions */
		return 1;								/* indicate found */
	}
	else										/* nope */
		return 0;								/* indicate not found */
}

/* Parse dialer output for CONNECT string from modem */

int doutCONN(char *buf)
{
	char speeds[32], *p;
	unsigned speed;

	p = doutWait(buf, "CONN", "/\r\n");			/* look for CONNECT */
	if (p != NULL) {							/* found it? */
		strcpy(speeds, strchr(p, ' '));			/* yup, copy the string */
		speed = atoi(speeds);					/* convert and save it */
		connected = 1;							/* update variables */
		connTime = 0;
		sprintf(connSpeed, "%d,%d",
				speed/1000, speed%1000);
		strcpy(connIP, EMPTY_IP);
		fl_set_object_label(					/* update indicators */
			fd_topFrame->timeText, EMPTY_TIME);
		updateCONN();
		fl_set_timer(ctimer, CT_INTERVAL);		/* start connection timer */
		linkPoll = 1;							/* clear link-poll activation */
		return 1;								/* indicate found */
	}
	else										/* nope */
		return 0;								/* indicate not found */
}

/* Parse dialer output for TIMEOUT string from dialer */

int doutTIMEOUT(char *buf)
{
	char *p;

	p = doutWait(buf, "TIME", "\n");
	if (p != NULL) {
		connected = 0;							/* invalidate connection */
		linkPoll = 0;							/* no link-poll activation */
		fl_set_timer(ctimer, 0.0);				/* stop connection timer */
		strcpy(connSpeed, EMPTY_SPEED);			/* reset speed indicator */
		updateCONN();							/* update display */
		possibleActions(DIALING);				/* mark possible actions */
		return 1;								/* indicate found */
	}
	else
		return 0;
}

/* Parse dialer output for usefull output */

void doutParse(char *buf)
{
	if (! dialerPID) doutPID(buf);				/* parse for dialer PID */
	if (! connected) doutCONN(buf);				/* for CONNECTED info */
	if (dialerON && connected)					/* and for TIMEOUT */
		doutTIMEOUT(buf);
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                              Misc callbacks                             |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doCTupdate(FL_OBJECT *obj, long param)
{
	char tout[16];

	connTime += (int)CT_INTERVAL;				/* increment by interval */
	strftime(tout, 16, "%H:%M:%S",				/* format time HH:MM:SS */
			 gmtime(&connTime));
	fl_set_object_label(fd_topFrame->timeText,	/* print it, and */
						tout);
	fl_set_timer(ctimer, CT_INTERVAL);			/* restart timer */
}

void doLPupdate(FL_OBJECT *obj, long param)
{
	char *p;
	static int justConnected = 1;

	if (pppdPID && (p = pppAddr()) != NULL) {	/* pppd running and link OK? */
		if (!linkOK) {							/* yes, just now? */
			strcpy(connIP, p);					/* yup, copy new IP address */
			updateCONN();						/* update indicators */
			linkOK = 1;							/* and mark link as OK */
			justConnected = 0;					/* init for next time */
			pppdPID = getPppdPID("ppp0");		/* get PID of spawned pppd */
		}
		fl_set_timer(ltimer, SP_INTERVAL);		/* restart timer */
	}
	else {
		if (pppdPID && justConnected) {			/* give it a second chance */
			justConnected = 0;					/* but not a third :) */
			fl_set_timer(ltimer, SP_INTERVAL);	/* restart status poll timer */
			return;
		}
		justConnected = 1;						/* init for next time */
		bprintf("PPP link is down.\n");			/* nope, print message */
		fl_set_timer(ctimer, 0.0);				/* disable timers */
		fl_set_timer(ltimer, 0.0);
		if (pppdPID)							/* if not dead already */
			kill(pppdPID, SIGINT);				/* terminate pppd process */
		pppdPID = 0;							/* indicate that it's gone */
		connected = 0;							/* update variables */
		connTime = 0;
		linkOK = 0;
		linkPoll = 0;
		strcpy(connSpeed, EMPTY_SPEED);
		strcpy(connIP, EMPTY_IP);
		updateCONN();							/* update indicators */
		possibleActions(DISCONNECTED);			/* mark possible actions */
		if (!userDiscon &&						/* if not user disconnect */
			p_xisprc->autoRedial) {				/* and auto-redial set, then */
			fl_call_object_callback(			/* "push Connect" again */
				fd_topFrame->conButton);
		}
		userDiscon = 0;							/* reset indicator */
	}
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |             Callback for browser updates from dialer output             |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doBUpdate(FL_OBJECT *obj, long param)
{
	int br, i, stat;
	char buf[MAXBUF_CHILD+1];
	static int justStarted = 1;

	if (dialerON) {								/* if dial in progress */
		br = read(dialerFD, &buf,MAXBUF_CHILD);	/* read a buffer full */
		if (br > 0) {							/* if read OK */
			buf[br] = 0;						/* indicate string end */
			doutParse(buf);						/* parse dialer output */
			bprintf("%s", buf);					/* print it */
			fl_set_timer(btimer, BU_INTERVAL);	/* restart timer */
			if (justStarted) justStarted = 0;	/* !justStarted any more */
		}
		else if (br < 0) {						/* read failed */
			if (errno != EAGAIN)				/* pipe output unavailable? */
				doErr("duBUpdate: read");		/* no, abort with diagnostic */
			fl_set_timer(btimer, BU_INTERVAL);	/* yes, restart timer */
		}
		else {									/* EOF on pipe output */
			if (justStarted) {					/* give xispdial another */
				justStarted = 0;				/* chance to open the pipe */
				fl_set_timer(btimer,			/* if it hasn't managed yet */
							 BU_INTERVAL);		/* due to machine load */
				return;
			}
			justStarted = 1;					/* reset for next time */
			dialerON = 0;						/* indicate dial done */
			dialerPID = 0;						/* and no dialer process */
			bflush();							/* flush browser output */
			if (connected)						/* check connection status */
				possibleActions(CONNECTED);		/* and mark possible actions */
			else
				possibleActions(DISCONNECTED);
			if (linkPoll)						/* activate link polling */
				fl_set_timer(ltimer,			/* timer if we have been */
							 SP_INTERVAL);		/* cleared to do so */
		}
		if (! pppdStat) {						/* only once */
			i = waitpid(pppdPID, &stat, 1);		/* get pppd return status */
			if (i < 0)							/* ret < 0 : wait failed */
				doErr("doBUpdate: waitpid");	/* abort with diagnostic */
			pppdStat = 1;						/* indicate status collected */
			if (stat) {							/* if pppd didn't fare well */
				bprintf("%s: %s %d!\n",			/* print a message */
					"doBUpdate",
					"pppd returned", stat);
				dialerON = 0;					/* indicate dial aborted */
				dialerPID = 0;					/* and no dialer process */
				fl_activate_object(				/* finally reactivate the */
					fd_topFrame->conButton);	/* connection button */
			}
		}
	}
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                      Callback routines for topFrame                     |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doConnect(FL_OBJECT *obj, long param)
{
	int fs;

	fl_deactivate_object(fd_topFrame->conButton);	/* disable button */
	fs = writeISPenv(p_xisprc);						/* create the env file */
	if (fs != 2) {									/* exactly 2 "%s"'s? */
		bprintf("Invalid number of '%%s' in the\n"	/* nope, print message */
			"user specified script lines!\n");
		fl_activate_object(fd_topFrame->conButton);	/* enable button */
		unlink(envfname);							/* delete the script */
		return;										/* and do nothing */
	}
	bprintf("Connection initiated: %s\n",
			strtime());
	pppd(PPPD, "file", PPPD_OPTIONS,				/* and exec pppd */
		 "0.0.0.0:0.0.0.0", "netmask",
		 "255.255.255.0", "connect",
		 PPPD_CONNECT, (char *)0 );
	return;
}

void doInterrupt(FL_OBJECT *obj, long param)
{
	fl_deactivate_object(fd_topFrame->intButton);	/* disable button */
	fl_call_object_callback(btimer);				/* flush/check status */
	if (dialerPID) {								/* dialer still there? */
		bprintf("\nDialing interrupted: %s\n",
				strtime());
		if (kill(dialerPID, SIGTERM))				/* yup, terminate it */
			doErr("doInterrupt: kill");
	}
	connected = 0;									/* invalidate connection */
	linkPoll = 0;									/* no link polling */
	return;
}

void doDisconnect(FL_OBJECT *obj, long param)
{
	userDiscon = 1;									/* set indicator */
	fl_deactivate_object(fd_topFrame->disButton);	/* disable button */
	bprintf("Disconnection: %s\n",
			strtime());
	pppdPID = getPppdPID("ppp0");					/* PID of spawned pppd */
	if (pppdPID)									/* interrupt pppd proc */
		kill(pppdPID, SIGINT);
	pppdPID = 0;									/* indicate no pppd proc */
	possibleActions(DISCONNECTED);					/* mark possible actions */
	fl_call_object_callback(ltimer);				/* update link status */
	return;
}

void doOptions(FL_OBJECT *obj, long param)
{
	int selection, i;
	char dflt[4], phone[MAXNUM_TELS*(MAXLEN_PHONE+1)+1] = {0},
		 ppasswd[MAXLEN_PASSWD+1] = {0}, *p;

	fl_deactivate_form(fd_topFrame->topFrame);
	selection = fl_get_menu(obj);
	switch (selection) {
		case OPTIONS_ISP: {
			fl_clear_browser(fd_ISPInfo->ISPBrowser);
			for (i=1; i<(MAXNUM_ISP+1); i++) {
				p = xispOptions[i].descr;
				fl_add_browser_line(fd_ISPInfo->ISPBrowser, (*p)?p:"~");
			}
			fl_show_form(fd_ISPInfo->ISPInfo, FL_PLACE_MOUSE,
						 FL_TRANSIENT, "ISP Selection");
			fd_ISPInfo->ldata = 0;
			fl_deactivate_object(fd_ISPInfo->ISPNameEdit);
			fl_select_browser_line(fd_ISPInfo->ISPBrowser, currentRC);
			break;
		}

		case OPTIONS_ACCOUNT: {
			for (i=0; i<p_xisprc->numPhones; i++) {
				if (i) strcat(phone, ";");
				strcat(phone, p_xisprc->phone[i]);
			}
			fl_set_input(fd_accountInfo->telInput, phone);
			fl_set_input(fd_accountInfo->accInput, p_xisprc->account);
			if (p_xisprc->passwd[0])
				pdecode(ppasswd, p_xisprc->passwd);
			fl_set_input(fd_accountInfo->pswInput, ppasswd);
			fl_show_form(fd_accountInfo->accountInfo, FL_PLACE_MOUSE,
						 FL_TRANSIENT, "Account Information");
			break;
		}

		case OPTIONS_DIALING: {
			sprintf(dflt, "%d", p_xisprc->maxAttempts);
			fl_set_input(fd_dialInfo->rtrInput, dflt);
			sprintf(dflt, "%d", p_xisprc->sleepDelay);
			fl_set_input(fd_dialInfo->dlyInput, dflt);
			for (i=0; i<MAXNUM_SLINES; i++) {
				fl_set_input(fd_dialInfo->scriptLine[i],
							 p_xisprc->sline[i]);
			}
			fd_dialInfo->ldata = 0;
			fl_show_form(fd_dialInfo->dialInfo, FL_PLACE_MOUSE,
						 FL_TRANSIENT, "Dialing Options");
		}

		default: break;
	}
	return;
}

void doHelp(FL_OBJECT *obj, long param)
{
	int selection, i;
	static int helpLoaded = 0;

	fl_deactivate_form(fd_topFrame->topFrame);
	selection = fl_get_menu(obj);
	switch (selection) {
		case HELP_ABOUT: {
			fl_show_form(fd_aboutInfo->aboutInfo, FL_PLACE_MOUSE,
						 FL_TRANSIENT, "About");
			break;
		}

		case HELP_GENERAL: {
			if (! helpLoaded) {
				for (i=0; i<MAXNUM_HELPLINES; i++)
					fl_add_browser_line(fd_helpInfo->helpBrowser,
										helpText[i]);
				helpLoaded = 1;
			}
			fl_show_form(fd_helpInfo->helpInfo, FL_PLACE_MOUSE,
						 FL_TRANSIENT, "General");
			break;
		}

		default: break;
	}
	return;
}

void doAutoRedial(FL_OBJECT *obj, long param)
{
	p_xisprc->autoRedial = fl_get_button(fd_topFrame->autoRedial);
	writeXisprc(currentRC);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                    Callback routines for accountInfo                    |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doTelNoInput(FL_OBJECT *obj, long param)
{
	static char phone[MAXNUM_TELS*(MAXLEN_PHONE+1)+1] = {0};

	strncpy(phone, (char *)fl_get_input(obj), MAXNUM_TELS*(MAXLEN_PHONE+1));
	fd_accountInfo->vdata = phone;
	return;
}

void doAccountInput(FL_OBJECT *obj, long param)
{
	static char account[MAXLEN_ACCOUNT+1] = {0};

	strncpy(account, (char *)fl_get_input(obj), MAXLEN_ACCOUNT);
	fd_accountInfo->vdata = account;
	return;
}

void doPasswdInput(FL_OBJECT *obj, long param)
{
	static char epasswd[MAXLEN_PASSWD+1] = {0};

	pencode(epasswd, (char *)fl_get_input(obj));
	fd_accountInfo->vdata = epasswd;
	return;
}

void doAccountOK(FL_OBJECT *obj, long param)
{
	char *p1, *p2;
	int i, nc;

	fl_call_object_callback(fd_accountInfo->telInput);
	p1 = fd_accountInfo->vdata;
	for (i=0; i<MAXNUM_TELS && *p1; i++) {
		p2 = strchr(p1, ';');
		if (p2 != NULL)
			nc = p2 - p1;
		else {
			nc = strlen(p1);
			p2 = p1 + nc - 1;
		}
		strncpy(p_xisprc->phone[i], p1, nc);
		p_xisprc->phone[i][nc] = 0;
		p1 = p2 + 1;
	}
	p_xisprc->numPhones = i;
	fl_call_object_callback(fd_accountInfo->accInput);
	strcpy(p_xisprc->account, (char *)fd_accountInfo->vdata);
	fl_call_object_callback(fd_accountInfo->pswInput);
	strcpy(p_xisprc->passwd, (char *)fd_accountInfo->vdata);
	writeXisprc(currentRC);
	fl_hide_form(fd_accountInfo->accountInfo);
	fl_activate_form(fd_topFrame->topFrame);
	return;
}

void doAccountCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_accountInfo->accountInfo);
	fl_activate_form(fd_topFrame->topFrame);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                     Callback routines for dialInfo                      |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doRetryInput(FL_OBJECT *obj, long param)
{
	char ns[4] = {0};

	strncpy(ns, (char *)fl_get_input(obj), 3);
	fd_dialInfo->ldata = atoi(ns);
	return;
}

void doDelayInput(FL_OBJECT *obj, long param)
{
	char ns[4] = {0};

	strncpy(ns, (char *)fl_get_input(obj), 3);
	fd_dialInfo->ldata = atoi(ns);
	return;
}

void doScriptInput(FL_OBJECT *obj, long param)
{
	static char line[MAXNUM_SLINES][MAXLEN_SLINE+1] = {{0}};

	strncpy(line[param],
			(char *)fl_get_input(fd_dialInfo->scriptLine[param]),
			MAXLEN_SLINE);
	fd_dialInfo->vdata = line;
	fd_dialInfo->ldata = param;
	return;
}

void doDialOK(FL_OBJECT *obj, long param)
{
	char *p = p_xisprc->sline[0];
	int i;

	fl_call_object_callback(fd_dialInfo->rtrInput);
	p_xisprc->maxAttempts = fd_dialInfo->ldata;
	fl_call_object_callback(fd_dialInfo->dlyInput);
	p_xisprc->sleepDelay = fd_dialInfo->ldata;
	for (i=0; i<MAXNUM_SLINES; i++)
		fl_call_object_callback(fd_dialInfo->scriptLine[i]);
	memcpy(p_xisprc->sline, fd_dialInfo->vdata,
		   MAXNUM_SLINES*(MAXLEN_SLINE+1));
	for (i=0; i<MAXNUM_SLINES && *p; ) p = p_xisprc->sline[++i];
	p_xisprc->numSlines = i;
	writeXisprc(currentRC);
	fl_hide_form(fd_dialInfo->dialInfo);
	fl_activate_form(fd_topFrame->topFrame);
	return;
}

void doDialCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_dialInfo->dialInfo);
	fl_activate_form(fd_topFrame->topFrame);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |               Callback routines for ISPInfo and renameISP               |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doISPPick(FL_OBJECT *obj, long param)
{
	if (! fd_ISPInfo->ldata)
		fl_activate_object(fd_ISPInfo->ISPNameEdit);
	fd_ISPInfo->ldata = fl_get_browser(obj);
	return;
}

void doISPPickNEdit(FL_OBJECT *obj, long param)
{
	fl_call_object_callback(fd_ISPInfo->ISPBrowser);
	fl_call_object_callback(fd_ISPInfo->ISPNameEdit);
	return;
}

void doISPOK(FL_OBJECT *obj, long param)
{
	if (fd_ISPInfo->ldata) {
		currentRC = fd_ISPInfo->ldata;
		p_xisprc = &xispOptions[currentRC];
		writeXisprc(currentRC);
		fl_set_object_label(fd_topFrame->ISPText, p_xisprc->descr);
		fl_set_button(fd_topFrame->autoRedial, p_xisprc->autoRedial);
	}
	fl_hide_form(fd_ISPInfo->ISPInfo);
	fl_activate_form(fd_topFrame->topFrame);
	return;
}

void doISPCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_ISPInfo->ISPInfo);
	fl_activate_form(fd_topFrame->topFrame);
	return;
}

void doISPNameEdit(FL_OBJECT *obj, long param)
{
	if (fd_ISPInfo->ldata) {
		writeXisprc(currentRC);
		currentRC = fd_ISPInfo->ldata;
		p_xisprc = &xispOptions[currentRC];
		fl_set_input(fd_renameISP->ISPNameInput, p_xisprc->descr);
		fl_deactivate_form(fd_ISPInfo->ISPInfo);
		fl_show_form(fd_renameISP->renameISP, FL_PLACE_MOUSE,
					 FL_TRANSIENT, "Rename ISP record");
	}
	return;
}

void doISPNameInput(FL_OBJECT *obj, long param)
{
	static char name[MAXLEN_DESCR+1] = {0};

	strncpy(name, (char *)fl_get_input(obj), MAXLEN_DESCR);
	fd_renameISP->vdata = name;
	return;
}

void doISPNameEditOK(FL_OBJECT *obj, long param)
{
	fl_call_object_callback(fd_renameISP->ISPNameInput);
	strcpy(p_xisprc->descr, (char *)fd_renameISP->vdata);
	fl_replace_browser_line(fd_ISPInfo->ISPBrowser,
							currentRC, p_xisprc->descr);
	fl_hide_form(fd_renameISP->renameISP);
	fl_activate_form(fd_ISPInfo->ISPInfo);
	return;
}

void doISPNameEditCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_renameISP->renameISP);
	fl_activate_form(fd_ISPInfo->ISPInfo);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |               Callback routines for aboutInfo and helpInfo              |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doAboutOK(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_aboutInfo->aboutInfo);
	fl_activate_form(fd_topFrame->topFrame);
	return;
}

void doHelpInfoOK(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_helpInfo->helpInfo);
	fl_activate_form(fd_topFrame->topFrame);
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |       Callback routines for exitDialog and window delete handlers       |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

void doExitOK(FL_OBJECT *obj, long param)
{
	if (dialerON && dialerPID) {
		kill(dialerPID, SIGTERM);
		sleep((int)BU_INTERVAL);
	}
	if (connected && pppdPID)
		kill(pppdPID, SIGINT);
	exit(0);
}

void doExitCancel(FL_OBJECT *obj, long param)
{
	fl_hide_form(fd_exitDialog->exitDialog);
	fl_activate_form(fd_topFrame->topFrame);
	return;
}

int deleteHandler(FL_FORM *form, void *data)
{
	fl_deactivate_form(fd_topFrame->topFrame);
	fl_show_form(fd_exitDialog->exitDialog, FL_PLACE_MOUSE,
				 FL_TRANSIENT, "Exit Dialog");
	return FL_IGNORE;
}

void doQuit(FL_OBJECT *obj, long param)
{
	fl_deactivate_form(fd_topFrame->topFrame);
	fl_show_form(fd_exitDialog->exitDialog, FL_PLACE_MOUSE,
				 FL_TRANSIENT, "Exit Dialog");
	return;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |                                 Main                                    |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

int main(int argc, char *argv[])
{
	Pixmap icon, mask;

	umask(077);										/* owner read/write only */
	pcode_init();									/* init passwd coding */
	initFnames();									/* initialize file names */
	readAllXisprc();								/* try reading RC file */
	dialerFD = namedPipe(pipefname);				/* create/open pipe */
	fl_initialize(&argc, argv, "X-ISP", NULL, 0);	/* initialize forms GUI */

	fd_topFrame = create_form_topFrame();			/* create all prog forms */
	fd_accountInfo = create_form_accountInfo();
	fd_dialInfo = create_form_dialInfo();
	fd_aboutInfo = create_form_aboutInfo();
	fd_exitDialog = create_form_exitDialog();
	fd_ISPInfo = create_form_ISPInfo();
	fd_renameISP = create_form_renameISP();
	fd_helpInfo = create_form_helpInfo();

	fl_set_atclose(deleteHandler, NULL);			/* fdelete() callback */
	fl_set_app_mainform(fd_topFrame->topFrame);		/* indicate master form */

	fl_show_form(fd_topFrame->topFrame,				/* realize it */
				 FL_PLACE_FREE, FL_FULLBORDER,
				 "X-ISP");
	fl_get_pixmap_pixmap(fd_topFrame->topIcon,		/* use as window icon */
						 &icon, &mask);				/* the same as that on */
	fl_winicon(FL_ObjWin(fd_topFrame->topIcon),		/* the main form */
			   icon, mask);

	possibleActions(DISCONNECTED);					/* mark possible actions */

	fl_do_forms();									/* and enter main loop */

	return 0;
}


/*+-------------------------------------------------------------------------+
  |                                                                         |
  |         Program forms created with fdesign and annotated by hand        |
  |                                                                         |
  +-------------------------------------------------------------------------+*/

FD_topFrame *create_form_topFrame(void)
{
  FL_OBJECT *obj;
  FD_topFrame *fdui = (FD_topFrame *) fl_calloc(1, sizeof(*fdui));

  fdui->topFrame = fl_bgn_form(FL_NO_BOX, 372, 299);
  obj = fl_add_box(FL_FLAT_BOX,0,0,372,299,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,13,12,90,40,"X-ISP");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcol(obj,FL_COL1);
    fl_set_object_lsize(obj,FL_HUGE_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_ENGRAVED_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->conButton = obj = fl_add_button(FL_NORMAL_BUTTON,13,138,80,30,
										"Connect");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doConnect,0);
  fdui->intButton = obj = fl_add_button(FL_NORMAL_BUTTON,13,178,80,30,
										"Interrupt");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doInterrupt,0);
  fdui->disButton = obj = fl_add_button(FL_NORMAL_BUTTON,13,218,80,30,
										"Disconnect");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDisconnect,0);
  fdui->quitButton = obj = fl_add_button(FL_NORMAL_BUTTON,13,258,80,30,
										 "Quit");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doQuit,0);
  obj = fl_add_box(FL_FRAME_BOX,111,10,248,39,"");
  fdui->optMenu = obj = fl_add_menu(FL_PULLDOWN_MENU,120,16,63,27,"Options");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doOptions,0);
	fl_set_menu(obj, "Select ISP|Account Data|Dialing Data");
  fdui->hlpMenu = obj = fl_add_menu(FL_PULLDOWN_MENU,311,16,35,27,"Help");
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doHelp,0);
	fl_set_menu(obj, "About|General");
  fdui->lstBrowser = obj = fl_add_browser(FL_NORMAL_BROWSER,111,63,247,108,"");
    fl_set_object_color(obj,FL_TOP_BCOL,FL_YELLOW);
    fl_set_browser_fontsize(obj,FL_NORMAL_SIZE);
  obj = fl_add_text(FL_NORMAL_TEXT,172,181,37,28,"ISP:");
    fl_set_object_boxtype(obj,FL_NO_BOX);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->ISPText = obj = fl_add_text(FL_NORMAL_TEXT,209,183,149,28,
									p_xisprc->descr);
    fl_set_object_boxtype(obj,FL_DOWN_BOX);
    fl_set_object_color(obj,FL_TOP_BCOL,FL_MCOL);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->autoRedial = obj = fl_add_checkbutton(FL_PUSH_BUTTON,145,181,29,29,
											  "ARD:");
    fl_set_object_color(obj,FL_COL1,FL_DARKCYAN);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAutoRedial,0);
	fl_set_button(obj, p_xisprc->autoRedial);
  obj = fl_add_text(FL_NORMAL_TEXT,107,257,49,28,"Speed:");
    fl_set_object_boxtype(obj,FL_NO_BOX);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,206,257,93,28,"Connect Time:");
    fl_set_object_boxtype(obj,FL_NO_BOX);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->timeText = obj = fl_add_text(FL_NORMAL_TEXT,299,259,59,28,EMPTY_TIME);
    fl_set_object_boxtype(obj,FL_DOWN_BOX);
    fl_set_object_color(obj,FL_TOP_BCOL,FL_MCOL);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->speedText = obj = fl_add_text(FL_NORMAL_TEXT,155,259,50,29,EMPTY_SPEED);
    fl_set_object_boxtype(obj,FL_DOWN_BOX);
    fl_set_object_color(obj,FL_TOP_BCOL,FL_MCOL);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,107,219,25,28,"IP:");
    fl_set_object_boxtype(obj,FL_NO_BOX);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->IPText = obj = fl_add_text(FL_NORMAL_TEXT,132,221,105,28,EMPTY_IP);
    fl_set_object_boxtype(obj,FL_DOWN_BOX);
    fl_set_object_color(obj,FL_TOP_BCOL,FL_MCOL);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,246,219,53,28,"Status:");
    fl_set_object_boxtype(obj,FL_NO_BOX);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_RIGHT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->statusText = obj = fl_add_text(FL_NORMAL_TEXT,299,221,59,28,"DISCON");
    fl_set_object_boxtype(obj,FL_DOWN_BOX);
    fl_set_object_color(obj,FL_TOP_BCOL,FL_MCOL);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->topIcon = obj = fl_add_pixmap(FL_NORMAL_PIXMAP,14,55,78,73,"");
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_pixmap_data(obj, xisp_xpm);

	btimer = fl_add_timer(FL_HIDDEN_TIMER,			/* add timer for browser */
						  0, 0, 0, 0, NULL);
	ctimer = fl_add_timer(FL_HIDDEN_TIMER,			/* connection time */
						  0, 0, 0, 0, NULL);
	ltimer = fl_add_timer(FL_HIDDEN_TIMER,			/* and link status */
						  0, 0, 0, 0, NULL);
    fl_set_object_callback(btimer, doBUpdate, 0);	/* register callbacks */
    fl_set_object_callback(ctimer, doCTupdate, 0);
    fl_set_object_callback(ltimer, doLPupdate, 0);
	fl_set_timer(btimer, 0.0);						/* initialize timers */
	fl_set_timer(ctimer, 0.0);
	fl_set_timer(ltimer, 0.0);

  fl_end_form();

  return fdui;
}

FD_accountInfo *create_form_accountInfo(void)
{
  FL_OBJECT *obj;
  FD_accountInfo *fdui = (FD_accountInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->accountInfo = fl_bgn_form(FL_NO_BOX, 237, 179);
  obj = fl_add_box(FL_FLAT_BOX,0,0,237,179,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->telInput = obj = fl_add_input(FL_NORMAL_INPUT,103,11,121,31,
									  "Phone no(s):");
    fl_set_input_maxchars(obj, MAXNUM_TELS*(MAXLEN_PHONE+1));
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doTelNoInput,0);
  fdui->accInput = obj = fl_add_input(FL_NORMAL_INPUT,103,54,121,31,
									  "Account name:");
    fl_set_input_maxchars(obj, MAXLEN_ACCOUNT);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAccountInput,0);
  fdui->pswInput = obj = fl_add_input(FL_SECRET_INPUT,103,97,121,31,
									  "Password:");
    fl_set_input_maxchars(obj, MAXLEN_PASSWD);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doPasswdInput,0);
  fdui->accOK = obj = fl_add_button(FL_RETURN_BUTTON,22,140,81,29,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAccountOK,0);
  fdui->accCancel = obj = fl_add_button(FL_NORMAL_BUTTON,135,140,81,29,
										"Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAccountCancel,0);
  fl_end_form();

  return fdui;
}

FD_dialInfo *create_form_dialInfo(void)
{
  FL_OBJECT *obj;
  FD_dialInfo *fdui = (FD_dialInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->dialInfo = fl_bgn_form(FL_NO_BOX, 230, 316);
  obj = fl_add_box(FL_FLAT_BOX,0,0,230,316,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->rtrInput = obj = fl_add_input(FL_NORMAL_INPUT,94,10,121,30,
									  "Max retries:");
    fl_set_input_maxchars(obj, 3);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doRetryInput,0);
  fdui->dlyInput = obj = fl_add_input(FL_NORMAL_INPUT,94,50,121,30,
									  "Delay (sec):");
    fl_set_input_maxchars(obj, 3);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDelayInput,0);
  fdui->dialOK = obj = fl_add_button(FL_RETURN_BUTTON,18,276,81,29,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDialOK,0);
  fdui->dialCancel = obj = fl_add_button(FL_NORMAL_BUTTON,131,276,81,29,
										 "Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doDialCancel,0);
  obj = fl_add_text(FL_NORMAL_TEXT,13,87,91,27,"Dialing script:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);

  obj = fl_add_box(FL_DOWN_BOX,17,117,197,146,"");
    fl_set_object_color(obj,FL_MCOL,FL_COL1);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->scriptLine[0] = obj = fl_add_input(FL_NORMAL_INPUT,21,122,188,17,"");
    fl_set_input_maxchars(obj, MAXLEN_SLINE);
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_color(obj,FL_MCOL,FL_MCOL);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,0);
  fdui->scriptLine[1] = obj = fl_add_input(FL_NORMAL_INPUT,21,139,188,17,"");
    fl_set_input_maxchars(obj, MAXLEN_SLINE);
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_color(obj,FL_MCOL,FL_MCOL);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,1);
  fdui->scriptLine[2] = obj = fl_add_input(FL_NORMAL_INPUT,21,156,188,17,"");
    fl_set_input_maxchars(obj, MAXLEN_SLINE);
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_color(obj,FL_MCOL,FL_MCOL);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,2);
  fdui->scriptLine[3] = obj = fl_add_input(FL_NORMAL_INPUT,21,173,188,17,"");
    fl_set_input_maxchars(obj, MAXLEN_SLINE);
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_color(obj,FL_MCOL,FL_MCOL);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,3);
  fdui->scriptLine[4] = obj = fl_add_input(FL_NORMAL_INPUT,21,190,188,17,"");
    fl_set_input_maxchars(obj, MAXLEN_SLINE);
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_color(obj,FL_MCOL,FL_MCOL);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,4);
  fdui->scriptLine[5] = obj = fl_add_input(FL_NORMAL_INPUT,21,207,188,17,"");
    fl_set_input_maxchars(obj, MAXLEN_SLINE);
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_color(obj,FL_MCOL,FL_MCOL);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,5);
  fdui->scriptLine[6] = obj = fl_add_input(FL_NORMAL_INPUT,21,224,188,17,"");
    fl_set_input_maxchars(obj, MAXLEN_SLINE);
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_color(obj,FL_MCOL,FL_MCOL);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,6);
  fdui->scriptLine[7] = obj = fl_add_input(FL_NORMAL_INPUT,21,241,188,17,"");
    fl_set_input_maxchars(obj, MAXLEN_SLINE);
    fl_set_object_boxtype(obj,FL_FLAT_BOX);
    fl_set_object_color(obj,FL_MCOL,FL_MCOL);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doScriptInput,7);

  fl_end_form();

  return fdui;
}

FD_aboutInfo *create_form_aboutInfo(void)
{
  FL_OBJECT *obj;
  FD_aboutInfo *fdui = (FD_aboutInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->aboutInfo = fl_bgn_form(FL_NO_BOX, 205, 177);
  obj = fl_add_box(FL_FLAT_BOX,0,0,205,177,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_pixmap(FL_NORMAL_PIXMAP,10,12,78,73,"");
    fl_set_object_resize(obj, FL_RESIZE_NONE);
	fl_set_pixmap_data(obj, xisp_xpm);
  obj = fl_add_text(FL_NORMAL_TEXT,88,18,74,23,"X-ISP  V.");
    fl_set_object_color(obj,FL_INDIANRED,FL_TOP_BCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,156,17,29,25,Version);
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,122,58,32,23,"by");
    fl_set_object_color(obj,FL_INDIANRED,FL_TOP_BCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,30,80,148,23,"Dimitrios P. Bouras");
    fl_set_object_color(obj,FL_INDIANRED,FL_TOP_BCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,106,37,60,23,"coded");
    fl_set_object_color(obj,FL_INDIANRED,FL_TOP_BCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->aboutOK = obj = fl_add_button(FL_RETURN_BUTTON,64,131,81,29,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doAboutOK,0);
  obj = fl_add_text(FL_NORMAL_TEXT,29,100,148,23,"dbouras@hol.gr");
    fl_set_object_color(obj,FL_INDIANRED,FL_TOP_BCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fl_end_form();

  return fdui;
}

FD_exitDialog *create_form_exitDialog(void)
{
  FL_OBJECT *obj;
  FD_exitDialog *fdui = (FD_exitDialog *) fl_calloc(1, sizeof(*fdui));

  fdui->exitDialog = fl_bgn_form(FL_NO_BOX, 230, 121);
  obj = fl_add_box(FL_FLAT_BOX,0,0,230,121,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  fdui->exitOK = obj = fl_add_button(FL_RETURN_BUTTON,19,75,81,29,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doExitOK,0);
  fdui->exitCancel = obj = fl_add_button(FL_NORMAL_BUTTON,132,75,81,29,
										 "Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doExitCancel,0);
  obj = fl_add_text(FL_NORMAL_TEXT,27,16,183,22,"Are you sure you want to");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  obj = fl_add_text(FL_NORMAL_TEXT,28,40,179,20,"EXIT   X-ISP?");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_CENTER|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fl_end_form();

  return fdui;
}

FD_ISPInfo *create_form_ISPInfo(void)
{
  FL_OBJECT *obj;
  FD_ISPInfo *fdui = (FD_ISPInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->ISPInfo = fl_bgn_form(FL_NO_BOX, 218, 222);
  obj = fl_add_box(FL_FLAT_BOX,0,0,218,222,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  obj = fl_add_text(FL_NORMAL_TEXT,9,20,67,20,"Pick ISP:");
    fl_set_object_color(obj,FL_INDIANRED,FL_MCOL);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
  fdui->ISPBrowser = obj = fl_add_browser(FL_HOLD_BROWSER,13,49,192,121,"");
    fl_set_object_color(obj,FL_TOP_BCOL,FL_DARKCYAN);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPPick,0);
    fl_set_browser_dblclick_callback(obj,doISPPickNEdit,0);
    fl_set_browser_fontsize(obj,FL_NORMAL_SIZE);
  fdui->ISPOK = obj = fl_add_button(FL_RETURN_BUTTON,12,181,81,29,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPOK,0);
  fdui->ISPCancel = obj = fl_add_button(FL_NORMAL_BUTTON,125,181,81,29,
										"Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPCancel,0);
  fdui->ISPNameEdit = obj = fl_add_button(FL_NORMAL_BUTTON,123,10,81,29,
										  "Rename");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPNameEdit,0);
  fl_end_form();

  return fdui;
}

FD_renameISP *create_form_renameISP(void)
{
  FL_OBJECT *obj;
  FD_renameISP *fdui = (FD_renameISP *) fl_calloc(1, sizeof(*fdui));

  fdui->renameISP = fl_bgn_form(FL_NO_BOX, 219, 112);
  obj = fl_add_box(FL_FLAT_BOX,0,0,219,112,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  fdui->ISPNameInput = obj = fl_add_input(FL_NORMAL_INPUT,12,27,194,31,
										  "Enter ISP description:");
    fl_set_input_maxchars(obj, MAXLEN_DESCR);
    fl_set_object_lcol(obj,FL_WHITE);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_TOP_LEFT);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE+FL_SHADOW_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPNameInput,0);
  fdui->ISPNameEditOK = obj = fl_add_button(FL_RETURN_BUTTON,12,70,81,29,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPNameEditOK,0);
  fdui->ISPNameEditCancel = obj = fl_add_button(FL_NORMAL_BUTTON,125,70,81,29,
												"Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doISPNameEditCancel,0);
  fl_end_form();

  return fdui;
}

FD_helpInfo *create_form_helpInfo(void)
{
  FL_OBJECT *obj;
  FD_helpInfo *fdui = (FD_helpInfo *) fl_calloc(1, sizeof(*fdui));

  fdui->helpInfo = fl_bgn_form(FL_NO_BOX, 507, 476);
  obj = fl_add_box(FL_FLAT_BOX,0,0,507,476,"");
    fl_set_object_color(obj,FL_INDIANRED,FL_COL1);
  fdui->helpBrowser = obj = fl_add_browser(FL_NORMAL_BROWSER,10,11,486,419,"");
    fl_set_object_color(obj,FL_TOP_BCOL,FL_YELLOW);
    fl_set_browser_fontsize(obj,FL_NORMAL_SIZE);
  fdui->helpInfoOK = obj = fl_add_button(FL_RETURN_BUTTON,416,439,81,29,"OK");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_resize(obj, FL_RESIZE_NONE);
    fl_set_object_callback(obj,doHelpInfoOK,0);
  fl_end_form();

  return fdui;
}

