/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "mutt_menu.h"

#include <string.h>
#include <stdlib.h>

typedef struct entry
{
  int tagged; /* has the user already added this alias to the list? */
  ALIAS *alias;
} ENTRY;

int alias_search (MUTTMENU *m, regex_t *re, int n)
{
  ENTRY *table = (ENTRY *) m->data;

  return (regexec (re, table[n].alias->name, 0, NULL, 0));
}

/* This is the callback routine from mutt_menuLoop() which is used to generate
 * a menu entry for the requested item number.
 */
void alias_entry (char *s, size_t slen, MUTTMENU *m, int num)
{
  char buf[SHORT_STRING] = { 0 };
  ENTRY *table = (ENTRY *) m->data;

  rfc822_write_address (buf, sizeof (buf), table[num].alias->addr);
  snprintf (s, slen, "%2d %c %-10s   %s", num+1,
	    table[num].tagged ? '*' : ' ',
	    table[num].alias->name,
	    buf);
}


static int addrbook_sort (const void *a, const void *b)
{
  ADDRESS *pa = ((ENTRY *) a)->alias->addr;
  ADDRESS *pb = ((ENTRY *) b)->alias->addr;

  if (pa->personal)
  {
    if (pb->personal)
      return (strcasecmp (pa->personal, pb->personal));
    return (-1);
  }
  else if (pb->personal)
    return (1);
  return (strcasecmp (pa->mailbox, pb->mailbox));
}

void mutt_alias_menu (char *buf, size_t buflen, ALIAS *aliases)
{
  ALIAS *aliasp;
  MUTTMENU *menu;
  ENTRY *AliasTable = NULL;
  int i, done = 0;
  char helpstr[SHORT_STRING];
  char tmp[16];

  if (!aliases)
  {
    mutt_error ("You have no aliases!");
    return;
  }

  /* tell whoever called me to redraw the screen when I return */
  set_option (OPTNEEDREDRAW);

  menu = mutt_new_menu ();
  menu->make_entry = alias_entry;
  menu->search = alias_search;
  menu->menu = MENU_ALIAS;
  menu->title = "Aliases";

  helpstr[0] = 0;
  mutt_make_help (tmp, sizeof (tmp), "Exit  ", MENU_ALIAS, OP_EXIT);
  strcat (helpstr, tmp);
  mutt_make_help (tmp, sizeof (tmp), "Select  ", MENU_ALIAS, OP_GENERIC_SELECT_ENTRY);
  strcat (helpstr, tmp);
  mutt_make_help (tmp, sizeof (tmp), "Help", MENU_ALIAS, OP_HELP);
  strcat (helpstr, tmp);
  menu->help = helpstr;

  /* count the number of aliases */
  for (aliasp = aliases; aliasp; aliasp = aliasp->next)
    menu->max++;

  menu->data = AliasTable = (ENTRY *) safe_calloc (menu->max, sizeof (ENTRY));

  for (i = 0, aliasp = aliases; aliasp; aliasp = aliasp->next, i++)
    AliasTable[i].alias = aliasp;

  qsort (AliasTable, i, sizeof (ENTRY), addrbook_sort);

  while (!done)
  {
    switch (mutt_menuLoop (menu))
    {
      case OP_GENERIC_SELECT_ENTRY:

	AliasTable[menu->current].tagged = !AliasTable[menu->current].tagged;
	if (option (OPTRESOLVE) && menu->current < menu->max - 1)
	{
	  menu->oldcurrent = menu->current++;
	  if (menu->current >= menu->top + PAGELEN)
	  {
	    menu->top = menu->current;
	    menu->redraw = REDRAW_INDEX | REDRAW_STATUS;
	  }
	  else
	    menu->redraw = REDRAW_MOTION_RESYNCH;
	}
	else
	  menu->redraw = REDRAW_CURRENT;
	break;

      case OP_EXIT:

	done = 1;
	break;
    }
  }

  for (i = 0; i < menu->max; i++)
    if (AliasTable[i].tagged)
      rfc822_write_address (buf, buflen, AliasTable[i].alias->addr);

  mutt_menuDestroy (&menu);
  safe_free ((void **) &AliasTable);
}
