/*************************************************************************
 *
 *  $RCSfile: evtque.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: jl $ $Date: 2001/03/16 15:25:50 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/


#include <sal/config.h>

#include <vos/evtque.hxx>
#include <vos/macros.hxx>
#include <vos/mutex.hxx>
#include <vos/conditn.hxx>
#include <vos/semaphor.hxx>
#include <vos/timer.hxx>
#include <vos/diagnose.hxx>


#include <hash_map>
#if STLPORT_VERSION<321
#include <slist.h>
#else
#include <slist>
#endif

#ifdef _USE_NAMESPACE
using namespace vos;
#endif

// forward declarations
class EventIdData;
class Event;

// STL types to store data efficient
typedef NAMESPACE_STD(slist)<IEventHandler*> 		HandlerList;
typedef NAMESPACE_STD(slist)<IEventQueueListener*> 	ListenerList;
typedef NAMESPACE_STD(slist)<Event*> 	 			EventQueue;
typedef NAMESPACE_STD(hash_map)
<
	EventId,
	EventIdData*,
	NAMESPACE_STD(hash)<EventId>,
	NAMESPACE_STD(equal_to)<EventId>
> 	EventIdMap;

// class for event objects to hold in the queue
class Event
{
public:
	Event(EventId _id, sal_Int32 _param, IReference*	_pObject)
		: id(_id), param(_param), pObject(_pObject)
		{ if (pObject) pObject->acquire(); }

	~Event()
		{ if (pObject) pObject->release(); }

	EventId		id;
	sal_Int32		param;
	IReference*	pObject;

private:
	Event(const Event&);
	Event& operator=(const Event&);
};

// class for event ids data
class EventIdData
{
public:
	EventIdData()
		{}

	EventIdData(const rtl::OUString& _description)
		: description(_description)
		{ // empty
		}

	::rtl::OUString	description;
	HandlerList				handlerList;
private:
	EventIdData(const EventIdData&);
	EventIdData& operator=(const EventIdData&);
};

#ifdef _USE_NAMESPACE
namespace vos {
#endif

// the implementation data for the event queue
struct EventQueue_Impl
{
	EventQueue_Impl()
		: eventIdCount(1), queueCountSemaphore(0) {}

	ListenerList 	listenerList;
	EventIdMap	 	eventIdMap;
	EventQueue		eventQueue;
	EventId			eventIdCount;
	OMutex			queueMutex;
	OSemaphore		queueCountSemaphore;
	OCondition		queueNotEmptyCondition;
};

#ifdef _USE_NAMESPACE
}
#endif

struct StaticEventIdData
{
	EventId id;
	const sal_Char*	description;
};

// add description of system events here
static StaticEventIdData predefinedData[] =
{
	{VOS_INVALID_EVENT_ID, "invalid event"}
};


/*****************************************************************************

	Default constructor

*****************************************************************************/
OEventQueue::OEventQueue()
{
	// initialize all predefined event ids

	m_pImpl = new EventQueue_Impl();

	for (sal_uInt32 i = 0; i < (sizeof(predefinedData) / sizeof(StaticEventIdData)); i++)
	{
		m_pImpl->eventIdMap[predefinedData[i].id] =
			new EventIdData(rtl::OUString::createFromAscii(predefinedData[i].description));
	}
}

/*****************************************************************************

	destructor

*****************************************************************************/
OEventQueue::~OEventQueue()
{
	// delete all stored id data
	for (EventIdMap::iterator idIter = m_pImpl->eventIdMap.begin();
		 idIter != m_pImpl->eventIdMap.end();
		 idIter++)
	{
		delete (*idIter).second;
	}

	// delete all stored events
	for (EventQueue::iterator eventIter = m_pImpl->eventQueue.begin();
		 eventIter != m_pImpl->eventQueue.end();
		 eventIter++)
	{
		delete (*eventIter);
	}

	delete m_pImpl;
}

/*****************************************************************************

	OEventQueue::registerListener()

*****************************************************************************/
sal_Bool OEventQueue::registerListener(IEventQueueListener* pListener)
{
	OGuard guard(&m_pImpl->queueMutex);

	sal_Bool ret = sal_False;

	ListenerList::iterator iter =
		NAMESPACE_STD(find)(m_pImpl->listenerList.begin(),
							m_pImpl->listenerList.end(),
							pListener);

	if (iter == m_pImpl->listenerList.end())
	{
		m_pImpl->listenerList.insert(m_pImpl->listenerList.end(), pListener);

		ret = sal_True;
	}

	return(ret);
}

/*****************************************************************************

	OEventQueue::deregisterListener()

*****************************************************************************/
sal_Bool OEventQueue::deregisterListener(IEventQueueListener* pListener)
{
	OGuard guard(&m_pImpl->queueMutex);

	sal_Bool ret = sal_False;

	ListenerList::iterator iter =
		NAMESPACE_STD(find)(m_pImpl->listenerList.begin(),
							m_pImpl->listenerList.end(),
							pListener);

	if (iter != m_pImpl->listenerList.end())
	{
		m_pImpl->listenerList.erase(iter);

		ret = sal_True;
	}

	return(ret);
}

/*****************************************************************************

	OEventQueue::registerId()

*****************************************************************************/
EventId OEventQueue::registerId(const rtl::OUString& description)
{
	OGuard guard(&m_pImpl->queueMutex);

	EventId id = m_pImpl->eventIdCount++;

	m_pImpl->eventIdMap[id] = new EventIdData(description);

	return(id);

}

/*****************************************************************************

	OEventQueue::deregisterId()

*****************************************************************************/
sal_Bool OEventQueue::deregisterId(EventId id)
{
	OGuard guard(&m_pImpl->queueMutex);

	sal_Bool ret = sal_False;

	EventIdMap::iterator iter = m_pImpl->eventIdMap.find(id);

	if (iter != m_pImpl->eventIdMap.end())
	{
		delete (*iter).second;

		m_pImpl->eventIdMap.erase(iter);

		ret	= sal_True;
	}

	return(ret);
}

/*****************************************************************************

	OEventQueue::getDescription()

*****************************************************************************/
sal_Bool OEventQueue::getDescription( EventId id, rtl::OUString& buffer ) //, sal_uInt32 bufferLen)
{
	OGuard guard(&m_pImpl->queueMutex);

	sal_Bool ret	= sal_False;

	EventIdMap::iterator iter = m_pImpl->eventIdMap.find(id);
	
	if (iter != m_pImpl->eventIdMap.end())
	{
/*
  if (buffer)
  {
  sal_uInt32 len = VOS_MIN(bufferLen - 1, (*iter).second->description.getLength());
  
  strncpy(buffer, (*iter).second->description.getStr(), len+1);
  }*/
		
		buffer = rtl::OUString((*iter).second->description.getStr());
		ret	= sal_True;
	}

	return(ret);
}

/*****************************************************************************

	OEventQueue::registerHandler()

*****************************************************************************/
sal_Bool OEventQueue::registerHandler(EventId id, IEventHandler* pHandler)
{
	sal_Bool ret = sal_False;

	if (id > VOS_INVALID_EVENT_ID)
	{
		OGuard guard(&m_pImpl->queueMutex);

		EventIdMap::iterator iter = m_pImpl->eventIdMap.find(id);

		if (iter == m_pImpl->eventIdMap.end())
		{
			// this id was not registered yet.
			// if the id is not predefined, the call will fail
			if (id <= VOS_MAX_PREDEF_EVENT_ID)
			{
				iter = m_pImpl->eventIdMap.insert(
					EventIdMap::value_type(id,
										   new EventIdData(rtl::OUString::createFromAscii("no description")))).first;
			}
		}

		if (iter != m_pImpl->eventIdMap.end())
		{
			// append handler to existing handler list
			HandlerList::const_iterator handlerIter =
				NAMESPACE_STD(find)((*iter).second->handlerList.begin(),
									(*iter).second->handlerList.end(),
									pHandler);

			if (handlerIter == (*iter).second->handlerList.end())
			{
				(*iter).second->handlerList.push_front(pHandler);
				ret = sal_True;
			}
		}
	}

	return(ret);
}

/*****************************************************************************

	OEventQueue::deregisterHandler()

*****************************************************************************/
sal_Bool OEventQueue::deregisterHandler(EventId id, IEventHandler* pHandler)
{
	OGuard guard(&m_pImpl->queueMutex);

	sal_Bool ret = sal_False;

	EventIdMap::const_iterator iter = m_pImpl->eventIdMap.find(id);

	if (iter != m_pImpl->eventIdMap.end())
	{
		HandlerList::iterator handlerIter =
			NAMESPACE_STD(find)((*iter).second->handlerList.begin(),
								(*iter).second->handlerList.end(),
								pHandler);

		if (handlerIter != (*iter).second->handlerList.end())
		{
			(*iter).second->handlerList.erase(handlerIter);
			ret = sal_True;
		}
		// else -> the handler was not registered for this id, call failed
	}
	// else -> this id was not registered yet, call failed

	return(ret);
}

/*****************************************************************************

	OEventQueue::getHandlerCount()

*****************************************************************************/
sal_uInt32 OEventQueue::getHandlerCount(EventId id) const
{
	OGuard guard(&m_pImpl->queueMutex);

	sal_uInt32 ret = 0;

	EventIdMap::const_iterator iter = m_pImpl->eventIdMap.find(id);

	if (iter != m_pImpl->eventIdMap.end())
	{
		ret = (*iter).second->handlerList.size();
	}
	return(ret);
}

/*****************************************************************************

	OEventQueue::postEvent()

*****************************************************************************/
EventHandle OEventQueue::postEvent(EventId id, sal_Int32 param, IReference* pObject)
{
	OGuard guard(&m_pImpl->queueMutex);

	Event* pEvent = new Event(id, param, pObject);

	// push event to queue, increment counter semaphore and
	// set condition, that indicates any events are pending
	m_pImpl->eventQueue.insert(m_pImpl->eventQueue.end(), pEvent);
	m_pImpl->queueCountSemaphore.release();
	m_pImpl->queueNotEmptyCondition.set();

	// call all queue listeners
	ListenerList::iterator iter = m_pImpl->listenerList.begin();

	while (iter != m_pImpl->listenerList.end())
	{
		(*iter++)->eventPosted(this);
	}

	return((EventHandle)pEvent);
}

/*****************************************************************************

	OEventQueue::removeEvent()

*****************************************************************************/
sal_Bool OEventQueue::removeEvent(EventHandle handle)
{
	OGuard guard(&m_pImpl->queueMutex);

	sal_Bool ret = sal_False;

	EventQueue::iterator iter =
		NAMESPACE_STD(find)(m_pImpl->eventQueue.begin(),
							m_pImpl->eventQueue.end(),
							(Event*) handle);

	if (iter != m_pImpl->eventQueue.end())
	{
		if (m_pImpl->queueCountSemaphore.tryToAcquire())
		{
			// specified event found and got ownership of it.
			// remove it from the queue, decrement counter semaphore
			// and - if counter reach zero - reset condition to indicate
			// that the queue is empty
			delete (*iter);
			m_pImpl->eventQueue.erase(iter);
			if (m_pImpl->eventQueue.size() == 0)
			{
				m_pImpl->queueNotEmptyCondition.reset();
			}
			ret = sal_True;
		}
		// else -> we found the event, but any other thread has
		// got ownership of it (while dispatching) by
		// decrementing the counter semaphore. This case is only
		// possible, if this was the last event in the queue.
	}

	return(ret);
}

/*****************************************************************************

	OEventQueue::dispatchEvent()

*****************************************************************************/
sal_Bool OEventQueue::dispatchEvent(sal_Int32 timeout)
{
	sal_Bool ret = sal_False;

	if (timeout > 0)
	{
		// wait specified time until queue is not empty
		ret = m_pImpl->queueCountSemaphore.tryToAcquire();

		if (!ret)
		{
			TTimeValue time(timeout);
			m_pImpl->queueNotEmptyCondition.wait(&time);
			// here is a potential leak:
			// if two threads waits for the condition with timeout
			// and one event is posted, _both_ thread may awake before
			// timeout is reached and only one of them will got an event.
			// The other will return sal_False _before_ the timeout is reached.
			ret = m_pImpl->queueCountSemaphore.tryToAcquire();
		}

		if (ret)
		{
			// we got the semaphore -> at least one event is pending
			ret = popAndDispatchEvent();
		}
	}
	else if (timeout < 0)
	{
		// block until queue is not empty
		m_pImpl->queueCountSemaphore.acquire();

		// we got the semaphore -> at least one event is pending
		ret = popAndDispatchEvent();
	}
	else	// timeout == 0
	{
		if (m_pImpl->queueCountSemaphore.tryToAcquire())
		{
			// we got the semaphore -> at least one event is pending
			ret = popAndDispatchEvent();
		}
	}

	return(ret);
}

/*****************************************************************************

	private OEventQueue::popAndDispatchEvent()

	implementation method

	pre: the caller has sucessfully decrement the counter semaphore
		 to take ownership of one event in the queue so size of queue
		 is greater zero.

*****************************************************************************/
sal_Bool OEventQueue::popAndDispatchEvent()
{
	EventIdData* pTmpIdData = NULL;
	Event* pEvent = NULL;
	sal_Bool ret = sal_False;

	{
		OGuard guard(&m_pImpl->queueMutex);

		if (m_pImpl->eventQueue.size())
		{
			pEvent = m_pImpl->eventQueue.front();

			m_pImpl->eventQueue.pop_front();

			EventIdMap::const_iterator iter = m_pImpl->eventIdMap.find(pEvent->id);

			if (iter != m_pImpl->eventIdMap.end())
				pTmpIdData = (*iter).second;

			if (m_pImpl->eventQueue.size() == 0)
			{
				m_pImpl->queueNotEmptyCondition.reset();
			}

			ret	= sal_True;
		}
		else
		{
			VOS_ENSHURE(sal_False, "OEventQueue::popAndDispatchEvent() precondition failed");
		}
	}

	if (pTmpIdData)
	{
		// call all handlers in reverse order of registration
		HandlerList::const_iterator handlerIter = pTmpIdData->handlerList.begin();
		sal_Bool goOn = sal_True;

		while (goOn && handlerIter != pTmpIdData->handlerList.end())
		{
			goOn = (*handlerIter++)->handleEvent(pEvent->id, pEvent->param, pEvent->pObject);
		}
	}
	delete pEvent;

	return ret;
}


/*****************************************************************************

	OEventQueue::getGlobalEventQueue()

*****************************************************************************/
OEventQueue* OEventQueue::getGlobalEventQueue()
{
	static OEventQueue globalEventQueue;

	return(&globalEventQueue);
}


