/*************************************************************************
 *
 *  $RCSfile: future.hxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 15:18:12 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/


#ifndef _VOS_FUTURE_HXX_
#define _VOS_FUTURE_HXX_

#ifndef _VOS_TYPES_HXX_
#	include <vos/types.hxx>
#endif
#ifndef _VOS_REFCOUNT_HXX_
#	include <vos/refcount.hxx>
#endif
#ifndef _VOS_CONDITN_HXX_
#	include <vos/conditn.hxx>
#endif
#ifndef _VOS_DIAGNOSE_HXX_
#	include <vos/diagnose.hxx>
#endif

#ifdef _USE_NAMESPACE
namespace vos
{     
#endif


/////////////////////////////////////////////////////////////////////////////
//
//  template class OFuture, template class OFutureRef
//
//      A future represents the result of an asynchronous operation.
//      It allows one or several "consumers" or "users" of that result
//      to be blocked only if they actually need to access the result and
//      only if the result has not yet been computed by the asynchronous
//      operation.
//      Consider a web-browser with seperate threads for downloading and
//      viewing pictures. The downloading thread offers its result (the
//      picture data) as a "future". If the display-threads trys to access
//      the data, it is blocked until the download-thread declares the
//      data as valid.
//
//      Now we might have several threads, unaware of each other, wanting to use 
//      the same data. How should one of these threads know when it's safe to 
//      delete the data represented by the future?
//      To solve this problem, we use reference-counting on the future object.
//      To hide the difficulties of calling addRef/reseaseRef operations from
//      the user, we use OFutureRef-Objects instead of OFutures. The first
//      created OFutureRef-Object actually creates the OFuture. Now we can make
//      as much copies of this OFutureRef-Object as are needed, with each copy only
//      the reference-count of the underlying OFuture is increased. Each thread
//      done with its OFutureRef can just delete it (or let it run out of scope,
//      if created on the stack). Only if the last OFutureRef is gone, the OFuture
//      will be deleted.
//


/////////////////////////////////////////////////////////////////////////////
//
//  template class OFuture
//

template <class T>
class OFuture : public VOS_NAMESPACE(OReferenceCounter, vos) 
{
public:

	OFuture() { }

	/** Assign data to the future.
		You should do this BEFORE declaring the futures data valid!
	*/
	void SAL_CALL setData(const T& data) 
		{ m_Data= data; }

	/** Access the data stored within the future.
		If the data has not been declared valid, the calling thread
		will be blocked.
	*/
	T SAL_CALL getData() 
		{
			// blocks only if data not complete
			m_Complete.wait();

			return m_Data;
		}
	
	/** Get a pointer to the data stored within the future.
		If the data has not been declared valid, the calling thread
		will be blocked.
	*/
	T* SAL_CALL getDataPtr() 
		{

			// blocks only if data not complete
			m_Complete.wait();
			
			// return result
			return &m_Data;
		}

	/** Declare the data stored in the future as valid.
		Any thread waiting on a getData-/getDataPtr-call
		will continue running now.
	*/
	void SAL_CALL dataValid() 
		{ m_Complete.set();	}

protected:
	T m_Data;
	OCondition m_Complete;		// initially not signaled
};


/////////////////////////////////////////////////////////////////////////////
//
//  template class OFutureRef
//

template <class T>
class OFutureRef : public VOS_NAMESPACE(OObject,vos) 
{
public:
	
	/** Creates an "empty" reference, use "create()" or
		assignment/copy-operator to make it a valid reference.
	*/
	OFutureRef() 
		{ m_pImpl= 0; }

	/// Overload copy-constr. to implement ref-counting.
	OFutureRef(OFutureRef<T>& ref) 
		{
			m_pImpl= ref.m_pImpl;
			m_pImpl->acquire();
		}

	/// decreases ref-count of underlying future
	~OFutureRef() 
		{
			if(m_pImpl) 
				m_pImpl->release();
		}

	/// Overload assignment-operator to implement ref-counting.
	OFutureRef<T>& SAL_CALL operator= (OFutureRef<T>& ref) 
		{
			if (m_pImpl) 
				m_pImpl->release();

			m_pImpl= ref.m_pImpl;
			m_pImpl->acquire();
			return *this;
		}

	/** Creates the underlying future. Don't try to call
		this twice on the same object.
	*/
	void SAL_CALL create() 
		{
			VOS_ASSERT(m_pImpl == 0);	// never create twice
										// alternative: deref impl and create new?
			m_pImpl= new OFuture<T>;
			m_pImpl->acquire();
		}

	/** Assign data to the future.
		You should do this BEFORE declaring the futures data valid!
	*/
	void SAL_CALL setData(const T& data) 
		{
			VOS_ASSERT(m_pImpl);
			m_pImpl->setData(data);
		}

	/** Assign data to the future.
		You should do this BEFORE declaring the futures data valid!
	*/
	const T& SAL_CALL operator= (const T& data) 
		{
			VOS_ASSERT(m_pImpl);
			m_pImpl->setData(data);
			return data;
		}


	/** Access the data stored within the future.
		If the data has not been declared valid, the calling thread
		will be blocked.
	*/
	SAL_CALL operator T()  
		{
			VOS_ASSERT(m_pImpl);
			return m_pImpl->getData();
		}

	/** Access the data stored within the future.
		If the data has not been declared valid, the calling thread
		will be blocked.
	*/
	T SAL_CALL getData() 
		{
			VOS_ASSERT(m_pImpl);
			return m_pImpl->getData();
		}
	
	/** Get a pointer to the data stored within the future.
		If the data has not been declared valid, the calling thread
		will be blocked.
	*/
	T* SAL_CALL getDataPtr() 
		{
			VOS_ASSERT(m_pImpl);
			return m_pImpl->getDataPtr();
		}

	/** Declare the data stored in the future as valid.
		Any thread waiting on a getData-/getDataPtr-call
		will continue running now.
	*/
	void SAL_CALL dataValid() 
		{
			VOS_ASSERT(m_pImpl);
			m_pImpl->dataValid();
		}

protected:
	OFuture<T>* m_pImpl;
};


#ifdef _USE_NAMESPACE
}
#endif

#endif	// _VOS_FUTURE_HXX_

