/*************************************************************************
 *
 *  $RCSfile: folder.cxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 17:03:05 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#define INCL_DOSPROCESS
#define INCL_DOSDEVICES
#define INCL_DOSFILEMGR
#define INCL_DOSERRORS
#define INCL_DOSMISC
#define INCL_DOSDEVIOCTL
#define INCL_WINWORKPLACE
#include <svpm.h>

#include <string.h>
#include <stdlib.h>
#include <itempath.hxx>
#include <longname.hxx>
#include <folder.hxx>
#include <extattr.hxx>

#ifndef _DEBUG_HXX
#include <debug.hxx>
#endif

#ifndef _EXTENDED_ATTRIBUTES_HXX
#include <ExtAttr.hxx>
#endif

#define PDATA ((FolderImpl*) pData)

static const char cBackSlash = '\\';
static const char szURL[] = ".url";
static BOOL	 bShowExt = TRUE;

//-----------------------------------------------------------------------------
// setIDPathData()
//-----------------------------------------------------------------------------

inline BOOL setIDPathData( ItemIDPath& rPath, const String& aName )
{
	PItemIDPathData pData = new ItemIDPathData;
	if( !pData ) return FALSE;
	
	// initialize data structure
	pData->nRefCount  = 0;
	pData->sPath	  = aName;
	pData->nImplFlags &= ~ITEM_IMPL_FLAG_IS_ABSOLUTE;

	rPath.SetData( pData, sizeof( ItemIDPathData ) );
	
	return TRUE;
}

/*------------------------------------------------------------------------------------
 * static methods
 *------------------------------------------------------------------------------------
 */


Folder* Folder::pRoot = 0;

void Folder::DestroyRoot()
{
	if(pRoot)
	{
		delete pRoot;
		pRoot = 0;
	}
}

Folder& Folder::GetRootFolder()
{
	if(!pRoot)
	{
		pRoot = new Folder(ItemIDPath(FOLDER_ROOT), FIND_FLAG_INCLUDE_FOLDER);
		DosExitList(EXLST_ADD | 0x00004200, (PFNEXITLIST) Folder::DestroyRoot);
	}

	return *pRoot;
}

BOOL Folder::IsAvailable()
{
	return TRUE;
}

/*------------------------------------------------------------------------------------
 * Constructor
 *------------------------------------------------------------------------------------
 */

Folder::Folder(const ItemIDPath &rPath, FindFlags aFlags)
{
	// check if rPath contains data
	ItemIDPathData * pItemData = (ItemIDPathData *) rPath.GetDataPtr();

	// test if item is valid and absolute
	if(pItemData && (pItemData->nImplFlags & ITEM_IMPL_FLAG_IS_ABSOLUTE))
	{
		String sPath = pItemData->sPath;
		if(sPath.Len() == 0)
		{
			if(WinFolderImpl::isAvailable())
				pData = new WorkplaceImpl(rPath, aFlags);
			else
				pData = new DrivesFolderImpl(rPath);
		}
        else
        {
            VolumeKind aVolumeKind;

            if(WinFolderImpl::isVirtual(sPath))
            {
                if(WinFolderImpl::isAvailable())
                {
                    BOOL bProgramsFolder = FALSE;

                    if(sPath.GetToken(0, '\\').Compare("<WP_PROGRAMS>") == COMPARE_EQUAL)
                    {
                        if(pItemData->sLocation.Len() == 0)
                        {
                            String aTmpStr = sPath.Upper();
                            aTmpStr.SetToken(0, '\\', "<WP_DESKTOP>");
                            pItemData->sLocation = WinFolderImpl::GetValidObject(aTmpStr);
                        }

                        if(sPath.GetTokenCount('\\') == 1)
                            bProgramsFolder = TRUE;
                    }

                    if(bProgramsFolder)
                        pData = new ProgFolderImpl(rPath, aFlags);
                    else
                        pData = new WinFolderImpl(rPath, aFlags);
                }
                else
                    pData = new InvalidFolderImpl(rPath);
            }
            else if(DrivesFolderImpl::GetVolumeKind(sPath, aVolumeKind))
            {
                if(aVolumeKind == VOLUME_KIND_FIXED && WinFolderImpl::isAvailable())
					pData = new WinFolderImpl(rPath, aFlags);
                else
                    pData = new DosFolderImpl(rPath, aFlags);
            }
            else
                pData = new InvalidFolderImpl(rPath);
        }
	}
	else
	{
		pData = new InvalidFolderImpl(rPath);

	}
}

/*------------------------------------------------------------------------------------
 * Destructor
 *------------------------------------------------------------------------------------
 */

Folder::~Folder()
{
	if(pData)
	{
		delete PDATA;
	}
}

/*------------------------------------------------------------------------------------
 * IsCaseSensitiv()
 *------------------------------------------------------------------------------------
 */

BOOL Folder::IsCaseSensitive() const
{
	return PDATA->isCaseSensitiv();
}

/*------------------------------------------------------------------------------------
 * GetItemIDPath()
 *------------------------------------------------------------------------------------
 */

ItemIDPath Folder::GetItemIDPath() const
{
	return PDATA->GetItemIDPath();
}

/*------------------------------------------------------------------------------------
 * GetNextItem()
 *------------------------------------------------------------------------------------
 */

BYTE Folder::GetNextItem(ItemIDPath& rPath)
{
    PItemIDPathData pIDPathData = new ItemIDPathData;

    BYTE nRet = PDATA->GetNextItem(*pIDPathData);
	if(nRet == FOLDER_OK)
	{
		pIDPathData->nRefCount	= 0;
        pIDPathData->nImplFlags = 0;

		rPath.SetData(pIDPathData, sizeof(ItemIDPathData));
		return nRet;
	}

	delete pIDPathData;
	return nRet;
}


/*------------------------------------------------------------------------------------
 * RestartEnum()
 *------------------------------------------------------------------------------------
 */

BOOL Folder::RestartEnum(FindFlags aFlags)
{
	return PDATA->RestartEnum(aFlags);
}

/*------------------------------------------------------------------------------------
 * BuildCRC()
 *------------------------------------------------------------------------------------
 */

UINT32 Folder::BuildCRC(FindFlags aFlags)
{
	return PDATA->BuildCRC(aFlags, FALSE);
}


/*------------------------------------------------------------------------------------
 * GetItemIDInfo()
 *------------------------------------------------------------------------------------
 */

BOOL Folder::GetItemIDInfo(const ItemIDPath& rPath, FastItemInfo& rInfo)
{
	if(PDATA->GetItemIDInfo(rPath, rInfo))
	{
		if(rInfo.aItemKind == ITEM_KIND_FILESYSTEM)
		{
			// search extension
			String aExt;
			USHORT n;

			// display name must not be empty
			DBG_ASSERT(rInfo.aDisplayName.Len() > 0, "DisplayName has zero length.");

			for( n = rInfo.aDisplayName.Len() - 1;
				 n > 0 && rInfo.aDisplayName.GetChar( n ) != '.';
				 n-- )
				;

			// extract extension if found
			if( n || rInfo.aDisplayName.GetChar( 0 ) == '.' )
				aExt = rInfo.aDisplayName.Copy( n );

			// cut off extension when extensions are not shown generally or if url
            if(aExt.Len() && !(rInfo.aAttributes & ITEM_FLAG_ISFOLDER))
            {
                if(aExt.ICompare(szURL) == COMPARE_EQUAL)
                {
                    rInfo.aAttributes |= ITEM_FLAG_ISLINK;
                    rInfo.aDisplayName.Erase( n );
                }
                else if(!bShowExt)
                {
                    rInfo.aDisplayName.Erase( n );
                }
			}
		}

		return TRUE;
	}

	return FALSE;
}


/*------------------------------------------------------------------------------------
 * GetFileInfo()
 *------------------------------------------------------------------------------------
 */

BOOL Folder::GetFileInfo(const ItemIDPath& rPath, FastFileInfo& rInfo)
{
	return PDATA->GetFileInfo(rPath, rInfo);
}

/*------------------------------------------------------------------------------------
 * GetLinkFileInfo()
 *------------------------------------------------------------------------------------
 */

BOOL Folder::GetLinkFileInfo(const ItemIDPath& rPath, LinkFileInfo& rInfo)
{
	return PDATA->GetLinkFileInfo(rPath, rInfo);
}

/*------------------------------------------------------------------------------------
 * GetVolumeInfo()
 *------------------------------------------------------------------------------------
 */

BOOL Folder::GetVolumeInfo(const ItemIDPath & rPath, VolumeInfo & rInfo)
{
	PItemIDPathData pIDPathData = (PItemIDPathData) rPath.GetDataPtr();

	if(pIDPathData && (pIDPathData->nImplFlags & ITEM_IMPL_FLAG_IS_VALID))
	{
		return DrivesFolderImpl::GetVolumeInfo(pIDPathData->sPath, rInfo);
	}

	return FALSE;
}

/*------------------------------------------------------------------------------------
 * IsValid()
 *------------------------------------------------------------------------------------
 */

BOOL Folder::IsValid() const
{
	if(pData)
		return PDATA->isValid();

	return FALSE;
}


/*------------------------------------------------------------------------------------
 * GetNetworkInfo()
 *------------------------------------------------------------------------------------
 */

BOOL Folder::GetNetworkInfo( const ItemIDPath & path, NetworkInfo & info )
{
	return FALSE;
}


/*------------------------------------------------------------------------------------
 * GetNameSpaceInfo()
 *------------------------------------------------------------------------------------
 */

BOOL Folder::GetNameSpaceInfo(const ItemIDPath& rPath, NameSpaceInfo& rInfo)
{
	return PDATA->GetNameSpaceInfo(rPath, rInfo );
}

/*------------------------------------------------------------------------------------
 * GetContextMenu()
 *------------------------------------------------------------------------------------
 */

IfcContextMenu * Folder::GetContextMenu(UINT32 nItems, const ItemIDPath *pIDPathList)
{
	if( nItems == 1 )
		return PDATA->GetContextMenu(pIDPathList[0]);
	else
		return NULL;
}

/*------------------------------------------------------------------------------------
 * RenameItem()
 *------------------------------------------------------------------------------------
 */

BOOL Folder::RenameItem(const ItemIDPath &rPath, ItemIDPath &rNewPath, const String &rNewName)
{
	FastItemInfo rInfo;
	String		 aNewName(rNewName);

	if(PDATA->GetItemIDInfo(rPath, rInfo))
	{
		// append extension if necessary
		if(rInfo.aItemKind == ITEM_KIND_FILESYSTEM)
		{
			String aExt(rInfo.aDisplayName);
		
			// find extension not showed
			USHORT n;
			for(n = aExt.Len() - 1;
				n > 0 && aExt.GetChar( n ) != '.';
				n-- )
				;

			// remove leading file name
			if( n ) aExt.Erase( 0, n );
		
			// get correct extension first
			if( (!bShowExt || aExt.ICompare( szURL ) == COMPARE_EQUAL)
				&& !(rInfo.aAttributes & ITEM_FLAG_ISFOLDER) )
			{
				// append extension if exists
				if(aExt.Len())
					aNewName += aExt;
			}
		}

		ItemIDPath aRelPath, aItem;

		if(rPath.Split(aRelPath, aItem))
		{
			ItemIDPath aNewItem;
			
			// let the parent folder do the renaming
			if(Folder(PDATA->GetItemIDPath() + aRelPath).RenameItem(aItem, aNewItem, aNewName))
			{
				rNewPath = aRelPath + aNewItem;
				return TRUE;
			}
		}
		else
		{
			// rPath is child of current folder
			return PDATA->RenameItem(rPath, rNewPath, aNewName);
		}
	}

	return FALSE;
}

/*------------------------------------------------------------------------------------
 * RenameFile()
 *------------------------------------------------------------------------------------
 */

BOOL Folder::RenameFile(const ItemIDPath& crOldPath, ItemIDPath& rNewPath, const String& crNewFileName )
{
    ItemIDPath aRelPath, aItem;

    if(crOldPath.Split(aRelPath, aItem))
    {
        ItemIDPath aNewItem;

        // let the parent folder do the renaming
        if(Folder(PDATA->GetItemIDPath() + aRelPath).RenameItem(aItem, aNewItem, crNewFileName))
        {
            rNewPath = aRelPath + aNewItem;
            return TRUE;
        }
    }
    else
    {
        // rPath is child of current folder
        return PDATA->RenameItem(crOldPath, rNewPath, crNewFileName);
    }

    return FALSE;
}

/*------------------------------------------------------------------------------------
 * DeleteItem()
 *------------------------------------------------------------------------------------
 */

BOOL Folder::DeleteItem(const ItemIDPath &rPath)
{
	return PDATA->DeleteItem(rPath);
}

/*------------------------------------------------------------------------------------
 * ShowExtensions()
 *------------------------------------------------------------------------------------
 */

void Folder::ShowExtensions( BOOL bFlag )
{
	bShowExt = bFlag;
}

/*------------------------------------------------------------------------------------
 * InstallChangeNotifier()
 *------------------------------------------------------------------------------------
 */

Link Folder::InstallChangeNotifier(const Link& rLink)
{

	if(PDATA->isThreadable())
	{
		Link aOldLink = PDATA->getLink();

		if(PDATA->isRunning())
		{
			PDATA->suspend();
		}
		else
		{
			PDATA->createSuspended();
			PDATA->setPriority(NAMESPACE_VOS(OThread)::TPriority_Lowest);
		}

		PDATA->setLink(rLink);
		PDATA->resume();

		return aOldLink;
	}

	return rLink;
}

/*------------------------------------------------------------------------------------
 * CreateShortcutInstance()
 *------------------------------------------------------------------------------------
 */

IfcShortcut *Folder::CreateShortcutInstance( const String& crLanguage )
{
    if(PDATA->isValid())
        return new IfcShortcutImpl(crLanguage, PDATA->GetItemIDPath());

    return NULL;
}

/*------------------------------------------------------------------------------------
 * SetTitle()
 *------------------------------------------------------------------------------------
 */

BOOL Folder::SetTitle( const ItemIDPath& crPath, const String crTitle )
{
    String   aFileName = (PDATA->GetItemIDPath() + crPath).GetHostNotationPath();
    StringEA aLongName (crTitle);

    aLongName.storeTo(aFileName, ".LONGNAME");
    return TRUE;
}

/***************************************************************************************
 *
 * DosFolderImpl - Implementation
 *
 ***************************************************************************************
 */

/*------------------------------------------------------------------------------------
 * Constructor
 *------------------------------------------------------------------------------------
 */

DosFolderImpl::DosFolderImpl(const ItemIDPath& rPath, const FindFlags aFlags) : FolderImpl(rPath)
{
	String aFileSystem;

	// query filesystem folder resides on
	DrivesFolderImpl::GetFileSystemName(m_aSearchMask, aFileSystem);

	m_bIsLongNameFS = TRUE;
	m_bIsCaseSensitiv = FALSE;

	if(aFileSystem == "FAT")
	{
		m_bIsLongNameFS = FALSE;
	}
	else if(aFileSystem == "NFS")
	{
		m_bIsCaseSensitiv = TRUE;
	}

	// this string is used whenever DosFindFirst() is called, so save it as a member
    m_aSearchMask = rPath.GetHostNotationPath();

    // GetHostNotationPath() appends \\ to drives
    if(m_aSearchMask.GetChar(m_aSearchMask.Len() - 1) != '\\')
        m_aSearchMask += '\\';

    m_aSearchMask += "*.*";

	m_nFlags = OS2FindFlags(aFlags);
	m_hDir	 = HDIR_CREATE;

	m_bIsRemote = FALSE;
	m_bIsRemovable = FALSE;


	// query volume kind
	VolumeKind aVolumeKind;
	if(DrivesFolderImpl::GetVolumeKind(m_aSearchMask, aVolumeKind))
	{
		switch(aVolumeKind)
		{
		case VOLUME_KIND_FLOPPY_525:
		case VOLUME_KIND_FLOPPY_35:
		case VOLUME_KIND_CDROM:
		case VOLUME_KIND_REMOVABLE:
			m_bIsRemovable = TRUE;
			break;

		case VOLUME_KIND_REMOTE:
			m_bIsRemote = TRUE;
			break;

		default:
			break;
		}
	}
}

/*------------------------------------------------------------------------------------
 * IsValid
 *------------------------------------------------------------------------------------
 */

BOOL DosFolderImpl::isValid() const
{
#if 0
	BOOL   aRet;
	USHORT nTokens = m_aPath.GetTokenCount( cBackSlash );

	// check if folder is root of a drive
	if( m_aPath.Len() == 2 && m_aPath.GetChar(1) == ':' )
	{
		VolumeInfo rInfo;
		
		return Folder::GetRootFolder().GetVolumeInfo( m_aPath, rInfo );
	}

	// check if folder is a	 UNC recources ( first two tokens are empty )
	else if( ( m_aPath.GetChar(0) == cBackSlash )  && ( m_aPath.GetChar(1) == cBackSlash )
			 && ( nTokens <= 4 ) )
	{
		// don't how to test further
		return nTokens == 4;
	}

	// make sure that file is really a directory
	else
	{
		FILESTATUS3 fsStatus;

		if( NO_ERROR == DosQueryPathInfo( m_aPath, FIL_STANDARD, &fsStatus, sizeof( fsStatus )))
			return fsStatus.attrFile & FILE_DIRECTORY;
	}
	
	return FALSE;
#endif

	return TRUE;
}


/*------------------------------------------------------------------------------------
 * OS2FindFlags
 *------------------------------------------------------------------------------------
 */

ULONG DosFolderImpl::OS2FindFlags(const FindFlags aFlags)
{
	ULONG nOS2Flags = FILE_NORMAL;

	// return invalid find flags when only non filesystem objects are searched for
	if(!(aFlags & FIND_FLAG_INCLUDE_FILES))
		return 0xFFFFFFFF;

	if(!(aFlags & FIND_FLAG_INCLUDE_NONFOLDER))
	{
		if(aFlags & FIND_FLAG_INCLUDE_FOLDER)
			nOS2Flags |= MUST_HAVE_DIRECTORY;
		else
			return 0xFFFFFFFF; // what are you looking for ?
	}
	else
		if(aFlags & FIND_FLAG_INCLUDE_FOLDER)
			nOS2Flags |= FILE_DIRECTORY;

	if( aFlags & FIND_FLAG_INCLUDE_HIDDEN )
		nOS2Flags |= FILE_HIDDEN;

	return nOS2Flags;
}

/*------------------------------------------------------------------------------------
 * GetNextItem
 *------------------------------------------------------------------------------------
 */

BOOL DosFolderImpl::GetNextItem( ItemIDPathData& rData )
{
	ULONG ulCount = 1;

	if( m_hDir == HDIR_CREATE )
	{
		if( NO_ERROR != DosFindFirst(m_aSearchMask, &m_hDir, m_nFlags, &rData.Cache.ffBuf,
									 sizeof(rData.Cache.ffBuf), &ulCount, FIL_STANDARD))
		{
			return FALSE;
		}

		// ignore . and ..
		if(strcmp(rData.Cache.ffBuf.achName, ".") && strcmp(rData.Cache.ffBuf.achName, ".."))
			rData.sPath = rData.Cache.ffBuf.achName;
	}

	while(rData.sPath.Len() == 0)
	{
		if(NO_ERROR != DosFindNext(m_hDir, &rData.Cache.ffBuf, sizeof(rData.Cache.ffBuf), &ulCount))
		{
			DosFindClose( m_hDir );
			m_hDir = HDIR_CREATE;
			return FALSE;
		}

		// ignore . and ..
		if(strcmp(rData.Cache.ffBuf.achName, ".") && strcmp(rData.Cache.ffBuf.achName, ".."))
			rData.sPath = rData.Cache.ffBuf.achName;
	}

	rData.nImplFlags = ITEM_IMPL_FLAG_FIND_INFO;

	// set location information if necessary
	if(!m_bIsLongNameFS)
	{
		rData.sLocation = Location();
		rData.sLocation += rData.Cache.ffBuf.achName;
	}

	return TRUE;
}





/*------------------------------------------------------------------------------------
 * RestartEnum()
 *------------------------------------------------------------------------------------
 */

BOOL DosFolderImpl::RestartEnum(const FindFlags aFlags)
{
	m_nFlags = OS2FindFlags(aFlags);

	if(m_hDir != HDIR_CREATE)
	{
		DosFindClose(m_hDir);
		m_hDir = HDIR_CREATE;
	}

	return TRUE;
}

/*------------------------------------------------------------------------------------
 * BuildCRC()
 *------------------------------------------------------------------------------------
 */

UINT32 DosFolderImpl::BuildCRC(const FindFlags aFlags, BOOL bBreak)
{
	APIRET rc;
	ULONG  ulCount = 1;
	UINT32 nCRC	   = 0;
	HDIR   hDir	   = HDIR_CREATE;
	static TimeValue aSleepTime = {0, 500};

	/*
	 * make sure that:
	 *
	 * 1) GEA2List is not overwritten by the data return from DosFindFirst(): this will cause
	 *	   DosFindNext to fail (rc == 87, ERROR_INVALID_PARAMETER)
	 * 2) GEA2List is not allocated on the stack: this will cause DodFindFirst() to fail
	 *	   (rc == 11, ERROR_BAD_FORMAT) for the structure is temporary modified.
	 */

	static PM_USHORT nBufSize	= 1024;
	static PM_USHORT nEANameLen = 9;

	PEAOP2	  pEAOP	   = (PEAOP2) new PM_BYTE[nBufSize + sizeof(GEA2LIST) + nEANameLen];
	PGEA2LIST pGEAList = (PGEA2LIST) ((PBYTE) pEAOP + nBufSize);

	// fill GEA2List structure
	pGEAList->cbList = sizeof(GEA2LIST) + nEANameLen;
	pGEAList->list[0].oNextEntryOffset = 0L;  // no more entries;
	pGEAList->list[0].cbName = nEANameLen;

	strncpy(pGEAList->list[0].szName, ".LONGNAME", pGEAList->list[0].cbName);
	pGEAList->list[0].szName[pGEAList->list[0].cbName] = '\0';

	// enable GEA2List
	pEAOP->fpGEA2List = pGEAList;

	rc = DosFindFirst(m_aSearchMask, &hDir, OS2FindFlags(aFlags), pEAOP,
					  nBufSize, &ulCount, FIL_QUERYEASFROMLIST);

	// iterate over all matching files
	for(BYTE n = 0; rc == NO_ERROR; n++)
	{
		PFILEFINDBUF3 pffdBuffer = (PFILEFINDBUF3) &pEAOP[1];
		PFEA2LIST	  pFEAList	 = (PFEA2LIST) ((PBYTE) &(pffdBuffer->attrFile) + sizeof(ULONG));
		PSZ			  pszFileName;
		PM_USHORT	  nLen;
		UINT32		 *pData;

		if(pFEAList->list[0].cbValue != 0)
		{
			// query display name
			PM_USHORT *pulMem = (PM_USHORT *) ((PBYTE) &pFEAList->list[1] + pFEAList->list[0].cbName);

			pszFileName = (PSZ) (pulMem + 2);
			nLen = pulMem[1];

			// for we don't need the data behind the EA, we can destruct it
			// and terminate the string.
			pszFileName[nLen] = '\0';
		}
		else
		{
			PBYTE pMem = (PBYTE) (pFEAList) + ((sizeof(FEA2LIST) + pFEAList->list[0].cbName + 3) & ~3);

			pszFileName = (PSZ) pMem + 1;
			nLen = * pMem;
		}

		if(nLen >= 4)
		{
			UINT32 *pInt32 = (UINT32 *) pszFileName;
			UINT32 *pLast  = (UINT32 *) (pszFileName + nLen - 4);

			// build a crc over long file names
			for(; pInt32 < pLast; pInt32++)
				nCRC ^= *pInt32;

			// don't forget last item
			nCRC ^= *pLast;
		}
		else
		{
			UCHAR szBuffer[sizeof(UINT32)] = {0};
			strncpy((char *) szBuffer, pszFileName, nLen);

			nCRC ^= *((UINT32 *) szBuffer);
		}

        // find extension
        char *cp = strrchr(pszFileName, '.');

        // ignore chaos storages
        if(!cp || (stricmp(++cp, "scc") && stricmp(++cp, "scs")))
        {
            // build CRC over last write date
            pData = (UINT32 *) &pffdBuffer->fdateLastWrite;
            nCRC ^= *pData;
        }

		if(n == 100)
		{
			// reset counter
			n = 0;

			if(bBreak)
				// make sure other threads can get some cpu time
				wait(aSleepTime);
		}

		rc = DosFindNext(hDir, pEAOP, nBufSize, &ulCount);
	}

	DosFindClose(hDir);
	delete[] pEAOP;

	return nCRC;
}

/*------------------------------------------------------------------------------------
 * isThreadable()
 *------------------------------------------------------------------------------------
 */

BOOL DosFolderImpl::isThreadable() const
{
	return !(m_bIsRemovable || m_bIsRemote);
}

/*------------------------------------------------------------------------------------
 * isCaseSensitiv()
 *------------------------------------------------------------------------------------
 */

BOOL DosFolderImpl::isCaseSensitiv() const
{
	return m_bIsCaseSensitiv;
}

/*------------------------------------------------------------------------------------
 * run()
 *------------------------------------------------------------------------------------
 */

void DosFolderImpl::run()
{
	static TimeValue aSleepTime = {5, 0};
	UINT32 n = BuildCRC(FIND_FLAG_INCLUDE_ALL, TRUE);

	wait(aSleepTime);

	while(schedule())
	{
		UINT32 nTmp = BuildCRC(FIND_FLAG_INCLUDE_ALL, TRUE);

		if(nTmp != n)
		{
			n = nTmp;
			getLink().Call(NULL);
		}

		wait(aSleepTime);
	}
}

/*------------------------------------------------------------------------------------
 * GetItemIDInfo()
 *------------------------------------------------------------------------------------
 */

BOOL DosFolderImpl::GetItemIDInfo(const ItemIDPath& rPath, FastItemInfo& rInfo)
{
	// need some information from the cache
    ItemIDPathData *pIDData = (ItemIDPathData *) rPath.GetDataPtr();

    if(!pIDData)
        return FALSE;

	// GetHostNotationPath() must be called on absolute ItemIDPaths
	String sPath = (GetItemIDPath() + rPath).GetHostNotationPath();

	/*
	 * first check if data already cached
	 */

	if(!(pIDData->nImplFlags & ITEM_IMPL_FLAG_FIND_INFO))
	{
		if(NO_ERROR == DosQueryPathInfo(sPath, FIL_STANDARD,
										&pIDData->Cache.ffBuf.fdateCreation,
										sizeof(FILESTATUS3)))
		{
			// the fields of FILEFINDBUF3, that are not filled here, are not used below.
			pIDData->nImplFlags &= ~ITEM_IMPL_FLAG_IS_CACHED;
			pIDData->nImplFlags |= ITEM_IMPL_FLAG_FIND_INFO;
		}
		else
		{
			return FALSE;
		}
	}

	// query longname title
	rInfo.aDisplayName = StringEA(sPath, ".LONGNAME");

	if(rInfo.aDisplayName.Len() == 0)
	{
		// filename is display name
		USHORT nTokens = pIDData->sPath.GetTokenCount();
		rInfo.aDisplayName = pIDData->sPath.GetToken( nTokens-1, cBackSlash );
	}

	rInfo.aItemKind = ITEM_KIND_FILESYSTEM;
	rInfo.aAttributes = ITEM_FLAG_COPYABLE | ITEM_FLAG_RENAMABLE;

	if(pIDData->Cache.ffBuf.attrFile & FILE_DIRECTORY)
		rInfo.aAttributes |=
			ITEM_FLAG_ISFOLDER	|
			ITEM_FLAG_HASSUBFOLDERS |
			ITEM_FLAG_HASITEMS;
    else
    {
		// do not test for .type EA for directories
		rInfo.aContent	= StringEA(sPath, ".TYPE");

        if(rInfo.aContent.Compare("UniformResourceLocator") == COMPARE_EQUAL)
        {
            rInfo.aAttributes |= ITEM_FLAG_ISLINK;
        }
    }

	if(!(pIDData->Cache.ffBuf.attrFile & FILE_READONLY))
		rInfo.aAttributes |=
			ITEM_FLAG_MOVABLE |
			ITEM_FLAG_DISCARDABLE;

	// if folder is already remote, the items are remote, too.
	if(m_bIsRemote)
	{
		rInfo.aAttributes |= ITEM_FLAG_ISREMOTE;
	}

	// check extended attributes for .ICON
	if(EA::queryName(sPath, ".ICON"))
        rInfo.aIconLocation = sPath;

	// replace ^ with blanks in DisplayName
	USHORT nIndex = 0;
	while( ( nIndex = rInfo.aDisplayName.SearchAndReplace( '^', ' ', nIndex ) )
		   != STRING_NOTFOUND )
        ;

	return TRUE;
}


/*------------------------------------------------------------------------------------
 * GetFileInfo()
 *------------------------------------------------------------------------------------
 */

BOOL DosFolderImpl::GetFileInfo(const ItemIDPath& rPath, FastFileInfo& rInfo)
{
	ItemIDPathData *pIDData = (ItemIDPathData *) rPath.GetDataPtr();

	if(pIDData == NULL)
		return FALSE;

	// information already in cache ?
	if(pIDData->nImplFlags & ITEM_IMPL_FLAG_FIND_INFO)
	{
		// set file name from cache
		rInfo.aFileName = pIDData->Cache.ffBuf.achName;

		getFileInfo((FILESTATUS3 *) &pIDData->Cache.ffBuf.fdateCreation, rInfo);
	}
	else
	{
		FILESTATUS3 fStatus;

		// get filesystem location
		String aFilePath = Location();
		aFilePath += pIDData->sPath;

		// set file name to last token of file path
		rInfo.aFileName = aFilePath.GetToken(aFilePath.GetTokenCount('\\') - 1, '\\');

		if(NO_ERROR != DosQueryPathInfo(aFilePath, FIL_STANDARD,
										&fStatus, sizeof(FILESTATUS3)))
		{
			return FALSE;
		}

		getFileInfo(&fStatus, rInfo);
	}

	if(!m_bIsLongNameFS && pIDData->sLocation.Len() == 0)
	{
		// set location
		pIDData->sLocation = Location();
		pIDData->sLocation += rInfo.aFileName;
	}

	return TRUE;
}

/*------------------------------------------------------------------------------------
 * RenameItem()
 *------------------------------------------------------------------------------------
 */

BOOL DosFolderImpl::RenameItem(const ItemIDPath& rPath, ItemIDPath& rNewPath, const String& rNewName)
{
	// need some information from the cache
	ItemIDPathData *pIDData = (ItemIDPathData *) rPath.GetDataPtr();

	if(pIDData->nImplFlags & ITEM_IMPL_FLAG_FIND_INFO)
	{
		// GetHostNotationPath() must be called on absolute ItemIDPaths
		String aSrcPath	 = (GetItemIDPath() + rPath).GetHostNotationPath();
		String aDestPath = GetItemIDPath().GetHostNotationPath();

        // append the new name directly
        USHORT nLen = aDestPath.Len();

        if(nLen && aDestPath.GetChar(nLen - 1))
            aDestPath += cBackSlash;

		aDestPath += rNewName;

		// in the easiest case the filesystem supports long filenames
		if(isLongNameFS(aSrcPath))
		{
			if(NO_ERROR == DosMove(aSrcPath, aDestPath))
			{
				ItemIDPath aParent;
				BOOL bRet = ItemIDPath(aDestPath).Split(aParent, rNewPath);

				// check if split works correctly
				DBG_ASSERT(bRet, "Rename: Split of destination path failed.\n");

				return TRUE;
			}
		}
		
		// the filename is short
		else if(isValidFileName(rNewName))
		{
			if(NO_ERROR == DosMove(aSrcPath, aDestPath))
			{
				ItemIDPath aParent;

				BOOL bRet = ItemIDPath(aDestPath).Split(aParent, rNewPath);

				// check if split works correctly
				DBG_ASSERT(bRet, "Rename: Split of destination path failed.\n");

				// create new or replace old EA
				StringEA(rNewName).storeTo(aDestPath, ".LONGNAME");
				
				return TRUE;
			}
		}

		// the long filename must be stored in the EAs
		else
		{
			CHAR szBuffer [CCHMAXPATHCOMP];
			CHAR * cp = szBuffer;

			// check if file already exists
			if(getValidPath(aDestPath, szBuffer, sizeof(szBuffer),
							aSrcPath.GetToken(aSrcPath.GetTokenCount(cBackSlash) - 1, cBackSlash)))
				return FALSE;

			/*
			 * the path of the current folder exists, so the only thing getValidPath can fail
			 * for is the nonexistance of the file with the long name specified. In this case,
			 * getValidPath() calls getValidFileName() and appends it at the end of the existing path.
			 * When getValidFileName() fails, the string returned from getValidPath() ends with a '\\'.
			 */
			cp += strlen(szBuffer) - 1;
			if(*cp == '\\')
				return FALSE;

			aDestPath = szBuffer;

			// rename file when short names differ
			if(aSrcPath.ICompare(aDestPath) != COMPARE_EQUAL)
			{
				if(NO_ERROR == DosMove(aSrcPath, aDestPath))
				{
					ItemIDPath aParent;
				
					BOOL bRet = ItemIDPath(aDestPath).Split(aParent, rNewPath);

					// check if split works correctly
					DBG_ASSERT(bRet, "Rename: Split of destination path failed.\n");
				}
				else
					return FALSE;
			}
			else
				rNewPath = rPath;

			// replace old EA if necessary
			StringEA(rNewName).storeTo(aDestPath, ".LONGNAME");

			return TRUE;
		}
	}
	
	return FALSE;
}


/*------------------------------------------------------------------------------------
 * DeleteItem()
 *------------------------------------------------------------------------------------
 */

BOOL DosFolderImpl::DeleteItem(const ItemIDPath &rPath)
{
	// need some information from the cache
	ItemIDPathData *pIDData = (ItemIDPathData *) rPath.GetDataPtr();

	if(pIDData->nImplFlags & ITEM_IMPL_FLAG_FIND_INFO)
	{
		// GetHostNotationPath() must be called on absolute ItemIDPaths
		String aPath = (GetItemIDPath() + rPath).GetHostNotationPath();

		if(aPath.Len())
		{
			APIRET rc;

			if(pIDData->Cache.ffBuf.attrFile & FILE_DIRECTORY)
				rc = DosDeleteDir(pIDData->sPath);
			else
				rc = DosDelete(pIDData->sPath);
		
			return rc == NO_ERROR;
		}
	}
	
	return FALSE;
}


/*------------------------------------------------------------------------------------
 * getFileTime()
 *------------------------------------------------------------------------------------
 */

inline void getFileTime(FileTime& aFileTime, const FDATE pfdDate, const FTIME pftTime)
{
	aFileTime.SetYear ( pfdDate.year + 1980 );
	aFileTime.SetMonth( pfdDate.month );
	aFileTime.SetDay  ( pfdDate.day );

	aFileTime.SetHour( pftTime.hours );
	aFileTime.SetMin ( pftTime.minutes );
	aFileTime.SetSec ( pftTime.twosecs * 2 );
}

/*------------------------------------------------------------------------------------
 * getFileInfo()
 *------------------------------------------------------------------------------------
 */

void DosFolderImpl::getFileInfo(const FILESTATUS3 *pfStatus, FastFileInfo& rInfo)
{
	rInfo.aFileSizeHigh = 0;
	rInfo.aFileSize		= pfStatus->cbFile;

	getFileTime(rInfo.aCreationTime,
				pfStatus->fdateCreation,
				pfStatus->ftimeCreation);
	getFileTime(rInfo.aLastAccessTime,
				pfStatus->fdateLastAccess,
				pfStatus->ftimeLastAccess);
	getFileTime(rInfo.aLastWriteTime,
				pfStatus->fdateLastWrite,
				pfStatus->ftimeLastWrite);

	if(pfStatus->attrFile & FILE_DIRECTORY)
	{
		rInfo.aFileKind	  = FILE_KIND_DIRECTORY;
		rInfo.aAttributes = FILE_FLAG_DIRECTORY;
	}
	else
	{
		rInfo.aFileKind	  = FILE_KIND_FILE;
		rInfo.aAttributes = 0;
	}

	if(pfStatus->attrFile & FILE_READONLY)
	   rInfo.aAttributes |= FILE_FLAG_READONLY;
	if(pfStatus->attrFile & FILE_HIDDEN)
	   rInfo.aAttributes |= FILE_FLAG_HIDDEN;
	if(pfStatus->attrFile & FILE_SYSTEM)
	   rInfo.aAttributes |= FILE_FLAG_SYSTEM;
	if(pfStatus->attrFile & FILE_ARCHIVED)
	   rInfo.aAttributes |= FILE_FLAG_ARCHIVE;

	// calculate executable flag from extension
	// this has to be changed to a type determination				   !!!!!!!!!!!!
	int nTokens = rInfo.aFileName.GetTokenCount('.');

	if(nTokens > 1)
	{
		String sExt = rInfo.aFileName.GetToken(nTokens - 1, '.');

		if(
		 sExt.ICompare( "EXE" ) == COMPARE_EQUAL ||
		 sExt.ICompare( "COM" ) == COMPARE_EQUAL ||
		 sExt.ICompare( "BAT" ) == COMPARE_EQUAL ||
		 sExt.ICompare( "CMD" ) == COMPARE_EQUAL
		)
			rInfo.aAttributes |= FILE_FLAG_EXECUTABLE;
	}
}
