/*************************************************************************
 *
 *  $RCSfile: OfficeConnect.java,v $
 *
 *  $Revision: 1.1 $
 *
 *  last change: $Author: kz $ $Date: 2003/05/13 10:28:21 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

// __________ Imports __________

// structs, const, ...
import com.sun.star.beans.PropertyValue;
import com.sun.star.uno.Any;

// interfaces
import com.sun.star.bridge.XUnoUrlResolver;
import com.sun.star.frame.XTerminateListener;
import com.sun.star.frame.TerminationVetoException;
import com.sun.star.frame.XDesktop;
import com.sun.star.lang.XMultiServiceFactory;

// exceptions
import com.sun.star.container.NoSuchElementException;
import com.sun.star.uno.Exception;
import com.sun.star.uno.RuntimeException;
import java.io.IOException;
import java.lang.InterruptedException;
import java.net.ConnectException;

// helper
import com.sun.star.uno.IBridge;
import com.sun.star.uno.UnoRuntime;

// others
import java.lang.*;
import java.io.*;

// __________ Implementation __________

/**
 * support ONE singleton uno connection to an office installation!
 * Can be used to open/use/close connection to uno environment.
 * ctor isn't available from outside. You should call static functions
 * to open or use internal set connection which is created one times only.
 *
 * @author      Andreas Schlns
 * @modified    10.07.2002 10:55
 */
public class OfficeConnect implements com.sun.star.frame.XTerminateListener
{
    // ____________________

    /**
     * static member
     *
     * @member m_aConnection        the singleton connection to the remote office instance
     *
     * member
     *
     * @member m_sOfficePath        system path to the office in case we start our own one
     * @member m_sHost              host on which the office runs
     * @member m_sPort              port on which the office will listen
     * @member m_nTries             count for trying to connect to the office
     * @member m_nTimeout           time in [milliseconds] we wait for startup of an office (in case we use our one one)
     * @member m_nTimeStep          time in [milliseconds] which is used to increase original timeout for every try we make
     * @member m_aOffice            reference to the office sub process, if we started our own one
     * @member m_aLog               log mechanism on top of rerouted std. in/out/err streams of the office process
     * @member m_xServiceManager    the remote service manager, which can be used to create remote uno services
     * @member m_bOwnCreated        true, if we started our own office and don't use any already running one
     */
    private static OfficeConnect                           m_aConnection    ;

    private        String                                  m_sOfficePath    ;
    private        String                                  m_sHost          ;
    private        String                                  m_sPort          ;
    private        int                                     m_nTries         ;
    private        int                                     m_nTimeout       ;
    private        int                                     m_nTimeStep      ;

    private        Process                                 m_aOffice        ;
    private        OfficeLog                               m_aLog           ;
    private        com.sun.star.lang.XMultiServiceFactory  m_xServiceManager;
    private        boolean                                 m_bOwnCreated    ;
    private        com.sun.star.frame.XDesktop             m_xTerminator    ;

    // ____________________

    /**
     * It's a generic way to get a connection to a remote office.
     * First we search for an existing ini file, which describe
     * the type of this connection. Only if this file couldn't be found,
     * we use default parameter to connect to an already running instance.
     *
     * @throw  [java.net.ConnectException]
     *          in case we couldn't establish a valid connection
     */
    public static synchronized void createConnection() throws java.net.ConnectException
    {
        if (m_aConnection==null)
        {
            /* TODO using of an ini file */

            // try default connection
            m_aConnection = new OfficeConnect("", "localhost", "8100", 3, 1000, 1000);
            m_aConnection.impl_connect();
        }
    }

    // ____________________

    /**
     * It tries to establish a new connection by using possible parameters
     * of the given command line.
     *
     * @param   lArguments
     *              a list of all command lne arguments, given by the outside main
     *
     * @throw   [java.net.ConnectException]
     *              in case we couldn't establish a valid connection
     */
    public static synchronized void createConnection( /*IN*/ String[] lArguments ) throws java.net.ConnectException
    {
        if (m_aConnection==null)
        {
            // Analyze command line parameters.
            String  sOffice   = new String (""         );
            Integer nTries    = new Integer(3          );
            Integer nTimeout  = new Integer(10000      );
            Integer nTimeStep = new Integer(0          );
            String  sHost     = new String ("localhost");
            String  sPort     = new String ("8000"     );

            for(int i=0; i<lArguments.length; ++i)
            {
                if(lArguments[i].startsWith("office=")==true)
                    sOffice = lArguments[i].substring(7);
                else
                if(lArguments[i].startsWith("try=")==true)
                    nTries = new Integer(lArguments[i].substring(4));
                else
                if(lArguments[i].startsWith("timeout=")==true)
                    nTimeout = new Integer(lArguments[i].substring(8));
                else
                if(lArguments[i].startsWith("timestep=")==true)
                    nTimeStep = new Integer(lArguments[i].substring(9));
                else
                if(lArguments[i].startsWith("host=")==true)
                    sHost = lArguments[i].substring(5);
                else
                if(lArguments[i].startsWith("port=")==true)
                    sPort = lArguments[i].substring(5);
            }

            if (sOffice.length()>1)
                sHost = "localhost";

            if (nTries.intValue()<=0)
                nTries = new Integer(3);

            if (nTimeout.intValue()<=0)
                nTimeout = new Integer(5000);

            if (nTimeStep.intValue()<0)
                nTimeStep = new Integer(0);

            // establish connection
            OfficeConnect.createConnection( sOffice              ,
                                            sHost                ,
                                            sPort                ,
                                            nTries.intValue()    ,
                                            nTimeout.intValue()  ,
                                            nTimeStep.intValue() );
        }
    }

    // ____________________

    /**
     * At first call we start an external office.
     * Then we try to connect our java application to it and
     * establish an uno connection. In case it's not possible
     * we try it several times. How offten can be specified from
     * outside. If no path to an office is given we supress starting
     * of an office by ourself. Then we use an already running one.
     *
     * @param  sOffice
     *          must be an absolute system path to an office which should be started
     *          and used
     *
     * @param  sHost
     *          hostname where an office runs and we will connect
     *          Will be ignored if parameter "sOffice" isn't empty.
     *          Because we start an own instance then and hostname will
     *          be fix = "localhost".
     *
     * @param  sPort
     *          port on which office can be found or should be started
     *
     * @param  nTries
     *          if we are forced to start our own office ... and it will fail
     *          this value allow us to try it mire then ones
     *
     * @param  nTimeout
     *          means the time we wait for finishing the start process of the office
     *          [millisecond]
     *
     * @throw  [java.net.ConnectException]
     *          in case we couldn't establish a valid connection
     */
    public static synchronized void createConnection(
                                        String sOffice   ,
                                        String sHost     ,
                                        String sPort     ,
                                        int    nTries    ,
                                        int    nTimeout  ,
                                        int    nTimeStep ) throws java.net.ConnectException
    {
        if (m_aConnection==null)
        {
            m_aConnection = new OfficeConnect(sOffice, sHost, sPort, nTries, nTimeout, nTimeStep);
            m_aConnection.impl_connect();
        }
    }

    // ____________________

    /**
     * close connection to remote office if it exist.
     */
    public static synchronized void disconnect()
    {
        if(m_aConnection==null)
            return;

        m_aConnection.impl_disconnect();
        m_aConnection = null;
    }

    // ____________________

    /**
     * Returns the default opened connection. They must be created before.
     * Otherwhise we return NULL here.
     *
     * @return  The default opened connection or <b>NULL</b> if no one exist.
     */
    public static synchronized OfficeConnect getDefaultConnection()
    {
        return m_aConnection;
    }

    // ____________________

    /**
     * May the outside user of this class whish to know, if
     * we created the office instance by ourself.
     * He can use it to decide if it's neccesary to terminate it too.
     */
    public static synchronized boolean isOwnOffice()
    {
        if (m_aConnection==null)
            return false;

        return m_aConnection.m_bOwnCreated;
    }

    // ____________________

    /**
     * It returns the internal saved uno service manager for our
     * default remote office connection.
     * Normaly an uno service should be create by using the create...() methods
     * of these OfficeConnect class. But sometimes it can be usefull
     * (e.g. to demonstrate API using inside the code) to use this service manager driectly.
     *
     * @return  a reference to the remote uno service manager for the current default connection.
     *          Note: The value can be <b>null</b> if no connection  already exist.
     */
    public static synchronized com.sun.star.lang.XMultiServiceFactory getDefaultSMGR()
    {
        if (m_aConnection==null)
            return null;

        return m_aConnection.getSMGR();
    }

    // ____________________

    /**
     * create uno components inside remote office process
     * After connection of these proccess to a running office we have access to remote service manager of it.
     * So we can use it to create all existing services. Use this method to create components by name and
     * get her interface. Casting of it to right target interface is part of your implementation.
     *
     * @param  aType
     *          describe class type of created service
     *          Returned object can be casted directly to this one.
     *          Uno query was done by this method automaticly.
     *
     * @param  sServiceSpecifier
     *          name of service which should be created
     *
     * @return An object which can be casted to the required interface.
     *         May its <NULL/> if creation was failed or no connection exist.
     */
    public static synchronized Object createRemoteInstance(Class aType, String sServiceSpecifier)
    {
        if (m_aConnection==null)
            return null;

        return m_aConnection.createInstance(aType,sServiceSpecifier);
    }

    // ____________________

    /**
     * same as "createRemoteInstance()" but supports additional parameter for initializing created object
     *
     * @param  lArguments
     *          optional arguments
     *          They are used to initialize new created service.
     *
     * @param  aType
     *          Description of Parameter
     *
     * @param  sServiceSpecifier
     *          Description of Parameter
     *
     * @return An object which can be casted to the required interface.
     *         May its <NULL/> if creation was failed or no connection exist.
     */
    public static synchronized Object createRemoteInstanceWithArguments(Class aType, String sServiceSpecifier, Any[] lArguments)
    {
        if (m_aConnection==null)
            return null;

        return m_aConnection.createInstanceWithArguments(aType,sServiceSpecifier,lArguments);
    }

    // ____________________

    /**
     * This method can be used to simulate a GPF.
     * That can be usefull to check e.g. the recovery functionality.
     * Note: After calling of this function the office will die and
     * can't be used any longer!
     */
    public static synchronized void GPF()
    {
        com.sun.star.util.URL[] aURL = new com.sun.star.util.URL[1];
        aURL[0]          = new com.sun.star.util.URL();
        aURL[0].Complete = new String("slot:6645");

        com.sun.star.util.XURLTransformer xParser = (com.sun.star.util.XURLTransformer)OfficeConnect.createRemoteInstance(
            com.sun.star.util.XURLTransformer.class,
            "com.sun.star.util.URLTransformer");

        if (xParser !=null)
            xParser.parseStrict(aURL);

        com.sun.star.frame.XDispatchProvider xProvider = (com.sun.star.frame.XDispatchProvider)OfficeConnect.createRemoteInstance(
            com.sun.star.frame.XDispatchProvider.class,
            "com.sun.star.frame.Desktop");

        com.sun.star.frame.XDispatch xDispatch = null;
        if (xProvider != null)
            xDispatch = xProvider.queryDispatch(aURL[0],"",0);

        if (xDispatch != null)
            xDispatch.dispatch(aURL[0], new com.sun.star.beans.PropertyValue[0]);
    }

    // ____________________

    /**
     * ctor
     * It set the given information on internal members only.
     * They will be used later in method "connect()".
     * But we correct the host information. Because if we got
     * an office path and should start our own office instance
     * it will be possible on the local machine only.
     * So we set it to "localhost" then.
     *
     * @param  sOffice
     *          must be an absolute system path to an office which should be started
     *          and used
     *
     * @param  sHost
     *          hostname where an office runs and we will connect
     *          Will be ignored if parameter "sOffice" isn't empty.
     *          Because we start an own instance then and hostname will
     *          be fix = "localhost".
     *
     * @param  sPort
     *          port on which office can be found or should be started
     *
     * @param  nTries
     *          if we are forced to start our own office ... and it will fail
     *          this value allow us to try it mire then ones
     *
     * @param  nTimeout
     *          means the time we wait for finishing the start process of the office
     *          [millisecond]
     */
    private OfficeConnect(
                String sOffice   ,
                String sHost     ,
                String sPort     ,
                int    nTries    ,
                int    nTimeout  ,
                int    nTimeStep )
    {
        m_sOfficePath = sOffice   ;
        m_sHost       = sHost     ;
        m_sPort       = sPort     ;
        m_nTries      = nTries    ;
        m_nTimeout    = nTimeout  ;
        m_nTimeStep   = nTimeStep ;
        if (sOffice.length()>0)
            m_sHost = "localhost";
    }

    // ____________________

    /**
     * Returns the uno service manager of the remote office.
     * Such manager can be used to create own services or to patch
     * a component registration base.
     *
     * @return  A reference to the uno service manager of the remote office
     *          instance.
     */
    public synchronized com.sun.star.lang.XMultiServiceFactory getSMGR()
    {
        return m_xServiceManager;
    }

    // ____________________

    /**
     * It uses the remote service manager, to register (means overwrite
     * the current registration for) the given service factory.
     *
     * @param   xServiceFactory
     *              points to the service and his factory, which wish
     *              to be registered
     *
     * @return  <b>TRUE</b> if registration succeeded.
     */
    public boolean patchServiceRegistry( /*IN*/ Object xServiceFactory )
    {
        com.sun.star.container.XSet xRegistry = (com.sun.star.container.XSet)UnoRuntime.queryInterface(
            com.sun.star.container.XSet.class,
            m_xServiceManager);

        if (xRegistry == null)
            return false;

        try
        {
            xRegistry.insert(xServiceFactory);
            return true;
        }
        catch(java.lang.Exception exUnknown)
        {
            return false;
        }
    }

    // ____________________

    /**
     * create uno components inside remote office process
     * After connection of these proccess to a running office we have access to remote service manager of it.
     * So we can use it to create all existing services. Use this method to create components by name and
     * get her interface. Casting of it to right target interface is part of your implementation.
     *
     * @param   aType
     *              describe the class type of created service
     *              Returned object can be casted directly to this one.
     *              Uno query was done by this method automaticly.
     *
     * @param   sServiceSpecifier
     *              name of service which should be created
     *
     * @return  An object which can be casted to the required interface.
     *          May its <b>NULL</b> if creation failed or the connection does not exist.
     */
    public synchronized Object createInstance( /*IN*/ Class  aType             ,
                                               /*IN*/ String sServiceSpecifier )
    {
        Object aResult = null;
        try
        {
            aResult = UnoRuntime.queryInterface( aType, m_xServiceManager.createInstance(sServiceSpecifier) );
        }
        catch (java.lang.Exception ex)
        {
            aResult = null;
        }
        return aResult;
    }

    // ____________________

    /**
     * same as {@link}createRemoteInstance(){@link} but supports additional parameter
     * to initialize the new created object
     *
     * @param   aType
     *              describe the class type of created service
     *              Returned object can be casted directly to this one.
     *              Uno query was done by this method automaticly.
     *
     * @param   sServiceSpecifier
     *              name of service which should be created
     *
     * @param   lArguments
     *              optional arguments, which are used to initialize the new created service
     *
     * @return  An object which can be casted to the required interface.
     *          May its <b>NULL</b> if creation failed, the connection does not exist or
     *          the service does not provide the right initialization method.
     */
    public synchronized Object createInstanceWithArguments( /*IN*/ Class                  aType             ,
                                                            /*IN*/ String                 sServiceSpecifier ,
                                                            /*IN*/ com.sun.star.uno.Any[] lArguments        )
    {
        Object aResult = null;
        try
        {
            aResult = UnoRuntime.queryInterface( aType, m_xServiceManager.createInstanceWithArguments( sServiceSpecifier, lArguments ));
        }
        catch (java.lang.Exception ex)
        {
            aResult = null;
        }
        return aResult;
    }

    // ____________________

    /**
     * We try to open the connection here.
     * After it was successfully you will find some internal set member; e.g. m_xServiceManager
     * wich means the remote uno service manager of connected office.
     * We start listening for office termination here too. So we prevent our java application
     * against a dieing office during using.
     *
     * @throw  [java.net.ConnectException]
     *          in case we couldn't establish a valid connection
     */
    private void impl_connect() throws java.net.ConnectException
    {
        m_aOffice = null;
        m_aLog    = null;

        for (int i=0; i<m_nTries; ++i)
        {
            // start the office - if neccessary!
            m_bOwnCreated = false;
            if (m_sOfficePath.length()>0)
            {
                impl_startOffice(m_sOfficePath,m_sPort,m_nTimeout);
                m_nTimeout   += m_nTimeStep;
                m_bOwnCreated = true;
            }

            // try to connect to an already running or
            // just created office instance
            impl_connectOffice(m_sHost,m_sPort);

            // check for success
            if (m_xServiceManager==null)
            {
                if (m_bOwnCreated)
                    impl_stopOffice();
                continue;
            }
            else
            {
                // We have a valid connection :-)
                break;
            }
        }

        // here we had no chance to make anything else.
        // We couldn't started an office nor connect to any running instance.
        // Our application couldn't work.
        // We throw right exception here.
        // Outside code can handle this situation.
        if (m_xServiceManager==null)
            throw new java.net.ConnectException("could not establish a connection to an office ...");

        impl_startListeningForTerminate();
    }

    // ____________________

    /**
     * Of course we must close our internal connection.
     * We stop listening for office termination here too.
     * In case we had created our own office instance we terminate it active!
     */
    private void impl_disconnect()
    {
        // We have to stop listening for office terminationm in every case ...
        impl_stopListeningForTerminate();

        // but we have to terminate the office only, if we created it by ourself
        // Then we have to terminate it via api correctly.
        // Used helper method reset internal member too.
        if (m_bOwnCreated)
            impl_terminateOffice();

        // otherwhise we used any external office instance.
        // So we have to forgt all connectsion to it
        // but have no rights to kill or terminate it!
        // So we set all internal member to null.
        // m_aOffice and m_aLog can't be valid here.
        // Otherwhise our bool m_bOwnCreated was setted wrong!
        else
        {
            m_xServiceManager = null ;
            m_xTerminator     = null ;
            m_bOwnCreated     = false;
        }
    }

    // ____________________

    /**
     * tries to connect to an external office instance
     *
     * @param  sHost
     *          hostname where an office runs and we will connect
     *
     * @param  sPort
     *          port on which office can be found
     */
    private void impl_connectOffice( String sHost, String sPort )
    {
        try
        {
            String sConnectString  = new String("uno:socket,host=");
                   sConnectString += sHost;
                   sConnectString += ",port=";
                   sConnectString += sPort;
                   sConnectString += ";urp;";                                 // enable oneway calls
                   //sConnectString += ";urp,Negotiate=0,ForceSynchronous=1;";  // disable oneway calls (experimental!)
                   sConnectString += "StarOffice.ServiceManager";

            System.out.print("try to connect to office with command \""+sConnectString+"\" ... ");

            com.sun.star.lang.XMultiServiceFactory xLocalServiceManager = com.sun.star.comp.helper.Bootstrap.createSimpleServiceManager();

            com.sun.star.bridge.XUnoUrlResolver    xURLResolver         = (com.sun.star.bridge.XUnoUrlResolver)UnoRuntime.queryInterface(
                                                                                com.sun.star.bridge.XUnoUrlResolver.class,
                                                                                xLocalServiceManager.createInstance("com.sun.star.bridge.UnoUrlResolver"));

            m_xServiceManager = (com.sun.star.lang.XMultiServiceFactory)UnoRuntime.queryInterface(
                                    com.sun.star.lang.XMultiServiceFactory.class,
                                    xURLResolver.resolve(sConnectString));
        }
        catch(java.lang.Exception ex)
        {
            System.out.println("got exception!");
            ex.printStackTrace();
            m_xServiceManager = null;
        }

        if (m_xServiceManager==null)
            System.out.println("failed");
        else
            System.out.println("OK");
    }

    // ____________________

    /**
     * here we start/stop listening on the remote office instance for his termination events
     * It's possible everytime (in case more then one external processes share the sam
     * office) that the office tries to shutdown during we use it. Then we stop it by
     * throwing a TeminateVetoException. In case we finish our own work on it, we
     * will deregister ourself. We terminate the office active only, if we started it by ourself.
     * Otherwhise we don't do anything.
     */
    private void impl_startListeningForTerminate()
    {
        if (m_xTerminator!=null)
            return;

        System.out.print("start listening for office termination ... ");
        m_xTerminator = (com.sun.star.frame.XDesktop)OfficeConnect.createRemoteInstance(
                            com.sun.star.frame.XDesktop.class,
                            "com.sun.star.frame.Desktop");

        if (m_xTerminator==null)
        {
            System.out.println("failed");
            return;
        }

        m_xTerminator.addTerminateListener(this);
        System.out.println("OK");
    }

    // ____________________

    private void impl_stopListeningForTerminate()
    {
        if (m_xTerminator==null)
            return;

        System.out.print("stop listening for office termination ... ");
        m_xTerminator.removeTerminateListener(this);
        m_xTerminator = null;
        System.out.println("OK");
    }

    // ____________________

    /**
     * tries to start an external office instance by using given path
     *
     * @param  sPath
     *          must be an absolute system path to an office which should be started
     *
     * @param  sPort
     *          port on which office should listen
     *
     * @param  nTimeout
     *          means the time we wait for finishing the start process of the office
     *          [milliseconds]
     */
    private void impl_startOffice( String sPath, String sPort, int nTimeout )
    {
        String[] lCommands = new String[4];
        lCommands[0]  = sPath;
        lCommands[1]  = new String("-accept=socket,host=localhost,port="+sPort+";urp;");
        lCommands[2]  = new String("-invisible");
        lCommands[3]  = new String("-nocrashreport");

        System.out.print("start office with command: \""+lCommands[0]+" "+lCommands[1]+" "+lCommands[2]+" "+lCommands[3]+" ... ");
        try
        {
            m_aOffice = Runtime.getRuntime().exec(lCommands);
            System.out.println("OK");
        }
        catch(java.io.IOException exIO)
        {
            m_aOffice=null;
            System.out.println("failed");
        }

        // read his output
        // Is required! see docu of used class!
        if (m_aOffice!=null)
        {
            m_aLog = new OfficeLog(m_aOffice);
            m_aLog.start();
            synchronized(this)
            {
                try
                {
                    System.out.print("look for busy log ... ");
                    while(m_aLog.isBusy())
                        wait(100);
                    System.out.println("finished");
                }
                catch(java.lang.InterruptedException exInterrupt) {}
            }
        }

        // Wait a little bit till the office has (hopefully) finished his startup
        try
        {
            System.out.println("wait for "+(nTimeout/1000)+" sec ...");
            Thread.sleep(nTimeout);
        }
        catch(java.lang.InterruptedException exBreaked) {}
    }

    // ____________________

    /**
     * tries to stop the external office instance
     * Don't mix it up with an office termination.
     * Here we "kill" the office process hard. And it's
     * allowed to do so only in case we started it by ourself ...
     * means by using method impl_startOffice() in combination
     * with a failed connection try.
     *
     * For normal office termination please use impl_terminateOffice().
     * It use the office api to shutdown it in the right way.
     * Of course listening for such termination events will be canceled
     * before too.
     */
    private void impl_stopOffice()
    {
        if (m_aLog!=null)
        {
            System.out.print("stop logging ...");
            synchronized(m_aLog)
            {
                m_aLog.notifyAll();
            }
            m_aLog = null;
            System.out.println("OK");
        }

        if (m_aOffice!=null)
        {
            System.out.print("stop office process ...");
            m_aOffice.destroy();
            try
            {
                m_aOffice.waitFor();
            }
            catch(java.lang.InterruptedException exInterruptProcess) {}
            m_aOffice = null;
            System.out.println("OK");
        }
    }

    // ____________________

    /**
     * tries to terminate the external office instance
     * Don't mix it up with an office kill as we do in method impl_stopOffice()!
     * Here we use the office api to shutdown it in the right way.
     * Of course listening for such termination events will be canceled
     * before too.
     */
    private void impl_terminateOffice()
    {
        // stop logging in case we use it
        if (m_aLog!=null)
        {
            System.out.print("stop logging ...");
            synchronized(m_aLog)
            {
                m_aLog.notifyAll();
            }
            m_aLog = null;
            System.out.println("OK");
        }

        // get copy of the remote desktop
        // BEFORE we stop listening
        // Because the used helper method release the internal member
        // reference. They is his indicator for a valid
        // listener connection!
        com.sun.star.frame.XDesktop xTerminator = m_xTerminator;
        impl_stopListeningForTerminate();
        // But in case we was no listener before ...
        // we should get a new reference to the desktop instance.
        if (xTerminator==null)
            xTerminator = (com.sun.star.frame.XDesktop)OfficeConnect.createRemoteInstance(
                                com.sun.star.frame.XDesktop.class,
                                "com.sun.star.frame.Desktop");
        xTerminator.terminate();

        // forget the process object for own started office instance
        // But don't "kill" it here!
        m_aOffice         = null ;
        m_xServiceManager = null ;
        m_bOwnCreated     = false;
    }

    // ____________________

    /**
     * These callbacks inform us about office termination.
     * Inside queryTermination() we can disagree by throwing an exception
     * TerminationVetoException. If we don't do it we must be aware that
     * the office can realy shutdown.
     * If we get the callback notifyTermination() - we must accept it.
     * Then we should terminate this java application too ...
     *
     * @param aSource
     *          describe the source of this event
     *          Should be the desktop instance, where we are registered.
     *          Normaly we can check it ... but its a bug, if somewhere
     *          else notify us, which we doesn't know!
     */
    public void queryTermination( com.sun.star.lang.EventObject aSource ) throws com.sun.star.frame.TerminationVetoException
    {
        System.out.println("terminate listener throw veto exception ...");
        throw new TerminationVetoException("in use by the java document converter ...");
    }

    // ____________________

    public void notifyTermination( com.sun.star.lang.EventObject aSource )
    {
        // We throw a veto exception everytime inside queryTermination().
        // Stopping of that can be reached only by deregistration of this instance
        // as a teminate listener! But then we shouldn't get this callback.
        // So nevertheless it's an unexpected situation if get it ...
        System.out.println("unexpected situation found! We got termination notify. But we throw veto for query ...");

        com.sun.star.frame.XDesktop xSource = (com.sun.star.frame.XDesktop)UnoRuntime.queryInterface(
                                                com.sun.star.frame.XDesktop.class,
                                                aSource.Source);
        if (xSource!=null)
            xSource.removeTerminateListener(this);
    }

    // ____________________

    /**
     * If we are a listener in generell, we have to listen for disposing
     * of our broadcaster too. Of course it's not specified, that this event
     * comes earlier then the terminate notifications ... but it's better to
     * react here too.
     * We use our central method shutdown() to terminate this thread.
     * So we can control our own shutdown in a better way and fore it for
     * different situations.
     *
     * @param aSource
     *          describe the source of this event
     *          Should be the desktop instance, where we are registered.
     *          Normaly we can check it ... but its a bug, if somewhere
     *          else notify us, which we doesn't know!
     */
    public void disposing(com.sun.star.lang.EventObject aSource)
    {
        System.out.println("terminate listener detected an unexpected situation - disposing! ...");

        com.sun.star.frame.XDesktop xSource = (com.sun.star.frame.XDesktop)UnoRuntime.queryInterface(
                                                com.sun.star.frame.XDesktop.class,
                                                aSource.Source);
        if (xSource!=null)
            xSource.removeTerminateListener(this);
    }
}
