/*************************************************************************
 *
 *  $RCSfile: xgdidemo.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: vg $ $Date: 2003/04/15 17:24:46 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/
#ifndef _SV_HXX
#include <sv.hxx>
#endif

#ifndef _SFXITEMS_HXX
#include <sfxitems.hxx>
#endif

#include <xout.hxx>

#pragma hdrstop

#include <time.h>

#include <stdlib.h>
#include <math.h>

#if OSL_DEBUG_LEVEL > 1
#include <\svx\inc\xpolyimp.hxx>
//#include <o:\sv\inc\poly.h>
#endif

#define MID_PENCOLOR    10
#define MID_PENRED      11
#define MID_PENBLUE     12
#define MID_PENYELLOW   13
#define MID_PENGREEN    14
#define MID_PENCYAN     15
#define MID_PENMAGENTA  16
#define MID_PENBROWN    17
#define MID_PENRED2     18
#define MID_PENGREEN2   19
#define MID_PENBLACK    20
#define MID_PENWHITE    21

#define MID_PENWIDTH    22
#define MID_PENSMALL    23
#define MID_PENSMALL2   24
#define MID_PENMEDIUM   25
#define MID_PENTHICK    27
#define MID_PENTHICK2   28

#define MID_FILE        30
#define MID_QUIT        31

#define SMALL_DOUBLE_VAL    10e-50


#define MAXPOLYPTS 100

// Resource Definitionen

#define BMP_FILL1 256

const double F_PI = 3.1415926535;

/*** FatLineParam *******************************************************/

class FatLineParam
{
 public:
	long    nPatSeg;
	long    nPatRemain;
	BOOL    bHasJoin;
	Point   aJoin1, aJoin2;
	double  fLength;
	long    nLineDx, nLineDy;
	long    nDxW, nDyW;

	FatLineParam(long nSeg = -1, long nRemain = 0) :
		nPatSeg(nSeg),
		nPatRemain(nRemain),
		bHasJoin(FALSE) {}
};


/************************************************************************/

void FatPolyLine(const XPolygon& rPoly, const long LinePattern[], long nWidth, OutputDevice& OutDev);
void FatLine(const Point& rStart, const Point& rEnd, const Point* pNext,
			 const long* pLinePattern, long nWidth, FatLineParam& rParam,
			 OutputDevice& rOutDev);
void CalcTangent(const Point& rCenter, Point& rPrev, Point& rNext);
void PointsToBezier(XPolygon& rXPoly, USHORT nPos);
XPolygon FitCurve(const XPolygon& rPoly, USHORT nPts, double fError);
XPolyPolygon GetCharOutline(USHORT nCharacter, OutputDevice& rOutDev);


/************************************************************************/

static long DefLinePattern[20];


/************************************************************************/

enum MarkerType { MARKER_RECT, MARKER_CROSS, MARKER_POINT };

void DrawMarker(const Point& aPos, USHORT nSize, MarkerType eMark,
				OutputDevice& rOutDev)
{
	Rectangle aRect(aPos - Point(nSize/2, nSize/2), Size(nSize, nSize));

	switch ( eMark )
	{
		case MARKER_RECT:
			rOutDev.DrawRect(aRect);
			break;
		case MARKER_CROSS:
			rOutDev.DrawLine(aRect.LeftCenter(), aRect.RightCenter());
			rOutDev.DrawLine(aRect.TopCenter() , aRect.BottomCenter());
			break;
		case MARKER_POINT:
			rOutDev.DrawPoint(aPos);
			break;
	}
}
/************************************************************************/

class MyApp : public Application
{
public:
	virtual void Main(int, char*[]);
};

// --- class MyWin -------------------------------------------------------

class MyWin : public WorkWindow
{
	XOutdevItemPool XPool;
	XOutputDevice   XOut;
	XPolygon        aBezier;
	XPolygon        aXPoly;
	USHORT          nPolyPts;
	Point           aLastPos;
	USHORT          nColorId;
	USHORT          nPenId;
	Timer           aTimer;
	long            dx[8], dy[8];
	short			nAngle;
	double          fFitBezError;
	long            nLineWidth;
	String			aFormText;

	void PntCheck(short nIndex, const Size& rSize);
	void InitTrans();

public:
	MyWin();

	virtual BOOL    Close();
	virtual void    KeyInput(const KeyEvent& rKEvt);
	virtual void    MouseButtonDown( const MouseEvent& rMEvt );
	virtual void    MouseButtonUp( const MouseEvent& rMEvt );
	virtual void    MouseMove( const MouseEvent& rMEvt );
	virtual void    Resize();
	virtual void    Paint( const Rectangle& rRect );
	long            FileSelect( Menu* pMenu );
	long            ColorSelect( Menu* pMenu );
	long            WidthSelect( Menu* pMenu );
	long            AniHdl(Timer *pTimer);
	void            ZoomIn();
	void            ZoomOut();
	void			DrawBezMark(const XPolygon& rXPoly);
	void			DrawFormText(const String& rText, const XPolygon& rXPoly);
	void AniFast();
	BOOL bStop;
};

// --- aMyApp ------------------------------------------------------------

MyApp aMyApp;

// --- MyWin::MyWin() ----------------------------------------------------

MyWin::MyWin() : WorkWindow( NULL, WB_STDWORK | WB_APP ), XPool(),
	XOut(this, &XPool), aBezier(9), aXPoly(MAXPOLYPTS), nPolyPts(0),
	fFitBezError(4.0), nLineWidth(20), nAngle(0),
	aFormText("AaBbCcDdEeFfGgHhIiJjKkLlMmNnOoPpQ")
{
	nColorId = MID_PENBLACK;
	nPenId = MID_PENSMALL;
	MapMode aMap;
	aMap.ChangeMapUnit(MAP_PIXEL);
	ChangeMapMode(aMap);
	Brush aBrush = GetFillInBrush();
	aBrush.ChangeColor(COL_BLACK);
//  ChangeFillInBrush(aBrush);
//  ChangeRasterOp(ROP_INVERT);
	aTimer.ChangeTimeout(10);
	aTimer.ChangeTimeoutHdl(LINK(this, MyWin, AniHdl));
//  aTimer.Start();
	DefLinePattern[0] = 10;
	DefLinePattern[1] = 10;
	DefLinePattern[2] = 10;
	DefLinePattern[3] = 10;
	DefLinePattern[4] = 30;
	DefLinePattern[5] = 10;
	DefLinePattern[6] = 0;
	bStop = TRUE;
	InitTrans();

	XPolygon aEndPoly(3);
	aEndPoly[0] = Point(10,  0);
	aEndPoly[1] = Point( 0, 30);
	aEndPoly[2] = Point(20, 30);
	XPolygon aStartPoly(Point(0,0), 100, 100);

	XLineStyleItem      aLStyle(XLINE_SOLID);
	XLineWidthItem      aWidth(0);
	XLineDashItem       aDash(String(), XDash(XDASH_RECT, 3, 7, 2, 40, 15));
	XLineColorItem      aColor(String(), Color(COL_BLACK));
	XLineAttrSetItem    aLineAttr(&XPool);
	XLineStartItem			aLineStart(String(), aStartPoly);
	XLineStartWidthItem		aLineStWdt(0);
	XLineStartCenterItem	aLineStCnt(TRUE);
	XLineEndItem			aLineEnd(String(), aEndPoly);
	XLineEndWidthItem		aLineEndWdt(0);
	XLineEndCenterItem		aLineEndCnt(FALSE);
	SfxItemSet& rLSet = aLineAttr.GetItemSet();
	rLSet.Put(aLStyle);
	rLSet.Put(aWidth);
	rLSet.Put(aDash);
	rLSet.Put(aColor);
	rLSet.Put(aLineStart);
	rLSet.Put(aLineStWdt);
	rLSet.Put(aLineStCnt);
	rLSet.Put(aLineEnd);
	rLSet.Put(aLineEndWdt);
	rLSet.Put(aLineEndCnt);
	XOut.SetLineAttr(aLineAttr);


	XFillAttrSetItem 	aFillAttr(&XPool);
	XFillStyleItem		aFStyle(XFILL_SOLID);
	XFillColorItem		aFColor(String(), Color(COL_LIGHTRED));
	XFillBitmapItem		aFBitmap(String(), Bitmap(BMP_FILL1));
	SfxItemSet& rFSet = aFillAttr.GetItemSet();
	rFSet.Put(aFStyle);
	rFSet.Put(aFColor);
	rFSet.Put(aFBitmap);
	XOut.SetFillAttr(aFillAttr);
}

void MyWin::InitTrans()
{
	for (short i = 0; i < 8; i++)
	{
		dx[i] = ((long) rand() * 17 / RAND_MAX) - 8;
		dy[i] = ((long) rand() * 17 / RAND_MAX) - 8;
	}
}

// --- MyWin::Close() ----------------------------------------------------

BOOL MyWin::Close()
{
	if ( !bStop )
	{
		bStop = TRUE;
		return FALSE;
	}
	aTimer.Stop();
	return WorkWindow::Close();
}

/************************************************************************/

BOOL IsInside(const XPolygon& rXPoly, const Point& rPnt)
{
	BOOL	bXFlag, bYFlag, bYFlagOld, bInsideFlag;
	long	nPx, nPy;
	USHORT  nPnt, nPrev;
	USHORT	nPntMax = rXPoly.GetPointCount() - 1;

	nPx = rPnt.X();
	nPy = rPnt.Y();

	nPrev = nPntMax;
	bYFlagOld = ( rXPoly[nPrev].Y() >= nPy ) ;
	bInsideFlag = FALSE;

	for (nPnt = 0; nPnt <= nPntMax; nPnt++)
	{
		bYFlag = ( rXPoly[nPnt].Y() >= nPy );
		if ( bYFlagOld != bYFlag )
		{
			bYFlagOld = ( rXPoly[nPrev].X() >= nPx );
			//
			if ( bXFlag == ( rXPoly[nPnt].X() >= nPx ) )
			{
				if ( bXFlag )
					bInsideFlag = !bInsideFlag;
			}
			else if ( (rXPoly[nPnt ].X() - (rXPoly[nPnt].Y() - nPy) *
					  (rXPoly[nPrev].X() - rXPoly[nPnt].X()) /
					  (rXPoly[nPrev].Y() - rXPoly[nPnt].Y())) >= nPx )
				bInsideFlag = !bInsideFlag;
		}
		bYFlagOld = bYFlag;
		nPrev = nPnt;
	}
	return bInsideFlag;
}

/************************************************************************/

BOOL IsInside(const XPolyPolygon& rXPPoly, const Point& rPnt)
{
	BOOL bInside = FALSE;

	for (USHORT i = 0; i < rXPPoly.Count(); i++)
		bInside ^= IsInside(rXPPoly[i], rPnt);

	return bInside;
}

/************************************************************************/

void MyWin::DrawBezMark(const XPolygon& rXPoly)
{
	Pen aPen = ChangePen(Pen(Color(COL_LIGHTRED), 1));

	for (USHORT i = 0; i < rXPoly.GetPointCount(); i++)
	{
		if ( i == rXPoly.GetPointCount()-1 )
			ChangePen(Pen(Color(COL_LIGHTBLUE), 1));
		if ( rXPoly.GetFlags(i) == XPOLY_CONTROL )
			DrawMarker(rXPoly[i], 5, MARKER_CROSS, *this);
		else
			DrawMarker(rXPoly[i], 5, MARKER_RECT , *this);
	}
	SetPen(aPen);
}


/************************************************************************/

void TiltX(XPolygon& rXPoly, long nY, double fSin, double fCos)
{
	USHORT	nPntCnt = rXPoly.GetPointCount();

	for (USHORT i = 0; i < nPntCnt; i++)
	{
		long nDy = rXPoly[i].Y() - nY;
		rXPoly[i].X() += (long)(fSin * nDy);
		rXPoly[i].Y() = nY + (long)(fCos * nDy);
	}
}

/************************************************************************/

void TiltX(XPolyPolygon& rXPPoly, double fSin, double fCos)
{
	long nY = rXPPoly.GetBoundRect().Top();

	for (USHORT i = 0; i < rXPPoly.Count(); i++)
		TiltX(rXPPoly[i], nY, fSin, fCos);
}

/************************************************************************/

void TiltY(XPolygon& rXPoly, long nX, double fSin, double fCos)
{
	USHORT	nPntCnt = rXPoly.GetPointCount();

	for (USHORT i = 0; i < nPntCnt; i++)
	{
		long nDx = rXPoly[i].X() - nX;
		rXPoly[i].X() = nX + (long)(fCos * nDx);
		rXPoly[i].Y() -= (long)(fSin * nDx);
	}
}

/************************************************************************/

void TiltY(XPolyPolygon& rXPPoly, double fSin, double fCos)
{
	long nX = rXPPoly.GetBoundRect().Left();

	for (USHORT i = 0; i < rXPPoly.Count(); i++)
		TiltY(rXPPoly[i], nX, fSin, fCos);
}

/************************************************************************/

void MyWin::DrawFormText(const String& rText, const XPolygon& rXPoly)
{
	Polygon	aPoly = XOutCreatePolygon(rXPoly, this);
	long	nTotalLen = 0;
	long	nTextLen = 0;
	long	nDist = 0;
	USHORT	nPntCnt = aPoly.GetSize();
	USHORT	nPnt;
	USHORT	nChar = 0;
	USHORT	nCharCnt = rText.Len();

	Font aFont("Times New Roman", Size(0,50));
	aFont.ChangeTransparent(TRUE);
	aFont.ChangeAlign(ALIGN_BOTTOM);
	Font aOldFont = ChangeFont(aFont);
	Size aTextSize = GetTextSize(rText);

	// Gesamtlaenge des Polygons berechnen
	for (nPnt = 1; nPnt < nPntCnt; nPnt++)
	{
		double	fDx = aPoly[nPnt].X() - aPoly[nPnt-1].X();
		double	fDy = aPoly[nPnt].Y() - aPoly[nPnt-1].Y();
		nTotalLen += (long) (sqrt(fDx * fDx + fDy * fDy) + 0.5);
	}
	// zentrieren bzw. rechtsbuendig
	long nSkip = (nTotalLen - aTextSize.Width()) / 2;
	nPnt = 1;

	while ( nChar < nCharCnt && nPnt < nPntCnt )
	{
		Point	aPos = aPoly[nPnt];
		double	fDx = aPos.X() - aPoly[nPnt-1].X();
		double	fDy = aPos.Y() - aPoly[nPnt-1].Y();
		double	fLen = sqrt(fDx * fDx + fDy * fDy);
		long	nLen = (long) (fLen + 0.5);
		nSkip -= nLen;
		nLen -= nTextLen;
		nTextLen = - nLen;

		if ( nLen > 0 && nSkip <= 0 )
		{
			USHORT nCnt = 0;

			do
			{	nCnt++;
				nTextLen = GetTextSize(rText, nChar, nCnt).Width();
			}
			while ( nChar + nCnt < nCharCnt && nTextLen < nLen );

			short nAngle = (short) (acos(fDx / fLen) * 1800 / F_PI + 0.5);
			if ( fDy > 0 )
				nAngle = 3600 - nAngle;
			aFont.ChangeLineOrientation(nAngle);
			SetFont(aFont);
			aPos.X() -= (long) ((fDx * nLen - fDy * nDist) / fLen);
			aPos.Y() -= (long) ((fDy * nLen + fDx * nDist) / fLen);
			DrawText(aPos, rText, nChar, nCnt);
			aFont.ChangeLineOrientation(0);
			SetFont(aFont);
			for (USHORT i = 0; i < nCnt; i++)
			{
				Point aPolyPos = aPos;
				long nW = GetTextSize(rText, nChar, i).Width();
				long nH = aTextSize.Height();
				aPolyPos.X() += (long) ((fDx * nW + fDy * nH) / fLen);
				aPolyPos.Y() += (long) ((fDy * nW - fDx * nH) / fLen);
/*				Point aTPos = aPos;
				aTPos.X() += (long) (fDx * nW / fLen);
				aTPos.Y() += (long) (fDy * nW / fLen);
				DrawText(aTPos, rText, nChar+i, 1);
*/				XPolyPolygon aChar = XOutGetCharOutline(rText[nChar+i], this);
				TiltY(aChar, - fDy/fLen, fDx/fLen);
				aChar.Translate(aPolyPos);
//				aChar.Rotate(aPolyPos, nAngle);
				XOut.DrawXPolyPolygon(aChar);
				DrawMarker(aPolyPos, 3, MARKER_CROSS, *this);
			}
			nChar += nCnt;
			nTextLen -= nLen;
		}
		if ( nSkip > 0 )	nTextLen = 0;
		else				nSkip = 0;
		nPnt++;
	}
	ChangeFont(aOldFont);
}

/************************************************************************/

void MyWin::KeyInput(const KeyEvent& rKEvt)
{
	WorkWindow::KeyInput(rKEvt);

	switch (rKEvt.GetKeyCode().GetCode())
	{
		case KEY_DOWN:
			break;
		case KEY_UP:
			break;
		case KEY_LEFT:
			break;
		case KEY_RIGHT:
			break;
		case KEY_DELETE:
			Invalidate();
			break;
		case KEY_SPACE:
			break;
		case KEY_TAB:
			break;
		case KEY_BACKSPACE:
			if ( bStop )
			{ bStop = FALSE; AniFast(); }
			else bStop = TRUE;
			nAngle = 0;
			Invalidate();
			break;
		case KEY_RETURN:
		{
/*            long start, duration;
			start = clock();
			for ( short i = 0; i < 200; i++ )
				XOut.DrawXPolygon(aBezier);
			duration = clock() - start;
			String aStr1("Zeit fuer 200 Beziers: ");
			aStr1 += duration;
			DrawText(Point(20, 20), aStr1);
*/			break;
		}
	}
	switch (rKEvt.GetCharCode())
	{
		case '+':
			break;
		case '-':
			break;
	}
}

// --- MyWin::MouseButtonDown() ------------------------------------------

void MyWin::MouseButtonDown( const MouseEvent& rMEvt )
{
	Point aNewPos = rMEvt.GetPosPixel();
	CaptureMouse();
	if ( nPolyPts < MAXPOLYPTS && aNewPos != aLastPos )
	{
		if ( nPolyPts == 0 )
			aXPoly.SetPointCount(0);
		aXPoly[nPolyPts++] = aNewPos;
		aLastPos = aNewPos;
		Pen aPen = ChangePen(Pen(Color(COL_BLACK), 1));
		DrawMarker(aNewPos, 3, MARKER_RECT, *this);
		SetPen(aPen);
	}

}

// --- MyWin::MouseButtonUp() --------------------------------------------

void MyWin::MouseButtonUp( const MouseEvent& rMEvt )
{
	if ( rMEvt.GetClicks() == 2 || nPolyPts == MAXPOLYPTS )
	{
		aXPoly = FitCurve(aXPoly, nPolyPts, fFitBezError);
		XOut.DrawXPolyLine(aXPoly);
//		DrawBezMark(aXPoly);
		DrawFormText(aFormText, aXPoly);
		nPolyPts = 0;
	}
	ReleaseMouse();
}

// --- MyWin::MouseMove() ------------------------------------------------

void MyWin::MouseMove( const MouseEvent& rMEvt )
{
	Point aPix(rMEvt.GetPosPixel());

	if ( rMEvt.GetMode() & MOUSE_DRAGMOVE )
	{
		long nDx = labs(aPix.X() - aLastPos.X());
		long nDy = labs(aPix.Y() - aLastPos.Y());

		if ( nDx + nDy > 6 && nPolyPts < MAXPOLYPTS )
		{
			Pen aPen = ChangePen(Pen(Color(COL_BLACK), 1));
			DrawMarker(aPix, 3, MARKER_RECT, *this);
			aXPoly[nPolyPts++] = aPix;
			aLastPos = aPix;
			SetPen(aPen);
		}
	}
}

// --- MyWin::Paint() ------------------------------------------------

void MyWin::Paint( const Rectangle& rRect )
{
	XOut.DrawXPolyLine(aXPoly);
	DrawFormText(aFormText, aXPoly);
}

// --- MyWin::Resize() ------------------------------------------------

void MyWin::Resize()
{
	Size aSize = GetOutputSizePixel();
	aBezier[8].X() = aBezier[0].X() = aSize.Width() / 10;
	aBezier[8].Y() = aBezier[0].Y() = aSize.Height() / 2 - aSize.Height() / 10;
	aBezier[1].X() = aSize.Width() / 3;
	aBezier[1].Y() = aSize.Height() / 10;
	aBezier[2].X() = aSize.Width() / 3 * 2;
	aBezier[2].Y() = aSize.Height() / 10;
	aBezier[3].X() = aSize.Width() - aSize.Width() / 10;
	aBezier[3].Y() = aSize.Height() / 2 - aSize.Height() / 10;
	aBezier[4].X() = aSize.Width() - aSize.Width() / 10;
	aBezier[4].Y() = aSize.Height() / 2 + aSize.Height() / 10;
	aBezier[5].X() = aSize.Width() / 3 * 2;
	aBezier[5].Y() = aSize.Height() - aSize.Height() / 10;
	aBezier[6].X() = aSize.Width() / 3;
	aBezier[6].Y() = aSize.Height() - aSize.Height() / 10;
	aBezier[7].X() = aSize.Width() / 10;
	aBezier[7].Y() = aSize.Height() / 2 + aSize.Height() / 10;
	aBezier.SetFlags(1, XPOLY_CONTROL);
	aBezier.SetFlags(2, XPOLY_CONTROL);
	aBezier.SetFlags(5, XPOLY_CONTROL);
	aBezier.SetFlags(6, XPOLY_CONTROL);
	Invalidate();
}

// ---------------------------------------------------

void MyWin::PntCheck(short nIndex, const Size& rSize)
{
	if ( aBezier[nIndex].X() < 0 )
	{
		aBezier[nIndex].X() = 0;
		dx[nIndex] = -dx[nIndex];
	}
	if ( aBezier[nIndex].X() >= rSize.Width() )
	{
		aBezier[nIndex].X() = rSize.Width() - 1;
		dx[nIndex] = -dx[nIndex];
	}
	if ( aBezier[nIndex].Y() < 0 )
	{
		aBezier[nIndex].Y() = 0;
		dy[nIndex] = -dy[nIndex];
	}
	if ( aBezier[nIndex].Y() >= rSize.Height() )
	{
		aBezier[nIndex].Y() = rSize.Height() - 1;
		dy[nIndex] = -dy[nIndex];
	}
}


long MyWin::AniHdl(Timer* pTimer)
{
	Size aSize = GetOutputSizePixel();
//  Paint(Rectangle(Point(0,0), aSize));
	for ( short i = 0; i < 8; i++ )
	{
		aBezier[i].X() += dx[i];
		aBezier[i].Y() += dy[i];
		PntCheck(i, aSize);
	}
	aBezier[8] = aBezier[0];
	Invalidate();
//  Paint(Rectangle(Point(0,0), aSize));
//  aTimer.Start();
	return 0;
}

void MyWin::AniFast()
{
	while ( !bStop )
	{
		Size aSize = GetOutputSizePixel();

		nAngle += 5;
		if ( nAngle >= 360 ) nAngle = 0;

		for ( short i = 0; i < 8; i++ )
		{
			aBezier[i].X() += dx[i];
			aBezier[i].Y() += dy[i];
			PntCheck(i, aSize);
		}
		aBezier[8] = aBezier[0];
		Invalidate();
		Update();
		aMyApp.Reschedule();
	}
}

/************************************************************************/

void MyWin::ZoomIn()
{
	MapMode aMap = GetMapMode();
	Fraction aFr(aMap.GetScaleX());
	long nSiz = aFr.GetNumerator() * 100 / aFr.GetDenominator();
	nSiz = nSiz * 3 / 2;
	if ( nSiz > 1000 ) nSiz = 1000;
	aFr = Fraction(nSiz, 100);
	aMap.ChangeScaleX(aFr);
	aMap.ChangeScaleY(aFr);
	SetMapMode(aMap);
	Invalidate();
}

/************************************************************************/

void MyWin::ZoomOut()
{
	MapMode aMap = GetMapMode();
	Fraction aFr(aMap.GetScaleX());
	long nSiz = aFr.GetNumerator() * 100 / aFr.GetDenominator();
	nSiz = nSiz * 2 / 3;
	if ( nSiz < 10 ) nSiz = 10;
	aFr = Fraction(nSiz, 100);
	aMap.ChangeScaleX(aFr);
	aMap.ChangeScaleY(aFr);
	SetMapMode(aMap);
	Invalidate();
}

// --- MyWin::FileSelect() -----------------------------------------------

long MyWin::FileSelect( Menu* pMenu )
{
	switch ( pMenu->GetCurItemId() )
	{
		case MID_QUIT:
			Close();
			break;
	}
	return TRUE;
}

// --- MyWin::ColorSelect() ----------------------------------------------

long MyWin::ColorSelect( Menu* pMenu )
{
	pMenu->CheckItem( nColorId, FALSE );
	nColorId = pMenu->GetCurItemId();
	pMenu->CheckItem( nColorId );
	Pen aPen = GetPen();

	switch( nColorId )
	{
		case MID_PENRED:
			 aPen.ChangeColor( Color( COL_RED ) );
			 break;
		case MID_PENBLUE:
			 aPen.ChangeColor( Color( COL_BLUE ) );
			 break;
		case MID_PENYELLOW:
			 aPen.ChangeColor( Color( COL_YELLOW ) );
			 break;
		case MID_PENGREEN:
			 aPen.ChangeColor( Color( COL_GREEN ) );
			 break;
		case MID_PENCYAN:
			 aPen.ChangeColor( Color( COL_CYAN ) );
			 break;
		case MID_PENMAGENTA:
			 aPen.ChangeColor( Color( COL_MAGENTA ) );
			 break;
		case MID_PENBROWN:
			 aPen.ChangeColor( Color( COL_BROWN ) );
			 break;
		case MID_PENRED2:
			 aPen.ChangeColor( Color( COL_LIGHTRED ) );
			 break;
		case MID_PENGREEN2:
			 aPen.ChangeColor( Color( COL_LIGHTGREEN ) );
			 break;
		case MID_PENBLACK:
			 aPen.ChangeColor( Color( COL_BLACK ) );
			 break;
		case MID_PENWHITE:
			 aPen.ChangeColor( Color( COL_WHITE ) );
			 break;
	}
	ChangePen( aPen );

	return TRUE;
}

// --- MyWin::WidthSelect() ----------------------------------------------

long MyWin::WidthSelect( Menu* pMenu )
{
	pMenu->CheckItem( nPenId, FALSE );
	nPenId = pMenu->GetCurItemId();
	pMenu->CheckItem( nPenId );

	switch ( nPenId )
	{
		case MID_PENSMALL:
			nLineWidth = 1;
			break;
		case MID_PENSMALL2:
			nLineWidth = 3;
			break;
		case MID_PENMEDIUM:
			nLineWidth = 6;
			break;
		case MID_PENTHICK:
			nLineWidth = 9;
			break;
		case MID_PENTHICK2:
			nLineWidth = 15;
			break;
	}
	return TRUE;
}

// --- MyApp::Main() -----------------------------------------------------

void MyApp::Main(int, char*[])
{
	MyWin       aMainWin;
	PopupMenu   aFileMenu;
	PopupMenu   aColorMenu;
	PopupMenu   aWidthMenu;
	MenuBar     aMenuBar;

	srand(1);

	aMenuBar.InsertItem( MID_FILE, "~Datei" );
	aMenuBar.InsertItem( MID_PENCOLOR, "~Farbe" );
	aMenuBar.InsertItem( MID_PENWIDTH, "~Breite" );

	aMenuBar.ChangePopupMenu( MID_FILE,     &aFileMenu );
	aMenuBar.ChangePopupMenu( MID_PENCOLOR, &aColorMenu );
	aMenuBar.ChangePopupMenu( MID_PENWIDTH, &aWidthMenu );

	aFileMenu.InsertItem( MID_QUIT, "~Ende..." );

	aColorMenu.InsertItem( MID_PENRED, "~Rot" );
	aColorMenu.InsertItem( MID_PENBLUE, "~Blau" );
	aColorMenu.InsertItem( MID_PENYELLOW, "~Gelb" );
	aColorMenu.InsertItem( MID_PENGREEN, "~Grn" );
	aColorMenu.InsertItem( MID_PENCYAN, "~Cyan" );
	aColorMenu.InsertItem( MID_PENMAGENTA, "~Magenta" );
	aColorMenu.InsertItem( MID_PENBROWN, "~Braun" );
	aColorMenu.InsertItem( MID_PENRED2, "~Hellrot" );
	aColorMenu.InsertItem( MID_PENGREEN2, "~Hellgrn" );
	aColorMenu.InsertItem( MID_PENBLACK, "~Schwarz" );
	aColorMenu.InsertItem( MID_PENWHITE, "~Wei" );

	aWidthMenu.InsertItem( MID_PENSMALL, "~Sehr klein" );
	aWidthMenu.InsertItem( MID_PENSMALL2, "~Klein" );
	aWidthMenu.InsertItem( MID_PENMEDIUM, "~Mittel" );
	aWidthMenu.InsertItem( MID_PENTHICK, "~Gro" );
	aWidthMenu.InsertItem( MID_PENTHICK2, "~Sehr Gro" );

	aFileMenu.PushSelectHdl( LINK( &aMainWin,
								   MyWin, FileSelect ) );
	aColorMenu.PushSelectHdl( LINK( &aMainWin,
									MyWin, ColorSelect ) );
	aWidthMenu.PushSelectHdl( LINK( &aMainWin,
									MyWin, WidthSelect ) );

	aColorMenu.CheckItem(MID_PENBLACK);
	aWidthMenu.CheckItem(MID_PENSMALL);

	ChangeAppMenu( &aMenuBar );

	aMainWin.SetText( "X_GDI Testprogramm" );
	aMainWin.Show();

	EnableSVLook();

	Execute();
}

/*** FatPoly ************************************************************/

void FatPolyLine(const XPolygon& rPoly, const long LinePattern[], long nWidth,
				 OutputDevice& rOutDev)
{
	USHORT nPntCnt = rPoly.GetPointCount();

	if ( nPntCnt > 1 )
	{
		FatLineParam aFlParam;
		const Point* pNextPoint;
		long nLineDx, nLineDy;

		Pen aPen = rOutDev.ChangePen(Pen(PEN_NULL));
		Brush aBrush = rOutDev.ChangeFillInBrush(Brush(aPen.GetColor()));
		do
		{   // Linien mit Laenge 0 nicht beruecksichtigen
			nLineDx = rPoly[1].X() - rPoly[0].X();
			nLineDy = rPoly[1].Y() - rPoly[0].Y();
		} while ( aFlParam.nLineDx == 0 && aFlParam.nLineDy == 0 );

		aFlParam.fLength = sqrt(nLineDx * nLineDx + nLineDy * nLineDy);
		aFlParam.nLineDx = nLineDx;
		aFlParam.nLineDy = nLineDy;
		double fWidth = nWidth / aFlParam.fLength;
		aFlParam.nDxW =   (long) (fWidth * nLineDy + 0.5);
		aFlParam.nDyW = - (long) (fWidth * nLineDx + 0.5);

		for (USHORT i = 0; i < nPntCnt - 1; i++)
		{
			if ( i+2 < nPntCnt )    pNextPoint = &rPoly[i+2];
			else                    pNextPoint = NULL;

			FatLine(rPoly[i], rPoly[i+1], pNextPoint, LinePattern, nWidth,
					aFlParam, rOutDev);
		}

		rOutDev.ChangePen(aPen);
		rOutDev.ChangeFillInBrush(aBrush);
	}
}

/*** FatLine ************************************************************/

void FatLine(const Point& rStart, const Point& rEnd, const Point* pNext,
			 const long* pLinePattern, long nWidth, FatLineParam& rParam,
			 OutputDevice& rOutDev)
{
	Polygon aPoly(4);
	BOOL    bLineComplete = FALSE;

	long nLineDx = rParam.nLineDx;
	long nLineDy = rParam.nLineDy;
	double fLength = rParam.fLength;
	if ( fLength == 0 )
		return;

	long nDxW = rParam.nDxW;
	long nDyW = rParam.nDyW;

	double  fDx = 0, fDy = 0;
	long    nDx, nDy;
	long    nSeg = rParam.nPatSeg;
	long    nPattern;

	// Linienmuster vorhanden?
	if ( !pLinePattern )
		nPattern = -1;
	// Angefangenes Segment zu Ende zeichnen?
	else if ( rParam.nPatRemain )
		nPattern = rParam.nPatRemain;
	else    // sonst naechstes Segment
	{
		nSeg++;
		if ( pLinePattern[nSeg] == 0 )  nSeg = 0;
		nPattern = pLinePattern[nSeg];
	}
	aPoly[0].X() = rStart.X() + nDxW / 2;
	aPoly[0].Y() = rStart.Y() + nDyW / 2;
	aPoly[1].X() = aPoly[0].X() - nDxW;
	aPoly[1].Y() = aPoly[0].Y() - nDyW;
	aPoly[2] = aPoly[1];
	aPoly[3] = aPoly[0];

	// An vorheriges Segment anschliessen?
	if ( rParam.bHasJoin )
	{
		aPoly[0] = rParam.aJoin1;
		aPoly[1] = rParam.aJoin2;
	}
	// Punkte zum Testen auf erreichen des Linienendes
	Point SegStart = rStart;
	Point SegEnd   = rStart;

	// Anschlusspunkte zunaechst im rechten Winkel an das Linienende
	rParam.aJoin1.X() = rEnd.X() + nDxW / 2;
	rParam.aJoin1.Y() = rEnd.Y() + nDyW / 2;
	rParam.aJoin2 = rParam.aJoin1;
	rParam.aJoin2.X() -= nDxW;
	rParam.aJoin2.Y() -= nDyW;

	// Anschlusslinie vorhanden?
	if ( pNext )
	{
		long nNextDx = pNext->X() - rEnd.X();
		long nNextDy = pNext->Y() - rEnd.Y();
		rParam.fLength = sqrt(nNextDx * nNextDx + nNextDy * nNextDy);

		if ( rParam.fLength > 0 )
		{
			// Berechnung des Anschlussuebergangs durch Auswertung
			// der Seitenverhaeltnisse in den Uebergangsdreiecken
			double fWidth = nWidth / rParam.fLength;
			long nNextDxW =   (long) (fWidth * nNextDy + 0.5);
			long nNextDyW = - (long) (fWidth * nNextDx + 0.5);
			long nDxA = nNextDxW - nDxW;
			long nDyA = nNextDyW - nDyW;
			long nDxU = nDxW + nNextDxW;
			long nDyU = nDyW + nNextDyW;
			double fJoin = nDxA * nDxA + nDyA * nDyA;
			double fDenom = nDxU * nDxU + nDyU * nDyU;
			if ( fDenom > 0 )   fJoin = sqrt(fJoin / fDenom) / 2;
			else                fJoin = 0;

			long nJoinDx = -(long) (fJoin * nDyW);
			long nJoinDy =  (long) (fJoin * nDxW);

			// Feststellen, ob Anschlusslinie links oder rechts
			// von der ersten Linie verlaeuft; < 0 -> links
			if ( (nLineDx * nNextDy - nLineDy * nNextDx) < 0 )
			{
				nJoinDx = -nJoinDx;
				nJoinDy = -nJoinDy;
			}
			rParam.aJoin1.X() += nJoinDx;
			rParam.aJoin1.Y() += nJoinDy;
			rParam.aJoin2.X() -= nJoinDx;
			rParam.aJoin2.Y() -= nJoinDy;
			rParam.bHasJoin = TRUE;

			rParam.nLineDx = nNextDx;
			rParam.nLineDy = nNextDy;
			rParam.nDxW = nNextDxW;
			rParam.nDyW = nNextDyW;
		}
	}
	else
		rParam.bHasJoin = FALSE;

	while ( !bLineComplete )
	{
		double fSegLength;
		if ( nPattern < 0 ) fSegLength = 1.0;
		else                fSegLength = (double) nPattern / fLength;
		fDx += fSegLength * nLineDx;
		fDy += fSegLength * nLineDy;
		nDx = (long) fDx;
		nDy = (long) fDy;
		fDx -= nDx;     // Rundungsfehler ausgleichen
		fDy -= nDy;
		aPoly[2].X() += nDx;
		aPoly[2].Y() += nDy;
		aPoly[3].X() += nDx;
		aPoly[3].Y() += nDy;
		SegEnd.X() += nDx;
		SegEnd.Y() += nDy;

		// wenn das rEnde ueberschritten wurde, hat das Vorzeichen
		// der Abstaende vom rEndpunkt gewechselt; durch Xor-Verknuepfung
		// wird dieser Wechsel festgestellt
		if ( ((SegEnd.X() - rEnd.X()) ^ (SegStart.X() - rEnd.X())) < 0 ||
			 ((SegEnd.Y() - rEnd.Y()) ^ (SegStart.Y() - rEnd.Y())) < 0 )
		{
			if ( nDx || nDy )
			{
				if ( labs(nDx) >= labs(nDy) )
				{
					long nDiffX = SegEnd.X() - rEnd.X();
					rParam.nPatRemain = nPattern * nDiffX / nDx;
				}
				else
				{
					long nDiffY = SegEnd.Y() - rEnd.Y();
					rParam.nPatRemain = nPattern * nDiffY / nDy;
				}
			}
			else
				rParam.nPatRemain = 0;

			rParam.nPatSeg = nSeg;
			aPoly[3] = rParam.aJoin1;
			aPoly[2] = rParam.aJoin2;
			bLineComplete = TRUE;
		}
		if ( !(nSeg & 0x1) )
			rOutDev.DrawPolygon(aPoly);

		aPoly[0] = aPoly[3];
		aPoly[1] = aPoly[2];
		SegStart = SegEnd;
		nSeg++;
		if ( pLinePattern )
		{
			if ( pLinePattern[nSeg] == 0 )
				nSeg = 0;
			nPattern = pLinePattern[nSeg];
		}
	}
}

/*************************************************************************
|*
|* Quadratwurzelberechnung auf Integer-Basis
|*
\************************************************************************/

ULONG LSqrt(ULONG nNum)
{
	ULONG nResult;
	ULONG nTry = nNum / 2;

	if ( nNum <= 1 )
		return nNum;
	do
	{   nResult = nTry;
		nTry = (nTry + nNum / nTry) / 2;
	} while ( nTry < nResult );
	return nResult;
}

/*************************************************************************
|*
|* Abstand zwischen zwei Punkten berechnen
|*
\************************************************************************/

ULONG CalcLineLength(const Point& aDiff)
{
	return LSqrt(aDiff.X() * aDiff.X() + aDiff.Y() * aDiff.Y());
}

/*************************************************************************
|*
|* Tangente fuer den Uebergang zwischen zwei Bezierkurven berechnen
|*  Center = End- bzw. Anfangspunkt der Bezierkurven
|*  Prev   = vorheriger Zugpunkt
|*  Next   = naechster Zugpunkt
|*
\************************************************************************/

void CalcTangent(const Point& rCenter, Point& rPrev, Point& rNext)
{
	ULONG nNextLength, nPrevLength, nAbsLength;
	Point aDiff(rNext - rPrev);

	nNextLength = CalcLineLength(rCenter - rNext);
	nPrevLength = CalcLineLength(rCenter - rPrev);
	nAbsLength  = CalcLineLength(aDiff);

	rNext = rCenter + (aDiff * nNextLength / nAbsLength);
	rPrev = rCenter - (aDiff * nPrevLength / nAbsLength);
}

/*************************************************************************
|*
|* Zugpunkte einer Bezierkurve aus zwei auf der Kurve liegenden
|* Punkten berechnen
|*
\************************************************************************/

void PointsToBezier(XPolygon& rXPoly, USHORT nPos)
{
	long    nFullLength, nPart1Length, nPart2Length;
	double  fX0, fY0, fX1, fY1, fX2, fY2, fX3, fY3;
	double  fTx1, fTx2, fTy1, fTy2;
	double  fT1, fU1, fT2, fU2, fV;

	fTx1 = rXPoly[nPos + 1].X();
	fTy1 = rXPoly[nPos + 1].Y();
	fTx2 = rXPoly[nPos + 2].X();
	fTy2 = rXPoly[nPos + 2].Y();
	fX0 = rXPoly[nPos + 0].X();   /* Startpunkt   */
	fY0 = rXPoly[nPos + 0].Y();
	fX3 = rXPoly[nPos + 3].X();   /* Endpunkt     */
	fY3 = rXPoly[nPos + 3].Y();

	nPart1Length = CalcLineLength(rXPoly[nPos+0] - rXPoly[nPos+1]);
	nPart2Length = nPart1Length + CalcLineLength(rXPoly[nPos+1] - rXPoly[nPos+2]);
	nFullLength  = nPart2Length + CalcLineLength(rXPoly[nPos+2] - rXPoly[nPos+3]);
	if ( nFullLength < 20 )
		return;

	if ( nPart2Length == nFullLength )
		nPart2Length -= 1;
	if ( nPart1Length == nFullLength )
		nPart1Length = nPart2Length - 1;
	if ( nPart1Length <= 0 )
		nPart1Length = 1;
	if ( nPart2Length <= 0 || nPart2Length == nPart1Length )
		nPart2Length = nPart1Length + 1;

	fT1 = (double) nPart1Length / nFullLength;
	fU1 = 1.0 - fT1;
	fT2 = (double) nPart2Length / nFullLength;
	fU2 = 1.0 - fT2;
	fV = 3 * (1.0 - (fT1 * fU2) / (fT2 * fU1));

	fX1 = fTx1 / (fT1 * fU1 * fU1) - fTx2 * fT1 / (fT2 * fT2 * fU1 * fU2);
	fX1 /= fV;
	fX1 -= fX0 * ( fU1 / fT1 + fU2 / fT2) / 3;
	fX1 += fX3 * ( fT1 * fT2 / (fU1 * fU2)) / 3;

	fY1 = fTy1 / (fT1 * fU1 * fU1) - fTy2 * fT1 / (fT2 * fT2 * fU1 * fU2);
	fY1 /= fV;
	fY1 -= fY0 * ( fU1 / fT1 + fU2 / fT2) / 3;
	fY1 += fY3 * ( fT1 * fT2 / (fU1 * fU2)) / 3;

	fX2 = fTx2 / (fT2 * fT2 * fU2 * 3) - fX0 * fU2 * fU2 / ( fT2 * fT2 * 3);
	fX2 -= fX1 * fU2 / fT2;
	fX2 -= fX3 * fT2 / (fU2 * 3);

	fY2 = fTy2 / (fT2 * fT2 * fU2 * 3) - fY0 * fU2 * fU2 / ( fT2 * fT2 * 3);
	fY2 -= fY1 * fU2 / fT2;
	fY2 -= fY3 * fT2 / (fU2 * 3);

	rXPoly[nPos + 1] = Point((long) fX1, (long) fY1);
	rXPoly[nPos + 2] = Point((long) fX2, (long) fY2);
	rXPoly.SetFlags(nPos + 1, XPOLY_CONTROL);
	rXPoly.SetFlags(nPos + 2, XPOLY_CONTROL);

//	if ( nPos > 0 )
//		CalcTangent(rXPoly[nPos + 0], rXPoly[nPos - 1], rXPoly[nPos + 1]);
}


/*************************************************************************
|*
|* An Algorithm for Automatically Fitting Digitized Curves
|* by Philip J. Schneider
|* from "Graphics Gems", Academic Press, 1990
|*
\************************************************************************/

class Point2D
{
 protected:
	double fX, fY;

 public:
	Point2D(double x = 0, double y = 0) : fX(x), fY(y) {}
	Point2D(const Point& rSVPoint) : fX(rSVPoint.X()), fY(rSVPoint.Y()) {}

	Point2D& operator=(const Point& rSVPoint)
	{ fX = rSVPoint.X(); fY = rSVPoint.Y(); return *this; }

	operator Point() { return Point((long)(fX+0.5), (long)(fY+0.5)); }

	double  X() const { return fX; }
	double  Y() const { return fY; }
	double& X() { return fX; }
	double& Y() { return fY; }

	double Length() const { return sqrt(fX * fX + fY * fY); }
	Point2D& Scale(double fNewLen);
	Point2D& Normalize();

	Point2D& operator+=(const Point2D& rPnt)
	{ fX += rPnt.fX; fY += rPnt.fY; return *this; }
	Point2D& operator-=(const Point2D& rPnt)
	{ fX -= rPnt.fX; fY -= rPnt.fY; return *this; }

	Point2D operator+(const Point2D& rPnt) const
	{ return Point2D(fX + rPnt.fX, fY + rPnt.fY); }
	Point2D operator-(const Point2D& rPnt) const
	{ return Point2D(fX - rPnt.fX, fY - rPnt.fY); }

	Point2D& operator*=(double fVal)
	{ fX *= fVal; fY *= fVal; return *this; }
	Point2D& operator/=(double fVal)
	{ fX /= fVal; fY /= fVal; return *this; }

	Point2D operator*(double fVal) const
	{ return Point2D(fX * fVal, fY * fVal); }
	Point2D operator/(double fVal) const
	{ return Point2D(fX / fVal, fY / fVal); }

	// Kreuzprodukt
	double operator*(const Point2D& rPnt) const
	{ return (fX * rPnt.fX + fY * rPnt.fY); }
};

typedef Point2D Vector2D;


Point2D& Point2D::Scale(double fNewLen)
{
	double fLen = Length();
	if ( fLen != 0.0 )
		*this *= fNewLen / fLen;
	return *this;
};

Point2D& Point2D::Normalize()
{
	double fLen = Length();
	if ( fLen != 0.0 )
		*this /= fLen;
	return *this;
};

/************************************************************************/

Point& operator+=(Point& rSVPoint, const Point2D& rPnt2D)
{
	Point aPoint(rPnt2D);
	return (rSVPoint += aPoint);
}

Point& operator-=(Point& rSVPoint, const Point2D& rPnt2D)
{
	Point aPoint(rPnt2D);
	return (rSVPoint -= aPoint);
}

Point operator+(const Point& rSVPoint, const Point2D& rPnt2D)
{
	Point aPoint(rPnt2D);
	return (rSVPoint + aPoint);
}

Point operator-(const Point& rSVPoint, const Point2D& rPnt2D)
{
	Point aPoint(rPnt2D);
	return (rSVPoint - aPoint);
}

/************************************************************************/

void FitCubic(const XPolygon& rPoly, USHORT nFirst, USHORT nLast,
			  Vector2D aTan1, Vector2D aTan2, double fError);
void GenerateBezier(const XPolygon& rPoly, USHORT nFirst, USHORT nLast,
					double *fParams, Vector2D aTan1, Vector2D aTan2);
Vector2D ComputeLeftTangent(const XPolygon& rPoly, USHORT nIndex);
Vector2D ComputeRightTangent(const XPolygon& rPoly, USHORT nIndex);
Vector2D ComputeCenterTangent(const XPolygon& rPoly, USHORT nIndex);
void ChordLengthParameterize(const XPolygon& rPoly, USHORT nFirst, USHORT nLast);
void Reparameterize(const XPolygon& rPoly, USHORT nFirst, USHORT nLast);
double ComputeMaxError(const XPolygon& rPoly, USHORT nFirst, USHORT nLast,
						double *fParams, USHORT* pSplitPoint);

static XPolygon*    pFitBezPoly;
static USHORT       nFitBezPolyPts;
static Polygon      aFitBezier(4);
static double       *pBezFitParams;

#define MAX_FITPOLY_SIZE 100

void AddBezierCurve(const Polygon& rBezier, BOOL bIsStraight)
{
	if ( (bIsStraight || rBezier[0] == rBezier[1] || rBezier[2] == rBezier[3] )
		 && nFitBezPolyPts <= MAX_FITPOLY_SIZE - 1 )
		(*pFitBezPoly)[nFitBezPolyPts++] = rBezier[3];
	else if ( nFitBezPolyPts <= MAX_FITPOLY_SIZE - 3 )
	{
		for (USHORT i = 1; i < 4; i++)
			(*pFitBezPoly)[nFitBezPolyPts++] = rBezier[i];
		pFitBezPoly->SetFlags(nFitBezPolyPts - 3, XPOLY_CONTROL);
		pFitBezPoly->SetFlags(nFitBezPolyPts - 2, XPOLY_CONTROL);
	}
}

/*
 *  FitCurve :
 *      Fit a Bezier curve to a set of digitized points
 */
XPolygon FitCurve(const XPolygon& rPoly, USHORT nPts, double fError)
{
	Vector2D aTan1, aTan2;  /*  Unit tangent vectors at endpoints */
	XPolygon aFitBezPoly(MAX_FITPOLY_SIZE);

	pFitBezPoly = &aFitBezPoly;
	nFitBezPolyPts = 1;
	aFitBezPoly[0] = rPoly[0];
	if ( nPts > 1 )
	{
		pBezFitParams = new double[nPts];
		aTan1 = ComputeLeftTangent(rPoly, 0);
		aTan2 = ComputeRightTangent(rPoly, nPts - 1);
		FitCubic(rPoly, 0, nPts - 1, aTan1, aTan2, fError);
		__DELETE(nPts) pBezFitParams;
	}
	aFitBezPoly.SetSize(nFitBezPolyPts);
	return aFitBezPoly;
}

/*
 *  FitCubic :
 *      Fit a Bezier curve to a (sub)set of digitized points
 */
void FitCubic(const XPolygon& rPoly, USHORT nFirst, USHORT nLast,
			  Vector2D aTan1, Vector2D aTan2, double fError)
{
	double      fMaxError;          // Maximum fitting error
	USHORT      nSplitPoint;        // Point to split point set at
	USHORT      nMaxIterations = 0; // Max times to try iterating
	Vector2D    aSplitTan;          // Unit tangent vector at splitPoint

	// Error below which to try iterating
	double  fIterationError = fError * fError;

	if ( (nLast - nFirst) == 1 )
	{
		aFitBezier[0] = rPoly[nFirst];
		aFitBezier[3] = rPoly[nLast];
		AddBezierCurve(aFitBezier, TRUE);
		return;
	}

	/* attempt to fit curve */
	ChordLengthParameterize(rPoly, nFirst, nLast);
	GenerateBezier(rPoly, nFirst, nLast, pBezFitParams, aTan1, aTan2);

	/*  Find max deviation of points to fitted curve */
	fMaxError = ComputeMaxError(rPoly, nFirst, nLast, pBezFitParams,
								&nSplitPoint);
	if ( fMaxError < fError )
	{
		AddBezierCurve(aFitBezier, FALSE);
		return;
	}

	/*  If error not too large, try some reparameterization  */
	/*  and iteration */
	if ( fMaxError < fIterationError )
	{
		for ( USHORT i = 0; i < nMaxIterations; i++ )
		{
			Reparameterize(rPoly, nFirst, nLast);
			GenerateBezier(rPoly, nFirst, nLast, pBezFitParams, aTan1, aTan2);
			fMaxError = ComputeMaxError(rPoly, nFirst, nLast,
										pBezFitParams, &nSplitPoint);

			if ( fMaxError < fError )
			{
				AddBezierCurve(aFitBezier, FALSE);
				return;
			}
		}
	}
	/* Fitting failed -- split at max error point and fit recursively */
	aSplitTan = ComputeCenterTangent(rPoly, nSplitPoint);
	FitCubic(rPoly, nFirst, nSplitPoint, aTan1, aSplitTan, fError);
	aSplitTan *= -1;
	FitCubic(rPoly, nSplitPoint, nLast, aSplitTan, aTan2, fError);
}

/*
 *  GenerateBezier :
 *  Use least-squares method to find Bezier control points for region.
 *
 */
void GenerateBezier(const XPolygon& rPoly, USHORT nFirst, USHORT nLast,
					double *fParams, Vector2D aTan1, Vector2D aTan2)
{
	Point2D     aFirst, aLast;
	Vector2D    V1, V2;
	double      fC[2][2];   // Matrix C
	double      fX0, fX1;
	double      fDetC0C1,   // Determinants of matrices
				fDetC0X,
				fDetXC1;
	double      fAlphaL,    // Alpha values, left and right
				fAlphaR;
	USHORT      i;

	fC[0][0] = 0.0;
	fC[0][1] = 0.0;
	fC[1][0] = 0.0;
	fC[1][1] = 0.0;
	fX0 = 0.0;
	fX1 = 0.0;

	aFirst = rPoly[nFirst];
	aLast = rPoly[nLast];

	for (i = nFirst; i <= nLast; i++)
	{
		double fU = fParams[i];
		double fV = 1.0 - fU;

		V1 = aTan1;
		V2 = aTan2;
		V1.Scale(3 * fU * fV * fV);
		V2.Scale(3 * fU * fU * fV);

		fC[0][0] += V1 * V1;        // Kreuzprodukt V1xV1
		fC[0][1] += V1 * V2;        // Kreuzprodukt V1xV2
		fC[1][0]  = fC[0][1];
		fC[1][1] += V2 * V2;        // Kreuzprodukt V2xV2

		Vector2D aTmpVec = rPoly[i];
		aTmpVec -= aFirst * (fV * fV * fV);
		aTmpVec -= aFirst * (3 * fU * fV * fV);
		aTmpVec -= aLast  * (3 * fU * fU * fV);
		aTmpVec -= aLast  * (fU * fU * fU);

		// Kreuzprodukte V1,V2 x aTmpVec
		fX0 += V1 * aTmpVec;
		fX1 += V2 * aTmpVec;
	}

	/* Compute the determinants of C and X  */
	fDetC0C1 = fC[0][0] * fC[1][1] - fC[1][0] * fC[0][1];
	fDetC0X  = fC[0][0] * fX1      - fC[0][1] * fX0;
	fDetXC1  = fX0      * fC[1][1] - fX1      * fC[0][1];

	/* Finally, derive alpha values */
	if ( fDetC0C1 == 0.0 )
		fDetC0C1 = (fC[0][0] * fC[1][1]) * SMALL_DOUBLE_VAL;
	if ( fDetC0C1 == 0.0 )
		fDetC0C1 = SMALL_DOUBLE_VAL;

	fAlphaL = fDetXC1 / fDetC0C1;
	fAlphaR = fDetC0X / fDetC0C1;

	aFitBezier[0] = rPoly[nFirst];
	aFitBezier[3] = rPoly[nLast];

	/*  If alpha negative, use the Wu/Barsky heuristic (see text) */
	if ( fAlphaL < 0.0 || fAlphaR < 0.0 )
	{
		Vector2D aVec = rPoly[nFirst] - rPoly[nLast];
		double fDist = aVec.Length() / 3.0;
		aFitBezier[1] = aFitBezier[0] + aTan1.Scale(fDist);
		aFitBezier[2] = aFitBezier[3] + aTan2.Scale(fDist);
		return;
	}

	/*  First and nLast control points of the Bezier curve are */
	/*  positioned exactly at the nFirst and nLast data points */
	/*  Control points 1 and 2 are positioned an alpha distance out */
	/*  on the tangent vectors, left and right, respectively */
	aFitBezier[1] = aFitBezier[0] + aTan1.Scale(fAlphaL);
	aFitBezier[2] = aFitBezier[3] + aTan2.Scale(fAlphaR);
}

/*
 *  EvaluateBezier :
 *      Evaluate a Bezier curve at a particular parameter value
 *
 */
Point2D EvaluateBezier(USHORT nDegree, const Polygon& rBezier, double fU)
{
	Point2D aTemp[4];       //Local copy of control points
	USHORT  i, j;

	for (i = 0; i <= nDegree; i++)
		aTemp[i] = rBezier[i];

	/* Triangle computation */
	for (i = 1; i <= nDegree; i++)
		for (j = 0; j <= nDegree - i; j++)
			aTemp[j] = aTemp[j] * (1.0 - fU) + aTemp[j+1] * fU;

	return aTemp[0];
}

/*
 *  NewtonRaphsonRootFind :
 *  Use Newton-Raphson iteration to find better root.
 */
double NewtonRaphsonRootFind(const Polygon& rBezier, const Point& rPoint,
							 double fU)
{
	double  fNumerator, fDenominator;
	Polygon Q1(3), Q2(2);       // Q' and Q''
	Point2D Qu, Q1u, Q2u;       // u evaluated at Q, Q', & Q''
	USHORT  i;

	/* Compute Q(u) */
	Qu = EvaluateBezier(3, rBezier, fU);

	/* Generate control vertices for Q' */
	for (i = 0; i <= 2; i++)
		Q1[i] = (rBezier[i+1] - rBezier[i]) * 3;

	/* Generate control vertices for Q'' */
	for (i = 0; i <= 1; i++)
		Q2[i] = (Q1[i+1] - Q1[i]) * 2.0;

	/* Compute Q'(u) and Q''(u) */
	Q1u = EvaluateBezier(2, Q1, fU);
	Q2u = EvaluateBezier(1, Q2, fU);
	Qu.X() -= rPoint.X();
	Qu.Y() -= rPoint.Y();

	/* Compute f(u)/f'(u) */
	fNumerator = Qu * Q1u;
	fDenominator = Q1u * Q1u + Qu * Q2u;

	/* u = u - f(u)/f'(u) */
	if ( fDenominator == 0.0 )
		return fU;
	else
		return (fU - (fNumerator/fDenominator));
}

/*
 * ComputeLeftTangent, ComputeRightTangent, ComputeCenterTangent :
 *Approximate unit tangents at endpoints and "center" of digitized curve
 */
Vector2D ComputeLeftTangent(const XPolygon& rPoly, USHORT nIndex)
{
	Vector2D aTangent = rPoly[nIndex + 1] - rPoly[nIndex];
	aTangent.Normalize();
	return aTangent;
}

Vector2D ComputeRightTangent(const XPolygon& rPoly, USHORT nIndex)
{
	Vector2D aTangent = rPoly[nIndex - 1] - rPoly[nIndex];
	aTangent.Normalize();
	return aTangent;
}

Vector2D ComputeCenterTangent(const XPolygon& rPoly, USHORT nIndex)
{
	Vector2D aTangent = rPoly[nIndex - 1] - rPoly[nIndex + 1];
	aTangent /= 2;
	aTangent.Normalize();
	return aTangent;
}

/*
 *  ChordLengthParameterize :
 *  Assign parameter values to digitized points
 *  using relative distances between points.
 */
void ChordLengthParameterize(const XPolygon& rPoly, USHORT nFirst, USHORT nLast)
{
	USHORT i;

	pBezFitParams[nFirst] = 0.0;

	for (i = nFirst + 1; i <= nLast; i++)
	{
		Vector2D aVec = rPoly[i-1] - rPoly[i];
		pBezFitParams[i] = pBezFitParams[i-1] + aVec.Length();
	}
	for (i = nFirst + 1; i <= nLast; i++)
		pBezFitParams[i] /= pBezFitParams[nLast];
}

/*
 *  Reparameterize:
 *  Given set of points and their parameterization, try to find
 *   a better parameterization.
 *
 */
void Reparameterize(const XPolygon& rPoly, USHORT nFirst, USHORT nLast)
{
	for (USHORT i = nFirst; i <= nLast; i++)
		pBezFitParams[i] = NewtonRaphsonRootFind(aFitBezier, rPoly[i],
												 pBezFitParams[i]);
}

/*
 *  ComputeMaxError :
 *  Find the maximum squared distance of digitized points
 *  to fitted curve.
*/
double ComputeMaxError(const XPolygon& rPoly, USHORT nFirst, USHORT nLast,
						double *fParams, USHORT* pSplitPoint)
{
	double      fMaxDist;       // Maximum error
	double      fDist;          // Current error
	Vector2D    aVec;           // Vector2D from point to curve

	*pSplitPoint = (nLast - nFirst + 1) / 2;
	fMaxDist = 0.0;

	for (USHORT i = nFirst + 1; i < nLast; i++)
	{
		aVec = EvaluateBezier(3, aFitBezier, fParams[i]);
		aVec -= rPoly[i];
		fDist = aVec * aVec;
		if ( fDist >= fMaxDist )
		{
			fMaxDist = fDist;
			*pSplitPoint = i;
		}
	}
	return (fMaxDist);
}

/************************************************************************/

void SplitBezier(const XPolygon& rBez, XPolygon& rLeft, XPolygon& rRight)
{
	rLeft[0] = rBez[0];
	rLeft[1] = (rBez[0] + rBez[1]) / 2;
	rLeft[2] = (rBez[0] + rBez[1] * 2 + rBez[2]) / 4;
	rLeft[3] = (rBez[0] + rBez[1] * 3 + rBez[2] * 3 + rBez[3]) / 8;

	rRight[0] = rLeft[3];
	rRight[1] = (rBez[3] + rBez[2] * 2 + rBez[1]) / 4;
	rRight[2] = (rBez[3] + rBez[2]) / 2;
	rRight[3] = rBez[3];
}

/************************************************************************/

BOOL IsStraight(const XPolygon& rBez)
{
	return TRUE;
}

/************************************************************************/

void DrawBezierSub(const XPolygon& rBez)
{
	if ( IsStraight(rBez) )
;//		DrawLine(rBez[0], rBez[3]);
	else
	{
		XPolygon aLeft(4), aRight(4);
		SplitBezier(rBez, aLeft, aRight);
		DrawBezierSub(aLeft);
		DrawBezierSub(aRight);
	}
}

