/*************************************************************************
 *
 *  $RCSfile: Office.java,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: hr $ $Date: 2003/06/30 15:30:11 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  the BSD license.
 *  
 *  Copyright (c) 2003 by Sun Microsystems, Inc.
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. Neither the name of Sun Microsystems, Inc. nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 *  FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 *  COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 *  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 *  OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 *  TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 *  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *     
 *************************************************************************/

import java.awt.Dimension;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;

import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.FileOutputStream;
import java.io.FileInputStream;

import com.sun.star.lang.XMultiServiceFactory;
import com.sun.star.frame.XFrame;
import com.sun.star.frame.XFrameActionListener;
import com.sun.star.frame.XModel;
import com.sun.star.frame.XStorable;
import com.sun.star.view.XSelectionSupplier;
import com.sun.star.beans.PropertyValue;
import com.sun.star.util.XModifiable;
import com.sun.star.uno.UnoRuntime;

/**
 * This class represents a primitive office bean.
 */
public class Office
	extends BasicOfficeBean
{
	public final static int	DEFAULT_WIDTH	= 600;
	public final static int	DEFAULT_HEIGHT	= 300;

	public final static short	SID_TOGGLEPREVIEW	= 5325;
	public final static short	SID_TOGGLEMENUBAR	= 6661;
	public final static short	SID_TOGGLEOBJECTBAR	= 5905;
	public final static short	SID_TOGGLETOOLBAR	= 5909;

	private boolean	mReadOnly			= false;
	private boolean	mPreviewVisible		= false;
	private boolean	mMenuBarVisible		= true;
	private boolean	mObjectBarVisible	= true;
	private boolean	mToolBarVisible		= true;

	private transient PropertyChangeListener	mPropertyChangeListener;

	/**
	 * Retrives the service factory.
	 *
	 * @return The service factory.
	 */
	public XMultiServiceFactory getServiceFactory()
	{
		return mServiceFactory;
	}

	/**
	 * Retrives the document frame.
	 *
	 * @return The document frame.
	 */
	public XFrame getFrame()
	{
		return mFrame;
	}

	/**
	 * Retrives the XModifiable interface of the document frame.
	 *
	 * @return The XModifiable interface of the document frame.
	 */
	public XModifiable getXModifiable()
	{
		return mModifiable;
	}

	/**
	 * Retrives the XSelectionSupplier interface of the document frame 
	 * controller.
	 *
	 * @return The XSelectionSupplier interface of the document frame 
	 * controller.
	 */
	public XSelectionSupplier getSelectionSupplier()
	{
		XSelectionSupplier	selection;
		selection	= (XSelectionSupplier)UnoRuntime.queryInterface(
			XSelectionSupplier.class, mFrame.getController());
		return selection;
	}

	/**
	 * Retrives the document prefered size.
	 *
	 * @return The document prefered size.
	 */
	public Dimension getPreferredSize()
	{
		return new Dimension(DEFAULT_WIDTH, DEFAULT_HEIGHT);
	}

	/**
	 * Retrives the read only property of the document.
	 *
	 * @return <code>true</code> if the document is read only.
	 */
	public boolean isReadOnly()
	{
		return mReadOnly;
	}

	/**
	 * Sets the read only property of the document.
	 *
	 * @param readonly if <code>true</code> the document becomes 
	 *	a read only document.
	 */
	public void setReadOnly(boolean readonly)
	{
		firePropertyChange("ReadOnly", new Boolean(mReadOnly),
			new Boolean(readonly));
		mReadOnly	= readonly;
	}

	/**
	 * Retrives the state of the document preview.
	 *
	 * @return <code>true</code> if the document preview is visible.
	 */
	public boolean isPreviewVisible()
	{
		return mPreviewVisible;
	}

	/**
	 * Sets the state of the document preview.
	 *
	 * @param visible if <code>true</code> the document preview is visible.
	 */
	public void setPreviewVisible(boolean visible)
	{
		if (mPreviewVisible != visible) {
			if (isDocumentLoaded() == true) {
				OfficeCommand command = new OfficeCommand(SID_TOGGLEPREVIEW);
				command.appendParameter("Preview", new Boolean(visible));
				command.execute(this);
			}
			firePropertyChange("PreviewVisible", new Boolean(mPreviewVisible),
				new Boolean(visible));
			mPreviewVisible	= visible;
		}
	}

	/**
	 * Retrives the visibility of the menu bar.
	 *
	 * @return <code>true</code> if the menu bar is visable.
	 */
	public boolean isMenuBarVisible()
	{
		return mMenuBarVisible;
	}

	/**
	 * Sets the visibility of the menu bar.
	 *
	 * @param visible The visibility flag.
	 */
	public void setMenuBarVisible(boolean visible)
	{
		if (mMenuBarVisible != visible) {
			if (isDocumentLoaded() == true) {
				OfficeCommand command = new OfficeCommand(SID_TOGGLEMENUBAR);
				command.appendParameter("MenuBar", new Boolean(visible));
				command.execute(this);
			}
			firePropertyChange("MenuBarVisible", new Boolean(mMenuBarVisible),
				new Boolean(visible));
			mMenuBarVisible	= visible;
		}
	}

	/**
	 * Retrives the visibility of the object bar.
	 *
	 * @return <code>true</code> if the object bar is visable.
	 */
	public boolean isObjectBarVisible()
	{
		return mObjectBarVisible;
	}

	/**
	 * Sets the visibility of the object bar.
	 *
	 * @param visible The visibility flag.
	 */
	public void setObjectBarVisible(boolean visible)
	{
		if (mObjectBarVisible != visible) {
			if (isDocumentLoaded() == true) {
				OfficeCommand command = new OfficeCommand(SID_TOGGLEOBJECTBAR);
				command.appendParameter("ObjectBarVisible", new Boolean(visible));
				command.execute(this);
			}
			firePropertyChange("ObjectBarVisible", new Boolean(mObjectBarVisible),
				new Boolean(visible));
			mObjectBarVisible	= visible;
		}
	}

	/**
	 * Retrives the visibility of the tool bar.
	 *
	 * @return <code>true</code> if the tool bar is visable.
	 */
	public boolean isToolBarVisible()
	{
		return mToolBarVisible;
	}

	/**
	 * Sets the visibility of the tool bar.
	 *
	 * @param visible The visibility flag.
	 */
	public void setToolBarVisible(boolean visible)
	{
		if (mToolBarVisible != visible) {
			if (isDocumentLoaded() == true) {
				OfficeCommand command = new OfficeCommand(SID_TOGGLETOOLBAR);
				command.appendParameter("ToolBarVisible", new Boolean(visible));
				command.execute(this);
			}
			firePropertyChange("ToolBarVisible", new Boolean(mToolBarVisible),
				new Boolean(visible));
			mToolBarVisible	= visible;
		}
	}

	/**
	 * Loads a document referenced by a URL.
	 *
	 * @param url The document's URL string. 
	 * @exception java.io.IOException if the document loading process has 
	 *	failed.
	 */
	public synchronized void load(String url)
		throws java.io.IOException
	{
		super.load(url);
		setMenuBarVisible(false);
	}

	/**
	 * Loads the document from the given InputStream.
	 *
	 * @param in The InputStream from which to load the document
	 * @see #save
	 */
	public void load(InputStream in)
		throws java.io.IOException
	{
		// BUG: Office 627: type detection needs to have the extension
		load(in, "tmp");
	}

	/**
	 * loads the document from the given stream.
	 * @see #save
	 */
	private void load(InputStream in, String sExtension)
		throws java.io.IOException
	{
		File tmpFile = java.io.File.createTempFile("officebean", sExtension);

		try {
			String tmpFileName = "file:///" + tmpFile.getAbsolutePath();

			byte[] buffer = new byte[1024];
			OutputStream o = new FileOutputStream(tmpFile);

			int len = 0;
			while ((len = in.read(buffer)) > 0)
				o.write(buffer, 0, len);
			o.close();

			load(tmpFileName);
		}
		catch (java.lang.Exception e) {
			e.printStackTrace(System.out);
		}
		tmpFile.delete();
	}

	/**
	* saves the document to the given stream.
	* @see #loadFromStream
	*/
	public void save(OutputStream o)
		throws java.io.IOException
	{
		if (mFrame == null)
			throw new java.io.IOException("no document loaded");

		XModel xModel = mFrame.getController().getModel();
		XStorable xStorable =
			(XStorable)UnoRuntime.queryInterface(XStorable.class, xModel);

		File tmpFile = File.createTempFile("officebean", "tmp");

		try {
			String tmpFileName = "file:///" + tmpFile.getAbsolutePath();
			xStorable.storeToURL(tmpFileName, new PropertyValue[0]);

			byte[] buffer = new byte[1024];
			InputStream in = new FileInputStream(tmpFile);

			int len = 0;
			while ((len = in.read(buffer)) > 0)
				o.write(buffer, 0, len);
			in.close();
		}
		catch (com.sun.star.io.IOException e) {
			throw new java.io.IOException(e.getMessage());
		}
		tmpFile.delete();
	}

	/**
	 * Sets a document referenced by a URL.
	 *
	 * @param url The document's URL string. 
	 */
	public void setDocumentURL(String url)
	{
		try {
			load(url);
		} catch (java.io.IOException ioe) {
			mDocumentURL	= null;
		}
	}

	/**
	 * Sends an office slot command to the document.
	 *
	 * @param id The office slot command identifier.
	 * @param args The array of the command arguments.
	 * @param asyn if <code>true</code> the office slot command will be 
	 *	executed asynchronously.
	 */
	public void sendSlotCommand(short id, PropertyValue args[], boolean asyn)
	{
		OfficeCommand	command	= new OfficeCommand(id, args);
		if (asyn == true)
			queue(command);
		else
			command.execute(this);
	}

	/**
	 * Retrives a property change listener.
	 *
	 * @return The property change listener.
	 */
	public PropertyChangeListener getPropertyChangeListener()
	{
		if (mPropertyChangeListener == null) {
			mPropertyChangeListener	= new DefaultPropertyChangeListener();
			addPropertyChangeListener(mPropertyChangeListener);
		}
		return mPropertyChangeListener;
	}

	/**
	 * Sets the new property chanche listener.
	 *
	 * @param listener The new property chanche listener.
	 */
	public void setPropertyChangeListener(PropertyChangeListener listener)
	{
		if (mPropertyChangeListener == listener)
			return;
		if (mPropertyChangeListener != null)
			removePropertyChangeListener(mPropertyChangeListener);
		mPropertyChangeListener	= listener;
		addPropertyChangeListener(mPropertyChangeListener);
	}

	/**
	 * Determines is the property chanche listener available.
	 *
	 * @return <code>true</code> if the property chanche listener is available.
	 */
	public boolean isPropertyChangeListenerNull()
	{
		return (mPropertyChangeListener == null);
	}

	/**
	 * Adds a frame action listener.
	 *
	 * @param listener The frame action listener.
	 */
	public void addFrameActionListener(XFrameActionListener listener)
	{
		mFrame.addEventListener(listener);
	}

	/**
	 * Removes a frame action listener.
	 *
	 * @param listener The frame action listener.
	 */
	public void removeFrameActionListener(XFrameActionListener listener)
	{
		mFrame.removeFrameActionListener(listener);
	}

	/**
	 * Closes the connection to the office.
	 */
	public void dispose()
	{
		closeConnection();
	}

	/**
	 * A stub implementation of the property change listener.
	 */
	private class DefaultPropertyChangeListener
		implements PropertyChangeListener
	{
		public void propertyChange(PropertyChangeEvent evt)
		{
		}
	}
}
