/*************************************************************************
 *
 *  $RCSfile: basepoly.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: aw $ $Date: 2002/05/31 11:26:03 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/
#ifndef _BASEPOLY_HXX
#include "basepoly.hxx"
#endif

#include <tools/debug.hxx>
#include <stdlib.h>
#include <string.h>

//////////////////////////////////////////////////////////////////////////////

class ImpNodeListDynamicMemoryEntry
{
	BYTE*						mpNodeArray;
	UINT32						mnRefCount;

public:
	ImpNodeListDynamicMemoryEntry(BYTE* pArr)
	:	mpNodeArray(pArr),
		mnRefCount(1L)
	{}
	~ImpNodeListDynamicMemoryEntry() { /*#90353#*/ delete [] mpNodeArray; }

	BOOL DecRefCount() { return !(--mnRefCount); }
	void IncRefCount() { mnRefCount++; }
};

//////////////////////////////////////////////////////////////////////////////

class ImpNodeListDynamicEntry
{
	ImpNodeListDynamicMemoryEntry*		mpMemEntry;
	BYTE*								mpNodeArray;
	UINT32								mnNodeIndex;
	UINT32								mnNodeCount;

public:
	// linear constructor
	ImpNodeListDynamicEntry(UINT32 nInd, UINT32 nCnt, BYTE* pArr)
	:	mnNodeIndex(nInd),
		mnNodeCount(nCnt),
		mpNodeArray(pArr)
	{
		// new mem entry
		mpMemEntry = new ImpNodeListDynamicMemoryEntry(mpNodeArray);
	}

	// split constructor
	ImpNodeListDynamicEntry(ImpNodeListDynamicEntry& rToBeSplitted, UINT32 nInd, 
		UINT32 nInnerInd, UINT32 nByteSize)
	:	mnNodeIndex(nInd),
		mnNodeCount(rToBeSplitted.mnNodeCount - nInnerInd),
		mpNodeArray(rToBeSplitted.mpNodeArray + (nByteSize * nInnerInd))
	{
		// get same mem entry and inc it's ref count
		mpMemEntry = rToBeSplitted.mpMemEntry;
		mpMemEntry->IncRefCount();
		
		// correct number of members in ToBeSplitted value
		rToBeSplitted.mnNodeCount = nInnerInd;
	}

	// counting destructor
	~ImpNodeListDynamicEntry() 
	{ 
		if(mpMemEntry->DecRefCount())
			delete mpMemEntry; 
	}

	UINT32 GetNodeIndex() const { return mnNodeIndex; }
	UINT32 GetNodeCount() const { return mnNodeCount; }
	BYTE* GetNodeArray() const { return mpNodeArray; }

	// for correction of index counts
	void AddToNodeIndex(UINT32 nNew) { mnNodeIndex += nNew; }
	void SubFromNodeIndex(UINT32 nNew) { mnNodeIndex -= nNew; }
};

//////////////////////////////////////////////////////////////////////////////

DECLARE_LIST(ImpNodeListDynamicEntryList, ImpNodeListDynamicEntry*);

//////////////////////////////////////////////////////////////////////////////

class ImpNodeListDynamicArray
{
	ImpNodeListDynamicEntryList		maList;
	UINT32							mnHintNodeIndex;
	UINT32							mnHintEntryIndex;
	ImpNodeListDynamicEntry*		mpHintEntry;
	BYTE*							mpHintValue;
	const NodeListDataDescriptor&	mrDD;

	// private functions
	void ImpSetHintValues(UINT32 nPos);
	void ImpGrantIsolateOfHintNode();

public:
	ImpNodeListDynamicArray(BYTE* pNodeArray, UINT32 nNodeCount, 
		const NodeListDataDescriptor& rDD);
	~ImpNodeListDynamicArray();

	void CopyToStatic(BYTE* pNodeArray);

	const BYTE&	GetConstReference(UINT32 nPos) const;
	BYTE& GetReference(UINT32 nPos);

	void InsertNodes(UINT32 nPos, UINT32 nCnt, const BYTE* pNodeArray, BOOL bOneArrayValue);
	void AppendNodes(UINT32 nCnt, const BYTE* pNodeArray, BOOL bOneArrayValue);
	void RemoveNodes(UINT32 nPos, UINT32 nCnt);
};

ImpNodeListDynamicArray::ImpNodeListDynamicArray(BYTE* pNodeArray, UINT32 nNodeCount, 
	const NodeListDataDescriptor& rDD)
:	mnHintNodeIndex(0L),
	mnHintEntryIndex(0L),
	mrDD(rDD)
{
	// create entry and add it
	mpHintEntry = new ImpNodeListDynamicEntry(0L, nNodeCount, pNodeArray);
	maList.Insert(mpHintEntry, LIST_APPEND);
	mpHintValue = mpHintEntry->GetNodeArray();
}

ImpNodeListDynamicArray::~ImpNodeListDynamicArray()
{
	while(maList.Count())
		delete maList.Remove(maList.Count() - 1);
}

void ImpNodeListDynamicArray::CopyToStatic(BYTE* pNodeArray)
{
	BYTE* pCurrentPtr = pNodeArray;

	// copy all points to it
	for(UINT32 a=0L;a<maList.Count();a++)
	{
		ImpNodeListDynamicEntry* pCand = maList.GetObject(a);
		memcpy(pCurrentPtr, pCand->GetNodeArray(), 
			pCand->GetNodeCount() * mrDD.GetByteSize());
		pCurrentPtr += pCand->GetNodeCount() * mrDD.GetByteSize();
	}
}

void ImpNodeListDynamicArray::ImpSetHintValues(UINT32 nPos)
{
	// access to identical value ?
	if(nPos == mnHintNodeIndex)
		return;

	// access to near DynamicEntry ?
	if(nPos >= mpHintEntry->GetNodeIndex())
	{
		if(nPos < mpHintEntry->GetNodeIndex() + mpHintEntry->GetNodeCount())
		{
			// access to SAME DynamicEntry! Calc new direct HintValue.
			mpHintValue = mpHintEntry->GetNodeArray() 
				+ (mrDD.GetByteSize() * (nPos - mpHintEntry->GetNodeIndex()));
			mnHintNodeIndex = nPos;
			return;
		}
		else
		{
			// access to next DynamicEntry ?
			if(mnHintEntryIndex + 1L < maList.Count())
			{
				mnHintEntryIndex++;
				mpHintEntry = maList.GetObject(mnHintEntryIndex);

				if(nPos < mpHintEntry->GetNodeIndex() + mpHintEntry->GetNodeCount())
				{
					// access inside next dynamic entry, fill hint values
					mpHintValue = mpHintEntry->GetNodeArray()
						+ (mrDD.GetByteSize() * (nPos - mpHintEntry->GetNodeIndex()));
					mnHintNodeIndex = nPos;
					return;
				}
			}
		}
	}
	else
	{
		// access to previous DynamicEntry ?
		if(mnHintEntryIndex > 0)
		{
			mnHintEntryIndex--;
			mpHintEntry = maList.GetObject(mnHintEntryIndex);

			if(nPos >= mpHintEntry->GetNodeIndex())
			{
				// access inside next dynamic entry, fill hint values
				mpHintValue = mpHintEntry->GetNodeArray()
					+ (mrDD.GetByteSize() * (nPos - mpHintEntry->GetNodeIndex()));
				mnHintNodeIndex = nPos;
				return;
			}
		}
	}

	// no hint could be used or there was none, calc hint values
	mnHintEntryIndex = 0L;
	UINT32 nHi(maList.Count());

	if(nHi)
	{
		nHi--;

		// binary search
		while(mnHintEntryIndex != nHi)
		{
			UINT32 nMid((mnHintEntryIndex + nHi) >> 1L);
			ImpNodeListDynamicEntry* pMid = maList.GetObject(nMid);

			if(nPos < pMid->GetNodeIndex())
			{
				nHi = nMid - 1L;
			}
			else
			{
				if(nPos >= pMid->GetNodeIndex() + pMid->GetNodeCount())
					mnHintEntryIndex = nMid + 1L;
				else
					mnHintEntryIndex = nHi = nMid;
			}
		}
	}

	mpHintEntry = maList.GetObject(mnHintEntryIndex);
	mpHintValue = mpHintEntry->GetNodeArray()
		+ (mrDD.GetByteSize() * (nPos - mpHintEntry->GetNodeIndex()));
	mnHintNodeIndex = nPos;
}

const BYTE&	ImpNodeListDynamicArray::GetConstReference(UINT32 nPos) const
{
	((ImpNodeListDynamicArray*)this)->ImpSetHintValues(nPos);
	return *mpHintValue;
}

BYTE& ImpNodeListDynamicArray::GetReference(UINT32 nPos)
{
	ImpSetHintValues(nPos);
	return *mpHintValue;
}

void ImpNodeListDynamicArray::AppendNodes(UINT32 nCnt, const BYTE* pNodeArray, BOOL bOneArrayValue)
{
	ImpNodeListDynamicEntry* pLast = maList.GetObject(maList.Count() - 1L);
	UINT32 nNewIndex(pLast->GetNodeIndex() + pLast->GetNodeCount());
	BYTE* pNewArray;

	// create new array and copy data to it
	pNewArray = new BYTE[nCnt * mrDD.GetByteSize()];

	if(bOneArrayValue)
	{
		BYTE* pCurrent = pNewArray;

		for(UINT32 a=0L;a<nCnt;a++)
		{
			memcpy(pCurrent, pNodeArray, mrDD.GetByteSize());
			pCurrent += mrDD.GetByteSize();
		}
	}
	else
		memcpy(pNewArray, pNodeArray, nCnt * mrDD.GetByteSize());
	
	// create new DynamicEntry and insert it at mnHintEntryIndex, shifting
	// the old points to back
	ImpNodeListDynamicEntry* mpNewEntry = new ImpNodeListDynamicEntry(nNewIndex, nCnt, pNewArray);
	maList.Insert(mpNewEntry, LIST_APPEND);
}

void ImpNodeListDynamicArray::ImpGrantIsolateOfHintNode()
{
	if(mpHintEntry->GetNodeIndex() != mnHintNodeIndex)
	{
		// split of npHitEntry, which has more than one point
		UINT32 nInnerIndex(mnHintNodeIndex - mpHintEntry->GetNodeIndex());

		// create new DynamicEntry at split position, insert in list
		// use splitting constructor; this corrects mpHintEntry, too
		ImpNodeListDynamicEntry* mpNewEntry = new ImpNodeListDynamicEntry(
			*mpHintEntry, mnHintNodeIndex, nInnerIndex, mrDD.GetByteSize()); 
		maList.Insert(mpNewEntry, mnHintEntryIndex + 1L);

		// set hint-values to new entry
		mnHintEntryIndex++;
		mpHintEntry = mpNewEntry;
	}
}

void ImpNodeListDynamicArray::InsertNodes(UINT32 nPos, UINT32 nCnt, const BYTE* pNodeArray, BOOL bOneArrayValue)
{
	// create hint values and Isolate nPos to the left, 
	// eventually a split is done at nPos
	ImpSetHintValues(nPos);
	ImpGrantIsolateOfHintNode();
	
	// now mpHintEntry->GetNodeIndex() == nPos is guaranteed.
	BYTE* pNewArray;

	// create new array and copy data to it
	pNewArray = new BYTE[nCnt * mrDD.GetByteSize()];

	if(bOneArrayValue)
	{
		BYTE* pCurrent = pNewArray;

		for(UINT32 a=0L;a<nCnt;a++)
		{
			memcpy(pCurrent, pNodeArray, mrDD.GetByteSize());
			pCurrent += mrDD.GetByteSize();
		}
	}
	else
		memcpy(pNewArray, pNodeArray, nCnt * mrDD.GetByteSize());

	// create new DynamicEntry and insert it at mnHintEntryIndex, shifting
	// the old points to back
	ImpNodeListDynamicEntry* mpNewEntry = new ImpNodeListDynamicEntry(nPos, nCnt, pNewArray);
	maList.Insert(mpNewEntry, mnHintEntryIndex);

	// correct index entries in all following DynamicEntries
	for(UINT32 a=mnHintEntryIndex; a<maList.Count();a++)
		maList.GetObject(a)->AddToNodeIndex(nCnt);

	// set HintValue to new points
	mpHintEntry = mpNewEntry;
	mpHintValue = mpHintEntry->GetNodeArray();
}

void ImpNodeListDynamicArray::RemoveNodes(UINT32 nPos, UINT32 nCnt)
{
	ImpNodeListDynamicEntry* pLast = maList.GetObject(maList.Count() - 1);
	UINT32 nNodeCount(pLast->GetNodeIndex() + pLast->GetNodeCount());
	UINT32 nBackIndex(nPos + nCnt);

	// need to split back position?
	if(nBackIndex < nNodeCount)
	{
		ImpSetHintValues(nBackIndex);
		ImpGrantIsolateOfHintNode();
	}

	// set index values to nPos and grant split
	ImpSetHintValues(nPos);
	ImpGrantIsolateOfHintNode();

	// now delete DynamicEntries while there is something to remove
	while(nCnt)
	{
		ImpNodeListDynamicEntry* pDeleteCand = maList.Remove(mnHintEntryIndex);
		nCnt -= pDeleteCand->GetNodeCount();
		delete pDeleteCand;
	}
	
	// correct index entries in all following DynamicEntries
	for(UINT32 a=mnHintEntryIndex; a<maList.Count();a++)
		maList.GetObject(a)->SubFromNodeIndex(nCnt);

	// reset HintValue to very first point
	mnHintNodeIndex = 0L;
	mnHintEntryIndex = 0L;

	if(maList.Count())
	{
		mpHintEntry = maList.GetObject(mnHintEntryIndex);
		mpHintValue = mpHintEntry->GetNodeArray();
	}
}

//////////////////////////////////////////////////////////////////////////////

class ImpNodeList
{
	BYTE*								mpNodeArray;
	ImpNodeListDynamicArray*			mpDynamicArray;
	UINT32								mnNodeCount;
	const NodeListDataDescriptor&		mrDD;

	// private functions
	void ImpForceDynamic();

public:
	ImpNodeList(const NodeListDataDescriptor& rDesc, UINT32 nCnt);
	ImpNodeList(const ImpNodeList& rSource);
	~ImpNodeList();

	const BYTE&	operator[](UINT32 nPos) const;
	BYTE& operator[](UINT32 nPos);

	BOOL operator==(const ImpNodeList& rPoly) const;
	BOOL operator!=(const ImpNodeList& rPoly) const;

	UINT32 GetNodeCount() const { return mnNodeCount; }
	BOOL IsSamePolyNodeType(const NodeListDataDescriptor& rDesc) const 
		{ return BOOL(&rDesc == &mrDD); }
	const NodeListDataDescriptor& GetPolyNodeType() const
		{ return mrDD; }

	void InsertNodes(UINT32 nPos, UINT32 nCnt, const BYTE* pNodeArray, 
		BOOL bOneArrayValue = FALSE);
	void RemoveNodes(UINT32 nPos, UINT32 nCnt);

	BOOL IsStatic() const { return !mpDynamicArray; }
	BOOL IsDynamic() const { return (BOOL)mpDynamicArray; }
	void ForceStatic();
};

void ImpNodeList::ImpForceDynamic()
{
	if(IsStatic() && mpNodeArray)
	{
		// get a DynamicEntryList
		mpDynamicArray = new ImpNodeListDynamicArray(mpNodeArray, mnNodeCount, mrDD);

		// set static array pointer to NULL
		mpNodeArray = 0L;
	}
}

ImpNodeList::ImpNodeList(const NodeListDataDescriptor& rDesc, UINT32 nCnt)
:	mpNodeArray(0L),
	mpDynamicArray(0L),
	mnNodeCount(nCnt),
	mrDD(rDesc)
{
	if(mnNodeCount)
	{
		mpNodeArray = new BYTE[mnNodeCount * mrDD.GetByteSize()];

		BYTE* pCurrentPtr = mpNodeArray;

		for(UINT32 a=0;a<mnNodeCount;a++)
		{
			memcpy(pCurrentPtr, &mrDD.GetInitData(), mrDD.GetByteSize());
			pCurrentPtr += mrDD.GetByteSize();
		}
	}
}

ImpNodeList::ImpNodeList(const ImpNodeList& rSource)
:	mpNodeArray(0L),
	mpDynamicArray(0L),
	mnNodeCount(rSource.mnNodeCount),
	mrDD(rSource.mrDD)
{
	if(mnNodeCount)
	{
		mpNodeArray = new BYTE[mnNodeCount * mrDD.GetByteSize()];

		if(rSource.IsDynamic())
		{
			rSource.mpDynamicArray->CopyToStatic(mpNodeArray);
		}
		else
		{
			memcpy(mpNodeArray, rSource.mpNodeArray, mnNodeCount * mrDD.GetByteSize());
		}
	}
}

ImpNodeList::~ImpNodeList()
{
	if(mpNodeArray)
	{
		/*#90353#*/ delete [] mpNodeArray;
		mpNodeArray = 0L;
	}
	else if(mpDynamicArray)
	{
		delete mpDynamicArray;
		mpDynamicArray = 0L;
	}
}

const BYTE&	ImpNodeList::operator[](UINT32 nPos) const
{
	DBG_ASSERT(nPos < mnNodeCount, "invalid index at const array access to ImpNodeList");
	if(IsDynamic())
		return mpDynamicArray->GetConstReference(nPos);
	return mpNodeArray[nPos];
}

BYTE& ImpNodeList::operator[](UINT32 nPos)
{
	DBG_ASSERT(nPos < mnNodeCount, "invalid index at array access to ImpNodeList");
	if(IsDynamic())
		return mpDynamicArray->GetReference(nPos);
	return mpNodeArray[nPos];
}

BOOL ImpNodeList::operator==(const ImpNodeList& rPoly) const
{
	DBG_ASSERT(IsSamePolyNodeType(rPoly.GetPolyNodeType()), "different poly type in ==operator of ImpNodeList");
	DBG_ASSERT(GetNodeCount() == rPoly.GetNodeCount(), "different point counts in ==operator of ImpNodeList");

	for(UINT32 a=0L;a<GetNodeCount();a++)
		if(!mrDD.IsNodeEqual((*this)[a], rPoly[a]))
			return FALSE;

	return TRUE;
}

BOOL ImpNodeList::operator!=(const ImpNodeList& rPoly) const
{
	DBG_ASSERT(IsSamePolyNodeType(rPoly.GetPolyNodeType()), "different poly type in !=operator of ImpNodeList");
	DBG_ASSERT(GetNodeCount() == rPoly.GetNodeCount(), "different point counts in !=operator of ImpNodeList");

	for(UINT32 a=0L;a<GetNodeCount();a++)
		if(mrDD.IsNodeEqual((*this)[a], rPoly[a]))
			return FALSE;

	return TRUE;
}

void ImpNodeList::InsertNodes(UINT32 nPos, UINT32 nCnt, const BYTE* pNodeArray, BOOL bOneArrayValue)
{
	if(nCnt)
	{
		ImpForceDynamic();

		if(nPos >= mnNodeCount)
			mpDynamicArray->AppendNodes(nCnt, pNodeArray, bOneArrayValue);
		else
			mpDynamicArray->InsertNodes(nPos, nCnt, pNodeArray, bOneArrayValue);

		mnNodeCount += nCnt;
	}
}

void ImpNodeList::RemoveNodes(UINT32 nPos, UINT32 nCnt)
{
	if(nCnt)
	{
		DBG_ASSERT(nPos+nCnt <= mnNodeCount, "invalid parameter range at ImpNodeList::RemoveNodes()");

		ImpForceDynamic();
		mpDynamicArray->RemoveNodes(nPos, nCnt);
		mnNodeCount -= nCnt;
	}
}

void ImpNodeList::ForceStatic()
{
	if(IsDynamic())
	{
		// create new static mem array and set it
		mpNodeArray = new BYTE[mnNodeCount * mrDD.GetByteSize()];

		// copy all points to it
		mpDynamicArray->CopyToStatic(mpNodeArray);

		// delete dynamic entries and list and set ptr to NULL
		delete mpDynamicArray;
		mpDynamicArray = 0L;
	}
}

//////////////////////////////////////////////////////////////////////////////

class ImpBasePolygon : public ImpNodeList
{
	// reference counter
	UINT32								mnRefCount;
	BOOL								mbClosed;

public:
	ImpBasePolygon(const NodeListDataDescriptor& rDesc, UINT32 nCnt, BOOL bClosed = FALSE)
	:	ImpNodeList(rDesc, nCnt),
		mnRefCount(1L),
		mbClosed(bClosed)
	{}
	ImpBasePolygon(const ImpBasePolygon& rSource)
	:	ImpNodeList(rSource),
		mnRefCount(1L),
		mbClosed(rSource.mbClosed)
	{}
	~ImpBasePolygon();

	BOOL operator==(const ImpBasePolygon& rPoly) const;
	BOOL operator!=(const ImpBasePolygon& rPoly) const;

	UINT32 ImpGetRefCount() const { return mnRefCount; }
	void ImpIncRefCount() { mnRefCount++; }
	void ImpDecRefCount() { mnRefCount--; }

	void RemoveDoubleNodes();
	void FlipOrientation();

	void CheckClosed();
	BOOL IsClosed() const { return mbClosed; }
	void SetClosed(BOOL bNew) { mbClosed = bNew; }
};

ImpBasePolygon::~ImpBasePolygon()
{
	DBG_ASSERT(mnRefCount==0L, "ImpBasePolygon with non-null RefCount deleted");
}

BOOL ImpBasePolygon::operator==(const ImpBasePolygon& rPoly) const
{
	if(mbClosed != rPoly.mbClosed)
		return FALSE;

	return ImpNodeList::operator==(rPoly);
}

BOOL ImpBasePolygon::operator!=(const ImpBasePolygon& rPoly) const
{
	if(ImpNodeList::operator!=(rPoly))
		return TRUE;

	return mbClosed != rPoly.mbClosed;
}

void ImpBasePolygon::RemoveDoubleNodes()
{
	CheckClosed();

	if(GetNodeCount() > 1L)
	{
		UINT32 nInd(1L);

		while(nInd < GetNodeCount())
		{
			if(GetPolyNodeType().IsNodeEqual((*this)[nInd], (*this)[nInd - 1L]))
				RemoveNodes(nInd, 1L);
			else
				nInd++;
		}
	}
}

void ImpBasePolygon::FlipOrientation()
{
	if(GetNodeCount() > 1L)
	{
		BYTE* pTemp = new BYTE[GetPolyNodeType().GetByteSize()];

		UINT32 nEnd(GetNodeCount() - 1L);
		UINT32 nCount(GetNodeCount() >> 1L);

		for(UINT32 a=0L;a<nCount;a++)
		{
			memcpy(pTemp, &(*this)[a], GetPolyNodeType().GetByteSize());
			memcpy(&(*this)[a], &(*this)[nEnd], GetPolyNodeType().GetByteSize());
			memcpy(&(*this)[nEnd++], pTemp, GetPolyNodeType().GetByteSize());
		}

		delete pTemp;
	}
}

void ImpBasePolygon::CheckClosed()
{
	if(GetNodeCount() > 1L)
	{
		while(GetNodeCount() > 1L 
			&& GetPolyNodeType().IsNodeEqual((*this)[0L], (*this)[GetNodeCount() - 1L]))
		{
			if(!IsClosed())
				SetClosed(TRUE);
			RemoveNodes(GetNodeCount() - 1L, 1L);
		}
	}
}

//////////////////////////////////////////////////////////////////////////////

void BasePolygon::CheckReference()
{
	if(mpImpPolygon->ImpGetRefCount() > 1L)
	{
		mpImpPolygon->ImpDecRefCount();
		mpImpPolygon = new ImpBasePolygon(*mpImpPolygon);
	}
}

void BasePolygon::CheckClosed()
{
	if(GetNodeCount() > 1L)
	{
		CheckReference();
		mpImpPolygon->CheckClosed();
	}
}

BasePolygon::BasePolygon(const NodeListDataDescriptor& rDesc, UINT32 nCnt, BOOL bClosed)
{
	mpImpPolygon = new ImpBasePolygon(rDesc, nCnt, bClosed);
}

BasePolygon::BasePolygon(const BasePolygon& rSource)
{
	mpImpPolygon = rSource.mpImpPolygon;
	rSource.mpImpPolygon->ImpIncRefCount();
}

BasePolygon::~BasePolygon()
{
	mpImpPolygon->ImpDecRefCount();
	
	if(!(mpImpPolygon->ImpGetRefCount()))
		delete mpImpPolygon;
}

UINT32 BasePolygon::GetNodeCount() const 
{ 
	return mpImpPolygon->GetNodeCount(); 
}

BOOL BasePolygon::IsSamePolyNodeType(const NodeListDataDescriptor& rDesc) const 
{ 
	return mpImpPolygon->IsSamePolyNodeType(rDesc); 
}

const NodeListDataDescriptor& BasePolygon::GetPolyNodeType() const
{
	return mpImpPolygon->GetPolyNodeType();
}

BasePolygon& BasePolygon::operator=(const BasePolygon& rPoly)
{
	DBG_ASSERT(IsSamePolyNodeType(rPoly.GetPolyNodeType()), "invalid operator= at BasePolygon");

	rPoly.mpImpPolygon->ImpIncRefCount();

	if(mpImpPolygon->ImpGetRefCount() > 1L)
		mpImpPolygon->ImpDecRefCount();
	else
		delete mpImpPolygon;
	
	mpImpPolygon = rPoly.mpImpPolygon;
	
	return *this;
}

BOOL BasePolygon::operator==(const BasePolygon& rPoly) const
{
	DBG_ASSERT(IsSamePolyNodeType(rPoly.GetPolyNodeType()), "invalid operator== at BasePolygon");
	
	if(rPoly.mpImpPolygon == mpImpPolygon)
		return TRUE;

	if(GetNodeCount() != rPoly.GetNodeCount())
		return FALSE;

	return (*rPoly.mpImpPolygon == *mpImpPolygon);
}

BOOL BasePolygon::operator!=(const BasePolygon& rPoly) const
{
	DBG_ASSERT(IsSamePolyNodeType(rPoly.GetPolyNodeType()), "invalid operator!= at BasePolygon");
	
	if(rPoly.mpImpPolygon != mpImpPolygon)
		return TRUE;

	if(GetNodeCount() != rPoly.GetNodeCount())
		return TRUE;

	return (*rPoly.mpImpPolygon != *mpImpPolygon);
}

BOOL BasePolygon::IsClosed() const
{
	return mpImpPolygon->IsClosed();
}

void BasePolygon::SetClosed(BOOL bNew)
{
	if(bNew != mpImpPolygon->IsClosed())
	{
		CheckReference();
		mpImpPolygon->SetClosed(bNew);
	}
}

void BasePolygon::RemoveNodes(UINT32 nPos, UINT32 nCnt)
{
	if(nCnt && nPos < GetNodeCount())
	{
		CheckReference();
		mpImpPolygon->RemoveNodes(nPos, nCnt);
	}
}

void BasePolygon::RemoveDoubleNodes()
{
	if(GetNodeCount() > 1L)
	{
		CheckReference();
		mpImpPolygon->RemoveDoubleNodes();
	}
}

void BasePolygon::FlipOrientation()
{
	if(GetNodeCount() > 1L)
	{
		CheckReference();
		mpImpPolygon->FlipOrientation();
	}
}

void BasePolygon::ForceStatic()
{
	if(mpImpPolygon->IsDynamic())
	{
		mpImpPolygon->ForceStatic();
	}
}

//////////////////////////////////////////////////////////////////////////////

BOOL BPolygon3D::IsNodeEqual(const BYTE& rA, const BYTE& rB)
{ 
	return ((const Vector3D&)rA == (const Vector3D&)rB); 
}

NodeListDataDescriptor BPolygon3D::aDescriptor(
	(BYTE&)aInitValue, sizeof(Vector3D), IsNodeEqual);
Vector3D BPolygon3D::aInitValue;

void BPolygon3D::InsertNodes(UINT32 nPos, UINT32 nCnt, 
	const Vector3D& rPointArray, BOOL bOneArrayValue)
{
	if(nCnt)
	{
		CheckReference();
		GetImpPolygon()->InsertNodes(nPos, nCnt, (BYTE*)&rPointArray, bOneArrayValue);
	}
}

void BPolygon3D::InsertNode(UINT32 nPos, const Vector3D& rPoint)
{
	CheckReference();
	GetImpPolygon()->InsertNodes(nPos, 1L, (BYTE*)&rPoint, TRUE);
}

//////////////////////////////////////////////////////////////////////////////

BOOL BPolygon2D::IsNodeEqual(const BYTE& rA, const BYTE& rB)
{ 
	return ((const Vector2D&)rA == (const Vector2D&)rB); 
}

NodeListDataDescriptor BPolygon2D::aDescriptor(
	(BYTE&)aInitValue, sizeof(Vector2D), IsNodeEqual);
Vector2D BPolygon2D::aInitValue;

void BPolygon2D::InsertNodes(UINT32 nPos, UINT32 nCnt, 
	const Vector2D& rPointArray, BOOL bOneArrayValue)
{
	if(nCnt)
	{
		CheckReference();
		GetImpPolygon()->InsertNodes(nPos, nCnt, (BYTE*)&rPointArray, bOneArrayValue);
	}
}

void BPolygon2D::InsertNode(UINT32 nPos, const Vector2D& rPoint)
{
	CheckReference();
	GetImpPolygon()->InsertNodes(nPos, 1L, (BYTE*)&rPoint, TRUE);
}

//////////////////////////////////////////////////////////////////////////////

BOOL BPolygonPoint::IsNodeEqual(const BYTE& rA, const BYTE& rB)
{ 
	return ((const Point&)rA == (const Point&)rB); 
}

NodeListDataDescriptor BPolygonPoint::aDescriptor(
	(BYTE&)aInitValue, sizeof(Point), IsNodeEqual);
Point BPolygonPoint::aInitValue;

void BPolygonPoint::InsertNodes(UINT32 nPos, UINT32 nCnt, 
	const Point& rPointArray, BOOL bOneArrayValue)
{
	if(nCnt)
	{
		CheckReference();
		GetImpPolygon()->InsertNodes(nPos, nCnt, (BYTE*)&rPointArray, bOneArrayValue);
	}
}

void BPolygonPoint::InsertNode(UINT32 nPos, const Point& rPoint)
{
	CheckReference();
	GetImpPolygon()->InsertNodes(nPos, 1L, (BYTE*)&rPoint, TRUE);
}

























