/*
    filemgmt.c
    Copyright (C) 1995, 1996 by Volker Lendecke
    Copyright (C) 1999  Petr Vandrovec
    Copyright (C) 1999  Roumen Petrov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Revision History:

	0.00  1995			Volker Lendecke
		Initial release.

	0.01  1999, June 1		Petr Vandrovec <vandrove@vc.cvut.cz>
		Splitted from ncplib.c.
		See ncplib.c for other contributors.

	0.02  1999, July		Petr Vandrovec <vandrove@vc.cvut.cz>
		New ncp_ns_* function group.

	0.03  1999, September		Roumen Petrov <rpetrov@usa.net>
		Added ncp tracing code.
		Different bugfixes.

	1.00  1999, November, 20	Petr Vandrovec <vandrove@vc.cvut.cz>
		Added license.

	1.01  2000, May 17		Bruce Richardson <brichardson@lineone.net>
		Added ncp_perms_to str and ncp_str_to_perms.
						
 */

#include "config.h"
#include "ncplib_i.h"
#include <ncp/nwcalls.h>

#include <string.h>
#include <errno.h>
#include <stdlib.h>

#include <libintl.h>
#define _(X) dgettext(PACKAGE, (X))
#define N_(X) (X)

#ifndef ENOPKG
#define ENOPKG	ENOSYS
#endif

/* Here the ncp calls begin
 */

#ifndef __MAKE_SULIB__
long
ncp_get_volume_info_with_number(struct ncp_conn *conn, int n,
				struct ncp_volume_info *target)
{
	long result;
	int len;

	ncp_init_request_s(conn, 44);
	ncp_add_byte(conn, n);

	if ((result = ncp_request(conn, 22)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	target->total_blocks = ncp_reply_dword_lh(conn, 0);
	target->free_blocks = ncp_reply_dword_lh(conn, 4);
	target->purgeable_blocks = ncp_reply_dword_lh(conn, 8);
	target->not_yet_purgeable_blocks = ncp_reply_dword_lh(conn, 12);
	target->total_dir_entries = ncp_reply_dword_lh(conn, 16);
	target->available_dir_entries = ncp_reply_dword_lh(conn, 20);
	target->sectors_per_block = ncp_reply_byte(conn, 28);

	memset(target->volume_name, 0, sizeof(target->volume_name));

	len = ncp_reply_byte(conn, 29);
	if (len > NCP_VOLNAME_LEN)
	{
		ncp_printf(_("ncpfs: volume name too long: %d\n"), len);
		ncp_unlock_conn(conn);
		return EIO;
	}
	memcpy(&(target->volume_name), ncp_reply_data(conn, 30), len);
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_get_volume_number(struct ncp_conn *conn, const char *name, int *target)
{
	long result;

	ncp_init_request_s(conn, 5);
	ncp_add_pstring(conn, name);

	if ((result = ncp_request(conn, 22)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*target = ncp_reply_byte(conn, 0);
	ncp_unlock_conn(conn);
	return 0;
}

NWCCODE
ncp_get_volume_name(struct ncp_conn *conn,
		    NWVOL_NUM volume,
		    char* name, size_t nlen) {
	NWCCODE result;
	size_t len;
	
	ncp_init_request_s(conn, 6);
	ncp_add_byte(conn, volume);
	if ((result = ncp_request(conn, 22)) != 0) {
		ncp_unlock_conn(conn);
		return result;
	}
	if (conn->ncp_reply_size < 1) {
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	len = ncp_reply_byte(conn, 0);
	if (conn->ncp_reply_size < 1 + len) {
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	if (name) {
		if (len >= nlen) {
			ncp_unlock_conn(conn);
			return NWE_BUFFER_OVERFLOW;
		}
		memcpy(name, ncp_reply_data(conn, 1), len);
		name[len] = 0;
	}
	ncp_unlock_conn(conn);
	return 0;
}

NWCCODE
NWGetVolumeName(struct ncp_conn *conn,
		NWVOL_NUM volume,
		char* name) {
	return ncp_get_volume_name(conn, volume, name, 17);
}

long
ncp_file_search_init(struct ncp_conn *conn,
		     int dir_handle, const char *path,
		     struct ncp_filesearch_info *target)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, dir_handle);
	ncp_add_pstring(conn, path);

	if ((result = ncp_request(conn, 62)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	target->volume_number = ncp_reply_byte(conn, 0);
	target->directory_id = ncp_reply_word_hl(conn, 1);
	target->sequence_no = ncp_reply_word_hl(conn, 3);
	target->access_rights = ncp_reply_byte(conn, 5);
	ncp_unlock_conn(conn);
	return 0;
}


long
ncp_file_search_continue(struct ncp_conn *conn,
			 struct ncp_filesearch_info *fsinfo,
			 int attributes, const char *name,
			 struct ncp_file_info *target)
{
	long result;

	ncp_init_request(conn);

	ncp_add_byte(conn, fsinfo->volume_number);
	ncp_add_word_hl(conn, fsinfo->directory_id);
	ncp_add_word_hl(conn, fsinfo->sequence_no);

	ncp_add_byte(conn, attributes);
	ncp_add_pstring(conn, name);

	if ((result = ncp_request(conn, 63)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	fsinfo->sequence_no = ncp_reply_word_hl(conn, 0);

	memset(target->file_name, 0, sizeof(target->file_name));
	memcpy(&(target->file_name), ncp_reply_data(conn, 4),
	       NCP_MAX_FILENAME);

	target->file_attributes = ncp_reply_byte(conn, 18);
	target->file_mode = ncp_reply_byte(conn, 19);
	target->file_length = ncp_reply_dword_hl(conn, 20);
	target->creation_date = ncp_reply_word_hl(conn, 24);
	target->access_date = ncp_reply_word_hl(conn, 26);
	target->update_date = ncp_reply_word_hl(conn, 28);
	target->update_time = ncp_reply_word_hl(conn, 30);

	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_get_finfo(struct ncp_conn *conn,
	      int dir_handle, const char *path, const char *name,
	      struct ncp_file_info *target)
{
	long result;

	struct ncp_filesearch_info fsinfo;

	if ((result = ncp_file_search_init(conn, dir_handle, path,
					   &fsinfo)) != 0)
	{
		return result;
	}
	if ((result = ncp_file_search_continue(conn, &fsinfo, 0, name,
					       target)) == 0)
	{
		return result;
	}
	if ((result = ncp_file_search_init(conn, dir_handle, path,
					   &fsinfo)) != 0)
	{
		return result;
	}
	return ncp_file_search_continue(conn, &fsinfo, aDIR, name, target);
}

long
ncp_open_file(struct ncp_conn *conn,
	      int dir_handle, const char *path,
	      int attr, int access,
	      struct ncp_file_info *target)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, dir_handle);
	ncp_add_byte(conn, attr);
	ncp_add_byte(conn, access);
	ncp_add_pstring(conn, path);

	if ((result = ncp_request(conn, 76)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(&(target->file_id), ncp_reply_data(conn, 0),
	       NCP_FILE_ID_LEN);

	memset(target->file_name, 0, sizeof(target->file_name));
	memcpy(&(target->file_name), ncp_reply_data(conn, 8),
	       NCP_MAX_FILENAME);

	target->file_attributes = ncp_reply_byte(conn, 22);
	target->file_mode = ncp_reply_byte(conn, 23);
	target->file_length = ncp_reply_dword_hl(conn, 24);
	target->creation_date = ncp_reply_word_hl(conn, 28);
	target->access_date = ncp_reply_word_hl(conn, 30);
	target->update_date = ncp_reply_word_hl(conn, 32);
	target->update_time = ncp_reply_word_hl(conn, 34);

	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_close_file(struct ncp_conn *conn, const char *file_id)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 0);
	ncp_add_mem(conn, file_id, 6);

	result = ncp_request(conn, 66);
	ncp_unlock_conn(conn);
	return result;
}

static int
ncp_do_create(struct ncp_conn *conn,
	      int dir_handle, const char *path,
	      int attr,
	      struct ncp_file_info *target,
	      int function)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, dir_handle);
	ncp_add_byte(conn, attr);
	ncp_add_pstring(conn, path);

	if ((result = ncp_request(conn, function)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(&(target->file_id), ncp_reply_data(conn, 0),
	       NCP_FILE_ID_LEN);

	memset(target->file_name, 0, sizeof(target->file_name));
	memcpy(&(target->file_name), ncp_reply_data(conn, 8),
	       NCP_MAX_FILENAME);

	target->file_attributes = ncp_reply_byte(conn, 22);
	target->file_mode = ncp_reply_byte(conn, 23);
	target->file_length = ncp_reply_dword_hl(conn, 24);
	target->creation_date = ncp_reply_word_hl(conn, 28);
	target->access_date = ncp_reply_word_hl(conn, 30);
	target->update_date = ncp_reply_word_hl(conn, 32);
	target->update_time = ncp_reply_word_hl(conn, 34);

	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_create_newfile(struct ncp_conn *conn,
		   int dir_handle, const char *path,
		   int attr,
		   struct ncp_file_info *target)
{
	return ncp_do_create(conn, dir_handle, path, attr, target, 77);
}

long
ncp_create_file(struct ncp_conn *conn,
		int dir_handle, const char *path,
		int attr,
		struct ncp_file_info *target)
{
	return ncp_do_create(conn, dir_handle, path, attr, target, 67);
}

long
ncp_erase_file(struct ncp_conn *conn,
	       int dir_handle, const char *path,
	       int attr)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, dir_handle);
	ncp_add_byte(conn, attr);
	ncp_add_pstring(conn, path);

	result = ncp_request(conn, 68);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_rename_file(struct ncp_conn *conn,
		int old_handle, const char *old_path,
		int attr,
		int new_handle, const char *new_path)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, old_handle);
	ncp_add_byte(conn, attr);
	ncp_add_pstring(conn, old_path);
	ncp_add_byte(conn, new_handle);
	ncp_add_pstring(conn, new_path);

	if ((result = ncp_request(conn, 69)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_create_directory(struct ncp_conn *conn,
		     int dir_handle, const char *path,
		     int inherit_mask)
{
	long result;

	ncp_init_request_s(conn, 10);
	ncp_add_byte(conn, dir_handle);
	ncp_add_byte(conn, inherit_mask);
	ncp_add_pstring(conn, path);

	result = ncp_request(conn, 22);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_delete_directory(struct ncp_conn *conn,
		     int dir_handle, const char *path)
{
	long result;

	ncp_init_request_s(conn, 11);
	ncp_add_byte(conn, dir_handle);
	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_pstring(conn, path);

	result = ncp_request(conn, 22);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_add_trustee(struct ncp_conn *conn,
		int dir_handle, const char *path,
		u_int32_t object_id, u_int8_t rights)
{
	long result;

	ncp_init_request_s(conn, 13);
	ncp_add_byte(conn, dir_handle);
	ncp_add_dword_hl(conn, object_id);
	ncp_add_byte(conn, rights);
	ncp_add_pstring(conn, path);

	result = ncp_request(conn, 22);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_delete_trustee(struct ncp_conn *conn,
		   int dir_handle, const char *path, u_int32_t object_id)
{
	long result;

	ncp_init_request_s(conn, 14);
	ncp_add_byte(conn, dir_handle);
	ncp_add_dword_hl(conn, object_id);
	ncp_add_byte(conn, 0);
	ncp_add_pstring(conn, path);

	result = ncp_request(conn, 22);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_get_trustee(struct ncp_conn *conn, u_int32_t object_id,
		u_int8_t vol, char *path,
		u_int16_t * trustee, u_int16_t * contin)
{
	long result;

	ncp_init_request_s(conn, 71);
	ncp_add_byte(conn, vol);
	ncp_add_word_hl(conn, *contin);
	ncp_add_dword_hl(conn, object_id);

	if ((result = ncp_request(conn, 23)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*contin = ncp_reply_word_hl(conn, 0);
	*trustee = ncp_reply_byte(conn, 6);
	strncpy(path, ncp_reply_data(conn, 8), ncp_reply_byte(conn, 7));
	path[ncp_reply_byte(conn, 7)] = 0;
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_rename_directory(struct ncp_conn *conn,
		     int dir_handle,
		     const char *old_path, const char *new_path)
{
	long result;

	ncp_init_request_s(conn, 15);
	ncp_add_byte(conn, dir_handle);
	ncp_add_pstring(conn, old_path);
	ncp_add_pstring(conn, new_path);

	result = ncp_request(conn, 22);
	ncp_unlock_conn(conn);
	return result;
}

static void
ncp_add_handle_path(struct ncp_conn *conn,
		    u_int8_t vol_num,
		    u_int32_t dir_base, u_int8_t dir_style,
		    const char *path)
{
	ncp_add_byte(conn, vol_num);
	ncp_add_dword_lh(conn, dir_base);
	ncp_add_byte(conn, dir_style);

	if (path) {
		ncp_add_byte(conn, 1);
		ncp_add_pstring(conn, path);
	} else
		ncp_add_byte(conn, 0);	
}
#endif	/* __MAKE_SULIB */

static void
ncp_add_handle_path2(struct ncp_conn *conn,
		     u_int8_t vol_num,
		     u_int32_t dir_base, int dir_style,
		     const unsigned char *encpath, int pathlen)
{
	ncp_add_byte(conn, vol_num);
	ncp_add_dword_lh(conn, dir_base);
	ncp_add_byte(conn, dir_style);	/* 1 = dir_base, 0xFF = no handle, 0 = handle */
	if (encpath) {
		ncp_add_mem(conn, encpath, pathlen);
	} else {
		ncp_add_byte(conn, 0);	/* empty path */
	}
}

static void
ncp_extract_file_info(void *structure, struct nw_info_struct *target)
{
	u_int8_t *name_len;
	const int info_struct_size = sizeof(struct nw_info_struct) - 257;

	memcpy(target, structure, info_struct_size);
	name_len = (u_int8_t*)structure + info_struct_size;
	target->nameLen = *name_len;
	strncpy(target->entryName, name_len + 1, *name_len);
	target->entryName[*name_len] = '\0';
	return;
}

#ifndef __MAKE_SULIB__
long
ncp_obtain_file_or_subdir_info(struct ncp_conn *conn,
			       u_int8_t source_ns,
			       u_int8_t target_ns,
			       u_int16_t search_attribs,
			       u_int32_t rim,
			       u_int8_t vol,
			       u_int32_t dirent,
			       const char* path,
			       struct nw_info_struct *target) {
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 6);
	ncp_add_byte(conn, source_ns);
	ncp_add_byte(conn, target_ns);
	ncp_add_word_lh(conn, search_attribs);
	ncp_add_dword_lh(conn, rim);
	ncp_add_handle_path(conn, vol, dirent, NCP_DIRSTYLE_DIRBASE, path);
	/* fn: 87 , subfn: 6 */
	if ((result = ncp_request(conn, 87)) == 0)
	{
		ncp_extract_file_info(ncp_reply_data(conn, 0), target);
	}
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_get_eff_directory_rights(struct ncp_conn *conn,
			     u_int8_t source_ns, u_int8_t target_ns,
			     u_int16_t search_attribs,
			     u_int8_t vol, u_int32_t dirent, const char *path,
			     u_int16_t * my_effective_rights)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 29);
	ncp_add_byte(conn, source_ns);
	ncp_add_byte(conn, target_ns);
	ncp_add_word_lh(conn, search_attribs);
	ncp_add_dword_lh(conn, 0);
	ncp_add_handle_path(conn, vol, dirent, NCP_DIRSTYLE_DIRBASE, path);
	/* fn: 87 , subfn: 29 */
	if ((result = ncp_request(conn, 87)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*my_effective_rights = ncp_reply_word_lh(conn, 0);
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_do_lookup2(struct ncp_conn *conn,
	       u_int8_t _source_ns,
	       struct nw_info_struct *dir,
	       const char *path,       /* may only be one component */
	       u_int8_t _target_ns,
	       struct nw_info_struct *target)
{
	u_int8_t vol_num;
	u_int32_t dir_base;
	long result;

	if (target == NULL)
	{
		return EINVAL;
	}
	if (dir == NULL)
	{

		/* Access a volume's root directory */
		ncp_init_request(conn);
		ncp_add_byte(conn, 22);		/* subfunction */
		ncp_add_byte(conn, _source_ns);
		ncp_add_byte(conn, _target_ns);
		ncp_add_byte(conn, 0);	/* reserved */
		ncp_add_byte(conn, 0);	/* reserved */
		/* path must be volume_name */
		ncp_add_handle_path(conn, 0, 0, NCP_DIRSTYLE_NOHANDLE, path);
		/* fn: 87 , subfn: 22 */
		if ((result = ncp_request(conn, 87)) != 0)
		{
			ncp_unlock_conn(conn);
			return result;
		}
		dir_base = ncp_reply_dword_lh(conn, 4);
		vol_num = ncp_reply_byte(conn, 8);
		ncp_unlock_conn(conn);
	} else
	{
		vol_num = dir->volNumber;
		dir_base = dir->dirEntNum;
	}

	return ncp_obtain_file_or_subdir_info(
		conn, _source_ns, _target_ns,
		0xFF, RIM_ALL,
		vol_num, dir_base, NULL,
		target);
}

long
ncp_do_lookup(struct ncp_conn *conn,
	      struct nw_info_struct *dir,
	      const char* path,
	      struct nw_info_struct *target) {
	return ncp_do_lookup2(conn, NW_NS_DOS, dir, path, NW_NS_DOS, target);
}

long
ncp_modify_file_or_subdir_dos_info(struct ncp_conn *conn,
				   struct nw_info_struct *file,
				   u_int32_t info_mask,
				   struct nw_modify_dos_info *info)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 7);	/* subfunction */
	ncp_add_byte(conn, 0);	/* dos name space */
	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_word_lh(conn, 0x8006);	/* search attribs: all */

	ncp_add_dword_lh(conn, info_mask);
	ncp_add_mem(conn, info, sizeof(*info));
	ncp_add_handle_path(conn, file->volNumber,
			    file->DosDirNum, NCP_DIRSTYLE_DIRBASE, NULL);
	/* fn: 87 , subfn: 7 */
	result = ncp_request(conn, 87);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_del_file_or_subdir(struct ncp_conn *conn,
		       struct nw_info_struct *dir, char *name)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 8);	/* subfunction */
	ncp_add_byte(conn, 0);	/* dos name space */
	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_word_lh(conn, 0x8006);	/* search attribs: all */
	ncp_add_handle_path(conn, dir->volNumber,
			    dir->DosDirNum, NCP_DIRSTYLE_DIRBASE, name);
	/* fn: 87 , subfn: 8 */
	result = ncp_request(conn, 87);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_open_create_file_or_subdir(struct ncp_conn *conn,
			       struct nw_info_struct *dir, char *name,
			       int open_create_mode,
			       u_int32_t create_attributes,
			       int desired_acc_rights,
			       struct nw_file_info *target)
{
	long result;

	target->opened = 0;

	ncp_init_request(conn);
	ncp_add_byte(conn, 1);	/* subfunction */
	ncp_add_byte(conn, 0);	/* dos name space */
	ncp_add_byte(conn, open_create_mode);
	ncp_add_word_lh(conn, 0x8006);
	ncp_add_dword_lh(conn, RIM_ALL);
	ncp_add_dword_lh(conn, create_attributes);
	/* The desired acc rights seem to be the inherited rights mask
	   for directories */
	ncp_add_word_lh(conn, desired_acc_rights);
	ncp_add_handle_path(conn, dir->volNumber,
			    dir->DosDirNum, NCP_DIRSTYLE_DIRBASE, name);
	/* fn: 87 , subfn: 1 */
	if ((result = ncp_request(conn, 87)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	target->opened = 1;
	target->server_file_handle = ncp_reply_dword_lh(conn, 0);
	target->open_create_action = ncp_reply_byte(conn, 4);
	ncp_extract_file_info(ncp_reply_data(conn, 6), &(target->i));
	ConvertToNWfromDWORD(target->server_file_handle, target->file_handle);

	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_initialize_search(struct ncp_conn *conn,
		      const struct nw_info_struct *dir,
		      int name_space,
		      struct ncp_search_seq *target)
{
	return ncp_initialize_search2(conn,
				      dir,
				      name_space,
				      NULL, 0,
				      target);
}

long
ncp_initialize_search2(struct ncp_conn *conn,
		       const struct nw_info_struct *dir,
		       int name_space,
		       const unsigned char *enc_subpath, int subpathlen,
		       struct ncp_search_seq *target)
{
	long result;

	if ((name_space < 0) || (name_space > 255))
	{
		return EINVAL;
	}
	memset(target, 0, sizeof(*target));

	ncp_init_request(conn);
	ncp_add_byte(conn, 2);	/* subfunction */
	ncp_add_byte(conn, name_space);
	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_handle_path2(conn, dir->volNumber,
			    dir->dirEntNum, NCP_DIRSTYLE_DIRBASE,
			    enc_subpath, subpathlen);

	/* fn: 87 , subfn: 2 */
	if ((result = ncp_request(conn, 87)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(&(target->s), ncp_reply_data(conn, 0), 9);
	target->name_space = name_space;

	ncp_unlock_conn(conn);
	return 0;
}

/* Search for everything */
long
ncp_search_for_file_or_subdir2(struct ncp_conn *conn,
			       int search_attributes,
			       u_int32_t RIM,
			       struct ncp_search_seq *seq,
			       struct nw_info_struct *target)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 3);	/* subfunction */
	ncp_add_byte(conn, seq->name_space);
	ncp_add_byte(conn, 0);	/* data stream (???) */
	ncp_add_word_lh(conn, search_attributes);	/* Search attribs */
	ncp_add_dword_lh(conn, RIM);	/* return info mask */
	ncp_add_mem(conn, &(seq->s), 9);
	if ((seq->name_space == NW_NS_NFS) || (seq->name_space == NW_NS_MAC))
		ncp_add_byte(conn, 0);
	else {
		ncp_add_byte(conn, 2);	/* 2 byte pattern */
		ncp_add_byte(conn, 0xff);	/* following is a wildcard */
		ncp_add_byte(conn, '*');
	}
	/* fn: 87 , subfn: 3 */
	if ((result = ncp_request(conn, 87)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	memcpy(&(seq->s), ncp_reply_data(conn, 0), sizeof(seq->s));
	ncp_extract_file_info(ncp_reply_data(conn, 10), target);

	ncp_unlock_conn(conn);
	return 0;
}

/* Search for everything */
long
ncp_search_for_file_or_subdir(struct ncp_conn *conn,
			      struct ncp_search_seq *seq,
			      struct nw_info_struct *target)
{
	return ncp_search_for_file_or_subdir2(conn, 0x8006, RIM_ALL, seq, target);
}

long
ncp_ren_or_mov_file_or_subdir(struct ncp_conn *conn,
			      struct nw_info_struct *old_dir, char *old_name,
			      struct nw_info_struct *new_dir, char *new_name)
{
	long result;

	if ((old_dir == NULL) || (old_name == NULL)
	    || (new_dir == NULL) || (new_name == NULL))
		return EINVAL;

	ncp_init_request(conn);
	ncp_add_byte(conn, 4);	/* subfunction */
	ncp_add_byte(conn, 0);	/* dos name space */
	ncp_add_byte(conn, 1);	/* rename flag */
	ncp_add_word_lh(conn, 0x8006);	/* search attributes */

	/* source Handle Path */
	ncp_add_byte(conn, old_dir->volNumber);
	ncp_add_dword_lh(conn, old_dir->DosDirNum);
	ncp_add_byte(conn, 1);
	ncp_add_byte(conn, 1);	/* 1 source component */

	/* dest Handle Path */
	ncp_add_byte(conn, new_dir->volNumber);
	ncp_add_dword_lh(conn, new_dir->DosDirNum);
	ncp_add_byte(conn, 1);
	ncp_add_byte(conn, 1);	/* 1 destination component */

	/* source path string */
	ncp_add_pstring(conn, old_name);
	/* dest path string */
	ncp_add_pstring(conn, new_name);
	/* fn: 87 , subfn: 4 */
	result = ncp_request(conn, 87);
	ncp_unlock_conn(conn);
	return result;
}

static int
ncp_do_read(struct ncp_conn *conn, const char *file_id,
	    u_int32_t offset, u_int16_t to_read,
	    char *target, int *bytes_read)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 0);
	ncp_add_mem(conn, file_id, 6);
	ncp_add_dword_hl(conn, offset);
	ncp_add_word_hl(conn, to_read);

	if ((result = ncp_request(conn, 72)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*bytes_read = ncp_reply_word_hl(conn, 0);

	memcpy(target, ncp_reply_data(conn, 2), *bytes_read);

	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_read(struct ncp_conn *conn, const char *file_id,
	 off_t offset, size_t count, char *target)
{
	const int bufsize = conn->i.buffer_size;
	size_t already_read = 0;

	while (already_read < count)
	{
		int read_this_time;
		int to_read = min(bufsize - (offset % bufsize),
				  count - already_read);

		if (ncp_do_read(conn, file_id, offset, to_read,
				target, &read_this_time) != 0)
		{
			return -1;
		}
		offset += read_this_time;
		target += read_this_time;
		already_read += read_this_time;

		if (read_this_time < to_read)
		{
			break;
		}
	}
	return already_read;
}

static int
ncp_do_write(struct ncp_conn *conn, const char *file_id,
	     u_int32_t offset, u_int16_t to_write,
	     const char *source, int *bytes_written)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 0);
	ncp_add_mem(conn, file_id, 6);
	ncp_add_dword_hl(conn, offset);
	ncp_add_word_hl(conn, to_write);
	ncp_add_mem(conn, source, to_write);

	if ((result = ncp_request(conn, 73)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*bytes_written = to_write;

	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_write(struct ncp_conn *conn, const char *file_id,
	  off_t offset, size_t count, const char *source)
{
	const int bufsize = conn->i.buffer_size;
	size_t already_written = 0;

	while (already_written < count)
	{
		int written_this_time;
		int to_write = min(bufsize - (offset % bufsize),
				   count - already_written);

		if (ncp_do_write(conn, file_id, offset, to_write,
				 source, &written_this_time) != 0)
		{
			return -1;
		}
		offset += written_this_time;
		source += written_this_time;
		already_written += written_this_time;

		if (written_this_time < to_write)
		{
			break;
		}
	}
	return already_written;
}

long
ncp_copy_file(struct ncp_conn *conn,
	      const char source_file[6],
	      const char target_file[6],
	      u_int32_t source_offset,
	      u_int32_t target_offset,
	      u_int32_t count,
	      u_int32_t * copied_count)
{
	long result;

	ncp_init_request(conn);

	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_mem(conn, source_file, 6);
	ncp_add_mem(conn, target_file, 6);
	ncp_add_dword_hl(conn, source_offset);
	ncp_add_dword_hl(conn, target_offset);
	ncp_add_dword_hl(conn, count);

	if ((result = ncp_request(conn, 74)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	*copied_count = ncp_reply_dword_hl(conn, 0);
	ncp_unlock_conn(conn);
	return 0;
}

long
ncp_dealloc_dir_handle(struct ncp_conn *conn, u_int8_t dir_handle)
{
	long result;

	ncp_init_request_s(conn, 20);
	ncp_add_byte(conn, dir_handle);

	result = ncp_request(conn, 22);
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_alloc_short_dir_handle2(struct ncp_conn *conn,
			    u_int8_t ns,
			    struct nw_info_struct *dir,
			    word alloc_mode,
			    byte * target)
{
	NWDIR_HANDLE dh;
	NWCCODE err;
	
	err = ncp_ns_alloc_short_dir_handle(conn, ns,
		NCP_DIRSTYLE_DIRBASE, dir->volNumber,
		dir->DosDirNum, NULL, 0,
		alloc_mode, &dh, NULL);
	if (!err) {
		if (target)
			*target = dh;
	}
	return err;
}

long
ncp_alloc_short_dir_handle(struct ncp_conn *conn,
			   struct nw_info_struct *dir,
			   u_int16_t alloc_mode,
			   u_int8_t* target) {
	return ncp_alloc_short_dir_handle2(conn, NW_NS_DOS, dir, alloc_mode, target);
}

long
ncp_ns_scan_salvageable_file(struct ncp_conn* conn, u_int8_t src_ns,
			     int dirstyle, 
			     u_int8_t vol_num, u_int32_t dir_base,
			     const unsigned char *encpath, int pathlen,
			     struct ncp_deleted_file* finfo,
			     char* name, int maxnamelen)
{
	long result;
	u_int8_t namelen;

	ncp_init_request(conn);
	ncp_add_byte(conn, 0x10);
	ncp_add_byte(conn, src_ns);
	ncp_add_byte(conn, 0);
	ncp_add_dword_lh(conn, RIM_NAME);
	ncp_add_dword_lh(conn, finfo->seq);
	ncp_add_handle_path2(conn, vol_num, dir_base, dirstyle, encpath, pathlen);
	result = ncp_request(conn, 0x57);
	if (result) {
		ncp_unlock_conn(conn);
		return result;
	}
	/* reply format:                        * == returned by RIM_NAME
		+00 u_int32_t lh	next sequence   *
		+04 u_int16_t		deletion time   *
		+06 u_int16_t		deletion date   *
                +08 u_int32_t lh	deletor ID      *
                +0C u_int32_t lh	volume          *
		+10 u_int32_t lh	directory base  *
		+14 u_int32_t		?
		+18 u_int32_t lh	attributes
		+1C u_int16_t hl	flags ?
		+1E u_int32_t lh	size
		+22 u_int32_t		?
		+26 u_int16_t		?
		+28 u_int16_t		creation time
		+2A u_int16_t		creation date
		+2C u_int32_t lh	creator ID
		+30 u_int16_t		modify time
		+32 u_int16_t		modify date
		+34 u_int32_t lh	modifier ID
		+38 u_int16_t		last access date
		+3A u_int16_t		last archive time
		+3C u_int16_t		last archive date
		+3E u_int32_t lh	last archiver ID
		+42 u_int16_t		inherited right mask
		+44 u_int8_t[0x18]	?
		+5C u_int32_t lh	owning namespace
		+60 u_int8_t[var]	name, length preceeded  *
	*/

	if (conn->ncp_reply_size < 0x61) {
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	finfo->seq = ncp_reply_dword_lh(conn, 0x00);
	finfo->vol = ncp_reply_dword_lh(conn, 0x0C);
	finfo->base = ncp_reply_dword_lh(conn, 0x10);
	if (name) {
		namelen = ncp_reply_byte(conn, 0x60);
		if (namelen >= maxnamelen) {
			result = ENAMETOOLONG;
			namelen = maxnamelen-1;
		}
		memcpy(name, ncp_reply_data(conn, 0x61), namelen);
		name[namelen] = 0;
	}
	ncp_unlock_conn(conn);
	return result;
}

long
ncp_ns_purge_file(struct ncp_conn* conn,
	          const struct ncp_deleted_file* finfo)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 18);
	ncp_add_byte(conn, NW_NS_DOS);
	ncp_add_byte(conn, 0);		/* reserved? */
	ncp_add_dword_lh(conn, finfo->seq);
	ncp_add_dword_lh(conn, finfo->vol);
	ncp_add_dword_lh(conn, finfo->base);
	/* fn: 87 , subfn: 18 */
	result = ncp_request(conn, 87);
	ncp_unlock_conn(conn);
	return result;
}

#endif	/* not __MAKE_SULIB__ */

struct ncpi_gfn_cookies {
		int	flag;
		int32_t	cookie1;
		int32_t	cookie2;
			};
static long
ncp_ns_get_full_name_int(struct ncp_conn* conn, u_int8_t src_ns, u_int8_t dst_ns,
		int dirstyle, u_int8_t vol_num, u_int32_t dir_base,
		const unsigned char* encpath, size_t pathlen,
		struct ncpi_gfn_cookies* cookies,
		unsigned char* name, size_t maxnamelen, unsigned char** begin)
{
	long result;
	size_t comps;
	unsigned char* putpos;
	unsigned char* getpos;
	unsigned char* getend;

	ncp_init_request(conn);
	ncp_add_byte(conn, 0x1C);
	ncp_add_byte(conn, src_ns);
	ncp_add_byte(conn, dst_ns);
	ncp_add_word_lh(conn, cookies->flag);
	ncp_add_dword_lh(conn, cookies->cookie1);
	ncp_add_dword_lh(conn, cookies->cookie2);
	ncp_add_handle_path2(conn, vol_num, dir_base, dirstyle, encpath, pathlen);
	result = ncp_request(conn, 0x57);
	if (result) {
		ncp_unlock_conn(conn);
		return result;
	}
	if (conn->ncp_reply_size < 14) {
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	cookies->flag = ncp_reply_word_lh(conn, 0);
	cookies->cookie1 = ncp_reply_dword_lh(conn, 2);
	cookies->cookie2 = ncp_reply_dword_lh(conn, 6);
	comps = ncp_reply_word_lh(conn, 12);
	getpos = ncp_reply_data(conn, 14);
	getend = getpos + ncp_reply_word_lh(conn, 10);
	putpos = name + maxnamelen;
	while (comps--) {
		size_t partl;

		if (getpos >= getend) {
			ncp_unlock_conn(conn);
			return NWE_INVALID_NCP_PACKET_LENGTH;
		}
		partl = *getpos++;
		if (getpos + partl > getend) {
			ncp_unlock_conn(conn);
			return NWE_INVALID_NCP_PACKET_LENGTH;
		}
		putpos -= partl+1;
		if (putpos < name) {
			ncp_unlock_conn(conn);
			return ENAMETOOLONG;
		}
		memcpy(putpos + 1, getpos, partl);
		*putpos = partl;
		getpos += partl;
	}
	ncp_unlock_conn(conn);
	*begin = putpos;
	return 0;
}

static long
ncp_ns_NW_to_path(char* name, size_t maxnamelen,
		  const unsigned char* encpath, const unsigned char* encend)
{
	char* nameend = name + maxnamelen;
	int pos = 0;

	while (encpath < encend) {
		int namel;

		if (pos >= 2) {
			if (name >= nameend) return ENAMETOOLONG;
			*name++ = '/';
		}	
		namel = *encpath++;
		if (encpath + namel > encend) {
			return NWE_INVALID_NCP_PACKET_LENGTH;
		}
		if (name + namel >= nameend) {
			return ENAMETOOLONG;
		}
		memcpy(name, encpath, namel);
		encpath += namel;
		name += namel;
		if (pos == 0) {
			if (name >= nameend) return ENAMETOOLONG;
			*name++ = ':';
		}
		pos++;
	}
	if (name >= nameend) return ENAMETOOLONG;
	*name = 0;
	return 0;
}

long
ncp_ns_get_full_name(struct ncp_conn* conn, u_int8_t src_ns, u_int8_t dst_ns, 
		     int dirstyle, u_int8_t vol_num, u_int32_t dir_base, 
		     const unsigned char* encpath, size_t pathlen,
		     char* name, size_t maxnamelen)
{
	unsigned char space[1024];
	struct ncpi_gfn_cookies cookie = { 0, -1, -1};
	size_t len = sizeof(space);
	long results;
	NWCCODE err;
	u_int16_t vers;

	err = NWGetFileServerVersion(conn, &vers);
	if (vers < 0x040B) {
		/* pre-NW4.11 */
		do {
			unsigned char* npos;

			results = ncp_ns_get_full_name_int(conn, src_ns, dst_ns,
					dirstyle, vol_num, dir_base, 
					encpath, pathlen,
					&cookie, space, len, &npos);
			if (results) return results;
			len = npos-space;
		} while (cookie.cookie2 != -1);
	} else {
		/* NW4.11 and later */
		struct nw_info_struct2 info;
		u_int32_t dent;

		err = ncp_ns_obtain_entry_info(conn, src_ns, 0x8006,
			1, vol_num, dir_base, encpath, pathlen,
			dst_ns, 
			RIM_NAME|RIM_ATTRIBUTES|RIM_DIRECTORY|
			RIM_PARENT_BASE_ID|RIM_COMPRESSED_INFO,
			&info, sizeof(info));
		if (err)
			return err;
		dent = info.Directory.dirEntNum;
		if (!(info.Attributes.Attributes & aDIR)) {
			len = len - info.Name.NameLength - 1;
			space[len] = info.Name.NameLength;
			memcpy(space+len+1, info.Name.Name, info.Name.NameLength);
			dent = info.ParentBaseID;
		}
		do {
			unsigned char* npos;

			results = ncp_ns_get_full_name_int(conn, dst_ns, dst_ns,
					1, info.Directory.volNumber,
					dent, 
					NULL, 0,
					&cookie, space, len, &npos);
			if (results) return results;
			len = npos-space;
		} while (cookie.cookie2 != -1);
	}
	return ncp_ns_NW_to_path(name, maxnamelen, space+len, space+sizeof(space));
}

int
ncp_path_to_NW_format(const char* path, unsigned char* buff, int buffsize) 
{
	int components = 0;
	unsigned char* pos = buff+1;
	buffsize--;

	if (path != NULL) {
		if (*path == '/') path++;	/* skip optional leading / */
		while (*path) {
			const char *c;
			const char *d;
			int   l;

			c = strchr(path, '/');
			if (!c) c=path+strlen(path);
			l = c-path;
			if (components == 0) {			/* volume */
				d = strchr(path, ':');	/* can be separated by :, / or :/ */
				if (!d) d=path+strlen(path);
				if (d < c) {
					c=d;
					if (c[1]=='/') c++;	/* skip optional / after : */
					l = d-path;
				}
			}
			if (l == 0) 
				return -EINVAL;
			if (l > 255) 
				return -ENAMETOOLONG;
			if ((l != 1)||(*path!='.')) {
				if (buffsize <= l) return -ENOBUFS;
				buffsize -= l+1;
				*pos++ = l;
				memcpy(pos, path, l);
				pos+=l;
				components++;
			}
			path = c;
			if (!*c) break;
			path++;
		}
	}
	*buff = components;
	return pos-buff;
}

long
ncp_obtain_file_or_subdir_info2(struct ncp_conn *conn,
			        u_int8_t source_ns, u_int8_t target_ns,
			        u_int16_t search_attribs, u_int32_t rim,
				int dir_style,
			        u_int8_t vol, u_int32_t dirent, const unsigned char *encpath, 
				int pathlen, struct nw_info_struct *target)
{
	long result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 6);
	ncp_add_byte(conn, source_ns);
	ncp_add_byte(conn, target_ns);
	ncp_add_word_lh(conn, search_attribs);
	ncp_add_dword_lh(conn, rim);
	ncp_add_handle_path2(conn, vol, dirent, dir_style, encpath, pathlen);
	/* fn: 87 , subfn: 6 */
	if ((result = ncp_request(conn, 87)) == 0)
	{
		ncp_extract_file_info(ncp_reply_data(conn, 0), target);
	}
	ncp_unlock_conn(conn);
	return result;
}

/* It MUST be possible to call this with rim==0 && dest==NULL (&& sizedest==0) ! */
static const u_int8_t* ncp_ns_extract_file_info(u_int32_t rim, const u_int8_t* src, size_t srclen,
		struct nw_info_struct2* dest, size_t sizedest) {
	if (rim & RIM_COMPRESSED_INFO) {
		if (rim & RIM_SPACE_ALLOCATED) {
			dest->SpaceAllocated = DVAL_LH(src, 0);
			src += 4;
		}
		if (rim & RIM_ATTRIBUTES) {
			dest->Attributes.Attributes = DVAL_LH(src, 0);
			dest->Attributes.Flags = WVAL_LH(src, 4);
			src += 6;
		}
		if (rim & RIM_DATA_SIZE) {
			dest->DataSize = DVAL_LH(src, 0);
			src += 4;
		}
		if (rim & RIM_TOTAL_SIZE) {
			dest->TotalSize.TotalAllocated = DVAL_LH(src, 0);
			dest->TotalSize.Datastreams = WVAL_LH(src, 4);
			src += 6;
		}
		if (rim & RIM_EXT_ATTR_INFO) {
			dest->ExtAttrInfo.DataSize = DVAL_LH(src, 0);
			dest->ExtAttrInfo.Count = DVAL_LH(src, 4);
			dest->ExtAttrInfo.KeySize = DVAL_LH(src, 8);
			src += 12;
		}
		if (rim & RIM_ARCHIVE) {
			dest->Archive.Time = WVAL_LH(src, 0);
			dest->Archive.Date = WVAL_LH(src, 2);
			dest->Archive.ID = DVAL_HL(src, 4);
			src += 8;
		}
		if (rim & RIM_MODIFY) {
			dest->Modify.Time = WVAL_LH(src, 0);
			dest->Modify.Date = WVAL_LH(src, 2);
			dest->Modify.ID = DVAL_HL(src, 4);
			dest->LastAccess.Date = WVAL_LH(src, 8);
			dest->LastAccess.Time = 0;
			src += 10;
		}
		if (rim & RIM_CREATION) {
			dest->Creation.Time = WVAL_LH(src, 0);
			dest->Creation.Date = WVAL_LH(src, 2);
			dest->Creation.ID = DVAL_HL(src, 4);
			src += 8;
		}
		if (rim & RIM_OWNING_NAMESPACE) {
			dest->OwningNamespace = DVAL_LH(src, 0);
			src += 4;
		}
		if (rim & RIM_DIRECTORY) {
			dest->Directory.dirEntNum = DVAL_LH(src, 0);
			dest->Directory.DosDirNum = DVAL_LH(src, 4);
			dest->Directory.volNumber = DVAL_LH(src, 8);
			src += 12;
		}
		if (rim & RIM_RIGHTS) {
			dest->Rights = WVAL_LH(src, 0);
			src += 2;
		}
		if (rim & RIM_REFERENCE_ID) {
			dest->ReferenceID = WVAL_LH(src, 0);
			src += 2;
		}
		if (rim & RIM_NS_ATTRIBUTES) {
			dest->NSAttributes = DVAL_LH(src, 0);
			src += 4;
		}
		if (rim & RIM_DATASTREAM_SIZES) {
			/* FIXME! */
			src += 8 * DVAL_LH(src, 0) + 4;
		}
		if (rim & RIM_DATASTREAM_LOGICALS) {
			/* FIXME! */
			src += 8 * DVAL_LH(src, 0) + 4;
		}
		if (rim & RIM_UPDATE_TIME) {
			dest->UpdateTime = DVAL_LH(src, 0);
			src += 4;
		}
		if (rim & RIM_DOS_NAME) {
			size_t len = dest->DOSName.NameLength = BVAL(src, 0);
			memcpy(dest->DOSName.Name, src+1, len);
			src += len+1;
		}
		if (rim & RIM_FLUSH_TIME) {
			dest->FlushTime = DVAL_LH(src, 0);
			src += 4;
		}
		if (rim & RIM_PARENT_BASE_ID) {
			dest->ParentBaseID = DVAL_LH(src, 0);
			src += 4;
		}
		if (rim & RIM_MAC_FINDER_INFO) {
			memcpy(dest->MacFinderInfo, src, 32);
			src += 32;
		}
		if (rim & RIM_SIBLING_COUNT) {
			dest->SiblingCount = DVAL_LH(src, 0);
			src += 4;
		}
		if (rim & RIM_EFFECTIVE_RIGHTS) {
			dest->EffectiveRights = DVAL_LH(src, 0);
			src += 4;
		}
		if (rim & RIM_MAC_TIMES) {
			dest->MacTimes.CreateTime = DVAL_LH(src, 0);
			dest->MacTimes.BackupTime = DVAL_LH(src, 4);
			src += 8;
		}
		if (rim & RIM_LAST_ACCESS_TIME) {
			dest->LastAccess.Time = WVAL_LH(src, 0);
			src += 2;
		}
		if (rim & RIM_NAME) {
			size_t len = BVAL(src, 0);
			memcpy(dest->Name.Name, src+1, dest->Name.NameLength = len);
			dest->Name.Name[len] = 0;
			src += 1 + len;
		}
	} else {
#define gpt(Q) ((size_t)&((struct nw_info_struct*)NULL)->Q)
		if (rim & 0x3FFFFFFF) {	/* anything requested... */
			if (srclen < gpt(nameLen))
				return NULL;
			dest->SpaceAllocated = DVAL_LH(src, gpt(spaceAlloc));
			dest->Attributes.Attributes = DVAL_LH(src, gpt(attributes));
			dest->Attributes.Flags = WVAL_LH(src, gpt(flags));
			dest->DataSize = DVAL_LH(src, gpt(dataStreamSize));
			dest->TotalSize.TotalAllocated = DVAL_LH(src, gpt(totalStreamSize));
			dest->TotalSize.Datastreams = WVAL_LH(src, gpt(numberOfStreams));
			dest->Creation.Time = WVAL_LH(src, gpt(creationTime));
			dest->Creation.Date = WVAL_LH(src, gpt(creationDate));
			dest->Creation.ID = DVAL_HL(src, gpt(creatorID));
			dest->Modify.Time = WVAL_LH(src, gpt(modifyTime));
			dest->Modify.Date = WVAL_LH(src, gpt(modifyDate));
			dest->Modify.ID = DVAL_HL(src, gpt(modifierID));
			dest->LastAccess.Date = WVAL_LH(src, gpt(lastAccessDate));
			dest->LastAccess.Time = 0;
			dest->Archive.Time = WVAL_LH(src, gpt(archiveTime));
			dest->Archive.Date = WVAL_LH(src, gpt(archiveDate));
			dest->Archive.ID = DVAL_HL(src, gpt(archiverID));
			dest->Rights = WVAL_LH(src, gpt(inheritedRightsMask));
			dest->Directory.dirEntNum = DVAL_LH(src, gpt(dirEntNum));
			dest->Directory.DosDirNum = DVAL_LH(src, gpt(DosDirNum));
			dest->Directory.volNumber = DVAL_LH(src, gpt(volNumber));
			dest->ExtAttrInfo.DataSize = DVAL_LH(src, gpt(EADataSize));
			dest->ExtAttrInfo.Count = DVAL_LH(src, gpt(EAKeyCount));
			dest->ExtAttrInfo.KeySize = DVAL_LH(src, gpt(EAKeySize));
			dest->OwningNamespace = DVAL_LH(src, gpt(NSCreator));
			if (rim & RIM_NAME) {
				size_t len = BVAL(src, gpt(nameLen));
				if (srclen < gpt(entryName) + len)
					return NULL;
				memcpy(dest->Name.Name, src+gpt(entryName), dest->Name.NameLength = len);
				dest->Name.Name[len] = 0;
				src += len + 1;
			}
		}
		src += gpt(nameLen);
#undef gpt
	}
	return src;
}

NWCCODE
ncp_ns_obtain_entry_info(struct ncp_conn *conn,
			 unsigned int source_ns,
			 unsigned int search_attribs,
			 int dir_style,
			 unsigned int vol, 
			 u_int32_t dirent, 
			 const unsigned char *encpath, size_t pathlen, 
			 unsigned int target_ns,
			 u_int32_t rim,
			 struct nw_info_struct2 *target, size_t sizeoftarget)
{
	NWCCODE result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 6);
	ncp_add_byte(conn, source_ns);
	ncp_add_byte(conn, target_ns);
	ncp_add_word_lh(conn, search_attribs);
	ncp_add_dword_lh(conn, rim);
	ncp_add_handle_path2(conn, vol, dirent, dir_style, encpath, pathlen);
	/* fn: 87 , subfn: 6 */
	if ((result = ncp_request(conn, 87)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	if (!ncp_ns_extract_file_info(rim, ncp_reply_data(conn, 0), conn->ncp_reply_size, target, sizeoftarget))
		result = NWE_INVALID_NCP_PACKET_LENGTH;
	ncp_unlock_conn(conn);
	return result;
}

#ifndef __MAKE_SULIB__
NWCCODE
ncp_ns_open_create_entry(struct ncp_conn *conn,
				/* input */
				unsigned int ns,
				unsigned int search_attributes,
				int dirstyle,
				unsigned int vol,
				u_int32_t dirent,
				const unsigned char* encpath, size_t pathlen,
				/* open specific */
				int datastream,
				int open_create_mode,
				u_int32_t create_attributes,
				u_int16_t desired_access_rights,
				/* what to return */
				u_int32_t rim,
				/* returned */
				struct nw_info_struct2* target, size_t sizeoftarget,
				u_int8_t* oc_action,
				u_int8_t* oc_callback,
				u_int8_t* file_handle)
{
	NWCCODE result;
	u_int32_t fhandle;

	ncp_init_request(conn);
	if (datastream == -1) {
		if (open_create_mode & OC_MODE_CALLBACK) {
			ncp_add_byte(conn, 32);
		} else {
			ncp_add_byte(conn, 1);
		}
		ncp_add_byte(conn, ns);
		ncp_add_byte(conn, open_create_mode);
		ncp_add_word_lh(conn, search_attributes);
	} else {
		if (open_create_mode & OC_MODE_CALLBACK) {
			ncp_add_byte(conn, 33);
		} else {
			ncp_add_byte(conn, 30);
		}
		ncp_add_byte(conn, ns);
		ncp_add_byte(conn, datastream);
		ncp_add_byte(conn, open_create_mode);		/* word_lh? */
		ncp_add_byte(conn, 0);				/* reserved */
		ncp_add_word_lh(conn, search_attributes);	/* dword_lh? */
		ncp_add_word_lh(conn, 0);			/* reserved */
	}
	ncp_add_dword_lh(conn, rim);
	ncp_add_dword_lh(conn, create_attributes);
	ncp_add_word_lh(conn, desired_access_rights);
	ncp_add_handle_path2(conn, vol, dirent, dirstyle, encpath, pathlen);
	/* fn: 87 , subfn: 32/1 or 33/30 */
	if ((result = ncp_request(conn, 87)) != 0)
	{
		ncp_unlock_conn(conn);
		return result;
	}
	if (conn->ncp_reply_size < 6) {
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	fhandle = ncp_reply_dword_lh(conn, 0);
	if (oc_action)
		*oc_action = ncp_reply_byte(conn, 4);
	if (oc_callback)
		*oc_callback = (open_create_mode & OC_MODE_CALLBACK) ? ncp_reply_byte(conn, 5) : 0;
	if (!ncp_ns_extract_file_info(rim, ncp_reply_data(conn, 6), conn->ncp_reply_size - 6, target, sizeoftarget))
		result = NWE_INVALID_NCP_PACKET_LENGTH;
	ncp_unlock_conn(conn);
	if (file_handle)
		ConvertToNWfromDWORD(fhandle, file_handle);
	return result;
}

NWCCODE
ncp_ns_modify_entry_dos_info(struct ncp_conn *conn,
				    /* input */
				    unsigned int ns,
				    unsigned int search_attributes,
				    int dirstyle,
				    unsigned int vol,
				    u_int32_t dirent,
				    const unsigned char* encpath, size_t pathlen,
				    /* what to do with entry */
				    u_int32_t mim,
				    const struct ncp_dos_info* info) 
{
	NWCCODE result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 7);
	ncp_add_byte(conn, ns);
	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_word_lh(conn, search_attributes);
	ncp_add_dword_lh(conn, mim);
	ncp_add_dword_lh(conn, info->Attributes);	/* do it faster?! */
	ncp_add_word_lh(conn, info->Creation.Date);
	ncp_add_word_lh(conn, info->Creation.Time);
	ncp_add_dword_hl(conn, info->Creation.ID);
	ncp_add_word_lh(conn, info->Modify.Date);
	ncp_add_word_lh(conn, info->Modify.Time);
	ncp_add_dword_hl(conn, info->Modify.ID);
	ncp_add_word_lh(conn, info->Archive.Date);
	ncp_add_word_lh(conn, info->Archive.Time);
	ncp_add_dword_hl(conn, info->Archive.ID);
	ncp_add_word_lh(conn, info->LastAccess.Date);
	ncp_add_word_lh(conn, info->Rights.Grant);
	ncp_add_word_lh(conn, info->Rights.Revoke);
	ncp_add_dword_lh(conn, info->MaximumSpace);
	ncp_add_handle_path2(conn, vol, dirent, dirstyle, encpath, pathlen);
	/* fn: 87 , subfn: 7 */
	result = ncp_request(conn, 87);
	ncp_unlock_conn(conn);
	return result;
}

NWCCODE
ncp_ns_obtain_namespace_info_format(struct ncp_conn *conn,
				  /* input */
				  unsigned int vol,
				  /* where to act */
				  unsigned int target_ns,
				  /* return buffer */
				  struct ncp_namespace_format* format, size_t sizeofformat) {
	NWCCODE result;
	
	ncp_init_request(conn);
	ncp_add_byte(conn, 23);
	ncp_add_byte(conn, target_ns);
	ncp_add_byte(conn, vol);
	/* fn: 87 , subfn: 23 */
	result = ncp_request(conn, 87);
	if (!result) {
		if (conn->ncp_reply_size < 146) {
			result = NWE_INVALID_NCP_PACKET_LENGTH;
		} else {
			void* data = ncp_reply_data(conn, 0);
			int i;

			format->Version = 0;
			format->BitMask.fixed = DVAL_LH(data, 0);
			format->BitMask.variable = DVAL_LH(data, 4);
			format->BitMask.huge = DVAL_LH(data, 8);
			format->BitsDefined.fixed = WVAL_LH(data, 12);
			format->BitsDefined.variable = WVAL_LH(data, 14);
			format->BitsDefined.huge = WVAL_LH(data, 16);
			for (i = 0; i < 32; i++)
				format->FieldsLength[i] = DVAL_LH(data, 18+i*4);
		}
	}
	ncp_unlock_conn(conn);
	return result;
}

NWCCODE
ncp_ns_obtain_entry_namespace_info(struct ncp_conn *conn,
					    /* input */
					    unsigned int source_ns,
					    unsigned int vol,
					    u_int32_t dirent,
					    /* where to act */
					    unsigned int target_ns,
					    /* what to return */
					    u_int32_t nsrim,
					    /* return buffer */
					    void* buffer, size_t* len, size_t maxlen) {
	NWCCODE result;
	
	ncp_init_request(conn);
	ncp_add_byte(conn, 19);
	ncp_add_byte(conn, source_ns);
	ncp_add_byte(conn, target_ns);
	ncp_add_byte(conn, 0);		/* ? */
	ncp_add_byte(conn, vol);
	ncp_add_dword_lh(conn, dirent);
	ncp_add_dword_lh(conn, nsrim);
	/* fn: 87 , subfn: 19 */	
	result = ncp_request(conn, 87);
	if (!result) {
		/* do this... */
		if (conn->ncp_reply_size > maxlen)
			result = NWE_BUFFER_OVERFLOW;
		else {
			if (len)
				*len = conn->ncp_reply_size;
			if (buffer)
				memcpy(buffer, ncp_reply_data(conn, 0), conn->ncp_reply_size);
		}
	}
	ncp_unlock_conn(conn);
	return result;
}

NWCCODE
ncp_ns_modify_entry_namespace_info(struct ncp_conn *conn,
					  /* input */
					  unsigned int source_ns,
					  unsigned int vol,
					  u_int32_t dirent,
					  /* where to act */
					  unsigned int target_ns,
					  /* what to set */
					  u_int32_t nsrim,
					  /* data buffer */
					  const void* buffer, size_t buflen) {
	
	NWCCODE result;
	
	ncp_init_request(conn);
	ncp_add_byte(conn, 25);
	ncp_add_byte(conn, source_ns);
	ncp_add_byte(conn, target_ns);
	ncp_add_byte(conn, vol);
	ncp_add_dword_lh(conn, dirent);
	ncp_add_dword_lh(conn, nsrim);
	ncp_add_mem(conn, buffer, buflen);
	/* fn: 87 , subfn: 25 */	
	result = ncp_request(conn, 87);
	ncp_unlock_conn(conn);
	return result;
}
				       
NWCCODE
ncp_ns_get_namespace_info_element(const struct ncp_namespace_format* nsformat,
			       u_int32_t nsrim,
			       const void* buffer,
			       size_t bufferlen,
			       unsigned int itemid,
			       void* item, size_t* itemlen, size_t itemmaxlen) {
	u_int32_t mask;
	size_t pos;
	u_int32_t mask2;
	const size_t* entlen;
	
	if (nsformat->Version != 0)
		return NWE_INVALID_LEVEL;
	if (itemid >= 32)
		return NWE_PARAM_INVALID;
	mask = 1 << itemid;
	if (!(mask & nsrim))
		return NCPLIB_INFORMATION_NOT_KNOWN;
	entlen = nsformat->FieldsLength;
	pos = 0;
	for (mask2 = 1; mask2 < mask; entlen++, mask2 <<= 1) {
		if (mask2 & nsrim) {
			if (nsformat->BitMask.variable & mask2) {
				if (pos >= bufferlen)
					return NWE_BUFFER_INVALID_LEN;
				pos += BVAL(buffer, 0) + 1;
			} else if (nsformat->BitMask.huge & mask2)
				return NCPLIB_NSFORMAT_INVALID;
			else
				pos += *entlen;	/* not defined and fixed */
			if (pos > bufferlen)
				return NWE_BUFFER_INVALID_LEN;
		}
	}
	if (nsformat->BitMask.huge & mask)
		return NCPLIB_NSFORMAT_INVALID;
	else {
		size_t len;
		
		if (nsformat->BitMask.variable & mask) {
			if (pos >= bufferlen)
				return NWE_BUFFER_INVALID_LEN;
			len = BVAL(buffer, 0) + 1;
		} else
			len = *entlen;
		if (pos + len > bufferlen)
			return NWE_BUFFER_INVALID_LEN;
		if (itemmaxlen < len)
			return NWE_BUFFER_OVERFLOW;
		if (itemlen)
			*itemlen = len;
		if (item)
			memcpy(item, ((const u_int8_t*)buffer)+pos, len);
	}
	return 0;
}

NWCCODE
ncp_ns_trustee_add(struct ncp_conn *conn,
		   /* input */
		   unsigned int ns,
		   unsigned int search_attributes,
		   int dirstyle,
		   unsigned int vol,
		   u_int32_t dirent,
		   const unsigned char* encpath, size_t pathlen,
		   /* trustee_add specific */
		   const TRUSTEE_INFO* trustees,
		   unsigned int object_count,
		   u_int16_t rights_mask)
{
	NWCCODE result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 10);	/* subfunction */
	ncp_add_byte(conn, ns);
	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_word_lh(conn, search_attributes);
	ncp_add_word_lh(conn, rights_mask);
	ncp_add_word_lh(conn, object_count);
	ncp_add_handle_path2(conn, vol, dirent, dirstyle, encpath, pathlen);
	if (ncp_add_seek(conn, 16 + 307)) {
		ncp_unlock_conn(conn);
		return NWE_BUFFER_OVERFLOW;
	}
	while (object_count != 0)
	{
		ncp_add_dword_hl(conn, trustees->objectID);
		ncp_add_word_lh(conn, trustees->objectRights);
		object_count -= 1;
		trustees += 1;
	}
	/* fn: 87 , subfn: 10 */
	result = ncp_request(conn, 87);
	ncp_unlock_conn(conn);
	return result;
}

NWCCODE
ncp_ns_trustee_del(struct ncp_conn *conn,
		   /* input */
		   unsigned int ns,
		   int dirstyle,
		   unsigned int vol,
		   u_int32_t dirent,
		   const unsigned char* encpath, size_t pathlen,
		   /* trustee_del specific */
		   const TRUSTEE_INFO* trustees,
		   unsigned int object_count)
{
	NWCCODE result;

	ncp_init_request(conn);
	ncp_add_byte(conn, 11);	/* subfunction */
	ncp_add_byte(conn, ns);
	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_word_lh(conn, object_count);
	ncp_add_handle_path2(conn, vol, dirent, dirstyle, encpath, pathlen);
	if (ncp_add_seek(conn, 12 + 307)) {
		ncp_unlock_conn(conn);
		return NWE_BUFFER_OVERFLOW;
	}
	while (object_count != 0)
	{
		ncp_add_dword_hl(conn, trustees->objectID);
		ncp_add_word_lh(conn, trustees->objectRights);
		object_count -= 1;
		trustees += 1;
	}
	/* fn: 87 , subfn: 11 */
	result = ncp_request(conn, 87);
	ncp_unlock_conn(conn);
	return result;
}

NWCCODE
ncp_ns_trustee_scan(struct ncp_conn *conn,
		    /* input */
		    unsigned int ns,
		    unsigned int search_attributes,
		    int dirstyle,
		    unsigned int vol,
		    u_int32_t dirent,
		    const unsigned char* encpath, size_t pathlen,
		    /* trustee_scan specific */
		    u_int32_t* iter,
		    TRUSTEE_INFO* trustees,
		    unsigned int *object_count)
{
	NWCCODE result;
	unsigned int objcnt;
	int pos;
	
	ncp_init_request(conn);
	ncp_add_byte(conn, 5);	/* subfunction */
	ncp_add_byte(conn, ns);
	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_word_lh(conn, search_attributes);
	ncp_add_dword_lh(conn, *iter);
	ncp_add_handle_path2(conn, vol, dirent, dirstyle, encpath, pathlen);
	/* fn: 87 , subfn: 5 */
	if ((result = ncp_request(conn, 87)) != 0) {
		ncp_unlock_conn(conn);
		return result;
	}
	if (conn->ncp_reply_size < 6) {
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	objcnt = ncp_reply_word_lh(conn, 4);
	if (conn->ncp_reply_size < 6 * objcnt + 6) {
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	*iter = ncp_reply_dword_lh(conn, 0);
	if (objcnt > *object_count) {
		objcnt = *object_count;
		result = NWE_BUFFER_OVERFLOW;
	} else
		*object_count = objcnt;
	pos = 6;
	while (objcnt) {
		trustees->objectID = ncp_reply_dword_hl(conn, pos);
		trustees->objectRights = ncp_reply_word_lh(conn, pos + 4);
		trustees++;
		pos += 6;
		objcnt--;
	}
	ncp_unlock_conn(conn);
	return result;
}

/* obsolete */
long
ncp_add_trustee_set(struct ncp_conn *conn,
		    u_int8_t volume_number, 
		    u_int32_t dir_entry,
		    u_int16_t rights_mask,
		    int object_count, 
		    const struct ncp_trustee_struct *rights)
{
	return ncp_ns_trustee_add(conn, NW_NS_DOS, 0x8006, 1,
		volume_number, dir_entry, NULL, 0, 
		(const TRUSTEE_INFO*)rights, object_count, rights_mask);
}

NWCCODE ncp_ns_alloc_short_dir_handle(struct ncp_conn *conn,
				      /* input */
				      unsigned int ns,
				      int dirstyle,
				      unsigned int vol,
		 		      u_int32_t dirent,
				      const unsigned char* encpath, size_t pathlen,
				      /* alloc short dir handle specific */
				      unsigned int allocate_mode,
				      /* output */
				      NWDIR_HANDLE *dirhandle,
				      NWVOL_NUM *ovol) {
	NWCCODE result;
	
	ncp_init_request(conn);
	ncp_add_byte(conn, 12);	/* subfunction */
	ncp_add_byte(conn, ns);
	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_word_lh(conn, allocate_mode);
	ncp_add_handle_path2(conn, vol, dirent, dirstyle, encpath, pathlen);
	/* fn: 87 , subfn: 12 */
	if ((result = ncp_request(conn, 87)) != 0) {
		ncp_unlock_conn(conn);
		return result;
	}
	/* NCP call returns 6 bytes, but we use first 2 only... */
	if (conn->ncp_reply_size < 2) {
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	if (dirhandle)
		*dirhandle = ncp_reply_byte(conn, 0);
	if (ovol)
		*ovol = ncp_reply_byte(conn, 1);
	ncp_unlock_conn(conn);
	return result;
}

/* volume services */
NWCCODE NWGetNSLoadedList(struct ncp_conn *conn,
			  NWVOL_NUM volume,
			  size_t maxNamespaces,
			  unsigned char* namespaces,
			  size_t *actual) {
	NWCCODE result;
	size_t nses;
	
	ncp_init_request(conn);
	ncp_add_byte(conn, 24);
	ncp_add_word_lh(conn, 0);
	ncp_add_byte(conn, volume);
	/* fn: 87 , subfn: 24 */
	result = ncp_request(conn, 87);
	if (result) {
		ncp_unlock_conn(conn);
		return result;
	}
	if (conn->ncp_reply_size < 2) {
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	nses = ncp_reply_word_lh(conn, 0);
	if (conn->ncp_reply_size < 2 + nses) {
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	if (namespaces) {
		if (nses > maxNamespaces)
			result = NWE_BUFFER_OVERFLOW;
		else
			memcpy(namespaces, ncp_reply_data(conn, 2), nses);
	}
	ncp_unlock_conn(conn);
	if (actual)
		*actual = nses;
	return result;
}

static NWCCODE
ncp_get_mount_volume_list_compat(struct ncp_conn *conn,
			  unsigned int ns,
			  unsigned int flags,
			  unsigned int *volnum,
			  unsigned int *itemcnt,
			  void* buffer, size_t* blen) {
	unsigned int vol;
				  
	while ((vol = (*volnum)++) < 256) {
		unsigned char nspc[256];
		size_t real;
		NWCCODE err;
		
		err = NWGetNSLoadedList(conn, vol, sizeof(nspc), nspc, &real);
		if (!err) {
			if (memchr(nspc, ns, real)) {
				DSET_LH(buffer, 0, vol);
				((unsigned char*)buffer) += 4;
				if (!(flags & 1)) {
					*blen = 4;
					return 0;
				}
				err = ncp_get_volume_name(conn, vol, ((unsigned char*)buffer)+1, 17);
				if (!err) {
					real = strlen(((char*)buffer)+1);
					*(unsigned char*)buffer = real;
					*blen = 4 + 1 + real;
					*itemcnt = 1;
					return 0;
				}
			}
		}
	}
	return NWE_SERVER_FAILURE;	
}

static NWCCODE
ncp_get_mount_volume_list(struct ncp_conn *conn,
			  unsigned int ns,
			  unsigned int flags,
			  unsigned int *volnum,
			  unsigned int *itemcnt,
			  void* buffer, size_t* blen) {
	NWCCODE result;
	unsigned int items;
	size_t slen;
	
	ncp_init_request_s(conn, 52);
	ncp_add_dword_lh(conn, *volnum);
	ncp_add_dword_lh(conn, flags);
	ncp_add_dword_lh(conn, ns);
	result = ncp_request(conn, 22);
	if (result) {
		ncp_unlock_conn(conn);
		return result;
	}
	if (conn->ncp_reply_size < 8) {
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	slen = conn->ncp_reply_size - 8;
	items = ncp_reply_dword_lh(conn, 0);
	if (slen < ((flags & 1)?6:4) * items) {
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	*itemcnt = items;
	*volnum = ncp_reply_dword_lh(conn, 4);
	if (slen > *blen) {
		slen = *blen;
		result = NWE_BUFFER_OVERFLOW;
	} else
		*blen = slen;
	if (buffer)
		memcpy(buffer, ncp_reply_data(conn, 8), slen);
	ncp_unlock_conn(conn);
	return result;
}

/* volume listing functions */

struct ncp_volume_list_handle {
	struct ncp_conn *conn;
	u_int32_t nextvol;
	unsigned int ns;
	unsigned int flags;
	NWCCODE err;
	unsigned int simple;
	unsigned int itemcnt;
	unsigned char* bufpos;
	unsigned char* buffer;
	unsigned char* bufend;
	ncpt_mutex_t mutex;
};
	
NWCCODE
ncp_volume_list_init(struct ncp_conn *conn,
		     unsigned int ns,
		     unsigned int flags,
		     NWVOL_HANDLE* handle) {
	NWCCODE result;
	NWVOL_HANDLE h = (NWVOL_HANDLE)malloc(sizeof(struct ncp_volume_list_handle));
	u_int16_t ver;
	
	if (!h)
		return ENOMEM;
	ncp_conn_store(conn);	/* prevent release connection, but allow disconnect... */
	h->conn = conn;
	h->nextvol = 0;
	h->ns = ns;
	h->flags = flags;
	h->itemcnt = 0;
	h->err = 0;
	result = NWGetFileServerVersion(conn, &ver);
	h->simple = result || (ver < 0x0400);
	ncpt_mutex_init(&h->mutex);
	*handle = h;
	return 0;
}

static NWCCODE
__ncp_volume_list_one(NWVOL_HANDLE h,
		      unsigned char** cptr,
		      unsigned char* end,
		      unsigned int* volnum,
		      char* volume,
		      size_t maxlen) {
	unsigned char* curr = *cptr;
	
	if (curr + 4 > end)
		return NWE_INVALID_NCP_PACKET_LENGTH;
	if (volnum)
		*volnum = DVAL_LH(curr, 0);
	curr += 4;
	if (h->flags & 1) {
		size_t namel;
		
		if (curr >= end)
			return NWE_INVALID_NCP_PACKET_LENGTH;
		namel = *curr++;
		if (curr + namel > end)
			return NWE_INVALID_NCP_PACKET_LENGTH;
		if (namel >= maxlen)
			return NWE_BUFFER_OVERFLOW;
		memcpy(volume, curr, namel);
		volume[namel] = 0;
		curr += namel;
	}
	*cptr = curr;
	return 0;		
}

NWCCODE
ncp_volume_list_next(NWVOL_HANDLE h, 
		     unsigned int* volnum,
		     char* volume, size_t maxlen) {
	NWCCODE result;

	ncpt_mutex_lock(&h->mutex);
	if (!h->itemcnt) {
		unsigned int itemcnt;
		unsigned char buffer[1024];
		size_t blen = sizeof(buffer);
		void* b;
				
		if (h->err) {
			result = h->err;
			goto quit;
		}
		if (h->simple)
			result = ncp_get_mount_volume_list_compat(h->conn,
				h->ns, h->flags & 0x0001,
				&h->nextvol,
				&itemcnt,
				buffer, &blen);
		else
			result = ncp_get_mount_volume_list(h->conn, 
				h->ns, 
				h->flags & 0x0001, 
				&h->nextvol,
				&itemcnt,
				buffer, &blen);
		if (result) {
			h->err = result;
			goto quit;
		}
		if (!itemcnt) {
			result = NWE_SERVER_FAILURE;
			goto quit;
		}
			
		/* let's build buffer */
		b = (void*)malloc(blen);
		if (!b) {
			result = ENOMEM;
			goto quit;
		}
		memcpy(b, buffer, blen);
		h->bufpos = h->buffer = b;
		h->bufend = b + blen;
		h->itemcnt = itemcnt;
		if (!h->nextvol)
			h->err = NWE_SERVER_FAILURE;
	}
	result = __ncp_volume_list_one(h, &h->bufpos, h->bufend, volnum, volume, maxlen);
	if (result)
		goto quit;
	if (!--h->itemcnt)
		free(h->buffer);
quit:;	
	ncpt_mutex_unlock(&h->mutex);
	return result;
}

NWCCODE
ncp_volume_list_end(NWVOL_HANDLE h) {
	ncpt_mutex_lock(&h->mutex);
	if (h->itemcnt)
		free(h->buffer);
	ncp_conn_release(h->conn);
	ncpt_mutex_destroy(&h->mutex);
	free(h);
	return 0;
}

/* directory listing functions */

static NWCCODE
__ncp_ns_search_init(struct ncp_conn* conn,
		     /* input */
		     unsigned int ns,
		     unsigned int dirstyle,
		     unsigned int vol,
		     NWDIR_ENTRY dirent,
		     const unsigned char* encpath, size_t enclen,
		     /* output */
		     struct nw_search_sequence* seq) {
	NWCCODE result;
	
	ncp_init_request(conn);
	ncp_add_byte(conn, 2);	/* subfunction */
	ncp_add_byte(conn, ns);
	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_handle_path2(conn, vol, dirent, dirstyle, encpath, enclen);
	/* fn: 87 , subfn: 2 */
	if ((result = ncp_request(conn, 87)) != 0) {
		ncp_unlock_conn(conn);
		return result;
	}
	if (conn->ncp_reply_size < 9) {
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	if (seq)
		memcpy(seq, ncp_reply_data(conn, 0), 9);
	ncp_unlock_conn(conn);
	return result;
}

static NWCCODE
__ncp_ns_search_next(struct ncp_conn* conn,
		     /* input */
		     unsigned int ns,
		     int datastream,
		     unsigned int search_attributes,
		     /* search next specific */
		     struct nw_search_sequence* seq,
		     u_int32_t rim,
		     const unsigned char* pattern, size_t patlen,
		     /* output */
		     void* buffer, size_t* blen) {
	NWCCODE result;
	size_t ulen;
	
	ncp_init_request(conn);
	ncp_add_byte(conn, 3);	/* subfunction */
	ncp_add_byte(conn, ns);
	ncp_add_byte(conn, datastream);
	ncp_add_word_lh(conn, search_attributes);
	ncp_add_dword_lh(conn, rim);
	ncp_add_mem(conn, seq, 9);
	ncp_add_mem(conn, pattern, patlen);
	/* fn: 87 , subfn: 3 */
	if ((result = ncp_request(conn, 87)) != 0) {
		ncp_unlock_conn(conn);
		return result;
	}
	if (conn->ncp_reply_size < 10) {
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	ulen = conn->ncp_reply_size - 10;
	if (buffer) {
		if (*blen < ulen) {
			ncp_unlock_conn(conn);
			return NWE_BUFFER_OVERFLOW;
		}
		memcpy(buffer, ncp_reply_data(conn, 10), ulen);
	}
	*blen = ulen;
	memcpy(seq, ncp_reply_data(conn, 0), 9);
	ncp_unlock_conn(conn);
	return result;
}		     
		     
static NWCCODE
__ncp_ns_search_next_set(struct ncp_conn* conn,
			 /* input */
			 unsigned int ns,
			 int datastream,
			 unsigned int search_attributes,
			 /* search next specific */
			 struct nw_search_sequence* seq,
			 u_int32_t rim,
			 const unsigned char* pattern, size_t patlen,
			 /* output */
			 unsigned int *itemcnt,
			 void* buffer, size_t* blen, unsigned char* more) {
	NWCCODE result;
	size_t ulen;
	
	ncp_init_request(conn);
	ncp_add_byte(conn, 20);	/* subfunction */
	ncp_add_byte(conn, ns);
	ncp_add_byte(conn, datastream);
	ncp_add_word_lh(conn, search_attributes);
	ncp_add_dword_lh(conn, rim);
	ncp_add_word_lh(conn, *itemcnt);
	ncp_add_mem(conn, seq, 9);
	ncp_add_mem(conn, pattern, patlen);
	/* fn: 87 , subfn: 20 */
	if ((result = ncp_request(conn, 87)) != 0) {
		ncp_unlock_conn(conn);
		return result;
	}
	if (conn->ncp_reply_size < 12) {
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	ulen = conn->ncp_reply_size - 12;
	if (buffer) {
		if (*blen < ulen) {
			ncp_unlock_conn(conn);
			return NWE_BUFFER_OVERFLOW;
		}
		memcpy(buffer, ncp_reply_data(conn, 12), ulen);
	}
	*blen = ulen;
	*itemcnt = ncp_reply_word_lh(conn, 10);
	if (more)
		*more = ncp_reply_byte(conn, 9);
	memcpy(seq, ncp_reply_data(conn, 0), 9);
	ncp_unlock_conn(conn);
	return result;
}		     
		     
struct ncp_directory_list_handle {
	struct ncp_conn *conn;
	ncpt_mutex_t mutex;
	struct nw_search_sequence searchseq;
	NWCCODE err;
	unsigned int ns;
	unsigned int search_attr;
	int datastream;
	u_int32_t rim;
	unsigned int searchset;
	unsigned int itemcnt;
	const unsigned char* bufpos;
	unsigned char buffer[1024];
	const unsigned char* bufend;
	unsigned char noteof;
	size_t patlen;
	unsigned char pattern[1];
};

NWCCODE
ncp_ns_search_init(struct ncp_conn* conn,
		   /* input */
		   unsigned int ns,
		   unsigned int search_attributes,
		   unsigned int dirstyle,
		   unsigned int vol,
		   NWDIR_ENTRY dirent,
		   const unsigned char* encpath, size_t enclen,
		   /* search specific */
		   int datastream,
		   const unsigned char* pattern, size_t patlen,
		   u_int32_t rim,
		   /* handle */
		   NWDIRLIST_HANDLE* rhandle) {
	NWDIRLIST_HANDLE h;
	NWCCODE err;
	struct nw_search_sequence sseq;
	
	err = __ncp_ns_search_init(conn, ns, dirstyle, vol, dirent,
			encpath, enclen, &sseq);
	if (err)
		return err;
	if (!pattern)
		patlen = 0;
	h = (NWDIRLIST_HANDLE)malloc(sizeof(*h) + patlen);
	if (!h)
		return ENOMEM;
	ncp_conn_store(conn);
	ncpt_mutex_init(&h->mutex);
	h->conn = conn;
	h->searchseq = sseq;
	h->err = 0;
	h->ns = ns;
	h->search_attr = search_attributes;
	h->rim = rim;
	h->datastream = datastream;
	h->patlen = patlen + 1;
	h->pattern[0] = patlen;
	h->itemcnt = 0;
	h->searchset = 1;
	h->noteof = 1;
	if (patlen)
		memcpy(h->pattern + 1, pattern, patlen);
	*rhandle = h;
	return 0;
}

NWCCODE
ncp_ns_search_next(NWDIRLIST_HANDLE h,
		   struct nw_info_struct2* target, size_t sizeoftarget) {
	NWCCODE err;
	const unsigned char* p;
	
	ncpt_mutex_lock(&h->mutex);
	if (!h->itemcnt) {
		size_t blen;
		
		if (!h->noteof) {
			err = NWE_SERVER_FAILURE;
			goto q;
		}
		blen = sizeof(h->buffer);
		if (h->searchset) {
			unsigned int itemcnt = 200;
			
			err = __ncp_ns_search_next_set(h->conn, h->ns, h->datastream, h->search_attr,
				&h->searchseq, h->rim, h->pattern, h->patlen,
				&itemcnt, h->buffer, &blen, &h->noteof);
			if (err)
				goto q;
			if (!itemcnt) {
				err = NWE_SERVER_FAILURE;
				goto q;
			}
			h->itemcnt = itemcnt;
		} else {
			h->rim |= RIM_NAME;
			err = __ncp_ns_search_next(h->conn, h->ns, h->datastream, h->search_attr,
				&h->searchseq, h->rim, h->pattern, h->patlen,
				h->buffer, &blen);
			if (err)
				goto q;
			h->itemcnt = 1;
		}
		h->bufpos = h->buffer;
		h->bufend = h->buffer + blen;
	}
	p = ncp_ns_extract_file_info(h->rim, 
			h->bufpos, h->bufend - h->bufpos, target, sizeoftarget);
	if (!p)	{
		err = NWE_INVALID_NCP_PACKET_LENGTH;
		h->itemcnt = 0;
	} else {
		err = 0;
		h->bufpos = p;
		h->itemcnt--;
	}
q:;
	ncpt_mutex_unlock(&h->mutex);
	return err;
}

NWCCODE
ncp_ns_search_end(NWDIRLIST_HANDLE h) {
	ncpt_mutex_lock(&h->mutex);
	ncp_conn_release(h->conn);
	ncpt_mutex_destroy(&h->mutex);
	free(h);
	return 0;
}

NWCCODE
ncp_get_file_size(
		NWCONN_HANDLE conn,
		const u_int8_t* fileHandle,
		ncp_off64_t* fileSize) {
	NWCCODE result;
	
	ncp_init_request(conn);
	ncp_add_byte(conn, 0);	/* reserved */
	ncp_add_mem(conn, fileHandle, 6);
	/* fn: 71 */
	result = ncp_request(conn, 71);
	if (result) {
		ncp_unlock_conn(conn);
		return result;
	}
	if (conn->ncp_reply_size < 4) {
		ncp_unlock_conn(conn);
		return NWE_INVALID_NCP_PACKET_LENGTH;
	}
	if (fileSize)
		*fileSize = ncp_reply_dword_hl(conn, 0);
	ncp_unlock_conn(conn);
	return result;
}

char* ncp_perms_to_str(char r[11], const u_int16_t rights)
{
        r[0] = '[';
        r[1] = ((rights & NCP_PERM_SUPER) != 0) ? 'S' : ' ';
        r[2] = ((rights & NCP_PERM_READ) != 0) ? 'R' : ' ';
        r[3] = ((rights & NCP_PERM_WRITE) != 0) ? 'W' : ' ';
        r[4] = ((rights & NCP_PERM_CREATE) != 0) ? 'C' : ' ';
        r[5] = ((rights & NCP_PERM_DELETE) != 0) ? 'E' : ' ';
        r[6] = ((rights & NCP_PERM_MODIFY) != 0) ? 'M' : ' ';
        r[7] = ((rights & NCP_PERM_SEARCH) != 0) ? 'F' : ' ';
        r[8] = ((rights & NCP_PERM_OWNER) != 0) ? 'A' : ' ';
        r[9] = ']';
        r[10] = '\0';
	return r;
}

/* The following function converts a rights string of format [SRWCEMFA]
   into an integer.  It will tolerate spaces, lower case and repeated 
   letters, even if this takes the length well over 10 characters, but 
   must be terminated with square brackets.  If such a string containing 
   spaces is given as a command line option it will have to be quoted. */

int ncp_str_to_perms(const char *r, u_int16_t *rights)
{
	u_int16_t result = 0;

	if (*r == '[') {
		do {
			++r;
			switch (*r) {
				case ' ' : 
				case ']' :
					break;
				case 's' :
				case 'S' : 
					result |= NCP_PERM_SUPER; break;
				case 'r' :
				case 'R' : 
					result |= NCP_PERM_READ; break;
				case 'w' :
				case 'W' : 
					result |= NCP_PERM_WRITE; break;
				case 'c' :
				case 'C' : 
					result |= NCP_PERM_CREATE; break;
				case 'e' :
				case 'E' : 
					result |= NCP_PERM_DELETE; break;
				case 'm' :
				case 'M' : 
					result |= NCP_PERM_MODIFY; break;
				case 'f' :
				case 'F' : 
					result |= NCP_PERM_SEARCH; break;
				case 'a' :
				case 'A' : 
					result |= NCP_PERM_OWNER; break;
				default :
					return -1;
			}
		} while (*r != ']');
		/* Now to be generous and ignore trailing spaces */
		do { ++r; } while (*r == ' ');
		if (*r == '\0') { 
			*rights = result; 
			return 0;
		}
	}
	return -1;
}
#endif
