/* source.c
 * - Source functions
 * Copyright (c) 1999 Jack Moffitt, Barath Raghavan, and Alexander Havng
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#ifdef _WIN32
#include <win32config.h>
#else
#include <config.h>
#endif
#endif

#include "definitions.h"
#include <stdio.h>
#include "definitions.h"

#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif

#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <sys/types.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>

#ifndef _WIN32
#include <sys/socket.h> 
#include <sys/wait.h>
#include <netinet/in.h>
#else
#include <io.h>
#include <winsock.h>
#define write _write
#define read _read
#define close _close
#endif

#include "avl.h"
#include "threads.h"
#include "icetypes.h"
#include "icecast.h"
#include "utility.h"
#include "ice_string.h"
#include "source.h"
#include "sock.h"
#include "log.h"
#include "connection.h"
#include "avl_functions.h"
#include "main.h"
#include "timer.h"
#include "alias.h"
#include "relay.h"
#include "client.h"
#include "commands.h"
#include "restrict.h"
#include "memory.h"
#include "admin.h"
#include "http.h"
#include "pool.h"

/* in microseconds */
#define READ_RETRY_DELAY 400
#define READ_TIMEOUT 500000

extern int running;
extern server_info_t info;

void source_login(connection_t *con, char *expr)
{
	char line[BUFSIZE], command[BUFSIZE], arg[BUFSIZE];
	char pass[BUFSIZE] ="";
	int go_on = 2;
	int connected = 1;
	int password_accepted = 0;
	int need_lock = 1;
	source_t *source;
	char *res;

	/* Normally, the type here is set to unknown,
	   and it's this function's job to set it
	   to encoder_e. But if we're called from
	   relay_pull, then that function has already
	   set the connection type to source_e, and
	   the source type to puller_e. In that case
	   we should neither change that, nor 
	   mess with put_source (); 
	   Also, a relay pull connection will not print
	   the SOURCE password /mount line, so we will only
	   get _one_ \n\n sequence, that's why we set go_on 
	   to 1. To get the mountname, the relay uses
	   the x-audiocast-mount. */
	if (con->type == unknown_connection_e)
	{
		if (info.throttle_on)
		{
			sock_write (con->sock, "HTTP/1.0 406 Not Acceptable (Bandwidth usage too high)");
			kick_connection (con, "Bandwidth usage too high (throttling)");
			return;
		}
		put_source(con);
		con->food.source->type = encoder_e;
		xa_debug (2, "DEBUG: Encoder logging in with [%s]", expr);

	} else if (con->food.source->type == puller_e)
	{
		xa_debug (2, "DEBUG: Puller loggin in with [%s]", expr);
		go_on = 1; /* Dealing with relay pull (all headers are already present)*/
		need_lock = 0;
	} else {
		xa_debug (2,"DEBUG: Icy encoder logging in with [%s]", expr);
	}
	
#ifdef HAVE_LIBWRAP
	if ((con->food.source->type == encoder_e) && (!sock_check_libwrap (con->sock, source_e)))
	{
		write_http_code_page (con, 403, "Forbidden");
		kick_connection (con, "Access Denied (tcp wrappers (source connection))");
		return;
	}
#endif
	if ((con->food.source->type == encoder_e) && !allowed (con, source_e))
	{
		write_http_code_page (con, 403, "Forbidden");
		kick_connection (con, "Access Denied (internal acl list (source connection))");
		return;
	}
	
	source = con->food.source;

	do {
		command[0] = '\0';
		arg[0] = '\0';

		if (splitc(line, expr, '\n') == NULL) {
			strncpy (line, expr, BUFSIZE);
			go_on = go_on - 1;

			if (source->type == puller_e)
				go_on = 0;

			else if ((go_on > 0) && (!source->protocol == icy_e && source->type == encoder_e))
			{
				sock_read_lines_np (con->sock, expr, BUFSIZE);
			}
		}
		
		/* The delimiter on the first line is ' ', but on the following lines
		   we want the part after the colon */
		if (go_on == 2)
			res = splitc (command, line, ' ');
		else
			res = splitc (command, line, ':');
		
		if (!res)
		{
			strncpy (command, line, BUFSIZE);
			arg[0] = '\0';
		} else {
			strncpy (arg, line, BUFSIZE);
		}
		
		if (line[0])
			xa_debug (2, "DEBUG: Source line: [%s] [%s]", command, arg);

		/* The first line looks like this */
		if (ice_strncmp(command, "SOURCE", 6) == 0)
		{
			if (splitc(pass, arg, ' ') == NULL) {
				sock_write_line (con->sock, "ERROR - Missing Mountpoint\r\n");
				kick_connection (con, "No Mountpoint supplied");
				connected = 0;
				return;
			}
			
			if (!password_match(info.encoder_pass, pass)) {
				sock_write_line (con->sock, "ERROR - Bad Password\r\n");
				kick_connection (con, "Bad Password");
				return;
			} else
				password_accepted = 1;
			
			if (!source->audiocast.mount)
				source->audiocast.mount = my_strdup(arg);

			{ /* Make sure it starts with / */
				char slash[BUFSIZE];
				if (source->audiocast.mount[0] != '/')
				{
					snprintf(slash, BUFSIZE, "/%s", source->audiocast.mount);
					nfree (source->audiocast.mount);
					source->audiocast.mount = my_strdup (slash);
				}
			}
			
			if (mount_exists (source->audiocast.mount) || (source->audiocast.mount[0] == '\0')) {
				sock_write_line (con->sock, "ERROR - Mount Point Taken or Invalid\r\n");
				kick_connection (con, "Invalid Mount Point");
				return;
			}

			if (source->protocol != icy_e)
				sock_write_line (con->sock, "OK");
			else if (source->type == encoder_e)
				go_on = 1;
		}
		
		else if (ice_strncmp(command, "icy-name", 8) == 0)
		{
			if (!source->audiocast.name)
				source->audiocast.name = ice_sprintf (info.nametemplate, arg);
			if (!source->audiocast.description)
				source->audiocast.description = nstrdup ("Icy protocol in use");
		}

		else if (ice_strncmp(command, "icy-genre", 9) == 0)
		{
			if (!source->audiocast.genre)
				source->audiocast.genre = my_strdup (arg);
		}

		else if (ice_strncmp(command, "icy-url", 7) == 0)
		{
			if (!source->audiocast.url)
				source->audiocast.url = my_strdup (arg);
		}

		else if (ice_strncmp(command, "icy-pub", 7) == 0)
			source->audiocast.public = atoi (arg);
		else if (ice_strncmp(command, "icy-br", 6) == 0)
			source->audiocast.bitrate = atoi (arg);

		else if (ice_strncmp(command, "x-audiocast-name", 16) == 0)
		{
			if (!source->audiocast.name)
				source->audiocast.name = ice_sprintf (info.nametemplate, arg);
		}

		else if (ice_strncmp(command, "x-audiocast-description", 23) == 0)
		{
			if (!source->audiocast.description)
				source->audiocast.description = ice_sprintf (info.descriptiontemplate, arg);
		}

		else if (ice_strncmp(command, "x-audiocast-url", 15) == 0)
		{
			if (!source->audiocast.url)
				source->audiocast.url = my_strdup (arg);
		}

		else if (ice_strncmp(command, "x-audiocast-genre", 17) == 0)
		{
			if (!source->audiocast.genre)
				source->audiocast.genre = my_strdup (arg);
		}

		else if (ice_strncmp(command, "x-audiocast-bitrate", 19) == 0)
			source->audiocast.bitrate = atoi (arg);
		else if (ice_strncmp(command, "x-audiocast-public", 18) == 0)
			source->audiocast.public = atoi (arg);
		else if (ice_strncmp(command, "x-audiocast-dumpfile", 20) == 0)
		{
			if (!source->dumpfile)
				source->dumpfile = my_strdup (arg);
		}

		else if (con->food.source->type != puller_e && ice_strncmp(command, "x-audiocast-mount", 17) == 0)
		{
			source->audiocast.mount = my_strdup (arg);
			if (arg[0] != '/')
			{
				char slash[BUFSIZE];
				snprintf (slash, BUFSIZE, "/%s", source->audiocast.mount);
				nfree (source->audiocast.mount);
				source->audiocast.mount = my_strdup (slash);
			}
		}

		else if (ice_strncmp(command, "x-audiocast-directory", 21) == 0)
		{
			char host[BUFSIZE];
			relay_id_t *rid;

			if (avl_count (con->food.source->relay_tree) < info.max_directories)
			{
				if ((splitc (host, arg, ':') == NULL))
				{
					sock_write_line (con->sock, "ERROR - Invalid directory server specification");
					kick_connection (con, "Invalid directory server specification");
					return;
				}
				
				rid = (relay_id_t *) nmalloc (sizeof (relay_id_t));
				xa_debug (1, "DEBUG: Adding directory-server: [%s] with id [%s] for %d", host, arg, con->id);
				rid->host = my_strdup (host);
				rid->id = atoi (arg);
				avl_insert (con->food.source->relay_tree, rid);
			}
		}

		else if (ice_strncmp(command, "x-audiocast-mimetype", 20) == 0)
		{
			if (!source->audiocast.streammimetype)
				source->audiocast.streammimetype = my_strdup (arg);
		}
		
		else if (ice_strncmp (command, "x-audiocast-contentid", 21) == 0)
		{
			if (source->audiocast.contentid == NULL)
				source->audiocast.contentid = my_strdup (arg);
		}

	} while ((go_on > 0) && connected);
	
	/* Relay did not tell mountpoint */
	if (connected && source->audiocast.mount == NULL)
	{
		sock_write_line (con->sock, "ERROR - Mount Point Taken or Invalid\r\n");
		kick_connection (con, "Invalid Mount Point");
		return;
	}
		
	/* Make sure we got everything */
	if (connected)
	{
		if (!source->audiocast.name)
			source->audiocast.name = nstrdup ("");
		if (!source->audiocast.url)
			source->audiocast.url = nstrdup ("");
		if (!source->audiocast.genre)
			source->audiocast.genre = nstrdup ("");
		if (!source->audiocast.description)
			source->audiocast.description = nstrdup ("");
		if (source->audiocast.bitrate == -1)
			source->audiocast.bitrate = 128;
		if (source->audiocast.public == -1)
			source->audiocast.public = 0;
	}
	
	if (connected && !password_accepted && con->food.source->type != puller_e)
	{
		sock_write_line (con->sock, "Missing Password\r\n");
		kick_connection(con, "Missing Password");
		return;
	}
	    
	if (connected) {

		if (info.num_sources + 1> info.max_sources)
		{
			sock_write_line (con->sock, "ERROR - Too many sources\r\n");
			kick_connection (con, "Server Full (too many streams)");
			return;
		}

		add_source ();

		if (!con->food.source->type == puller_e)
			sock_write_line (con->sock, "OK");

		source->connected = SOURCE_CONNECTED;
		source->info.streamtitle = nstrdup (info.streamtitle);
		source->info.streamurl = nstrdup (info.streamurl);

		write_log (LOG_DEFAULT, "Accepted encoder on mountpoint %s from %s. %d sources connected", 
			   source->audiocast.mount, con_host (con), info.num_sources);

		thread_mutex_lock(&info.source_mutex);
		avl_insert(info.sources, con);
		thread_mutex_unlock(&info.source_mutex);
		
		{
			connection_t *clicon, *scon = get_pending_mount (source->audiocast.mount);
			
			if (scon)
			{
				write_log (LOG_DEFAULT, "Assigning listeners from pending source %d", scon->id);
				while ((clicon = avl_get_any_node (scon->food.source->clients)))
					move_to (clicon, con);
				scon->food.source->connected = SOURCE_KILLED;
			}
		}
			
			
		if (con->food.source->type == puller_e) {
			thread_create("Source Thread", source_func, (void *)con);
			return;
		} else {
			/* change thread name */
			thread_rename("Source Thread");

			source_func(con);
		}
	}

	write_log (LOG_DEFAULT, "WARNING: Thread exiting in source_login(), this should not happen");
	thread_exit(0);
}

/* This function is started as a new thread for every connected
   and accepted source.
   A lot can be said about termination of this thread.
   Either it gets killed by another thread, using the kick_connection (thiscon,..), 
   which should set the connected value to SOURCE_KILLED, and let this thread
   exit on it's on with the close_connection at the end.
   Or it kills itself, when the encoder dies, and then it should call kick_connection (thiscon,..),
   on itself, setting the value of connected to SOURCE_KILLED, and exit through close_connection () */
void *
source_func(void *conarg)
{
	source_t *source;
	avl_traverser trav = {0};
	connection_t *clicon, *con = (connection_t *)conarg;
	mythread_t *mt;
	int i;

	source = con->food.source;
	con->food.source->thread = thread_self();
	thread_init();

	mt = thread_get_mythread ();

	sock_set_blocking(con->sock, SOCK_NONBLOCK);

	if (source->dumpfile) {
		source->dumpfd = open_for_writing (source->dumpfile);
		if (source->dumpfd == -1) {
			write_log (LOG_DEFAULT, "WARNING: Could not open dumpfile %s for source %d [%d]", source->dumpfile, con->id, errno);
			nfree(source->dumpfile);
			source->dumpfile = NULL;
		}
	}

	while (thread_alive (mt) && ((source->connected == SOURCE_CONNECTED) || (source->connected == SOURCE_PAUSED)))
	{
		source_get_new_clients (source);

		add_chunk(con);
		
		for (i = 0; i < 10; i++) {
			
			if (source->connected != SOURCE_CONNECTED)
				break;

			thread_mutex_lock(&source->mutex);
			
			zero_trav (&trav);
			
			while ((clicon = avl_traverse(source->clients, &trav)) != NULL) {
			  
				if (source->connected == SOURCE_KILLED || source->connected == SOURCE_PAUSED)
					break;
				
				source_write_to_client (source, clicon);
				
			}
			
			thread_mutex_unlock(&source->mutex);

			if (mt->ping == 1)
				mt->ping = 0;
		}

		thread_mutex_lock (&info.double_mutex);
		thread_mutex_lock (&source->mutex);
		kick_dead_clients (source);
		thread_mutex_unlock (&source->mutex);
		thread_mutex_unlock(&info.double_mutex);
	}
	
	thread_mutex_lock (&info.double_mutex);
	
	thread_mutex_lock (&info.source_mutex);
	thread_mutex_lock (&source->mutex);

	close_connection (con, &info);

	thread_mutex_unlock (&info.source_mutex);
	thread_mutex_unlock (&info.double_mutex);

	thread_exit (0);
	return NULL;
}

source_t *
create_source()
{
	source_t *source = (source_t *)nmalloc(sizeof(source_t));
	memset(source,0,sizeof(source_t));
	source->type = unknown_source_e;
	return source;
}

void 
put_source(connection_t *con)
{
	register int i;
	socklen_t sin_len;
	source_t *source = create_source();

	con->food.source = source;
	zero_stats (&source->stats);
	source->connected = SOURCE_UNUSED;
	source->type = unknown_source_e;
	thread_create_mutex(&source->mutex);
	source->dumpfile = NULL;
	source->dumpfd = -1;
	source->audiocast.name = NULL;
	source->audiocast.genre = NULL;
	source->audiocast.bitrate = -1;
	source->audiocast.url = NULL;
	source->audiocast.mount = NULL;
	source->audiocast.description = NULL;
	source->audiocast.streammimetype = NULL;
	source->audiocast.public = -1;
	source->protocol = xaudiocast_e;
	source->cid = 0;
	source->clients = avl_create (compare_connection, &info);
	source->relay_tree = avl_create (compare_relay_ids, &info);
	source->num_clients = 0;
	source->priority = 0;
	zero_song_info (&source->info);

	for (i = 0; i < CHUNKLEN; i++)
	{
		source->chunk[i].clients_left = 0;
		source->chunk[i].len = 0;
		source->chunk[i].metalen = 0;
	}
	sin_len = 1;

	con->type = source_e;
}

void
add_source ()
{
	info.num_sources++;
	info.hourly_stats.source_connections++;
}

void
del_source ()
{
	info.num_sources--;
}

/* Must have mount, source and double mutex to call this */
connection_t *
find_mount_with_req (request_t *req)
{
	char pathbuf[BUFSIZE] = "";
	char tempbuf[256] = "";

	connection_t *con;
	request_t *alias;
	request_t search;

	int true = 0;

	avl_traverser trav = {0};
	
	if (!req || !req->path || !req->host)
	{
		write_log (LOG_DEFAULT, "WARNING: find_mount_with_req called with NULL request!");
		return NULL;
	}

	if (ice_strcmp (req->path, "/list.cgi") == 0)
		return (connection_t *) &info;

	xa_debug (1, "DEBUG: Looking for [%s] on host [%s] on port %d", req->path, req->host, req->port);
	xa_debug (1, "DEBUG: Searching local aliases");

	/* First look for aliases */
	alias = get_alias (req);
	
	if (alias && hostname_local (req->host) &&
	    hostname_local (alias->host) && (req->port == alias->port))
	{
		xa_debug (1, "DEBUG: Found alias [%s:%d%s] -> [%s:%d%s]", req->host, req->port, req->path,
			   alias->host, alias->port, alias->path);
		return find_mount_with_req (alias);
	}

	xa_debug (1, "DEBUG: Search local mount points");

	snprintf(pathbuf, BUFSIZE, "%s:%d%s", req->host[0] ? req->host : "localhost", req->port, req->path);

	while ((con = avl_traverse(info.sources, &trav)) != NULL) 
	{
		true = 0; /* Sorry about this.. damn we've got many ugly cases */

		xa_debug(2, "DEBUG: Looking on mount [%s]", con->food.source->audiocast.mount);

		zero_request(&search);

		generate_http_request(con->food.source->audiocast.mount, &search);

		strncpy(tempbuf, con->food.source->audiocast.mount, 252);
		strcat(tempbuf, ".mp3");

		if (search.path[0] && (ice_strcasecmp(search.host, req->host) == 0) && (search.port == req->port) && ((ice_strcmp(search.path, req->path) == 0) || (ice_strcmp(tempbuf, req->path) == 0))) {
			true = 1;
		} else if (con->food.source->audiocast.mount[0] == '/') {	/* regular mount */
			if ((hostname_local(req->host) && ((ice_strcmp(con->food.source->audiocast.mount, req->path) == 0) || (ice_strcmp(tempbuf, req->path) == 0))))
				true = 1;
		} else {
			if ((hostname_local(req->host) && ((ice_strcmp(con->food.source->audiocast.mount, req->path + 1) == 0) || (ice_strcmp(tempbuf, req->path + 1) == 0))))
				true = 1;
		}
		
		if (true) {
			xa_debug(1, "DEBUG: Found local mount for [%s]", req->path);
			return con;
		} 
	}
	
	xa_debug(1, "DEBUG: Searching remote aliases");

	/* Time for on demand relays */
	alias = get_alias (req);

	if (alias)
	{
		connection_t *sourcecon; 
		xa_debug (1, "DEBUG: Found alias [%s:%d%s] -> [%s:%d%s]", req->host, req->port, req->path,
			  alias->host, alias->port, alias->path);
		sourcecon = find_mount_with_req (alias);
		if (!sourcecon) /* No trans */
		{
			int err;

			thread_mutex_unlock (&info.mount_mutex);
			thread_mutex_unlock (&info.source_mutex);
			thread_mutex_unlock (&info.double_mutex);

			if ((sourcecon = relay_pull_stream (alias, &err)))
			{
				write_log (LOG_DEFAULT, "Found [%s:%d%s] -> [%s:%d%s], pulling relay", req->host, req->port, req->path,
					   alias->host, alias->port, alias->path);
				sourcecon->food.source->type = on_demand_pull_e;
			} else {
				write_log (LOG_DEFAULT, "On demand relay for [%s:%d%s] -> [%s:%d%s] failed, damn!", req->host, req->port, req->path,
					   alias->host, alias->port, alias->path);
			}

			thread_mutex_lock (&info.double_mutex);
			thread_mutex_lock (&info.mount_mutex);
			thread_mutex_lock (&info.source_mutex);
			return sourcecon;

		} else {
			return sourcecon;
		}
	}

	if (info.transparent_proxy && !hostname_local (req->host))
	{
		connection_t *sourcecon;
		int err;

		xa_debug (1, "DEBUG: Trying transparent proxy with host: [%s]", req->host);
		if ((sourcecon = relay_pull_stream (req, &err)))
		{
			sourcecon->food.source->type = on_demand_pull_e;
			return sourcecon;
		} else {
			write_log (LOG_DEFAULT, "Transparent proxy relay for [%s:%d%s] failed", req->host, req->port, req->path);
			return NULL;
		}
	}

	return NULL;
}

/* Just return the first source, or NULL if no sources are connected */
connection_t *
get_default_mount()
{
	avl_traverser trav = {0};
	connection_t *con, *max = NULL;

	thread_mutex_lock (&info.source_mutex);
	
	while ((con = avl_traverse(info.sources, &trav)))
	{
		if (con->food.source->connected == SOURCE_CONNECTED)
		{
			if (!max || con->food.source->priority > max->food.source->priority)
				max = con;
		}
	}
	
	thread_mutex_unlock (&info.source_mutex);
	
	return max;
}

void
add_chunk (connection_t *con)
{
	int read_bytes;
	int len;
	int tries;

	if (con->food.source->chunk[con->food.source->cid].clients_left > 0)
	{
		connection_t *smaller = get_twin_mount_wl (con->food.source);
#ifndef OPTIMIZE
		xa_debug (2, "DEBUG: Kicking trailing clients [%d] on id %d", con->food.source->chunk[con->food.source->cid].clients_left, 
			con->food.source->cid);
#endif
		thread_mutex_lock (&info.double_mutex);
		thread_mutex_lock (&con->food.source->mutex);

		kick_clients_on_cid (con->food.source, smaller);

		thread_mutex_unlock (&con->food.source->mutex);
		thread_mutex_unlock (&info.double_mutex);
	}
	
	len = 0;
	read_bytes = 0;
	tries = 0;

	do {
		errno = 0;
#ifdef _WIN32
	        sock_set_blocking(con->sock, SOCK_BLOCK);
#endif

		len = recv(con->sock, con->food.source->chunk[con->food.source->cid].data + read_bytes, SOURCE_READSIZE - read_bytes, 0);
		
		xa_debug (5, "DEBUG: Source received %d bytes in try %d, total %d, errno: %d", len, tries, read_bytes, errno);

#ifdef _WIN32
		sock_set_blocking(con->sock, SOCK_NONBLOCK);
#endif
		
		if (con->food.source->connected == SOURCE_KILLED)
			return;
		
		if ((len == 0) || ((len == -1) && (!is_recoverable(errno)))) {
			if (info.client_timeout > 0 && con->food.source->connected != SOURCE_KILLED) {
				/* Set this source as pending (not connected) */
				pending_connection (con);
				
				/* Sleep for client_timeout seconds. If during that time this source is set to
				SOURCE_KILLED, return false */
				if (pending_source_signoff (con))
				{
					thread_mutex_lock (&info.double_mutex);
					thread_mutex_lock (&con->food.source->mutex);
					move_clients_to_default_mount (con);
					kick_connection (con, "Client timeout exceeded, removing source");
					thread_mutex_unlock (&con->food.source->mutex);
					thread_mutex_unlock (&info.double_mutex);
					return;
				} else {
					/* Someone has stolen all our clients, let's just die */
					thread_mutex_lock (&info.double_mutex);
					thread_mutex_lock (&con->food.source->mutex);
					kick_connection (con, "Lost all clients to new source");
					thread_mutex_unlock (&con->food.source->mutex);
					thread_mutex_unlock (&info.double_mutex);
					return;
				}
			} else {
				/* This is where the current thread kills itself off, probably because 
				   the encoder signed off. */
				thread_mutex_lock (&info.double_mutex);
				thread_mutex_lock (&con->food.source->mutex);
				write_log (LOG_DEFAULT, "Source %s signed off, moving clients to default mount", 
					   con->food.source->audiocast.mount);
				
				if (!info.kick_clients)
					move_clients_to_default_mount (con);
				kick_connection (con, "Source signed off (killed itself)");
				thread_mutex_unlock (&con->food.source->mutex);
				thread_mutex_unlock (&info.double_mutex);
				return;
			}
		} else if (len > 0) {
			read_bytes += len;
			stat_add_read(&con->food.source->stats, len);
			info.hourly_stats.read_bytes += len;
		} else {
			my_sleep(READ_RETRY_DELAY * 1000);
		}

		tries++;
		
	} while ((info.use_meta_data && (read_bytes < SOURCE_READSIZE)) 
		 || ((!info.use_meta_data) && ((double)read_bytes < ((double)SOURCE_READSIZE*3.0)/4.0) && (tries < (READ_TIMEOUT / READ_RETRY_DELAY))));

	/* make sure we actually got some data, and it's not a zombie connection */
	if (read_bytes <= 0) {
		write_log(LOG_DEFAULT, "Didn't receive data from source after %d microseconds, assuming it died...", tries * READ_RETRY_DELAY);
		
		/* Set this source as pending (not connected) */
		pending_connection (con);
		
		if (info.client_timeout > 0) {
			/* Sleep for client_timeout seconds. If during that time this source is set to SOURCE_KILLED, return false */
			if (pending_source_signoff(con)) {
				thread_mutex_lock (&info.double_mutex);
				thread_mutex_lock(&con->food.source->mutex);
				move_clients_to_default_mount(con);
				kick_connection(con, "Client timeout exceeded, removing source");
				thread_mutex_unlock(&con->food.source->mutex);
				thread_mutex_unlock (&info.double_mutex);
				return;
			} else {
				/* Someone has stolen all our clients, let's just die */
				thread_mutex_lock(&con->food.source->mutex);
				kick_connection(con, "Lost all clients to new source");
				thread_mutex_unlock(&con->food.source->mutex);
				return;
			}
		} else {
			/* This is where the current thread kills itself off, probably because the encoder signed off. */
			thread_mutex_lock (&info.double_mutex);
			thread_mutex_lock(&con->food.source->mutex);
			write_log(LOG_DEFAULT, "Source %s died, moving clients to default mount", con->food.source->audiocast.mount);
			
			if (!info.kick_clients)
				move_clients_to_default_mount(con);
			kick_connection(con, "Source died");
			thread_mutex_unlock(&con->food.source->mutex);
			thread_mutex_unlock (&info.double_mutex);
			return;
		}
	}
	
	thread_mutex_lock (&con->food.source->mutex);
	/* Dumpfile */
	if (con->food.source->dumpfile && con->food.source->dumpfd != -1)
	{
		errno = 0;
		if (write(con->food.source->dumpfd, con->food.source->chunk[con->food.source->cid].data, read_bytes) <= 0)	{
			write_log(LOG_DEFAULT, "Writing to dumpfile %s for source %d failed [%d] (closing file)", con->food.source->dumpfile, con->id, errno);
			fd_close(con->food.source->dumpfd);
			nfree(con->food.source->dumpfile);
			con->food.source->dumpfile = NULL;
			con->food.source->dumpfd = -1;
		}
	}
	thread_mutex_unlock (&con->food.source->mutex);

#ifndef OPTIMIZE
	xa_debug (4, "-------add_chunk: Chunk %d was [%d] bytes", con->food.source->cid, read_bytes );
#endif
	
	/* Add metadata to this chunk */
	if (info.use_meta_data) {
		int ulen, metalen = ice_strlen (con->food.source->info.streamtitle) 
			+ ice_strlen (con->food.source->info.streamurl) + 29;

		if (metalen >= MAXMETADATALENGTH) {
			write_log (LOG_DEFAULT, "WARNING: Avoided buffer overrun in metadata. Metadata to long!");
			metalen = 0;
		}
		
		snprintf((char *)&con->food.source->chunk[con->food.source->cid].data[read_bytes + 1],
			 (SOURCE_BUFFSIZE + MAXMETADATALENGTH),
			 "StreamTitle='%s';StreamUrl='%s';%c", con->food.source->info.streamtitle, 
			 con->food.source->info.streamurl, '\0');
		
		ulen = (1 + (metalen + 1) / 16) * 16;
		con->food.source->chunk[con->food.source->cid].data[read_bytes] = 
			ulen ? (unsigned char) ((double)ulen / 16.0) : (unsigned char) 0;
		/* First length byte and last null byte is added */
		con->food.source->chunk[con->food.source->cid].metalen = read_bytes + ulen + 1;

		xa_debug (4, 
		    "DEBUG: Adding chunk: metachunk: [%s], metalen: %d, complete length: %d, read: %d, ulen: %d, ulenc: %d",
			  &con->food.source->chunk[con->food.source->cid].data[read_bytes + 1],
			  metalen + 2, con->food.source->chunk[con->food.source->cid].metalen, read_bytes, ulen,
			  (unsigned int) ((double)ulen / 16.0));
	}
	
	con->food.source->chunk[con->food.source->cid].len = read_bytes;
	con->food.source->chunk[con->food.source->cid].clients_left = con->food.source->num_clients;
	con->food.source->cid = (con->food.source->cid + 1) % CHUNKLEN;
	
}

/* Hey, now even I can understand this :) */
void 
write_chunk(source_t *source, connection_t *clicon)
{
	int i = 0;
	long int write_bytes = 0, len = 0;

	/* Try to write 2 times */
	for (i = 0; i < 2; i++)
	{
		/* We caught up with the source */
		if (source->cid == clicon->food.client->cid) /* No more data available */
			return;

		/* This is how much we should be writing to the client */
		len = source->chunk[clicon->food.client->cid].len - clicon->food.client->offset;
		
		xa_debug (5, "DEBUG: write_chunk(): Try: %d, writing chunk %d to client %d, len(%d) - offset(%d) == %d", i, clicon->food.client->cid, clicon->id, 
			  source->chunk[clicon->food.client->cid].len, clicon->food.client->offset, len);
		
		if (len < 0 || source->chunk[clicon->food.client->cid].len == 0)
		{
#ifndef OPTIMIZE
			xa_debug (5, "DEBUG: write_chunk: Empty chunk [%d] [%d]", source->chunk[clicon->food.client->cid].len,
				  clicon->food.client->offset );
#endif
			source->chunk[clicon->food.client->cid].clients_left--;
			clicon->food.client->cid = (clicon->food.client->cid + 1) % CHUNKLEN;
			clicon->food.client->offset = 0;
			continue; /* Perhaps for some reason the source read a zero sized chunk but the next one is ok.. */
		} 
		
		/* If we're switching sources (virgin == 2), finish exactly one frame */
		if (clicon->food.client->virgin == 2)
		{
			move_client (clicon, source, -1);
			return;
		}
		
		write_bytes = write_data (clicon, source);

		if (write_bytes < 0)
		{
#ifndef OPTIMIZE
			xa_debug (5, "DEBUG: client: [%2d] errors: [%3d]", clicon->id, client_errors (clicon->food.client));
#endif
			if (is_recoverable (0 - write_bytes))
				continue;
			break; /* Safe to assume that the client is kicked out due to socket error */
		}

		clicon->food.client->write_bytes += write_bytes;
		info.hourly_stats.write_bytes += write_bytes;
		stat_add_write (&source->stats, write_bytes);
		
		if (write_bytes + clicon->food.client->offset >= source->chunk[clicon->food.client->cid].len) {
			source->chunk[clicon->food.client->cid].clients_left--;
			clicon->food.client->cid = (clicon->food.client->cid + 1) % CHUNKLEN;
			clicon->food.client->offset = 0;
		} else {
			
			clicon->food.client->offset += write_bytes;
#ifndef OPTIMIZE
			xa_debug (5, "DEBUG: client %d only read %d of %d bytes", clicon->id, write_bytes, 
				  source->chunk[clicon->food.client->cid].len - clicon->food.client->offset);
#endif
		}
	}
	
	xa_debug (4, "DEBUG: client %d tried %d times, now has %d errors %d chunks behind source", clicon->id, i,
		  client_errors (clicon->food.client), source->cid < clicon->food.client->cid ? source->cid+CHUNKLEN - clicon->food.client->cid : source->cid - clicon->food.client->cid);
	
	if (clicon->food.client->alive == CLIENT_DEAD)
		return;
}

/* Hey, now even I can understand this :) */
void 
write_chunk_with_metadata (source_t *source, connection_t *clicon)
{
	int i = 0;
	long int write_bytes = 0, len = 0, sourcelen = 0, justnull = 0;

	/* Try to write 2 times */
	for (i = 0; i < 2; i++)
	{
		/* We caught up with the source */
		if (source->cid == clicon->food.client->cid) /* No more data available */
			return;

		if (source->info.udpseqnr == clicon->food.client->udpseqnr) {
			sourcelen = source->chunk[clicon->food.client->cid].len + 1;
			justnull = 1;
		} else {
			sourcelen = source->chunk[clicon->food.client->cid].metalen;
		}

		/* This is how much we should be writing to the client */
		len = sourcelen - clicon->food.client->offset;

		xa_debug (5, "DEBUG: write_chunk(): Try: %d, writing chunk %d to client %d, len(%d) - offset(%d) == %d",
			  i, clicon->food.client->cid, clicon->id, sourcelen, clicon->food.client->offset, len);
		
		if (len < 0 || sourcelen == 0)
		{
#ifndef OPTIMIZE
			xa_debug (5, "DEBUG: write_chunk: Empty chunk [%d] [%d]", sourcelen, 
				  clicon->food.client->offset);
#endif
			source->chunk[clicon->food.client->cid].clients_left--;
			clicon->food.client->cid = (clicon->food.client->cid + 1) % CHUNKLEN;
			clicon->food.client->offset = 0;
			continue;
		} 
		
		/* If we're switching sources (virgin == 2), finish exactly one frame */
		if (clicon->food.client->virgin == 2)
		{
			move_client (clicon, source, -1);
			return;
		}
		
		write_bytes = write_data_with_metadata (clicon, source, justnull);
		
		if (write_bytes < 0)
		{
#ifndef OPTIMIZE
			xa_debug (5, "DEBUG: client: [%2d] errors: [%3d]", clicon->id, client_errors (clicon->food.client));
#endif
			if (is_recoverable (0 - write_bytes))
				continue;
			break; /* Safe to assume that the client is kicked out due to socket error */
		}
		
		clicon->food.client->write_bytes += write_bytes;
		info.hourly_stats.write_bytes += write_bytes;
		stat_add_write (&source->stats, write_bytes);
		
		if (write_bytes + clicon->food.client->offset >= sourcelen) {
			if (justnull == 0)
				clicon->food.client->udpseqnr++;
			source->chunk[clicon->food.client->cid].clients_left--;
			clicon->food.client->cid = (clicon->food.client->cid + 1) % CHUNKLEN;
			clicon->food.client->offset = 0;
		} else {
			
			clicon->food.client->offset += write_bytes;
#ifndef OPTIMIZE
			xa_debug (5, "DEBUG: client %d only read %d of %d bytes", clicon->id, write_bytes, 
				  sourcelen - clicon->food.client->offset);
#endif
		}
	}
	
	xa_debug (4, "DEBUG: client %d tried %d times, now has %d errors %d chunks behind source", clicon->id, i,
		  client_errors (clicon->food.client), source->cid < clicon->food.client->cid ? source->cid+CHUNKLEN - clicon->food.client->cid : source->cid - clicon->food.client->cid);
	
	if (clicon->food.client->alive == CLIENT_DEAD)
		return;
}

void 
kick_clients_on_cid(source_t *source, connection_t *smaller)
{
	avl_traverser trav = {0};
	connection_t *clicon;

	unsigned int max = avl_count (source->clients) * avl_count (source->clients) + 2;

	xa_debug (3, "Clearing cid %d", source->cid);

	zero_trav (&trav);

#if !defined(SAVE_CPU) || !defined(OPTIMIZE)
	xa_debug (5, "DEBUG: In function kick_clients_on_cid. Source has %d clients", max);
#endif
	while (max >= 0)
	{
		clicon = avl_traverse (source->clients, &trav);
		if (!clicon)
			break;
		
		if (client_errors (clicon->food.client) >= (CHUNKLEN - 1) && clicon->food.client->alive != CLIENT_DEAD)
		    
		{
			if (smaller)
			{
				write_log (LOG_DEFAULT, "Moving client %d [%s] from mount %s to lower bitrate on mount %s",
					   clicon->id, con_host (clicon), source->audiocast.mount, smaller->food.source->audiocast.mount);
				move_to (clicon, source); /* Should be safe to do so here */
			} else {
				kick_connection (clicon, "Client cannot sustain sufficient bandwidth");
			}
			zero_trav (&trav); /* Start from the top of the tree */
		}
		max--;
	}
	source->chunk[source->cid].clients_left = 0;
#if !defined(SAVE_CPU) || !defined(OPTIMIZE)
	xa_debug (5, "DEBUG: leaving function kick_clients_on_cid");
#endif
}

/* 
 * Can't be removing clients inside the loop which handles all the
 * write_chunk()s, instead we kick all the dead ones for each chunk. 
 */
void 
kick_dead_clients(source_t *source)
{
	avl_traverser trav = {0};
	connection_t *clicon = NULL;

	int max = avl_count (source->clients) * avl_count (source->clients) + 2;
	
#if !defined(SAVE_CPU) || !defined(OPTIMIZE)
	xa_debug (5, "DEBUG: In function kick_dead_clients. Will run %d laps", max);
#endif

	/* Check for too many errors */
	while ((clicon = avl_traverse (source->clients, &trav))) {
		if (client_errors (clicon->food.client) >= (CHUNKLEN - 1)) {
			connection_t *smaller = get_twin_mount_wl (source);
			if (smaller)
			{
				write_log (LOG_DEFAULT, 
					   "Moving client %d [%s] from mount %s to lower bitrate on mount %s",
					   clicon->id, con_host (clicon), source->audiocast.mount, 
					   smaller->food.source->audiocast.mount);
				move_to_smaller_twin (source, clicon);
			}
			else
				kick_connection (clicon, "Too many errors (client not receiving data fast enough)");
		}
	}
	
	zero_trav (&trav);

	while (max >= 0) 
	{
		clicon = avl_traverse (source->clients, &trav);
		if (!clicon)
			break;
		
		if (clicon->food.client->alive == CLIENT_MOVE) {
			connection_t *smaller;
			if (source->connected == SOURCE_KILLED || source->connected == SOURCE_UNUSED || 
			    source->connected == SOURCE_PENDING)
				smaller = get_twin_mount (source);
			else
				smaller = get_twin_mount_wl (source);

			if (smaller)
			{
				write_log (LOG_DEFAULT, "Moving client %d [%s] from mount %s to lower bitrate on mount %s",
					   clicon->id, con_host (clicon), source->audiocast.mount, smaller->food.source->audiocast.mount);
				move_to (clicon, smaller);
				zero_trav (&trav);
			}
			else
				kick_connection (clicon, "Smaller source stream signed off");
			/* No zero_trav() here cause it's not removed from the list (close_connection() does that) */
		}
		
		if (clicon->food.client->alive == CLIENT_DEAD) {
			close_connection (clicon, &info);
			zero_trav (&trav);
		}
		
		max--;
	}

#if !defined(SAVE_CPU) || !defined(OPTIMIZE)
	xa_debug (5, "DEBUG: leaving function kick_dead_clients, %d laps left", max);
#endif
}
			
void 
move_clients_to_default_mount(connection_t *con)
{
	connection_t *clicon, *default_source = get_twin_mount_wl (con->food.source);

	if (!default_source)
		default_source = get_default_mount ();
	/* This might look like a deadlock, but get_default_mount() can never return
	   this mountpoint since this one is PENDING.. still.. it doesn't hurt to check */
	if (con != default_source && default_source) {
		thread_mutex_lock(&default_source->food.source->mutex);
		while ((clicon = avl_get_any_node(con->food.source->clients)))
			move_to(clicon, default_source);
		thread_mutex_unlock(&default_source->food.source->mutex);
	}
}

int
originating_id (connection_t *sourcecon, char *dshost)
{
	avl_traverser trav = {0};
	relay_id_t *rip;

	if (!sourcecon || !dshost)
	{
		xa_debug (1, "WARNING: originating_id called with NULL pointers");
		return -1;
	}

	if (sourcecon->food.source->type == encoder_e)
	{
		xa_debug (2, "DEBUG: originating_id () return -1 cause type is encoder_e");
		return -1;
	}

	while ((rip = avl_traverse (sourcecon->food.source->relay_tree, &trav)))
	{
		if (!rip || !rip->host)
		{
			write_log (LOG_DEFAULT, "ERROR: Erroneous host in relay id");
			continue;
		}
		if (ice_strcasecmp (dshost, rip->host) == 0)
		{
			xa_debug (2, "DEBUG: originating_id() returning %d, found hit", rip->id);
			return rip->id;
		}
	}

	xa_debug (2, "DEBUG: originating_id() for %s didn't match any hosts, returning -1", dshost);
	return -1;
}

void
move_client (connection_t *clicon, source_t *source, int meta_after)
{
	int olen = find_frame_ofs (source);
	int write_bytes;
	
	errno = 0;
	write_bytes = sock_write_bytes_or_kick (clicon->sock, clicon, &source->chunk[clicon->food.client->cid].data[clicon->food.client->offset], min (olen, meta_after));

	if (write_bytes < 0)
		return; /* It's dead */
	
	if (write_bytes > 0)
	{
		clicon->food.client->write_bytes += write_bytes;
		info.hourly_stats.write_bytes += write_bytes;
		source->stats.write_bytes += write_bytes;
	}
	
	avl_delete(source->clients, clicon);
	del_client(clicon, source);
	clicon->food.client->virgin = 1;
	pool_add (clicon);
	util_increase_total_clients();
	return;
}

int
write_data (connection_t *clicon, source_t *source)
{
	int write_bytes;
	
	if (source->chunk[clicon->food.client->cid].len - clicon->food.client->offset <= 0)
		return 0;
	
	write_bytes = sock_write_bytes_or_kick(clicon->sock, clicon, &source->chunk[clicon->food.client->cid].data[clicon->food.client->offset], 
					       source->chunk[clicon->food.client->cid].len - clicon->food.client->offset);
	
#ifndef OPTIMIZE
	xa_debug (4, "DEBUG: client %d in write_data(). Function write() returned %d of %d bytes, client on chunk %d (+%d), source on chunk %d", clicon->id, write_bytes,
		  source->chunk[clicon->food.client->cid].len - clicon->food.client->offset, clicon->food.client->cid, clicon->food.client->offset,
		  source->cid);
#endif
	if (write_bytes < 0)
		return 0 - errno;
	return write_bytes;
}

int
write_data_with_metadata (connection_t *clicon, source_t *source, int justnull)
{
	int write_bytes, res = 0;
	char nullc[1];

	nullc[0] = '\0';

	if (justnull == 1) {
		int len = source->chunk[clicon->food.client->cid].len - clicon->food.client->offset;
		
		write_bytes = sock_write_bytes_or_kick (
			clicon->sock,
			clicon,
			&source->chunk[clicon->food.client->cid].data[clicon->food.client->offset],
			len);
		
		xa_debug (4, "DEBUG: 1: Wrote %d initial bytes (len: %d offset: %d", write_bytes, source->chunk[clicon->food.client->cid].len, clicon->food.client->offset);

		if (write_bytes == len) {
			res = sock_write_bytes_or_kick (clicon->sock, clicon, nullc, 1);
		}
		
		if (res > 0)
			write_bytes += res;

		xa_debug (4, "DEBUG: 2: Wrote %d supplemental bytes", res);

	} else {
		write_bytes = sock_write_bytes_or_kick (
			clicon->sock, 
			clicon, 
			&source->chunk[clicon->food.client->cid].data[clicon->food.client->offset], 
			source->chunk[clicon->food.client->cid].metalen - clicon->food.client->offset);

		xa_debug (4, "DEBUG: 3: Wrote %d initial bytes (len: %d offset: %d", write_bytes, source->chunk[clicon->food.client->cid].metalen, clicon->food.client->offset);
	}
	
#ifndef OPTIMIZE
	xa_debug (4, "DEBUG: client %d in write_data_with_metadata(). Function write() returned %d of %d bytes, client on chunk %d (+%d), source on chunk %d", clicon->id, write_bytes,
		  source->chunk[clicon->food.client->cid].len - clicon->food.client->offset, clicon->food.client->cid, clicon->food.client->offset,
		  source->cid);
#endif
	if (write_bytes < 0)
		return 0 - errno;
	return write_bytes;
}

connection_t *
get_source_with_mount (const char *mount)
{
	avl_traverser trav = {0};
	connection_t *travcon;
	char* alias;

	thread_mutex_lock (&info.source_mutex);
	
	while ((travcon = avl_traverse (info.sources, &trav)))
	{
		if (! (alias = strchr (travcon->food.source->audiocast.mount, '/')))
			alias = travcon->food.source->audiocast.mount;
		if ((ice_strcmp (alias, mount) == 0) ||
		    ((mount[0] != '/') && (ice_strcmp (&mount[0], &alias[1]) == 0)))
		{
			thread_mutex_unlock (&info.source_mutex);
			return travcon;
		}
	}
	
	thread_mutex_unlock (&info.source_mutex);
	return NULL;
}

connection_t *
get_source_from_host (connection_t *con)
{
	avl_traverser trav = {0};
	connection_t *travcon;

	thread_mutex_lock (&info.source_mutex);

	while ((travcon = avl_traverse (info.sources, &trav)))
	{
		if (hosteq (con, travcon))
		{
			thread_mutex_unlock (&info.source_mutex);
			return travcon;
		}
	}

	thread_mutex_unlock (&info.source_mutex);

	return NULL;
}

void
describe_source (const com_request_t *req, const connection_t *sourcecon)
{
	const source_t *source;
	char buf[BUFSIZE];
	
	if (!req || !sourcecon)
	{
		xa_debug (1, "WARNING: describe_source(): called with NULL pointers");
		return;
	}
	
	if (sourcecon->type != source_e)
	{
		xa_debug (1, "WARNING: describe_source(): called with invalid type");
		return;
	}

	describe_connection (req, sourcecon);
	
	source = sourcecon->food.source;

	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_START, "Misc source info:");
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Connected: %s", source->connected ? "yes" : "no");
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Source Type: %s", sourcetype_to_string (source->type));

	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Protocol: %s", source->protocol == icy_e ? "ICY" : "x-audiocast");
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Number of clients: %lu", source->num_clients);
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Dumpfile and dumpfd: %s : %d", 
			  source->dumpfile ? source->dumpfile : "(null)", source->dumpfd);
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Stream priority: %d", source->priority);
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Stream title: %s", 
			  source->info.streamtitle ? source->info.streamtitle : "(null)");
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Stream url: %s", source->info.streamurl ? source->info.streamurl : "(null)");
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Stream msg: %s", source->info.streammsg ? source->info.streammsg : "(null)");
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Stream length: %ld", source->info.streamlength);
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Source name: %s", source->audiocast.name ? source->audiocast.name : "(null)");
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Source genre: %s", source->audiocast.genre ? source->audiocast.genre : "(null)");
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Source bitrate: %d", source->audiocast.bitrate);
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Source url: %s", source->audiocast.url ? source->audiocast.url : "(null)");
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Source mount: %s", source->audiocast.mount ? source->audiocast.mount : "(null)");
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Source content id: %s", source->audiocast.contentid ? source->audiocast.contentid : "(null");
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Source description: %s", source->audiocast.description ? source->audiocast.description : "(null)");
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Megabytes read: %lu", source->stats.read_megs);
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Megabytes written: %lu", source->stats.write_megs);
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Client connections: %lu", source->stats.client_connections);
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Client connect time: %s", nice_time_minutes (source->stats.client_connect_time, buf));
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Average client connect time: %s", connect_average (source->stats.client_connect_time, source->stats.client_connections, buf));
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Average client transfer: %lu", transfer_average (source->stats.write_megs, source->stats.client_connections));
	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_MISC, "Stream Song Counter: %lu", source->info.udpseqnr);

	admin_write_line (req, ADMIN_SHOW_DESCRIBE_SOURCE_END, "End of source info");
}

const char source_protos[2][12] = { "icy", "x-audiocast" };
const char source_types[5][16] = { "encoder", "pulling relay", "on demand relay", "file transfer", "unknown source" };

const char *
sourceproto_to_string (protocol_t proto)
{
	return source_protos[proto];
}

const char *
sourcetype_to_string (source_type_t type)
{
	return source_types[type];
}

/* What we want to do here is give the client the best possible
 * chunk in the source to start from. Where he suffers the least
 * from both his own slow network connection, and discrepanices
 * in the source feed. I used to set this to the chunk where the
 * source was, minus one, but I think it's better to find a place
 * in the middle somewhere. I leave this for later testing though.
 */
int
start_chunk (source_t *source)
{
	return source->cid > 0 ? source->cid - 1 : CHUNKLEN - 1;
}

connection_t *get_twin_mount_proc(source_t *source)
{
	connection_t *target, *max = NULL;
	avl_traverser trav = {0};

	while ((target = avl_traverse(info.sources, &trav))) {
		if ((source->audiocast.contentid != NULL) && (source->audiocast.contentid[0]) &&
		    (target->food.source->audiocast.contentid != NULL) &&
		    (ice_strcmp (source->audiocast.contentid, target->food.source->audiocast.contentid) == 0) && 
		    (target->food.source->audiocast.bitrate <= source->audiocast.bitrate) &&
		    (!thread_equal(source->thread, target->food.source->thread))) {
			
			max = target;
			
			break;
		}
	}

	return max;
}

/* Must have source mutex to call this! */
connection_t *get_twin_mount(source_t *source)
{
	return get_twin_mount_proc(source);
}

connection_t *get_twin_mount_wl(source_t *source)
{
	connection_t *max = NULL;

	thread_mutex_lock(&info.source_mutex);

	max = get_twin_mount_proc(source);

	thread_mutex_unlock(&info.source_mutex);
	
	return max;
}

void move_to_smaller_twin(source_t *source, connection_t *clicon)
{
	clicon->food.client->alive = CLIENT_MOVE;
}

void
source_write_to_client (source_t *source, connection_t *clicon)
{
	client_t *client;

	if (!clicon || !source) {
		xa_debug (1, "WARNING: source_write_to_client() called with NULL pointers");
		return;
	}
	
	client = clicon->food.client;

	if (client->alive == CLIENT_DEAD)
		return;
	
	if (client->virgin == CLIENT_PAUSED || client->virgin == -1)
		return;
	
	if (client->virgin == CLIENT_UNPAUSED)	{
		client->cid = start_chunk (source);
		client->offset = find_frame_ofs (source);
		client->virgin = 0;
	}
	
	if (client->virgin == 1) {
		client->cid = start_chunk (source); 
		client->offset = find_frame_ofs(source);
		xa_debug (2, "Client got offset %d", client->offset);
		client->virgin = 0;
		source->num_clients = source->num_clients + (unsigned long int)1;
	}
	
	if (clicon->food.client->use_icy_metadata && info.use_meta_data)
	  write_chunk_with_metadata (source, clicon);
	else
	  write_chunk (source, clicon);
}

void
source_get_new_clients (source_t *source)
{
	connection_t *clicon;
	
	while ((clicon = pool_get_my_clients (source))) {
		xa_debug (1, "DEBUG: source_get_new_clients(): Accepted client %d", clicon->id);
		avl_insert (source->clients, clicon);
	}
}

int
source_get_id (char *argument)
{
	connection_t *con;
	
	if (!argument || !argument[0])
		return -1;
	
	if (isdigit((int)argument[0]) && strchr (argument, '.') == NULL && strchr (argument, '/') == NULL)
		return atoi (argument);
	
	con = find_source_with_mount (argument);
	
	if (con)
		return con->id;
	
	return -1;
}
