/* 
** Copyright (C) 1994, 1995 Enterprise Integration Technologies Corp.
**         VeriFone Inc./Hewlett-Packard. All Rights Reserved.
** Kevin Hughes, kev@kevcom.com 3/11/94
** Kent Landfield, kent@landfield.com 4/6/97
** 
** This program and library is free software; you can redistribute it and/or 
** modify it under the terms of the GNU (Library) General Public License 
** as published by the Free Software Foundation; either version 2 
** of the License, or any later version. 
** 
** This program is distributed in the hope that it will be useful, 
** but WITHOUT ANY WARRANTY; without even the implied warranty of 
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
** GNU (Library) General Public License for more details. 
** 
** You should have received a copy of the GNU (Library) General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA 
*/

#include "hypermail.h"
#include <ctype.h>

#include "setup.h"
#include "struct.h"
#include "printfile.h"
#include "print.h"

#include "threadprint.h"

/* Uses threadlist to find the next message after
 * msgnum in the thread containing msgnum.
 * Returns NULL if there are no more messages in 
 * the thread.
 */
struct emailinfo *nextinthread(int msgnum)
{
  struct reply *rp = threadlist;

  for (rp = threadlist; (rp != NULL) && (rp->msgnum != msgnum); rp = rp->next) {
    ;
  }

  if (rp == NULL) {
                /* msgnum not found in threadlist */
    return NULL;
  }

  rp = rp->next;

  if ((rp == NULL) || (rp->frommsgnum == -1)) {
                /* end of thread - no next message */
    return NULL;
  }

  return rp->data;
}


/*
** Output a menu line with hyperlinks for table display
**
** All parameters are inputs only.  Parameters:
**   fp       : HTML output file pointer
**   idx      : Type of page this menu is for.
**   about    : "NONE" or else the URL of info about this archive.
**   archives : "NONE" or else the URL of more hypermail archives.
**   currentid: "" or else the id of the "current" message.
**   cursub   : "" or else the subject of the "current" message.
**   pos      : Called at the top or bottom of the page.
*/

void fprint_menu(FILE *fp, mindex_t idx, char *about, char *archives,
                 char *currentid, char *cursub, int pos)
{
  char *ptr;
  fprintf(fp, "<DIV ALIGN=CENTER>\n<TABLE BORDER=2 WIDTH=\"100%%\">\n<TR>\n");

  if (set_mailcommand) {
    if (set_hmail) {
      ptr=makemailcommand("mailto:$TO", set_hmail, "", "");
      fprintf(fp, "<TH><A HREF=\"%s\">%s</A></TH>\n",
	      ptr?ptr:"", lang[MSG_NEW_MESSAGE]);
      if(ptr)
        free(ptr);
      
      if((currentid != NULL && currentid[0] != '\0') ||
	 (cursub    != NULL && cursub[0]    != '\0')) {

        ptr=makemailcommand(set_mailcommand, set_hmail,
                            currentid, cursub);
	fprintf(fp, "<TH><A HREF=\"%s\">%s</A></TH>\n",
		ptr?ptr:"", lang[MSG_REPLY]);
        if(ptr)
          free(ptr);
      }
    }
  }
  
   if (set_about)
      fprintf(fp,"<TH><A HREF=\"%s\">%s</A></TH>\n", set_about,
                  lang[MSG_ABOUT_THIS_LIST]);

   if (idx != NO_INDEX && !set_reverse) {
      if (pos == PAGE_TOP)
          fprintf(fp,"<TH><A HREF=\"#end\">%s</A></TH>\n",
                  lang[MSG_END_OF_MESSAGES]);
      else
          fprintf(fp,"<TH><A NAME=\"end\" HREF=\"#\">%s</A></TH>\n",
                  lang[MSG_START_OF_MESSAGES]);
   }
           
   if (idx != DATE_INDEX)
      fprintf(fp,"<TH><A HREF=\"%s\">%s</A></TH>\n", datename,
                  lang[MSG_DATE_VIEW]);

   if (idx != THREAD_INDEX)
      fprintf(fp,"<TH><A HREF=\"%s\">%s</A></TH>\n", thrdname,
                  lang[MSG_THREAD_VIEW]);

   if (idx != SUBJECT_INDEX)
      fprintf(fp,"<TH><A HREF=\"%s\">%s</A></TH>\n", subjname,
                  lang[MSG_SUBJECT_VIEW]);

   if (idx != AUTHOR_INDEX)
      fprintf(fp,"<TH><A HREF=\"%s\">%s</A></TH>\n", authname,
                  lang[MSG_AUTHOR_VIEW]);

   if (set_archives)
      fprintf(fp,"<TH><A HREF=\"%s\">%s</A></TH>\n", set_archives,
                  lang[MSG_OTHER_GROUPS]);

   fprintf(fp, "</TR>\n</TABLE>\n</DIV>\n");
}

/*----------------------------------------------------------------------------*/

void fprint_summary(FILE *fp, long first_d, long last_d, int num)
{
   fprintf(fp, "<DIV ALIGN=CENTER>\n");
   fprintf(fp, "<TABLE>\n<TR>\n  <TH COLSPAN=4>%d %s</TH>\n</TR>\n",
                 num, lang[MSG_MESSAGES]);
   fprintf(fp, "<TR>\n");
   fprintf(fp, "  <TH>%s:</TH><TD><EM>%s</EM></TD>\n",
                getdatestr(first_d), lang[MSG_STARTING]);
   fprintf(fp, "  <TH>%s:</TH><TD><EM>%s</EM></TD>\n",
                getdatestr(last_d), lang[MSG_ENDING]);
   fprintf(fp, "</TR>\n</TABLE>\n</DIV>\n");
}

/*----------------------------------------------------------------------------*/

void print_index_header_links(FILE *fp,char *archives,char *about,int called,int amountmsgs)
{
    /* 
    ** Print out the links for
    ** 
    **      About this archive 
    **      Most recent messages 
    **      Messages sorted by: [ date ][ subject ][ author ] 
    **      Other mail archives 
    **
    ** as appropriate.
    */
  char *ptr;
    
    if (set_mailcommand && set_hmail) {
      ptr=makemailcommand("mailto:$TO", set_hmail, "", "");
      fprintf(fp,"<STRONG><A HREF=\"%s\">%s</A></STRONG>\n",
              ptr?ptr:"", lang[MSG_NEW_MESSAGE]);
      if(ptr)
        free(ptr);
    }

    if (about) 
        fprintf(fp,"<STRONG><A HREF=\"%s\">%s</A></STRONG><br>\n",
           about, lang[MSG_ABOUT_THIS_ARCHIVE]);

    if (!set_reverse && (called != FROM_AUTHOR && called != FROM_SUBJECT))
        fprintf(fp, "<STRONG><A HREF=\"#end\">%s</A></STRONG><br>\n",
           lang[MSG_MOST_RECENT_MESSAGES]);

    fprintf(fp, "<STRONG>%d %s %s:</STRONG> \n",amountmsgs, lang[MSG_ARTICLES], lang[MSG_SORTED_BY]);

    if (called != FROM_AUTHOR)
        fprintf(fp,"<A HREF=\"%s\">[ %s ]</A>\n",authname,lang[MSG_AUTHOR]);

    if (called != FROM_DATE)
        fprintf(fp, "<A HREF=\"%s\">[ %s ]</A>\n", datename,lang[MSG_DATE]);

    if (called != FROM_THREAD)
        fprintf(fp, "<A HREF=\"%s\">[ %s ]</A>\n", thrdname,lang[MSG_THREAD]);

    if (called != FROM_SUBJECT)
        fprintf(fp,"<A HREF=\"%s\">[ %s ]</A>\n",subjname,lang[MSG_SUBJECT]);

    if (archives) 
        fprintf(fp,"<br><STRONG><A HREF=\"%s\">%s</A></STRONG>\n",
                    archives, lang[MSG_OTHER_MAIL_ARCHIVES]);

    fprintf(fp, "<P>\n");
}

void print_index_footer_links(FILE *fp, char *archives, int called,int amountmsgs)
{
    /* 
    ** Print out the links for
    ** 
    **      Messages sorted by: [ date ][ subject ][ author ] 
    **      Other mail archives 
    **
    ** as appropriate.
    */

    fprintf(fp, "\n");
    fprintf(fp, "<STRONG>%d %s %s:</STRONG> \n",amountmsgs, lang[MSG_ARTICLES], lang[MSG_SORTED_BY]);

    if (called != FROM_AUTHOR)
        fprintf(fp,"<A HREF=\"%s\">[ %s ]</A>\n",authname,lang[MSG_AUTHOR]);

    if (called != FROM_DATE)
        fprintf(fp, "<A HREF=\"%s\">[ %s ]</A>\n", datename,lang[MSG_DATE]);

    if (called != FROM_THREAD)
        fprintf(fp, "<A HREF=\"%s\">[ %s ]</A>\n", thrdname,lang[MSG_THREAD]);

    if (called != FROM_SUBJECT)
        fprintf(fp,"<A HREF=\"%s\">[ %s ]</A>\n",subjname,lang[MSG_SUBJECT]);

    if (archives) 
        fprintf(fp,"<BR><STRONG><A HREF=\"%s\">%s</A></STRONG>\n",
                    archives, lang[MSG_OTHER_MAIL_ARCHIVES]);
    
    fprintf(fp, "\n\n");
}

/*
** Prints a comment in the file.
*/

void printcomment(FILE *fp, char *label, char *value)
{
  if(value && *value)
    /* we only save the non-NULL ones */
    fprintf(fp, "<!-- %s=\"%s\" -->\n", label, value);
}

/*
** Prints a line of HTML. Used to filter off unwanted tags from the output.
*/

void printhtml(FILE *fp, char *line)
{
  /* the banned tags ignore all of them that begin exactly like "<tag" or
     "</tag" (case insensitive). Any other letters within the tags destroy
     this filter capability. */

  char *banned_tags[]={"html>"}; /* ending >-letter must be included */
  unsigned int i;
  int index;
  
  while(*line) {
    if('<' != *line)
      fputc(*line++, fp);
    else {
      if('/' == line[1])
        index=2;
      else
        index=1;
      for(i=0; i<sizeof(banned_tags)/sizeof(banned_tags[0]);i++) {
        if(!strncasecmp(&line[index],
                        banned_tags[i],
                        strlen(banned_tags[i]))) {
          line+=strlen(banned_tags[i])+index;
          break;
        }
      }
      if(sizeof(banned_tags)/sizeof(banned_tags[0]) == i) {
        fputc(*line++, fp);
        if(2 == index)
          fputc(*line++, fp);
      }
    }
  }

}

/*
** Pretty-prints the dates in the index files.
*/

void printdates(FILE *fp, struct header *hp)
{
  char *subj;
  if (hp != NULL) {
    struct emailinfo *em=hp->data;
    printdates(fp, hp->left);
    subj=convchars(em->subject);
    if(set_indextable) {
      fprintf(fp,"<tr><td><A HREF=\"%.4d.%s\">%s</A></td>"
              "<td><A NAME=\"%d\"><EM>%s</EM></A></td>"
              "<td>%s</td></tr>\n",
              em->msgnum, set_htmlsuffix, subj,
              em->msgnum, em->name,
              getdatestr(em->date));
    }
    else {
      fprintf(fp,"<LI><A HREF=\"%.4d.%s\"><STRONG>%s</STRONG></A>&nbsp;"
              "<A NAME=\"%d\"><EM>%s</EM></A>&nbsp;<EM>(%s)</EM>\n",
              em->msgnum, set_htmlsuffix,
              subj, em->msgnum,em->name,
              getdatestr(em->date));
    }
    free(subj);
    printdates(fp, hp->right);
  }
}

int showheader(char *header)
{
   return (inlist(set_show_headers, header));
}

/*
** Converts URLs, email addresses, and message IDs in a line to links,
** mail commands, and articles respectively. Write everything to the
** file pointed to by 'fp'.
*/

void ConvURLs(FILE *fp,
	      char *line,
	      char *mailid,
	      char *mailsubject)
{
    char *parsed;
    char *newparse;

#if 0
    int i, j, status, subjmatch;
    char inreply[REPYSTRLEN], name[NAMESTRLEN], subject[SUBJSTRLEN];
    /* Daniel's note, is this kludge thing actually ever used?
       If it is, then make it deal with the non-static-buffer way later */
    if (!strncasecmp(line, "In-Reply-To:", 12) ||
          !strncasecmp(line, "References:", 11) ||
          ((!strncmp(line, "    ", 4) || line[0] == '\t')
          && strchr(line, '<') && strchr(line, '>') && strchr(line, '@'))) {

        strcpymax(inreply, getreply(line), REPYSTRLEN);
        strcpymax(tmpline4, convchars(line), MAXLINE);

        if (inreply[0] != '\0') {
            status = hashreplylookup(inreply, name, subject, &subjmatch);
            if (status != -1) {
                for (i = j = 0; tmpline4[i] != ' ' &&
                     j < MAXLINE && tmpline4[i] != '\t'; i++)
                     tmpline3[j++] = tmpline4[i];

                while (isspace(tmpline4[i])) {
                     tmpline3[j++] = ' ';
                     i++;
                }
                tmpline3[j] = '\0';

                sprintf(tmpline3, "%s<A HREF=\"%.4d.%s\">",
                        tmpline3, status, set_htmlsuffix);
                for (j = strlen(tmpline3); tmpline4[i] &&
                     tmpline4[i] != '\n' && j < MAXLINE; i++)
                     tmpline3[j++] = tmpline4[i];

                tmpline3[j] = '\0';
                strcat(tmpline3, "</A>\n");
                return tmpline3;
            }
        }
        return tmpline4;
    }
#endif

    parsed = parseurl(line);
    if (use_mailcommand) {
        /* Exclude headers that are not mail type headers */
#if 0
      /* These will never match anyway */
      if (strcasestr(parsed, "Message-ID") || 
	  strcasestr(parsed, "Supersedes"))
	return parsed;
#endif

      if (parsed && *parsed) {

	newparse =
	  parseemail(parsed,     /* source */
		     mailid,       /* mail's Message-Id: */
		     mailsubject); /* mail's Subject: */
      
	free(parsed);

	parsed = newparse;
      }
    }

    if(parsed) {
      /* write it to the file! */
      fwrite(parsed, strlen(parsed), 1, fp);
#if 0 /* debug */
      /* write it to the file! */
      fwrite(parsed, strlen(parsed), 1, stderr);
#endif
      free(parsed);
    }
}


/*
** The heuristics for displaying an otherwise ordinary line (a non-quote,
** non-sig, non-inhtml, non-blank line) if 'showhtml' is true (which really
** means we're *not* doing <PRE> to format the text) have changed so that
** indented text shows up correctly.  First, leading spaces become HTML
** non-breaking spaces (&nbsp;).  In order for this to work, those lines
** must have been preceded by a <BR> or <P>.  We accomplish this by checking
** ahead when we print each line... if we determine that the next line
** should be broken before, we do so.
*/

void printbody(FILE *fp, struct body *bp, char *id, char *subject)
{
    int insig, inblank;
    int inhtml;
    char inheader=FALSE; /* we always start in a mail header */
    int pre=FALSE;

    if (set_showhr)
        fprintf(fp, "<HR>\n");

    printcomment(fp, "body", "start");

    if (!set_showhtml) {
      fprintf(fp, "<PRE>\n");
      pre=TRUE;
    }
#if 0
    else
      printf("SHOWHTML is ON\n");
#endif
    inblank = 1;
    insig = 0;

    inhtml = 0;

    while (bp != NULL) {
          if (bp->html) {
            /* already in HTML, don't touch */
            if(pre) {
                fprintf(fp, "</PRE>\n");
                pre=FALSE;              
            }
            printhtml(fp, bp->line);
            inheader = FALSE; /* this can't be a header if already in HTML */
            bp = bp->next;
            continue;
          }

          if(bp->header) {
            char head[128];
            if(!inheader) {
              if(!pre && !set_showhtml && set_showheaders) {
                fprintf(fp, "<PRE>\n");
                pre=TRUE;
              }
            }
            inheader=TRUE;
            if(sscanf(bp->line, "%127[^:]", head) &&
               set_show_headers && !showheader(head)) {
              /* the show header keyword has been used, then we skip all those
                 that aren't mentioned! */
              if(isalnum(*head) || !set_showheaders) {
                /* this check is only to make sure that the last line among 
                   the headers (the "\n" one) won't be filtered off */
                bp = bp->next;
                continue;
              }
            }
          }
          else {
            if(inheader) {
              if(set_showhtml) {
                if(pre)
                  fprintf(fp, "</PRE>\n");
                pre=FALSE;
                fprintf(fp, "<P>\n");
              }
            }
            inheader=FALSE;
          }

#ifdef REMOVED_990310
          if (bp->attached && bp->header) {
              fprintf(fp, "%s%s", bp->line,
                      pre?"":"<br>\n"); /* only show <br> if not in <PRE>
                                         since in <PRE> mode, the existing
                                         newlines are enough! */
            bp = bp->next;
            continue;            
          }
#endif

        if (((bp->line)[0] != '\n') && (bp->header && !set_showheaders)) {
            bp = bp->next;
            continue;
        }

        if ((bp->line)[0] == '\n' && inblank) {
            bp = bp->next;
            continue;
        }
        else
            inblank = 0;

        if (!strncasecmp(bp->line, "<HTML>", 6)) {
            inhtml = 1;
            if (!set_showhtml && pre) {
                fprintf(fp, "</PRE>\n");
                pre=FALSE;
            }
        }
        else if (!strncasecmp(bp->line, "</HTML>", 7)) {
            inhtml = 0;
            if (set_showhtml)
                fprintf(fp, "%s", bp->line);
            else {
	      char *ptr;
	      fprintf(fp, "%s<BR>", ptr=convchars(bp->line));
	      free(ptr);
	    }
            if (!set_showhtml && !pre) {
                fprintf(fp, "<PRE>\n");
                pre=TRUE;
            }
            bp = bp->next;
            continue;
        }

        if (inhtml) {
            if (set_showhtml)
                fprintf(fp, "%s", bp->line);
            else {
	      char *ptr;
	      fprintf(fp, "%s<BR>", ptr=convchars(bp->line));
	      free(ptr);
	    }
            bp = bp->next;
            continue;
        }
        if (set_showhtml) {
            if (!strcmp(bp->line, "--\n") ||
                !strcmp(bp->line, "-- \n") ||
                !strcmp(bp->line, "----\n") ||
                !strcmp(bp->line, "---\n")) {
                insig = 1;
                if(!pre) {
                  fprintf(fp, "<PRE>\n");
                  pre=TRUE;
                }
            }

            if ((bp->line)[0] == '\n')
                fprintf(fp, "<P>");
            else {
                if (insig) {
		  ConvURLs(fp, bp->line, id, subject);
                }
                else if (isquote(bp->line)) {
                    fprintf(fp, "%s",
                           (set_iquotes)?"<EM>":"");
		    
		    ConvURLs(fp, bp->line, id, subject);

		    fprintf(fp, "%s<BR>\n", (set_iquotes)?"</EM>":"");

		    /* Daniel changed this 17th of November 1998 
		       We're gonna make this deal with infinite buffer sizes! */
		    
                }
                else if ((bp->line)[0] != '\0') {
#if 1
                    char *sp;
                    sp = bp->line;
                    while (*sp && (*sp == ' ' || *sp == '\t')) {
                        if (*sp == '\t')
                            fprintf(fp, "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;");
                        else
                            fprintf(fp, "&nbsp;");
                        sp++;
                    }
                    ConvURLs(fp, sp, id, subject);
#else
                    int sp;
                    for (sp = 0;isspace(bp->line[sp]);sp++){
                        if (bp->line[sp] == '\t')
                            fprintf(fp, "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;");
                        else
                            fprintf(fp, "&nbsp;");
                    }
                    fprintf(fp, "%s",convurls(&bp->line[sp],id,subject));
#endif

                    /*
                    ** Determine whether we should break.
                    ** We could check for leading spaces
                    ** or quote lines, but in general,
                    ** non-alphanumeric lines should be
                    ** broken before.
                    */
 
                    if ((set_showbr && !bp->header) ||
                        ((bp->next != NULL) && !isalnum(bp->next->line[0])))
                      fprintf(fp, "<BR>");
                    if(!bp->header) {
                      fprintf(fp, "\n");
                    }
                }

            }
        }
        else if ((bp->line)[0] != '\0') {
	  ConvURLs(fp, bp->line, id, subject);
        }
        bp = bp->next;

    }

    if (pre) 
        fprintf(fp, "</PRE>\n");

    fprintf(fp, "<P>");
    printcomment(fp, "body", "end");
    if (set_showhr)
        fprintf(fp, "<HR>\n");
}

/*
** Printing...the other main part of this program!
** This writes out the articles, beginning with the number startnum.
*/

void writearticles(char *dir,
		   char *label,
		   char *about,
		   int overwrite,
		   int startnum,
		   int maxnum)
{
    int num, skip, subjmatch, newfile;

    struct emailinfo *email;
    struct emailinfo *email2;

    char filename[MAXFILELEN];
    struct body *bp, *status;
    struct reply *rp;
    FILE *fp;
    char *ptr;

    num = startnum;

    if (set_showprogress)
        printf("%s \"%s\"...    ", lang[MSG_WRITING_ARTICLES], dir);

    while ((bp = hashnumlookup(num, &email))) {

        sprintf(filename, "%s%s%.4d.%s", 
                dir, (dir[strlen(dir) - 1] == '/') ? "" : "/", 
                num, set_htmlsuffix);

        /*
        ** Determine to overwrite files or not
        */

        if (isfile(filename))
            newfile = 0;
        else
            newfile = 1;

	skip=0;
        if (!newfile && !overwrite) {
            skip = 1; /* is this really necessary with continue ??? */
            num++;
            continue;
        }
        else {
            if ((fp = fopen(filename, "w")) == NULL) {
                sprintf(errmsg,"%s \"%s\".",lang[MSG_COULD_NOT_WRITE],filename);
                progerr(errmsg);
            }
        }

        /*
        ** Create the comment fields necessary for incremental updating
        */

        print_msg_header(fp, label,
                         email->subject, dir,
                         email->name, email->emailaddr, email->msgid,
                         email->charset, filename);

        printcomment(fp, "received", email->fromdatestr);
        printcomment(fp, "isoreceived", secs_to_iso(email->fromdate));
        printcomment(fp, "sent", email->datestr);
        printcomment(fp, "isosent", secs_to_iso(email->date));
        printcomment(fp, "name", email->name);
        printcomment(fp, "email", email->emailaddr);
        printcomment(fp, "subject", ptr=convchars(email->subject));
        if(ptr)
          free(ptr);
        printcomment(fp, "id", email->msgid);
        printcomment(fp, "charset", email->charset);
        printcomment(fp, "inreplyto", ptr=convchars(email->inreplyto));
        if(ptr)
          free(ptr);

        if (set_usetable) {
            fprint_menu(fp, NO_INDEX, about, "",
                        email->msgid, email->subject,
                        PAGE_TOP);
            fprintf(fp, "<P>\n");
        }

        /*
        ** Print the message's author info and date.
        */

        fprintf(fp, "<STRONG>Subject: </STRONG>%s<BR>\n", email->subject);
        if (!strcmp(email->name, email->emailaddr)) {
            if (use_mailcommand) {
              ptr=makemailcommand(set_mailcommand,
                                  email->emailaddr,
                                  email->msgid,
                                  email->subject);
              fprintf(fp, "<A HREF=\"%s\">", ptr?ptr:"");
              if(ptr)
                free(ptr);
              fprintf(fp, "<EM>%s</EM></A><BR>\n", email->name);
            }
            else
              fprintf(fp, "<EM>%s</EM><BR>\n", email->name);
        }
        else {
          if (use_mailcommand) {  
            ptr=makemailcommand(set_mailcommand,
                                email->emailaddr,
                                email->msgid,
                                email->subject);
            fprintf(fp,"<STRONG>From: </STRONG>%s (<A HREF=\"%s\">",
                    email->name, ptr?ptr:"");
            if(ptr)
              free(ptr);
            fprintf(fp, "<EM>%s</EM></A>)<BR>\n", email->emailaddr);
          }
          else
            fprintf(fp, "<STRONG>From: </STRONG>%s (<EM>%s</EM>)<BR>\n",
                    email->name, email->emailaddr);
        }
        fprintf(fp, "<STRONG>Date: </STRONG>%s\n<P>\n", getdatestr(email->date));
 
        /*
        ** This is here because it looks better here. The table looks
        ** better before the Author info. This stuff should be in 
        ** printfile() so it could be laid out as the user wants...
        */

        if (!set_usetable) {
            fprintf(fp,"<UL>\n");
            if (set_mailcommand && set_hmail) {
              ptr=makemailcommand("mailto:$TO", set_hmail, "", "");
              fprintf(fp,"<STRONG><A HREF=\"%s\">%s</A></STRONG>\n",
                      ptr?ptr:"", lang[MSG_NEW_MESSAGE]);
              if(ptr)
		free(ptr);

              if ((email->msgid && email->msgid[0]) ||
                  (email->subject && email->subject[0])) {
                ptr=makemailcommand(set_mailcommand, set_hmail,
                                    email->msgid,
                                    email->subject);
                fprintf(fp, "<LI><STRONG><A HREF=\"%s\">%s</A></STRONG>\n",
                        ptr?ptr:"", lang[MSG_REPLY]);
                if(ptr)
                  free(ptr);
              }
            }
            fprintf(fp,"<LI><STRONG>%s:</STRONG> \n",
                       lang[MSG_SORTED_BY]);
            fprintf(fp,"<A HREF=\"%s#%d\">[ %s ]</A>\n", 
                       datename, num, lang[MSG_DATE]);
            fprintf(fp,"<A HREF=\"%s#%d\">[ %s ]</A>\n", 
                       thrdname, num, lang[MSG_THREAD]);
            fprintf(fp,"<A HREF=\"%s#%d\">[ %s ]</A>\n", 
                       subjname, num, lang[MSG_SUBJECT]);
            fprintf(fp,"<A HREF=\"%s#%d\">[ %s ]</A>\n", 
                       authname,num, lang[MSG_AUTHOR]);
        }

        /*
        ** Should we print message links ?
        */

        if (set_show_msg_links) {
          printcomment(fp, "next", "start");
          if (set_usetable)
            fprintf(fp, "<UL>\n");
    
          /*
          ** Is there a next message?
          */
    
          status = hashnumlookup(num + 1, &email2);
          if (status) {
            fprintf(fp,"<LI><STRONG>%s:</STRONG> ",lang[MSG_NEXT_MESSAGE]);

            ptr=convchars(email2->subject);
            fprintf(fp, "<A HREF=\"%.4d.%s\">%s: \"%s\"</A>\n",
                    num + 1, set_htmlsuffix, email2->name,
                    ptr?ptr:"");
            if(ptr)
              free(ptr);
          }
    
          /*
          ** Is there a previous message?
          */
          
          status = hashnumlookup(num - 1, &email2);
          if (status) {
            fprintf(fp,"<LI><STRONG>%s:</STRONG> ",
                    lang[MSG_PREVIOUS_MESSAGE]);
            fprintf(fp,"<A HREF=\"%.4d.%s\">%s: \"%s\"</A>\n",
                    num - 1, set_htmlsuffix, email2->name,
                    ptr=convchars(email2->subject));
            if(ptr)
              free(ptr);
          }
    
          /*
          ** Is this message a reply to another?
          */
    
          if (email->inreplyto[0]) {
            email2 = hashreplylookup(email->msgnum, email->inreplyto, &subjmatch);
            if (email2) {
              if (subjmatch)
                fprintf(fp,"<LI><STRONG>%s:</STRONG>",
                        lang[MSG_MAYBE_IN_REPLY_TO]);
              else
                fprintf(fp,"<LI><STRONG>%s:</STRONG>",
                        lang[MSG_IN_REPLY_TO]);
              fprintf(fp," <A HREF=\"%.4d.%s\">", email2->msgnum,
                      set_htmlsuffix);
              fprintf(fp,"%s: \"%s\"</A>\n",
                      email2->name,
                      ptr=convchars(email2->subject));
              if(ptr)
                free(ptr);
            }
          }
    
          /*
          ** Is there a message next in the thread?
          */
    
          printcomment(fp, "nextthread", "start");
          
          email2 = nextinthread(email->msgnum);
          if (email2) {
            fprintf(fp, "<LI><STRONG>%s:</STRONG> ",
                    lang[MSG_NEXT_IN_THREAD]); 
            fprintf(fp, "<A HREF=\"%.4d.%s\">", 
                    email2->msgnum, set_htmlsuffix);
            fprintf(fp, "%s: \"%s\"</A>\n",
                    email2->name,
                    ptr=convchars(email2->subject));
            if(ptr)
              free(ptr);
          }
            
          /*
          ** Does this message have replies? If so, print them all!
          */
          
          if (set_showreplies) {
            for (rp = replylist; rp != NULL; rp = rp->next) {
              if (rp->msgnum == num) {
                if (rp->maybereply)
                  fprintf(fp,"<LI><STRONG>%s:</STRONG>",
                          lang[MSG_MAYBE_REPLY]);
                else
                  fprintf(fp, "<LI><STRONG>%s:</STRONG>",
                          lang[MSG_REPLY]);
                fprintf(fp," <A HREF=\"%.4d.%s\">",
                        rp->frommsgnum,
                        set_htmlsuffix);
                fprintf(fp,"%s: \"%s\"</A>\n",
                        rp->data->name,
                        ptr =convchars(rp->data->subject));
                free(ptr);
              }
            }
            printcomment(fp, "reply", "end");
          }
        }

        fprintf(fp, "</UL>\n");


        /*
        ** Finally...print the body!
        */

        printbody(fp, bp, email->msgid, email->subject);

        /*
        ** Should we print out the message links ?
        */

        if (set_show_msg_links) {
          fprintf(fp, "<P>\n<UL>\n");
    
          printcomment(fp, "next", "start");
    
          status = hashnumlookup(num + 1, &email2);

          if (status != NULL) {
            fprintf(fp,"<LI><STRONG>%s:</STRONG> ",lang[MSG_NEXT_MESSAGE]);
            fprintf(fp,"<A HREF=\"%.4d.%s\">%s: \"%s\"</A>\n",
                    num + 1, set_htmlsuffix, email2->name, 
                    ptr =convchars(email2->subject));
            free(ptr);
          }
    
          status = hashnumlookup(num - 1, &email2);
          if (status) {
            fprintf(fp,"<LI><STRONG>%s:</STRONG> ",
                    lang[MSG_PREVIOUS_MESSAGE]);
            fprintf(fp,"<A HREF=\"%.4d.%s\">%s: \"%s\"</A>\n",
                    num - 1, set_htmlsuffix, email2->name, 
                    ptr=convchars(email2->subject));
            free(ptr);
          }
          
          if (*email->inreplyto) {
            email2 = hashreplylookup(email->msgnum, email->inreplyto, &subjmatch);
            if (email2) {
              if (subjmatch)
                fprintf(fp,"<LI><STRONG>%s:</STRONG>",
                        lang[MSG_MAYBE_IN_REPLY_TO]);
              else
                fprintf(fp,"<LI><STRONG>%s:</STRONG>",
                        lang[MSG_IN_REPLY_TO]);
              fprintf(fp," <A HREF=\"%.4d.%s\">",
                      email2->msgnum, set_htmlsuffix);
              fprintf(fp,"%s: \"%s\"</A>\n", email2->name,
                      ptr=convchars(email2->subject));
              free(ptr);
            }
          }
    
          printcomment(fp, "nextthread", "start");
    
          email2 = nextinthread(email->msgnum);
          if (email2) {
            fprintf(fp, "<LI><STRONG>%s:</STRONG> ",
                    lang[MSG_NEXT_IN_THREAD]); 
            fprintf(fp, "<A HREF=\"%.4d.%s\">", 
                    email2->msgnum, set_htmlsuffix);
            fprintf(fp, "%s: \"%s\"</A>\n",
                    email2->name,
                    ptr=convchars(email2->subject));
            if(ptr)
              free(ptr);
          }

          if (set_showreplies) {
            /* FIXME:
               This should be cleaned up to use threadprint.c functions */

            for (rp = replylist; rp != NULL; rp = rp->next) {
              if (rp->msgnum == num) {
                if (rp->maybereply)
                  fprintf(fp, "<LI><STRONG>%s:</STRONG>",
                          lang[MSG_MAYBE_REPLY]);
                else
                  fprintf(fp, "<LI><STRONG>%s:</STRONG>",
                          lang[MSG_REPLY]);
                fprintf(fp, " <A HREF=\"%.4d.%s\">", 
                        rp->frommsgnum, set_htmlsuffix);
                fprintf(fp, "%s: \"%s\"</A>\n",
                        rp->data->name,
                        ptr=convchars(rp->data->subject));
                free(ptr);
              }
            }
            printcomment(fp, "reply", "end");
          }
          fprintf(fp, "</UL>\n");
        }

        if (set_usetable)
            fprint_menu(fp, NO_INDEX, about, "",
                        email->msgid, email->subject,PAGE_BOTTOM);

        printfooter(fp, mhtmlfooterfile, set_label, set_dir,
                    email->subject, filename);

        fclose(fp);

        if (newfile && chmod(filename, set_filemode) == -1) {
            sprintf(errmsg, "%s \"%s\": %o.", 
                lang[MSG_CANNOT_CHMOD],filename, set_filemode);
            progerr(errmsg);
        }

        if (maxnum && !(num % 5) && set_showprogress) {
            printf("\b\b\b\b%03.0f%c",
		   ((float) num / (float) maxnum) * 100,'%');
            fflush(stdout);
        }

        num++;
    }
    if (set_showprogress)
        printf("\b\b\b\b    \n");
}


/*
** Write the date index...
*/

void writedates(char *dir, char *label, char *archives, char *about,
		int amountmsgs)
{
    int newfile;
    char filename[MAXFILELEN];
    FILE *fp;

    sprintf(filename, "%s%s%s", dir,
	(dir[strlen(dir) - 1] == '/') ? "" : "/", datename);

    if (isfile(filename))
        newfile = 0;
    else
        newfile = 1;

    if ((fp = fopen(filename, "w")) == NULL) {
        sprintf(errmsg, "%s \"%s\".", lang[MSG_COULD_NOT_WRITE],filename);
        progerr(errmsg);
    }

    if (set_showprogress)
        printf("%s \"%s\"...", lang[MSG_WRITING_DATE_INDEX],filename);

    /*
    ** Print out the index file header 
    */
    print_index_header(fp, set_label, set_dir, lang[MSG_BY_DATE], datename);

    if (!set_usetable) {
        /* 
        ** Print out archive information links at the top of the index
        */ 
        print_index_header_links(fp, set_archives, set_about, FROM_DATE,amountmsgs);
    
        if (set_showhr)
            fprintf(fp, "<HR>\n");

        /*
        ** Printout the Dates for the Starting and Ending messages 
        ** in the archive, along with a count of the messages.
        */

        fprintf(fp, "<STRONG>%s:</STRONG> <EM>%s</EM><BR>\n", 
                    lang[MSG_STARTING], getdatestr(firstdatenum));
        fprintf(fp, "<STRONG>%s:</STRONG> <EM>%s</EM><BR>\n", 
                    lang[MSG_ENDING], getdatestr(lastdatenum));
    }
    else {
        fprint_menu(fp, DATE_INDEX, about, archives, "", "", PAGE_TOP);
        fprint_summary(fp, firstdatenum, lastdatenum, amountmsgs);
        if (set_showhr)
            fprintf(fp, "<HR>\n");
    }

    /*
    ** Print out the actual message index lists. Here's the beef.
    */

    if (set_indextable) {
	fprintf(fp, "<center><table><tr><td><u><strong>%s</strong></u></td><td><u><strong>%s</strong></u></td><td><u><strong> %s</strong></u></td></tr>\n",
		lang[MSG_CSUBJECT], lang[MSG_CAUTHOR], lang[MSG_CDATE]);
    	printdates(fp, datelist);
	fprintf(fp, "</table></center>\n<p>\n");
    }
    else {
    	fprintf(fp, "<UL>\n");
    	printdates(fp, datelist);
    	fprintf(fp, "</UL>\n<P>\n");
    }

    if (!set_usetable) {
        /*
        ** Printout the dates for the last message in the archive
        ** along with the date the archive was generated on.
        */

        fprintf(fp,"<A NAME=\"end\"><STRONG>%s:</STRONG></A> ",
                         lang[MSG_LAST_MESSAGE_DATE]);
        fprintf(fp,"<EM>%s</EM><BR>\n", getdatestr(lastdatenum));
        fprintf(fp,"<STRONG>%s:</STRONG> <EM>%s</EM><P>\n",
                         lang[MSG_ARCHIVED_ON], getlocaltime());
        if (set_showhr)
            fprintf(fp, "<HR>\n");

        /* 
        ** Print out archive information links at the bottom of the index
        */ 
        print_index_footer_links(fp, archives, FROM_DATE,amountmsgs);
    }
    else {
        fprint_menu(fp, DATE_INDEX, about, archives, "", "", PAGE_BOTTOM);
    }

    /* 
    ** Print the index page footer.
    */
    printfooter(fp, ihtmlfooterfile, set_label, set_dir, lang[MSG_BY_DATE],
                datename);

    fclose(fp);

    if (newfile && chmod(filename, set_filemode) == -1) {
        sprintf(errmsg, "%s \"%s\": %o.", 
                lang[MSG_CANNOT_CHMOD],filename, set_filemode);
        progerr(errmsg);
    }

    if (set_showprogress)
        putchar('\n');
}

#if 0
/*
** While printing the thread index, prints any replies to replies.
*/

void checkreplies(FILE *fp, int num, int level)
{
  struct reply *rp;
  char *ptr;
  
  for (rp = replylist; rp != NULL; rp = rp->next) {
    if (rp->msgnum == num) {
      if (level < set_thrdlevels)
        fprintf(fp, "<UL>\n");
      fprintf(fp,"<LI><A HREF=\"%.4d.%s\">",
              rp->frommsgnum, set_htmlsuffix);
      printedlist = (struct printed *)
        markasprinted(printedlist, rp->frommsgnum);
      fprintf(fp,"<STRONG>%s</STRONG></A> ",
              ptr=convchars(rp->data->subject));
      free(ptr);
      fprintf(fp, "<A NAME=\"%d\"><EM>%s</EM></A>\n",
              rp->frommsgnum, rp->data->name);
      checkreplies(fp, rp->frommsgnum, level + 1);
      if (level < set_thrdlevels)
        fprintf(fp, "</UL>\n");
    }
  }
}

/*
** Prints the article pointers in the thread index by date.
*/

void printthreads(FILE *fp, struct header *hp)
{
  int hasreply;
  struct reply *rp;

  if (hp != NULL) {
    printthreads(fp, hp->left);

    for (hasreply = 0, rp = replylist; rp != NULL; rp = rp->next) {
      if (rp->frommsgnum == hp->data->msgnum) {
        hasreply = 1;
        break;
      }
    }

    if (!hasreply &&
        !wasprinted(printedlist, hp->data->msgnum)) {
      char *ptr;
      fprintf(fp, "<LI><A HREF=\"%.4d.%s\"><STRONG>%s</STRONG></A> ",
              hp->data->msgnum,
              set_htmlsuffix,
              ptr=convchars(hp->data->subject));
      free(ptr);
#if 0
      fprintf(fp,"<A NAME=\"%d\"><EM>%s</EM></A>\n",hp->msgnum,hp->name);
#endif
#if 1
      fprintf(fp,"<A NAME=\"%d\"><EM>%s</EM></A> <small>(%s)</small>\n",
              hp->data->msgnum, hp->data->name, getdatestr(hp->data->date));
#endif
      printedlist = (struct printed *)
        markasprinted(printedlist, hp->data->msgnum);

      checkreplies(fp, hp->data->msgnum, 1);
    }
    printthreads(fp, hp->right);
  }
}
#endif

/*
** Write the thread index...
*/

void writethreads(char *dir, char *label, char *archives, char *about,
		  int amountmsgs)
{
    int newfile;
    char filename[MAXFILELEN];
    FILE *fp;

    struct printed *pp;

    while (printedlist) { /* cleanup needed ?? */
         pp = printedlist;
         printedlist = printedlist->next;
         free(pp);
    }

    printedlist = NULL;

    sprintf(filename, "%s%s%s", 
            dir, (dir[strlen(dir) - 1] == '/') ? "" : "/", thrdname);

    if (isfile(filename))
        newfile = 0;
    else
        newfile = 1;

    if ((fp = fopen(filename, "w")) == NULL) {
        sprintf(errmsg, "%s \"%s\".", lang[MSG_COULD_NOT_WRITE],filename);
        progerr(errmsg);
    }

    if (set_showprogress)
        printf("%s \"%s\"...", lang[MSG_WRITING_THREAD_INDEX],filename);

    print_index_header(fp, label, dir, lang[MSG_BY_THREAD], thrdname);

    if (!set_usetable) {
        /* 
        ** Print out the index page links 
        */ 
        print_index_header_links(fp, archives, about, FROM_THREAD,amountmsgs);
        
        if (set_showhr)
            fprintf(fp, "<HR>\n");
    
        fprintf(fp,"<STRONG>%s:</STRONG> <EM>%s</EM><BR>\n",
                   lang[MSG_STARTING], getdatestr(firstdatenum));
        fprintf(fp,"<STRONG>%s:</STRONG> <EM>%s</EM><BR>\n", 
                   lang[MSG_ENDING], getdatestr(lastdatenum));
    }
    else {
        fprint_menu(fp, THREAD_INDEX, about, archives, "", "", PAGE_TOP);
        fprint_summary(fp, firstdatenum, lastdatenum, amountmsgs);
        if (set_showhr)
            fprintf(fp, "<HR>\n");
    }
#if 0
    fprintf(fp, "<UL>\n");
    printthreads(fp, datelist);
    fprintf(fp, "</UL>\n<P>\n");

    fprintf(fp, "<!--\n");
    fprintf(fp, "-->\n");
#endif
    if (set_indextable) {
	fprintf(fp, "<center><table><tr><td><u><strong>%s</strong></u></td><td><u><strong>%s</strong></u></td><td><u><strong> %s</strong></u></td></tr>\n",
		lang[MSG_CSUBJECT], lang[MSG_CAUTHOR], lang[MSG_CDATE]);
    	print_all_threads(fp);
	fprintf(fp, "</table></center>\n<p>\n");
    }
    else {
      fprintf(fp, "<UL>\n");
      print_all_threads(fp);
      fprintf(fp, "</UL>\n<P>\n");
    }

    if (!set_usetable) {
        fprintf(fp,"<A NAME=\"end\"><STRONG>%s:</STRONG></A> ",
                   lang[MSG_LAST_MESSAGE_DATE]);
        fprintf(fp,"<EM>%s</EM><BR>\n", getdatestr(lastdatenum));
        fprintf(fp,"<STRONG>%s:</STRONG> <EM>%s</EM><P>\n",
                   lang[MSG_ARCHIVED_ON], getlocaltime());
             
        if (set_showhr)
            fprintf(fp, "<HR>\n");
    
        /* 
        ** Print out archive information links at the bottom of the index
        */ 
        print_index_footer_links(fp, archives, FROM_THREAD,amountmsgs);
    
    }
    else {
        fprint_menu(fp, THREAD_INDEX, about, archives, "", "", PAGE_BOTTOM);
    }

    printfooter(fp, ihtmlfooterfile, set_label, set_dir, lang[MSG_BY_THREAD],
                thrdname);

    fclose(fp);

    if (newfile && chmod(filename, set_filemode) == -1) {
        sprintf(errmsg, "%s \"%s\": %o.", 
                lang[MSG_CANNOT_CHMOD],filename, set_filemode);
        progerr(errmsg);
    }

    if (set_showprogress)
        putchar('\n');
}

/*
** Print the subject index pointers alphabetically.
*/

void printsubjects(FILE *fp, struct header *hp, char **oldsubject)
{
  char *subj;

  if (hp != NULL) {
    printsubjects(fp, hp->left, oldsubject);

    subj = convchars(hp->data->unre_subject);

    if (strcasecmp(hp->data->unre_subject, *oldsubject)) {
      if(set_indextable) {
        fprintf(fp, "<tr><td colspan=3><strong>%s</strong></td></tr>\n", subj);
      }
      else {
        fprintf(fp, "<LI><STRONG>%s</STRONG>\n", subj);
      }
    }
    if(set_indextable) {
      fprintf(fp, "<tr><td>&nbsp;</td>"
              "<td><A HREF=\"%.4d.%s\">%s</A></td>"
              "<td><A NAME=\"%d\">%s</A></td></tr>\n",
              hp->data->msgnum, set_htmlsuffix, hp->data->name,
              hp->data->msgnum, getdatestr(hp->data->date));
    }
    else {
      fprintf(fp, "<UL><LI><A HREF=\"%.4d.%s\">%s</A> "
              "<A NAME=\"%d\"><EM>(%s)</EM></A></UL>\n",
              hp->data->msgnum, set_htmlsuffix, hp->data->name,
              hp->data->msgnum, getdatestr(hp->data->date));
    }

    *oldsubject = hp->data->unre_subject;

    free(subj);

    printsubjects(fp, hp->right, oldsubject);
  }
}

/*
** Prints the subject index.
*/

void writesubjects(char *dir, char *label, char *archives, char *about,
		   int amountmsgs)
{
    int newfile;
    char filename[MAXFILELEN];
    FILE *fp;

    sprintf(filename, "%s%s%s", 
            dir, (dir[strlen(dir) - 1] == '/') ? "" : "/", subjname);

    if (isfile(filename))
        newfile = 0;
    else
        newfile = 1;

    if ((fp = fopen(filename, "w")) == NULL) {
        sprintf(errmsg, "%s \"%s\".", lang[MSG_COULD_NOT_WRITE], filename);
        progerr(errmsg);
    }

    if (set_showprogress)
        printf("%s \"%s\"...", lang[MSG_WRITING_SUBJECT_INDEX],filename);

    print_index_header(fp, set_label, set_dir, lang[MSG_BY_SUBJECT], subjname);

    if (!set_usetable) {
        /* 
        ** Print out the index page links 
        */ 
        print_index_header_links(fp, set_archives, set_about, FROM_SUBJECT,amountmsgs);
    
        if (set_showhr)
            fprintf(fp, "<HR>\n");

        fprintf(fp, "<STRONG>%s:</STRONG> <EM>%s</EM><BR>\n",
                    lang[MSG_STARTING], getdatestr(firstdatenum));
        fprintf(fp, "<STRONG>%s:</STRONG> <EM>%s</EM><BR>\n", 
                    lang[MSG_ENDING], getdatestr(lastdatenum));
    }
    else {
        fprint_menu(fp, SUBJECT_INDEX, set_about, set_archives, "", "", PAGE_TOP);
        fprint_summary(fp, firstdatenum, lastdatenum, amountmsgs);
        if (set_showhr)
            fprintf(fp, "<HR>\n");
    }

    if (set_indextable) {
	fprintf(fp, "<center><table><tr><td><u><strong>%s</strong></u></td><td><u><strong>%s</strong></u></td><td><u><strong> %s</strong></u></td></tr>\n",
		lang[MSG_CSUBJECT], lang[MSG_CAUTHOR], lang[MSG_CDATE]);
    }
    else {
    	fprintf(fp, "<UL>\n");
    }
    {
      char *oldsubject=""; /* dummy to start with */
      printsubjects(fp, subjectlist, &oldsubject);
    }
    if (set_indextable) {
	fprintf(fp, "</table></center>\n<p>\n");
    }
    else {
    	fprintf(fp, "</UL>\n<P>\n");
    }

    if (!set_usetable) {
        fprintf(fp, "<STRONG>%s:</STRONG> <EM>%s</EM><BR>\n",
                    lang[MSG_LAST_MESSAGE_DATE], getdatestr(lastdatenum));
        fprintf(fp, "<STRONG>%s:</STRONG> <EM>%s</EM><P>\n", 
                    lang[MSG_ARCHIVED_ON], getlocaltime());

        if (set_showhr)
            fprintf(fp, "<HR>\n");

        /* 
        ** Print out archive information links at the bottom of the index
        */ 
        print_index_footer_links(fp, set_archives, FROM_SUBJECT,amountmsgs);
    }
    else {
        fprint_menu(fp, SUBJECT_INDEX, set_about, set_archives, "", "", PAGE_BOTTOM);
    }

    printfooter(fp, ihtmlfooterfile, set_label, set_dir, lang[MSG_BY_SUBJECT],
                subjname);

    fclose(fp);

    if (newfile && chmod(filename, set_filemode) == -1) {
        sprintf(errmsg, "%s \"%s\": %o.", 
                lang[MSG_CANNOT_CHMOD],filename, set_filemode);
        progerr(errmsg);
    }

    if (set_showprogress)
        putchar('\n');
}

/*
** Prints the author index links sorted alphabetically.
*/

void printauthors(FILE *fp, struct header *hp, char **oldname)
{
  char *subj;

  if (hp != NULL) {
    printauthors(fp, hp->left, oldname);

    subj = convchars(hp->data->subject);

    if (strcasecmp(hp->data->name, *oldname)) {
      if(set_indextable)
        fprintf(fp, "<tr><td colspan=3><STRONG>%s</STRONG></td></tr>",
                hp->data->name);
      else
        fprintf(fp, "<LI><STRONG>%s</STRONG>\n", hp->data->name);
    }

    if(set_indextable) {
      fprintf(fp,"<tr><td>&nbsp;</td><td><A HREF=\"%.4d.%s\">%s</A></td>"
              "<td><A NAME=\"%d\">%s</A></td></tr>\n",
              hp->data->msgnum,
              set_htmlsuffix, subj,
              hp->data->msgnum,
	      getdatestr(hp->data->date));
    }
    else {
      fprintf(fp,"<UL><LI><A HREF=\"%.4d.%s\">%s</A>&nbsp;"
              "<A NAME=\"%d\"><EM>(%s)</EM></A></UL>\n",
              hp->data->msgnum,
              set_htmlsuffix, subj,
              hp->data->msgnum,
	      getdatestr(hp->data->date));
    }
    free(subj);

    *oldname = hp->data->name; /* avoid copying */
    printauthors(fp, hp->right, oldname);
  }
}

/*
** Prints the author index file and links sorted alphabetically.
*/

void writeauthors(char *dir, char *label, char *archives, char *about,
		  int amountmsgs)
{
    int newfile;
    char filename[MAXFILELEN];
    FILE *fp;

    sprintf(filename, "%s%s%s", 
            dir, (dir[strlen(dir) - 1] == '/') ? "" : "/", authname);

    if (isfile(filename))
        newfile = 0;
    else
        newfile = 1;

    if ((fp = fopen(filename, "w")) == NULL) {
        sprintf(errmsg, "%s \"%s\".", lang[MSG_COULD_NOT_WRITE], filename);
        progerr(errmsg);
    }

    if (set_showprogress)
        printf("%s \"%s\"...", lang[MSG_WRITING_AUTHOR_INDEX], filename);

    print_index_header(fp, set_label, set_dir, lang[MSG_BY_AUTHOR], authname);

    if (!set_usetable) {
        /* 
        ** Print out the index page links 
        */ 
        print_index_header_links(fp, set_archives, set_about, FROM_AUTHOR,amountmsgs);
        
        if (set_showhr)
            fprintf(fp, "<HR>\n");
    
        fprintf(fp,"<STRONG>%s:</STRONG> <EM>%s</EM><BR>\n",
                   lang[MSG_STARTING], getdatestr(firstdatenum));
        fprintf(fp,"<STRONG>%s:</STRONG> <EM>%s</EM><BR>\n", 
                   lang[MSG_ENDING], getdatestr(lastdatenum));
    }
    else {
        fprint_menu(fp, AUTHOR_INDEX, about, archives, "", "", PAGE_TOP);
        fprint_summary(fp, firstdatenum, lastdatenum, amountmsgs );        
        if (set_showhr)
            fprintf(fp, "<HR>\n");
    }

    if (set_indextable) {
	fprintf(fp, "<center><table><tr><td><u><strong>%s</strong></u></td><td><u><strong>%s</strong></u></td><td><u><strong> %s</strong></u></td></tr>\n",
		lang[MSG_CAUTHOR], lang[MSG_CSUBJECT], lang[MSG_CDATE]);
    }
    else {
    	fprintf(fp, "<UL>\n");
    }
    {
      char *prevauthor="";
      printauthors(fp, authorlist, &prevauthor);
    }
    if (set_indextable) {
	fprintf(fp, "</table></center>\n<p>\n");
    }
    else {
    	fprintf(fp, "</UL>\n<P>\n");
    }

    if (!set_usetable) {
        fprintf(fp, "<STRONG>%s:</STRONG> <EM>%s</EM><BR>\n",
                    lang[MSG_LAST_MESSAGE_DATE], getdatestr(lastdatenum));
        fprintf(fp,"<STRONG>%s:</STRONG> <EM>%s</EM><P>\n",
                    lang[MSG_ARCHIVED_ON], getlocaltime());

        if (set_showhr)
            fprintf(fp, "<HR>\n");

        /* 
        ** Print out archive information links at the bottom of the index
        */ 
        print_index_footer_links(fp, archives, FROM_AUTHOR,amountmsgs);
    }
    else {
        fprint_menu(fp, AUTHOR_INDEX, set_about, archives, "", "", PAGE_BOTTOM);
    }

    printfooter(fp, ihtmlfooterfile, set_label, set_dir, lang[MSG_BY_AUTHOR],
                authname);

    fclose(fp);

    if (newfile && chmod(filename, set_filemode) == -1) {
        sprintf(errmsg, "%s \"%s\": %o.", 
                lang[MSG_CANNOT_CHMOD],filename, set_filemode);
        progerr(errmsg);
    }

    if (set_showprogress)
        putchar('\n');
}
