/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: stage1.cpp,v 1.1.2.2 2004/07/09 02:02:43 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */
/***********************************************************************
 * stage1.cpp 
 *
 * A simple self-extracting installer 
 *
 ***********************************************************************
 *
 * Create the tarball (or whatever archive type you're using).
 * Determine the size of the tarball.
 * Define ARCHIVE_SIZE in archive_info.h to be the tarball size.
 * Define ARCHIVE_TYPE_TGZ (or one of the others) in archive_info.h
 *     if the tarball is a .tgz archive.
 * Compile this program.
 * When run, it untarrs the tarball and attempts to run a second-stage
 *     installer inside it.
 * The second-stage installer may be called setup (binary), setup.sh
 *     bash shell), setup.pl (perl), or setup.py (python), and it will 
 *     run the first it finds and exit.
 *
 * KEEP THIS THING GENERIC AND VERY SIMPLE , putting package-specific items
 *     in the second-stage installer.  The second-stage installer
 *     is responsible for all cleanup.
 *
 ***********************************************************************/

#include "hxtypes.h"

#ifdef _WIN32
#include <process.h>
#include <direct.h>
#ifndef mkdir
#define mkdir(n,p) \
    mkdir(n)
#endif /* mkdir */

#elif defined _UNIX
#if defined(_FREEBSD) || defined(_OPENBSD) || defined(_NETBSD) || defined(_MAC_UNIX)
#define TCGETS TIOCGETA
#define TCSETS TIOCSETA
#elif defined _AIX43
#include <sys/termio.h>
#endif /* _FREEBSD || _OPENBSD || _NETBSD || _MAC_UNIX */

#include <termios.h>
#include <unistd.h>
#include <sys/ioctl.h>
#endif /* _WIN32 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "package_info.h"
#include "stage1.h"
#include "prodinstlib.h"


/***********************************************************************
 * Defines
 */
#ifdef _WIN32
#define PATH_SEP "\\"
#else
#define PATH_SEP "/"
#endif

#ifndef DESTDIR
#define DESTDIR "." PATH_SEP "hxsetup"
#endif

#ifndef SECOND_STAGE_PROG
#define SECOND_STAGE_PROG "Bin" PATH_SEP "setup"
#endif

#ifndef TMP_EXTRACT_PROG
#define TMP_EXTRACT_PROG "xxextract.tmp"
#endif

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif


/***********************************************************************
 * Globals
 */
char* g_szTmpDir = (char*)DESTDIR;
char* g_szProg=0;

#ifdef _CONSOLE
    struct termios g_oldtty;
#endif //_CONSOLE

Stage1::Stage1() : m_ulInstFlags(0)
{
    memset(m_szUserPass, 0, MAX_PASS);
}

Stage1::~Stage1()
{
}


#ifdef _CONSOLE
/************************************************************************
 * Stage1::Main - first stage executable of the self-extracting installer
 *
 * Basic idea:
 *   Open the binary file we're currently executing (argv[0]).
 *   Seek to the end minus the size of the archive (.tar.gz/whatever) file.
 *   Read the data and pipe it to the extraction utility.
 *   Run the second-stage installer if found in the archive.
 */
int
Stage1::Main(int nArchiveSize, int nArchiveProgSize, char* szArchiveType,
             int argc, char* argv[], const char* szPassword)
{
    int nRet = 0;
    char* pArchiveData = new char[nArchiveSize + 1];
    char* pArchiveProgData = new char[nArchiveProgSize + 1];
    FILE* fp=0;
    int nBytes=0;
    char szCmd[1024];
    bool bUsePipe = TRUE;
    const char* szTmpArchive=0;
    char szExtractProg[1024];
    szExtractProg[0] = '\0';
    m_szPassword = szPassword;

    ParseCmdLine(argc, argv);

    // Check the password
    if(m_szPassword && strcmp(m_szUserPass, m_szPassword) != 0 &&
        ((m_ulInstFlags & INST_NON_INTERACTIVE) || !PasswordPrompt()))
    {
        Bail(0, NULL);
    }

    if(!(m_ulInstFlags & INST_SILENT))
    {
        printf("Extracting files for %s installation...", 
	       PackageInfo::Name());
        fflush(0);
    }

    fp = fopen(argv[0], "r");

    // if archive prog is included in the archive, extract it
    if (nArchiveProgSize)
    {
        fseek(fp, -(nArchiveSize + nArchiveProgSize), SEEK_END);
        nBytes = fread(pArchiveData, sizeof(char), nArchiveProgSize, fp);
        if (nBytes != nArchiveProgSize)
        {
            if(!(m_ulInstFlags & INST_SILENT))
            {
                printf("\nError: short read (%d/%d bytes) (%s)\n",
                        nBytes, nArchiveProgSize, strerror(errno));
            }
            nRet = -1;
        }
        unlink(TMP_EXTRACT_PROG);
        FILE* fp2 = fopen(TMP_EXTRACT_PROG, "w");
        fwrite(pArchiveData, sizeof(char), nArchiveProgSize, fp2);
        fclose (fp2);
        chmod (TMP_EXTRACT_PROG, 0755);
        char szCurDir[1024];
        memset (szCurDir, 0, 1024);
        getcwd(szCurDir, 1023);
        sprintf(szExtractProg, "%s%s%s", szCurDir, PATH_SEP, TMP_EXTRACT_PROG);
    }

    // read the archive file (.rzt, .tgz, whatever)
    fseek(fp, -(nArchiveSize), SEEK_END);
    nBytes = fread(pArchiveData, sizeof(char), nArchiveSize, fp);
    fclose(fp);

    if (nBytes != nArchiveSize)
    {
        if(!(m_ulInstFlags & INST_SILENT))
        {
            printf("\nError: short read (%d/%d bytes) (%s)\n",
                nBytes, nArchiveSize, strerror(errno));
        }
        nRet = -1;
    }

    system ("rm -rf " DESTDIR);
    if (mkdir(DESTDIR, 0700) != 0)
        Bail(-1, "\nError: mkdir failed");
    if (chmod(DESTDIR, 0700) != 0)
        Bail(-1, "\nError: chmod failed");

#if defined(_OPENBSD)
    const char* szTarOpts = "xf";
#else
    const char* szTarOpts = "xof";
#endif

    if (strcmp(szArchiveType, "bz2") == 0)
    {
        // TAR+BZIP2
        // should work on any unix system with tar
        sprintf(szCmd, "%s -dc | (cd %s; tar %s -)", szExtractProg, DESTDIR, szTarOpts);
    }
    else if (strcmp(szArchiveType, "tgz") == 0)
    {
        // TAR+GZIP
        // should work on any unix system with gunzip and tar
        sprintf(szCmd, "gunzip -c | (cd %s; tar %s -)", DESTDIR, szTarOpts);
    }
    else if (strcmp(szArchiveType, "tz") == 0)
    {
        // TAR+COMPRESS
        // archives are pretty big but should work on any unix system with tar
        sprintf(szCmd, "uncompress | (cd %s; tar %s -)", DESTDIR, szTarOpts);
    }
    else if (strcmp(szArchiveType, "zip") == 0)
    {
        // ZIP
        // XXX untested, experimental, unfinished, don't use yet...
        bUsePipe = FALSE;
        szTmpArchive = "rsarchive.zip";
        sprintf(szCmd, "cd %s; %s %s; rm -f %s\n",
                DESTDIR, szExtractProg, szTmpArchive, szTmpArchive);
    }
    else
    {
        Bail (-1, "\nError: Internal error -- unknown archive type\n");
    }

    if (bUsePipe)
    {
        fp = popen(szCmd, "w");
    }
    else
    {
        char szFile[1024];
        sprintf (szFile, "%s%s%s", DESTDIR, PATH_SEP, szTmpArchive);
        fp = fopen(szFile, "w");
    }

    // we read it in 20 chunks so it's easy to get "."'s printed
    int nWriteSize = nArchiveSize / 20;
    for (int nPos = 0; nPos < nBytes; nPos += nArchiveSize / 20)
    {
        if(!(m_ulInstFlags & INST_SILENT))
        {
            printf(".");
            fflush(0);
        }
        if (nPos + nArchiveSize / 20 > nArchiveSize)
        {
            nWriteSize = nArchiveSize - nPos;
        }
        fwrite(pArchiveData + nPos, sizeof(char), nWriteSize, fp);
    }
    if (bUsePipe)
    {
        pclose(fp);
    }
    else
    {
        fclose(fp);
        if(!(m_ulInstFlags & INST_SILENT))
        {
            //printf ("\n(%s)\n", szCmd);
            printf ("\nDecompressing files... (please wait)\n");
        }
        system(szCmd);
    }

    if (nArchiveProgSize)
    {
        unlink(TMP_EXTRACT_PROG);
    }

    if(!(m_ulInstFlags & INST_SILENT))
    {
        printf("\n");
    }
    RunSecondStage(argc, argv); //only returns if second stage installer not found
    exit(nRet);
    return 0; //shut-up HP
}


/************************************************************************
 * Stage1::RunSecondStage - invoke the second-stage installer
 *
 * Note, does not return if successful. Second-stage installer
 * is responsible for all cleanup.
 */
void
Stage1::RunSecondStage(int argc, char* argv[])
{
    char szCmd[1024];
    struct stat info;
    char** newargv = new char* [argc + 1];
    memcpy (newargv, argv, (argc+1) * sizeof(char*));


    //printf("Checking for second stage installer...");
    fflush(0);

    if (chdir(DESTDIR) == 0)
    {
        //
        // check for install binary first
        //
        sprintf(szCmd, ".%s%s", PATH_SEP, SECOND_STAGE_PROG);
#ifdef _WIN32 //win32 not really supported yet...maybe someday
        strcat(szCmd, ".exe");
#endif
        if (stat(szCmd, &info) == 0)
        {
            chmod(szCmd, 0755);
            newargv[0] = szCmd;
            execvp(szCmd, newargv);
        }

        //
        // check for install scripts in various languages next
        //
        sprintf(szCmd, ".%s%s.sh", PATH_SEP, SECOND_STAGE_PROG);
        if (stat (szCmd, &info) == 0)
        {
            newargv[0] = szCmd;
            execvp("bash", newargv);
        }

        sprintf(szCmd, ".%s%s.pl", PATH_SEP, SECOND_STAGE_PROG);
        if (stat (szCmd, &info) == 0)
        {
            newargv[0] = szCmd;
            execvp("perl", newargv);
        }

        sprintf(szCmd, ".%s%s.py", PATH_SEP, SECOND_STAGE_PROG);
        if (stat (szCmd, &info) == 0)
        {
            newargv[0] = szCmd;
            execvp("python", newargv);
        }
    }

    delete[] newargv;

    if(!(m_ulInstFlags & INST_SILENT))
    {
        printf("Warning: second stage installer not found!\n");
        fflush(0);
    }
}
#endif // _CONSOLE


/************************************************************************
 * Stage1::Bail - print a message and exit
 */
void
Stage1::Bail (int nRet, const char* szMessage)
{
    if (!(m_ulInstFlags & INST_SILENT) && szMessage)
    {
        printf("%s", szMessage);
        fflush(0);
    }
    exit(nRet);
}


/************************************************************************
 * Stage1::PrintVersion - Print the version
 */
void
Stage1::PrintVersion(void)
{
    printf("%s\n", ProductInstaller::VersionBanner());
}


/************************************************************************
 * Stage1::PrintUsage - Print command-line usage information
 */
void
Stage1::PrintUsage(const char* szProg)
{
    fprintf(stderr, "\nUsage: %s [options]\n"
        "\nWhere options are:\n%s\n", szProg, 
        PackageInfo::CommandLineHelpString(m_szPassword ? TRUE : FALSE));
}


/************************************************************************
 * Stage1::ParseCmdLine - Parse the command-line options
 *
 * We pass our command-line options to the second-stage installer,
 * so most of them will be parsed by it.  We just grab -h and -v
 * here so we don't have to extract the archive before handling them.
 */
void
Stage1::ParseCmdLine(int argc, char* argv[])
{
    int i=0;
    char** arg=0;

    // strip off any path information
    g_szProg = argv[0] + strlen(argv[0]);
    while (g_szProg > argv[0] && *g_szProg != '/' && *g_szProg != '\\')
    {
        --g_szProg;
    }
    if (*g_szProg == '/' || *g_szProg == '\\')
    {
        ++g_szProg;
    }

    
    // First, scan all args for -v or -h; if there we want to exit now...
    for (i = argc, arg = argv; i; arg++, i--)
    {
        if (!strcmp(*arg, "--version") ||
            !strcmp(*arg, "-v"))
        {
            PrintVersion();
            Bail(-1,"");
        }
        if (!strcmp(*arg, "--help") ||
            !strcmp(*arg, "-h") ||
            !strcmp(*arg, "-?") ||
            !strcmp(*arg, "/?")) // for win32-heads
        {
            PrintVersion();
            PrintUsage();
            Bail(-1,"");
        }
    }

    for (i = argc, arg = argv; i; arg++, i--)
    {
        if(!strcmp(*arg, "--silent") ||
           !strcmp(*arg, "-s") ||
           !strcmp(*arg, "/s"))
        {
            m_ulInstFlags |= INST_SILENT | INST_NON_INTERACTIVE;
        }

        else if(!strcmp(*arg, "--progress-only") ||
                !strcmp(*arg, "-p"))
        {
            m_ulInstFlags |= INST_NON_INTERACTIVE;
        }

        else if(!strcmp(*arg, "--password"))
        {
            if(i > 1)
            {
                i--; 
                arg++;
                strncpy(m_szUserPass, *arg, MAX_PASS-1);
            }
        }
    }
    // any other args will be parsed by the second-stage installer
    // so ignore them

}

#ifdef _CONSOLE
void
Stage1::SetEchoOff(void)
{
    ioctl(0, TCGETS, &g_oldtty);
    struct termios tty = g_oldtty;
    tty.c_lflag &= ~(ICANON | ECHO | ECHOE | ECHOK | ECHONL);
    tty.c_cc[VMIN] = 1;
    tty.c_cc[VTIME] = 0;
    ioctl(0, TCSETS, &tty);
}

void
Stage1::SetEchoOn(void)
{
    ioctl(0, TCSETS, &g_oldtty);
}

BOOL
Stage1::PasswordPrompt(void)
{
    BOOL bMatch = FALSE;

    SetEchoOff();
    UINT8 i = 0;

    for(;;)
    {
        printf("Please enter the installation password: ");
        
        if(!fgets(m_szUserPass, MAX_PASS, stdin))
        {
            break;
        }

        // Trim the newline
        m_szUserPass[strlen(m_szUserPass) - 1] = '\0';
        if(strcmp(m_szUserPass, m_szPassword) == 0)
        {
            bMatch = TRUE;
            break;
        }

        if(++i < MAX_PASS_ATTEMPTS)
        {
            printf("\nThe password you entered is incorrect.\n");
            sleep(1);
        }
        else
        {
           printf("\nThe maximum number of installation attempts has "
                  "been exceeded.\n");
           break;
        }
    }

    printf("\n");
    SetEchoOn();
    return bMatch;
}
#endif // _CONSOLE

