/*
 *  PGGraphvizOptionController.m
 *  graphviz
 *
 *  Created by Glen Low on Fri Dec 26 2003.
 *  Copyright (c) 2003, Pixelglow Software. All rights reserved.
 *  http://www.pixelglow.com/graphviz/
 *  graphviz@pixelglow.com
 *
 *  Redistribution and use in source and binary forms, with or without modification, are permitted
 *  provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions
 *    and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
 *    and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *  * Neither the name of Pixelglow Software nor the names of its contributors may be used to endorse or
 *    promote products derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#import "PGFontColorTableView.h"
#import "PGFontTextFieldCell.h"
#import "PGGraphvizOptionController.h"
#import "PGGraphvizDocument.h"
#import "PGGraphvizSchema.h"
#import "PGNilColor.h"
#import "PGTableViewDataSource.h"
#import "PGSourceTableColumn.h"
#import "PGDescriptor.h"
#import "PGOption.h"

@implementation PGGraphvizOptionController

- (void) awakeFromNib
	{
		[[document layout] bindToPopUpButton: layout];

		[[document scale] bindToTextField: scale];
		[[document scale] bindToControl: scaleStepper];
	
		[[document position] bindToPopUpButton: position];

		[[document verbose] bindToControl: verbose];		
		[[document invertY] bindToControl: flipVertical];
		[[document reduceGraph] bindToControl: reduceGraph];

		NSArray* documentGraphDefaults = [document graphDefaults];
		NSArray* documentNodeDefaults = [document nodeDefaults];
		NSArray* documentEdgeDefaults = [document edgeDefaults];
		
		// source each table from the options in document
		graphDefaultSource_ = [[PGTableViewDataSource alloc] initWithDataSource: documentGraphDefaults];
		nodeDefaultSource_ = [[PGTableViewDataSource alloc] initWithDataSource: documentNodeDefaults];
		edgeDefaultSource_ = [[PGTableViewDataSource alloc] initWithDataSource: documentEdgeDefaults];
		
		[graphDefaults setDataSource: graphDefaultSource_];
		[nodeDefaults setDataSource: nodeDefaultSource_];
		[edgeDefaults setDataSource: edgeDefaultSource_];

		[graphDefaults setDelegate: self];
		[nodeDefaults setDelegate: self];
		[edgeDefaults setDelegate: self];
		
		// each option in each table will describe itself for its tooltip
		int index;
		for (index = 0; index < [documentGraphDefaults count]; ++index)
			[graphDefaults addToolTipRect: [graphDefaults rectOfRow: index]
				owner: [[documentGraphDefaults objectAtIndex: index] tip]
				userData: nil];
		for (index = 0; index < [documentNodeDefaults count]; ++index)
			[nodeDefaults addToolTipRect: [nodeDefaults rectOfRow: index]
				owner: [[documentNodeDefaults objectAtIndex: index] tip]
				userData: nil];
		for (index = 0; index < [documentEdgeDefaults count]; ++index)
			[edgeDefaults addToolTipRect: [edgeDefaults rectOfRow: index]
				owner: [[documentEdgeDefaults objectAtIndex: index] tip]
				userData: nil];
				
		// if no alternating row background available (pre-10.3), make sure both vertical and horizontal grid lines show
		if (![graphDefaults respondsToSelector: @selector (usesAlternatingRowBackgroundColors)])
			[graphDefaults setDrawsGrid: YES];
		if (![nodeDefaults respondsToSelector: @selector (usesAlternatingRowBackgroundColors)])
			[nodeDefaults setDrawsGrid: YES];
		if (![nodeDefaults respondsToSelector: @selector (usesAlternatingRowBackgroundColors)])
			[nodeDefaults setDrawsGrid: YES];

		NSNotificationCenter* defaultCenter = [NSNotificationCenter defaultCenter];
						
		layout_ = [document layout];
		[defaultCenter
			addObserver: self
			selector: @selector (layoutOptionDidChange:)
			name: OptionDidChange
			object: layout_];

		scale_ = [document scale];
		[defaultCenter
			addObserver: self
			selector: @selector (scaleOptionDidChange:)
			name: OptionDidChange
			object: scale_];			

		BOOL isNeatoLayout = [[layout_ stringValue] isEqual: @"neato"];
		[position setEnabled: isNeatoLayout];
		[reduceGraph setEnabled: isNeatoLayout];
	}

- (void) layoutOptionDidChange: (NSNotification*) notification
	{
		BOOL isNeatoLayout = [[layout_ stringValue] isEqual: @"neato"];
		[position setEnabled: isNeatoLayout];
		[reduceGraph setEnabled: isNeatoLayout];
	}

- (void) scaleOptionDidChange: (NSNotification*) notification
	{
		id scaleValue = [scale_ objectValue];
		[scale setObjectValue: scaleValue];
		[scaleStepper setObjectValue: scaleValue];
	}

- (void) tableView: (NSTableView*) table updateColor: (id) sender
	{
		id selectedObjectValue = nil;
		int selectedRow = [table selectedRow];
		if (selectedRow != -1)
			{
				NSTableColumn* objectValueColumn = [table tableColumnWithIdentifier: @"objectValue"];
				if ([[objectValueColumn dataCellForRow: selectedRow] tag] == PGColorTag)
					selectedObjectValue = [[table dataSource]
						tableView: table
						objectValueForTableColumn: objectValueColumn
						row: selectedRow];
			}
		[sender setColor:
			[selectedObjectValue isKindOfClass: [NSColor class]] ? selectedObjectValue : [PGNilColor nilColor]];
	}
	
- (void) tableView: (NSTableView*) table updateFont: (id) sender
	{
		id selectedObjectValue = nil;
		int selectedRow = [table selectedRow];
		if (selectedRow != -1)
			{
				NSTableColumn* objectValueColumn = [table tableColumnWithIdentifier: @"objectValue"];
				if ([[objectValueColumn dataCellForRow: selectedRow] tag] == PGFontTag)
					selectedObjectValue = [[table dataSource]
						tableView: table
						objectValueForTableColumn: objectValueColumn
						row: selectedRow];
			}
			
		// set the panel font. NOTE: we don't try to set the font manager font, since that doesn't accept a nil font
		[sender setPanelFont:
			[selectedObjectValue isKindOfClass: [NSFont class]] && ![[PGFontTextFieldCell nilFont] isEqual: selectedObjectValue] ? selectedObjectValue : nil
			isMultiple: NO];
	}
	
- (void) tableView: (NSTableView*) table changeColor: (id) sender
	{
		int selectedRow = [table selectedRow];
		if (selectedRow != -1)
			{
				NSTableColumn* objectValueColumn = [table tableColumnWithIdentifier: @"objectValue"];
				if ([[objectValueColumn dataCellForRow: selectedRow] tag] == PGColorTag)
					{
						id oldObjectValue = [[table dataSource]
							tableView: table
							objectValueForTableColumn: objectValueColumn
							row: selectedRow];
						NSColor* newColor = [sender color];
						
						if ([oldObjectValue isKindOfClass: [NSColor class]] || ![newColor isKindOfClass: [PGNilColor class]])
							{
								// forget any editing that took place
								BOOL editing = [[table window] fieldEditor: NO forObject: table] == [[table window] firstResponder];
								if (editing)
									[[table window] endEditingFor: table];
								
								[[table dataSource]
									tableView: table
									setObjectValue: newColor
									forTableColumn: objectValueColumn
									row: selectedRow];
								[table setNeedsDisplayInRect:
									[table frameOfCellAtColumn: [table columnWithIdentifier: @"objectValue"] row: selectedRow]];

								if (editing)
									[[table window] makeFirstResponder: table];

							}
					}
			}
			
	}

- (void) tableView: (NSTableView*) table changeFont: (id) sender
	{
		int selectedRow = [table selectedRow];
		if (selectedRow != -1)
			{
				NSTableColumn* objectValueColumn = [table tableColumnWithIdentifier: @"objectValue"];
				if ([[objectValueColumn dataCellForRow: selectedRow] tag] == PGFontTag)
					{
						id oldObjectValue = [[table dataSource]
							tableView: table
							objectValueForTableColumn: objectValueColumn
							row: selectedRow];
						if (![oldObjectValue isKindOfClass: [NSFont class]])
							oldObjectValue = [PGFontTextFieldCell nilFont];
							
						// forget any editing that took place
						BOOL editing = [[table window] fieldEditor: NO forObject: table] == [[table window] firstResponder];
						if (editing)
							[[table window] endEditingFor: table];
													
						[[table dataSource]
							tableView: table
							setObjectValue: [sender convertFont: oldObjectValue]
							forTableColumn: objectValueColumn
							row: selectedRow];
						[table setNeedsDisplayInRect:
							[table frameOfCellAtColumn: [table columnWithIdentifier: @"objectValue"] row: selectedRow]];
							
						if (editing)
							[[table window] makeFirstResponder: table];
						
					}
			}
			
	}
	
- (unsigned int) tableView: (NSTableView*) table validModesForFontPanel: (NSFontPanel*) fontPanel
	{
	    return NSFontPanelFaceModeMask | NSFontPanelCollectionModeMask;	// don't want font sizes
	}

- (void) dealloc
	{
		[[NSNotificationCenter defaultCenter] removeObserver: self];
				
		[graphDefaultSource_ release];
		[nodeDefaultSource_ release];
		[edgeDefaultSource_ release];
		[super dealloc];
	}
	
@end
