/*
 *  PGTasker.m
 *  graphviz
 *
 *  Created by Glen Low on Tue Dec 23 2003.
 *  Copyright (c) 2003, Pixelglow Software. All rights reserved.
 *  http://www.pixelglow.com/graphviz/
 *  graphviz@pixelglow.com
 *
 *  Redistribution and use in source and binary forms, with or without modification, are permitted
 *  provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions
 *    and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
 *    and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *  * Neither the name of Pixelglow Software nor the names of its contributors may be used to endorse or
 *    promote products derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
 
#import "PGTasker.h"

@implementation PGTasker

- (id) copyWithZone: (NSZone*) zone
	{
		return [self retain];   // we are immutable
	}

- (id) initWithLaunchPaths: (NSArray*) launchPaths arguments: (NSArray*) arguments currentDirectoryPath: (NSString*) path
	{
		if ((self = [super init]))
			{
				NSNotificationCenter* defaultCenter = [NSNotificationCenter defaultCenter];
				tasks_ = [[NSMutableArray alloc] init];
				status_ = 0;
				
				NSPipe* lastInputOutput = [NSPipe pipe];
				input_ = [[lastInputOutput fileHandleForWriting] retain];
				
				int index;
				for (index = 0; index < [launchPaths count]; ++index)
					{
						NSTask* newTask = [[NSTask alloc] init];
						
						[newTask setLaunchPath: [launchPaths objectAtIndex: index]];
						[newTask setArguments: [arguments objectAtIndex: index]];
						if (path)
							[newTask setCurrentDirectoryPath: path];
						
						NSPipe* inputOuput = [NSPipe pipe];
						NSPipe* error = [NSPipe pipe];
						
						if (lastInputOutput)
							[newTask setStandardInput: lastInputOutput];
						[newTask setStandardOutput: inputOuput];
						[newTask setStandardError: error];
						
						lastInputOutput = inputOuput;
												
						[defaultCenter
							addObserver: self
							selector: @selector (getError:)
							name: NSFileHandleReadCompletionNotification
							object: [error fileHandleForReading]];
						[[error fileHandleForReading] readInBackgroundAndNotify];
						[self retain];

						[defaultCenter
							addObserver: self
							selector: @selector (getStatus:)
							name: NSTaskDidTerminateNotification
							object: newTask];
						[self retain];
						
						[tasks_ addObject: newTask];
						[newTask release];
					}
					
				[defaultCenter
					addObserver: self
					selector: @selector (getOutput:)
					name: NSFileHandleReadToEndOfFileCompletionNotification
					object: [lastInputOutput fileHandleForReading]];
				[[lastInputOutput fileHandleForReading] readToEndOfFileInBackgroundAndNotify];
				[self retain];
				
			}
		return self;
	}
	
- (NSString*) commandLine
	{
		NSMutableArray* commands = [NSMutableArray arrayWithCapacity: [tasks_ count]];
		
		NSEnumerator* eachTask = [tasks_ objectEnumerator];
		NSTask* nextTask;
		
		while ((nextTask = [eachTask nextObject]))
			[commands addObject:
				[NSString stringWithFormat: @"%@ %@\n",
				[[nextTask launchPath] lastPathComponent],
				[[nextTask arguments] componentsJoinedByString: @" "]]];
				
		return [commands componentsJoinedByString: @" | "];
	}
	
- (int) terminationStatus
	{
		return status_;
	}
	
- (void) startWithData: (NSData*) data
	{
		NSEnumerator* eachTask = [tasks_ objectEnumerator];
		NSTask* nextTask;

		while ((nextTask = [eachTask nextObject]))
			[nextTask launch];
			
		if (data)
			{
				[input_ writeData: data];
				[input_ closeFile];
			}

		[[NSNotificationCenter defaultCenter] postNotificationName: TaskerDidStart
			object: self];
	}
	
- (void) stop
	{
		NSEnumerator* eachTask = [tasks_ objectEnumerator];
		NSTask* nextTask;

		while ((nextTask = [eachTask nextObject]))
			[nextTask terminate];
	}
	
- (void) getOutput: (NSNotification*) notification
	{
		NSData* data = [[notification userInfo] objectForKey: NSFileHandleNotificationDataItem];
		NSNotificationCenter* defaultCenter = [NSNotificationCenter defaultCenter];

		[defaultCenter removeObserver: self
			name: NSFileHandleReadToEndOfFileCompletionNotification
			object: [notification object]];
		[self autorelease];

		[defaultCenter postNotificationName: TaskerDidOutput
			object: self
			userInfo: [NSDictionary dictionaryWithObject: data forKey: TaskerDataItem]];
	}

- (void) getError: (NSNotification*) notification
	{
		NSNotificationCenter* defaultCenter = [NSNotificationCenter defaultCenter];
		NSData* data = [[notification userInfo] objectForKey: NSFileHandleNotificationDataItem];
		if ([data length])
			[[notification object] readInBackgroundAndNotify];
		else
			{
				[defaultCenter removeObserver: self
					name: NSFileHandleReadCompletionNotification
					object: [notification object]];
				[self autorelease];
			}

		[defaultCenter postNotificationName: TaskerDidError object: self
			userInfo: [NSDictionary dictionaryWithObject: data forKey: TaskerDataItem]];
	}

- (void) getStatus: (NSNotification*) notification
	{
		NSTask* terminated = [notification object];
		
		NSNotificationCenter* defaultCenter = [NSNotificationCenter defaultCenter];
		[defaultCenter removeObserver: self
			name: NSTaskDidTerminateNotification
			object: terminated];
		[self autorelease];
		
		if (status_ == 0)
			status_ = [terminated terminationStatus];
			
		[tasks_ removeObject: terminated];
		if ([tasks_ count] == 0)
			[defaultCenter postNotificationName: TaskerDidStatus object: self];
	};
	
- (void) dealloc
	{
		[input_ release];
		[tasks_ release];
		[super dealloc];
	}
@end

NSString* TaskerDidStart = @"TaskerDidStart";
NSString* TaskerDidOutput = @"TaskerDidOutput";
NSString* TaskerDidError = @"TaskerDidError";
NSString* TaskerDidStatus = @"TaskerDidStatus";

NSString* TaskerDataItem = @"TaskerDataItem";

