/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* chaninfo-dialog.cpp
 *
 * Copyright (C) 2005 Takuo KITAME.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *
 * Author: Takuo KITAME.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "servmgr.h"
#include "channel.h"

#include <gnome.h>
#include <glade/glade.h>

#include "utils.h"
#include "chaninfo-dialog.h"
#include "gnome-peercast.h"

#define GLADE_FILE_NAME "gnome-peercast.glade"
#define DELAY_UPDATE 5000 /* 1 sec */


enum {
	LIST_HOST,
	LIST_FOREGROUND,
	LIST_HOPS,
	LIST_LISTENERS,
	LIST_RELAYS,
	LIST_UPTIME,
	LIST_FIREWALLED,
	LIST_AGENT,
	LIST_UPDATE,
	N_COLUMNS
};

static void
dialog_response_cb (GtkDialog *dlg, gint id, gpointer data)
{
	ChaninfoDialog *cd = (ChaninfoDialog *)data;

	switch (id) {
	case GTK_RESPONSE_APPLY:
		cd->applyMetaInfo ();
		return;
		break;
	}
	gtk_widget_destroy (GTK_WIDGET (dlg));
	delete cd;
}

static gboolean
update_info (gpointer data)
{
	ChaninfoDialog *cd = (ChaninfoDialog *)data;

	cd->setInfo ();

	return TRUE;
}

static gboolean
update_hit_list (gpointer data)
{
	ChaninfoDialog *cd = (ChaninfoDialog *)data;

	cd->setHitList ();

	return TRUE;
}

static void
add_columns (GtkTreeView *treeview)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;

	/* IP:HOST */
	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("IP:HOST"), renderer,
							"text", LIST_HOST,
							"foreground", LIST_FOREGROUND,
							NULL);
	gtk_tree_view_column_set_resizable (col, TRUE);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Hops"), renderer,
							"text", LIST_HOPS,
							NULL);
	g_object_set (G_OBJECT (renderer), "xalign", 1.0, NULL);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Listeners"), renderer,
							"text", LIST_LISTENERS,
							NULL);
	g_object_set (G_OBJECT (renderer), "xalign", 1.0, NULL);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Relays"), renderer,
							"text", LIST_RELAYS,
							NULL);
	g_object_set (G_OBJECT (renderer), "xalign", 1.0, NULL);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Uptime"), renderer,
							"text", LIST_UPTIME,
							NULL);
	g_object_set (G_OBJECT (renderer), "xalign", 1.0, NULL);
	gtk_tree_view_append_column (treeview, col);

	/* FireWalled */
	renderer = gtk_cell_renderer_toggle_new ();
	g_object_set (G_OBJECT (renderer), "activatable", FALSE, NULL);
	col = gtk_tree_view_column_new_with_attributes (_("Firewalled"), renderer,
							"active", LIST_FIREWALLED,
							NULL);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Agent"), renderer,
							"text", LIST_AGENT,
							NULL);
	gtk_tree_view_column_set_resizable (col, TRUE);
	gtk_tree_view_append_column (treeview, col);

	renderer = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Update"), renderer,
							"text", LIST_UPDATE,
							NULL);
	g_object_set (G_OBJECT (renderer), "xalign", 1.0, NULL);
	gtk_tree_view_append_column (treeview, col);

}

static GtkTreeModel *
create_model (void)
{
	GtkListStore * store;

	store = gtk_list_store_new (N_COLUMNS,
				    G_TYPE_STRING,  /* HOST */
				    G_TYPE_STRING,  /* FOREGROUND */
				    G_TYPE_UINT,    /* HOPS */
				    G_TYPE_UINT,    /* LISTENERS */
				    G_TYPE_UINT,    /* RELAYS */
				    G_TYPE_STRING,  /* UPTIME */
				    G_TYPE_BOOLEAN, /* FIREWALLED */
				    G_TYPE_STRING,  /* AGENT */
				    G_TYPE_STRING   /* UPDATE */
		);
	return GTK_TREE_MODEL (store);
}


GtkWidget *
ChaninfoDialog::getWidget (const gchar *name)
{
	return glade_xml_get_widget (xml, name);
}

void
ChaninfoDialog::applyMetaInfo (void)
{
	GtkWidget *w;
	const gchar *tmp;
	ChanInfo newInfo = channel->info;

	w = getWidget ("entry_genre");
	gtk_entry_set_editable (GTK_ENTRY (w), FALSE);
	tmp = gtk_entry_get_text (GTK_ENTRY (w));
	newInfo.genre = tmp;

	w = getWidget ("entry_desc");
	gtk_entry_set_editable (GTK_ENTRY (w), FALSE);
	tmp = gtk_entry_get_text (GTK_ENTRY (w));
	newInfo.desc = tmp;

	w = getWidget ("entry_url");
	gtk_entry_set_editable (GTK_ENTRY (w), FALSE);
	tmp = gtk_entry_get_text (GTK_ENTRY (w));
	newInfo.url = tmp;

	w = getWidget ("entry_comment");
	gtk_entry_set_editable (GTK_ENTRY (w), FALSE);
	tmp = gtk_entry_get_text (GTK_ENTRY (w));
	newInfo.comment = tmp;

	w = getWidget ("entry_artist");
	gtk_entry_set_editable (GTK_ENTRY (w), FALSE);
	tmp = gtk_entry_get_text (GTK_ENTRY (w));
	newInfo.track.artist = tmp;

	w = getWidget ("entry_title");
	gtk_entry_set_editable (GTK_ENTRY (w), FALSE);
	tmp = gtk_entry_get_text (GTK_ENTRY (w));
	newInfo.track.title = tmp;

	w = getWidget ("entry_album");
	gtk_entry_set_editable (GTK_ENTRY (w), FALSE);
	tmp = gtk_entry_get_text (GTK_ENTRY (w));
	newInfo.track.album = tmp;

	w = getWidget ("entry_tgenre");
	gtk_entry_set_editable (GTK_ENTRY (w), FALSE);
	tmp = gtk_entry_get_text (GTK_ENTRY (w));
	newInfo.track.genre = tmp;

	w = getWidget ("entry_contact");
	gtk_entry_set_editable (GTK_ENTRY (w), FALSE);
	tmp = gtk_entry_get_text (GTK_ENTRY (w));
	newInfo.track.contact = tmp;

	channel->updateInfo (newInfo);
}

void
ChaninfoDialog::disableApply (void)
{
	GtkWidget *w;
	const gchar *entries[] = {
		"entry_genre",
		"entry_desc",
		"entry_url",
		"entry_comment",
		"entry_artist",
		"entry_title",
		"entry_album",
		"entry_tgenre",
		"entry_contact",
		NULL };

	for (gint i = 0; entries[i]; i++) {
		w = getWidget (entries[i]);
		gtk_entry_set_editable (GTK_ENTRY (w), FALSE);
	}
	w = getWidget ("button_commit");
	gtk_widget_set_child_visible (w, FALSE);
}

void
ChaninfoDialog::setHitList (void)
{
	GtkTreeIter iter;
	GtkListStore *store;
	ChanHitList *chl;
	gchar buf[1024];
	GString *str;
	GnomePeercastApp *gap = (GnomePeercastApp *)peercastApp;

	store = GTK_LIST_STORE (gtk_tree_view_get_model (treeview));
	gtk_list_store_clear (store);

	chl = chanMgr->findHitList (channel->info);
	if (!chl) return;

	ChanHit *hit = chl->hit;
	while (hit) {
		gchar *color;
		gchar uptime[64];
		gchar update[64];
		gchar hostname[256];
		gchar verstr[16];

		if (!hit->host.ip || hit->dead) {
			hit = hit->next;
			continue;
		}

		hit->rhost[0].toStr (buf);
		str = g_string_new (buf);
		if (gap->getConfigBool (CONF_GUI_PREFIX"/name_resolve_hitlist") &&
		    ClientSocket::getHostname (hostname, hit->rhost[0].ip)) {
			g_string_append_printf (str, "[%s]", hostname);
		}

		if (hit->rhost[1].isValid ()) {
			hit->rhost[1].toStr (buf);
			g_string_append_printf (str, "/%s", buf);
		}
		if (hit->firewalled && hit->numRelays == 0)
			color = gap->getConfigString (CONF_GUI_PREFIX"/color_firewalled");
		else if (hit->firewalled)
			color = gap->getConfigString (CONF_GUI_PREFIX"/color_push_relay");
		else
			color = gap->getConfigString (CONF_GUI_PREFIX"/color_unfirewalled");
		make_time_str (uptime, sizeof (update), hit->upTime);
		make_time_str (update, sizeof (update), sys->getTime() - hit->time);

		if (hit->version)
			g_snprintf (verstr, sizeof (verstr), "v0.%d", hit->version);
		else
			strcpy (verstr, _("Unknown"));
		gtk_list_store_append (store, &iter);
		gtk_list_store_set (store, &iter,
				    LIST_HOST, str->str,
				    LIST_FOREGROUND, color,
				    LIST_HOPS, hit->numHops,
				    LIST_LISTENERS, hit->numListeners,
				    LIST_RELAYS, hit->numRelays,
				    LIST_FIREWALLED, hit->firewalled,
				    LIST_AGENT, verstr,
				    LIST_UPTIME, uptime,
				    LIST_UPDATE, update,
				    -1);
		g_string_free (str, TRUE);
		g_free (color);

		hit = hit->next;
	}
}

void
ChaninfoDialog::setInfo (void)
{
	GtkWidget *w;
	gchar buf[1024];
	gchar *tmp;

	/* channel */
	g_snprintf (buf, sizeof (buf),
		    "<span weight=\"bold\"><b>%s</b></span>",
		    channel->info.name.cstr ());
	w = getWidget ("label_name");
	gtk_label_set_markup (GTK_LABEL (w), buf);
	w = getWidget ("entry_genre");
	gtk_entry_set_text (GTK_ENTRY (w), channel->info.genre.cstr ());
	w = getWidget ("entry_desc");
	gtk_entry_set_text (GTK_ENTRY (w), channel->info.desc.cstr ());
	if (! channel->info.url.isEmpty ()) {
		w = getWidget ("entry_url");
		gtk_entry_set_text (GTK_ENTRY (w), channel->info.url.cstr ());
		w = getWidget ("href_url");
		gtk_widget_set_sensitive (w, TRUE);
		gnome_href_set_url (GNOME_HREF (w), channel->info.url.cstr ());
	} else {
		w = getWidget ("href_url");
		gtk_widget_set_sensitive (w, FALSE);
	}

	w = getWidget ("href_id");
	channel->info.id.toStr(buf);
	gnome_href_set_text (GNOME_HREF (w), buf);
	tmp = gen_local_url (channel->info, TRUE);
	gnome_href_set_url (GNOME_HREF (w), tmp);
	gtk_tooltips_set_tip (tips, w, tmp, NULL);
	g_free (tmp);

	w = getWidget ("label_type");
	gtk_label_set_text (GTK_LABEL (w),
			    ChanInfo::getTypeStr(channel->info.contentType));
	w = getWidget ("label_rate");
	tmp = g_strdup_printf ("%d kbps", channel->info.bitrate);
	gtk_label_set_text (GTK_LABEL (w), tmp);
	g_free (tmp);
	w = getWidget ("label_source");
	if (channel->sourceURL.isEmpty ()) {
		channel->sourceHost.host.toStr(buf);
		tmp = g_strdup_printf ("%s - %s [%s]",
				       channel->getSrcTypeStr(),
				       ChanInfo::getProtocolStr(channel->info.srcProtocol),
				       buf);

		gtk_label_set_text (GTK_LABEL (w), tmp);
		g_free (tmp);
	} else {
		gtk_label_set_text (GTK_LABEL (w), channel->sourceURL.cstr());
	}
	w = getWidget ("label_status");
	gtk_label_set_text (GTK_LABEL (w), channel->getStatusStr ());
	w = getWidget ("label_pos");
	g_snprintf (buf, sizeof (buf), "%d", channel->streamPos);
	gtk_label_set_text (GTK_LABEL (w), buf);
	w = getWidget ("label_head");
	g_snprintf (buf, sizeof (buf), "%d (%d bytes)",
		    channel->headPack.pos, channel->headPack.len);
	gtk_label_set_text (GTK_LABEL (w), buf);
	w = getWidget ("label_time");
	if (channel->info.lastPlayStart)
		make_time_str (buf, sizeof (buf),
			       sys->getTime() - channel->info.lastPlayStart);
	else
		g_snprintf (buf, sizeof (buf),"-");
	gtk_label_set_text (GTK_LABEL (w), buf);


	/* track */
	w = getWidget ("entry_artist");
	gtk_entry_set_text (GTK_ENTRY (w), channel->info.track.artist.cstr ());
	w = getWidget ("entry_title");
	gtk_entry_set_text (GTK_ENTRY (w), channel->info.track.title.cstr ());
	w = getWidget ("entry_tgenre");
	gtk_entry_set_text (GTK_ENTRY (w), channel->info.track.genre.cstr ());
	w = getWidget ("entry_album");
	gtk_entry_set_text (GTK_ENTRY (w), channel->info.track.album.cstr ());
	w = getWidget ("entry_contact");
	gtk_entry_set_text (GTK_ENTRY (w), channel->info.track.contact.cstr ());
	w = getWidget ("href_track");
	if (channel->info.track.contact.isEmpty ())
		gtk_widget_set_sensitive (w, FALSE);
	else {
		gtk_widget_set_sensitive (w, TRUE);
		gnome_href_set_url (GNOME_HREF (w),
				    channel->info.track.contact.cstr ());
	}
}

ChaninfoDialog::ChaninfoDialog (Channel *ch, GtkWindow *window)
{
	GtkTreeModel *model = create_model ();
	GtkWidget *dialog;

	debug_print (1, "New object: ChaninfoDialog");

	channel = ch;
	parent = window;

	tips = gtk_tooltips_new ();

	xml = load_glade_xml (GLADE_FILE_NAME, "chaninfo_dialog");
	dialog = getWidget ("chaninfo_dialog");
	gtk_widget_hide (dialog); /* hide at once */
	gtk_window_set_transient_for (GTK_WINDOW (dialog), parent);

	signalConnect ("chaninfo_response",
		       G_CALLBACK (dialog_response_cb), this);

	treeview = GTK_TREE_VIEW (getWidget ("hit_list"));
	gtk_tree_view_set_model (treeview, model);
	add_columns (treeview);

	setInfo ();
	setHitList ();

	if (channel->type != Channel::T_BROADCAST) {
		disableApply ();
	}
	gtk_widget_show (dialog);
}

