/*
 * RatPwCache.c --
 *
 *	This file contains password caching routines
 *
 * TkRat software and its included text is Copyright 1996-2000 by
 * Martin Forssn
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include <sys/stat.h>
#include <unistd.h>
#include "rat.h"
#include "env_unix.h"

/*
 * For the memory cache
 */
typedef struct CachedPasswd {
    int onDisk;
    char *host;
    unsigned int port;
    char *user;
    char *service;
    char *password;
    struct CachedPasswd *next;
    Tcl_TimerToken token;
} CachedPasswd;
static CachedPasswd *cache = NULL;
static int initialized = 0;
static char *filename = NULL;

/*
 * Local functions
 */
static void ReadDisk(Tcl_Interp *interp);
static void WriteDisk(Tcl_Interp *interp);
static void TouchEntry(Tcl_Interp *interp, CachedPasswd *cp);
static Tcl_TimerProc ErasePasswd;


/*
 *----------------------------------------------------------------------
 *
 * ReadDisk --
 *
 *      Read cached passwords from disk
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Updates the local cached list
 *
 *
 *----------------------------------------------------------------------
 */

void
ReadDisk(Tcl_Interp *interp)
{
    char *name, **argv;
    CachedPasswd *cp;
    char buf[1024];
    int argc;
    FILE *fp;
    Tcl_DString exp;

    name = Tcl_GetVar2(interp, "option", "pwcache_file", TCL_GLOBAL_ONLY);
    if (NULL == name) {
	return;
    }
    Tcl_TranslateFileName(interp, name, &exp);
    filename = cpystr(Tcl_DStringValue(&exp));
    Tcl_DStringFree(&exp);
    initialized = 1;
    if (NULL == (fp = fopen(filename, "r"))) {
	return;
    }
    while (fgets(buf, sizeof(buf), fp), !feof(fp)) {
	if (TCL_OK != Tcl_SplitList(interp, buf, &argc, &argv) || 5 != argc) {
	    continue;
	}
	cp = (CachedPasswd*)ckalloc(sizeof(CachedPasswd)+strlen(buf));
	cp->onDisk = 1;
	cp->host = (char*)cp + sizeof(CachedPasswd);
	strcpy(cp->host, argv[0]);
	cp->port = atoi(argv[1]);
	cp->user = cp->host + strlen(cp->host)+1;
	strcpy(cp->user, argv[2]);
	cp->service = cp->user + strlen(cp->user)+1;
	strcpy(cp->service, argv[3]);
	cp->password = cp->service + strlen(cp->service)+1;
	strcpy(cp->password, argv[4]);
	cp->next = cache;
	cache = cp;
	ckfree(argv);
    }
    fclose(fp);
    return;
}


/*
 *----------------------------------------------------------------------
 *
 * WriteDisk --
 *
 *      Write the cache to disk. Only those entries marked to be stored on
 *	disk are actually output.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Rewrites the disk cache.
 *
 *
 *----------------------------------------------------------------------
 */

void
WriteDisk(Tcl_Interp *interp)
{
    CachedPasswd *cp;
    char c, buf[64];
    FILE *fp;
    struct stat sbuf;
    int i, fd;
    Tcl_DString ds;

    if (-1 < (fd = open(filename, O_WRONLY))) {
	fstat(fd, &sbuf);
	c = 0;
	for (i=0; i<sbuf.st_size; i++) {
	    write(fd, &c, 1);
	}
	close(fd);
	unlink(filename);
    }
    if (NULL == (fp = fopen(filename, "w"))) {
	return;
    }
    fchmod(fileno(fp), 0600);
    Tcl_DStringInit(&ds);
    for (cp = cache; cp; cp = cp->next) {
	if (cp->onDisk) {
	    Tcl_DStringAppendElement(&ds, cp->host);
	    sprintf(buf, "%d", cp->port);
	    Tcl_DStringAppendElement(&ds, buf);
	    Tcl_DStringAppendElement(&ds, cp->user);
	    Tcl_DStringAppendElement(&ds, cp->service);
	    Tcl_DStringAppendElement(&ds, cp->password);
	    fprintf(fp, "%s\n", Tcl_DStringValue(&ds));
	    Tcl_DStringSetLength(&ds, 0);
	}
    }
    fclose(fp);
    Tcl_DStringFree(&ds);
    return;
}


/*
 *----------------------------------------------------------------------
 *
 * TouchEntry --
 *
 *     Touch an entry in the cache. This only affects entries that are
 *	going to timeout.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	May rewrite the cached password file
 *
 *
 *----------------------------------------------------------------------
 */

void
TouchEntry(Tcl_Interp *interp, CachedPasswd *cp)
{
    int timeout;

    if (cp->onDisk) {
	return;
    }
    if (cp->token) {
	Tcl_DeleteTimerHandler(cp->token);
    }
    Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "cache_passwd_timeout",
	       TCL_GLOBAL_ONLY), &timeout);
    if (timeout) {
	cp->token =
		Tcl_CreateTimerHandler(timeout*1000,ErasePasswd,(ClientData)cp);
    }
}


/*
 *----------------------------------------------------------------------
 *
 * ErasePasswd --
 *
 *      Earase a password from the cache
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Modifies the in-memory cache
 *
 *
 *----------------------------------------------------------------------
 */

static void
ErasePasswd(ClientData clientData)
{
    CachedPasswd **tcp, *cp = (CachedPasswd*)clientData;

    memset(cp->password, 0, strlen(cp->password));
    for (tcp = &cache; *tcp != cp; tcp = &(*tcp)->next);
    *tcp = cp->next;
    ckfree(cp);
}


/*
 *----------------------------------------------------------------------
 *
 * RatGetCachedPassword --
 *
 *      get a cached password
 *
 * Results:
 *	Returns a pointer to a static area containing the password,
 *	or NULL if no suitable cached password was found.
 *
 * Side effects:
 *	may read the cached password file
 *
 *
 *----------------------------------------------------------------------
 */

char*
RatGetCachedPassword(Tcl_Interp *interp, const char *host, unsigned int port,
	const char *user, const char *service)
{
    CachedPasswd *cp;

    if (0 == initialized) {
	ReadDisk(interp);
    }
    if (port == 0) {
	if (!strcmp(service, "pop3")) {
	    port = 110;
	} else if (!strcmp(service, "imap")) {
	    port = 143;
	}
    }
    for (cp = cache; cp; cp = cp->next) {
	if (!strcmp(cp->host, host)
		&& cp->port == port
		&& !strcmp(cp->user, user)
		&& !strcmp(cp->service, service)) {
	    TouchEntry(interp, cp);
	    return cp->password;
	}
    }
    return NULL;
}


/*
 *----------------------------------------------------------------------
 *
 * RatCachePassword --
 *
 *      Cache a password
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	May rewrite the cached password file
 *
 *
 *----------------------------------------------------------------------
 */

void
RatCachePassword(Tcl_Interp *interp, const char *host, unsigned int port,
	const char *user, const char *service, const char *passwd, int store)
{
    CachedPasswd *cp;

    if (0 == initialized) {
	ReadDisk(interp);
    }
    cp = (CachedPasswd*)ckalloc(sizeof(CachedPasswd)+strlen(host)+strlen(user)+
	    strlen(service)+strlen(passwd)+4);
    cp->onDisk = store;
    cp->host = (char*)cp + sizeof(CachedPasswd);
    strcpy(cp->host, host);
    cp->port = port;
    cp->user = cp->host+strlen(host)+1;
    strcpy(cp->user, user);
    cp->service = cp->user+strlen(user)+1;
    strcpy(cp->service, service);
    cp->password = cp->service+strlen(service)+1;
    strcpy(cp->password, passwd);
    cp->next = cache;
    cp->token = NULL;
    cache = cp;
    if (store) {
	WriteDisk(interp);
    } else {
	TouchEntry(interp, cp);
    }
    return;
}


/*
 *----------------------------------------------------------------------
 *
 * RatPasswdCachePurge --
 *
 *      Purge the password cache. If disk_also is true the both the disk and
 *	memory caches are purged. If disk_also is false, the only the
 *	memory cache is purged.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	May rewrite the cached password file
 *
 *
 *----------------------------------------------------------------------
 */

void
RatPasswdCachePurge(Tcl_Interp *interp, int disk_also)
{
    CachedPasswd *cp, *cpn;

    if (0 == initialized) {
	ReadDisk(interp);
    }
    for (cp = cache; cp; cp = cpn) {
	cpn = cp->next;
	memset(cp->password, 0, strlen(cp->password));
	if (cp->token) {
	    Tcl_DeleteTimerHandler(cp->token);
	}
	ckfree(cp);
    }
    cache = NULL;
    if (disk_also) {
	WriteDisk(interp);
    }
    return;
}

int
RatPasswdCachePurgeCmd(ClientData clientData, Tcl_Interp *interp, int objc,
	Tcl_Obj *CONST objv[])
{
    RatPasswdCachePurge(interp, 1);
    return TCL_OK;
}
