/*
 * Ras server for OpenGate
 * 
 * Copyright (c) Egoboo Ltd. 1999-2000
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open Gatekeeper
 *
 * The Initial Developer of the Original Code is Egoboo Ltd.
 *
 * $Log: RasServer.cxx,v $
 * Revision 1.22  2000/06/06 09:07:07  aunitt
 * Put in fixes for Cisco equipment sending zero length RAS addresses in keep
 * alive RRQs.
 *
 * Revision 1.21  2000/05/25 14:26:13  aunitt
 * Added work around for Cisco gateways sending 0 length call address in keep
 * alive RRQs.
 *
 * Revision 1.20  2000/05/23 10:11:41  aunitt
 * Added support for static gateway prefixes.
 *
 * Revision 1.19  2000/05/15 19:02:30  aunitt
 * Added IP address to logging and generally tidied it up.
 *
 * Revision 1.18  2000/05/12 14:03:14  aunitt
 * Reworked logging code.
 *
 * Revision 1.17  2000/05/05 11:16:36  aunitt
 * Added support for bandwidth management.
 *
 * Revision 1.16  2000/05/02 10:22:16  aunitt
 * Added support for reading time to live from RRQ.
 * Backed out local address changes as they broke other things.
 *
 * Revision 1.15  2000/05/01 17:03:02  aunitt
 * Fixed typo.
 *
 * Revision 1.14  2000/04/29 12:54:03  aunitt
 * Improved working of sending LRQs in a multi homed environment.
 *
 * Revision 1.13  2000/04/27 17:50:35  aunitt
 * Added support for creation of messages needed for registration timeout handling.
 *
 * Revision 1.12  2000/04/26 17:07:19  aunitt
 * Added intial support for registration time to live features.
 *
 * Revision 1.11  2000/04/21 13:40:02  aunitt
 * Now sends LRQs to all neighbours in parallel.
 *
 * Revision 1.10  2000/04/20 18:47:07  aunitt
 * Added support for sending LRQs to neighbouring Gatekeepers to support calling
 * to other zones and cooperating with their Gatekeepers.
 * Fixed bug in responding negatively to multicast LRQs if we don't know the
 * endpoint - we should just keep quiet.
 *
 * Revision 1.9  2000/04/12 13:28:02  aunitt
 * Changed to use new convenience functions in AddrUtils.
 * Fixed bug in giving wrong bandwidth replies.
 * Fixed bug in giving wrong destination address in ACF to answerer when calls
 * are GKRouted.
 *
 * Revision 1.8  2000/04/10 19:14:09  aunitt
 * Added support for Gatekeeper Routed Calls.
 * Moved environment data into new environment object.
 *
 * Revision 1.7  2000/04/05 15:17:52  aunitt
 * Added RAS logging module from Tony Genovese.
 *
 * Revision 1.6  2000/03/23 00:02:17  aunitt
 * Removed bogus comment.
 *
 * Revision 1.5  2000/03/21 21:15:48  aunitt
 * Added work around for NetMeeting asking for too little bandwidth.
 * Fixed bug in not replying with the correct protocol id.
 * Added more stringent checks for the supplied destination address on ACF.
 * Added more informational messages.
 *
 * Revision 1.4  2000/03/12 21:19:49  aunitt
 * Fixed bug in not handling  messages from endpoints that were not registered. Added endpoint id to unregistration request.
 *
 * Revision 1.3  2000/02/27 20:45:41  aunitt
 * Fixed problems with replying to wrong RAS address
 *
 * Revision 1.2  2000/02/15 20:48:51  aunitt
 * Added support for using system log. Added proper support for answer call field on ARQ.
 *
 * Revision 1.1.1.1  2000/02/01 22:25:40  aunitt
 * Initial revision
 *
 *
 */

#include <ptlib.h>
#include <ptlib/sockets.h>
#include <ptlib/svcproc.h>
#if (_MSC_VER >= 1200)
#include <q931.h>
#include <h245.h>               // To stop "ostream" ambiguous symbol error
#endif
#include "AddrUtils.h"
#include "EndpointTabl.h"
#include "RasServer.h"
#include "Defaults.h"


static unsigned MySeqNumber = 0;     // Sequence number for messages we send....

RasServer::RasServer( const Environ & Environ,
                            WORD      Port
                    ) : MyEnviron( Environ )
{
    if ( Environ.LocalAddr != INADDR_ANY )
    	GatekeeperAddr = AddrUtils::ConvertToH225TransportAddr( Environ.LocalAddr, Port );
    else
    {
        PIPSocket::Address GkAddr;
        PIPSocket::GetHostAddress( GkAddr );
        GatekeeperAddr = AddrUtils::ConvertToH225TransportAddr( GkAddr, Port );
    }
}

RasServer::~RasServer()
{
}

static BOOL AliasIsMemberOf( const H225_ArrayOf_AliasAddress & Aliases,
                             const H225_AliasAddress           Alias
                           )
// Task: to return TRUE iff the given alias is a member of the given array....
{
	for ( PINDEX i=0; i < Aliases.GetSize(); ++i )
	{
		if ( Aliases[i] == Alias )
			return TRUE;
	}

	return FALSE;
}

static void AliasDifference( const H225_ArrayOf_AliasAddress & Aliases,
                             const H225_ArrayOf_AliasAddress & ToRemove,
                                   H225_ArrayOf_AliasAddress & Difference
                           )
// Task: to return the array of aliases created by taking the original array
//       and removing the array of aliases given as the 2nd parameter
{
	size_t Diff = 0;
	Difference.SetSize(Diff);

	for ( PINDEX Source=0; Source < Aliases.GetSize(); ++Source )
	{
		if ( !AliasIsMemberOf( ToRemove, Aliases[Source] ) )
		{
			Difference.SetSize(Diff+1);
			Difference[Diff] = Aliases[Source];
			++Diff;
		}
	}
}

static void GetReplyTo(       EndpointTable *           EPTable,
                        const H225_EndpointIdentifier & Id,
                              H225_TransportAddress &   ReplyTo
                      )
// Task: to return the reply to address for given endpoint
{
	try
	{
		Endpoint EP = EPTable->FindByEndpointId( Id );
		// Extract the first IP address from the list of requested addresses....
		AddrUtils::GetIPAddress(ReplyTo, EP.GetRasAddresses());
	}
	catch( EndpointTable::NotFoundError )
	{
		PSYSTEMLOG( Error, "Failed to find valid reply to address" );
	}
}

static BOOL DoRCF( const H225_RegistrationRequest &  RegReq,
	                     H225_RasMessage &           Reply,
                   const H225_EndpointIdentifier &   Id,
                   const H225_ArrayOf_AliasAddress & Aliases,
                   const H225_GatekeeperIdentifier & GKId,
                   const H225_TimeToLive &           TTL
                 )
// Task: to confirm the given registration request sending a possibly new Id and array
//       of aliases
{
	Reply.SetTag(H225_RasMessage::e_registrationConfirm);
	H225_RegistrationConfirm & RegConf = Reply;
	H225_ArrayOf_AliasAddress AliasesToSend;

	RegConf.m_requestSeqNum        = RegReq.m_requestSeqNum;
	RegConf.m_protocolIdentifier.SetValue(GKDefaults::H225_ProtocolID);
	RegConf.m_callSignalAddress    = RegReq.m_callSignalAddress;
	AliasDifference( Aliases, RegReq.m_terminalAlias, AliasesToSend );
	if ( AliasesToSend.GetSize() > 0 )
	{
		RegConf.IncludeOptionalField( H225_RegistrationConfirm::e_terminalAlias);
		RegConf.m_terminalAlias = AliasesToSend;
	}
	RegConf.IncludeOptionalField( H225_RegistrationConfirm::e_gatekeeperIdentifier);
	RegConf.m_gatekeeperIdentifier = GKId;
	RegConf.m_endpointIdentifier   = Id;
	RegConf.IncludeOptionalField( H225_RegistrationConfirm::e_timeToLive );
	RegConf.m_timeToLive           = TTL;	

	return TRUE;
}

static BOOL DoRRJ( const H225_RegistrationRequest &      RegReq,
                         H225_RasMessage &               Reply,
                   const H225_RegistrationRejectReason & Reason,
                   const H225_GatekeeperIdentifier &     GKId
                 )
{
	Reply.SetTag(H225_RasMessage::e_registrationReject);
	H225_RegistrationReject & RegRej = Reply;

	RegRej.m_requestSeqNum        = RegReq.m_requestSeqNum;
	RegRej.m_protocolIdentifier.SetValue(GKDefaults::H225_ProtocolID);
	RegRej.m_rejectReason         = Reason;
	RegRej.IncludeOptionalField( H225_RegistrationReject::e_gatekeeperIdentifier);
	RegRej.m_gatekeeperIdentifier = GKId;

	return TRUE;
}

static BOOL IncludesIPAddress( const H225_ArrayOf_TransportAddress & Addr )
// Task: to return true iff the given array includes an IP address
{
	for( PINDEX i=0; i<Addr.GetSize(); ++i )
	{
		if ( Addr[i].GetTag() == H225_TransportAddress::e_ipAddress )
			return TRUE;
	}

	return FALSE;
}

static void GetRRQReplyTo( const H225_RegistrationRequest & RegReq,
                                 H225_TransportAddress &    ReplyTo
                         )
// Task: to return the reply to address for the given RRQ request
{
    try
    {
        // Extract the first IP address from the list of requested addresses....
	    AddrUtils::GetIPAddress(ReplyTo, RegReq.m_rasAddress);
	}
	catch( AddrUtils::IPAddrNotFoundError )
	{
	    // Assume address will already have been set to a sensible default
	}
}

static PTimeInterval GetTimeInterval( const H225_TimeToLive & TTL )
// Task: to convert from a H225 time to live value to a PTimeInterval
{
    PTimeInterval Result( 0, TTL.GetValue() );
    return Result;
}

static BOOL OnRRQ( const H225_RegistrationRequest &  RegReq,
                         H225_RasMessage &           Reply,	    
                         EndpointTable *             EPTable,
                   const H225_GatekeeperIdentifier & GKId,
                         H225_TransportAddress &     ReplyTo
                 )
// Task: to handle a registration request message....
{
    GetRRQReplyTo( RegReq, ReplyTo );

	// Check if they really want me
	if ( ( RegReq.HasOptionalField( H225_RegistrationRequest::e_gatekeeperIdentifier ) ) &&
	     ( RegReq.m_gatekeeperIdentifier != GKId )
	   )
	{
	    PSYSTEMLOG( Info, "RRQ: They asked for a different gatekeeper" );
		return DoRRJ( RegReq, 
		              Reply, 
		              H225_RegistrationRejectReason::e_undefinedReason,
		              GKId
		            );
	}
	
	// Check if keep alive
	if ( ( RegReq.HasOptionalField( H225_RegistrationRequest::e_keepAlive ) ) &&
	     ( RegReq.m_keepAlive )	
	   )
	{
		// An already registered endpoint is saying it's still alive
		try
		{
			if ( !RegReq.HasOptionalField(H225_RegistrationRequest::e_endpointIdentifier) )
				return DoRRJ( RegReq, 
			                  Reply,
			                  H225_RegistrationRejectReason::e_fullRegistrationRequired,
			                  GKId
			                );
			
			Endpoint EP = EPTable->FindByEndpointId( RegReq.m_endpointIdentifier );
			
			// Check it really is the same endpoint
			// Cisco will send zero length addresses....
			if ( ( ( RegReq.m_rasAddress.GetSize() == 0 ) ||
			       ( EP.GetRasAddresses() == RegReq.m_rasAddress )
			     )
			     &&
			     ( ( RegReq.m_callSignalAddress.GetSize() == 0 ) ||
			       ( EP.GetCallSigAddresses() == RegReq.m_callSignalAddress )
			     )
			   )
			{
			    EPTable->ResetTTL( RegReq.m_endpointIdentifier );
			    return DoRCF( RegReq,
			                  Reply,
			                  RegReq.m_endpointIdentifier,
			                  RegReq.m_terminalAlias,
    			              GKId,
	    		              EPTable->GetTimeToLive( RegReq.m_endpointIdentifier )
		    	            );
			}
		}
		catch( EndpointTable::NotFoundError )
		{
			// Reject it....
			return DoRRJ( RegReq, 
			              Reply, 
			              H225_RegistrationRejectReason::e_fullRegistrationRequired,
			              GKId
			            );
		}
		
	}

	// Check they've given us an IP RAS address, otherwise we're a bit buggered
	if ( !IncludesIPAddress( RegReq.m_rasAddress ) )
	{
    	PSYSTEMLOG( Info, "RRQ: I can only use IP addresses for RAS" );
		return DoRRJ( RegReq, 
		              Reply, 
		              H225_RegistrationRejectReason::e_invalidRASAddress,
		              GKId
		            );
	}

	// Try and add or update the given endpoint to our table of registered endpoints
	try
	{
		Endpoint NewEP( RegReq.m_rasAddress,
		                RegReq.m_callSignalAddress,
                        RegReq.m_terminalType,
		                RegReq.m_terminalAlias,
				        ( RegReq.HasOptionalField(H225_RegistrationRequest::e_endpointIdentifier) ?
		                  RegReq.m_endpointIdentifier :
		                  EPTable->GenerateEndpointId()
		                ),
		                RegReq.m_endpointVendor
		              );
	
        if ( RegReq.HasOptionalField(H225_RegistrationRequest::e_timeToLive) )		
            EPTable->Insert( NewEP, ReplyTo, GetTimeInterval( RegReq.m_timeToLive ) );
        else
    		EPTable->Insert( NewEP, ReplyTo );
		return DoRCF( RegReq,
		              Reply,
		              NewEP.GetId(),
		              NewEP.GetAliases(),
		              GKId,
		              EPTable->GetTimeToLive( NewEP.GetId() )
		            );
	}
	catch( EndpointTable::RegistrationError RegErr )
	{
	    PSYSTEMLOG( Info, "RRQ: Error in registering the endpoint" );
		// Error in registering
		return DoRRJ( RegReq, Reply, RegErr.Reason, GKId );
	}
}

static void RemoveAliases(       Endpoint &                  EP,
                           const H225_ArrayOf_AliasAddress & AliasesToRemove
                         )
// Task: to remove the given aliases from the given endpoint
{
	H225_ArrayOf_AliasAddress NewAliases;
	H225_ArrayOf_AliasAddress Existing = EP.GetAliases();
	PINDEX Last = 0;

	for( PINDEX i=0; i<Existing.GetSize(); ++i )
	{
		BOOL ToBeRemoved = FALSE;
		for( PINDEX j=0; j<AliasesToRemove.GetSize(); ++j )
		{
			if ( Existing[i] == AliasesToRemove[j] )
			{
				ToBeRemoved = TRUE;
				break;
			}
		}
		if ( !ToBeRemoved )
		{
			NewAliases.SetSize( Last+1 );
			NewAliases[Last] = Existing[i];
			++Last;
		}
	}

	EP.UpdateAlias( NewAliases );
}

static BOOL DoUCF( const H225_UnregistrationRequest &  UnregReq,
    	                 H225_RasMessage &             Reply
                 )
// Task: to confirm the given unregistration request
{
	Reply.SetTag(H225_RasMessage::e_unregistrationConfirm);
	H225_UnregistrationConfirm & UnregConf = Reply;

	UnregConf.m_requestSeqNum = UnregReq.m_requestSeqNum;

	return TRUE;
}

static BOOL DoURJ( const H225_UnregistrationRequest & UnregReq,
                         H225_RasMessage &            Reply,
                   const H225_UnregRejectReason &     Reason
                 )
{
	Reply.SetTag(H225_RasMessage::e_unregistrationReject);
	H225_UnregistrationReject & UnregRej = Reply;

	UnregRej.m_requestSeqNum      = UnregReq.m_requestSeqNum;
	UnregRej.m_rejectReason       = Reason;

	return TRUE;
}

static BOOL OnURQ( const H225_UnregistrationRequest & UnregReq,
                         H225_RasMessage &            Reply,	    
                         EndpointTable *              EPTable,
                         H225_TransportAddress &      ReplyTo
                 )
// Task: to handle a unregistration request message....
{
	GetReplyTo( EPTable, UnregReq.m_endpointIdentifier , ReplyTo );
	
	try
	{
		// Firstly find the endpoint
		Endpoint EP = EPTable->FindByCallSigAddr( UnregReq.m_callSignalAddress );

		if ( UnregReq.HasOptionalField( H225_UnregistrationRequest::e_endpointAlias ) )
		{
			// Endpoint wants to remove some aliases, not necessarily the whole
			// registration
			RemoveAliases( EP, UnregReq.m_endpointAlias );
			
			// Check if any aliases left
			if ( EP.GetAliases().GetSize() > 0 )
			{
				// Update the endpoint in the table
				try
				{
					EPTable->Insert( EP, ReplyTo );
				}
				catch( EndpointTable::RegistrationError )
				{
					return DoURJ( UnregReq, 
					              Reply, 
					              H225_UnregRejectReason::e_undefinedReason 
					            );
				}
			}
			else
			{
				// Remove it as no aliases left
				EPTable->Remove( EP );
			}
			
			return DoUCF( UnregReq, Reply );
		}
		else
		{
			// Plain old unregister....
			EPTable->Remove( EP );
			return DoUCF( UnregReq, Reply );
		}
	}	
	catch( EndpointTable::NotFoundError )
	{
	    PSYSTEMLOG( Info, "URQ: Endpoint is not registered" );
		// Endpoint is not registered with us....
		return DoURJ( UnregReq, Reply, H225_UnregRejectReason::e_notCurrentlyRegistered );
	}
	catch( EndpointTable::InconsistentError )
	{	
	    PSYSTEMLOG( Info, "RRQ: Trying to unregister aliases for different endpoints" );
		// Naughty boy, trying to unregister the addresses of multiple endpoints
		return DoURJ( UnregReq, Reply, H225_UnregRejectReason::e_undefinedReason );
	}

	// Catch all error - something went horribly wrong
	return DoURJ( UnregReq, Reply, H225_UnregRejectReason::e_undefinedReason );
}

static BOOL DoACF( const H225_AdmissionRequest & AdmReq,
	                     H225_RasMessage &       Reply,
	                     bool                    GKRouted,
                   const H225_TransportAddress & Destination,
                         BOOL                    TypeKnown,
                   const H225_EndpointType &     DestType,
                   const H225_BandWidth &        Bandwidth
                 )
// Task: to confirm the given admission request
{
	Reply.SetTag(H225_RasMessage::e_admissionConfirm);
	H225_AdmissionConfirm & AdmConf = Reply;

	AdmConf.m_requestSeqNum         = AdmReq.m_requestSeqNum;
	AdmConf.m_bandWidth             = Bandwidth;
	if ( GKRouted )
	    AdmConf.m_callModel         = H225_CallModel::e_gatekeeperRouted;
	else
       	AdmConf.m_callModel         = H225_CallModel::e_direct;
	AdmConf.m_destCallSignalAddress = Destination;
	if ( TypeKnown )
	{
		AdmConf.IncludeOptionalField( H225_AdmissionConfirm::e_destinationType );
		AdmConf.m_destinationType = DestType;
	}

	return TRUE;
}

static BOOL DoARJ( const H225_AdmissionRequest &      AdmReq,
	                     H225_RasMessage &            Reply,
                   const H225_AdmissionRejectReason & Reason
                 )
// Task: to reject the given admission request
{
	Reply.SetTag(H225_RasMessage::e_admissionReject);
	H225_AdmissionReject & AdmRej = Reply;

	AdmRej.m_requestSeqNum = AdmReq.m_requestSeqNum;
	AdmRej.m_rejectReason  = Reason;

	return TRUE;
}

static unsigned InsertCall( CallTable *                     CTable,
                            const H225_AdmissionRequest &   AdmissReq,
                            const H225_EndpointIdentifier & CallerId,
                            const H225_EndpointIdentifier & CalledId,
                            unsigned                        RequestedBandwidth
                          )
// Task: to insert the given call into the call table, returns the amount of
//       bandwidth actually allocated
{
    // Insert with call identifier if possible are it is "more" unique
    if ( AdmissReq.HasOptionalField( H225_AdmissionRequest::e_callIdentifier ) )
    {
        CallDetails Call( AdmissReq.m_callReferenceValue,
                          AdmissReq.m_callIdentifier,
	                      CallerId,
	                      CalledId,
	                      RequestedBandwidth
		                );
		CTable->Insert( Call );
		return Call.GetBandwidth();
    }
    else
    {
        CallDetails Call( AdmissReq.m_callReferenceValue,
	                      CallerId,
	                      CalledId,
	                      RequestedBandwidth
		                );
		CTable->Insert( Call );
		return Call.GetBandwidth();
    }
}

static bool CallIsKnown( CallTable *                    CTable,
                         const H225_AdmissionRequest &  AdmissReq,
                         unsigned &                     Bandwidth
                       )
// Task: to return true iff the call specified by the given admission request is already
//       in our call table
{
    try
    {
        if ( AdmissReq.HasOptionalField( H225_AdmissionRequest::e_callIdentifier ) )
        {
            CallDetails Call = CTable->FindById( AdmissReq.m_callIdentifier );
            Bandwidth = Call.GetBandwidth();
        }
        else
        {
            CallDetails Call = CTable->FindByRef( AdmissReq.m_callReferenceValue );
            Bandwidth = Call.GetBandwidth();
        }
        return true;
    }
    catch ( CallTable::NotFoundError )
    {
        return false;
    }
}

static unsigned InsertCallIfNotPresent( EndpointTable *                 EPTable,
                                        CallTable *                     CTable,
                                        const H225_AdmissionRequest &   AdmissReq,
                                        unsigned                        RequestedBandwidth
                                      )
// Task: to insert the given call into the call table, if it isn't already present,
//       returns bandwidth actually allocated
//       Note will throw errors if we cannot create a new endpoint or if we can't add
//       the call to the call table....
{
    // Check if it's already there
    unsigned Bandwidth;

    if ( CallIsKnown( CTable, AdmissReq, Bandwidth ) )
        return Bandwidth;

    // Call isn't already known so we need to add it

    // Firstly make sure we add the new endpoint into the endpoint table
    H225_EndpointIdentifier CallerId = EPTable->GenerateEndpointId();

    if ( AdmissReq.HasOptionalField( H225_AdmissionRequest::e_srcCallSignalAddress ) )
    {
        // Assume calling function will catch errors
        H225_ArrayOf_TransportAddress SrcAddr;
        SrcAddr.SetSize(1);
        SrcAddr[0] = AdmissReq.m_srcCallSignalAddress;

        Endpoint NewEP( SrcAddr, AdmissReq.m_srcInfo, CallerId );
        EPTable->Insert( NewEP, AdmissReq.m_srcCallSignalAddress );
    }
    else
    {
        // We have to manufacture a bogus registration address as we don't have one.
        // But that doesn't really matter as by definition this endpoint isn't
        // registered with us....
        PIPSocket::Address BogusAddr("0.0.0.0");
        H225_TransportAddress Bogus = AddrUtils::ConvertToH225TransportAddr( BogusAddr, 0 );

        Endpoint NewEP( AdmissReq.m_srcInfo, CallerId );
        EPTable->Insert( NewEP, Bogus );
    }

    // Now insert it into the call table
    return InsertCall( CTable,
                       AdmissReq,
                       CallerId,
                       AdmissReq.m_endpointIdentifier,
                       RequestedBandwidth
                     );
}

static void BuildLRQ(       PUDPSocket *                ReplySocket,
                      const PIPSocket::Address &        MyAddr,
                      const H225_ArrayOf_AliasAddress & DestAlias,
                            H225_RasMessage &           Mesg
                    )
// Task: to build an LRQ message
{
    Mesg.SetTag( H225_RasMessage::e_locationRequest );
    H225_LocationRequest &  LRQ = Mesg;
    PIPSocket::Address      ReplyAddr;
    WORD                    ReplyPort;

    ReplySocket->GetLocalAddress( ReplyAddr, ReplyPort );
    ReplyAddr = MyAddr;     // ReplyAddr is probably INADDR_ANY
    LRQ.m_requestSeqNum     = MySeqNumber++;
    LRQ.m_destinationInfo   = DestAlias;
    LRQ.m_replyAddress      = AddrUtils::ConvertToH225TransportAddr( ReplyAddr, ReplyPort );
}

static BOOL SendLRQ( const Environ &                    Environ,
                           PUDPSocket *                 MySocket,
                           H225_RasMessage &            LRQ,
                     const PIPSocket::Address &         MyAddr,
                     const PIPSocket::Address &         GatekeeperAddr,
                     const H225_ArrayOf_AliasAddress &  DestAlias
                   )
// Task: to construct and send an LRQ to the given Gatekeeper to ask for the given alias
{
   	PBYTEArray      Buffer;
	PPER_Stream     WriteStream;

    MySocket->Connect( GatekeeperAddr );
    BuildLRQ( MySocket, MyAddr, DestAlias, LRQ );
    H225_TransportAddress Dest =
	            AddrUtils::ConvertToH225TransportAddr( GatekeeperAddr,
	                                                   GKDefaults::RASPort
	                                                 );
	Environ.Log->LogNote("Sending Location Request");
    Environ.Log->LogRasMsg( LRQ, OpengateLog::Sending, Dest );
   	LRQ.Encode( WriteStream );
    WriteStream.CompleteEncoding();
    return MySocket->Write( WriteStream.GetPointer(), WriteStream.GetSize() );
}

static BOOL DecodeLCF( const H225_RasMessage &      Mesg,
                       EndpointTable *              EPTable,
                       H225_EndpointIdentifier &    DestId,
               		   H225_TransportAddress &      DestAddr,
                       BOOL &                       TypeKnown,
		               H225_EndpointType &          DestType
		              )
// Task: to attempt to decode the given message as an LCF. If it is an LCF, return true
//       fill in the parameters, and create an entry in our endpoint table.
{
    if ( Mesg.GetTag() == H225_RasMessage::e_locationConfirm )
    {
        const H225_LocationConfirm & LCF = Mesg;

        DestAddr = LCF.m_callSignalAddress;
        DestId   = EPTable->GenerateEndpointId();

        H225_ArrayOf_TransportAddress EPRasAddr;
        H225_ArrayOf_TransportAddress EPCallAddr;
        EPRasAddr.SetSize(1);
        EPRasAddr[0]  = LCF.m_rasAddress;
        EPCallAddr.SetSize(1);
        EPCallAddr[0] = DestAddr;

        H225_ArrayOf_AliasAddress DestInfo;
        if ( LCF.HasOptionalField( H225_LocationConfirm::e_destinationInfo ) )
            DestInfo = LCF.m_destinationInfo;

        try
	    {
	        // We need to mark the endpoint as discovered by LCF because we don't want
	        // to sent if a URQ on shutdown and we want to purge it from the table after
	        // a timeout....
	        if ( LCF.HasOptionalField( H225_LocationConfirm::e_destinationType ) )
            {
                TypeKnown = TRUE;
                DestType  = LCF.m_destinationType;
                Endpoint NewEP( EPRasAddr,
		                        EPCallAddr,
		                        DestType,
		                        DestInfo,
		                        DestId,
		                        false
		                      );
       		    EPTable->Insert( NewEP, DestAddr );
            }
            else
            {
                TypeKnown = FALSE;
                Endpoint NewEP( EPRasAddr,
		                        EPCallAddr,
		                        DestInfo,
		                        DestId,
		                        false
             	              );
       		    EPTable->Insert( NewEP, DestAddr );
            }

    	    return TRUE;
	    }
	    catch( EndpointTable::RegistrationError RegErr )
	    {
	        PSYSTEMLOG( Error, "Error in registering the endpoint from neighbour lookup" );
		    // Error in registering
	    }
    }

    return FALSE;
}

static BOOL ReceiveLCF( const Environ &                 Environ,
                              PSocket &                 MySocket,
                              H225_EndpointIdentifier & DestId,
               		          H225_TransportAddress &   DestAddr,
                              BOOL &                    TypeKnown,
		                      H225_EndpointType &       DestType
		              )
// Task: to receive a message on the given socket and if it is an LCF, return true and
//       fill in the passed parameters
{
    static const int BufferSize = 4096;
	PBYTEArray  Buffer(BufferSize);
	PPER_Stream ReadStream(Buffer);

	// We need to set the read timeout to a sensible value.....
	PTimeInterval Timeout(0,3);     // Try 3 seconds, should read from config
	MySocket.SetReadTimeout( Timeout );
	
	if ( MySocket.Read( ReadStream.GetPointer(), ReadStream.GetSize() ) )
	{
        H225_RasMessage Reply;
        if ( Reply.Decode( ReadStream ) )
        {
            H225_TransportAddress Sender;   // Bogus value - to be sorted out!
   			Environ.Log->LogNote("Received Location Confirm");
   			Environ.Log->LogRasMsg( Reply, OpengateLog::Receiving, Sender );
            return DecodeLCF( Reply,
                              Environ.EPTable,
                              DestId,
                     	      DestAddr,
                              TypeKnown,
		                      DestType
		                    );
        }
	}
	return FALSE;
}		

static PSocket::SelectList BuildSelectList( const PSocketList List )
// Task: to build a select list as a copy of the given socket list
{
    PSocket::SelectList Result;

    for( PINDEX i=0; i < List.GetSize(); ++i )
    {
        Result += List[i];
    }

    return Result;
}

static BOOL NeighboursKnowAlias( const Environ &                    Environ,
                                 const H225_ArrayOf_AliasAddress &  DestAlias,
                                       H225_EndpointIdentifier &    DestId,
                               		   H225_TransportAddress &      DestAddr,
                                   	   BOOL &                       TypeKnown,
		                               H225_EndpointType &          DestType
		                       )
// Task: to ask our neighbouring gatekeepers (via an LRQ) whether they know the
//       specified destination. If known this function will return TRUE and set up
//       the details in the passed parameters.
{
    AddressList Neighbours = Environ.GetNeighbours();
    PSocketList WaitForReplyFromList;   // Note - AllowDeletedObjects should be on
                                        // so when this list is emptied all sockets
                                        // should be deleted

    // Firstly send an LRQ to all our neighbours
    for ( AddressList::const_iterator i = Neighbours.begin(); i < Neighbours.end(); ++i )
    {
        PUDPSocket * Socket = new PUDPSocket( GKDefaults::RASPort );
        H225_RasMessage Request;
        if ( SendLRQ( Environ, Socket, Request, Environ.LocalAddr, *i, DestAlias ) )
            WaitForReplyFromList.Append( Socket );
        else
            delete Socket;
    }

   	PTimeInterval ReadTimeout(0,3);     // Try 3 seconds, should read from config

    // Now wait for all the replies
    while( !WaitForReplyFromList.IsEmpty() )
    {
        PSocket::SelectList SelectList = BuildSelectList( WaitForReplyFromList );
        PSocket::Select( SelectList, ReadTimeout );
        if ( SelectList.IsEmpty() )
        {
            // Timeout
            PSYSTEMLOG( Warning, "No reply from neighbours to LRQ" );
            // We really should retry sending the LRQs.... TODO...
            WaitForReplyFromList.RemoveAll();
        }
        else
        {
            // Check all sockets with messages waiting
            for ( PINDEX i=0; i < SelectList.GetSize(); ++i )
            {
                if ( ReceiveLCF( Environ,
                                 SelectList[i],
                                 DestId,
                                 DestAddr,
                                 TypeKnown,
                                 DestType
                               )
                   )
                {
                    // Got an LCF !!!
                    WaitForReplyFromList.RemoveAll();
                    return TRUE;
                }
                else
                {
                    // Okay not an LCF, but we got some response so remove this socket
                    // from the list of sockets we are waiting to respond....
                    if ( !WaitForReplyFromList.Remove( &SelectList[i] ) )
                        PSYSTEMLOG( Error, "Failed to remove socket from list" );
                }
            }
        }
    }

    return FALSE;
}		

static BOOL OnARQ( const H225_AdmissionRequest &     AdmissReq,
                         H225_RasMessage &           Reply,	    
                         EndpointTable *             EPTable,
                         CallTable *                 CTable,
                   const H225_GatekeeperIdentifier & GKId,
                         H225_TransportAddress &     ReplyTo,
                   const H225_TransportAddress &     MyCallAddr,
                         bool                        GKRouted,
                   const Environ &                   Environ
                 )
// Task: to handle an admission request message....
{
    GetReplyTo( EPTable, AdmissReq.m_endpointIdentifier, ReplyTo );

	// Check if they really want me
	if ( ( AdmissReq.HasOptionalField( H225_AdmissionRequest::e_gatekeeperIdentifier ) ) &&
	     ( AdmissReq.m_gatekeeperIdentifier != GKId )
	   )
	{
	    PSYSTEMLOG( Info, "ARQ: They asked for a different gatekeeper" );
		return DoARJ( AdmissReq, 
		              Reply, 
		              H225_AdmissionRejectReason::e_undefinedReason
		            );
	}

	try
	{
	    if ( ( AdmissReq.m_answerCall ) &&
	         ( AdmissReq.HasOptionalField( H225_AdmissionRequest::e_destCallSignalAddress ) )
    	   )
    	{
	        // They are answering a call from another endpoint....
    	    H225_EndpointType   DestType;   // Not used!!

            // We  need to insert in the call table to monitor bandwidth IF it
            // isn't already there...
    		H225_BandWidth AllocatedBandwidth;
            AllocatedBandwidth.SetValue( InsertCallIfNotPresent( EPTable,
                                                                 CTable,
                                                                 AdmissReq,
                                                                 AdmissReq.m_bandWidth.GetValue()
                                                               )
                                       );
   	
       		return DoACF( AdmissReq,
	    	              Reply,
		                  GKRouted,
                          AdmissReq.m_destCallSignalAddress,
		                  false,
		                  DestType,
    		              AllocatedBandwidth
	    	            );
    	}
		
		Endpoint                Caller = EPTable->FindByEndpointId(AdmissReq.m_endpointIdentifier);
		H225_TransportAddress   DestAddr;
		BOOL                    TypeKnown = FALSE;
		H225_EndpointType       DestType;
		H225_EndpointIdentifier CalledId;

		if ( AdmissReq.HasOptionalField( H225_AdmissionRequest::e_destCallSignalAddress ) )
		{
		    // They've supplied the destination address so let's use it
			try
			{
			    H225_ArrayOf_TransportAddress AkaDest;
			    AkaDest.SetSize(1);
			    AkaDest[0] = AdmissReq.m_destCallSignalAddress;
			
			    Endpoint Called = EPTable->FindByCallSigAddr( AkaDest );
			    if ( AdmissReq.HasOptionalField( H225_AdmissionRequest::e_destinationInfo ) )
			    {
			        // They've supplied both address and aliases, check they are the same guy!
			        if ( !Called.HasAllAliases( AdmissReq.m_destinationInfo ) )
			        {
                        PSYSTEMLOG( Info, "ARQ: They supplied a destination address and alias, but not for the same destination" );
			            return DoARJ( AdmissReq,
				                      Reply,
        				              H225_AdmissionRejectReason::e_aliasesInconsistent
		        		            );
			        }
			    }
                DestAddr = AdmissReq.m_destCallSignalAddress;
                CalledId = Called.GetId();
			}
			catch( EndpointTable::NotFoundError )
			{
			    PSYSTEMLOG( Info, "ARQ: They supplied a destination address that is not registered with us" );
				// Called party is not registered with us....
				return DoARJ( AdmissReq,
				              Reply,
				              H225_AdmissionRejectReason::e_calledPartyNotRegistered
				            );
			}
       	}
		else
		{
			if ( !AdmissReq.HasOptionalField( H225_AdmissionRequest::e_destinationInfo ) ) 
			{
    			PSYSTEMLOG( Info, "ARQ: They haven't supplied either a destination address or an alias" );
				// You haven't told me who you want!
				return DoARJ( AdmissReq, 
				              Reply, 
				              H225_AdmissionRejectReason::e_incompleteAddress
				            );		

				// This may not be the correct reject reason?
			}

			// Try and find the destination
			try
			{
				Endpoint Called = EPTable->FindByAlias( AdmissReq.m_destinationInfo );
				// Use the IP address in preference, but if we can't find an
				// IP address, just give them the first address and hope
				// they can handle it!
				H225_TransportAddress_ipAddress DestIPAddr;
				if ( Called.GetCallSigIPAddress(DestIPAddr) )
				{
					DestAddr.SetTag( H225_TransportAddress::e_ipAddress );
					(H225_TransportAddress_ipAddress &)DestAddr = DestIPAddr;
				}
				else
				{
					const H225_ArrayOf_TransportAddress & Addrs =
					    Called.GetCallSigAddresses();
					DestAddr = Addrs[0];
				}
				TypeKnown = TRUE;
				DestType  = Called.GetType();
				CalledId  = Called.GetId();
			}
			catch( EndpointTable::NotFoundError )
			{
    			PSYSTEMLOG( Info, "ARQ: Supplied destination alias is not registered with us" );
				// Called party is not registered with us, try other gatekeepers we know
				// about and if they don't know reject..
				
				if ( !NeighboursKnowAlias( Environ,
				                           AdmissReq.m_destinationInfo,
				                           CalledId,
				                           DestAddr,
				                           TypeKnown,
				                           DestType
				                         )
				   )
				    return DoARJ( AdmissReq,
				                  Reply,
				                  H225_AdmissionRejectReason::e_calledPartyNotRegistered
				                );
			}

		}
						
		// NOTE: We do the adjustment because NetMeeting asks for a bandwidth of 1
		// and if you only grant that it has a bit of a fit!
		unsigned RequestedBandwidth = max( AdmissReq.m_bandWidth.GetValue(),
		                                   Environ.MinBandwidth()
		                                 );

		// Now add the call to the call table
		H225_BandWidth AllocatedBandwidth;
        AllocatedBandwidth.SetValue( InsertCall( CTable,
                                                 AdmissReq,
                                                 AdmissReq.m_endpointIdentifier,
                                                 CalledId,
                                                 RequestedBandwidth
                                               )
                                   );

		return DoACF( AdmissReq, 
		              Reply,
		              GKRouted,
		              ( GKRouted ? MyCallAddr : DestAddr ),
		              TypeKnown,
		              DestType,
		              AllocatedBandwidth
		            );
	}
	catch( EndpointTable::NotFoundError )
	{
    	PSYSTEMLOG( Info, "ARQ: Caller is not registered with us" );
		// Caller is not registered with us....
		return DoARJ( AdmissReq, Reply, H225_AdmissionRejectReason::e_callerNotRegistered );
	}
	catch( EndpointTable::InconsistentError )
	{
    	PSYSTEMLOG( Info, "ARQ: Supplied destination aliases do not all refer to the same endpoint" );
		// The caller is asking for aliases that map to multiple
		// endpoints
		return DoARJ( AdmissReq,
		              Reply,
		              H225_AdmissionRejectReason::e_aliasesInconsistent
		            );		
	}
	catch( EndpointTable::RegistrationError )
	{
	    PSYSTEMLOG( Info, "ARQ: Error in registering foreign endpoint" );
   		return DoARJ( AdmissReq,
		              Reply,
		              H225_AdmissionRejectReason::e_aliasesInconsistent
		            );	
	}
	catch( CallTable::BandwidthUnavailable )
	{
	    PSYSTEMLOG( Info, "ARQ: Bandwidth unavailable" );
	    // Our maximum bandwidth would be exceeded
	    return DoARJ( AdmissReq, Reply, H225_AdmissionRejectReason::e_requestDenied );
	}
	catch( CallTable::RequestInconsistent )
	{
	    PSYSTEMLOG( Info, "ARQ: Inconsistent call details" );
	    return DoARJ( AdmissReq, Reply, H225_AdmissionRejectReason::e_invalidEndpointIdentifier );
	}

	return FALSE;
}


static BOOL DoDCF( const H225_DisengageRequest & DisReq,
	                     H225_RasMessage &       Reply
                 )
// Task: to confirm the given disengage request
{
	Reply.SetTag(H225_RasMessage::e_disengageConfirm);
	H225_DisengageConfirm & DisConf = Reply;

	DisConf.m_requestSeqNum = DisReq.m_requestSeqNum;

	return TRUE;
}

static BOOL DoDRJ( const H225_DisengageRequest &      DisReq,
    	                 H225_RasMessage &            Reply,
                   const H225_DisengageRejectReason & Reason
                 )
// Task: to reject the given disengage request
{
	Reply.SetTag(H225_RasMessage::e_disengageReject);
	H225_DisengageReject & DisRej = Reply;

	DisRej.m_requestSeqNum = DisReq.m_requestSeqNum;
	DisRej.m_rejectReason  = Reason;

	return TRUE;
}

static BOOL OnDRQ( const H225_DisengageRequest &     DisReq,
                         H225_RasMessage &           Reply,	    
                         EndpointTable *             EPTable,
                         CallTable *                 CTable,
                         H225_TransportAddress &     ReplyTo
                 )
// Task: to handle a disengage request message....
{
    GetReplyTo( EPTable, DisReq.m_endpointIdentifier, ReplyTo );

	// We are not tracking calls at the moment, so not much to do...
	try
	{
		Endpoint EP = EPTable->FindByEndpointId( DisReq.m_endpointIdentifier );
		if ( DisReq.HasOptionalField( H225_DisengageRequest::e_callIdentifier ) )
		    CTable->Remove( DisReq.m_callIdentifier );
		else
		    CTable->Remove( DisReq.m_callReferenceValue );
		return DoDCF( DisReq, Reply );
	}
	catch( EndpointTable::NotFoundError )
	{
    	PSYSTEMLOG( Info, "DRQ: Caller is not registered with us" );
		// Caller is not registered with us....
		return DoDRJ( DisReq, Reply, H225_DisengageRejectReason::e_notRegistered );
	}

	return FALSE;
}

static BOOL DoBCF( const H225_BandwidthRequest & BandReq,
	                     H225_RasMessage &       Reply,
                   const H225_BandWidth &        Bandwidth
                 )
// Task: to confirm the given bandwidth request
{
	Reply.SetTag(H225_RasMessage::e_bandwidthConfirm);
	H225_BandwidthConfirm & BandConf = Reply;

	BandConf.m_requestSeqNum = BandReq.m_requestSeqNum;
	BandConf.m_bandWidth     = Bandwidth;

	return TRUE;
}

static BOOL DoBRJ( const H225_BandwidthRequest & BandReq,
	                     H225_RasMessage &       Reply,
                   const H225_BandRejectReason & Reason,
                   const H225_BandWidth &        AllowedBandwidth
                 )
// Task: to reject the given bandwidth request
{
	Reply.SetTag(H225_RasMessage::e_bandwidthReject);
	H225_BandwidthReject & BandRej = Reply;

	BandRej.m_requestSeqNum    = BandReq.m_requestSeqNum;
	BandRej.m_rejectReason     = Reason;
	BandRej.m_allowedBandWidth = AllowedBandwidth;

	return TRUE;
}

static BOOL OnBRQ( const H225_BandwidthRequest &     BandReq,
                         H225_RasMessage &           Reply,	    
                         EndpointTable *             EPTable,
                         CallTable *                 CTable,
                   const H225_GatekeeperIdentifier & GKId,
                         H225_TransportAddress &     ReplyTo
                 )
// Task: to handle a bandwidth request message....
{
    GetReplyTo( EPTable, BandReq.m_endpointIdentifier, ReplyTo );

	// Check if they really want me
	if ( ( BandReq.HasOptionalField( H225_BandwidthRequest::e_gatekeeperIdentifier ) ) &&
	     ( BandReq.m_gatekeeperIdentifier != GKId )
	   )
	{
    	PSYSTEMLOG( Info, "BRQ: They asked for a different gatekeeper" );
		return DoBRJ( BandReq, 
		              Reply, 
		              H225_BandRejectReason::e_undefinedReason,
		              CTable->GetAllowedBandwidth()
		            );
	}

	try
	{
	    bool RequestGranted;
	
		Endpoint EP = EPTable->FindByEndpointId( BandReq.m_endpointIdentifier );

		if ( BandReq.HasOptionalField( H225_BandwidthRequest::e_callIdentifier ) )
		    RequestGranted = CTable->AlterBandwidth( BandReq.m_callIdentifier,
		                                             BandReq.m_endpointIdentifier,
		                                             BandReq.m_bandWidth.GetValue()
		                                           );
        else
            RequestGranted = CTable->AlterBandwidth( BandReq.m_callReferenceValue,
		                                             BandReq.m_endpointIdentifier,
		                                             BandReq.m_bandWidth.GetValue()
		                                           );		
        if ( RequestGranted )		
    		return DoBCF( BandReq, Reply, BandReq.m_bandWidth );
        else
            return DoBRJ( BandReq,
                          Reply,
                          H225_BandRejectReason::e_undefinedReason,     // Not sure of the correct reason
                          CTable->GetAllowedBandwidth()
                        );
	}
	catch( EndpointTable::NotFoundError )
	{
    	PSYSTEMLOG( Info, "BRQ: Caller is not registered with us" );
		// Caller is not registered with us....
		return DoBRJ( BandReq,
		              Reply,
		              H225_BandRejectReason::e_notBound,
		              CTable->GetAllowedBandwidth()
		            );
	}
	catch( CallTable::NotFoundError )
	{
    	PSYSTEMLOG( Info, "BRQ: Call not found" );
    	// Couldn't find the requested call
		return DoBRJ( BandReq,
		              Reply,
		              // Not sure of the correct reason to pick....
		              H225_BandRejectReason::e_undefinedReason,
		              CTable->GetAllowedBandwidth()
		            );
	}
	catch( CallTable::RequestInconsistent )
	{
    	PSYSTEMLOG( Info, "BRQ: Inconsistent request" );
    	// Call and endpoint are not consistent
		return DoBRJ( BandReq,
		              Reply,
		              H225_BandRejectReason::e_invalidPermission,
		              CTable->GetAllowedBandwidth()
		            );
	}
}

static BOOL DoLCF( const H225_LocationRequest &  LocReq,
	                     H225_RasMessage &       Reply,
                   const H225_TransportAddress & CallSignalAddr,
                   const H225_TransportAddress & RasAddr,
                   const H225_EndpointType &     DestType 
                 )
// Task: to confirm the given location request
{
	Reply.SetTag(H225_RasMessage::e_locationConfirm);
	H225_LocationConfirm & LocConf = Reply;

	LocConf.m_requestSeqNum     = LocReq.m_requestSeqNum;
	LocConf.m_callSignalAddress = CallSignalAddr;
	LocConf.m_rasAddress        = RasAddr;
	LocConf.IncludeOptionalField( H225_LocationConfirm::e_destinationType );
	LocConf.m_destinationType   = DestType;

	return TRUE;
}

static BOOL DoLRJ( const H225_LocationRequest &      LocReq,
	                     H225_RasMessage &           Reply,
	                     bool                        IsMulticast,
                   const H225_LocationRejectReason & Reason
                 )
// Task: to reject the given bandwidth request
{
    if ( IsMulticast )
        // We shouldn't respond negatively to multicast LRQs`
        return FALSE;

	Reply.SetTag(H225_RasMessage::e_locationReject);
	H225_LocationReject & LocRej = Reply;

	LocRej.m_requestSeqNum = LocReq.m_requestSeqNum;
	LocRej.m_rejectReason  = Reason;

	return TRUE;
}

static void GetBestAddress( const H225_ArrayOf_TransportAddress & Addresses,
                                  H225_TransportAddress &         BestAddress
                          )
// Task: to extract the "best" address from the given array - i.e. pick the ip address
//       in preference
{
	for( PINDEX i=0; i<Addresses.GetSize(); ++i )
	{
		if ( Addresses[i].GetTag() == H225_TransportAddress::e_ipAddress )
		{
			BestAddress = Addresses[i];
			return;
		}
	}

	// No IP address, just pick the first
	BestAddress = Addresses[0];
}

static void GetLRQReplyTo( const H225_LocationRequest &  LocReq,
                                 H225_TransportAddress & ReplyTo
                         )
// Task: to return the reply to address for the given LRQ request
{
	ReplyTo = LocReq.m_replyAddress;
}

static BOOL OnLRQ( const H225_LocationRequest &     LocReq,
                         H225_RasMessage &          Reply,	
                         EndpointTable *            EPTable,
                         bool                       IsMulticast,
                         H225_TransportAddress &    ReplyTo
                 )
// Task: to handle a location request message....
{
    GetLRQReplyTo( LocReq, ReplyTo );

	// Check if the endpoint is registered if it is specified
	// Note: an unregistered endpoint can send us an LRQ, but we will assume that
	// if the endpoint is specified in the message they want us to check it!
	if ( LocReq.HasOptionalField( H225_LocationRequest::e_endpointIdentifier ) )
	{
		try
		{
			Endpoint EP = EPTable->FindByEndpointId( LocReq.m_endpointIdentifier );
		}
		catch( EndpointTable::NotFoundError )
		{
    		PSYSTEMLOG( Info, "LRQ: Caller is not registered with us" );
			// Caller is not registered with us....
			return DoLRJ( LocReq,
			              Reply,
			              IsMulticast,
			              H225_LocationRejectReason::e_notRegistered
			            );
		}
	}	

	// Now search for the destination they want....
	try
	{
		Endpoint WantedEP = EPTable->FindByAlias( LocReq.m_destinationInfo );
		H225_TransportAddress CallSigAddr;
		H225_TransportAddress RasAddr;
	
		GetBestAddress( WantedEP.GetCallSigAddresses(), CallSigAddr );
		GetBestAddress( WantedEP.GetRasAddresses(), RasAddr );
		return DoLCF( LocReq, Reply, CallSigAddr, RasAddr, WantedEP.GetType() );
	}	
	catch( EndpointTable::NotFoundError )
	{
	    PSYSTEMLOG( Info, "LRQ: Destination is not registered with us" );
		// Endpoint is not registered with us....
		return DoLRJ( LocReq,
		              Reply,
		              IsMulticast,
		              H225_LocationRejectReason::e_requestDenied
		            );
	}
	catch( EndpointTable::InconsistentError )
	{	
    	PSYSTEMLOG( Info, "LRQ: Supplied destination aliases do not all refer to the same endpoint" );
		// Aliases registered to multiple endpoints
		return DoLRJ( LocReq,
		              Reply,
		              IsMulticast,
		              H225_LocationRejectReason::e_aliasesInconsistent
		            );
	}
	
}

static BOOL DoGCF( const H225_GatekeeperRequest &    GKReq,
    	                 H225_RasMessage &           Reply,
                   const H225_GatekeeperIdentifier & GKId,
                   const H225_TransportAddress &     GKAddr                  
                 )
// Task: to confirm the given gatekeeper request
{
	Reply.SetTag(H225_RasMessage::e_gatekeeperConfirm);
	H225_GatekeeperConfirm & GKConf = Reply;

	GKConf.m_requestSeqNum        = GKReq.m_requestSeqNum;
	GKConf.m_protocolIdentifier.SetValue(GKDefaults::H225_ProtocolID);
	GKConf.IncludeOptionalField( H225_GatekeeperConfirm::e_gatekeeperIdentifier);
	GKConf.m_gatekeeperIdentifier = GKId;
	GKConf.m_rasAddress           = GKAddr;

	return TRUE;
}

static BOOL DoGRJ( const H225_GatekeeperRequest &      GKReq,
	                     H225_RasMessage &             Reply,
                   const H225_GatekeeperRejectReason & Reason,
                   const H225_GatekeeperIdentifier &   GKId
                 )
// Task: to reject the given gatekeeper request
{
	Reply.SetTag(H225_RasMessage::e_gatekeeperReject);
	H225_GatekeeperReject & GKRej = Reply;

	GKRej.m_requestSeqNum        = GKReq.m_requestSeqNum;
	GKRej.m_protocolIdentifier.SetValue(GKDefaults::H225_ProtocolID);
	GKRej.IncludeOptionalField( H225_GatekeeperReject::e_gatekeeperIdentifier);
	GKRej.m_gatekeeperIdentifier = GKId;
	GKRej.m_rejectReason         = Reason;

	return TRUE;
}

static void GetGRQReplyTo( const H225_GatekeeperRequest & GKReq,
                                 H225_TransportAddress &  ReplyTo
                         )
// Task: to return the reply to address for the given GRQ request
{
	ReplyTo = GKReq.m_rasAddress;
}

static BOOL OnGRQ( const H225_GatekeeperRequest &    GKReq,
                         H225_RasMessage &           Reply,	    
                   const H225_GatekeeperIdentifier & GKId,
                   const H225_TransportAddress &     GKAddr,
                         H225_TransportAddress &     ReplyTo
                 )
// Task: to handle a gatekeeper request message....
{	
    GetGRQReplyTo( GKReq, ReplyTo );

	// Check if they really want me
	if ( ( GKReq.HasOptionalField( H225_GatekeeperRequest::e_gatekeeperIdentifier ) ) &&
	     ( GKReq.m_gatekeeperIdentifier != GKId )
	   )
	{
    	PSYSTEMLOG( Info, "GRQ: They asked for a different gatekeeper" );
		return DoGRJ( GKReq, 
		              Reply, 
		              H225_GatekeeperRejectReason::e_resourceUnavailable,
		              GKId
		            );
	}

	return DoGCF( GKReq, Reply, GKId, GKAddr );
}

static BOOL OnIRR( const H225_InfoRequestResponse & IRR,
                         EndpointTable *            EPTable
                 )
// Task: to handle an info request response message
{
    EPTable->ResetTTL( IRR.m_endpointIdentifier );
    return TRUE;
}

BOOL RasServer::HandleRasRequest( const H225_RasMessage &       Request,
                                        bool                    IsMulticast,
                                        H225_RasMessage &       Reply,
                                        H225_TransportAddress & ReplyTo
                      	        )
// Task: to handle the given Ras request
{
	switch (Request.GetTag())
	{
    	case H225_RasMessage::e_registrationRequest :
			return OnRRQ( Request, Reply, MyEnviron.EPTable, MyEnviron.MyId,
			              ReplyTo
			            );
		case H225_RasMessage::e_unregistrationRequest :
			return OnURQ( Request, Reply, MyEnviron.EPTable, ReplyTo );
		case H225_RasMessage::e_admissionRequest :
			return OnARQ( Request, Reply, MyEnviron.EPTable, MyEnviron.CTable,
			              MyEnviron.MyId, ReplyTo, MyEnviron.MyCallAddr,
			              MyEnviron.IsGatekeeperRouted(), MyEnviron
			            );
		case H225_RasMessage::e_bandwidthRequest :
			return OnBRQ( Request, Reply, MyEnviron.EPTable, MyEnviron.CTable,
			              MyEnviron.MyId, ReplyTo
			            );
		case H225_RasMessage::e_disengageRequest :
			return OnDRQ( Request, Reply, MyEnviron.EPTable, MyEnviron.CTable,
			              ReplyTo
			            );
		case H225_RasMessage::e_locationRequest :
			return OnLRQ( Request, Reply, MyEnviron.EPTable, IsMulticast, ReplyTo );
			
		case H225_RasMessage::e_gatekeeperRequest :
			return OnGRQ( Request, Reply, MyEnviron.MyId, GatekeeperAddr, ReplyTo );
			
		case H225_RasMessage::e_infoRequestResponse :
		    OnIRR( Request, MyEnviron.EPTable );
		    return FALSE;
		
  		case H225_RasMessage::e_unregistrationConfirm :
		    return FALSE;	
		case H225_RasMessage::e_unregistrationReject :
		    return FALSE;
		
		case H225_RasMessage::e_gatekeeperConfirm :
		case H225_RasMessage::e_gatekeeperReject :

		case H225_RasMessage::e_registrationConfirm :
		case H225_RasMessage::e_registrationReject :

		case H225_RasMessage::e_admissionConfirm :
		case H225_RasMessage::e_admissionReject :

		case H225_RasMessage::e_bandwidthConfirm :
		case H225_RasMessage::e_bandwidthReject :

		case H225_RasMessage::e_disengageConfirm :
		case H225_RasMessage::e_disengageReject :

		case H225_RasMessage::e_locationConfirm :
		case H225_RasMessage::e_locationReject :
		case H225_RasMessage::e_infoRequest :

		case H225_RasMessage::e_nonStandardMessage :
		case H225_RasMessage::e_unknownMessageResponse :
		case H225_RasMessage::e_requestInProgress :
		case H225_RasMessage::e_resourcesAvailableIndicate :
		case H225_RasMessage::e_resourcesAvailableConfirm :
		case H225_RasMessage::e_infoRequestAck :
 		case H225_RasMessage::e_infoRequestNak :
		default :
			// We should really return an unknown message response
			// for messages we don't understand, but I've got no idea
			// how to extract the message sequence number....
			MyEnviron.Log->LogNote("Unknown message Request");
			PSYSTEMLOG( Info, "Unknown mesg" );
			return FALSE;
       }
}

void RasServer::MakeUnregMessage( const Endpoint &                  EP,
                                  const H225_UnregRequestReason &   Reason,
                                        H225_RasMessage &           Mesg
				                )
// Task: to construct an unregistration request for the given endpoint
{
	Mesg.SetTag(H225_RasMessage::e_unregistrationRequest);
	H225_UnregistrationRequest & UnregReq = Mesg;
	
	UnregReq.m_requestSeqNum        = MySeqNumber++;
	UnregReq.m_callSignalAddress    = EP.GetCallSigAddresses();
	UnregReq.IncludeOptionalField(H225_UnregistrationRequest::e_gatekeeperIdentifier);
	UnregReq.m_gatekeeperIdentifier = MyEnviron.MyId;
	UnregReq.IncludeOptionalField(H225_UnregistrationRequest::e_endpointIdentifier);
	UnregReq.m_endpointIdentifier   = EP.GetId();
	if ( Reason.GetTag() != H225_UnregRequestReason::e_undefinedReason )
	{
	    UnregReq.IncludeOptionalField(H225_UnregistrationRequest::e_reason);
	    UnregReq.m_reason           = Reason;
	}
}

void RasServer::MakeUnregMessage( const Endpoint &                  EP,
                                        H225_RasMessage &           Mesg
				                )
// Task: to construct an unregistration request for the given endpoint
{
    H225_UnregRequestReason Reason;
    Reason.SetTag(H225_UnregRequestReason::e_undefinedReason);
    MakeUnregMessage( EP, Reason, Mesg );
}

void RasServer::MakeIRR( const H225_CallReferenceValue &    CRV,
		                       H225_RasMessage &            Mesg
		               )
// Task: to construct an IRR request for the given call reference value
{
    Mesg.SetTag(H225_RasMessage::e_infoRequest);
    H225_InfoRequest & IRR = Mesg;

    IRR.m_requestSeqNum         = MySeqNumber++;
    IRR.m_callReferenceValue    = CRV;
    IRR.IncludeOptionalField(H225_InfoRequest::e_replyAddress);
    IRR.m_replyAddress          = GatekeeperAddr;
    IRR.RemoveOptionalField(H225_InfoRequest::e_callIdentifier);
}