/*
 * main.cxx
 *
 * PWLib application source file for OhPhone
 *
 * A H.323 "net telephone" application.
 *
 * Copyright (c) 1998-2000 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open H323 Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Portions of this code were written with the assisance of funding from
 * Vovida Networks, Inc. http://www.vovida.com.
 *
 * Contributor(s): ______________________________________.
 *                 Derek J Smithies (derek@indranet.co.nz)
 *
 * $Log: main.cxx,v $
 * Revision 1.237  2001/10/23 02:40:46  dereks
 * Fix bracket problem in Cu30 release of code.
 *
 * Revision 1.236  2001/10/23 02:21:39  dereks
 * Initial release CU30 video codec.
 * Add --videotest option, to display raw video, but not invoke a call.
 *
 * Revision 1.235  2001/09/25 03:18:09  dereks
 * Add code from Tiziano Morganti to set bitrate for H261 video codec.
 * Use --videobitrate n         Thanks for your code - good work!!
 *
 * Revision 1.234  2001/09/14 06:02:40  robertj
 * Added ability for number that does not match a speed dial to be passed
 *   to the gatekeeper for processing, thanks Chih-Wei Huang
 *
 * Revision 1.233  2001/08/24 13:57:37  rogerh
 * Delete the listener if StartListener() fails.
 *
 * Revision 1.232  2001/08/22 01:30:21  robertj
 * Resolved confusion with YUV411P and YUV420P video formats, thanks Mark Cooke.
 *
 * Revision 1.231  2001/08/10 10:06:10  robertj
 * No longer need SSL to have H.235 security.
 *
 * Revision 1.230  2001/08/08 05:09:09  dereks
 * Add test for presence/absence of SDL if the --videoreceive sdl option is used.
 * Thanks for the suggestion Greg Hosler
 * Break up the creation of the video grabber into individual components.
 * Add PTRACE commands to report on failures.
 *
 * Revision 1.229  2001/08/07 05:02:58  robertj
 * Added command line argument for H.235 gatekeeper password.
 *
 * Revision 1.228  2001/07/12 06:25:08  rogerh
 * create seperate variables for ulaw and alaw codecs
 *
 * Revision 1.227  2001/07/06 01:50:10  robertj
 * Changed memory check code to be conditionally compiled by
 *   PMEMORY_CHECK and not just _DEBUG.
 *
 * Revision 1.226  2001/05/17 07:11:29  robertj
 * Added more call end types for common transport failure modes.
 *
 * Revision 1.225  2001/05/10 23:47:45  robertj
 * Added trim to transfer command so can have leading spaces.
 *
 * Revision 1.224  2001/05/09 04:59:02  robertj
 * Bug fixes in H.450.2, thanks Klein Stefan.
 *
 * Revision 1.223  2001/05/09 04:07:53  robertj
 * Added more call end codes for busy and congested.
 *
 * Revision 1.222  2001/05/02 16:30:35  rogerh
 * Tidy up comments
 *
 * Revision 1.221  2001/05/01 17:07:32  rogerh
 * Allow CIF images to be transmitted by ordering the capabilities correctly
 * The previous re-ordering code did not work as the codec names had changed
 *
 * Revision 1.220  2001/05/01 05:00:38  robertj
 * Added command to do H.450.x call transfer and hold functions.
 *
 * Revision 1.219  2001/03/20 23:42:55  robertj
 * Used the new PTrace::Initialise function for starting trace code.
 *
 * Revision 1.218  2001/03/12 03:52:09  dereks
 * Tidy up fake video user interface. use --videodevice fake now.
 *
 * Revision 1.217  2001/03/08 23:53:26  robertj
 * Changed the fake video test pattern numbers from -1, -2, -3 to 0, 1 & 2 due
 *   to -1 now being a default channel value for all video devices.
 *
 * Revision 1.216  2001/03/07 01:47:45  dereks
 * Initial release of SDL (Simple DirectMedia Layer, a cross-platform multimedia library),
 * a video library code.
 *
 * Revision 1.215  2001/03/03 05:59:16  robertj
 * Major upgrade of video conversion and grabbing classes.
 *
 * Revision 1.214  2001/02/23 00:34:44  robertj
 * Added ability to add/display non-standard data in setup PDU.
 *
 * Revision 1.213  2001/02/06 07:44:20  robertj
 * Removed ACM codecs.
 *
 * Revision 1.212  2001/01/25 07:16:28  robertj
 * Fixed spurious memory leak. It is OK for the trace file to never be deleted.
 *
 * Revision 1.211  2001/01/24 06:25:46  robertj
 * Altered volume control range to be percentage, ie 100 is max volume.
 *
 * Revision 1.210  2001/01/18 12:24:19  rogerh
 * Fix bug which prevented lookback being selected on OSs without IXJ (eg BSD)
 *
 * Revision 1.209  2001/01/16 13:52:19  rogerh
 * Fix bug when reordering or deleting codecs. Found by "gnome" <gnome@21cn.com>
 *
 * Revision 1.208  2001/01/09 02:09:11  craigs
 * Added extra logging
 * Fixed problem with autodisconnect mode
 *
 * Revision 1.207  2001/01/05 14:55:53  rogerh
 * remove a warning from non ixj systems
 *
 * Revision 1.206  2000/12/19 22:35:53  dereks
 * Install revised video handling code, so that a video channel is used.
 * Code now better handles the situation where the video grabber could not be opened.
 *
 * Revision 1.205  2000/12/16 21:54:31  eokerson
 * Added DTMF generation when User Input Indication received.
 *
 * Revision 1.204  2000/11/13 22:31:38  craigs
 * Fixed per connection options
 *
 * Revision 1.203  2000/11/10 04:09:43  craigs
 * Changed to display AEC settings as text
 * Fixed improved CIDCW support
 * Removed misleading G728 defines
 *
 * Revision 1.202  2000/11/06 02:10:50  eokerson
 * Added support for AGC on IXJ devices.
 *
 * Revision 1.201  2000/10/26 21:29:20  dereks
 * Add --gsmframes parameter, for setting the number of gsmframes in one ethernet packet.
 * Default value is 4. Lower audio latency can be achieved with a value of 1
 *
 * Revision 1.200  2000/10/22 20:27:39  rogerh
 * Add more HAVE_IXJ #ifdefs to make code build on non-linux systems (eg FreeBSD)
 * Submittd by Blaz Zupan <blaz@amis.net>
 *
 * Revision 1.199  2000/10/16 08:50:12  robertj
 * Added single function to add all UserInput capability types.
 *
 * Revision 1.198  2000/10/13 01:47:59  dereks
 * Include command line option for setting the number of transmitted video
 * frames per second.   use --videotxfps n  (default n is 10)
 *
 * Revision 1.197  2000/09/29 00:07:38  craigs
 * Added G711 frame size options
 *
 * Revision 1.196  2000/09/27 03:06:13  dereks
 * Added lots of PTRACE statements to xlib code.
 * Removed X videoMutex from main.cxx & main.h
 * Removed some weird display issues from X code.
 *
 * Revision 1.195  2000/09/24 23:30:18  craigs
 * Removed debugging messages
 *
 * Revision 1.194  2000/09/23 07:20:49  robertj
 * Fixed problem with being able to distinguish between sw and hw codecs in LID channel.
 *
 * Revision 1.193  2000/09/22 01:35:55  robertj
 * Added support for handling LID's that only do symmetric codecs.
 *
 * Revision 1.192  2000/09/22 01:30:14  robertj
 * MSVC compatibility.
 *
 * Revision 1.191  2000/09/22 00:30:52  craigs
 * Enhanced autoDisconnect and autoRepeat functions
 *
 * Revision 1.190  2000/09/21 00:42:50  craigs
 * Changed play volume on sound cards to use PCM mixer channel
 *
 * Revision 1.189  2000/09/20 21:27:01  craigs
 * Fixed problem with default connection options
 *
 * Revision 1.188  2000/09/13 23:58:11  dereks
 * Corrected bug in video display. Now correctly handles 8, 16, 32 bit colour
 * Correctly handles 8 bit grayscale.
 *
 * Revision 1.187  2000/09/08 06:50:06  craigs
 * Added ability to set per-speed dial options
 * Fixed problem with transmit-only endpoints
 *
 * Revision 1.186  2000/09/01 02:13:08  robertj
 * Added ability to select a gatekeeper on LAN via it's identifier name.
 *
 * Revision 1.185  2000/08/30 23:43:27  robertj
 * Added -C option to set IXJ country code.
 * Added -c option for caller id, was documented in help but only had long version.
 *
 * Revision 1.184  2000/08/30 23:21:20  robertj
 * Fixed MSVC warnings.
 *
 * Revision 1.183  2000/08/30 05:14:49  craigs
 * Really fixed problem with setting quicknet volumes on startup
 *
 * Revision 1.182  2000/08/30 04:55:11  craigs
 * Added ability to bind to discrete interfaces
 * Fixed problem with initial audio settings
 *
 * Revision 1.181  2000/08/30 04:16:50  robertj
 * Fixed MSVC warning.
 *
 * Revision 1.180  2000/08/30 01:52:53  craigs
 * New IXJ volume code with pseudo-log scaling
 *
 * history deleted
 *
 * Revision 1.1  1998/12/14 09:13:19  robertj
 * Initial revision
 *
 */

#include <ptlib.h>

#include "main.h"
#include "gsmcodec.h"
#include "lpc10codec.h"
#include "mscodecs.h"
#include "h261codec.h"
#include "videoio.h"
#include "h323pdu.h"
#ifdef HAS_CU30
#include "cu30codec.h"
#endif

#ifdef P_LINUX
#include "vidlinux.h"
#include <sys/resource.h>
#endif

#define FICTITOUS_VIDEO "fake"

#if defined(P_FREEBSD) || defined(P_OPENBSD)
#define	DEFAULT_VIDEO	"/dev/bktr0"
#endif

#ifndef DEFAULT_VIDEO
#define	DEFAULT_VIDEO	"/dev/video0"
#endif

#ifdef HAS_X11
#include "xlibvid.h"
#endif

#ifdef HAS_SDL
#include "sdlvid.h"
#endif

#ifdef HAS_OSS
#define	DEFAULT_MIXER	"/dev/mixer"
#include	<linux/soundcard.h>
#endif

// uncomment below if to include xJack G729 code
#ifdef _WIN32
#define	G729
#endif


#ifdef HAS_IXJ
static const char * AECLevelNames[] = { "Off", "Low", "Medium", "High", "Auto AEC", "Auto AEC/AGC" };
#endif


#include "version.h"

PCREATE_PROCESS(OhPhone);

#define	DEFAULT_TIMEOUT	60000
#define	LAST_CALL_COUNT	16

class RingThread : public PThread
{
  PCLASSINFO(RingThread, PThread);

  public:
    RingThread(MyH323EndPoint & ep)
      : PThread(1000, NoAutoDeleteThread),
        endpoint(ep)
      { Resume(); }

    void Main()
      { endpoint.HandleRinging(); }

  protected:
    MyH323EndPoint & endpoint;
};

#define new PNEW


///////////////////////////////////////////////////////////////

OhPhone::OhPhone()
  : PProcess("Open H323 Project", "OhPhone",
             MAJOR_VERSION, MINOR_VERSION, BUILD_TYPE, BUILD_NUMBER)
{
}


OhPhone::~OhPhone()
{
}


void OhPhone::Main()
{
  //PArgList & args = GetArguments();
  PConfigArgs args(GetArguments());

  args.Parse(
             "a-auto-answer."        "-no-auto-answer."
             "b-bandwidth:"          "-no-bandwidth."
             "B-forward-busy:"       "-no-forward-busy."
#ifdef HAS_IXJ
             "c-callerid."           "-no-callerid."
             "C-country:"            "-no-country."
#endif
             "d-autodial:"           "-no-autodial."
             "D-disable:"
             "e-silence."            "-no-silence."
             "f-fast-disable."       "-no-fast-disable."
             "F-forward-always:"     "-no-forward-always."
             "g-gatekeeper:"   
             "G-gatekeeper-id:"
             "h-help."
             "i-interface:"          "-no-interface."
             "j-jitter:"             "-no-jitter."
             "l-listen."
             "n-no-gatekeeper."
             "N-forward-no-answer:"  "-no-forward-no-answer."
             "-answer-timeout:"      "-no-answer-timeout."
#if PTRACING
             "o-output:"             "-no-output."
#endif
             "p-proxy:"              "-no-proxy."
             "-password:"            "-no-password."
  	     "-listenport:"	    "-no-listenport."
	     "-connectport:"	    "-no-connectport."
	     "-connectring:"	    "-no-connectring."
	     "-port:"		    "-no-port."
             "P-prefer:"
#ifdef HAS_IXJ
             "q-quicknet:"           "-no-quicknet."
#endif
             "r-require-gatekeeper." "-no-require-gatekeeper."
             "-save."
             "-setup-param:"         "-no-setup-param."
             "s-sound:"              "-no-sound."
             "-sound-in:"            "-no-sound-in."
             "-sound-out:"           "-no-sound-out."
             "-sound-buffers:"       "-no-sound-buffers."

#ifdef HAS_OSS
             "-sound-mixer:"         "-no-sound-mixer."
             "-sound-recchan:"       "-no-sound-recchan."
             "-sound-recvol:"        "-no-sound-recvol."
             "-sound-playvol:"       "-no-sound-playvol."
#endif
#ifdef PMEMORY_CHECK
	     "-setallocationbreakpoint:"
#endif
             "T-h245tunneldisable."  "-no-h245tunneldisable."
#if PTRACING
             "t-trace."              "-no-trace."
#endif
             "-tos:"                 "-no-tos."
             "u-user:"               "-no-user."
             "v-verbose:"            "-no-verbose."

#ifdef HAS_IXJ
             "-aec:"                 "-no-aec."
             "-disable-menu."        "-no-disable-menu."
             "-dial-after-hangup."   "-no-dial-after-hangup."
             "-callerid."            "-no-callerid."
             "-calleridcw."          "-no-calleridcw."
	     "-autohook."            "-no-autohook."
             "-quicknet-recvol:"     "-no-quicknet-recvol."
             "-quicknet-playvol:"    "-no-quicknet-playvol."
#endif

             "-g728."                "-no-g728."
#ifdef	G729
             "-g729."                "-no-g729."
#endif
             "-gsm."                 "-no-gsm."
	     "-gsmframes:"           "-no-gsmframes."
             "-g711-ulaw."           "-no-g711-ulaw."
             "-g711-alaw."           "-no-g711-alaw."
	     "-g711frames:"          "-no-g711frames."
             "-g7231."               "-no-g7231."
             "-h261:"                "-no-h261."
             "-playvol:"             "-no-playvol."
             "-recvol:"              "-no-recvol."
	     "-ringfile:"
	     "-ringdelay:"

             "-videotransmit."       "-no-videotransmit."
             "-videolocal."          "-no-videolocal."
             "-videosize:"           "-no-videosize."
             "-videoformat:"         "-no-videoformat."
             "-videoinput:"          "-no-videoinput."              
             "-videodevice:"         "-no-videodevice."              

             "-videoreceive:"        "-no-videoreceive."
             "-videoquality:"        "-no-videoquality."
             "-videotxquality:"      "-no-videotxquality."
             "-videoidle:"           "-no-videoidle."
             "-videopip."            "-no-videopip."
             "-videofill:"           "-no-videofill."
             "-videotxfps:"          "-no-videotxfps."
             "-videobitrate:"        "-no-videobitrate." 
             "-videotest."           "-no-videotest."
#ifdef HAS_CU30
             "-videocu30stats:"      "-no-videocu30stats."
             "-videocu30."           "-no-videocu30."
#endif
             "-autodisconnect:"
             "-autorepeat:"

          , FALSE);

#if PMEMORY_CHECK
  if (args.HasOption("setallocationbreakpoint"))
    PMemoryHeap::SetAllocationBreakpoint(args.GetOptionString("setallocationbreakpoint").AsInteger());
#endif


  int verbose = 255;
  if (args.HasOption('v'))
    verbose = args.GetOptionString('v').AsInteger();

  if (verbose >= 3)
    cout << GetName()
         << " Version " << GetVersion(TRUE)
         << " by " << GetManufacturer()
         << " on " << GetOSClass() << ' ' << GetOSName()
         << " (" << GetOSVersion() << '-' << GetOSHardware() << ")\n\n";

#if PTRACING
  PTrace::Initialise(args.GetOptionCount('t'),
                     args.HasOption('o') ? (const char *)args.GetOptionString('o') : NULL);
#endif

  if (args.HasOption('h') || (!args.HasOption('l') && args.GetCount() == 0)) {
    cout << "Usage : " << GetName() << " [options] -l\n"
            "      : " << GetName() << " [options] [-p host] hostname/alias\n"
	    "\n   where:  hostname/alias = Remote host/alias to call\n"
            "\nOptions:\n"
            "  -a --auto-answer        : Automatically answer incoming calls\n"
            "  -d --autodial host      : Autodial host if phone off hook\n"
            "  -h --help               : Display this help message.\n"
            "  -l --listen             : Only listen for incoming calls\n"
            "  -v --verbose n          : Set amount of information displayed (0=none)\n"
            "  --disable-menu          : Disable internal menu\n"
            "  --ringfile filename     : Set sound file for \"ring\" annunciation\n"
            "  --ringdelay seconds     : Set delay between playing above file\n"
            "  --save                  : Save parameters in configuration file.\n"

            "\nGatekeeper options:\n"
            "  -g --gatekeeper host    : Specify gatekeeper host.\n"
            "  -G --gatekeeper-id name : Specify gatekeeper by ID.\n"
            "  -n --no-gatekeeper      : Disable gatekeeper discovery.\n"
            "  -r --require-gatekeeper : Exit if gatekeeper discovery fails.\n"
            "     --password pwd       : Password for gatekeeper H.235 authentication.\n"
            "  -p --proxy host         : Proxy/Gateway hostname/ip address\n"

            "\nDivert options:\n"
            "  -F --forward-always party    : Forward to remote party.\n"
            "  -B --forward-busy party      : Forward to remote party if busy.\n"
            "  -N --forward-no-answer party : Forward to remote party if no answer.\n"
            "     --answer-timeout time     : Time in seconds till forward on no answer.\n"

            "\nProtocol options:\n"
            "  -i --interface ipaddr   : Select interface to bind to for incoming connections (default is all interfaces)\n"
            "  --listenport            : Port to listen on for incoming connections (default 1720)\n"
            "  --connectport port      : Port to connect to for outgoing connections (default 1720)\n"
            "  --connectring num       : Distinctive ring number to send to remote - 0 (default) to 7\n"
            "  -b --bandwidth bps      : Limit bandwidth usage to bps bits/second\n"
            "  -f --fast-disable       : Disable fast start\n"
            "  -T --h245tunneldisable  : Disable H245 tunnelling.\n"
            "  -u --user name          : Set local alias name(s) (defaults to login name)\n"
            "  --tos n                 : Set IP Type of Service byte to n\n"
            "  --setup-param string    : Arbitrary data to be put into H.225 Setup PDU\n"

            "\nAudio options:\n"
            "  -e --silence            : Disable silence detection for GSM and software G.711\n"
            "  -j --jitter delay       : Set jitter buffer to delay milliseconds\n"
            "  --recvol n              : Set record volume\n"
            "  --playvol n             : Set play volume\n"

            "\nVideo transmit options:\n"
            "  --videodevice dev       : Select video capture device (default " DEFAULT_VIDEO ")\n"
            "  --videotransmit         : Enable video transmission\n"
            "  --videolocal            : Enable local video window\n"
            "  --videosize size        : Sets size of transmitted video window\n"
            "                             size can be small (default) or large\n"
            "  --videoformat type      : Set capture video format\n"
            "                             can be auto (default) pal or ntsc\n"
            "  --videoinput num        : Select capture video input (default is 0)\n"
            "  --videotxquality n      : Select sent video quality,(def 9). 1(good)<=n<=31\n" 
            "  --videofill n           : Select number of updated background blocks per frame 2(def)<=n<=99\n"
            "  --videotxfps n          : Maximum number of transmitted video frames per sec 2<10(def)<30\n"
            "  --videobitrate n        : Enable constant bitrate.   16< 256(def) <2048 kbit/s (net bw)\n"

            "\nVideo receive options:\n"
            "  --videoquality n        : Set received video quality hint - 0 <= n <= 31\n"
            "  --videoreceive viddev   : Receive video to following device\n"
            "                          :      null     do nothing\n"
            "                          :      ppm      create sequence of PPM files\n"
#ifdef HAS_VGALIB
            "                          :      svga256  256 colour VGA (Linux only)\n"
            "                          :      svga     full colour VGA (Linux only)\n"
#endif
#ifdef HAS_SDL
            "                          :      sdl      Use Simple DirectMedia Library\n"
            " --videopip               : Local video is displayed in adjacent smaller window\n"
#endif
#ifdef HAS_X11
      	    "                          :      x11       automatically pick best X11 mode\n"
	          "                          :      x1124     X11 using 24 bit colour\n"
	          "                          :      x1116     X11 using 16 bit colour\n"
	          "                          :      x118      X11 using 8 bit grey scale\n"
            "  --videopip              : Local video is displayed in corner of received video\n"
#endif   // HAS_X11

            "\nVideo options:\n"
            "  --videotest             : Display local video. Exit after 10 seconds. NO h323 call\n"
#ifdef HAS_CU30
            "  --videocu30             : Enable Cu30 codec\n"
            "  --videocu30stats n      : Collect stats for n frames, to optimise subsequent calls. (100-10000)\n"
#endif

            "\nSound card options:\n"
            "  -s --sound device       : Select sound card input/output device\n"
            "  --sound-in device       : Select sound card input device (overrides --sound)\n"
            "  --sound-out device      : Select sound card output device (overrides --sound)\n"
            "  --sound-buffers n       : Set sound buffer depth (default=2)\n"

#ifdef HAS_OSS
            "  --sound-mixer device    : Select sound mixer device (default is " DEFAULT_MIXER ")\n"
            "  --sound-recchan device  : Select sound mixer channel (default is mic)\n"
            "  --sound-recvol n        : Set record volume for sound card only (overrides --recvol)\n"
            "  --sound-playvol n       : Set play volume for sound card only (overrides --playvol)\n"
#endif

#ifdef HAS_IXJ
            "\nQuicknet card options:\n"
            "  -q -quicknet dev        : Use device (number or full device name)\n"
            "  -C --country name       : Set the country code for Quicknet device\n"
            "  --aec n                 : Set Audio Echo Cancellation level (0..3)\n"
            "  --autohook              : Don't use hook switch (for PhoneCard)\n"
            "  -c --callerid           : Enable caller id display\n"
            "  --calleridcw            : Enable caller id on call waiting display\n"
            "  --dial-after-hangup     : Present dial tone after remote hang up\n"
            "  --quicknet-recvol n     : Set record volume for Quicknet card only (overrides recvol)\n"
            "  --quicknet-playvol n    : Set play volume for Quicknet card only (overrides playvol)\n"
#endif

            "\nAudio Codec options:\n"
            "  -D --disable codec      : Disable the specified codec (may be used multiple times)\n"
            "  -P --prefer codec       : Prefer the specified codec (may be used multiple times)\n"
	    "  --g711frames count      : Set the number G.711 frames in capabilities (default 30)\n"
	    "  --gsmframes count       : Set the number GSM frames in capabilities (default 4)\n"
#ifdef HAS_IXJ
            "  --g7231                 : Set G.723.1 as preferred codec\n"
#endif
            "  --gsm                   : Set GSM 06.10 as preferred codec (default)\n"
            "  --g711-ulaw             : Set G.711 uLaw as preferred codec\n"
            "  --g711-alaw             : Set G.711 ALaw as preferred codec\n"
            "  --g728                  : Set G.728 as preferred codec\n"
#ifdef	G729
            "  --g729                  : Set G.729 as preferred codec\n"
#endif
            "  --g7231                 : Set G.723.1 as preferred codec\n"

#if PTRACING || PMEMORY_CHECK
            "\nDebug options:\n"
#endif
#if PTRACING
            "  -t --trace              : Enable trace, use multiple times for more detail\n"
            "  -o --output             : File for trace output, default is stderr\n"
#endif
#ifdef PMEMORY_CHECK
	    "  --setallocationbreakpoint n : Enable breakpoint on memory allocation n\n"
#endif
            << endl;
    return;
  }

  BOOL hasMenu = !args.HasOption("disable-menu");

  int autoRepeat;
  if (!args.HasOption("autorepeat"))
    autoRepeat = -1;
  else {
    autoRepeat = args.GetOptionString("autorepeat").AsInteger();
    if (autoRepeat < 1) {
      cerr << "autorepeat must be >= 1" << endl;
      return;
    }
    hasMenu = FALSE;
  }

  args.Save("save");

  MyH323EndPoint * endpoint = new MyH323EndPoint;
  if (endpoint->Initialise(args, verbose, hasMenu)) {
    if(!args.HasOption("videotest")) {
      if (autoRepeat < 0) {
	if (args.HasOption('l')) {
	  if (verbose >= 2) 
	    cout << "Waiting for incoming calls for \"" << endpoint->GetLocalUserName() << "\"\n";
	} else 
	  endpoint->MakeOutgoingCall(args[0], args.GetOptionString('p'));
	endpoint->AwaitTermination();
      } else {
	int i;
	endpoint->terminateOnHangup = TRUE;
	for (i = 1; i <= autoRepeat; i++) {
	  if (!args.HasOption('l')) {
	    cerr << "Making automatic call " << i << endl;
	    endpoint->MakeOutgoingCall(args[0], args.GetOptionString('p'));
	  }
	  endpoint->AwaitTermination();

	  cout << "Call #" << i;
#ifdef P_LINUX
	  struct rusage usage;
	  if (getrusage(RUSAGE_SELF, &usage) == 0)
	    cout << ": memory = " << usage.ru_ixrss << ", " << usage.ru_idrss;
#endif
	  cout << endl;
	  if (i == autoRepeat)
	    break;
	}  
      }
    }else      
      endpoint->TestVideoGrabber();
  } //initialised OK

  delete endpoint;

  if (verbose >= 3)
    cout << GetName() << " ended." << endl;
}


///////////////////////////////////////////////////////////////

BOOL MyH323EndPoint::Initialise(PConfigArgs & args, int _verbose, BOOL _hasMenu)
{
  PINDEX i;

  verbose = _verbose;
  hasMenu = _hasMenu;
  uiState = uiDialtone;

  // get local username
  if (args.HasOption('u')) {
    PStringArray aliases = args.GetOptionString('u').Lines();
    SetLocalUserName(aliases[0]);
    for (i = 1; i < aliases.GetSize(); i++)
      AddAliasName(aliases[i]);
  }

  // get ports
  WORD port       = H323ListenerTCP::DefaultSignalPort;
  WORD listenPort = port;
  if (!args.GetOptionString("port").IsEmpty())
    port = (WORD)args.GetOptionString("port").AsInteger();


#if 0
  PSoundChannel::writeDebug = TRUE;
  PSoundChannel::readDebug  = TRUE;
  OpalIxJDevice::writeDebug = TRUE;
  OpalIxJDevice::readDebug  = TRUE;
#endif

  defaultCallOptions.jitter      = GetMaxAudioDelayJitter();
  defaultCallOptions.connectPort = port;
  defaultCallOptions.connectRing = 0;

  if (!defaultCallOptions.Initialise(args))
    return FALSE;

  currentCallOptions = defaultCallOptions;

  //////////

  terminateOnHangup    = !args.HasOption('l');
  autoAnswer           = args.HasOption('a');
  alwaysForwardParty   = args.GetOptionString('F');
  busyForwardParty     = args.GetOptionString('B');
  noAnswerForwardParty = args.GetOptionString('N');
  noAnswerTime         = args.GetOptionString("answer-timeout", "30").AsUnsigned();
  dialAfterHangup      = args.HasOption("dial-after-hangup");
  setupParameter       = args.GetOptionString("setup-param");

  noAnswerTimer.SetNotifier(PCREATE_NOTIFIER(OnNoAnswerTimeout));

  if (args.HasOption("tos"))
    SetRtpIpTypeofService(args.GetOptionString("tos").AsUnsigned());

  if (args.HasOption('b')) {
    initialBandwidth = args.GetOptionString('b').AsUnsigned();
    if (initialBandwidth == 0) {
      cerr << "Illegal bandwidth specified." << endl;
      return FALSE;
    }
  }

  if (args.HasOption("sound-buffers")) {
    soundChannelBuffers = args.GetOptionString("sound-buffers", "2").AsUnsigned();
    if (soundChannelBuffers < 2 || soundChannelBuffers > 99) {
      cerr << "Illegal sound buffers specified." << endl;
      return FALSE;
    }
  }

  if (verbose >= 3) {
    cout << "Local username: " << GetLocalUserName() << "\n"
         << "TerminateOnHangup is " << terminateOnHangup << "\n"
         << "Auto answer is " << autoAnswer << "\n"
         << "DialAfterHangup is " << dialAfterHangup << "\n"
         << defaultCallOptions
         << endl;

  }

  if (args.HasOption("autodial")) {
    autoDial = args.GetOptionString("autodial");
    if (verbose >= 3)
      cout << "Autodial is set to "  << autoDial << "\n";
  }

  if (args.HasOption("h261")) {
    cout << "warning: --h261 option has been replaced by --videoreceive and --videotransmit" << endl;
    videoReceiveDevice = args.GetOptionString("h261");
  } else if (args.HasOption("videoreceive")) 
    videoReceiveDevice = args.GetOptionString("videoreceive");

  if (!videoReceiveDevice.IsEmpty()) {
#ifndef HAS_SDL                                               
    if (videoReceiveDevice *= "sdl") {                        
      cout << "Warning --videoreceive device is SDL, but SDL  is not installed" << endl;
      cout << "       Recompile the source code, after installing SDL." << endl;
    }                                                         
#endif                                                        
    if (   !(videoReceiveDevice *= "ppm")
        && !(videoReceiveDevice *= "null")
#ifdef HAS_VGALIB
        && !(videoReceiveDevice *= "svga")
        && !(videoReceiveDevice *= "svga256")
#endif
#ifdef HAS_SDL
        && !(videoReceiveDevice *= "sdl")
#endif
#ifdef HAS_X11
        && !(videoReceiveDevice *= "x1132")
        && !(videoReceiveDevice *= "x1124")
        && !(videoReceiveDevice *= "x1116")
        && !(videoReceiveDevice *= "x118")
        && !(videoReceiveDevice *= "x11")
        && !(videoReceiveDevice *= "x1132s")
        && !(videoReceiveDevice *= "x1124s")
        && !(videoReceiveDevice *= "x1116s")
        && !(videoReceiveDevice *= "x118s")
        && !(videoReceiveDevice *= "x11s")
#endif
        ) {
        cerr << "Unknown video receive device \"" << videoReceiveDevice << "\"" << endl;
        return FALSE;
    }
#ifdef HAS_SDL
    SDLVideoDevice::InitializeStaticVariables();
#endif
    
    if (!args.HasOption("videoquality")) 
      videoQuality = -1;
    else {
      videoQuality = args.GetOptionString("videoquality").AsInteger();
      videoQuality = PMAX(0, PMIN(31, videoQuality));
    }
  }

  videoPIP   = FALSE;
  videoSize = 0; //Default is small.
 
  autoStartTransmitVideo = args.HasOption("videotransmit");
  if (autoStartTransmitVideo) {

    videoDevice = DEFAULT_VIDEO;
    videoFake = FALSE;    
    if (args.HasOption("videodevice")) {      
      videoDevice = args.GetOptionString("videodevice");
      if (videoDevice == FICTITOUS_VIDEO)
        videoFake = TRUE;
    }
    
    if (args.GetOptionString("videosize") *= "large")
      videoSize = 1;
    cerr << "Set video size to be "<<videoSize<<endl;

    videoInput = 0; 
    if (args.HasOption("videoinput")) {      
      videoInput = args.GetOptionString("videoinput").AsInteger();
      if (videoInput<0) {
        cout << "User interface has changed.\n"
             << "Select fictitous video device with --videodevice "FICTITOUS_VIDEO"\n"
             << "Use videoinput argument of 0, 1, 2, ...\n"
             << "For backwards compatability, negative inputs are currently supported\n";
        videoFake = TRUE;
        videoInput= -1 - videoInput;    //-1 ==> 0, -2 ==>1, etc
      }
    }
  
    videoIsPal = TRUE;
    if (args.HasOption("videoformat"))
      videoIsPal = args.GetOptionString("videoformat") *= "pal";

    videoLocal = args.HasOption("videolocal");
    if(args.HasOption("videopip")) {
       videoPIP   = TRUE;
       videoLocal = TRUE;
    }
  
    if (!args.HasOption("videotxquality")) 
      videoTxQuality = -1; 
    else {
      videoTxQuality = args.GetOptionString("videotxquality").AsInteger();
      videoTxQuality = PMAX(1, PMIN(31, videoTxQuality));
    }

    if (args.HasOption("videofill")) {      
      videoFill = args.GetOptionString("videofill").AsInteger();
      videoFill = PMAX(2, PMIN(99, videoFill));
    } else
      videoFill = 2;

    if(args.HasOption("videocu30stats")) {
      videoCu30Stats = args.GetOptionString("videocu30stats").AsInteger();
      videoCu30Stats = PMAX(10, PMIN(10000, videoCu30Stats));
    } else
      videoCu30Stats = 0; // Dont record stats.
 
    if (args.HasOption("videotxfps")) {
      videoFramesPS = args.GetOptionString("videotxfps").AsInteger();
      videoFramesPS = PMAX(2,PMIN(30,videoFramesPS));  
    } else
      videoFramesPS=10;   //default value.
   
    if (args.HasOption("videobitrate")) {
      videoBitRate = args.GetOptionString("videobitrate").AsInteger();
      videoBitRate = 1024*PMAX(16,PMIN(2048,videoBitRate));
    } else
      videoBitRate = 0;   //default value.  
  }

  if (verbose >= 3) {
    if (videoReceiveDevice.IsEmpty())
      cout << "Video receive disabled" << endl << endl;
    else {
      cout << "Video receive using device : " << videoReceiveDevice << endl;
      cout << "Video receive quality hint : " << videoQuality << endl << endl;
    }
    if (!autoStartTransmitVideo)
      cout << "Video transmit disabled" << endl << endl;
    else {
      cout << "Video transmit enabled with local video window " << (videoLocal ? "en" : "dis") << "abled" << endl;
      cout << "Video transmit size is " << ((videoSize == 1) ? "large" : "small") << endl;
      cout << "Video capture using input " << videoInput << endl;
      cout << "Video capture using format " << (videoIsPal ? "PAL" : "NTSC") << endl;
      cout << "Video picture in picture of local video "<< (videoPIP ? "en" : "dis") << "abled" << endl;
      cout << "Video transmit quality is "<<videoTxQuality<<endl;
      cout << "Video background fill blocks "<<videoFill<<endl;
      cout << "Video transmit frames per sec "<<videoFramesPS<<endl;
      cout << "Video bitrate "<<videoBitRate<<" bps" << endl;
    }
#ifdef HAS_CU30
    if( args.HasOption("videocu30") || args.HasOption("videocu30stats")) {
      cout << "Video codec Cu30 enabled."<<endl;
      cout << "Video Cu30 statitistics " ;
      if(videoCu30Stats>0)
	cout << "enabled. " << videoCu30Stats << "frames." << endl ;
      else
	cout << "disabled" << endl;
    }
#endif
    cout << endl;
  }

#ifdef HAS_IXJ
  if (args.HasOption('q')) {
    PString ixjDevice = args.GetOptionString('q');
    if (xJack.Open(ixjDevice)) {
      if (verbose >= 3)
        cout << "Using Quicknet " << xJack.GetName() << '\n';
      xJack.SetLineToLineDirect(0, 1, FALSE);
      xJack.EnableAudio(0, TRUE);

      callerIdEnable = args.HasOption("callerid");
      if (verbose >= 3)
        cout << "Caller ID set to " << callerIdEnable << endl;

      callerIdCallWaitingEnable = args.HasOption("calleridcw");
      if (verbose >= 3)
        cout << "Caller ID on call waiting set to " << callerIdCallWaitingEnable << endl;

      if (args.HasOption('C'))
        xJack.SetCountryCodeName(args.GetOptionString('C'));
      if (verbose >= 3)
        cout << "Country set to " << xJack.GetCountryCodeName() << endl;

      int aec = 0;
      if (args.HasOption("aec")) {
        aec = args.GetOptionString("aec").AsUnsigned();
        xJack.SetAEC(0, (OpalLineInterfaceDevice::AECLevels)aec);
        if (verbose >= 3)
          cout << "AEC set to " << AECLevelNames[aec] << endl;
      } else {
        xJack.SetAEC(0, OpalLineInterfaceDevice::AECMedium);
        if (verbose >= 3)
          cout << "AEC set to default" << endl;
      }

      PString volStr;

      if((OpalLineInterfaceDevice::AECLevels)aec != OpalLineInterfaceDevice::AECAGC) {
        if (args.HasOption("quicknet-recvol"))
          volStr = args.GetOptionString("quicknet-recvol");
        else if (args.HasOption("recvol")) 
          volStr = args.GetOptionString("recvol");

        unsigned recvol;
        if (!volStr.IsEmpty()) {
          recvol = volStr.AsInteger();
          xJack.SetRecordVolume(0, recvol);
        } else {
          xJack.GetRecordVolume(0, recvol);
        }
        if (verbose >= 3) {
          cout << "Recording volume set to " << recvol << endl;
        }
      }
      if (args.HasOption("quicknet-playvol"))
        volStr = args.GetOptionString("quicknet-playvol");
      else if (args.HasOption("playvol")) 
        volStr = args.GetOptionString("playvol");

      unsigned playvol;
      if (!volStr.IsEmpty()) {
        playvol = volStr.AsInteger();
        xJack.SetPlayVolume(0, playvol);
      } else {
        xJack.GetPlayVolume(0, playvol);
      }

      if (verbose >= 3) {
        cout << "Playing volume set to " << playvol << endl;
      }

      autoHook = args.HasOption("autohook");
      if (autoHook)
        xJack.StopTone(0);
      if (verbose >= 3)
        cout << "Autohook set to " << autoHook << endl;
    }
    else {
      cerr << "Could not open " << ixjDevice << ": ";
      int code = xJack.GetErrorNumber();
      if (code == EBADF)
        cerr << "check that the Quicknet driver is installed correctly" << endl;
      else if (code == EBUSY)
        cerr << "check that the Quicknet driver is not in use" << endl;
      else {
        PString errStr = xJack.GetErrorText();
        if (errStr.IsEmpty())
          errStr = psprintf("error code %i", code);
        cerr << errStr << endl;
      }
      return FALSE;
    }
  }

#endif

  if (args.HasOption("ringfile"))
    ringFile = args.GetOptionString("ringfile");
  if (args.HasOption("ringdelay"))
    ringDelay = args.GetOptionString("ringdelay").AsInteger();
  else
    ringDelay = 5;

#ifdef HAS_IXJ
  if ( !xJack.IsOpen() ) {
#endif
  
    if (!SetSoundDevice(args, "sound", PSoundChannel::Recorder))
      return FALSE;
    if (!SetSoundDevice(args, "sound", PSoundChannel::Player))
      return FALSE;
    if (!SetSoundDevice(args, "sound-in", PSoundChannel::Recorder))
      return FALSE;
    if (!SetSoundDevice(args, "sound-out", PSoundChannel::Player))
      return FALSE;

    if (verbose >= 3)
      cout << "Sound output device: \"" << GetSoundChannelPlayDevice() << "\"\n"
              "Sound  input device: \"" << GetSoundChannelRecordDevice() << "\"\n";

#ifdef HAS_OSS
    if (!InitialiseMixer(args, verbose))
      return FALSE;
#endif

#ifdef HAS_IXJ
  } // endif
#endif

  // The order in which capabilities are added to the capability table
  // determines which one is selected by default.

#ifdef HAS_IXJ
  if (xJack.IsOpen())
    H323_LIDCapability::AddAllCapabilities(xJack, capabilities, 0, 0);
#endif

  int g711Frames = 30;
  if (args.HasOption("g711frames")) {
    g711Frames = args.GetOptionString("g711frames").AsInteger();
    if (g711Frames <= 10 || g711Frames > 240) {
      cerr << "error: G.711 frame size must be in range 10 to 240" << endl;
      g711Frames = 30;
    }
  }

  int gsmFrames = 4;
  if (args.HasOption("gsmframes")) {
    gsmFrames = args.GetOptionString("gsmframes").AsInteger();
    if (gsmFrames < 1 || gsmFrames > 7) {
      cerr << "error: GSM frame size must be in range 1 to 7" << endl;
      gsmFrames = 4;
    }
  }
  if (verbose >= 3) {
    cout <<"G.711 frame size: " << g711Frames << endl;
    cout <<"GSM frame size: " << gsmFrames << endl;
  }

  H323_GSM0610Capability * gsmCap; 
  SetCapability(0, 0, gsmCap = new H323_GSM0610Capability);
  gsmCap->SetTxFramesInPacket(gsmFrames);

  MicrosoftGSMAudioCapability * msGsmCap;
  SetCapability(0, 0, msGsmCap = new MicrosoftGSMAudioCapability);
  msGsmCap->SetTxFramesInPacket(gsmFrames);

  H323_G711Capability * g711uCap;
  SetCapability(0, 0, g711uCap = new H323_G711Capability(H323_G711Capability::muLaw));
  g711uCap->SetTxFramesInPacket(g711Frames);

  H323_G711Capability * g711aCap;
  SetCapability(0, 0, g711aCap = new H323_G711Capability(H323_G711Capability::ALaw));
  g711aCap->SetTxFramesInPacket(g711Frames);

  SetCapability(0, 0, new H323_LPC10Capability(*this));

  PStringArray toRemove = args.GetOptionString('D').Lines();
  PStringArray toReorder = args.GetOptionString('P').Lines();

  static const char * const oldArgName[] = {
    "g7231",   "g729",  "g728",  "gsm", "g711-ulaw", "g711-alaw"
  };
  static const char * const capName[] = {
    "G.723.1", "G.729", "G.728", "GSM", "G.711-uLaw", "G.711-ALaw"
  };

  for (i = 0; i < PARRAYSIZE(oldArgName); i++) {
    if (args.HasOption(PString("no-")+oldArgName[i]))
      toRemove[toRemove.GetSize()] = capName[i];
    if (args.HasOption(oldArgName[i]))
      toReorder[toReorder.GetSize()] = capName[i];
  }

  capabilities.Remove(toRemove);
  capabilities.Reorder(toReorder);
  
  //AddCapability puts the codec into the list of codecs we can send
  //SetCapability puts the codec into the list of codecs we can send and receive
#ifdef HAS_CU30
  PFilePath  fileName= PProcess::Current().GetConfigurationFile();
  PString statsDir = fileName.GetDirectory();   //Statistics files ("y" "u" "v" and "mc") have to be here.  
  INT _width,_height;
  _width= 176 << videoSize;
  _height= 144 << videoSize;
  cerr<<"SetVideo size to "<<_width<<" x " <<_height<<endl;

  if( args.HasOption("videocu30") || args.HasOption("videocu30stats") ) {
    if (!videoReceiveDevice.IsEmpty())
      SetCapability(0, 0, new H323_Cu30Capability(*this, statsDir, _width, _height, videoCu30Stats));
    else 
      if (autoStartTransmitVideo)
        AddCapability(new H323_Cu30Capability(*this, statsDir, _width, _height, videoCu30Stats));
  }

#endif

  //Make sure the CIF and QCIF capabilities are in the correct order
  if (!videoReceiveDevice.IsEmpty()) {
    if (videoSize == 1) {
      SetCapability(0, 1, new H323_H261Capability(0, 4, FALSE, FALSE, 6217));
      SetCapability(0, 1, new H323_H261Capability(2, 0, FALSE, FALSE, 6217));
    } else {
      SetCapability(0, 1, new H323_H261Capability(2, 0, FALSE, FALSE, 6217));
      SetCapability(0, 1, new H323_H261Capability(0, 4, FALSE, FALSE, 6217));
    }
  } else if (autoStartTransmitVideo) {
    if (videoSize == 1) {
      AddCapability(new H323_H261Capability(0, 4, FALSE, FALSE, 6217)); //CIF
      AddCapability(new H323_H261Capability(2, 0, FALSE, FALSE, 6217)); //QCIF
    } else {
      AddCapability(new H323_H261Capability(2, 0, FALSE, FALSE, 6217)); //QCIF
      AddCapability(new H323_H261Capability(0, 4, FALSE, FALSE, 6217)); //CIF
    }
  }

  H323_UserInputCapability::AddAllCapabilities(capabilities, 0, P_MAX_INDEX);
  //SetCapability(0, P_MAX_INDEX, new H323_T120Capability);

  if (verbose >= 4)
    cout <<  "Codecs (in preference order):\n" << setprecision(2) << capabilities << endl << endl;


  if (!args.GetOptionString("listenport").IsEmpty())
    listenPort = (WORD)args.GetOptionString("listenport").AsInteger();

  PStringArray interfaceList;
  if (!args.GetOptionString('i').IsEmpty()) 
    interfaceList = args.GetOptionString('i').Lines();

  PString interfacePrintable;

  // if no interfaces specified, then bind to all interfaces with a single Listener
  // otherwise, bind to specific interfaces
  if (interfaceList.GetSize() == 0) {
    PIPSocket::Address interfaceAddress(INADDR_ANY);
    H323ListenerTCP * listener = new H323ListenerTCP(*this, interfaceAddress, listenPort);
    if (!StartListener(listener)) {
      cerr <<  "Could not open H.323 listener port on "
           << listener->GetListenerPort() << endl;
      delete listener;
      return FALSE;
    }
    interfacePrintable = psprintf("ALL:%i", listenPort);
  } else {
    for (i = 0; i < interfaceList.GetSize(); i++) {

      PString interfaceStr = interfaceList[i];
      WORD interfacePort = listenPort;

      PINDEX pos;
      if ((pos = interfaceStr.Find(':')) != P_MAX_INDEX) {
        interfacePort = (WORD)interfaceStr.Mid(pos+1).AsInteger();
        interfaceStr = interfaceStr.Left(pos);
      }
      interfacePrintable &= interfaceStr + ":" + PString(PString::Unsigned, interfacePort);
      PIPSocket::Address interfaceAddress(interfaceStr);

      H323ListenerTCP * listener = new H323ListenerTCP(*this, interfaceAddress, interfacePort);
      if (!StartListener(listener)) {
        cerr << "Could not open H.323 listener port on "
             << interfaceAddress << ":" << interfacePort << endl;
	delete listener;
        return FALSE;
      }
    }
  }

  if (verbose >= 3) 
    cout << "Listening interfaces : " << interfacePrintable << endl;

#ifdef SetGatekeeperPassword
  // Initialise the security info
  if (args.HasOption("password")) {
    SetGatekeeperPassword(args.GetOptionString("password"));
    cout << "Enabling H.235 security access to gatekeeper." << endl;
  }
#endif

  if (args.HasOption('g')) {
    PString gkName = args.GetOptionString('g');
    H323TransportUDP * rasChannel;
    if (args.GetOptionString('i').IsEmpty())
      rasChannel  = new H323TransportUDP(*this);
    else {
      PIPSocket::Address interfaceAddress(args.GetOptionString('i'));
      rasChannel  = new H323TransportUDP(*this, interfaceAddress);
    }
    if (SetGatekeeper(gkName, rasChannel)) {
      if (verbose >= 3) 
        cout << "Gatekeeper set: " << *gatekeeper << endl;
    } else {
      cerr << "Error registering with gatekeeper at \"" << gkName << '"' << endl;
      return FALSE;
    }
  }
  else if (args.HasOption('G')) {
    PString gkIdentifier = args.GetOptionString('G');
    if (verbose >= 2)
      cout << "Searching for gatekeeper with id \"" << gkIdentifier << "\" ..." << flush;
    if (LocateGatekeeper(gkIdentifier)) {
      if (verbose >= 3) 
        cout << "Gatekeeper set: " << *gatekeeper << endl;
    } else {
      cerr << "Error registering with gatekeeper at \"" << gkIdentifier << '"' << endl;
      return FALSE;
    }
  }
  else if (!args.HasOption('n') || args.HasOption('r')) {
    if (verbose >= 2)
      cout << "Searching for gatekeeper..." << flush;
    if (DiscoverGatekeeper(new H323TransportUDP(*this))) {
      if (verbose >= 2) 
        cout << "\nGatekeeper found: " << *gatekeeper << endl;
    } else {
      if (verbose >= 2)
        cerr << "\nNo gatekeeper found." << endl;
      if (args.HasOption("require-gatekeeper")) 
        return FALSE;
    }
  }

  ringThread = NULL;

  if (!args.HasOption("autodisconnect"))
    autoDisconnect = 0;
  else {
    autoDisconnect = args.GetOptionString("autodisconnect").AsInteger();
    if (autoDisconnect < 0) {
      cerr << "autodisconnect must be > 0" << endl;
      return FALSE;
    }
  }

  return TRUE;
}
  

#ifdef HAS_OSS
BOOL MyH323EndPoint::InitialiseMixer(PConfigArgs & args, int _verbose)
{
  mixerDev = -1;

  // make sure mixer isn't disabled
  if (args.HasOption("no-sound-mixer"))
    return TRUE;

  PString mixerDeviceName = DEFAULT_MIXER;
  if (args.HasOption("mixer"))
    mixerDeviceName = args.GetOptionString("sound-mixer");

  mixerDev = ::open(mixerDeviceName, O_RDWR);
  if (mixerDev < 0) {
    cerr << "warning: Cannot open mixer device " << mixerDeviceName
         << ": " << ::strerror(errno) << endl;
    return TRUE;
  }

  char * mixerChanNames[] = SOUND_DEVICE_NAMES;
  int numMixerChans = SOUND_MIXER_NRDEVICES;

  // get the current record channel setting, and save it
  if (::ioctl(mixerDev, SOUND_MIXER_READ_RECSRC, &savedMixerRecChan) < 0) {
    cerr << "warning: cannot get current mixer record channels" << endl;
    savedMixerRecChan = -1;
  }

  // if the user specified a record channel, then find it
  // otherwise, find the currently select record channel
  if (args.HasOption("sound-recchan")) {
    PCaselessString mixerRecChanName = args.GetOptionString("sound-recchan");
    int i;
    for (i = 0; i < numMixerChans; i++) 
      if (mixerRecChanName *= mixerChanNames[i])
        break;
    if (i == numMixerChans) {
      cerr << "error: Cannot find record mixer channel " << mixerDeviceName << endl;
      return FALSE;
    }
    mixerRecChan = i;
  } else {
    int i;
    for (i = 0; i < numMixerChans; i++) 
      if (savedMixerRecChan & (1 << i))
        break;
    if (i == numMixerChans)
      mixerRecChan = SOUND_MIXER_MIC;
    else 
      mixerRecChan = i;
  } 

  PString volStr;
  if (args.HasOption("sound-recvol"))
    volStr = args.GetOptionString("sound-recvol");
  else if (args.HasOption("recvol")) 
    volStr = args.GetOptionString("recvol");

  if (volStr.IsEmpty()) {
    ::ioctl(mixerDev, MIXER_READ(mixerRecChan), &ossRecVol);
    ossRecVol &= 0xff;
  } else {
    ossRecVol = (unsigned)volStr.AsReal();
    int volVal = ossRecVol | (ossRecVol << 8);
    ::ioctl(mixerDev, MIXER_WRITE(mixerRecChan), &volVal);
  }

  if (args.HasOption("sound-playvol"))
    volStr = args.GetOptionString("sound-playvol");
  else if (args.HasOption("playvol")) 
    volStr = args.GetOptionString("playvol");

  if (volStr.IsEmpty()) {
    ::ioctl(mixerDev, SOUND_MIXER_READ_VOLUME, &ossPlayVol);
    ossPlayVol &= 0xff;
  } else {
    ossPlayVol = (unsigned)volStr.AsReal();
    int volVal = ossPlayVol | (ossPlayVol << 8);
    ::ioctl(mixerDev, SOUND_MIXER_WRITE_PCM, &volVal);
  }

  if (verbose >= 3) {
    cout << "Recording using mixer channel " << mixerChanNames[mixerRecChan] << endl;
    cout << "Record volume is " << ossRecVol << endl;
    cout << "Play volume is " << ossPlayVol << endl;
  }

  return TRUE;
}
#endif


BOOL MyH323EndPoint::SetSoundDevice(PConfigArgs & args,
                                    const char * optionName,
                                    PSoundChannel::Directions dir)
{
  if (!args.HasOption(optionName))
    return TRUE;

  PString dev = args.GetOptionString(optionName);

  if (dir == PSoundChannel::Player) {
    if (SetSoundChannelPlayDevice(dev))
      return TRUE;
  }
  else {
    if (SetSoundChannelRecordDevice(dev))
      return TRUE;
  }

  cerr << "Device for " << optionName << " (\"" << dev << "\") must be one of:\n";

  PStringArray names = PSoundChannel::GetDeviceNames(dir);
  for (PINDEX i = 0; i < names.GetSize(); i++)
    cerr << "  \"" << names[i] << "\"\n";

  return FALSE;
}


H323Connection * MyH323EndPoint::CreateConnection(unsigned callReference)
{
  return new MyH323Connection(*this, callReference,
                              currentCallOptions.noFastStart,
                              currentCallOptions.noH245Tunnelling,
                              (WORD)currentCallOptions.jitter,
                              verbose);
}


BOOL MyH323EndPoint::OnIncomingCall(H323Connection & connection,
                                    const H323SignalPDU & setupPDU,
                                    H323SignalPDU &)
{
  // get the default call options
  currentCallOptions = defaultCallOptions;

  // get remote address so we can call back later
  PString lastCallingParty = connection.GetSignallingChannel()->GetRemoteAddress().GetHostName();

  PConfig config("Callers");
  int index = config.GetInteger("index");
  PString lastLastCallingParty = config.GetString(PString(PString::Unsigned, index));
  index = (index + 1) % LAST_CALL_COUNT;
  PTime now;
  PString indexStr = PString(PString::Unsigned, index);
  config.SetString(indexStr,           lastCallingParty);
  config.SetString(indexStr + "_Time", now.AsString());
  config.SetString("index",            indexStr);

  // Check for setup parameter
  if (setupPDU.m_h323_uu_pdu.HasOptionalField(H225_H323_UU_PDU::e_nonStandardData)) {
    PString param = setupPDU.m_h323_uu_pdu.m_nonStandardData.m_data.AsString();
    if (!param)
      cout << "Received non-standard parameter data in Setup PDU: \"" << param << "\"." << endl;
  }


  if (!alwaysForwardParty.IsEmpty()) {
    cout << "Forwarding call to \"" << alwaysForwardParty << "\"." << endl;
    return !connection.ForwardCall(alwaysForwardParty);
  }

  // incoming call is accepted if no call in progress 
  // unless the xJack is open and phone is off onhook

  if (!currentCallToken.IsEmpty())
    cout << "WARNING: current call token not empty" << endl;

  if (currentCallToken.IsEmpty()
#ifdef HAS_IXJ
        && !(xJack.IsOpen() && (!autoHook && xJack.IsLineOffHook(OpalIxJDevice::POTSLine)))
#endif
  ) {
    // get the current call token
    currentCallToken = connection.GetCallToken();
    return TRUE;
  }

  if (busyForwardParty.IsEmpty()) {
    PTime now;
    cout << "Incoming call from \"" << connection.GetRemotePartyName() << "\" rejected at " << now << ", line busy!" << endl;
    connection.ClearCall(H323Connection::EndedByLocalBusy);
#ifdef HAS_IXJ
#ifdef IXJCTL_VMWI
    if (callerIdCallWaitingEnable) {
      PString callerId = ((MyH323Connection &)connection).GetCallerIdString();
      cout << "Sending caller on call waiting ID " << callerId << endl;
      xJack.SendCallerIDOnCallWaiting(OpalIxJDevice::POTSLine, callerId);
    }
#endif
#endif
    return FALSE;
  }

  cout << "Forwarding call to \"" << busyForwardParty << "\"." << endl;
  return !connection.ForwardCall(busyForwardParty);
}


BOOL MyH323EndPoint::OnConnectionForwarded(H323Connection & /*connection*/,
                                           const PString & forwardParty,
                                           const H323SignalPDU & /*pdu*/)
{
  if (MakeCall(forwardParty, currentCallToken)) {
    cout << GetLocalUserName() << " is being forwarded to host " << forwardParty << endl;
    return TRUE;
  }

  cout << "Error forwarding call to \"" << forwardParty << '"' << endl;
  return FALSE;
}


void MyH323EndPoint::OnNoAnswerTimeout(PTimer &, INT)
{
  H323Connection * connection = FindConnectionWithLock(currentCallToken);
  if (connection != NULL) {
    cout << "Forwarding call to \"" << noAnswerForwardParty << "\"." << endl;
    connection->ForwardCall(noAnswerForwardParty);
    connection->Unlock();
  }
}

void MyH323EndPoint::OnConnectionEstablished(H323Connection & connection,
                                             const PString & /*token*/)
{
  cout << "Call with \"" << connection.GetRemotePartyName() << "\" established." << endl;
  uiState = uiCallInProgress;
}

void MyH323EndPoint::OnAutoDisconnect(PTimer &, INT)
{
  if (currentCallToken.IsEmpty()) 
    cout << "Autodisconnect occurred without current call token" << endl;
  else {
    ClearCall(currentCallToken);
    cout << "Autodisconnect triggered" << endl;
  }
}

void MyH323EndPoint::TriggerDisconnect()
{
  // we have an autodisconnect timer specified, start the timer
  if (autoDisconnect <= 0)
    PTRACE(2, "Main\tAuto disconnect not triggered");
  else {
    PTRACE(2, "Main\tAuto disconnect triggered");
    autoDisconnectTimer.SetNotifier(PCREATE_NOTIFIER(OnAutoDisconnect));
    autoDisconnectTimer = PTimeInterval(autoDisconnect * 100);
  }
}


void MyH323EndPoint::OnConnectionCleared(H323Connection & connection, const PString & clearedCallToken)
{
  // stop any ringing that is occurring
  StopRinging();

  // ignore connections that are not the current connection
  if (clearedCallToken != currentCallToken)
    return;

  // update values for current call token and call forward call token:
  if (!callTransferCallToken) {
    // after clearing the first call during a call proceeding,
    // the call transfer call token becomes the new call token
    currentCallToken = callTransferCallToken;
    callTransferCallToken = PString();
  }
  else
    currentCallToken = PString(); // indicate that our connection is now cleared

  // indicate call has hungup
  uiState = uiCallHungup;

  if (verbose != 0) {

    BOOL printDuration = TRUE;

    PString remoteName = '"' + connection.GetRemotePartyName() + '"';

    switch (connection.GetCallEndReason()) {
      case H323Connection::EndedByCallForwarded :
        printDuration = FALSE;   // Don't print message here, was printed when forwarded
        break;
      case H323Connection::EndedByRemoteUser :
        cout << remoteName << " has cleared the call";
        break;
      case H323Connection::EndedByCallerAbort :
        cout << remoteName << " has stopped calling";
        break;
      case H323Connection::EndedByRefusal :
        cout << remoteName << " did not accept your call";
        break;
      case H323Connection::EndedByRemoteBusy :
        cout << remoteName << " was busy";
        break;
      case H323Connection::EndedByRemoteCongestion :
        cout << "Congested link to " << remoteName;
        break;
      case H323Connection::EndedByNoAnswer :
        cout << remoteName << " did not answer your call";
        break;
      case H323Connection::EndedByTransportFail :
        cout << "Call with " << remoteName << " ended abnormally";
        break;
      case H323Connection::EndedByCapabilityExchange :
        cout << "Could not find common codec with " << remoteName;
        break;
      case H323Connection::EndedByNoAccept :
        cout << "Did not accept incoming call from " << remoteName;
        break;
      case H323Connection::EndedByAnswerDenied :
        cout << "Refused incoming call from " << remoteName;
        break;
      case H323Connection::EndedByNoUser :
        cout << "Gatekeeper could find user " << remoteName;
        break;
      case H323Connection::EndedByNoBandwidth :
        cout << "Call to " << remoteName << " aborted, insufficient bandwidth.";
        break;
      case H323Connection::EndedByUnreachable :
        cout << remoteName << " could not be reached.";
        break;
      case H323Connection::EndedByHostOffline :
        cout << remoteName << " is not online.";
        break;
      case H323Connection::EndedByNoEndPoint :
        cout << "No phone running for " << remoteName;
        break;
      case H323Connection::EndedByConnectFail :
        cout << "Transport error calling " << remoteName;
        break;
      default :
        cout << "Call with " << remoteName << " completed";
    }

    if (printDuration)
      cout << ", duration "
           << setprecision(0) << setw(5)
           << (PTime() - connection.GetConnectionStartTime())
           << endl;
  }

  if (!hasMenu && terminateOnHangup) {
    exitFlag.Signal();
    exitFlag.Wait();
  }
}


BOOL MyH323EndPoint::OpenAudioChannel(H323Connection & connection,
                                         BOOL isEncoding,
                                         unsigned bufferSize,
                                         H323AudioCodec & codec)
{
  codec.SetSilenceDetectionMode(currentCallOptions.noSilenceSuppression ?
                                   H323AudioCodec::NoSilenceDetection :
                                   H323AudioCodec::AdaptiveSilenceDetection);

#ifdef HAS_IXJ
  if (xJack.IsOpen()) {
    PTRACE(2, "xJack\tAttaching channel to codec");
    return codec.AttachChannel(new OpalLineChannel(xJack, OpalIxJDevice::POTSLine, codec));
  }
#endif

  if (H323EndPoint::OpenAudioChannel(connection, isEncoding, bufferSize, codec))
    return TRUE;

  cerr << "Could not open sound device ";
  if (isEncoding)
    cerr << GetSoundChannelRecordDevice();
  else
    cerr << GetSoundChannelPlayDevice();
  cerr << " - Check permissions or full duplex capability." << endl;

  return FALSE;
}


BOOL MyH323EndPoint::OpenVideoChannel(H323Connection & connection,
                                     BOOL isEncoding,
                                     H323VideoCodec & codec)
{
  PVideoChannel      * channel = new PVideoChannel;
  PVideoOutputDevice * device  = NULL;
  PVideoInputDevice  * grabber = NULL;
  
  PString nameStr = isEncoding ? PString("Local") : connection.GetRemotePartyName();

  if (isEncoding) {
    PAssert(autoStartTransmitVideo, "video encoder created without enable");
    codec.SetTxQualityLevel(videoTxQuality);
    codec.SetBackgroundFill(videoFill);   
    codec.SetAverageBitRate(videoBitRate);

    unsigned newFrameWidth,newFrameHeight;              
    newFrameWidth =  352>>(1-videoSize);                
    newFrameHeight = 288>>(1-videoSize);                

    if (videoFake)
      grabber = new PFakeVideoInputDevice();
    else
      grabber = new PVideoInputDevice();
    
    if ( !grabber->Open(videoDevice, FALSE)) {
      PTRACE(3,"Failed to open the camera device");
      goto errVideo;
    }
    if ( !grabber->SetVideoFormat(
         videoIsPal ? PVideoDevice::PAL : PVideoDevice::NTSC)) {
      PTRACE(3,"Failed to set format to " << (videoIsPal ? "PAL" : "NTSC"));
      goto errVideo;
    }
    if ( !grabber->SetChannel(videoInput) ) {
      PTRACE(3,"Failed to set channel to "<<videoInput);
      goto errVideo;
    }
    if ( !grabber->SetColourFormatConverter("YUV420P") ) {
      PTRACE(3,"Failed to set format to yuv420p");
      goto errVideo;
    }
    if (  !grabber->SetFrameRate(videoFramesPS)) {
      PTRACE(3,"Failed to set framerate to " << videoFramesPS );
      goto errVideo;
    }
    if  ( !grabber->SetFrameSizeConverter(newFrameWidth,newFrameHeight,FALSE) ) {
      PTRACE(3, "Failed to set framesize to "<<newFrameWidth<<"x"<<newFrameHeight);
      goto errVideo;
    }
    PTRACE(3,"OpenVideoChannel\t done. Successfully opened a video camera");
    goto exitVideo;
 
  errVideo:
    delete grabber;
    grabber = (PVideoInputDevice *) new PFakeVideoInputDevice(); //This one never fails.
    grabber->SetColourFormat("YUV420P");
    grabber->SetVideoFormat(PVideoDevice::PAL);  // not actually used for fake video.
    grabber->SetChannel(100);     //NTSC test image.
    grabber->SetFrameRate(10);
    grabber->SetFrameSize(newFrameWidth,newFrameHeight);
    PTRACE(3,"Made a fictitious video camera showing NTSC test frame");
 
  exitVideo:
    grabber->Start();
    channel->AttachVideoReader(grabber);
  }

  if ((!isEncoding) || videoLocal) {
    PAssert(!videoReceiveDevice.IsEmpty(), "video display created without device type");

#ifdef HAS_SDL
     // Dump received video to SDL 
     if (videoReceiveDevice *= "sdl")
       device = new SDLVideoDevice(nameStr,isEncoding,videoPIP);
#endif

#ifdef HAS_X11
    // Dump received video to X11 window
    if (videoReceiveDevice.Left(3) *= "x11") {
      PString str = videoReceiveDevice;
      BOOL shared = str.Right(1) *= "s";
      str = str.Mid(3);
      if (!shared)
         device = new XlibVideoDevice(nameStr,isEncoding,videoPIP);
      else {
        str = str.Left(str.GetLength()-1); 
        device = new ShmXlibVideoDevice(nameStr,isEncoding,videoPIP);
      }
      int depth = str.AsInteger();
      if (depth > 0) 
        ((GenericXlibVideoDevice *)device)->ForceDepth(depth);
    }
#endif
#ifdef WIN32
    // code to specify windows 32 display device,
    // which uses MS windows conversion routines.
#endif
    // Dump video to PPM files
    //can have two ppm video devices. 99==local    00==received.
    if (videoReceiveDevice *= "ppm")
      device = new PPMVideoOutputDevice(isEncoding? 99:0);
  }
#ifdef HAS_VGALIB    //vgalib can only do receive video device.
  if (!isEncoding) {
     // Dump received video to VGA 
     if (videoReceiveDevice *= "svga")
       device = new LinuxSVGAFullOutputDevice();

     else if (videoReceiveDevice *= "svga256")
       device = new LinuxSVGA256OutputDevice();
  }
#endif
  // Dump video to nowhere
  if ((videoReceiveDevice *= "null") || (isEncoding&&(!videoLocal))) 
    device = new NullVideoOutputDevice();

  if (device == NULL)  
    PError << "unknown video output device \"" << videoReceiveDevice << "\"" << endl;
  PAssert(device != NULL, "NULL video device");
  

  device->SetFrameSize(352>>(1-videoSize), 288>>(1-videoSize));
  //device->SetFrameSize(codec.GetWidth(), codec.GetHeight());

  channel->AttachVideoPlayer(device);
 
  //Select true, delete video chanel on closing codec.
  return codec.AttachChannel(channel,TRUE);
}


H323Connection * MyH323EndPoint::SetupTransfer(const PString & token,
                                               const PString & callIdentity,
                                               const PString & remoteParty,
                                               PString & newToken)
{
  H323Connection * conn = H323EndPoint::SetupTransfer(token,
                                                      callIdentity, 
                                                      remoteParty,
                                                      newToken);
  callTransferCallToken = newToken;
  return conn;
}


//
//  if gateway is empty, then dest is assumed to be a IP address and optional port
//  if gateway is non-empty, then gateway is assumed to be an IP address and optional port, and
//  dest is passed to the gateway as the e164 address
//
void MyH323EndPoint::MakeOutgoingCall(const PString & dest,
                                      const PString & gateway)
{
  MakeOutgoingCall(dest, gateway, defaultCallOptions);
}

void MyH323EndPoint::MakeOutgoingCall(const PString & dest,
                                      const PString & gateway,
                                      CallOptions callOptions)
{
  currentCallOptions = callOptions;

  PString fullAddress;

  if (!gateway)
    fullAddress = gateway;
  else 
    fullAddress = dest;

  if ((fullAddress.Find(':') == P_MAX_INDEX) && (callOptions.connectPort != H323ListenerTCP::DefaultSignalPort))
    fullAddress += psprintf(":%i", currentCallOptions.connectPort);

  if (!gateway)
    fullAddress = dest.Trim() + '@' + fullAddress;


  if (!MakeCall(fullAddress, currentCallToken)) {
    cout << "Error making call to \"" << fullAddress << '"' << endl;
    return;
  }

  PConfig config("Calls");
  int index = config.GetInteger("index");
  PString lastCalledParty = config.GetString(PString(PString::Unsigned, index));
  index = (index + 1) % LAST_CALL_COUNT;
  PTime now;
  PString indexStr = PString(PString::Unsigned, index);
  config.SetString(indexStr,           fullAddress);
  config.SetString(indexStr + "_Time", now.AsString());
  config.SetString("index",            indexStr);

  cout << GetLocalUserName() << " is calling host " << fullAddress << endl;
  uiState = uiConnectingCall;
}

void MyH323EndPoint::NewSpeedDial(const PString & ostr)
{
  PString str = ostr;
  PINDEX idx = str.Find(' ');
  if (str.IsEmpty() || (idx == P_MAX_INDEX)) {
    cout << "Must specify speedial number and string" << endl;
    return;
  }

  PString key  = str.Left(idx).Trim();
  PString data = str.Mid(idx).Trim();

  PConfig config("Speeddial");
  config.SetString(key, data);

  cout << "Speedial " << key << " set to " << data << endl;
}

void MyH323EndPoint::ListSpeedDials()
{
  PConfig config("Speeddial");
  PStringList keys = config.GetKeys();
  if (keys.GetSize() == 0) {
    cout << "No speed dials defined" << endl;
    return;
  }

  PINDEX i;
  for (i = 0; i < keys.GetSize(); i++) 
    cout << keys[i] << ":   " << config.GetString(keys[i]) << endl;
}

//
// StartCall accepts any of the following types of arguments
//    speedial '#'      lookup the string in the registry, and continue processing
//    ipaddress         dial this IP address or hostname
//    num ' ' gateway   dial the number using the specified gateway
//    

void MyH323EndPoint::StartCall(const PString & ostr)
{
  PString str = ostr.Trim();
  if (str.IsEmpty()) 
    cout << "Must supply hostname to connect to!\n";

  CallOptions callOptions = defaultCallOptions;

  // check for speed dials, and match wild cards as we go
  PString key, prefix;
  if ((str.GetLength() > 1) && (str[str.GetLength()-1] == '#')) {

    key = str.Left(str.GetLength()-1).Trim();

    str = PString();
    PConfig config("Speeddial");
    PINDEX p;
    for (p = key.GetLength(); p > 0; p--) {

      PString newKey = key.Left(p);
      prefix = newKey;
      PINDEX q;

      // look for wild cards
      str = config.GetString(newKey + '*').Trim();
      if (!str.IsEmpty())
        break;

      // look for digit matches
      for (q = p; q < key.GetLength(); q++)
        newKey += '?';
      str = config.GetString(newKey).Trim();
      if (!str.IsEmpty())
        break;
    }
    if (str.IsEmpty()) {
      cout << "Speed dial \"" << key << "\" not defined";
      if (gatekeeper != NULL) {
        cout << ", trying gatekeeper ..." << endl;
        MakeOutgoingCall(key, PString(), callOptions);
        return;
      }
      else
        cout << endl;
    }
    else if ((p = str.Find('(')) != P_MAX_INDEX) {
      PString argStr = str.Mid(p);
      if (argStr.GetLength() > 0 && argStr[argStr.GetLength()-1] == ')')
        argStr = argStr.Mid(1, argStr.GetLength()-2);
      PArgList strArgs(argStr,
                       "f-fast-disable."
                       "T-h245tunneldisable."
                       "e-silence."
                       "j-jitter:"
                       "-connectport:"
                       "-connectring:");
      callOptions.Initialise(strArgs);
      str = str.Left(p);
      cout << "Per connection call options set: " << argStr << endl
           << callOptions
           << endl;
    }
  }

  if (!str.IsEmpty()) {
    PINDEX idx = str.Find(' ');
    if (idx == P_MAX_INDEX) {
      if (!key && (str[0] == '@'))
        MakeOutgoingCall(key, str.Mid(1), callOptions);
      else if (!key && !prefix && (str[0] == '%')) {
        if (key.Left(prefix.GetLength()) == prefix)
          key = key.Mid(prefix.GetLength());
        MakeOutgoingCall(key, str.Mid(1), callOptions);
      } else
        MakeOutgoingCall(str, PString(), callOptions);
    } else {
      PString host = str.Left(idx).Trim();
      PString gw   = str.Mid(idx).Trim();
      MakeOutgoingCall(host, gw, callOptions);
    }
    return;
  }

  uiState = MyH323EndPoint::uiCallHungup;
}

void MyH323EndPoint::AwaitTermination()
{
  PThread * userInterfaceThread = NULL;
  if (hasMenu)
    userInterfaceThread = new UserInterfaceThread(*this);

#ifndef HAS_IXJ

  exitFlag.Wait();

#else

  if (!xJack.IsOpen())
    exitFlag.Wait();
  else {

    speakerphoneSwitch = FALSE;
    BOOL oldOnHook     = TRUE;
    BOOL oldRealOnHook = TRUE;
    int  olduiState    = uiStateIllegal;
    PTime offHookTime;

    PString digits;

    // poll the handset every 100ms looking for state changes
    while (!exitFlag.Wait(100)) {

      // lock the user interface state whilst we change it
      uiStateMutex.Wait();

      // get real hook state
      BOOL realOnHook = !xJack.IsLineOffHook(OpalIxJDevice::POTSLine);
      BOOL onHook     = realOnHook;

      // if in speakerphone mode,
      if (speakerphoneSwitch) {

        // if the phone is onhook then don't look at the real hook state
        if (realOnHook)
          onHook = FALSE;

        // if the handset just went offhook, then get out of speakerphone mode
        else if (realOnHook != oldRealOnHook) {
          speakerphoneSwitch = FALSE;
          xJack.EnableAudio(0, TRUE);
          if (verbose > 1)
            cout << "Speakerphone off" << endl;
        }
      }

      // handle onhook/offhook transitions
      if (onHook != oldOnHook) {
        if (onHook) {
          digits = PString();
          HandleHandsetOnHook();
        } else {
          HandleHandsetOffHook();
          offHookTime = PTime();
        }
        HandleStateChange(onHook);
        olduiState = uiState;
      } 

      // handle timeouts and DTMF digits
      if (!onHook) {
        HandleHandsetTimeouts(offHookTime);
        HandleHandsetDTMF(digits);
      }

      // handle any state changes
      if (uiState != olduiState) {
        offHookTime = PTime();
        HandleStateChange(onHook);
      }

      // save hook state so we can detect changes
      oldOnHook     = onHook;
      oldRealOnHook = realOnHook;

      // save the old UI state so we can detect changes
      olduiState = uiState;

      uiStateMutex.Signal();
    }
  }
#endif

  if (userInterfaceThread != NULL) {
    userInterfaceThread->Terminate();
    userInterfaceThread->WaitForTermination();
    delete userInterfaceThread;
  }
}

#ifdef HAS_IXJ

#if 0
static char * stateNames[] = {
  "Dialtone",
  "AnsweringCall",
  "ConnectingCall",
  "WaitingForAnswer",
  "CallInProgress",
  "CallHungup",
  "StateIllegal"
};
#endif


void MyH323EndPoint::HandleStateChange(BOOL onHook)
{
  switch (uiState) {

    // dialtone whilst no call active
    case uiDialtone:
      if (onHook)
        xJack.RingLine(0, 0);
      else if (autoHook) {
        xJack.StopTone(0);
      } else {
        cout << "Playing dialtone" << endl;
        xJack.PlayTone(0, OpalLineInterfaceDevice::DialTone);
      }
      break;

    // no tone whilst waiting for remote party to connect
    case uiConnectingCall:
      if (onHook)
        xJack.RingLine(0, 0);
      else
        xJack.StopTone(0);
      break;

    // when connected, play ring tone
    case uiWaitingForAnswer:
      if (onHook)
        xJack.RingLine(0, 0);
      else {
        cout << "Playing ringtone" << endl;
        xJack.PlayTone(0, OpalLineInterfaceDevice::RingTone);
      }
      break;

    // when a call is in progress, stop all tones and remove DTMF tones from the stream
    case uiCallInProgress:
      if (onHook) 
        xJack.RingLine(0, 0);
      else {
        xJack.StopTone(0);
        xJack.SetRemoveDTMF(0, TRUE);
      }
      break;

    // remote end has hungup
    case uiCallHungup:
      if (terminateOnHangup)
        exitFlag.Signal();

      if (autoHook || onHook) {
        uiState = uiDialtone;
        xJack.RingLine(0, 0);
      } else {
        if (dialAfterHangup) 
          uiState = uiDialtone;
        else 
          xJack.PlayTone(OpalIxJDevice::POTSLine, OpalLineInterfaceDevice::BusyTone);
      }
      break;

    case uiAnsweringCall:
      if (autoHook || !onHook) 
        xJack.StopTone(0);
      else {
        xJack.SetCallerID(OpalIxJDevice::POTSLine, "");
        if (callerIdEnable) {
          MyH323Connection * connection = (MyH323Connection *)FindConnectionWithLock(currentCallToken);
          if (connection != NULL) {
            xJack.SetCallerID(OpalIxJDevice::POTSLine, connection->GetCallerIdString());
            connection->Unlock();
          }
        }
        xJack.RingLine(0, 0x33);
      }
      break;

   default:
      if (!onHook || autoHook) 
        xJack.StopTone(0);
      else
        xJack.RingLine(0, 0);
      break;
  }
}

void MyH323EndPoint::HandleHandsetOffHook()
{
  if (verbose > 1)
    cout << "Offhook - ";

//  if (speakerphoneSwitch) {
//    if (verbose > 1)
//      cout << "speakerphone off - ";
//    speakerphoneSwitch = FALSE;
//    xJack.EnableAudio(0, TRUE);
//  }
     
  switch (uiState) {

    case uiDialtone:
      if (!autoDial) {
        if (verbose > 1)
          cout << "auto-dialing " << autoDial << endl;
        StartCall(autoDial);
      } else {
        if (verbose > 1)
          cout << "dialtone" << endl;
      }
    break;
  
    case uiConnectingCall:
      if (verbose > 1)
        cout << "call connecting" << endl;
      break;   
  
    case uiAnsweringCall:
      if (verbose > 1)
        cout << "answering call" << endl;
      AnswerCall(H323Connection::AnswerCallNow);
      break;
  
    case uiWaitingForAnswer:
      if (verbose > 1)
        cout << "waiting for remote answer" << endl;
      break;

    case uiCallInProgress:  // should never occur!
      if (verbose > 1)
        cout << "call in progress" << endl;
      break;
  
    default:
      if (verbose > 1)
        cout << "not sure!" << endl;
      break;
  }
}

void MyH323EndPoint::HandleHandsetOnHook()
{
  if (uiState == uiCallHungup)
    uiState = uiDialtone;

  if (verbose > 1) 
    cout << "Onhook - ";

//#ifndef OLD_IXJ_DRIVER
//  if (speakerphoneSwitch) {
//    xJack.EnableAudio(0, FALSE);
//    if (verbose > 1)
//      cout << "speakerphone on." << endl;
//  } else
//#endif
//  {
    if (verbose > 1)
      cout << "ending call." << endl;
    ClearCall(currentCallToken);
//  }
  speakerphoneSwitch = FALSE;
}


void MyH323EndPoint::HandleHandsetDTMF(PString & digits)
{
  char newDigit = xJack.ReadDTMF(0);
  if (newDigit != '\0') {
    digits += newDigit;
    if (!digits) {
      switch (uiState) {
        case uiCallInProgress:
          {
            if (verbose > 1)
              cout << "Sending user indication message " << digits << endl;
            H323Connection * connection = FindConnectionWithLock(currentCallToken);
            if (connection != NULL) {
              connection->SendUserInput(digits);
              connection->Unlock();
            }
            digits = PString();
          }
          break;
    
        case uiDialtone:
          xJack.StopTone(0);
          if (digits.GetLength() > 0) {
            PINDEX i;
            for (i = 0; i < digits.GetLength(); i++)
              if (!isdigit(digits[i]))
                break;
            BOOL allDigits = i == digits.GetLength();

            // handle strings ending in '#'
            if (digits[digits.GetLength()-1] == '#')  {

              // if pressed '#', then redial last number
              if (digits.GetLength() == 1) {
                PConfig config("Calls");
                int index = config.GetInteger("index");
                PString lastCalledParty = config.GetString(PString(PString::Unsigned, index));
                if (lastCalledParty.IsEmpty()) 
                  cout << "No last called party to dial" << endl;
                else {
                  if (!MakeCall(lastCalledParty, currentCallToken)) 
                    cout << "Error making call to \"" << lastCalledParty << '"' << endl;
                  else {
                    if (verbose > 1)
                      cout << "Redialling last number at " << lastCalledParty << endl;
                    uiState = uiConnectingCall;
                  }
                }
              }

              // if pressed '*#', then redial last caller
              else if ((digits.GetLength() == 2) && (digits[0] == '*')) {
                PConfig config("Callers");
                int index = config.GetInteger("index");
                PString lastCallingParty = config.GetString(PString(PString::Unsigned, index));
                if (lastCallingParty.IsEmpty())
                  cout << "No last calling party to dial" << endl;
                else {
                  if (!MakeCall(lastCallingParty, currentCallToken)) 
                    cout << "Error making call to \"" << lastCallingParty << '"' << endl;
                  else {
                    if (verbose > 1)
                      cout << "Calling back last party at " << lastCallingParty << endl;
                    uiState = uiConnectingCall;
                  }
                }
              }

              // if string starts with '*', then convert to IP address
              else if ((digits.GetLength() >= 9) && (digits[0] == '*')) {
                digits = digits.Mid(1, digits.GetLength()-2);
                digits.Replace('*', '.', TRUE);
                StartCall(digits);

              // if there are some digits, then use them as a speed dial
              } else if (digits.GetLength() > 1)
                StartCall(digits);

              // clear out the dialled digits
              digits = PString();

            } else if (allDigits && (digits.GetLength() == 10)) {
              while (digits[0] == '0')
                digits = digits.Mid(1);
              if (digits.GetLength() > 0) {
                StartCall(digits);
                digits = PString();
              }
            }
          }
          break;
  
        default:
          break;
      }
    }
  }
}


void MyH323EndPoint::HandleHandsetTimeouts(const PTime & offHookTime)
{
  int timeout = 0;
  
  switch (uiState) {

    case uiDialtone:
      timeout = DEFAULT_TIMEOUT;
      break;

    case uiConnectingCall:
      timeout = DEFAULT_TIMEOUT;
      break;

    case uiWaitingForAnswer:
      timeout = DEFAULT_TIMEOUT;
      break;

    default:
      break;
  }

  if (timeout > 0) {
    PTime now;
    if ((now - offHookTime) > timeout) {
      if (verbose > 1)
        cout << "Operation timed out" << endl;
      ClearCall(currentCallToken);
      uiState = uiCallHungup;
    }
  }
}

#endif

void MyH323EndPoint::HandleUserInterface()
{
  PConsoleChannel console(PConsoleChannel::StandardInput);

  PTRACE(2, "OhPhone\tUser interface thread started.");

  PStringStream help;
  help << "Select:\n"
          "  0-9 : send user indication message\n"
          "  C   : connect to remote host\n"
          "  T   : Transfer to another host\n"
          "  O   : Hold call\n"
          "  S   : Display statistics\n"
          "  H   : Hang up phone\n"
          "  L   : List speed dials\n"
          "  I   : Show call history\n"
          "  D   : Create new speed dial\n"
          "  {}  : Increase/reduce record volume\n"
          "  []  : Increase/reduce playback volume\n"
          "  V   : Display current volumes\n"
#ifdef HAS_IXJ
          "  A   : turn AEC up/down\n"
#endif
          "  E   : Turn silence supression on/off\n"
          "  F   : Forward call calls to address\n"
         ;

#ifdef HAS_IXJ
#ifndef OLD_IXJ_DRIVER
  if (xJack.IsOpen())
    help << "  P   : Enter speakerphone mode\n";
#endif
#endif

  help << "  X   : Exit program\n";

  for (;;) {

    // display the prompt
    cout << "Command ? " << flush;

    // terminate the menu loop if console finished
    char ch = (char)console.peek();
    if (console.eof()) {
      if (verbose)
        cerr << "\nConsole gone - menu disabled" << endl;
      return;
    }

    if (isdigit(ch)) {
      H323Connection * connection = FindConnectionWithLock(currentCallToken);
      if (connection == NULL) {
        cout << "No call in progress\n";
        console.ignore(INT_MAX, '\n');
      } else {
        PString str;
        console >> str;
        cout << "Sending user indication: " << str << endl;
        connection->SendUserInput(str);
        connection->Unlock();
      }
    }
    else {
      console >> ch;
      switch (tolower(ch)) {
        case '?' :
          cout << help << endl;
          break;

        case 'x' :
        case 'q' :
          cout << "Exiting." << endl;
          ClearAllCalls();
          uiState = uiDialtone;
          exitFlag.Signal();
          console.ignore(INT_MAX, '\n');
          return;

        case 'h' :
          if (!currentCallToken) {
            cout << "Hanging up call." << endl;
            if (!ClearCall(currentCallToken))
              cout << "Could not hang up current call!\n";
            speakerphoneSwitch = FALSE;
          }
          console.ignore(INT_MAX, '\n');
          break;

        case 't' :
          if (!currentCallToken) {
            PString str;
            console >> str;
            cout << "Transferring call to " << str << endl;
            TransferCall(currentCallToken, str.Trim());
          }
          else
            console.ignore(INT_MAX, '\n');
          break;

        case 'o' :
          if (!currentCallToken) {
            cout << "Holding call." << endl;
            HoldCall(currentCallToken, TRUE);
          }
          console.ignore(INT_MAX, '\n');
          break;

        case 'y' :
          AnswerCall(H323Connection::AnswerCallNow);
          console.ignore(INT_MAX, '\n');
          break;

        case 'n' :
          AnswerCall(H323Connection::AnswerCallDenied);
          console.ignore(INT_MAX, '\n');
          break;

        case 'c' :
          if (!currentCallToken.IsEmpty()) 
            cout << "Cannot make call whilst call in progress\n";
          else {
            PString str;
            console >> str;
            StartCall(str.Trim());
          }
          break;

        case 'l' :
          ListSpeedDials();
          break;

        case 'd' :
          {
            PString str;
            console >> str;
            NewSpeedDial(str.Trim());
          }
          break;

        case 'e' :
          if (currentCallToken.IsEmpty())
            cout << "No call in progress" << endl;
          else {
            H323Connection * connection = FindConnectionWithLock(currentCallToken);
            if (connection == NULL) 
              cout << "No connection active.\n";
            else {
              connection->Unlock();
              H323Channel * chan = connection->FindChannel(RTP_Session::DefaultAudioSessionID, FALSE);
              if (chan == NULL)
                cout << "Cannot find audio channel" << endl;
              else {
                H323Codec * rawCodec  = chan->GetCodec();
                if (!rawCodec->IsDescendant(H323AudioCodec::Class()))
                  cout << "Audio channel is not audio!" << endl;
                else {
                  H323AudioCodec * codec = (H323AudioCodec *)rawCodec;
                  H323AudioCodec::SilenceDetectionMode mode = codec->GetSilenceDetectionMode();
                  if (mode == H323AudioCodec::AdaptiveSilenceDetection) {
                    mode = H323AudioCodec::NoSilenceDetection;
                    cout << "Silence detection off" << endl;
                  } else {
                    mode = H323AudioCodec::AdaptiveSilenceDetection;
                    cout << "Silence detection on" << endl;
                  }
                  codec->SetSilenceDetectionMode(mode);
                }
              }
            //  connection->Unlock();
            }
          }
          break;

        case 's' :
          if (currentCallToken.IsEmpty())
            cout << "No call in progress" << endl;
          else {
            H323Connection * connection = FindConnectionWithLock(currentCallToken);
            if (connection == NULL) 
              cout << "No connection statistics available.\n";
            else {
              PTime now;
              PTime callStart = connection->GetConnectionStartTime();
              cout << "Connection statistics:\n   "
                   << "Remote party     : " << connection->GetRemotePartyName() << "\n   "
                   << "Start            : " << callStart << "\n   "
                   << "Duration         : " << setw(5) << setprecision(0) << (now - callStart) << " mins\n   "
                   << "Round trip delay : " << connection->GetRoundTripDelay().GetMilliSeconds() << " msec"
                   << endl;
              RTP_Session * session = connection->GetSession(RTP_Session::DefaultAudioSessionID);
              if (session == NULL)
                cout << "No RTP session statistics available.\n";
              else
                cout << "RTP session statistics:\n   "

                     << session->GetPacketsSent() << '/'
                     << session->GetOctetsSent() 
                     << " packets/bytes sent\n   "

                     << session->GetMaximumSendTime() << '/'
                     << session->GetAverageSendTime() << '/'
                     << session->GetMinimumSendTime() << " max/avg/min send time\n   "

                     << session->GetPacketsReceived() << '/'
                     << session->GetOctetsReceived()
                     << " packets/bytes received\n   "

                     << session->GetMaximumReceiveTime() << '/'
                     << session->GetAverageReceiveTime() << '/'
                     << session->GetMinimumReceiveTime() << " max/avg/min receive time\n   "

                     << session->GetPacketsLost() << '/'
                     << session->GetPacketsOutOfOrder() << '/'
                     << session->GetPacketsTooLate() 
                     << " packets dropped/out of order/late\n   "

		     << endl;


              connection->Unlock();
            }
          }
          break;

#ifdef HAS_IXJ
#ifndef OLD_IXJ_DRIVER
        case 'p' :
          if (xJack.IsOpen()) {
            speakerphoneSwitch = !speakerphoneSwitch;
            xJack.EnableAudio(0, !speakerphoneSwitch);
            if (verbose > 1)
              cout << "Speakerphone " 
                   << (speakerphoneSwitch ? "on" : "off")
                   << endl;
          }
          console.ignore(INT_MAX, '\n');
          break;
#endif

        case 'a' :
          if (xJack.IsOpen()) {
            int aec = xJack.GetAEC(0);
            if (ch == 'a')
              aec--;
            else
              aec++;
            if (aec < 0)
              aec = OpalLineInterfaceDevice::AECAGC;
            else if (aec > OpalLineInterfaceDevice::AECAGC)
              aec = OpalLineInterfaceDevice::AECOff;

            xJack.SetAEC(0, (OpalLineInterfaceDevice::AECLevels)aec);

	    if(aec == OpalLineInterfaceDevice::AECAGC ||
	       (ch == 'a' && aec == OpalLineInterfaceDevice::AECHigh) ||
	       (ch == 'A' && aec == OpalLineInterfaceDevice::AECOff)) {
              unsigned recvol;
	      xJack.GetRecordVolume(0, recvol);
              if (verbose > 2)
                cout << "New volume level is " << recvol << endl;
	    }
            if (verbose > 2)
              cout << "New AEC level is " << AECLevelNames[aec] << endl;
          } else
              cout <<"AEC change ignored as xJack closed"<<endl;
          break;
#endif
        case 'v' :
#ifdef HAS_IXJ
          if (xJack.IsOpen()) {
            unsigned vol;
            xJack.GetPlayVolume(0, vol);
            cout << "Play volume is " << vol << endl;
            xJack.GetRecordVolume(0, vol);
            cout << "Record volume is " << vol << endl;
          }
#endif
#if defined(HAS_IXJ) && defined(HAS_OSS)
          else
#endif
#ifdef HAS_OSS
          {
            cout << "Play volume is " << ossPlayVol << endl;
            cout << "Record volume is " << ossRecVol << endl;
          }
#endif
          break;

        case '[' :
        case ']' :
        case '{' :
        case '}' :
#ifdef HAS_IXJ
          if (xJack.IsOpen()) {
            unsigned vol;
            if (ch == '{' || ch == '}')
              xJack.GetRecordVolume(0, vol);
            else
              xJack.GetPlayVolume(0, vol);

            // adjust volume up or down
            vol += ((ch == '[') || (ch == '{')) ? -5 : 5;
            if (vol < 0)
              vol = 0;
            else if (vol > 100)
              vol = 100;

            // write to hardware
            if (ch == '{' || ch == '}') {
              xJack.SetRecordVolume(0, vol);
              if (verbose > 2)
               cout << "Record volume is " << vol << endl;
            } else {
              xJack.SetPlayVolume(0, vol);
              if (verbose > 2)
               cout << "Play volume is " << vol << endl;
            }
          }
#endif

#if defined(HAS_IXJ) && defined(HAS_OSS)
          else
#endif
#ifdef HAS_OSS
          {
            int vol;
            if (ch == '{' || ch == '}')
              vol = ossRecVol;
            else
              vol = ossPlayVol;
            
            vol += ((ch == '[') || (ch == '{')) ? -5 : 5;
            if (vol < 0)
              vol = 0;
            else if (vol > 100)
              vol = 100;

            if (mixerDev >= 0)  {
              int volVal = vol | (vol << 8);
              if (ch == '{' || ch == '}') {
                ossRecVol = vol;
                ::ioctl(mixerDev, MIXER_WRITE(mixerRecChan), &volVal);
                cout << "Record volume is " << ossRecVol << endl;
              } else {
                ossPlayVol = vol;
                ::ioctl(mixerDev, SOUND_MIXER_WRITE_PCM, &volVal);
                cout << "Play volume is " << ossPlayVol << endl;
              }
            } else
              cout << "Audio setting change ignored as mixer device disabled"<<endl;
          }
#endif
          break;

        case 'f' :
          console >> alwaysForwardParty;
          alwaysForwardParty = alwaysForwardParty.Trim();
          if (!alwaysForwardParty)
            cout << "Forwarding all calls to \"" << alwaysForwardParty << '"' << endl;
          else
            cout << "Call forwarding of all calls disabled." << endl;
          break;

        case 'i' :
        case 'I' :
          {
            PString title;
            if (ch == 'i')
              title   = "Callers";
            else 
              title   = "Calls";
            cout << title << endl;
            PConfig config(title);
            int index = config.GetInteger("index");
            PINDEX i;
            for (i = 0; i < LAST_CALL_COUNT; i++) {
              PString indexStr = PString(PString::Unsigned, index);
              PString number = config.GetString(indexStr);
              if (number.IsEmpty())
                continue;
              cout << indexStr
                   << ": "
                   << number
                   << " at "
                   << config.GetString(indexStr + "_Time")
                   << endl;
              if (index == 0)
                index = LAST_CALL_COUNT-1;
              else
                index--;
            }
          }
          break;
 
        default:
          cout << "Unknown command " << ch << endl;
          console.ignore(INT_MAX, '\n');
          break;
      }
    }
  }
}


void MyH323EndPoint::AnswerCall(H323Connection::AnswerCallResponse response)
{
  if (uiState != uiAnsweringCall)
    return;

  StopRinging();

  H323Connection * connection = FindConnectionWithLock(currentCallToken);
  if (connection == NULL)
    return;

  connection->AnsweringCall(response);
  connection->Unlock();

  if (response == H323Connection::AnswerCallNow) {
    cout << "Accepting call." << endl;
    uiState = uiCallInProgress;
  } else {
    cout << "Rejecting call." << endl;
    uiState = uiCallHungup;
  }
}


void MyH323EndPoint::HandleRinging()
{
  PSoundChannel dev(GetSoundChannelPlayDevice(), PSoundChannel::Player);
  if (!dev.IsOpen()) {
    PTRACE(2, "Cannot open sound device for ring");
    return;
  }

  if (ringDelay < 0) {
    PTRACE(2, "Playing " << ringFile);
    dev.PlayFile(ringFile, TRUE);
  } else {
    PTimeInterval delay(0, ringDelay);
    PTRACE(2, "Playing " << ringFile << " with repeat of " << delay << " ms");
    do {
      dev.PlayFile(ringFile, TRUE);
    } while (!ringFlag.Wait(delay));
  }
}


void MyH323EndPoint::StartRinging()
{
  PAssert(ringThread == NULL, "Ringing thread already present");

  if (!noAnswerForwardParty)
    noAnswerTimer = PTimeInterval(0, noAnswerTime);

  if (!ringFile)
    ringThread = new RingThread(*this);
}


void MyH323EndPoint::StopRinging()
{
  noAnswerTimer.Stop();

  if (ringThread == NULL)
    return;

  ringFlag.Signal();
  ringThread->WaitForTermination();
  delete ringThread;
  ringThread = NULL;
}

void MyH323EndPoint::SendDTMF(const char * tone)
{
#ifdef HAS_IXJ
  xJack.PlayDTMF(0, tone, 200, 100);
#endif
}
///////////////////////////////////////////////////////////////

BOOL CallOptions::Initialise(PArgList & args)
{
  // set default connection options
  noFastStart          = args.HasOption('f');
  noH245Tunnelling     = args.HasOption('T');
  noSilenceSuppression = args.HasOption('e');

  if (args.HasOption("connectring"))
    connectRing = args.HasOption("connectring");

  if (!args.GetOptionString("connectport").IsEmpty())
    connectPort = (WORD)args.GetOptionString("connectport").AsInteger();

  if (args.HasOption('j')) {
    int newjitter = args.GetOptionString('j').AsUnsigned();
    if (newjitter >= 20 && newjitter <= 10000)
      jitter = newjitter;
    else {
      cerr << "Jitter should be between 20 milliseconds and 10 seconds." << endl;
      return FALSE;
    }
  }

  return TRUE;
}

void CallOptions::PrintOn(ostream & strm) const
{
  strm << "FastStart is " << !noFastStart << "\n"
       << "H245Tunnelling is " << !noH245Tunnelling << "\n"
       << "SilenceSupression is " << !noSilenceSuppression << "\n"
       << "Jitter buffer: "  << jitter << " ms\n"
       << "Connect port: " << connectPort << "\n";
}


void MyH323EndPoint::TestVideoGrabber()
{
  MyH323Connection tempConnection(*this,2,FALSE,FALSE,FALSE,FALSE);

 H323Capability * cap;
 PString capName;
#ifdef HAS_CU30
  if(videoCu30Stats) 
    capName="Cu30";
  else
#endif 
    if(videoSize==0)
      capName="H.261-QCIF";
    else
      capName="H.261-CIF";
  
  cap=GetCapabilities().FindCapability(capName);
  if(cap==NULL){
    cout << "Unable to find the codec associated with \"" << capName << 
           "\". Exiting"<<endl;
    return;
  }

  H323Codec *Codec=  cap->CreateCodec(H323Codec::Encoder);

  autoStartTransmitVideo = TRUE;    //Force these options to be on. Guarantees that OpenVideoChannel works.
  videoLocal = TRUE;


 OpenVideoChannel(tempConnection,TRUE,*((H323VideoCodec *)Codec) );

  int bytesInFrame = ( ((H323VideoCodec *)Codec)->GetWidth() * ((H323VideoCodec *)Codec)->GetHeight() * 3 )>>1;
  u_char *videoBuffer = new u_char[bytesInFrame];    


  PVideoChannel *videoChannel =  (PVideoChannel *)Codec->GetRawDataChannel();

  PINDEX  i,maxIterations;
  if(videoCu30Stats)
    maxIterations= videoCu30Stats+5;
  else
    maxIterations = 100;


  for( i=0;i<maxIterations; i++) {
    PTRACE(3,"Video\t Iteration "<<i<<" of test video program");
    ((PVideoChannel *)videoChannel)->DisplayRawData(videoBuffer);
#ifdef HAS_CU30
    if(videoCu30Stats)
      ((H323_Cu30Codec *)Codec)->RecordStatistics(videoBuffer);
#endif
  }

  delete videoBuffer;
  delete Codec;

  return;
}

///////////////////////////////////////////////////////////////

MyH323Connection::MyH323Connection(MyH323EndPoint & ep,
                                   unsigned callReference,
                                   BOOL disableFastStart,
                                   BOOL disableTunneling,
                                   WORD jitter,
                                   int _verbose)
  : H323Connection(ep, callReference, disableFastStart, disableTunneling),
    myEndpoint(ep),
    verbose(_verbose)
{
  SetMaxAudioDelayJitter(jitter);
  channelsOpen = 0;
}


BOOL MyH323Connection::OnSendSignalSetup(H323SignalPDU & setupPDU)
{
  if (!myEndpoint.setupParameter) {
    setupPDU.m_h323_uu_pdu.IncludeOptionalField(H225_H323_UU_PDU::e_nonStandardData);
    setupPDU.m_h323_uu_pdu.m_nonStandardData.m_nonStandardIdentifier.SetTag(H225_NonStandardIdentifier::e_h221NonStandard);
    endpoint.SetH221NonStandardInfo(setupPDU.m_h323_uu_pdu.m_nonStandardData.m_nonStandardIdentifier);
    setupPDU.m_h323_uu_pdu.m_nonStandardData.m_data = myEndpoint.setupParameter;
  }
  return TRUE;
}


H323Connection::AnswerCallResponse
     MyH323Connection::OnAnswerCall(const PString & caller,
                                    const H323SignalPDU & /*setupPDU*/,
                                    H323SignalPDU & /*connectPDU*/)
{
  PTRACE(1, "H225\tOnWaitForAnswer");

  PTime now;

  if (myEndpoint.autoAnswer) {
    cout << "Automatically accepting call at " << now << endl;
    return AnswerCallNow;
  }

  myEndpoint.currentCallToken = GetCallToken();
  myEndpoint.uiState = MyH323EndPoint::uiAnsweringCall;


  cout << "Incoming call from \""
       << caller
       << "\" at "
       << now
       << ", answer call (Y/n)? "
       << flush;

  myEndpoint.StartRinging();
  
  return AnswerCallPending;
}


BOOL MyH323Connection::OnStartLogicalChannel(H323Channel & channel)
{
  if (!H323Connection::OnStartLogicalChannel(channel))
    return FALSE;

  if (verbose >= 2) {
    cout << "Started logical channel: ";

    switch (channel.GetDirection()) {
      case H323Channel::IsTransmitter :
        cout << "sending ";
        break;

      case H323Channel::IsReceiver :
        cout << "receiving ";
        break;

      default :
        break;
    }

    cout << channel.GetCapability() << endl;  
  }

  if (channel.GetDirection() == H323Channel::IsReceiver) {
    int videoQuality = myEndpoint.videoQuality;
    if (channel.GetCodec()->IsDescendant(H323VideoCodec::Class()) && (videoQuality >= 0)) {

      //const H323_H261Capability & h261Capability = (const H323_H261Capability &)channel.GetCapability();
      //if (!h261Capability.GetTemporalSpatialTradeOffCapability()) 
      //  cout << "Remote endpoint does not allow video quality configuration" << endl;
      //else {
        cout << "Requesting remote endpoint to send video quality " << videoQuality << "/31" << endl;
  
        // kludge to wait for channel to ACK to be sent
        PThread::Current()->Sleep(2000);
  
        H323ControlPDU pdu;
        H245_CommandMessage & command = pdu.Build(H245_CommandMessage::e_miscellaneousCommand);
  
        H245_MiscellaneousCommand & miscCommand = command;
        miscCommand.m_logicalChannelNumber = (unsigned)channel.GetNumber();
        miscCommand.m_type.SetTag(H245_MiscellaneousCommand_type::e_videoTemporalSpatialTradeOff);
        PASN_Integer & value = miscCommand.m_type;
        value = videoQuality;
        WriteControlPDU(pdu);
      //}
    }
  }

  // adjust the count of channels we have open
  channelsOpen++;

  PTRACE(2, "Main\tchannelsOpen = " << channelsOpen);

  // if we get to two channels (one open, one receive), then (perhaps) trigger a timeout for shutdown
  if (channelsOpen == 2)
    myEndpoint.TriggerDisconnect();

  return TRUE;
}

void MyH323Connection::OnClosedLogicalChannel(H323Channel & channel)
{
  channelsOpen--;
  H323Connection::OnClosedLogicalChannel(channel);
}

BOOL MyH323Connection::OnAlerting(const H323SignalPDU & /*alertingPDU*/,
                                  const PString & username)
{
  PAssert((myEndpoint.uiState == MyH323EndPoint::uiConnectingCall) ||
          (myEndpoint.uiState == MyH323EndPoint::uiWaitingForAnswer) ||
          !myEndpoint.callTransferCallToken,
     psprintf("Alerting received in state %i whilst not waiting for incoming call!", myEndpoint.uiState));

  if (verbose > 0)
    cout << "Ringing phone for \"" << username << "\" ..." << endl;

  myEndpoint.uiState = MyH323EndPoint::uiWaitingForAnswer;

  return TRUE;
}


void MyH323Connection::OnUserInputString(const PString & value)
{
  cout << "User input received: \"" << value << '"' << endl;
#ifdef HAS_IXJ
  myEndpoint.SendDTMF(value);
#endif
}


PString MyH323Connection::GetCallerIdString() const
{
  H323TransportAddress addr = GetControlChannel().GetRemoteAddress();
  PIPSocket::Address ip;
  WORD port;
  addr.GetIpAndPort(ip, port);
  DWORD decimalIp = (ip[0] << 24) +
                    (ip[1] << 16) +
                    (ip[2] << 8) +
                     ip[3];
  PString remotePartyName = GetRemotePartyName();
  PINDEX bracket = remotePartyName.Find('[');
  if (bracket != P_MAX_INDEX)
    remotePartyName = remotePartyName.Left(bracket);
  bracket = remotePartyName.Find('(');
  if (bracket != P_MAX_INDEX)
    remotePartyName = remotePartyName.Left(bracket);
  return psprintf("%010li\t\t", decimalIp) + remotePartyName;
}

// End of File ///////////////////////////////////////////////////////////////
