#!/usr/bin/perl

# neat.cgi
$VERSION = "4.8";
# October 1, 2001
#
# Part of NEAT
# by jason blakey - jblakey@frogboy.net

##################### CONFIGURATION ########################
# Where does the neat.options file live?
$OPTIONS = "/usr/local/netsaint/neat4/neat4.options";
################### END CONFIGURATION ######################

# Load our configuration file...
do ($OPTIONS) or die ("Can't load $OPTIONS: $!");

# Use the CGI.pm lib...
use CGI":standard";

# Set our PATH.
$ENV{PATH} = "/usr/bin:/usr/sbin";

# Check if @USERS is defined, and if it is, check the HTTP username
# that is calling this page...
if (@REMOTE_USERS) {

	# Initialize the allowed_to_connect to zero...
	$allowed_to_connect = 0;

	# Get the current username from the environment variables...
	$current_username = $ENV{REMOTE_USER};

	# Step through each username in remote_uses, compare it to $current_username...
	foreach $username (@REMOTE_USERS) {
		if ($current_username eq $username) {
			# If we get a match, the user is allowed to connect...
			$allowed_to_connect = 1;
			last;
		}
	}

	# If allowed to connect is not set, then the user is not allowed to connect...
	if (! $allowed_to_connect) {
		draw_page ("message", "<H2>Permission Denied</H2> If you should have access, talk to your sys admin.");
		exit ();
	}
}


# Declare the %ENTS global variable...
local (%ENTS);

# Declare the %VER_TAGS global hash...
local (%VER_TAGS);

# Read in our entity definitions file...
$successful = read_entity_defs ($ENTITY_DEFS);

# Exit with warning if not successful...
if (! $successful) {
	draw_page ("message", "ERROR - could not load the entity definition file $ENTITY_DEFS");
	exit ();
}

# Read in our config files...
$successful = read_configs (@CONFIGS);

# Exit with warning if not successful...
if (! $successful) {
	draw_page ("message", "ERROR - could not load the config files @CONFIGS");
	exit ();
}

# Get what mode we are running in...
$mode = param ("mode");

# If no mode set, just draw the main page...
if (! $mode) {
	draw_page ("view");
}

# See if we are running in configuration verification mode...
elsif ($mode eq "verify") {

	# Double check that the netsaint binary and the configuration file
	# Defined in the $VERIFY_COMMAND actually exist...

	# Verify the syntax of the verify command...
	if ($VERIFY_COMMAND !~ /(\S+)\s+\S+\s+(\S+)/) {
		draw_page ("message", "ERROR - your verify command syntax is incorrect. Fix this in $OPTIONS");
		exit();
	}

	# Pull the binary and config_file out of the varify command...
	$VERIFY_COMMAND =~ /(\S+)\s+\S+\s+(\S+)/;
	$binary = $1;
	$config_file = $2;

	# Make sure the config file is readable...
	# Determine our uid (real name from /etc/passwd, not numeric)
	$uid = getpwuid ($>);

	if (! (-r $config_file)) {
		draw_page ("message", "ERROR - userid $uid cannot read your config_file $config_file.  Fix the permissions");
		exit ();
	}

	# And make sure the binary is executable...
	if (! (-X $binary)) {
		draw_page ("message", "ERROR - userid $uid cannot run $binary to verify your config. This is probably your httpd userid. Fix this");
		exit();
	}

	# Execute the verify command...
	$results = `$VERIFY_COMMAND`;

	# If we have no results, say so...	
	if (!$results) {
		draw_page ("message", "ERROR - no results - very strange...");
		exit ();
	}

	# Add some html formatting to the results...
	$results = "<PRE>" . $results . "<PRE>\n";

	# If we didn't see any serious problems...
	if ($results =~ /No serious problems/) {

		# Say so, and offer the restart option...
		$results .= <<EOF;
</TD>
</TR>
<TR BGCOLOR=$COLOR1>
<TD COLSPAN=5>
<CENTER>
<FORM ACTION="$CGIPATH/cmd.cgi" METHOD=POST>
<INPUT TYPE=HIDDEN NAME="cmd_typ" VALUE="13">
<INPUT TYPE=SUBMIT VALUE="Restart NetSaint / Activate Changes">
</CENTER>
</FORM>
EOF
		draw_page ("message", $results);
	}

	# Otherwise, there was a problem... say so..
	else {
		$results .= "<HR><FONT COLOR=RED><H2>DID NOT PASS VERIFICATION - FIX THIS</H2></FONT>";
		draw_page ("message", $results);
	}
}

# See if we are running in create mode...
elsif ($mode eq "create") {

	# Figure out what entity type we are creating...
	$entity_type = param ("entity_type");

	# Draw the proper page for entity creation...
	draw_page ("create", $entity_type);
}

# Or in view mode...
elsif ($mode eq "view") {
	$entity_type = param ("entity_type");
	$entity_instance = param ("entity_instance");

	draw_page ("view", $entity_type, $entity_instance);
}


# or in modify mode...
elsif ($mode eq "modify") {
	$entity_type = param ("entity_type");
	$entity_instance = param ("entity_instance");

	# Draw the page...
	draw_page ("modify", $entity_type, $entity_instance);
}

# or in delete mode...
elsif ($mode eq "delete") {

	# Check to see if the delete_confirm checkbox was checked...
	$delete_confirm = param ("delete_confirm");

	if (! $delete_confirm) {
		draw_page ("message", "To delete that entity, please check the confirm checkbox, and try again");
		exit ();
	}

	# Read the entity_instance and entity_type...
	$entity_instance = param ("entity_instance");
	$entity_type = param ("entity_type");

	# We need to parse and load in the deletion_rules...
	$successful = read_deletion_rules ($DELETION_RULES);

	if (! $successful) {
		draw_page ("message", "ERROR - could not load deletion rules $DELETION_RULES: $!");
		exit ();
	}

	# Delete entity is a recursive subroutine...
	$success = delete_entity ($entity_type, $entity_instance);

	if (! $success) {
		draw_page ("message", "Unable to delete $entity_type $entity_instance");
		exit ();
	}

	# Write the config files...
	foreach $config (@CONFIGS) {
		write_config ($config);
	}

	# Draw the page...
	draw_page ("message", "Deleted Successful");

	# If we need to, record this modification...
	if ($MOD_LOGFILE and $current_username) {
		$success = record_mod ($MOD_LOGFILE, $current_username, "Deletion");

		# Print a warning message if we can't write to the $MOD_LOGFILE...
		if (! $success) {
			print STDERR "Could not write to $MOD_LOGFILE\n";
		}
	}

	elsif ($MOD_LOGFILE) {
		$success = record_mod ($MOD_LOGFILE, "UNKNOWN", "Deletion");

		# Print a warning message if we can't write to the $MOD_LOGFILE...
		if (! $success) {
			print STDERR "Could not write to $MOD_LOGFILE\n";
		}
	}
}

# or if we're writing an entry...
elsif ($mode eq "write") {

	# We need to read in our input verification file...
	$successful = read_ver_tags ($VER_TAGS);

	if (! $successful) {
		draw_page ("message", "ERROR - could not load the input verification tags file $VER_TAGS");
		exit ();
	}

	# Get our entity_type and CONFIG name...
	$entity_type = param ("entity_type");
	$config = param ("config");

	if ($entity_instance = param ("entity_instance")) {
		# Then we are dealing with a modification...
		# Delete the old entry, then do the addition...
		# Use delete, not undef...
		delete ($ENTS{$entity_type}->{INSTANCES}->{$entity_instance});

		# Remember that this was a modification...
		$action = "Modification";
	}

	# Otherwise, this was a deletion...
	else {
		$action = "Creation";
	}

	# For both a create and a modify, do this section...

	# Initialize some vars...
	$num_uds = 0;

	# For each variable defined in the definition of this entity_type...
	$num_vars = $#{$ENTS{$entity_type}->{DEFINITION}};

	foreach $var_num (0 .. $num_vars) {

		# Get the variable_name for this var_num...
		$variable_name = $ENTS{$entity_type}->{DEFINITION}->[$var_num]->{NAME};

		# Check to see if the variable is a list or a single item....
		if ($ENTS{$entity_type}->{DEFINITION}->[$var_num]->{TYPE} eq "MULTIPLE") {
			# If it is, then we have a list to deal with

			# get the list from the CGI form...
			@variable = param ($variable_name);

			# and make a string outa it, delimited by ,'s...
			$variable = join (",", @variable);
		}

		# Otherwise, it's a plain old variable...
		else {
			$variable = param ($variable_name);
		}

		# Check to see if we need to supply a default "0" value for unchecked check-boxes...
		if ($ENTS{$entity_type}->{DEFINITION}->[$var_num]->{FORM_ELEMENT} eq "CHECKBOX") {
			# The most efficient variable initialization i've seen...
			# from Programming Perl (3rd Edition)
			$variable ||= "0";
		}

		# Figure out if this variable is a unique descriptor for this entity type...
		$ud = $ENTS{$entity_type}->{DEFINITION}->[$var_num]->{UD};

		if ($ud) {
			# If this is a unique descriptor, we need to check existing instances, and see if they match...
			# If they do, we need to raise an alarm...

			# Increment our number of unique_descriptors for this entity_type...
			++$num_uds;

			# Get a list of instances for this entity_type...
			@instances = keys (%{$ENTS{$entity_type}->{INSTANCES}});

			# For each instance...
			foreach $instance (@instances) {

				# Look up the contents of the field...
				$field_contents = $ENTS{$entity_type}->{INSTANCES}->{$instance}->{FIELDS}->[$var_num];

				# And compare them...
				if ($field_contents eq $variable) {
					# If they match, then we have to remember that we had a match...
					$ud_match_count{$instance}++;
				}
			}
		}

		# Run through the checks for input validation...
		# Get the input verification tags for this entity_type/field...
		$ver_tags = $ENTS{$entity_type}->{DEFINITION}->[$var_num]->{VER_TAGS};

		@ver_tags = split (",", $ver_tags, -1);

		foreach $tag (@ver_tags) {
			# Look up the regex for the current tag...
			$regex = $VER_TAGS{$tag};

			if ($variable !~ /$regex/) {
				# Figure out the nice name for the offending field...
				$nice_name = $ENTS{$entity_type}->{DEFINITION}->[$var_num]->{NICE_NAME};

				# If the variable is empty, it contains "nothing"
				$variable ||= "nothing";

				# Draw the message page...
				draw_page ("message", "<B>Field $nice_name</B> contains <B>$variable</B>,which does not pass the $tag check.");

				# And exit...
				exit ();
			}
		}

		# And now store it...
		# Hopefully, we will never have 1000000 instances in netsaint...
		# If so, time to move on to a new tool...
		$ENTS{$entity_type}->{INSTANCES}->{1000000}->{FIELDS}->[$var_num] = $variable;

	}

	# Now, we check to see if we had if we had as many ud matches as there were ud's defined... 
	@instances = keys (%{$ENTS{$entity_type}->{INSTANCES}});
	foreach $instance (@instances) {

		# If the number of unique descriptors for an entity type is the same as the count for a particular instance...
		if ($num_uds == $ud_match_count{$instance}) {
			# We have a duplicate - inform and exit...
			draw_page ("message", "Sorry, there is already an instance defined with those unique descriptors");
			exit ();
		}

	}

	# Remember which config file this guy is writing to...
	$ENTS{$entity_type}->{INSTANCES}->{1000000}->{CONFIG} = $config;

	# Now, write the host files...
	$successful = write_config ($config);

	# If we weren't successful, print an error message...
	if (! $successful) {
		draw_page ("message", "ERROR - did not successfully write config file $config");
	}

	# Otherwise, report success...
	else {
		draw_page ("message", "Config file $config sucessfully updated - $action successful");
	}

	# If we need to, record this modification...
	if ($MOD_LOGFILE and $current_username) {
		$success = record_mod ($MOD_LOGFILE, $current_username, $action);

		# Print a warning message if we can't write to the $MOD_LOGFILE...
		if (! $success) {
			print STDERR "Could not write to $MOD_LOGFILE\n";
		}

	}

	elsif ($MOD_LOGFILE) {
		$success = record_mod ($MOD_LOGFILE, "UNKNOWN", $action);

		# Print a warning message if we can't write to the $MOD_LOGFILE...
		if (! $success) {
			print STDERR "Could not write to $MOD_LOGFILE\n";
		}
	}
}

# Otherwise, print an error...
else {
	draw_page ("message", "ERROR - no such mode as $mode");
}


# And exit...
exit ();


#################################################################################
#################################################################################
# SUBROUTINES
#################################################################################
#################################################################################

# This subroutine will parse the entity_defs file, and store it in memory...
sub read_entity_defs {
	my ($defs) = $_[0];
	my ($line, $entity_type, @fields, $var_num, $good_stuff);

	# Open the $defs file for reading...
	if (! open(DEFS, "< $defs")) {
		# If we can't open the file, return a 0...
		return (0);
	}

	while (defined ($line = <DEFS>)) {
		# Skip comments...
		next if ($line =~ /^#/);

		# Skip the closing brace...
		next if ($line =~ /}/);

		# See if we are beginning a entity definition...
		if ($line =~ /(\S+)\s+{/) {
			$entity_type = $1;

			# Reset the variable number counter...
			$var_num = 0;
		}

		# See if we are examining a data-bearing line...
		elsif ($line =~ /\s*(\S+\:.*)/) {

			$good_stuff = $1;

			# And split it into its fields...
			@fields = split (/:/, $good_stuff, -1);

			# Check to make sure there are 11 fields (10 being the highest index of @fields)...
			if ($#fields != 10) {
				draw_page ("message", "ERROR: Not the right number of fields (9) in entity_defs file $DEFS on line $.<BR>$line");
				exit ();
			}

			# Store the information in the global hash $ENTS...
			$ENTS{$entity_type}->{DEFINITION}->[$var_num]->{NAME} = $fields[0];
			$ENTS{$entity_type}->{DEFINITION}->[$var_num]->{UD} = $fields[1];
			$ENTS{$entity_type}->{DEFINITION}->[$var_num]->{TYPE} = $fields[2];
			$ENTS{$entity_type}->{DEFINITION}->[$var_num]->{NICE_NAME} = $fields[3];
			$ENTS{$entity_type}->{DEFINITION}->[$var_num]->{VER_TAGS} = $fields[4];
			$ENTS{$entity_type}->{DEFINITION}->[$var_num]->{FORM_ELEMENT} = $fields[5];
			$ENTS{$entity_type}->{DEFINITION}->[$var_num]->{HTML_OPTIONS} = $fields[6];
			$ENTS{$entity_type}->{DEFINITION}->[$var_num]->{LINKED_TO} = $fields[7];
			$ENTS{$entity_type}->{DEFINITION}->[$var_num]->{EF_GLUE} = $fields[8];
			$ENTS{$entity_type}->{DEFINITION}->[$var_num]->{EXTRA_FIELDS} = $fields[9];
			$ENTS{$entity_type}->{DEFINITION}->[$var_num]->{DEFAULT_VALUE} = $fields[10];

			# Increment our variable number counter...
			++$var_num;
		}
	}

	# And return...
	return (1);
}

################################################################################################
# This subroutine reads in multiple config files, and stores them in the $ENTS data structure...
################################################################################################
sub read_configs {
	my (@configs) = @_;
	my ($instance_num) = 0;
	my ($field_num, $config, $line, $entity_type, $short_name, $rest, @fields);

	# First we check each config for read/writability...
	foreach $config (@configs) {
		if (! -r $config and ! -w $config) {
			# If we fail, return failure...
			return (0);
		}
	}
			
	foreach $config (@configs) {
		if (! open (CONFIG, "< $config")) {
			# If we can't open the config file in question, return failure...
			return (0);
		}	

		while (defined ($line = <CONFIG>)) {
			next if ($line =~ /^#/);

			# Chomp the newline off the end of the line...
			chomp ($line);

			if ($line =~ /(\S+)\[(\S+)\]=(.*)/) {
				$entity_type = $1;
				$short_name = $2;
				$rest = $3;

				# Split the rest into its fields...
				# Set the limit to -1, to specify arbitrarily large limit...
				# Otherwise, trailing empty fields get dropped...
				# See Programming Perl, 3rd Edition, Page 794...
				# Tricky!

				@fields = split (/;/, $rest, -1);

				# Push $short_name onto the left hand side of the @fields array...
				unshift (@fields, $short_name);

				# Initialize the field number counter...
				$field_num = 0;

				# Remember which HOSTFILE this instance came from...
				$ENTS{$entity_type}->{INSTANCES}->{$instance_num}->{CONFIG} = $config;

				# Remember this information...
				foreach $field (@fields) {
					$ENTS{$entity_type}->{INSTANCES}->{$instance_num}->{FIELDS}->[$field_num] = $field;
					++$field_num;
				}

				# Increment our entity instance counter...
				++$instance_num;
			}
		}

		# Close the current config file...
		close (CONFIG);
	}

	# Return success...
	return (1);
}

#########################################################################################
# This subroutine will read int the input verifcation regexes...
sub read_ver_tags {
	my ($tag_file) = $_[0];
	my ($name, $regex, $line);

	if (! open (REGEXES, "< $tag_file")) {
		return (0);
	}

	while (defined ($line = <REGEXES>)) {
		#skip comments...
		next if ($line =~ /^#/);

		chomp ($line);

		# if the line contains a definition...
		if ($line =~ /^(\S+)\s+(.*)/) {
			$name = $1;
			$regex = $2;

			# Store it...
			$VER_TAGS{$name} = $regex;
		}
	}

	# Close the regex definition file...
	close (REGEXES);

	# Return success...
	return (1);
}


##########################################################################################
# This subroutine reads in the deletion rules for deleting entities...
##########################################################################################
sub read_deletion_rules {
	my ($rules) = $_[0];
	my ($entity_type, $rule_num, $mode, $entity, $myfield, $yourfield);

	# If we can't open the rules file, return failure...
	if (! open (RULES, "< $rules")) {
		return (0);
	}

	while (defined ($line = <RULES>)) {
		# Skip comments...
		next if $line =~ /^#/;
		next if $line =~ /}/;

		if ($line =~ /^(\S+)\s+{/) {
			$entity_type = $1;
			$rule_num = 0;
		}

		# Otherwise, if the line is a valid definition line...
		elsif ($line =~ /\S+/) {

			# Chomp off the end of line character...
			chomp ($line);

			# Split it into it's parts, and then store 'em...
			($mode, $entity, $myfield, $yourfield) = split (/:/, $line, -1);
			$ENTS{$entity_type}->{DEL_RULES}->[$rule_num]->{DEL_MODE} = $mode;
			$ENTS{$entity_type}->{DEL_RULES}->[$rule_num]->{ENTITY_TYPE} = $entity;
			$ENTS{$entity_type}->{DEL_RULES}->[$rule_num]->{MYFIELD} = $myfield;
			$ENTS{$entity_type}->{DEL_RULES}->[$rule_num]->{YOURFIELD} = $yourfield;

			# Increment the rule counter...
			++$rule_num;
		}
	}

	# Return success...
	return (1);
}

#########################################################################################
# This subroutine will delete an entity, and all subentities, following the rules...
#########################################################################################

sub delete_entity {
	my ($entity_type) = $_[0];
	my ($entity_instance) = $_[1];
	my ($num_rules, $rule_num, $del_mode, $del_entity_type, $del_myfield, $del_yourfield);
	my ($field_contents, @instances, $instance, $field);
	my (@parts, @newparts, $part, $newparts, $type);

	# Look up the number of deletion rules for this entity type...
	$num_rules = $#{$ENTS{$entity_type}->{DEL_RULES}};

	# For each rule...
	foreach $rule_num (0 .. $num_rules) {

		# Look up the information on this rule...
		$del_mode = $ENTS{$entity_type}->{DEL_RULES}->[$rule_num]->{DEL_MODE};
		$del_entity_type = $ENTS{$entity_type}->{DEL_RULES}->[$rule_num]->{ENTITY_TYPE};
		$del_myfield = $ENTS{$entity_type}->{DEL_RULES}->[$rule_num]->{MYFIELD};
		$del_yourfield = $ENTS{$entity_type}->{DEL_RULES}->[$rule_num]->{YOURFIELD};

		# Figure out the entity_instance for this del_entity....
		$field_contents = $ENTS{$entity_type}->{INSTANCES}->{$entity_instance}->{FIELDS}->[$del_myfield];

		# Check if there is further recursion necessary to execute this rule...
		if ($del_mode eq "REMOVE") {

			# Now, we have to check if we are dealing with a MULTIPLE field or a SINGLE...

			$type = $ENTS{$entity_type}->{DEFINITION}->[$del_myfield]->{TYPE};

			if ($type eq "MULTIPLE") {
				# Then we are dealing with a list, need to split and step through...

				@parts = split (/,/, $field_contents, -1);
			}

			else {
				push (@parts, $field_contents);
			}

			foreach $part (@parts) {

				# Find the instances that match...
				@instances = get_instances ($del_entity_type, $del_yourfield, $part);

				# For each instance...	
				foreach $instance (@instances) {
					# Call delete_entity again, with this entity_type...
					# And remember this deletion in @deletions...
					delete_entity ($del_entity_type, $instance);
				}
			}
		}

		elsif ($del_mode eq "UNREF") {
			# We need to remove the reference in question...
			
			# Get a list of the instances we need to modify...
			@instances = get_instances ($del_entity_type, $del_yourfield, $field_contents);

			foreach $instance (@instances) {
				# Retrieve the field in question...

				$field = $ENTS{$del_entity_type}->{INSTANCES}->{$instance}->{FIELDS}->[$del_yourfield];

				# Modify it...
				@parts = split (/,/, $field, -1);

				foreach $part (@parts) {
					next if ($part eq $field_contents);

					# Otherwise, save the part in new parts...
					push (@newparts, $part);
				}

				# Rejoin everything...
				$newparts = join (",", @newparts);

				# And put it back in...
				$ENTS{$del_entity_type}->{INSTANCES}->{$instance}->{FIELDS}->[$del_yourfield] = $newparts;
			}
		}
	}

	# Finally, delete the entity in question....
	delete ($ENTS{$entity_type}->{INSTANCES}->{$entity_instance});

	return ("Done");
}

##########################################################################################
# This subroutine takes the name of a config as input, and writes out that config file
##########################################################################################
sub write_config {
	my ($config) = $_[0];
	my (@entity_types, $entity_type, @instance_nums, $instance_num);
	my ($inst_name, $instance, $num_vars, $i, $var, @instances);
	my($configname, $match);


	# First, we check to make sure that $config is a valid configuration file...
	$match = 0;
	foreach $configname (@CONFIGS) {
		if ($configname eq $config) {
			$match = 1;
			last;
		}
	}

	# If we didn't get a match...
	if (! $match) {
		# Return failure...
		return();
	}

	# Find out what kind of entity types we have...
	@entity_types = sort (keys (%ENTS));

	# Step through em...
	foreach $entity_type (@entity_types) {

		# Get an index of the indstances...
		@instance_nums = sort (keys (%{$ENTS{$entity_type}->{INSTANCES}}));

		# And step through those...
		foreach $instance_num (@instance_nums) {

			# Figure out what the stored_config is for this instance...
			$stored_config = $ENTS{$entity_type}->{INSTANCES}->{$instance_num}->{CONFIG};

			# If this instance lives in the config we're interested in...
			if ($stored_config eq $config) {

				# We got one...build the instance...
				$inst_name = $ENTS{$entity_type}->{INSTANCES}->{$instance_num}->{FIELDS}->[0];
				$instance = "$entity_type"."[$inst_name]=";
				$num_vars = $#{$ENTS{$entity_type}->{INSTANCES}->{$instance_num}->{FIELDS}};

				# Step through each variable...
				foreach $var_num (1 .. $num_vars) {

					# Skip fields which are part of the extra fields of another field...
					if ($ENTS{$entity_type}->{DEFINITION}->[$var_num]->{EXTRA_FIELDS}) {
						if ($ENTS{$entity_type}->{DEFINITION}->[$var_num]->{EXTRA_FIELDS} =~ /PART/) {
							next;
						}
					}

					# Get our variable value...
					$var = $ENTS{$entity_type}->{INSTANCES}->{$instance_num}->{FIELDS}->[$var_num];

					# Normally, we don't have to worry about multi-field variables...
					$have_to_do_this = 0;

					# Check to see if we need to worry about special extra fields...
					$extra_fields = $ENTS{$entity_type}->{DEFINITION}->[$var_num]->{EXTRA_FIELDS};

					if ($extra_fields) {
						@extra_fields = split (/,/, $extra_fields);

						# Check if any of these values are defined ... if any are, we have to do this...
						foreach $extra_field (@extra_fields) {
							if ($ENTS{$entity_type}->{INSTANCES}->{$instance_num}->{FIELDS}->[$extra_field]) {
								$have_to_do_this = 1;
								last;
							}
						}
					}

					# If we have to do this, build the instance...
					if ($have_to_do_this ) {

						# Determine what the glue character is...
						$glue = $ENTS{$entity_type}->{DEFINITION}->[$var_num]->{EF_GLUE};

						# Initialize $string...
						undef ($string);
						# And start off with the string being $var...
						$string = $var;

						# Step through each field, and add the contents to $string (delim'ed by the glue)...
						foreach $fieldnum (@extra_fields) {
							$string .= $glue;

							$value = $ENTS{$entity_type}->{INSTANCES}->{$instance_num}->{FIELDS}->[$fieldnum];
							$string .= $value;
						}

						# And then complete the entry...
						$instance .= "$string;";
					}
					
					# Otherwise, just add the value to $instance...
					else {
						$instance .= "$var;";
					}
				}
				# Chop the extra ; off the instance...
				chop ($instance);

				# push the new entry into @instances...
				push (@instances, $instance);
			}
		}
	}

	# and now, write the config...
	if (! open (CONFIG, "> $config")) {
		draw_page ("message", "Can't open config file $config for writing: $!");
		exit ();
	}

	$current_entity_type = "";

	# Write each instance to the config file...
	foreach $instance (@instances) {
		$instance =~ /(\S+)\[/;
		$entity_type = $1;

		if ($entity_type ne $current_entity_type) {
			print CONFIG "\n# \U$entity_type ENTRIES\n\n";
			$current_entity_type = $entity_type;
		}
		print CONFIG "$instance\n";
	}
	close (CONFIG);

	# And return...
	return (1);
}


##########################################################################################
# This subroutine takes input of type, and draws the appropriate page (checking for other
# appropriate variables).
##########################################################################################
sub draw_page {

	# Determine the type of page to draw (create, modify, view)...
	my ($type) = $_[0];

	# Determine the $entity_type and $entity_instance ... these may not be set...
	my ($entity_type) = $_[1];
	my ($entity_instance) = $_[2];

	print <<EOF;
Content-type: text/html;

<HTML>
<HEAD>
<TITLE>
EOF

	############################## MESSAGE #######################################

	if ($type eq "message") {
		my ($message) = $_[1];

		print <<EOF;
NEAT Message
</TITLE>

<BODY BGCOLOR=White>
<CENTER>
<TABLE>
<TR BGCOLOR=$COLOR1><TD WIDTH=400 COLSPAN=5><CENTER><H2><FONT COLOR=$TEXTCOLOR1>NetSaint Easy Administration Tool Version $VERSION</FONT></H2></CENTER></TD></TR>
<TR BGCOLOR=$COLOR1><TD COLSPAN=5><FONT COLOR=$TEXTCOLOR1><H3><B>Informational Message</B></H3></FONT></TD></TR>
<TR BGCOLOR=$COLOR2><TD COLSPAN=5><FONT COLOR=$TEXTCOLOR2>$message</FONT></TD></TR>
EOF
	}

	############################### CREATE #########################################

	# Otherwise, see if we are creating an entry...
	elsif ($type eq "create") {

	# Check to make sure that an entity type was chosen...
	if (! $entity_type) {
			draw_page ("message", "ERROR - please choose an entity type");
			exit ();
	}

		my ($uppercase) = ucfirst ($entity_type);

		print <<EOF;
$uppercase Creation
</TITLE>

<BODY BGCOLOR=White>

<CENTER>

<TABLE CELLPADDING=2>
<TR><TD COLSPAN=5 BGCOLOR=$COLOR1><CENTER><H2><FONT COLOR=$TEXTCOLOR1>$uppercase Creation</FONT></H2></CENTER></TD></TR>
<FORM ACTION="$CGIPATH/neat.cgi" METHOD=POST>
<INPUT TYPE=HIDDEN NAME="mode" VALUE="write">
<INPUT TYPE=HIDDEN NAME="entity_type" VALUE="$entity_type">
<TR BGCOLOR=$COLOR2><TD><FONT COLOR=$TEXTCOLOR2>Select Config File to Write To:</FONT></TD>
<TD COLSPAN=4><SELECT NAME="config">
EOF
		# Print out the list of config files, so the user can choose which to 
		# write this entry to...
		foreach $config (@CONFIGS) {
			print "<OPTION VALUE=\"$config\">$config</OPTION>\n";
		}
		print "</SELECT></TD></TR>\n";

		# Figure out how many variables are in the DEFINITION for this entity_type...
		$num_variables = $#{$ENTS{$entity_type}->{DEFINITION}};

		foreach $index (0 .. $num_variables) {	
			$variable = $ENTS{$entity_type}->{DEFINITION}->[$index]->{NAME};
			$nice_name = $ENTS{$entity_type}->{DEFINITION}->[$index]->{NICE_NAME};
			$form_element = $ENTS{$entity_type}->{DEFINITION}->[$index]->{FORM_ELEMENT};
			$html_options = $ENTS{$entity_type}->{DEFINITION}->[$index]->{HTML_OPTIONS};
			$default_value = $ENTS{$entity_type}->{DEFINITION}->[$index]->{DEFAULT_VALUE};
			$linked_to = $ENTS{$entity_type}->{DEFINITION}->[$index]->{LINKED_TO};
			$type = $ENTS{$entity_type}->{DEFINITION}->[$index]->{TYPE};

			# If the form element is the SELECT element...
			if ($form_element eq "SELECT") {

				print <<EOF;

<TR BGCOLOR=$COLOR2><TD><FONT COLOR=$TEXTCOLOR2>$nice_name</FONT></TD>
<TD COLSPAN=4><SELECT NAME="$variable" $html_options>
<OPTION VALUE="">None</OPTION>
EOF
				# See if link_to contains a link to an entity, or a link to an array...

				# If its linked to an array...
				if ($linked_to =~ /^\@(\S+)/) {
					# Figure out what our array name is...
					$array_name = $1;

					# Step through each element in this array...
					foreach $array_entry (@{$array_name}) {

						# And if the array element is the same as the default_value...
						if ($array_entry eq $default_value) {
							# Make it selected...
							print "<OPTION VALUE=\"$array_entry\" SELECTED>$array_entry</OPTION>\n";
						}
						else {
							# Otherwise, don't make it selected...
							print "<OPTION VALUE=\"$array_entry\">$array_entry</OPTION>\n";
						}
					}
				}

				else {
					# link_to links to an entity...

					# Figure out the entity type and the field number...
					($entity, $fnum) = split (/,/, $linked_to);

					# Figure out how many instances we have for this SELECT...
					# and get the instances sorted on the name field...
					@instances = sort_by_field ($entity, 0);
	
					# For each instance of this entity_type...
					foreach $instance_num (@instances) {
	
						# Figure out what the short name is for this instance...
						$link_value = $ENTS{$entity}->{INSTANCES}->{$instance_num}->{FIELDS}->[$fnum];
	
						# If this short name is equal to the default_value...
						if ($link_value eq $default_value) {
							# make it selected...
							print "<OPTION VALUE=\"$link_value\" SELECTED>$link_value</OPTION>\n";
						}
						else {
							# Otherwise, don't make it selected...
							print "<OPTION VALUE=\"$link_value\">$link_value</OPTION>\n";
						}
					}
				}
				print "</SELECT></TD></TR>\n";
			}

			elsif ($form_element eq "TEXT") {
				print "<TR BGCOLOR=$COLOR2><TD><FONT COLOR=$TEXTCOLOR2>$nice_name</FONT></TD><TD COLSPAN=4><INPUT TYPE=TEXT NAME=\"$variable\" $html_options VALUE=\"$default_value\"></TD></TR>\n";
			}
			elsif ($form_element eq "CHECKBOX") {
				print "<TR BGCOLOR=$COLOR2><TD><FONT COLOR=$TEXTCOLOR2>$nice_name</FONT></TD><TD COLSPAN=4><INPUT TYPE=CHECKBOX NAME=\"$variable\" $html_options VALUE=1 $default_value></TD></TR>\n";
			}
			else {
				print "ERROR - Encountered form element which is not supported...\n";
				print "<BR>form element is $form_element\n";
				exit ();
			}
		}

		print <<EOF;
<TR BGCOLOR=$COLOR1>
<TD COLSPAN=5><CENTER>
<TABLE>
<TR>
<TD><CENTER><INPUT TYPE=SUBMIT VALUE="Create $uppercase"></CENTER></FORM></TD>
<TD><CENTER><FORM><INPUT TYPE=BUTTON VALUE="Cancel $uppercase Creation" onClick="location.href='$CGIPATH/neat.cgi'"></FORM></CENTER></TD></TR>
</TABLE>
</TD>
</TR>
EOF
	}

	################################### MODIFY ###############################################

	# Elsif the type is modify...
	elsif ($type eq "modify") {
		my ($field_num);
		my ($uppercase) = ucfirst($entity_type);

		# Declare variables here...

		print <<EOF;
$uppercase Modification Instance $entity_instance
</TITLE>

<BODY BGCOLOR=White>

<CENTER>

<TABLE CELLPADDING=2>
<TR BGCOLOR=$COLOR1>
<TD COLSPAN=5 BGCOLOR=$COLOR1><CENTER><H2><FONT COLOR=$TEXTCOLOR1>$uppercase Modification</FONT></H2></CENTER></TD>
</TR>
<FORM ACTION="$CGIPATH/neat.cgi" METHOD=POST>

<INPUT TYPE=HIDDEN NAME="entity_type" VALUE="$entity_type">
<INPUT TYPE=HIDDEN NAME="entity_instance" VALUE="$entity_instance">
<INPUT TYPE=HIDDEN NAME="mode" VALUE="write">

<TR BGCOLOR=$COLOR2><TD><FONT COLOR=$TEXTCOLOR2>Select Config File to Write To:</FONT></TD>
<TD COLSPAN=4><SELECT NAME="config">
EOF
		# Print out the list of config files, so the user can choose which to 
		# write this entry to...
		foreach $config (@CONFIGS) {
			# If the current config file is the same as the instances home config file, make
			# it selected...
			if ($config eq $ENTS{$entity_type}->{INSTANCES}->{$entity_instance}->{CONFIG}) {
				print "<OPTION VALUE=\"$config\" SELECTED>$config</OPTION>\n";
			}

			else {
				print "<OPTION VALUE=\"$config\">$config</OPTION>\n";
			}
		}

		print "</SELECT></TD></TR>\n";

		# Figure out how many variables are in the DEFINITION for this entity_type...
		$num_variables = $#{$ENTS{$entity_type}->{DEFINITION}};

		foreach $index (0 .. $num_variables) {
			$variable = $ENTS{$entity_type}->{DEFINITION}->[$index]->{NAME};
			$nice_name = $ENTS{$entity_type}->{DEFINITION}->[$index]->{NICE_NAME};
			$form_element = $ENTS{$entity_type}->{DEFINITION}->[$index]->{FORM_ELEMENT};
			$html_options = $ENTS{$entity_type}->{DEFINITION}->[$index]->{HTML_OPTIONS};
			$linked_to = $ENTS{$entity_type}->{DEFINITION}->[$index]->{LINKED_TO};
			$type = $ENTS{$entity_type}->{DEFINITION}->[$index]->{TYPE};
			$extra_fields = $ENTS{$entity_type}->{DEFINITION}->[$index]->{EXTRA_FIELDS};

			# Now, look up the current contents of that field...
			$current_value = $ENTS{$entity_type}->{INSTANCES}->{$entity_instance}->{FIELDS}->[$index];

			# If $extra_fields contains PART (denoting that this is a part of a larger variable...
			if ($extra_fields eq "PART") {
				# Then this is a part - print the part for this variable...
				$current_value = $extra_parts{$index};
			}

			# Otherwise, if it has extra_fields and is NOT a part...
			elsif ($extra_fields) {
				# Then we have to worry about extra_fields and extra_fields_glue...

				# Undef the extra_part hash...
				undef (%extra_parts);

				# Find out what the glue characters are...
				$glue = $ENTS{$entity_type}->{DEFINITION}->[$index]->{EF_GLUE};

				# Split the extra fields into the separate fields...
				@extra_fields = split (/,/, $extra_fields);

				# Split the current_value into the $current_value, and the rest...
				($current_value, @rest) = split (/$glue/, $current_value, -1);

				# Gonna have to remember the rest...
				# This is kinda kludgy...
				# Actually, this whole section is kinda kludgy...
				# Wish ethan had been a little more stringent with his entity definition style...

				foreach $field_num (0 .. $#extra_fields) {
					# If we've hit the end, throw what's left into the last extra_parts...
					# NOTE: this only makes a difference when our number of $extra_fields is less than the
					# number of entries in $rest...

					if ($field_num == $#extra_fields) {
						# Join 'em back together...
						$new_rest = join ($glue, @rest);
						$extra_parts{$extra_fields[$field_num]} = $new_rest; 
					}

					else {
						# Take the next item off @rest, and store it in $extra_parts...
						$extra_parts{$extra_fields[$field_num]} = shift (@rest);
					}
				}

			}

			# See if we've got a list or a single element to deal with...
			if ($type eq "MULTIPLE") {
				# If so, split the $current_value into its parts...
				@parts = split (/,/, $current_value);
			
				undef (%parts);
				
				# And store this information in %parts...
				foreach $part (@parts) {
					$parts{$part} = 1;
				}
			}
			else {
				# Make $current_value the only entry in %parts...
				undef (%parts);
				$parts{$current_value} = 1;
			}

			# If the form element is the SELECT element...
			if ($form_element eq "SELECT") {

				print <<EOF;
<TR BGCOLOR=$COLOR2>
<TD><FONT COLOR=$TEXTCOLOR2>$nice_name</FONT></TD>
<TD COLSPAN=4>
<SELECT NAME="$variable" $html_options>
<OPTION VALUE="">None</OPTION>
EOF

				# See if link_to contains a link to an entity,field , or a link to an array...

				# If its linked to an array...
				if ($linked_to =~ /^\@(\S+)/) {
					# Figure out what our array name is...
					$array_name = $1;

					# Step through each element in this array...
					foreach $array_entry (@{$array_name}) {

						# And if the array element is the same as the default_value...
						if ($parts{$array_entry}) {
							# Make it selected...
							print "<OPTION VALUE=\"$array_entry\" SELECTED>$array_entry</OPTION>\n";
						}
						else {
							# Otherwise, don't make it selected...
							print "<OPTION VALUE=\"$array_entry\">$array_entry</OPTION>\n";
						}
					}
				}

				else {
					# link_to links to an entity,fieldnum

					($entity, $fnum) = split (/,/, $linked_to);

					# Figure out how many instances we have for this SELECT...
					@instances = sort_by_field ($entity, 0);
	
					# For each instance of this entity_type...
					foreach $instance_num (@instances) {

						# Figure out what the short name is for this instance...
						$link_value = $ENTS{$entity}->{INSTANCES}->{$instance_num}->{FIELDS}->[$fnum];

						# If this short name is equal to the current_value...
						if ($parts{$link_value}) {
							# make it selected...
							print "<OPTION VALUE=\"$link_value\" SELECTED>$link_value</OPTION>\n";
						}
						else {
							# Otherwise, don't make it selected...
							print "<OPTION VALUE=\"$link_value\">$link_value</OPTION>\n";
						}
					}
				}
				print "</SELECT></TD></TR>\n";
			}

			# Otherwise if the form_element is TEXT...
			elsif ($form_element eq "TEXT") {
				print "<TR BGCOLOR=$COLOR2><TD><FONT COLOR=$TEXTCOLOR2>$nice_name</FONT></TD><TD COLSPAN=4><INPUT TYPE=TEXT NAME=\"$variable\" $html_options VALUE=\"$current_value\"></TD></TR>\n";
			}

			# Elsif the form_element is a CHECKBOX...
			elsif ($form_element eq "CHECKBOX") {
				if ($current_value == 1) {
					print "<TR BGCOLOR=$COLOR2><TD><FONT COLOR=$TEXTCOLOR2>$nice_name</FONT></TD><TD COLSPAN=4><INPUT TYPE=CHECKBOX NAME=\"$variable\" $html_options VALUE=1 CHECKED></TD></TR>\n";
				}
				else {

					print "<TR BGCOLOR=$COLOR2><TD><FONT COLOR=$TEXTCOLOR2>$nice_name</FONT></TD><TD COLSPAN=4><INPUT TYPE=CHECKBOX NAME=\"$variable\" $html_options VALUE=1></TD></TR>\n";
				}
			}
			else {
				print "ERROR - Encountered form element which is not supported...\n";
				print "<BR>form element is $form_element\n";
				exit ();
			}
		}

		print <<EOF;
<TR BGCOLOR=$COLOR1>
<TD COLSPAN=5><CENTER>
<TABLE>
<TR>
<TD><CENTER><INPUT TYPE=SUBMIT VALUE="Modify $uppercase"></FORM></CENTER></TD>
<TD><CENTER><FORM><INPUT TYPE=BUTTON VALUE="Cancel $uppercase Modification" onClick="location.href='$CGIPATH/neat.cgi'"></FORM></CENTER></TD></TR>
</TR>
</TABLE>
</CENTER>
</TD>
</TR>
EOF
	}

	#################################### VIEW ###########################################
	# Otherwise, if we are printing a view page...
	elsif ($type eq "view") {

		# Figure the date and time (pretty versions)...
		($date, $time) = datetime (1);

		# Print the header...
		print <<EOF;
NEAT Version $VERSION
</TITLE>
</HEAD>

<BODY BGCOLOR=White>

<CENTER>
<TABLE CELLPADDING=2>
<TR><TD COLSPAN=5 BGCOLOR=$COLOR1><CENTER><H1><FONT COLOR=$TEXTCOLOR1><A HREF="http://www.netsaint.org"><FONT COLOR=$TEXTCOLOR1>NetSaint</FONT></A> Easy Administration Tool Version $VERSION</FONT><CENTER></H1></TD></TR>
<TR><TD COLSPAN=5 BGCOLOR=$COLOR1><CENTER><FONT COLOR=$TEXTCOLOR1>by <A HREF="MAILTO:jblakey\@frogboy.net">jason blakey</A> -  jblakey\@frogboy.net</FONT></CENTER></TD></TR>
<TR>
<TD COLSPAN=5 BGCOLOR=$COLOR1>
<TABLE>
<TR>
<TD VALIGN=TOP>
<FORM ACTION="$CGIPATH/neat.cgi" METHOD=POST>
<FONT COLOR=$TEXTCOLOR1>Create a new</FONT>  
<INPUT TYPE=HIDDEN NAME="mode" VALUE="create">
<SELECT NAME="entity_type" onChange="submit()">
<OPTION VALUE="">Choose Type</OPTION>
EOF

		# Get a list of entity types...
		@entity_types = sort (keys (%ENTS));

		# Print each one as an option...
		foreach $entity_type (@entity_types) {
			print "<OPTION VALUE=\"$entity_type\">$entity_type</OPTION>\n";
		}

		print <<EOF;

</SELECT><FONT COLOR=$TEXTCOLOR1>instance. </FONT></FORM></TD>
<TD WIDTH=100>
&nbsp
</TD>

<TD>
EOF

		# Check to see if the verify command is defined...
		if ($VERIFY_COMMAND) {
			# If so, offer the verification choice...
			# Print the verifcation button
			print <<EOF;
<FORM ACTION="$CGIPATH/neat.cgi" METHOD=POST>
<INPUT TYPE=HIDDEN NAME="mode" VALUE="verify">
<INPUT TYPE=SUBMIT VALUE="Verify Config / Restart NetSaint">
</FORM>
EOF
		}
		else {
			# Print the restart button
	print <<EOF;
<FORM ACTION="$CGIPATH/cmd.cgi" METHOD=POST>
<INPUT TYPE=HIDDEN NAME="cmd_typ" VALUE="13">
<INPUT TYPE=SUBMIT VALUE="Restart NetSaint / Activate Changes">
</FORM>
EOF
		}

		print <<EOF;
</TD>

</TR>
</TABLE>
</TD>
</TR>
EOF

		#If there is a current username defined (HTTPD authentication used), print it...
		if ($current_username) {
			print <<EOF;

<TR>
<TD COLSPAN=5 BGCOLOR=$COLOR1><CENTER><FONT COLOR=$TEXTCOLOR1>Logged in as $current_username : Page Generated at $time on $date</FONT></CENTER></TD>
</TR>
EOF
		}

		# Otherwise, don't... just print the date and time...
		else {
			print <<EOF;

<TR>
<TD COLSPAN=5 BGCOLOR=$COLOR1><CENTER><FONT COLOR=$TEXTCOLOR1>Page Generated at $time on $date</FONT></CENTER></TD>
</TR>
EOF
		}

		#################################### INTRO PAGE ####################################
		# IF we don't have an entity type or entity instance... then just print the intro page...
		if (! $entity_type and ! $entity_instance) {

			# Initialize variables for this section of code...
			my (@types, $entity_type, @instances, $instance, $instance_name, $instance_desc);
			my (@timeperiod_instances, $timeperiod_instance, $timeperiod_name, $timeperiod_desc);
			my ($instance_host, $instance_service, $instance_first, $instance_last, $instance_hostgroup);
			my (@host_instances, $host_name, @hostgroup_instances, $hostgroup_instance, $hostlist);
			my (@hosts, $host, $loner, @loners, @contact_instances, $contact_name);
			my (@contactgroup_instances, $contactgroup_instance, @contacts, $contact);

			# Abandon all hope ye who go beyond here!

			# Now, first, we  deal with hostgroups and contactgroups...
			@types = ("hostgroup", "contactgroup");

			foreach $entity_type (@types) {
	
				# Print a header...
				print "<TR><TD COLSPAN=5 BGCOLOR=$COLOR1><CENTER><H3><B><FONT COLOR=$TEXTCOLOR1>\u$entity_type Instances</FONT></B></H3></CENTER></TD></TR>\n";
	
				# Get a list of instances for that entity_type...
				@instances = sort_by_field ($entity_type, 0);
	
	
				# look at each, and print a line...
				foreach $instance (@instances) {
					$instance_name = $ENTS{$entity_type}->{INSTANCES}->{$instance}->{FIELDS}->[0];
					$instance_desc = $ENTS{$entity_type}->{INSTANCES}->{$instance}->{FIELDS}->[1];

					print <<EOF;
<TR BGCOLOR=$COLOR2><TD><FONT COLOR=$TEXTCOLOR2><B>$instance_name</B></FONT></TD>
<TD>
<FONT COLOR=$TEXTCOLOR2>$instance_desc</FONT>
</TD>
<TD><A HREF="$CGIPATH/neat.cgi?mode=modify&entity_type=$entity_type&entity_instance=$instance"><FONT COLOR=$TEXTCOLOR2>View/Modify</FONT></A></TD>
<TD><A HREF="$CGIPATH/neat.cgi?mode=view&entity_type=$entity_type&entity_instance=$instance"><FONT COLOR=$TEXTCOLOR2>List Members</FONT></A></TD>
<TD>
<FORM ACTION="$CGIPATH/neat.cgi">
<INPUT TYPE=HIDDEN NAME="mode" VALUE="delete">
<INPUT TYPE=HIDDEN NAME="entity_type" VALUE="$entity_type">
<INPUT TYPE=HIDDEN NAME="entity_instance" VALUE="$instance">
<INPUT TYPE=CHECKBOX NAME="delete_confirm" VALUE="yes">
<FONT SIZE=-2><INPUT TYPE=SUBMIT VALUE="Delete"></FONT>
</FORM>
</TD>
</TR>
EOF
				}
			}

			# Next, we look at the timeperiods...
			print "<TR BGCOLOR=$COLOR1><TD COLSPAN=5><CENTER><H3><B><FONT COLOR=$TEXTCOLOR1>Timeperiod Instances</FONT></B></H3></TD></TR>\n";

			# Get our array of time period instance numbers...
			@timeperiod_instances = sort_by_field ("timeperiod", 0);

			# And print a line for each one...
			foreach $timeperiod_instance (@timeperiod_instances) {

				$timeperiod_name = $ENTS{timeperiod}->{INSTANCES}->{$timeperiod_instance}->{FIELDS}->[0];
				$timeperiod_desc = $ENTS{timeperiod}->{INSTANCES}->{$timeperiod_instance}->{FIELDS}->[1];
				print <<EOF;

<TR BGCOLOR=$COLOR2><TD><FONT COLOR=$TEXTCOLOR2><B>$timeperiod_name</B></FONT></TD>
<TD><FONT COLOR=$TEXTCOLOR2>$timeperiod_desc</FONT></TD>
<TD>
<A HREF="$CGIPATH/neat.cgi?mode=modify&entity_type=timeperiod&entity_instance=$timeperiod_instance"><FONT COLOR=$TEXTCOLOR2>View/Modify</FONT>
</TD>
<TD>&nbsp</TD>
<TD>
<FORM ACTION="$CGIPATH/neat.cgi" METHOD=POST>
<INPUT TYPE=HIDDEN NAME="mode" VALUE="delete">
<INPUT TYPE=HIDDEN NAME="entity_type" VALUE="timeperiod">
<INPUT TYPE=HIDDEN NAME="entity_instance" VALUE="$timeperiod_instance">
<INPUT TYPE=CHECKBOX NAME="delete_confirm" VALUE="yes">
<FONT SIZE="-2"><INPUT TYPE=SUBMIT VALUE="Delete"></FONT>
</FORM>
</TD>
</TR>
EOF
			}
	
			# Next, we need to worry about commands... - would like to have a seperate listing screen...
			if ($ENTS{command}->{INSTANCES}) {
				print <<EOF;
<TR BGCOLOR=$COLOR1><TD COLSPAN=5><CENTER>
<A HREF="$CGIPATH/neat.cgi?mode=view&entity_type=command"><H3><B><FONT COLOR=$TEXTCOLOR1>View/Modify Command Instances</FONT></B></H3></A>
</CENTER></TD></TR>
EOF
			}
			
			# and finally hostgroup escalations....
			if ($ENTS{hostgroupescalation}->{INSTANCES}) {
				print "<TR><TD COLSPAN=5 BGCOLOR=$COLOR1><CENTER><H3><B><FONT COLOR=$TEXTCOLOR1>HostGroup Escalation Instances</FONT></B></H3></CENTER></TD></TR>\n";
				@instances = sort_by_field ("hostgroupescalation", 0);
	
				foreach $instance (@instances) {
					$instance_hostgroup = $ENTS{hostgroupescalation}->{INSTANCES}->{$instance}->{FIELDS}->[0];
					$instance_range = $ENTS{hostgroupescalation}->{INSTANCES}->{$instance}->{FIELDS}->[1];

					print <<EOF;
<TR BGCOLOR=$COLOR2>
<TD><FONT COLOR=$TEXTCOLOR2><B>$instance_hostgroup</B></FONT></TD>
<TD><FONT COLOR=$TEXTCOLOR2>Notify Between Checks $instance_range</FONT></TD>
<TD>
<A HREF="$CGIPATH/neat.cgi?mode=modify&entity_type=hostgroupescalation&entity_instance=$instance"><FONT COLOR=$TEXTCOLOR2>View/Modify</FONT></A>
</TD>
<TD>&nbsp</TD>
<TD>
<FORM ACTION="$CGIPATH/neat.cgi" METHOD=POST>
<INPUT TYPE=HIDDEN NAME="mode" VALUE="delete">
<INPUT TYPE=HIDDEN NAME="entity_type" VALUE="hostgroupescalation">
<INPUT TYPE=HIDDEN NAME="entity_instance" VALUE="$instance">
<INPUT TYPE=CHECKBOX NAME="delete_confirm" VALUE="yes">
<FONT SIZE="-2"><INPUT TYPE=SUBMIT VALUE="Delete"></FONT>
</FORM>
</TD>
</TR>
EOF

				}
			}
	
			# Next, we worry about hosts not part of hostgroups and contacts not part of a contact group...

	
			# First, we get a list of hosts...
			@host_instances = sort_by_field ("host", 0);
	
			# look at each host...
			CHECK: foreach $host_instance (@host_instances) {
				# figure this hosts name...
				$host_name = $ENTS{host}->{INSTANCES}->{$host_instance}->{FIELDS}->[0];
	
				# Now, look through hostgroups until we find one that contains this host...
				# If we don't find a hostgroup which contains this host, print 'im...
	
				@hostgroup_instances = keys (%{$ENTS{hostgroup}->{INSTANCES}});
	
				foreach $hostgroup_instance (@hostgroup_instances) {
					$hostlist = $ENTS{hostgroup}->{INSTANCES}->{$hostgroup_instance}->{FIELDS}->[3];
	
					@hosts = split (/,/, $hostlist);
	
					foreach $host (@hosts) {
	
						if ($host eq $host_name) {
							next CHECK;
						}
					}
				}
				# If we get to this point in the code, the current $host is NOT part of a hostgroup...
				# So we print it out...
				$loner = <<EOF;
<TR BGCOLOR=$COLOR2>
<TD COLSPAN=2><FONT COLOR=$TEXTCOLOR2>Host not part of hostgroup: <B>$host_name</B></FONT></TD>
<TD>
<A HREF="$CGIPATH/neat.cgi?mode=modify&entity_type=host&entity_instance=$host_instance"><FONT COLOR=$TEXTCOLOR2>View/Modify</FONT></A>
</TD>
<TD>&nbsp</TD>
<TD>
<FORM ACTION="$CGIPATH/neat.cgi" METHOD=POST>
<INPUT TYPE=HIDDEN NAME="mode" VALUE="delete">
<INPUT TYPE=HIDDEN NAME="entity_type" VALUE="host">
<INPUT TYPE=HIDDEN NAME="entity_instance" VALUE="$host_instance">
<INPUT TYPE=CHECKBOX NAME="delete_confirm" VALUE="yes">
<FONT SIZE="-2"><INPUT TYPE=SUBMIT VALUE="Delete"></FONT>
</FORM>
</TD>  
</TR>
EOF
				push (@loners, $loner);
			}

			# Now, we worry about contacts not part of contact groups...
	
			# First, we get a list of contacts...
			@contact_instances = sort_by_field ("contact", 0);
	
			# look at each host...
			CHECK: foreach $contact_instance (@contact_instances) {
				# figure this contacts name...
				$contact_name = $ENTS{contact}->{INSTANCES}->{$contact_instance}->{FIELDS}->[0];
	
				# Now, look through contactgroups until we find one that contains this contact...
				# If we don't find a contactgroup which contains this contact, print 'im...
	
				@contactgroup_instances = keys (%{$ENTS{contactgroup}->{INSTANCES}});
	
				foreach $contactgroup_instance (@contactgroup_instances) {
					$contactlist = $ENTS{contactgroup}->{INSTANCES}->{$contactgroup_instance}->{FIELDS}->[2];
	
					@contacts = split (/,/, $contactlist);
	
					foreach $contact (@contacts) {
	
						if ($contact eq $contact_name) {
							next CHECK;
						}
					}
				}
				# If we get to this point in the code, the current $host is NOT part of a hostgroup...
				# So we print it out...

				$loner = <<EOF;
<TR BGCOLOR=$COLOR2>
<TD COLSPAN=2><FONT COLOR=$TEXTCOLOR2>Contact not part of contactgroup: <B>$contact_name</B></FONT></TD>
<TD>
<A HREF="$CGIPATH/neat.cgi?mode=modify&entity_type=contact&entity_instance=$contact_instance"><FONT COLOR=$TEXTCOLOR2>View/Modify</FONT></A>
</TD>
<TD>&nbsp</TD>
<TD>
<FORM ACTION="$CGIPATH/neat.cgi" METHOD=POST>
<INPUT TYPE=HIDDEN NAME="mode" VALUE="delete">
<INPUT TYPE=HIDDEN NAME="entity_type" VALUE="contact">
<INPUT TYPE=HIDDEN NAME="entity_instance" VALUE="$contact_instance">
<INPUT TYPE=CHECKBOX NAME="delete_confirm" VALUE="yes">
<FONT SIZE="-2"><INPUT TYPE=SUBMIT VALUE="Delete"></FONT>
</FORM>
</TD>  
</TR>
EOF
				push (@loners, $loner);
			}


			# If any loners exist, print 'em out...
			if (@loners) {

				print "<TR BGCOLOR=$COLOR1><TD COLSPAN=5><CENTER><H3><B><FONT COLOR=$TEXTCOLOR1>Hosts / Contacts NOT Part of a Group</FONT></B></H3></CENTER></TD></TR>\n";
				foreach $loner (@loners) {
					print $loner;
				}
			}


		}

		################################################# HOSTGROUP ##############################
		elsif ($entity_type eq "hostgroup") {

			my ($hostgroupname, $hosts, @hosts, $host, $instance_num, $host_name, $host_desc);

			# Lookup the name of this hostgroup...
			$hostgroupname = $ENTS{hostgroup}->{INSTANCES}->{$entity_instance}->{FIELDS}->[0];

			# Print the page banner...
			print <<EOF;
<TR BGCOLOR=$COLOR1><TD COLSPAN=5><B><FONT COLOR=$TEXTCOLOR1><H3>Viewing Members of Hostgroup $hostgroupname</H3></FONT></B></TD></TR>
EOF

			# Get the list of hosts in this hostgroup...
			$hosts = $ENTS{hostgroup}->{INSTANCES}->{$entity_instance}->{FIELDS}->[3];

			# Split the list into an array...
			@hosts = split (/,/, $hosts);
			@hosts = sort (@hosts);

			# And for each host in hosts...
			foreach $host (@hosts) {

				# Call get_instances in list mode because there should only be one match...
				($instance_num) = get_instances ("host", 0, $host);

				# Figure the hostname and hostdesc...
				$host_name = $ENTS{host}->{INSTANCES}->{$instance_num}->{FIELDS}->[0];
				$host_desc = $ENTS{host}->{INSTANCES}->{$instance_num}->{FIELDS}->[1];

				print <<EOF;

<TR BGCOLOR=$COLOR2><TD><FONT COLOR=$TEXTCOLOR2><B>$host_name</B></FONT></TD>
<TD><FONT COLOR=$TEXTCOLOR2>$host_desc</FONT></TD>
<TD><A HREF="$CGIPATH/neat.cgi?mode=modify&entity_type=host&entity_instance=$instance_num"><FONT COLOR=$TEXTCOLOR2>View/Modify</FONT></A></TD>
<TD><A HREF="$CGIPATH/neat.cgi?mode=view&entity_type=host&entity_instance=$instance_num"><FONT COLOR=$TEXTCOLOR2>List Services</FONT></A></TD>
<TD>
<FORM ACTION="$CGIPATH/neat.cgi" METHOD=POST>
<INPUT TYPE=HIDDEN NAME="mode" VALUE="delete">
<INPUT TYPE=HIDDEN NAME="entity_type" VALUE="host">
<INPUT TYPE=HIDDEN NAME="entity_instance" VALUE="$instance_num">
<INPUT TYPE=CHECKBOX NAME="delete_confirm" VALUE="yes">
<FONT SIZE="-2"><INPUT TYPE=SUBMIT VALUE="Delete"></FONT>
</FORM>
</TD> 
</TR>
EOF
			}
		}

		################################### CONTACTGROUP ###################################
		elsif ($entity_type eq "contactgroup") {

			my ($contactgroupname, $contacts, @contacts, $contact, $instance_num, $contact_name);
			my ($contact_desc);

			# Get the contactgroup name...
			$contactgroupname = $ENTS{contactgroup}->{INSTANCES}->{$entity_instance}->{FIELDS}->[0];

			# Print the page banner...
			print <<EOF;
<TR BGCOLOR=$COLOR1><TD COLSPAN=5><B><FONT COLOR=$TEXTCOLOR1><H3>Viewing Members of Contactgroup $contactgroupname</H3></FONT></B></TD></TR>
EOF

			$contacts = $ENTS{contactgroup}->{INSTANCES}->{$entity_instance}->{FIELDS}->[2];
			@contacts = split (/,/, $contacts);
			@contacts = sort (@contacts);

			foreach $contact (@contacts) {

				# Call get_instances in list mode because there should only be one match...
				($instance_num) = get_instances ("contact", 0, $contact);

				$contact_name = $ENTS{contact}->{INSTANCES}->{$instance_num}->{FIELDS}->[0];
				$contact_desc = $ENTS{contact}->{INSTANCES}->{$instance_num}->{FIELDS}->[1];

				print <<EOF;
<TR BGCOLOR=$COLOR2><TD><FONT COLOR=$TEXTCOLOR2><B>$contact_name</B></FONT></TD>
<TD><FONT COLOR=$TEXTCOLOR2>$contact_desc</FONT></TD>
<TD COLSPAN=2>
<A HREF="$CGIPATH/neat.cgi?mode=modify&entity_type=contact&entity_instance=$instance_num"><FONT COLOR=$TEXTCOLOR2>View/Modify</FONT></A>
</TD>
<TD>
<FORM ACTION="$CGIPATH/neat.cgi" METHOD=POST>
<INPUT TYPE=HIDDEN NAME="mode" VALUE="delete">
<INPUT TYPE=HIDDEN NAME="entity_type" VALUE="contact">
<INPUT TYPE=HIDDEN NAME="entity_instance" VALUE="$instance_num">
<INPUT TYPE=CHECKBOX NAME="delete_confirm" VALUE="yes">
<FONT SIZE="-2"><INPUT TYPE=SUBMIT VALUE="Delete"></FONT>
</FORM>
</TD> 
</TR>
EOF
			}
		}

		############################### COMMAND ##############################################
		elsif ($entity_type eq "command") {
			my (@command_instances, $command_instance);

			print <<EOF;
<TR BGCOLOR=$COLOR1><TD COLSPAN=5><B><FONT COLOR=$TEXTCOLOR1><H3>Viewing Defined Commands</H3></FONT></B></TD></TR>
EOF

			# Figure out the command instancees...
			@command_instances = sort_by_field ("command", 0);

			# For each of these command instances...
			foreach $command_instance (@command_instances) {

				# Determine the command name...
				$command_name = $ENTS{command}->{INSTANCES}->{$command_instance}->{FIELDS}->[0];

				# and print the entry...
				print <<EOF;
<TR BGCOLOR=$COLOR2><TD><FONT COLOR=$TEXTCOLOR2><B>$command_name</B></FONT></TD>
<TD>
<A HREF="$CGIPATH/neat.cgi?mode=modify&entity_type=command&entity_instance=$command_instance"><FONT COLOR=$TEXTCOLOR2>View/Modify</FONT>
</TD>
<TD COLSPAN=3>
<FORM ACTION="$CGIPATH/neat.cgi" METHOD=POST>
<INPUT TYPE=HIDDEN NAME="mode" VALUE="delete">
<INPUT TYPE=HIDDEN NAME="entity_type" VALUE="command">
<INPUT TYPE=HIDDEN NAME="entity_instance" VALUE="$command_instance">
<INPUT TYPE=CHECKBOX NAME="delete_confirm" VALUE="yes">
<FONT SIZE="-2"><INPUT TYPE=SUBMIT VALUE="Delete"></FONT>
</FORM>
</TD> 
</TR>
EOF
			}
		}

		########################## HOST ######################################################
		elsif ($entity_type eq "host") {
			my ($hostname, @service_instances, $service_desc);

			# We have to locate the services under this host...

			# Figure out the name for this host...
			$hostname = $ENTS{host}->{INSTANCES}->{$entity_instance}->{FIELDS}->[0];

			# Print the page banner...
			print <<EOF;
<TR BGCOLOR=$COLOR1><TD COLSPAN=5><H3><B><FONT COLOR=$TEXTCOLOR1>Viewing Services Defined for $hostname</FONT></B></H3></TD></TR>
EOF

			# Get the instances, with the optional sort on field 1 of service...
			@service_instances = get_instances ("service", 0, $hostname, 1);

			foreach $instance (@service_instances) {
				#figure the description of this instance...
				$service_desc = $ENTS{service}->{INSTANCES}->{$instance}->{FIELDS}->[1];

				print <<EOF;

<TR BGCOLOR=$COLOR2><TD><FONT COLOR=$TEXTCOLOR2><B>$service_desc</B></FONT></TD>
<TD><A HREF="$CGIPATH/neat.cgi?mode=modify&entity_type=service&entity_instance=$instance"><FONT COLOR=$TEXTCOLOR2>View/Modify</FONT></A></TD>
<TD COLSPAN=3>
<FORM ACTION="$CGIPATH/neat.cgi" METHOD=POST>
<INPUT TYPE=HIDDEN NAME="mode" VALUE="delete">
<INPUT TYPE=HIDDEN NAME="entity_type" VALUE="service">
<INPUT TYPE=HIDDEN NAME="entity_instance" VALUE="$instance">
<INPUT TYPE=CHECKBOX NAME="delete_confirm" VALUE="yes">
<FONT SIZE="-2"><INPUT TYPE=SUBMIT VALUE="Delete"></FONT>
</FORM>
</TD> 
</TR>
EOF
			}
		}


		else {
			print "<TR><TD>Error</TD></TR>\n";
		}
	}


	# Finish printing the page...
	print <<EOF;
<TR BGCOLOR=$COLOR1><TD COLSPAN=5><CENTER><A HREF="$CGIPATH/neat.cgi"><B><FONT COLOR=$TEXTCOLOR1>Back to Main Page</FONT></A></B>
</TD></TR>
</TABLE>
</CENTER>
</BODY>
</HTML>
EOF

	# And return...
	return ();
}

####################################################################################
# This subroutine will returnt the current date and time in one of two formats...
####################################################################################
sub datetime {
	my ($sec, $min, $hour, $dayofmonth, $month, $year) = localtime (time ());
		 
	# Fix the year...
	$year = $year + 1900;

	# Fix the month...
	$month++;
						   
	# Replace a single digit with a 0 and that digit eg - 9 -> 09
	map (s/^(\d)$/0$1/, $month, $dayofmonth, $sec, $min, $hour);
								    
	# If we have no calling arguments - return the naked date and time...
	if (! @_) {
		return ("$year$month$dayofmonth", "$hour$min$sec");
	}
     
	# Otherwise, return a slighly nicer looking date and time...
	else {
		return ("$year/$month/$dayofmonth", "$hour:$min:$sec");
	}
}

###################################################################################
# This subroutine will sort the $ENTS{$entity_type}->{INSTANCES} hash, and
# return an array of keys, sorted alphabetically on the field defined in the 
# subroutine call...
###################################################################################
sub sort_by_field {
	my ($entity_type) = $_[0];
	my ($field_num) = $_[1];
	my (@sorted_keys);

	# Pretty hairy sort ... sort the keys on the contents of $field_num

	@sorted_keys = sort 	{ 
							$ENTS{$entity_type}->{INSTANCES}->{$a}->{FIELDS}->[$field_num] 
							cmp 
							$ENTS{$entity_type}->{INSTANCES}->{$b}->{FIELDS}->[$field_num] 
							} 
							keys (%{$ENTS{$entity_type}->{INSTANCES}});


	return (@sorted_keys);
}

###############################################################################
# This subroutine returns a list of entity instance numbers, where the 
# entity_type supplied contains the match string in the field num supplied...
###############################################################################
sub get_instances {
	my ($entity_type) = $_[0];
	my ($field_num) = $_[1];
	my ($string) = $_[2];

	if ($_[3]) {
		my ($optional_sort_field) = $_[3];
	}

	my (@instances, @parts, $part, $instance, $field_contents, $type, @instance_list);

	# Get a list of instances in the entity_type we are interested in...
	@instances = keys (%{$ENTS{$entity_type}->{INSTANCES}});

	# Determine is the field in question is a SINGLE or a MULTIPLE...
	$type = $ENTS{$entity_type}->{DEFINITION}->[$field_num]->{TYPE};

	# Step through them...
	foreach $instance (@instances) {

		# Get the contents of the field we are trying to match against...
		$field_contents = $ENTS{$entity_type}->{INSTANCES}->{$instance}->{FIELDS}->[$field_num];

		if ($type eq "MULTIPLE") {
			# Then we have a list we have to breakup and deal with...

			@parts = split (/,/, $field_contents, -1);

			# Look at each part, and if we match, then store this instance number....
			foreach $part (@parts) {
				if ($part eq $string) {
					push (@instance_list, $instance);
					last;
				}		
			}
		}

		else {
			# It's just a simple SINGLE...

			# Check if the field_contents equal the string we are looking for...
			if ($field_contents eq $string) {
				# if yes, return the instance_num...
				push (@instance_list, $instance);	
			}
		}
	}

	# Sort the @instance_list if $optional_sort_field is defined...

	if ($optional_sort_field) {	
		@instance_list = sort	{
			$ENTS{$entity_type}->{INSTANCES}->{$a}->{FIELDS}->[$optional_sort_field]
			cmp
			$ENTS{$entity_type}->{INSTANCES}->{$b}->{FIELDS}->[$optional_sort_field]
			} @instance_list;
	}

	# Return the results...
	return (@instance_list);
}

###############################################################################################
# This subroutine outputs a message to a logfile
############################################################################################### 
sub record_mod {
	my ($logfile) = $_[0];
	my ($username) = $_[1];
	my ($action) = $_[2];

	my ($date, $time) = datetime ();

	# Make sure we can open the logfile for appending...
	if (! open (LOGFILE, ">> $logfile")) {
		# if not, return failure...
		return (0);
	}

	# Print the message to the logfile...
	print LOGFILE "$date	$time	$username	$action\n";

	# Close the logfile...
	close (LOGFILE);

	# And return success...
	return (1);
}
