// Copyright (c) 2000-2001 Brad Hughes <bhughes@trolltech.com>
//
// Use, modification and distribution is allowed without limitation,
// warranty, or liability of any kind.
//

#ifndef   __decoder_hh
#define   __decoder_hh

class Decoder;
class DecoderFactory;

#include "eqpreset.h"

#include <qevent.h>
#include <qthread.h>
#include <qptrlist.h>

class QObject;
class QIODevice;

class Buffer;
class Recycler;
class Output;
class Visual;


class DecoderEvent : public QCustomEvent
{
public:
    enum Type { Decoding = (QEvent::User + 100), Stopped, Finished, Error };

    DecoderEvent(Type t)
	: QCustomEvent(t), error_msg(0)
    { ; }

    DecoderEvent(QString *e)
	: QCustomEvent(Error), error_msg(e)
    { ; }

    ~DecoderEvent()
    {
	if (error_msg)
	    delete error_msg;
    }

    const QString *errorMessage() const { return error_msg; }


private:
    QString *error_msg;
};


class Decoder : public QThread {
public:
    virtual ~Decoder();

    // Standard Decoder API
    virtual bool initialize() = 0;
    virtual double lengthInSeconds() = 0;
    virtual void seek(double) = 0;
    virtual void stop() = 0;

    // Equalizer
    virtual bool isEQSupported() const = 0;
    virtual void setEQEnabled(bool) = 0;
    virtual void setEQ(const EqPreset &preset) { eq = preset; }
    const EqPreset &EQ() const { return eq; }

    DecoderFactory *factory() const { return fctry; }

    void addListener(QObject *);
    void removeListener(QObject *);

    QIODevice *input() { return in; }
    Output *output() { return out; }
    void setInput(QIODevice *);
    void setOutput(Output *);

    QMutex *mutex() { return &mtx; }
    QWaitCondition *cond() { return &cnd; }

    void setBlockSize(unsigned int sz) { blksize = sz; }
    unsigned int blockSize() const { return blksize; }

    // static methods
    static QStringList all();
    static bool supports(const QString &);
    static void registerFactory(DecoderFactory *);
    static Decoder *create(const QString &, QIODevice *,
			   Output *, bool = FALSE);
    static QString filter();


protected:
    Decoder(DecoderFactory *, QIODevice *, Output *);

    void dispatch(const DecoderEvent &);
    void error(const QString &);


private:
    DecoderFactory *fctry;

    QPtrList<QObject> listeners;
    QIODevice *in;
    Output *out;
    EqPreset eq;

    QMutex mtx;
    QWaitCondition cnd;

    int blksize;
};


// DecoderFactories - All of them

class DecoderFactory
{
public:
    virtual bool supports(const QString &source) const = 0;
    virtual const QString &extension() const = 0; // file extension, ie. ".mp3" or ".ogg"
    virtual const QString &description() const = 0; // file type, ie. "MPEG Audio Files"
    virtual Decoder *create(QIODevice *, Output *, bool) = 0;
};


#if defined(XINGMP3)
class DecoderMP3Factory : public DecoderFactory
{
public:
    bool supports(const QString &) const;
    const QString &extension() const;
    const QString &description() const;
    Decoder *create(QIODevice *, Output *, bool);
};
#endif


#if defined(OGGVORBIS)
class DecoderOggFactory : public DecoderFactory
{
public:
    bool supports(const QString &) const;
    const QString &extension() const;
    const QString &description() const;
    Decoder *create(QIODevice *, Output *, bool);
};
#endif

#if defined(MAD)
class DecoderMADFactory : public DecoderFactory
{
public:
    bool supports(const QString &) const;
    const QString &extension() const;
    const QString &description() const;
    Decoder *create(QIODevice *, Output *, bool);
};
#endif

#endif // __decoder_hh
