/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Galeon includes */
#include "galeon.h"
#include "session.h"
#include "bookmarks.h"
#include "bookmarks_editor.h"
#include "eel-gconf-extensions.h"
#include "window.h"
#include "embed.h"
#include "glade.h"
#include "misc_general.h"

/* system includes */
#include <time.h>
#include <unistd.h>
#include <string.h>
#include <gtk/gtk.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnome/libgnome.h>
#include <libgnome/gnome-i18n.h>
#include <gnome-xml/parser.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/xmlmemory.h>
#include <gtkmozembed.h>

/*
 *  Session code 
 *
 *  Copyright (C) 2000 Matthew Aubury
 *
 *  This code implements the idea of "sessions", meaning the state of
 *  all the open browsers with respect to the URLs they're pointed at.
 *
 *  This should be useful in two ways:
 *    -- You need to quit X but have got lots of windows open
 *       with stuff you want to read, but you don't want to
 *       bookmark it all, just come back and see it later.
 *    -- By autosaving sessions we should offer a degree of 
 *       automatic crash recovery. This should mitigate some
 *       of the Mozilla/Galeon instability problems.
 *
 *  NOTE: this code won't do the right thing with frames yet. This is
 *  an ongoing problem with frames (they can't easily be referenced by
 *  URL) which is shared with bookmarks.
 *
 *  This code is quite derived from the history code,
 *  kudos to whoever did that...
 *
 *  -- MattA
 */

/*
 *  Hmm, I think we should really be using the gnome-client /
 *  gnome-session stuff to handle this properly. -- MattA 16/4/2001
 */

/* local function prototypes */
static xmlDocPtr session_load (const gchar *filename);
static void session_resume (xmlDocPtr doc, gboolean use_bookmarks);
static void session_close_all_windows (void);
static gint session_server_timeout_cb (gpointer data);

/** The global list of all GaleonWindow structures */
GList *all_windows = NULL;
gint window_count = 0;

/* session quit timeout id */
gint quit_timeout_id = 0;

/* true when we don't want to autosave */
gboolean dont_autosave_session = FALSE;

/* recently-loaded sessions.  we were keeping this stored in gconf, but it
 * doesn't sync changes to the listing quickly enough, and the whole thing
 * turns into a mess.  now, we just use gconf to load the list at startup
 * and save it at exit */
GList *recent_sessions = NULL;
gint num_recent_sessions = 0;

/**
 * session_autosave: save to the "crash recovery" session file
 */
void
session_autosave (void)
{
	gchar *filename;

	/* don't do anything if galeon is exiting, or all the windows are
	 * being closed for some other reason (e.g. session load) */
	if (dont_autosave_session) return;

	/* make the session filename string */
	filename = g_strconcat (g_get_home_dir (),
				"/.galeon/session_crashed.xml", NULL);

	/* save it out */
	session_save_to (filename);

	/* free allocated string */
	g_free (filename);
}

/**
 * session_autoresume: check to see if there's a crashed session and load it
 */
gint
session_autoresume (void)
{
	gchar *filename;
	xmlDocPtr session;
	gint crashed, saved, dialog_button;
	gint action = 2; /* 0 = restore, 1 = create bookmarks, 2 = discard,
			    3 = abort */
	GladeXML *gxml;
	GtkWidget *dialog, *restore_radio, *create_radio, *discard_radio;

	/* see if there's a crashed or saved session to resume */
	crashed = eel_gconf_get_integer (CONF_CRASH_CRASHED);
	saved = eel_gconf_get_integer (CONF_STATE_SESSION_SAVED);
	eel_gconf_set_integer (CONF_STATE_SESSION_SAVED, 0);

	if (crashed == FALSE && saved == FALSE)
	{
		return FALSE;
	}

	/* see if there's a session file to resume from */
	if (crashed)
	{
		filename = g_strconcat (g_get_home_dir (),
					"/.galeon/session_crashed.xml", NULL);
	}
	else
	{
		filename = g_strconcat (g_get_home_dir (),
					"/.galeon/session_saved.xml", NULL);
	}
	if (access (filename, F_OK) != 0)
	{
		g_free(filename);
		return FALSE;
	}
	
	/* load it */
	session = session_load (filename);
	g_free (filename);
	if (session == NULL)
	{
		return FALSE;
	}

	/* abort if no windows in session */
	if (session->root->childs == NULL)
	{
		xmlFreeDoc (session);
		return FALSE;
	}

	/* resume saved session */
	if (saved)
	{
		/* resume and free */
		session_resume (session, FALSE);
		xmlFreeDoc (session);
		return TRUE;
	}

	/* create dialog */
	gxml = glade_widget_new ("galeon.glade", "crash_recovery_dialog",
				 NULL, NULL);
	dialog = glade_xml_get_widget (gxml, "crash_recovery_dialog");
	restore_radio = glade_xml_get_widget (gxml, "restore_radio");
	create_radio = glade_xml_get_widget (gxml, "create_radio");
	discard_radio = glade_xml_get_widget (gxml, "discard_radio");
	gtk_object_unref (GTK_OBJECT (gxml));

	/* select the radio button that the user used last time */
	switch (eel_gconf_get_integer (CONF_STATE_RECOVERY_METHOD))
	{
		/* restore */
		case 0:
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
				restore_radio), TRUE);
			break;
		/* create bookmarks */
		case 1:
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
				create_radio), TRUE);
			break;
		/* discard */
		default:
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (
				discard_radio), TRUE);
			break;
	}

	/* display the dialog */
	gtk_widget_show_all (GNOME_DIALOG (dialog)->vbox);
	dialog_button = gnome_dialog_run (GNOME_DIALOG (dialog));

	/* if the user clicked the "start galeon" button... */
	if (dialog_button == 0)
	{
		/* find which radio button was selected */
		if (gtk_toggle_button_get_active (
					GTK_TOGGLE_BUTTON (restore_radio)))
			action = 0;
		else if (gtk_toggle_button_get_active (
					GTK_TOGGLE_BUTTON (create_radio)))
			action = 1;
		else
			action = 2;

		/* save state */
		eel_gconf_set_integer (CONF_STATE_RECOVERY_METHOD, action);
	}
	/* if the user clicked the "don't start galeon" button... */
	else if (dialog_button == 1)
	{
		action = 3;
	}
	/* if the user closed the dialog without clicking either button,
	 * we'll assume that they want to discard the session. */

	/* close the dialog, if it's still open */
	if (dialog_button != -1)
		gnome_dialog_close (GNOME_DIALOG (dialog));

	/* perform the requested action */
	switch (action)
	{
		/* open in windows */
		case 0:
			session_resume (session, FALSE);
			xmlFreeDoc (session);
			return TRUE;
		/* add bookmarks */
		case 1:
			session_resume (session, TRUE);
			xmlFreeDoc (session);
			return TRUE;
		/* discard */
		case 2:
			xmlFreeDoc (session);
			return FALSE;
		/* abort */
		default:
			/* exit immediately instead of returning to the
			 * main loop -- don't unset the crashed flag, thus
			 * enabling the user to resume the session later */
			galeon_exit (FALSE, FALSE);
			return FALSE;
	}
}

/**
 * session_save_to: record the session state to the give location
 */
void
session_save_to (const gchar *filename)
{
	GList *w;
	xmlNodePtr root_node;
	xmlNodePtr window_node;
	xmlNodePtr embed_node;
	xmlDocPtr doc;
	gint x = 0, y = 0, width = 0, height = 0;
	gchar buffer[32];
	gint i, num_embeds;

        doc = xmlNewDoc ("1.0");

	/* create and set the root node for the session */
        root_node = xmlNewDocNode (doc, NULL, "session", NULL);
        xmlDocSetRootElement (doc, root_node);

	/* iterate through all the windows */
	for (w = all_windows; w != NULL; w = g_list_next (w))
	{
		/* get this item */
		GaleonWindow *window = (GaleonWindow *)(w->data);
		if (window->magic != GALEON_WINDOW_MAGIC)
		{
			g_warning ("not a valid galeon window");
			continue;
		}

		/* skip if for some reason it doesn't contain any embeds */
		if (window->embed_list == NULL)
		{
			g_warning ("window with no GaleonEmbeds");
			continue;
		}

		/* make a new XML node */
		window_node = xmlNewDocNode (doc, NULL, "window", NULL);

		/* get window geometry */
		gdk_window_get_size (GTK_WIDGET (window->wmain)->window,
				     &width, &height);
		gdk_window_get_root_origin (GTK_WIDGET (window->wmain)->window,
					    &x, &y);

		/* set window properties */
		snprintf(buffer, 32, "%d", x);
		xmlSetProp (window_node, "x", buffer);
		snprintf(buffer, 32, "%d", y);
		xmlSetProp (window_node, "y", buffer);
		snprintf(buffer, 32, "%d", width);
		xmlSetProp (window_node, "width", buffer);
		snprintf(buffer, 32, "%d", height);
		xmlSetProp (window_node, "height", buffer);

		/* iterate through all the embeds in the order that they
		 * appear in the notebook */
		num_embeds = g_list_length (window->embed_list);
		for (i = 0; i < num_embeds; i++)
		{
			GtkWidget *mozembed;
			GaleonEmbed *embed;

			mozembed = gtk_notebook_get_nth_page (
				GTK_NOTEBOOK (window->notebook), i);
			embed = (GaleonEmbed *) gtk_object_get_data (
				GTK_OBJECT (mozembed), "GaleonEmbed");

			/* skip if embed is not valid */
			if (!embed || embed->magic != GALEON_EMBED_MAGIC)
			{
				g_warning ("Invalid GaleonEmbed");
				continue;
			}

			/* skip if it's a XUL dialog */
			if (embed->is_chrome)
			{
				continue;
			}

			/* skip if the wrapper is NULL, can't do anything
			 * with such a crippled embed */
			if (embed->wrapper == NULL)
			{
				g_warning ("GaleonEmbed with no GaleonWrapper");
				continue;
			}

			/* make a new XML node */
			embed_node = xmlNewDocNode (doc, NULL,
						    "embed", NULL);

			/* store title in the node */
			xmlSetProp (embed_node, "title", embed->title_utf8);

			/* special case where user has opened a new window
			 * or tab and entered a location, but it hasn't
			 * loaded, and modified_location hasn't been
			 * updated -- get URL from location entry */
			if (embed->is_active &&
			    (!embed->location || embed->location[0] == '\0' ||
			     !strcmp (embed->location, "about:blank")) &&
			    !embed->modified_location)
			{
				GaleonWindow *window;
				gchar *entry_text;

				window = embed->parent_window;
				entry_text = gtk_editable_get_chars (
					GTK_EDITABLE (window->location_entry),
					0, -1);
				xmlSetProp (embed_node, "url", entry_text);
				g_free (entry_text);
			}
			/* otherwise, if there's a modified location and no
			 * page is currently loaded, use the modified
			 * location. */
			else if ((!embed->location ||
				  embed->location[0] == '\0' ||
				  !strcmp (embed->location, "about:blank"))
				 && embed->modified_location)
			{
				xmlSetProp (embed_node, "url",
					    embed->modified_location);
			}
			/* otherwise, use the actual location. */
			else xmlSetProp (embed_node, "url", embed->location);

			/* insert node into the tree */
			xmlAddChild (window_node, embed_node);
		}

		/* add window into tree */
		if (g_list_length (window->embed_list) != 0)
		{
			xmlAddChild (root_node, window_node);
		}
	}

	/* save it all out to disk */
        xmlSaveFile (filename, doc);
	xmlFreeDoc (doc);
}

/**
 * session_load: load a session into a new XML document
 */
static xmlDocPtr
session_load (const gchar *filename)
{
	xmlDocPtr doc;

	if (access (filename, R_OK))
	{
		g_warning ("Unable to open session file \"%s\" for reading",
			   filename);
		return NULL;
	}

	/* read the session file */
        doc = xmlParseFile (filename);
        if (!doc)
	{
		g_warning ("Unable to parse session file \"%s\"", filename);
		return NULL;
	}

	return doc;
}

/**
 * session_resume: open browsers to resume the session state
 */
static void
session_resume (xmlDocPtr doc, gboolean use_bookmarks)
{
	xmlNodePtr window_node;
	xmlNodePtr embed_node;
	gchar *url;
	gchar *title;
	BookmarkItem *cat = NULL; 
	BookmarksEditorControls *controls;
	gint x = 0, y = 0, width = 0, height = 0;
	GdkWindow *gdk_window;
	GaleonWindow *window = NULL;
	GaleonEmbed *embed;
	gint embed_count = 0;

	if (use_bookmarks)
	{
		time_t time_num;
		struct tm *time_struct;
		gchar cat_title[128];

		/* form the string */
		time_num = time (NULL);
		time_struct = localtime (&time_num);
		strftime (cat_title, 128, _("URLs recovered on %Y-%m-%d %X"),
			  time_struct);
		/* FIXME: should time_struct be freed here? */

		/* create the folder */
		cat = bookmarks_add_folder (cat_title, NULL, NULL,
					    BOOKMARK_ADD_NEVER_PROMPT_TITLE);
	}

	/* iterate over session windows */
        for (window_node = doc->root->childs; window_node != NULL;
		window_node = window_node->next)
	{
		if (window_node->childs == NULL)
		{
			g_warning ("window with no GaleonEmbeds");
			continue;
		}

		if (!use_bookmarks)
		{
			/* create a default window */
			window = window_create (DEFAULT_CHROME, NULL);

			/* get the size of the window */
			x      = xmlGetIntProp (window_node, "x");
			y      = xmlGetIntProp (window_node, "y");
			width  = xmlGetIntProp (window_node, "width");
			height = xmlGetIntProp (window_node, "height");
			
			/* set the size and position of the window */
			gtk_window_set_default_size  
				(GTK_WINDOW (window->wmain), width, height);
			gtk_widget_set_uposition (GTK_WIDGET (window->wmain),
						  x, y);
			window->set_size = TRUE;
		}
		
		/* iterate over session embeds */
		for (embed_node = window_node->childs; embed_node != NULL;
		     embed_node = embed_node->next)
		{
			/* get the properties */
			url = xmlGetProp (embed_node, "url");
			title = xmlGetProp (embed_node, "title");

			if (use_bookmarks)
			{
				/* add the bookmark */
				bookmarks_add_bookmark (title, url, cat, NULL,
					BOOKMARK_ADD_NEVER_PROMPT_TITLE);
			}
			else
			{
				/* open a browser pointing to this URL */
				embed = embed_create_in_window (window,
					NULL, url,
					EMBED_CREATE_FORCE_APPEND |
					EMBED_CREATE_NEVER_JUMP |
					EMBED_CREATE_RAISE_WINDOW |
					EMBED_CREATE_DONT_AUTOSAVE_INITIALLY);
				embed_set_visibility (embed, TRUE);
				embed_count++;
			}

			/* free allocated strings */
			if (url != NULL) xmlFree (url);
			if (title != NULL) xmlFree (title);
		}

		if (!use_bookmarks)
		{
			gdk_window = GTK_WIDGET (window->wmain)->window;
			gdk_window_move (gdk_window, x, y);
		}
	}
 
	if (embed_count == 0)
	{
		embed_create_in_window (NULL, NULL, NULL,
					EMBED_CREATE_LOAD_DEFAULT_URL |
					EMBED_CREATE_FORCE_JUMP |
					EMBED_CREATE_RAISE_WINDOW);
	}

	/* Add an empty window bookmark to the temp bookmarks window, just
	   in case all the saved bookmarks are crashers */
	if (use_bookmarks)
	{
		controls = bookmarks_editor_show_dock (all_windows->data);
		bookmarks_editor_select_bookmark (controls, cat);
	}
}

void 
session_add_window (GaleonWindow *window)
{
	/* add it to the list of windows */
	all_windows = g_list_append (all_windows, window);

	/* Increment window count */
	window_count++;
}

void 
session_remove_window (GaleonWindow *window)
{
	/* remove window from list */
	all_windows = g_list_remove (all_windows, window);

	/* Decrement window count */
	window_count--;

	/* quit if this was the last window to go and
	 * we're not in server mode */
	if (all_embeds == NULL && !galeon_server_mode
	    && window_count ==0)
	{
		galeon_exit (TRUE, TRUE);
	}
}

void
session_save (void)
{
	gchar *filename;
	
	/* make the session filename string */
	filename = g_strconcat (g_get_home_dir (), 
				"/.galeon/session_saved.xml", NULL);

	/* save it out */
	session_save_to (filename);
	
	/* set config marker */
	eel_gconf_set_integer (CONF_STATE_SESSION_SAVED, 1);
	
	g_free (filename);
}

void 
session_quit (gboolean auto_save)
{
	/* set variable so we don't autosave for every embed close */
	dont_autosave_session = TRUE;

	/* save the session, if it was requested or if the user has the
	 * "always save" option enabled */
	if (auto_save ||
	    eel_gconf_get_boolean (CONF_GENERAL_ALWAYS_SAVE_SESSION)) {
		session_save ();
	}

	if (all_windows != NULL)
	{
		/* close all windows, this should quit */
		session_close_all_windows ();
	}
}

static void
session_close_all_windows (void)
{
	GList *copy;

	dont_autosave_session = TRUE;

	/* close all windows: this in turn will close all embeds */
	/* at the closing of the last window galeon_exit will be called */
	copy = g_list_copy (all_windows);
	g_list_foreach (copy, (GFunc)window_close, NULL);
	g_list_free (copy);

	/* wait for pending destruction events */
	while (gtk_events_pending ())
	{
		gtk_main_iteration ();
	}

	/* check we've been successful in shutting everything down */
	g_assert (g_list_length (all_windows) == 0);

	dont_autosave_session = FALSE;
}

/**
 * session_load_from: closes all current windows and opens the ones saved
 * in the file.  returns TRUE if the load is successful, and FALSE
 * otherwise.  if lenient is true, the function is more forgiving -- if the
 * given file is not readable, it looks in the default sessions dir and
 * adds a .xml extension.
 */
gboolean
session_load_from (const gchar *filename, gboolean lenient)
{
	gchar *modified_filename;
	xmlDocPtr session;

	if (lenient)
	{
		/* first try the given filename */
		modified_filename = g_strdup (filename);
		if (access (modified_filename, R_OK))
		{
			/* if that didn't work, try looking in the sessions
			 * directory */
			g_free (modified_filename);
			modified_filename =
				g_strconcat (g_get_home_dir (),
					     "/.galeon/sessions/",
					     filename, NULL);
			if (access (modified_filename, R_OK))
			{
				/* if that didn't work, try looking in the
				 * sessions dir again, with .xml appended
				 * to the name */
				g_free (modified_filename);
				modified_filename =
					g_strconcat (g_get_home_dir (),
						     "/.galeon/sessions/",
						     filename, ".xml", NULL);
				if (access (modified_filename, R_OK))
				{
					/* if that didn't work, give up */
					g_free (modified_filename);
					g_warning ("Unable to find session "
						   "file \"%s\"", filename);
					return FALSE;
				}
			}
		}
	}
	else modified_filename = g_strdup (filename);

	/* open the session and free the filename string */
	session = session_load (modified_filename);
	g_free (modified_filename);

	if (session != NULL) 
	{
		gboolean old_server_mode;
		/* Switch to serveer mode to avoid exiting from galeon
		 * when all windows are closed. This is a bit hacky... 
		 */
		old_server_mode = galeon_server_mode;
		galeon_server_mode = TRUE;
		/* close all windows */
		session_close_all_windows ();
		/* reopen windows... */
		session_resume (session, FALSE);
		xmlFreeDoc (session);
		/* restore server mode */
		galeon_server_mode = old_server_mode;

		return TRUE;
	} 
	else
	{
		gchar *s = g_strdup_printf 
			(_("Couldn't load session from \"%s\""), filename);
		GtkWidget *d = gnome_error_dialog (s);
		g_free (s);
		gnome_dialog_run_and_close (GNOME_DIALOG (d));

		return FALSE;
	}
}

/**
 * session_history_load: loads the list of recently-accessed session
 * filenames from prefs
 */
void
session_history_load (void)
{
	gint i, num;

	/* return if we've already loaded the sessions */
	if (recent_sessions != NULL) return;

	/* load number of sessions */
	num = eel_gconf_get_integer (CONF_HISTORY_SESSION_COUNT);

	/* load sessions */
	for (i = 0; i < num; i++) 
	{
		gchar *key, *file;

		key = g_strdup_printf (CONF_HISTORY_SESSION_PATH"/File%d", i);
		file = eel_gconf_get_string (key);

		if (file && g_file_exists (file))
		{
			recent_sessions = g_list_append (recent_sessions, file);
			num_recent_sessions++;
		}
		else g_free (file);

		g_free (key);
	}
}

/**
 * session_history_save: saves the list of recently-accessed session
 * filenames out to prefs
 */
void
session_history_save (void)
{
	gint i;
	GList *l;

	/* save the items */
	for (l = recent_sessions, i = 0; i < num_recent_sessions;
	     l = l->next, i++) 
	{
		gchar *key;

		/* this shouldn't happen, but i guess it can't hurt to
		 * check... */
		if (!l)
		{
			g_print ("session_history_save: missing session file "
				 "at position %d\n", i);
			break;
		}

		key = g_strdup_printf (CONF_HISTORY_SESSION_PATH"/File%d", i);
		eel_gconf_set_string (key, l->data);
		g_free (key);
	}

	/* save the number of items */
	eel_gconf_set_integer (CONF_HISTORY_SESSION_COUNT, num_recent_sessions);
}

/**
 * session_history_add: modifies the list of recently used sessions,
 * adding the filename at the top
 */
void
session_history_add (const gchar *filename)
{
	GList *l;
	gint max;

	/* remove the file from the list if it's already present.  this is
	 * a bit of a duplication of session_history_remove(), but we don't
	 * want to update the gui twice */
	for (l = recent_sessions; l; l = l->next) 
	{
		if (!strcmp (l->data, filename))
		{
			g_free (l->data);
			recent_sessions =
				g_list_remove_link (recent_sessions, l);
			num_recent_sessions--;
		}		
	}

	/* add the new item */
	recent_sessions = g_list_prepend (recent_sessions,
					  g_strdup (filename));
	num_recent_sessions++;

	/* find the maximum number of items */
	max = eel_gconf_get_integer (CONF_HISTORY_SESSION_MAXITEMS);

	/* shorten the list if necessary */
	while (num_recent_sessions > max)
	{
		l = g_list_last (recent_sessions);
		g_free (l->data);
		recent_sessions = g_list_remove_link (recent_sessions, l);
		num_recent_sessions--;
	}

	/* update the GUI */
	window_session_history_update_all ();
}

/**
 * session_history_remove: modifies the list of recently used sessions,
 * removing the given filename
 */
void
session_history_remove (const gchar *filename)
{
	GList *l;

	/* remove the file */
	for (l = recent_sessions; l; l = l->next) 
	{
		if (!strcmp (l->data, filename))
		{
			g_free (l->data);
			recent_sessions =
				g_list_remove_link (recent_sessions, l);
			num_recent_sessions--;
		}		
	}

	/* update the GUI */
	window_session_history_update_all ();
}

/**
 * session_open_confirmation: displays a confirmation dialog warning the
 * user that existing windows will be closed if a session is opened.
 * returns TRUE if the user chooses to proceed anyway, and false otherwise.
 */
gboolean
session_open_confirmation (void)
{
	gboolean retval = TRUE;

	if (eel_gconf_get_boolean (CONF_STATE_SESSION_OPEN_CONFIRMATION))
	{
		GladeXML *gxml;
		GtkWidget *dialog, *check;
		gint dialog_button;
		gboolean never_show = FALSE;

		/* create dialog */
		gxml = glade_widget_new ("galeon.glade",
					 "session_open_confirmation_dialog",
					 NULL, NULL);
		dialog = glade_xml_get_widget (gxml,
				"session_open_confirmation_dialog");
		check = glade_xml_get_widget (gxml, "never_show_check");
		gtk_object_unref (GTK_OBJECT (gxml));

		/* display the dialog */
		gtk_widget_show_all (GNOME_DIALOG (dialog)->vbox);
		dialog_button = gnome_dialog_run (GNOME_DIALOG (dialog));

		/* if the dialog is still open, get the state of the toggle
		 * button and then close the dialog */
		if (dialog_button > -1)
		{
			never_show = gtk_toggle_button_get_active (
					GTK_TOGGLE_BUTTON (check));
			gnome_dialog_close (GNOME_DIALOG (dialog));
		}

		/* if they chose no or closed the dialog, return FALSE */
		if (dialog_button != 0) retval = FALSE;

		/* otherwise, check if they chose to never show the warning
		 * again */
		else if (never_show == TRUE)
		{
			eel_gconf_set_boolean (
				CONF_STATE_SESSION_OPEN_CONFIRMATION, FALSE);
		}
	}

	return retval;
}

void
session_server_start_timeout ()
{
	if (quit_timeout_id > 0)
		session_server_stop_timeout ();
	if (all_embeds != NULL)
		return;
	if (galeon_server_timeout == 0)
		return;

	if (galeon_server_timeout < 10)
		/* I don't wnat strange races when closing an embed */
		galeon_server_timeout = 10;

	quit_timeout_id = gtk_timeout_add 
		(galeon_server_timeout * 1000,
		 session_server_timeout_cb,
		 NULL);
}

void 
session_server_stop_timeout ()
{
	if (quit_timeout_id == 0)
		return;

	gtk_timeout_remove (quit_timeout_id);
	quit_timeout_id = 0;
}

static gint
session_server_timeout_cb (gpointer data)
{
	if (all_embeds != NULL
	    || window_count != 0)
		return FALSE;

	galeon_server_mode = FALSE;

	session_quit (FALSE);
	galeon_exit (TRUE, TRUE);

	/* not really important what we return here */
	return FALSE; 
}
