/*
 *  Copyright (C) 2001 Jorn Baayen 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  TODO a security page. I won't do it myself since i have no clue
 *  about that stuff, but patches are very welcome :P
 */

#include "galeon.h"
#include "dialog.h"
#include "mozilla.h"
#include "page_info.h"
#include "embed.h"
#include "state.h"
#include "eel-gconf-extensions.h"
#include "glade.h"
#include "misc_gui.h"

#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <gtk/gtkclist.h>
#include <gtk/gtkentry.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-app.h>
#include <libgnomeui/gnome-app-helper.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-stock.h>

typedef struct
{
	gchar *url;
	gchar *title;
} RowInfo;

/* function prototypes */
void page_information_close_clicked_cb (GtkWidget *widget,
					PageInfoDialog *dialog);
void page_info_images_clist_select_row_cb (GtkCList *clist, gint row,
					   gint column, GdkEventButton *event,
					   PageInfoDialog *dialog);
void page_info_images_clist_unselect_row_cb (GtkCList *clist, gint row,
					     gint column, GdkEventButton *event,
					     PageInfoDialog *dialog);
void page_info_clist_click_column_cb (GtkCList *clist, gint column,
				      PageInfoDialog *dialog);
gboolean page_info_generic_clist_button_press_event_cb (GtkWidget *widget,
					                GdkEventButton *event,
					                PageInfoDialog *dialog);
gboolean page_information_delete_cb (GtkWidget *widget, GdkEvent *event,
				     PageInfoDialog *dialog);

/**
 * page_info_show_dialog: show page info dialog
 */
void
page_info_show_dialog (GaleonEmbed *embed, PageInfoPage page)
{
	GladeXML *gxml;
	PageInfoDialog *dialog = g_new0 (PageInfoDialog, 1);
	gchar *tmp;
	GList *list, *l;

	/* build dialog from glade file */
	gxml = glade_widget_new ("galeon.glade", 
				 "page_information_dialog",
				 &(dialog->dialog), dialog);

	dialog->embed = embed;

	/* get notebook widget and switch to the right page */
	dialog->notebook = glade_xml_get_widget (gxml, "page_info_notebook");
	gtk_notebook_set_page (GTK_NOTEBOOK (dialog->notebook), page);

	/* get paned */
	dialog->paned = glade_xml_get_widget (gxml, "page_info_paned");

	/* general page */
	dialog->title =
		glade_xml_get_widget (gxml, "page_info_title_entry");
	tmp = mozilla_get_frame_title (embed);
	if (tmp)
		gtk_entry_set_text (GTK_ENTRY (dialog->title), tmp);
	g_free (tmp);
	dialog->url =
		glade_xml_get_widget (gxml, "page_info_url_entry");
	tmp = mozilla_get_document_url (embed);
	if (tmp)
		gtk_entry_set_text (GTK_ENTRY (dialog->url), tmp);
	g_free (tmp);
	dialog->date =
		glade_xml_get_widget (gxml, "page_info_date_entry");
	tmp = mozilla_get_last_modified (embed);
	if (tmp && !strcmp (tmp, "January 1, 1970 GMT"))
	{
		g_free (tmp);
		tmp = g_strdup (_("Unknown"));
	}
	if (tmp)
		gtk_entry_set_text (GTK_ENTRY (dialog->date), tmp);
	g_free (tmp);

	/* images page */
	dialog->images_clist =
		glade_xml_get_widget (gxml, "page_info_images_clist");
	list = mozilla_list_images (embed);
	for (l = list; l; l = g_list_next (l))
	{
		ImageListItem *i = (ImageListItem *) l->data;
		RowInfo *ri = g_new0 (RowInfo, 1);
		gchar *row[5];
		gint r;
		
		row[0] = ri->url = i->url;
		row[1] = g_strdup_printf ("%d", i->width);
		row[2] = g_strdup_printf ("%d", i->height);
		row[3] = i->title;
		row[4] = i->alt;

		r = gtk_clist_append (GTK_CLIST (dialog->images_clist), row);
		gtk_clist_set_row_data (GTK_CLIST (dialog->images_clist), r,
					ri);

		if (i->title && strlen (i->title) > 0)
		{
			ri->title = i->title;
			g_free (i->alt);
		}
		else
		{
			ri->title = i->alt;
			g_free (i->title);
		}
		
		g_free (row[1]);
		g_free (row[2]);
		g_free (i);
	}
	g_list_free (list);
	dialog->image =
		glade_xml_get_widget (gxml, "page_info_images_pixmap");
	gtk_widget_hide (dialog->image);

	/* forms page */
	dialog->forms_clist =
		glade_xml_get_widget (gxml, "page_info_forms_clist");
	list = mozilla_list_forms (embed);
	for (l = list; l; l = g_list_next (l))
	{
		FormListItem *i = (FormListItem *) l->data;
		RowInfo *ri = g_new0 (RowInfo, 1);
		gchar *row[3];
		gint r;
		
		row[0] = ri->url = i->action;
		row[1] = i->method;
		row[2] = ri->title = i->name;

		r = gtk_clist_append (GTK_CLIST (dialog->forms_clist), row);
		gtk_clist_set_row_data (GTK_CLIST (dialog->forms_clist), r,
					ri);

		g_free (i->method);
		g_free (i);
	}
	g_list_free (list);

	/* stylesheets page */
	dialog->css_clist =
		glade_xml_get_widget (gxml, "page_info_stylesheets_clist");
	list = mozilla_list_css (embed);
	for (l = list; l; l = g_list_next (l))
	{
		LinkListItem *i = (LinkListItem *) l->data;
		RowInfo *ri = g_new0 (RowInfo, 1);
		gchar *row[2];
		gint r;
		
		row[0] = ri->url = i->url;
		row[1] = ri->title = i->title;

		r = gtk_clist_append (GTK_CLIST (dialog->css_clist), row);
		gtk_clist_set_row_data (GTK_CLIST (dialog->css_clist), r,
					ri);

		g_free (i);
	}
	g_list_free (list);

	/* links page */
	dialog->links_clist =
		glade_xml_get_widget (gxml, "page_info_links_clist");
	list = mozilla_list_links (embed);
	
	for (l = list; l; l = g_list_next (l))
	{
		LinkListItem *i = (LinkListItem *) l->data;
		RowInfo *ri = g_new0 (RowInfo, 1);
		gchar *row[3];
		gint r;
		
		row[0] = ri->url = i->url;
		row[1] = ri->title = i->title;
		row[2] = i->rel;

		r = gtk_clist_append (GTK_CLIST (dialog->links_clist), row);
		gtk_clist_set_row_data (GTK_CLIST (dialog->links_clist), r,
					ri);

		g_free (i->rel);
		g_free (i);
	}
	g_list_free (list);

	/* load saved state */
	state_load_column_widths (dialog->images_clist, "page_info_images");
	gtk_clist_set_sort_column (GTK_CLIST (dialog->images_clist),
				   eel_gconf_get_integer
				   	(CONF_STATE_PAGE_INFO_IMAGES_COLUMN));
	gtk_clist_set_sort_type (GTK_CLIST (dialog->images_clist),
				 eel_gconf_get_boolean
			 		(CONF_STATE_PAGE_INFO_IMAGES_ORDER));
	state_load_column_widths (dialog->forms_clist, "page_info_forms");
	gtk_clist_set_sort_column (GTK_CLIST (dialog->forms_clist),
				   eel_gconf_get_integer
				   	(CONF_STATE_PAGE_INFO_FORMS_COLUMN));
	gtk_clist_set_sort_type (GTK_CLIST (dialog->forms_clist),
				 eel_gconf_get_boolean
			 		(CONF_STATE_PAGE_INFO_FORMS_ORDER));
	state_load_column_widths (dialog->css_clist, "page_info_css");
	gtk_clist_set_sort_column (GTK_CLIST (dialog->css_clist),
				   eel_gconf_get_integer
				   	(CONF_STATE_PAGE_INFO_CSS_COLUMN));
	gtk_clist_set_sort_type (GTK_CLIST (dialog->css_clist),
				 eel_gconf_get_boolean
			 		(CONF_STATE_PAGE_INFO_CSS_ORDER));
	state_load_column_widths (dialog->links_clist, "page_info_links");
	gtk_clist_set_sort_column (GTK_CLIST (dialog->links_clist),
				   eel_gconf_get_integer
				   	(CONF_STATE_PAGE_INFO_LINKS_COLUMN));
	gtk_clist_set_sort_type (GTK_CLIST (dialog->links_clist),
				 eel_gconf_get_boolean
			 		(CONF_STATE_PAGE_INFO_LINKS_ORDER));
	state_load_pane_pos (dialog->paned, "page_info");
	gtk_clist_sort (GTK_CLIST (dialog->images_clist));
	gtk_clist_sort (GTK_CLIST (dialog->forms_clist));
	gtk_clist_sort (GTK_CLIST (dialog->css_clist));
	gtk_clist_sort (GTK_CLIST (dialog->links_clist));

	/* finish up */
	gtk_widget_show (dialog->dialog);
	dialog_set_parent (dialog->dialog, embed->parent_window->wmain);
	gtk_object_unref (GTK_OBJECT (gxml));
}

/**
 * page_information_close_clicked_cb: close clicked
 */
void
page_information_close_clicked_cb (GtkWidget *widget,
				   PageInfoDialog *dialog)
{
	GtkWidget *w = dialog->dialog;
	page_information_delete_cb (widget, NULL, dialog);
	gtk_widget_destroy (w);
}

/**
 * page_information_delete_cb: window deleted
 */
gboolean
page_information_delete_cb (GtkWidget *widget,
			    GdkEvent *event,
		            PageInfoDialog *dialog)
{
	RowInfo *ri;
	gint i;

	/* save state */
	state_save_column_widths (dialog->images_clist, "page_info_images");
	eel_gconf_set_integer (CONF_STATE_PAGE_INFO_IMAGES_COLUMN,
				GTK_CLIST (dialog->images_clist)->sort_column);
	eel_gconf_set_boolean (CONF_STATE_PAGE_INFO_IMAGES_ORDER,
				GTK_CLIST (dialog->images_clist)->sort_type);
	state_save_column_widths (dialog->forms_clist, "page_info_forms");
	eel_gconf_set_integer (CONF_STATE_PAGE_INFO_FORMS_COLUMN,
				GTK_CLIST (dialog->forms_clist)->sort_column);
	eel_gconf_set_boolean (CONF_STATE_PAGE_INFO_FORMS_ORDER,
				GTK_CLIST (dialog->forms_clist)->sort_type);
	state_save_column_widths (dialog->css_clist, "page_info_css");
	eel_gconf_set_integer (CONF_STATE_PAGE_INFO_CSS_COLUMN,
				GTK_CLIST (dialog->css_clist)->sort_column);
	eel_gconf_set_boolean (CONF_STATE_PAGE_INFO_CSS_ORDER,
				GTK_CLIST (dialog->css_clist)->sort_type);
	state_save_column_widths (dialog->links_clist, "page_info_links");
	eel_gconf_set_integer (CONF_STATE_PAGE_INFO_LINKS_COLUMN,
				GTK_CLIST (dialog->links_clist)->sort_column);
	eel_gconf_set_boolean (CONF_STATE_PAGE_INFO_LINKS_ORDER,
				GTK_CLIST (dialog->links_clist)->sort_type);
	state_save_pane_pos (dialog->paned, "page_info");

	/* free clist row data */
	for (i = GTK_CLIST (dialog->images_clist)->rows - 1; i >= 0; i--)
	{
		ri = gtk_clist_get_row_data
			(GTK_CLIST (dialog->images_clist), i);
		if (!ri) continue;

		g_free (ri->url);
		g_free (ri->title);
		g_free (ri);
	}
	for (i = GTK_CLIST (dialog->links_clist)->rows - 1; i >= 0; i--)
	{
		ri = gtk_clist_get_row_data
			(GTK_CLIST (dialog->links_clist), i);
		if (!ri) continue;

		g_free (ri->url);
		g_free (ri->title);
		g_free (ri);
	}
	for (i = GTK_CLIST (dialog->css_clist)->rows - 1; i >= 0; i--)
	{
		ri = gtk_clist_get_row_data
			(GTK_CLIST (dialog->css_clist), i);
		if (!ri) continue;

		g_free (ri->url);
		g_free (ri->title);
		g_free (ri);
	}
	for (i = GTK_CLIST (dialog->forms_clist)->rows - 1; i >= 0; i--)
	{
		ri = gtk_clist_get_row_data
			(GTK_CLIST (dialog->forms_clist), i);
		if (!ri) continue;

		g_free (ri->url);
		g_free (ri->title);
		g_free (ri);
	}

	/* free dialog struct */
	g_free (dialog->selected_url);
	dialog->selected_url = NULL;
	dialog->selected_row = NULL;
	dialog->image = NULL;
	g_free (dialog);
	dialog = NULL;

	return TRUE;
}

/**
 * page_info_images_clist_select_row_cb: row selected in the images clist
 */
void
page_info_images_clist_select_row_cb (GtkCList *clist, gint row,
				      gint column, GdkEventButton *event,
				      PageInfoDialog *dialog)
{
	gint t;
	RowInfo *ri = gtk_clist_get_row_data (clist, row);
	SetPixmapInfo *info;
	gchar *path;

	/* create a tmp. file */
	path = g_strconcat (g_get_tmp_dir (),
			    "/galeon-page-info-XXXXXX", NULL);
	if ((t = mkstemp (path)) < 0)
	{
		g_free (path);
		return;
	}
	close (t);

	g_free (dialog->selected_url);
	dialog->selected_url = g_strdup (ri->url);
	dialog->selected_row = ri;
	
	info = g_new0 (SetPixmapInfo, 1);
	info->url = ri->url;
	info->pixmap = dialog->image;
	info->selected_url = dialog->selected_url;

	gtk_widget_queue_resize (dialog->image->parent);

	/* save to tmp location and update pixmap */
	mozilla_save_url (NULL, ri->url, path, ACTION_SET_PIXMAP,
			  (gpointer) info);

	g_free (path);
}

/**
 * page_info_images_clist_unselect_row_cb: row deselected in the images clist
 */
void
page_info_images_clist_unselect_row_cb (GtkCList *clist, gint row,
					gint column, GdkEventButton *event,
					PageInfoDialog *dialog)
{
	gtk_widget_hide (dialog->image);
	gtk_widget_queue_resize (dialog->image->parent);
	g_free (dialog->selected_url);
	dialog->selected_url = NULL;
	dialog->selected_row = NULL;
}

/**
 * page_info_clist_click_column_cb: clist column clicked; sort
 */
void
page_info_clist_click_column_cb (GtkCList *clist, gint column,
				 PageInfoDialog *dialog)
{
	if (clist->sort_column == column)
	{
		if (clist->sort_type == GTK_SORT_ASCENDING)
			gtk_clist_set_sort_type (clist, GTK_SORT_DESCENDING);
		else
			gtk_clist_set_sort_type (clist, GTK_SORT_ASCENDING);
	}
	gtk_clist_set_sort_column (clist, column);
	gtk_clist_sort (clist);
}

/**
 * page_info_generic_clist_button_press_event_cb: present a nice contextmenu
 * allowing you to save the selected url
 */
gboolean
page_info_generic_clist_button_press_event_cb (GtkWidget *widget,
					       GdkEventButton *event,
					       PageInfoDialog *dialog)
{
	gchar *url, *title;
	GaleonEmbed *embed = dialog->embed;

	/* get the selection */
	if (GTK_IS_CLIST (widget))
	{
		gint row, column;
		RowInfo *ri;
		GtkCList *clist = GTK_CLIST (widget);
		gtk_clist_get_selection_info (clist, event->x, event->y,
					      &row, &column);
		ri = gtk_clist_get_row_data (clist, row);
		if (!ri) return FALSE;
		url = ri->url;
		title = ri->title;
	}
	else
	{
		RowInfo *ri = dialog->selected_row;
		if (!ri) return FALSE;
		url = ri->url;
		title = ri->title;
	}

	if (!url) return FALSE;
	if (!embed_exists (embed)) embed = NULL;

	/* load url */
	if ((event->button == 2 || event->type == GDK_2BUTTON_PRESS)
	    && GTK_IS_CLIST (widget))
	{
		embed_activate_link_mouse (embed, NULL, url, event);
		return TRUE;
	}

	/* contextmenu */
	if (event->button == 3)
	{
		GtkWidget *menu;
		gint action;

		/* context menu */
		static GnomeUIInfo menu_uiinfo[] =
		{
			GNOMEUIINFO_ITEM_STOCK (N_("Open in a new window"),
						NULL, NULL, GNOME_STOCK_MENU_NEW),
			GNOMEUIINFO_ITEM_STOCK (N_("Open in a new tab"),
						NULL, NULL, GNOME_STOCK_MENU_NEW),
			GNOMEUIINFO_ITEM_STOCK (N_("Save as..."),
						NULL, NULL, GNOME_STOCK_MENU_SAVE_AS),
			GNOMEUIINFO_ITEM_STOCK (N_("Copy location"),
						NULL, NULL, GNOME_STOCK_MENU_COPY),
			GNOMEUIINFO_ITEM_STOCK (N_("Add bookmark"),
						NULL, NULL, GNOME_STOCK_MENU_BLANK),
			GNOMEUIINFO_END
		};
	
		/* show context menu */
		menu = misc_gui_new_popup_menu_lock_accels (menu_uiinfo);
		action = gnome_popup_menu_do_popup_modal 
			(GTK_WIDGET (menu), NULL, NULL, event, NULL);

		/* do appropiate action */
		switch (action)
		{
		case 0:
			embed_activate_link (embed, NULL,
					     url, LINKSTATE_NEWWIN);
			break;
		case 1:
			embed_activate_link (embed, NULL,
					     url, LINKSTATE_NEWTAB);
			break;
		case 2:
			/* pass NULL since we dont want the saved
			 * message to show up in the embed */
			embed_save_image (NULL, url, TRUE);
			break;
		case 3:
			if (embed == NULL && all_embeds)
				embed = (GaleonEmbed *)(all_embeds->data);
			if (embed != NULL)
				embed_copy_text_to_clipboard (url, embed);
			break;
		case 4:
			bookmarks_add_bookmark (title, url, NULL,
					GTK_WINDOW (dialog->dialog), 0);
			break;
		default:
			break;
		}

		/* destroy the popup menu */
		gtk_widget_unref (menu);
		
		return TRUE;
	}

	return FALSE;
}
