/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/* Galeon includes */
#include "galeon.h"
#include "GaleonAutomation.h"
#include "prefs.h"
#include "history.h"
#include "embed.h"
#include "session.h"
#include "bookmarks.h"
#include "window.h"
#include "dialog.h"
#include "favicon.h"
#include "gestures.h"
#include "js_console.h"
#include "mime.h"
#include "mozilla.h"
#include "misc_gui.h"

#include <gtkmozembed.h>
#include <glade/glade.h>
#include <libgnome/gnome-util.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-client.h>
#include <libgnomeui/gnome-init.h>
#include <libgnomeui/gnome-geometry.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <liboaf/liboaf.h>
#ifdef ENABLE_APPLET
#include <applet-widget.h>
#endif
#ifdef ENABLE_GNOME_FILE_SELECTOR
#include <bonobo/bonobo-main.h>
#endif /* !ENABLE_GNOME_FILE_SELECTOR */
#ifdef ENABLE_NAUTILUS_VIEW
/* FIXME: include only the needed bits */
#include <bonobo.h>
#include <libnautilus/libnautilus.h>
#include "galeon-nautilus-view.h"
#endif

/* local function prototypes */
static gint translate_url_arguments (poptContext context, gchar ***url);
static CORBA_ORB corba_init (int *argc, char *argv[]);
static gboolean  new_view_on_running_shell (gint n_urls, gchar **url);
static void new_shell (CORBA_ORB orb);
static gboolean galeon_init (int argc, char *argv[]);
static void create_default_browser (void);
static inline ViewState get_viewstate (void);

#ifdef ENABLE_NAUTILUS_VIEW
static BonoboObject *galeon_nautilus_view_new (BonoboGenericFactory *factory, const char *id, gpointer data);
static void galeon_init_factory (void);
#endif

/* import from Galeon-impl.c */
Galeon_Browser impl_Galeon_Browser__create (PortableServer_POA poa, 
					    CORBA_Environment *corba_env);

/* local function prototypes */
gboolean client_save_yourself_cb (GnomeClient *client, gint phase, 
				  GnomeSaveStyle save_style, gint shutdown,
				  GnomeInteractStyle interact_style, gint fast);
void client_die_cb (GnomeClient *client, gpointer data);

/* global variables */
gboolean galeon_server_mode = FALSE;
gint galeon_server_timeout = -1;
gboolean galeon_panel_mode = FALSE;
gboolean galeon_exiting = FALSE;

Galeon_Browser galeon_automation_client = NULL;
#ifdef ENABLE_NAUTILUS_VIEW
BonoboGenericFactory *galeon_nautilusview_factory = NULL;
#endif

/* file local variables */
CORBA_Environment corba_env;                 /* corba environment for errors */
static gboolean open_in_existing   = FALSE;  /* load in existing window?     */
static gboolean open_in_new_tab    = FALSE;  /* force open in a new tab?     */
static gboolean noraise            = FALSE;  /* no raise                     */
static gboolean open_in_new_window = FALSE;  /* force open in a new window?  */
static gboolean open_fullscreen    = FALSE;  /* open galeon in full screen ? */
static gchar *session_filename     = NULL;   /* the session filename         */
static gchar *geometry_string      = NULL;   /* the geometry string          */
static gchar *bookmark_url         = NULL;   /* the temp bookmark to add     */
static gboolean close_option       = FALSE;  /* --close                      */
static gboolean quit_option        = FALSE;  /* --quit                       */

/* mozilla version detected at runtime */
int mozilla_version=0;

/* command line argument parsing structure */
static struct poptOption popt_options[] =
{
	{ NULL, '\0', POPT_ARG_INCLUDE_TABLE, &oaf_popt_options, 0, NULL,
	  NULL },
	{ "new-tab", 'n', POPT_ARG_NONE, &open_in_new_tab, 0,
	  N_("Open a new tab in an existing Galeon window"),
	  NULL },
	{ "new-window", 'w', POPT_ARG_NONE, &open_in_new_window, 0,
	  N_("Open a new window in an existing Galeon process"),
	  NULL },
	{ "noraise", '\0', POPT_ARG_NONE, &noraise, 0,
	  N_("Do not raise the window when opening a page in an existing Galeon process"),
	  NULL },
	{ "fullscreen", 'f', POPT_ARG_NONE, &open_fullscreen, 0,
	  N_("Run Galeon in full screen mode"),
	  NULL },
	{ "existing", 'x', POPT_ARG_NONE, &open_in_existing, 0,
	  N_("Attempt to load URL in existing Galeon window"),
	  NULL },
	{ "load-session", 'l', POPT_ARG_STRING, &session_filename, 0,
	  N_("Load the given session file"),
	  N_("FILE") },
#ifdef ENABLE_APPLET
	{ "panel", 'p', POPT_ARG_NONE, &galeon_panel_mode, 0,
	  N_("Open as a panel applet"),
	  NULL },		  
#endif
	{ "server", 's', POPT_ARG_NONE, &galeon_server_mode, 0,
	  N_("Don't open any windows; instead act as a server "
	     "for quick startup of new Galeon instances"),
	  NULL },		  
	{ "server-timeout", 'e', POPT_ARG_INT, &galeon_server_timeout, 0,
	  N_("Time of inactivity (in seconds) until the server "
	     "shuts down itself. Defaults to one hour, use 0 to disable."),
	  N_("TIME") },		  
	{ "add-bookmark", 't', POPT_ARG_STRING, &bookmark_url,
	  0, N_("Add a bookmark (don't open any window)"), 
	  N_("URL")},
	{ "geometry", 'g', POPT_ARG_STRING, &geometry_string,
	  0, N_("Create the initial window with the given geometry.\n"
		"see X(1) for the GEOMETRY format"),
	  N_("GEOMETRY")},
	{ "close", 'c', POPT_ARG_NONE, &close_option, 0,
	  N_("Close all Galeon windows"),
	  NULL },
	{ "quit", 'q', POPT_ARG_NONE, &quit_option, 0,
	  N_("Same as --close, but exits server mode too"),
	  NULL },

	/* terminator, must be last */
	{ NULL, 0, 0, NULL, 0, NULL, NULL }
};

/**
 * main: main entry point
 *
 * Set values to standard values, and change later by reading the 
 * options defined at the command line.
 * 
 * Initialize localization, glade, gnome, etc.
 */
int
main (int argc, char *argv[])
{
	GaleonEmbed *last_embed = NULL;
	gboolean tabbed_mode;
	poptContext context;
	gboolean no_default;
	CORBA_ORB orb;
	gint i, n_urls;
	gchar **url;

	 /* make stdout line buffered - we only use it for debug info */
	setvbuf (stdout, NULL, _IOLBF, 0);

#ifdef ENABLE_NLS
	/* initialise localisation */
	bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
	textdomain (PACKAGE);
#endif

#ifdef ENABLE_APPLET
	/* panel mode? this check has to be done here since we
	 * don't use gnome_init_* when in panel mode */
	for (i = 1; i < argc; i++)
	{
		if (strcmp (argv[i], "-p") == 0 || 
		    strcmp (argv[i], "--panel") == 0)
		{
			galeon_panel_mode = TRUE;
		}
	}

	/* initialise GNOME */
	if (galeon_panel_mode)
	{
		applet_widget_init (PACKAGE, VERSION, argc, argv, 
				    popt_options, 0, &context);
	}
	else
#endif
	{
		gnome_init_with_popt_table (PACKAGE, VERSION, argc, argv,
					    popt_options /* oaf_popt_options*/
					    , 0, &context);
	}

	if (galeon_panel_mode && galeon_server_mode)
	{
		galeon_server_mode = FALSE;
		g_message (_("Panel mode. "
			     "Ignoring server mode option."));
	}

	/* set the server timeout to an hour by default if the option 
	 * was not present */
	if (galeon_server_timeout < 0)
	{
		galeon_server_timeout = 1 * 60 * 60;
	}
	
	/* load arguments that aren't regular options (urls to load) */
	n_urls = translate_url_arguments (context, &url);

	/* initialise CORBA */
	orb = corba_init (&argc, argv);
#ifdef ENABLE_NAUTILUS_VIEW
	bonobo_activate ();
#endif

	/* check if galeon is already running and use that instead */
	if (!galeon_panel_mode && !galeon_server_mode)
	{
		if (new_view_on_running_shell (n_urls, url))
		{
			/* done */
			return 0;
		} 
	}

	/* if no running shell exist, quit and close prefs
	 * are invalid
	 */
	if (close_option || quit_option)
	{
		g_message (_("Nothing to close: "
			     "no opened Galeon windows found."));
		return 0;
	}

	/* start a new CORBA shell */
	new_shell (orb);

	/* initialise required external and internal Galeon services */
	no_default = galeon_init (argc, argv);

	/* load the session if requested */
	if (session_filename)
		session_load_from (session_filename, TRUE);

	/* add a temp bookmark and exit if needed */
	if (bookmark_url != NULL)
	{
		bookmarks_load ();
		/* this will save the bookmarks too -- Matthew */
		bookmarks_add_bookmark (NULL, bookmark_url, NULL, NULL,
					BOOKMARK_ADD_NEVER_PROMPT_TITLE);
		return 0;
	}
	
	/* check if urls should be opened in tabs, and if so, find an embed */
	if (get_viewstate () != VIEWSTATE_WINDOW) tabbed_mode = TRUE;
	else tabbed_mode = FALSE;
	if (tabbed_mode)
	{
		if (all_windows)
		{
			GaleonWindow *window = all_windows->data;
			if (window && window->magic == GALEON_WINDOW_MAGIC)
			{
				if (window->active_embed &&
				    window->active_embed->magic ==
				    	GALEON_EMBED_MAGIC)
				{
					last_embed = window->active_embed;
				}
			}
		}
	}

	/* open all the command-line supplied urls */
	for (i = 0; i < n_urls; i++)
	{
		last_embed = embed_create_after_embed (last_embed,
			!tabbed_mode, url[i], EMBED_CREATE_RAISE_WINDOW |
			EMBED_CREATE_FORCE_APPEND);
		if (open_fullscreen && last_embed->parent_window)
		{
			window_set_fullscreen_mode (last_embed->parent_window, 
						    TRUE);
		}
	}
	g_strfreev (url);

	/* check if browser exists, or create default one */
	if (!no_default && !galeon_server_mode && !galeon_panel_mode)
	{
		create_default_browser ();
	}

	/* start quit timeout if no window has been opened */
	if (galeon_server_mode && (all_embeds == NULL))
	{
		session_server_start_timeout ();
	}

	/* start main loop */
#ifdef ENABLE_APPLET
	if (galeon_panel_mode)
	{
		/* build panel and run special main loop */
		panel_main ();
	}
	else
#endif
	{
		/* enter the main GTK event processing loop */
#ifdef ENABLE_NAUTILUS_VIEW	       
		bonobo_main ();
#else
		gtk_main ();
#endif
	}

	/* absolutely no more mozembeds - this happens after the
           mainloop has returned to make sure that all timers have
           expired and everything has been handled. */
	gtk_moz_embed_pop_startup ();

	eel_gconf_monitor_remove ("/apps/galeon");
	eel_gconf_monitor_remove ("/apps/nautilus/preferences");

	/* exit cleanly */
	return 0;
}

/**
 * translate_url_arguments: gather URL arguments and expand them fully
 * with realpath if they're filenames
 */
static gint
translate_url_arguments (poptContext context, gchar ***urls)
{
	gchar buffer[PATH_MAX];
	gchar **args;
	gint i, n;

	/* any context remaining? */
	if (context == NULL)
	{
		*urls = NULL;
		return 0;
	}

	/* get the args and check */
	args = (gchar **) poptGetArgs (context);
	if (args == NULL)
	{
		poptFreeContext (context);
		*urls = NULL;
		return 0;
	}

	/* count args */
	for (n = 0; args[n] != NULL; n++)
		/* nothing */;

	/* allocate pointer array */
	*urls = g_new0 (gchar *, n + 1);
	
	/* translate each one */
	for (i = 0; i < n; i++)
	{
		/* try to expand as files */
		if (realpath (args[i], buffer) != NULL)
		{
			(*urls)[i] = g_strconcat ("file://", buffer, NULL);
		}
		else
		{
			(*urls)[i] = g_strdup (args[i]);
		}
	}
	poptFreeContext (context);
	(*urls)[i] = NULL;

	/* return the number of urls */
	return n;
}

/**
 * corba_init: initialise Gnorba (for gtm) and OAF (for remote
 * Galeon startups).
 */
static CORBA_ORB
corba_init (int *argc, char *argv[])
{
	CORBA_ORB orb;

	/* catch exceptions */
	CORBA_exception_init (&corba_env);

	/* initialise OAF */
	orb = oaf_init (*argc, argv);

#ifdef ENABLE_GNOME_FILE_SELECTOR
        if ( !bonobo_init (orb, NULL, NULL)) {
                g_warning ("Cannot initialize Bonobo component framework !");
        }
#else
#ifdef ENABLE_NAUTILUS_VIEW
        if ( !bonobo_init (orb, NULL, NULL)) {
                g_warning ("Cannot initialize Bonobo component framework !");
        }
#endif
#endif /* !ENABLE_GNOME_FILE_SELECTOR */
        
	/* return handle */
	return orb;
}

/**
 * new_view_on_running_shell: 
 *
 * Create new view on existing shell. 
 * Returns false if no shell is currently running.
 */
static gboolean 
new_view_on_running_shell (gint n_urls, gchar **url)
{
	CORBA_Object client;
	ViewState viewstate;
	gboolean result = TRUE;
	gint i;
	gchar *default_url;
	
	/* compute target viewstate */
	viewstate = get_viewstate ();

	/* try to find an existing shell to run on */
	client = oaf_activate ("repo_ids.has('IDL:galeon/browser:1.0')", NULL,
			       OAF_FLAG_EXISTING_ONLY, NULL, &corba_env);

	/* check if we found one */
	if (CORBA_Object_is_nil (client, &corba_env))
	{
		return FALSE;
	}

	/* load the session if requested */
	if (session_filename)
	{
		Galeon_Browser_loadSession (client, session_filename,
					    &corba_env);
		return TRUE;
	}

	/* if found and we're given a bookmark to add... */
	if (bookmark_url != NULL)
	{
		Galeon_Browser_addTempBookmark (client, bookmark_url,
						&corba_env);
		return TRUE;
	}

	if (close_option || quit_option)
	{
		Galeon_Browser_quit (client, quit_option, &corba_env);
		return TRUE;
	}

	/* otherwise we'll be opening windows, so warn */
	g_message (_("Galeon already running, using existing process"));

	/* provided with urls? */
	if (n_urls == 0)
	{
		default_url = embed_get_default_url (NULL);
		/* no, open a default window */
		result = Galeon_Browser_loadurl (client, default_url, viewstate,
						 geometry_string ?
						 geometry_string : "",
						 open_fullscreen,
						 &corba_env);
		g_free (default_url);
	}

	/* open all of the urls */
	for (i = 0; i < n_urls; i++)
	{
		result = Galeon_Browser_loadurl (client, url[i], viewstate, 
						 geometry_string ?
						 geometry_string : "",
						 open_fullscreen,
						 &corba_env);
		if (!result)
		{
			break;
		}
	}

#if 0
	/* check */
	if (result)
	{
		GtkWidget *dialog = gnome_error_dialog 
			(_("Galeon remote startup failed.\n\n"
			   "Try exiting other OAF applications\n"
			   "(e.g. Evolution and Nautilus), and running\n"
			   "\"oaf-slay\" from the command line."));
			 
		gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
	}
#endif

	/* done */
	return TRUE;
}

/**
 * new_shell: create a new CORBA shell
 */
static void 
new_shell (CORBA_ORB orb)
{
	PortableServer_POA poa;
	gchar *reg_id;

	/* make a client */
	poa = (PortableServer_POA) CORBA_ORB_resolve_initial_references 
		(orb, "RootPOA", &corba_env);
	galeon_automation_client = impl_Galeon_Browser__create (poa,
								&corba_env);

	/* check */
	if (galeon_automation_client == NULL)
	{
		g_warning (_("Galeon object already active, "
			     "server registration failed\n"));
		return;
	}

	/* register server with oaf */
	reg_id = GALEON_AUTOMATION_IID;
	oaf_active_server_register (reg_id, galeon_automation_client);

	PortableServer_POAManager_activate
		(PortableServer_POA__get_the_POAManager (poa, &corba_env), 
		 &corba_env);

}

/**
 * galeon_init: initialise Galeon and the services on which it depends
 */
static gboolean
galeon_init (int argc, char *argv[])
{
	static gchar *restart_argv[] = { "galeon", "--server", NULL };
	GnomeClient *session_client;
	gboolean session_recovery;
	gboolean vfs_success;
	gboolean new_user;

	/* preload the prefs */
	/* it also enables notifiers support */
	eel_gconf_monitor_add ("/apps/galeon");
	eel_gconf_monitor_add ("/apps/nautilus/preferences");

	/* initialise GLADE */
	glade_gnome_init ();

	/* initialise embed */
	if (!pushed_startup)
	{
		embed_startup_init ();
	}

	/* initialise GNOME VFS */
	vfs_success = gnome_vfs_init ();
	g_assert (vfs_success);

	/* check if this is the first time this user has run Galeon */
	new_user = newuser_check ();

	/* initialise Galeon itself */
	misc_gui_colors_init ();
	prefs_load ();
	history_init ();
	bookmarks_init ();
	downloader_dialog_init ();
	favicon_key_files_init ();
	gestures_init ();
	js_console_init ();
	session_history_load ();
#if ENABLE_NAUTILUS_VIEW
	galeon_init_factory ();
#endif

	/* resume a crashed session, if any */
	session_recovery = session_autoresume ();

	/* attach session-manager callbacks */
	session_client = gnome_master_client ();
	gnome_client_set_restart_command 
		(session_client, galeon_server_mode ? 2 : 1, restart_argv);
	gnome_client_set_restart_style (GNOME_CLIENT (session_client),
					GNOME_RESTART_IF_RUNNING);
	gtk_signal_connect (GTK_OBJECT (session_client), "save_yourself",
			    (GtkSignalFunc)client_save_yourself_cb, NULL);
	gtk_signal_connect (GTK_OBJECT (session_client), "die",
			    (GtkSignalFunc)client_die_cb, NULL);

	/* if this flag doesn't get cleared, then Galeon crashed */
	eel_gconf_set_integer (CONF_CRASH_CRASHED, TRUE);

	/* return saying whether or not to override default window creation */
	return (new_user || session_recovery);
}

/**
 * create_default_browser: called at the end to make sure at least one
 * browser is created in normal modes.
 */
static void
create_default_browser (void)
{
	GaleonEmbed *embed;
	GaleonWindow *window;
	gint xpos, ypos;
	gint width, height;
	
	/* don't bother if there's already at least one */
	if (g_list_length (all_embeds) != 0)
	{
		return;
	}

	/* create an empty window */
	window = window_create (DEFAULT_CHROME, NULL);
	
	/* collect geometry information */
	if (gnome_parse_geometry (geometry_string, &xpos, &ypos,
				       &width, &height))
	{
		/* setup size and show it */
		gtk_window_set_default_size (GTK_WINDOW (window->wmain), 
					     width, height);
		gtk_widget_set_uposition (GTK_WIDGET (window->wmain),
					  xpos, ypos);
		window->set_size = TRUE;
	}
	
	/* build the default embed inside it */
	embed = embed_create_in_window (window, NULL, NULL,
					EMBED_CREATE_RAISE_WINDOW |
					EMBED_CREATE_LOAD_DEFAULT_URL);
	
	/* show as fullscreen? */
	/* THE WINDOW NEEDS TO BE VISIBLE HERE */
	if (open_fullscreen)
	{
		window_set_fullscreen_mode (window, TRUE);
	}
	
	window_grab_location_focus (window);
}

/**
 * get_viewstate:
 *
 * Determine the viewstate. A viewstate defines how the URI should be opened.
 */
static inline ViewState
get_viewstate (void)
{
	ViewState result;
	gboolean tabbed_mode;

	tabbed_mode = eel_gconf_get_boolean (CONF_TABS_TABBED);

	/* figure out whether we should open in an existing browser, a
	 * window, or a tab, and whether this was explicitly requested or
	 * just defaulted to */
	if (open_in_existing)
		result = VIEWSTATE_EXISTING;
	else if (open_in_new_tab)
		result = VIEWSTATE_TAB_FORCE;
	else if (open_in_new_window || !tabbed_mode)
		result = VIEWSTATE_WINDOW;
	else
		result = VIEWSTATE_TAB_DEFAULT;

	/* see if noraise was requested */
	if (result == VIEWSTATE_TAB_FORCE && noraise)
		result = VIEWSTATE_TAB_FORCE_NORAISE;
	else if (result == VIEWSTATE_TAB_DEFAULT && noraise)
		result = VIEWSTATE_TAB_DEFAULT_NORAISE;

	return result;
}

#ifdef ENABLE_NAUTILUS_VIEW
static void 
galeon_init_factory (void)
{
	BonoboGenericFactory *galeon_nautilusview_factory;
	galeon_nautilusview_factory = bonobo_generic_factory_new_multi
	       	(GALEON_NAUTILUS_VIEW_IID,
	         galeon_nautilus_view_new, NULL);
	if (!BONOBO_IS_GENERIC_FACTORY (galeon_nautilusview_factory))
		g_error ("Couldn't create the factory!");
}

static BonoboObject *
galeon_nautilus_view_new (BonoboGenericFactory *factory, const char *id, 
			  gpointer data)
{
	if (galeon_exiting) 
		return NULL;
	else
		return galeon_nautilus_view_new_component ();
}

#endif

void 
galeon_exit (gboolean return_from_main_loop, gboolean unset_crashed_flag)
{
	extern GList *mozilla_notifiers;
	
	/**
	 * First of all, unregister with OAF
	 */
	oaf_active_server_unregister (GALEON_AUTOMATION_IID, 
				      galeon_automation_client);
#ifdef ENABLE_NAUTILUS_VIEW
/* TODO: I can't find how to get the CORBA object from a BonoboGenericFactory
        oaf_active_server_unregister (GALEON_NAUTILUS_VIEW_IID,
	                              [galeon_nautilusview_factory]);
*/
#endif
	galeon_exiting = TRUE;

	if (unset_crashed_flag)
		eel_gconf_set_integer (CONF_CRASH_CRASHED, FALSE);

	/* unset the this-session-only download dir pref */
	eel_gconf_set_string (CONF_STATE_LAST_DOWNLOAD_DIR, "");

	/* shut down galeon subsystems */
	prefs_destroy (); /* this will sync prefs if dialog is open */
	mime_db_shutdown ();
	bookmarks_exit ();
	history_exit ();
	mozilla_save_prefs ();
	favicon_key_files_close ();
	js_console_exit ();
	session_history_save ();

	/* check... */
	g_assert (g_list_length (all_embeds) == 0);
	g_assert (g_list_length (all_windows) == 0);

	/* shut down GNOME subsystems */
	gnome_vfs_shutdown ();

	/* remove notifiers for mozilla prefs */
	galeon_notification_remove (&mozilla_notifiers);

	/* return from the main loop if requested */
	if (return_from_main_loop) 
		gtk_main_quit();
	else
		/* otherwise, just exit */
		gtk_exit (0);
}

/**
 * client_save_yourself_cb: called during window manager session shutdown,
 * we use our internal exit-with-session functionality to restore next
 * time the user starts Galeon.
 */
gboolean
client_save_yourself_cb (GnomeClient *client, gint phase, 
			 GnomeSaveStyle save_style, gint shutdown,
			 GnomeInteractStyle interact_style, gint fast)
{
	/* save session */
	session_save ();

	/* we're done */
	return TRUE;
}

/**
 * client_die_cb: called during session shutdown.  merely causes galeon to
 * exit cleanly.
 */
void
client_die_cb (GnomeClient *client, gpointer data)
{
	galeon_exit (TRUE, TRUE);
}
