/*
 *  Copyright (C) 2001 Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __bookmarks_editor_h
#define __bookmarks_editor_h

#include <gtk/gtkclist.h>
#include <gtk/gtkctree.h>
#include "bookmarks.h"


typedef struct _BookmarksEditorControls  BookmarksEditorControls;

/**
 * BookmarksSearchIn: Search in bookmarks "search in"
 */
typedef enum {
	BOOKMARKS_SEARCH_IN_NAME,
	BOOKMARKS_SEARCH_IN_NICKS,
	BOOKMARKS_SEARCH_IN_NOTES,
	BOOKMARKS_SEARCH_IN_URL
} BookmarksSearchIn;

/**
 * BookmarksSearchMethod: Search in bookmarks search method
 */
typedef enum {
	BOOKMARKS_SEARCH_METHOD_CONTAINS,
	BOOKMARKS_SEARCH_METHOD_STARTS_WITH,
	BOOKMARKS_SEARCH_METHOD_ENDS_WITH,
	BOOKMARKS_SEARCH_METHOD_IS,
	BOOKMARKS_SEARCH_METHOD_IS_NOT,
	BOOKMARKS_SEARCH_METHOD_DOES_NOT_CONTAIN
} BookmarksSearchMethod;

/**
 * BookmarksFindDialog: Find in bookmarks dialog
 */
typedef struct {
	/* widgets */
	GtkWidget *dialog;
	GtkWidget *prev_button;
	GtkWidget *next_button;
	GtkWidget *criterium1_optionmenu;
	GtkWidget *criterium2_optionmenu;
	GtkWidget *gnomeentry;
	GtkWidget *entry;

	/* data */
	gchar *str;
	BookmarksSearchIn search_in;
	BookmarksSearchMethod search_method;
	BookmarksEditorControls *controls;
} BookmarksFindDialog;

/**
 * BookmarksEditorControlsType: types of bookmark editor dialogs
 */
typedef enum
{
	STANDARD_BOOKMARKS_EDITOR,
	DOCKED_BOOKMARKS_EDITOR,
	REMOVE_BOOKMARKS_EDITOR
} BookmarksEditorControlsType;

/**
 * BookmarksEditorControls: representation of a bookmarks editor.
 */
struct _BookmarksEditorControls
{
	BookmarksEditorControlsType type;

	/* the root folder being edited here */
	BookmarkItem *root_bookmark;

	/* the widgets of the editor */
	GtkWidget *dialog;
	GtkWidget *menubar;
	GtkWidget *toolbar;
	GtkWidget *menubar_dock;
	GtkWidget *toolbar_dock;
	GtkWidget *ctree;
	GtkWidget *vpane;
	GtkWidget *edit_frame;
	GtkWidget *name_entry;
	GtkWidget *url_entry;
	GtkWidget *smarturl_entry;
	GtkWidget *nick_entry;
	GtkWidget *pixmap_entry;
	GtkWidget *pixmap_file_entry;
	GtkWidget *create_toolbar_toggle;
	GtkWidget *create_context_menu_toggle;
	GtkWidget *notes_text;
	GtkWidget *search_entry;
	GtkWidget *time_added_entry;
	GtkWidget *time_modified_entry;
	GtkWidget *time_visited_entry;

	BookmarkItem *selection;    /* The current selected bookmark */
	BookmarkItem *last_pressed; /* For dnd */
	gboolean lock_edit_controls; /* Don't update bookmark when editing 
				      * fields. Set to TRUE to avoid recursion
				      */
	/* widgets we need to update at various stages*/
	GtkWidget *edit_menuitem;
	GtkWidget *dock_menuitem;
	GtkWidget *edit_button;

	GtkWidget *set_default_folder_menuitem;
	
	GaleonWindow *window;

	/* the clipboard data owned by this editor, if any */
	gchar *clipboard;

	/* the position in the ctree of the search */
	GList *search_position;

	/* find dialog */
	BookmarksFindDialog *find_dialog;
};


/**
 * Pair of controls / tree_item for a bookmark
 */
struct _BookmarkTreeItem
{
	BookmarksEditorControls *controls;
	GtkCTreeNode *tree_item; 	
};



/**
 * Functions
 */

gint bookmarks_move_bookmark (BookmarksEditorControls *controls,
			      BookmarkItem *b, int where);

void bookmarks_editor_place_tree_item (BookmarksEditorControls *controls,
				       BookmarkItem *b);
void bookmarks_editor_place_tree_items (BookmarkItem *b);
void bookmarks_editor_update_tree_item (BookmarksEditorControls *controls,
					BookmarkItem *b);
void bookmarks_editor_update_tree_items (BookmarkItem *b);
void bookmarks_editor_remove_tree_item (BookmarksEditorControls *controls,
					BookmarkItem *b);
void bookmarks_editor_remove_tree_items (BookmarkItem *b);
void bookmarks_editor_select_bookmark (BookmarksEditorControls *controls,
				       BookmarkItem *b);

BookmarksEditorControls *bookmarks_editor_show_dialog (GaleonWindow *window);
void bookmarks_editor_hide_dialog (BookmarksEditorControls *controls);
BookmarksEditorControls *bookmarks_editor_show_dock (GaleonWindow *window);

GtkCTreeNode *bookmarks_get_tree_item (BookmarkItem *b,
				       BookmarksEditorControls *c);
void bookmarks_set_tree_item (BookmarkItem *b, BookmarksEditorControls *c,
			      GtkCTreeNode *n);
void bookmarks_create_edit_items (GtkMenu *popup,
				  BookmarksEditorControls *controls);
void bookmarks_editor_set_root (BookmarksEditorControls *controls, 
		BookmarkItem *b);
void bookmarks_editors_refresh (void);
void bookmarks_editor_freeze_all (void);
void bookmarks_editor_thaw_all (void);

gboolean bookmarks_editor_search (BookmarksEditorControls *controls, 
				  gchar *str,
			          BookmarksSearchIn search_in,
			          BookmarksSearchMethod method,
			          gboolean backwards, gboolean wrap);
gboolean bookmarks_editor_search_matches (BookmarkItem *b, gchar *str,
					  BookmarksSearchIn search_in,
					  BookmarksSearchMethod search_method);
void bookmarks_editor_popup_menu (BookmarksEditorControls *controls,
				  GtkWidget *w);
BookmarksEditorControls *bookmarks_editor_remove_bookmarks_window (GList
								   *bookmarks);
void bookmarks_editor_remove_bookmarks_window_destroy (BookmarksEditorControls
						       *controls);



#endif
