/*
 *  Copyright (C) 2000 Nate Case 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Galeon CORBA interface implementation code */

#include "galeon.h"
#include "GaleonAutomation.h"
#include "embed.h"
#include "bookmarks.h"
#include "prefs.h"
#include "window.h"
#include "session.h"
#include "misc_gui.h"

#include <math.h>
#include <stdlib.h>
#include <gtkmozembed.h>
#include <libgnome/gnome-defs.h>
#include <libgnomeui/gnome-geometry.h>
#include <libgnomeui/gnome-winhints.h>

typedef struct
{
	POA_Galeon_Browser servant;
	PortableServer_POA poa;
} impl_POA_Galeon_Browser;


Galeon_Browser impl_Galeon_Browser__create(	PortableServer_POA poa,
						CORBA_Environment * ev);


/* interface method prototypes */

static CORBA_boolean
impl_Galeon_Browser_loadurl(impl_POA_Galeon_Browser * servant,
		 	     CORBA_char * url,
			     Galeon_ViewState viewstate,
			     CORBA_char * geometry,
			     CORBA_boolean fullscreen,
			     CORBA_Environment * ev);

static CORBA_boolean
impl_Galeon_Browser_configure(impl_POA_Galeon_Browser * servant,
				CORBA_Environment * ev);

static CORBA_boolean
impl_Galeon_Browser_addTempBookmark (impl_POA_Galeon_Browser * servant,
				     CORBA_char * url,
				     CORBA_Environment * ev);

static CORBA_boolean
impl_Galeon_Browser_quit(impl_POA_Galeon_Browser * servant,
			 CORBA_boolean disableServer, CORBA_Environment * ev);

static CORBA_boolean
impl_Galeon_Browser_loadSession (impl_POA_Galeon_Browser * servant,
				 CORBA_char * filename,
				 CORBA_Environment * ev);

/*** epv structures ***/
static PortableServer_ServantBase__epv impl_Galeon_Browser_base_epv =
{
	NULL,		/* _private data */
	NULL,		/* finalize routine */
	NULL,		/* default_POA routine */
};

static POA_Galeon_Browser__epv impl_Galeon_Browser_epv =
{
	NULL,		/* _private */
	(gpointer) & impl_Galeon_Browser_loadurl,
	(gpointer) & impl_Galeon_Browser_configure,
	(gpointer) & impl_Galeon_Browser_addTempBookmark,
	(gpointer) & impl_Galeon_Browser_quit,
	(gpointer) & impl_Galeon_Browser_loadSession,
};

/*** vepv structures ***/
static POA_Galeon_Browser__vepv impl_Galeon_Browser_vepv =
{
	&impl_Galeon_Browser_base_epv,
	&impl_Galeon_Browser_epv,
};


/*** Stub implementations ***/
Galeon_Browser 
impl_Galeon_Browser__create(PortableServer_POA poa, CORBA_Environment * ev)
{
	Galeon_Browser retval;
	impl_POA_Galeon_Browser *newservant;
	PortableServer_ObjectId *objid;

	newservant = g_new0(impl_POA_Galeon_Browser, 1);
	newservant->servant.vepv = &impl_Galeon_Browser_vepv;
	newservant->poa = poa;

	POA_Galeon_Browser__init((PortableServer_Servant) newservant, ev);
	objid = PortableServer_POA_activate_object(poa, newservant, ev);
	CORBA_free(objid);

	retval = PortableServer_POA_servant_to_reference(poa, newservant, ev);

	return retval;
}

static CORBA_boolean
impl_Galeon_Browser_loadurl(impl_POA_Galeon_Browser * servant,
			  CORBA_char * url,
			  Galeon_ViewState viewstate,
			  CORBA_char * geometry,
			  CORBA_boolean fullscreen,
			  CORBA_Environment * ev)
{
	gint workspace;
	/* make min. workspace distances start out with huge numbers */
	gint window_dist = 999999, icon_window_dist = 999999;
	GaleonEmbed *prev_embed = NULL, *icon_prev_embed = NULL;
	GaleonWindow *window = NULL;
	GList *li;
	gint xpos, ypos, width, height;

	/* don't let anybody create new embeds while exiting */
	if (galeon_exiting) return FALSE;

#ifdef DEBUG_CORBA	
	g_print("CORBA: loadurl(%s, %d, %s, %d)\n", url, viewstate, geometry,
		fullscreen);
#endif
	/* get the current workspace number */
	workspace = gnome_win_hints_get_current_workspace ();

	/* find the closest iconified and non-iconified windows */
	for (li = all_windows; li != NULL; li = g_list_next (li))
	{
		window = li->data;
		return_val_if_not_window (window, FALSE);

		/* if the window is visible */
		if (misc_gui_gdk_window_is_visible (window->wmain->window))
		{
			if (window->active_embed != NULL &&
			    !(window->active_embed->is_chrome) &&
			    (prev_embed == NULL ||
			     abs (gnome_win_hints_get_workspace (window->wmain)
				  - workspace) < window_dist))
			{
				prev_embed = window->active_embed;
				window_dist =
					abs (gnome_win_hints_get_workspace (
							window->wmain) -
					     workspace);
			}
		}
		/* otherwise, it's iconified... */
		else
		{
			if (window->active_embed != NULL &&
			    !(window->active_embed->is_chrome) &&
			    (prev_embed == NULL ||
			     abs (gnome_win_hints_get_workspace (window->wmain)
				  - workspace) < icon_window_dist))
			{
				icon_prev_embed = window->active_embed;
				icon_window_dist =
					abs (gnome_win_hints_get_workspace (
							window->wmain) -
					     workspace);
			}
		}
	}

	/* if we've defaulted to opening in a tab, rather than being
	 * requested by the user... */
	if ((viewstate == Galeon_TAB_DEFAULT ||
	     viewstate == Galeon_TAB_DEFAULT_NORAISE))
	{
		/* if there's no visible window on the current desktop,
		 * open in a new window instead */
		if (window_dist)
		{
			viewstate = Galeon_WINDOW;
		}
	}
	/* otherwise, if the user specifically requested opening in a tab
	 * or an existing browser... */
	else if (viewstate != Galeon_WINDOW)
	{
		/* if there's no visible embed, but there's an iconified
		 * one, open in the iconified embed */
		if (!prev_embed && icon_prev_embed)
		{
			prev_embed = icon_prev_embed;
		}
		/* otherwise, if we can't find _any_ embed, open in a new
		 * window */
		else if (!prev_embed && !icon_prev_embed)
		{
			viewstate = Galeon_WINDOW;
		}
	}

	/* handle the different viewstates */
	switch (viewstate)
	{
		/* open in existing browser */
		case Galeon_EXISTING:
			embed_load_url (prev_embed, url);
			break;

		/* open in new window */
		case Galeon_WINDOW:
			window = window_create (DEFAULT_CHROME, NULL);

			/* parse geometry string */
			if (gnome_parse_geometry (geometry, &xpos, &ypos,
		 			               &width, &height))
			{
				gtk_window_set_default_size
					(GTK_WINDOW (window->wmain), 
					 width, height);
				gtk_widget_set_uposition
					(GTK_WIDGET (window->wmain),
					 xpos, ypos);

				window->set_size = TRUE;
			}

			embed_create_in_window (window, prev_embed, url, 0);

			window_grab_location_focus (window);

			break;

		/* open in tab */
		case Galeon_TAB_FORCE:
		case Galeon_TAB_DEFAULT:
			embed_create_after_embed (prev_embed, FALSE, url,
						  EMBED_CREATE_FORCE_JUMP |
						  EMBED_CREATE_RAISE_WINDOW);
			break;

		/* open in tab, and don't raise the window */
		case Galeon_TAB_FORCE_NORAISE:
		case Galeon_TAB_DEFAULT_NORAISE:
			embed_create_after_embed (prev_embed, FALSE, url,
						  EMBED_CREATE_FORCE_JUMP);
			break;
	}	 

	/* go into fullscreen mode if requested */
	if (window && fullscreen)
	{
		window_set_fullscreen_mode (window, TRUE);
	}

	return TRUE;
}

static CORBA_boolean
impl_Galeon_Browser_configure(impl_POA_Galeon_Browser * servant,
				CORBA_Environment * ev)
{
	CORBA_boolean retval = TRUE;

#ifdef DEBUG_CORBA
	g_print("CORBA: configure()\n");
#endif

	return retval;
}

static CORBA_boolean
impl_Galeon_Browser_addTempBookmark (impl_POA_Galeon_Browser * servant,
				     CORBA_char * url,
				     CORBA_Environment * ev)
{
	CORBA_boolean retval = TRUE;

	/* adding bookmarks while exiting will probably crash */
	if (galeon_exiting) return FALSE;

	bookmarks_add_bookmark (NULL, url, NULL, NULL,
				BOOKMARK_ADD_NEVER_PROMPT_TITLE);
	
	return retval;
}

static CORBA_boolean
impl_Galeon_Browser_quit (impl_POA_Galeon_Browser * servant,
			  CORBA_boolean disableServer, CORBA_Environment * ev)
{
	CORBA_boolean retval = TRUE;
	GList *l;
	
	
	/* nothing to do when we are already exiting */
	if (galeon_exiting) return FALSE;
	
	if (disableServer == TRUE) 
	{
		galeon_server_mode = FALSE;
	}
	
	if (all_windows == NULL)
	{
		/* make sure that we quit */
		if (!galeon_server_mode)
		{
			session_quit (FALSE);
			galeon_exit (TRUE, TRUE);
		}
	}
	else
	{
		/* close all windows */
		l = g_list_copy (all_windows);
		g_list_foreach (l, (GFunc) window_close, NULL);
		g_list_free (l);
	}

	return retval;
}

static CORBA_boolean
impl_Galeon_Browser_loadSession (impl_POA_Galeon_Browser * servant,
				 CORBA_char * filename,
				 CORBA_Environment * ev)
{
	CORBA_boolean retval = TRUE;


	/* don't let anybody create new embeds while exiting */
	if (galeon_exiting) return FALSE;

	if (session_open_confirmation ())
		session_load_from (filename, TRUE);
	
	return retval;
}
