/* -*- Mode: C++; c-file-style: "stroustrup"; indent-tabs-mode: nil -*- */
/*
 * DbAdmin.cc
 *   Class for database interaction from the CGI and the
 *   command line admin tool.
 *
 * $Id: DbAdmin.cc,v 1.25 2002/03/30 22:21:17 benoit Exp $
 *
 * Copyright (c) 2000 Remi Lefebvre <remi@dhis.net>
 *                and Benoit Joly <benoit@dhis.net>
 *
 * Licensed under the GPLv2
 */

#include <iostream>

extern "C" {
#include <cgi.h>
}

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "DbPsql.h"
#include "DnsBind.h"

#include "DbAdmin.h"

DbAdmin::DbAdmin(Logger *log, char *dbname, char *dbuser, char *dbpass,
                 char *domain, char *banned, char *reserved)
{
    this->log = log;
    this->db = new DbPsql(log, dbname, dbuser, dbpass);
    this->dns = new DnsBind(log);

    this->manager = new DdtManager(log, db, dns);

    this->check = new Validate(banned, reserved);
}

DbAdmin::~DbAdmin()
{
    if (log) delete log;
    if (db) delete db;
    if (dns) delete dns;

    if (manager) delete manager;

    if (check) delete check;
}

/**
 * verify if password matches the one from database and fills
 * @param id user account id of the account
 * @param Pass cleartext password to verify for account 'id'
 * @param PassMd5 md5 hash of Pass. Function will first check it up. If 
 *        it doesn't matches, it tries the cleartext pass.
 */
int DbAdmin::checkPasswd(int id, char *Pass, char *PassMd5)
{
    unsigned char Digest[MD5_DIGEST_LENGTH];
    char dbPass[MD5_DIGEST_LENGTH*2 + 1];
 
    if (manager->fetchAccountInfo(id, "adminPassword", dbPass,
                                  sizeof (dbPass)) == false)
    {
        return -1;
    }

    // if aPass is already hashed and matches
    if (strncasecmp(dbPass, PassMd5, (MD5_DIGEST_LENGTH*2 + 1)) == 0)
    {
        manager->markAlive(id);
        return 0;
    }

    if (Pass == NULL)
    {
        return -1;
    }
    
    // hash provided password
    MD5((unsigned char *) Pass, strlen (Pass), Digest);
    for (int i = 0; i < MD5_DIGEST_LENGTH; i++)
    {
        sprintf (&(PassMd5[i*2]), "%02x", Digest[i]);
    }

    // if new hash matches
    if (strncasecmp(dbPass, PassMd5, (MD5_DIGEST_LENGTH*2 + 1)) == 0)
    {
        manager->markAlive(id);
        return 0;
    }
    return -1;
}

int DbAdmin::hashPassword (const char *Pass, const char *PassConfirm, 
                           char *PassMd5)
{
    unsigned char Digest[MD5_DIGEST_LENGTH];

    // check if passwords matches
    if (Pass != NULL && PassConfirm != NULL)
    {
        if (strcmp(Pass, PassConfirm) != 0)
        {
            return -1; // Passwords don't match
        }
    }
    else // NULL password somewhere
    {
        return -1;
    }

    // hash passwords
    MD5 ((unsigned char *) Pass, strlen (Pass), Digest);
	
    for (int i = 0; i < MD5_DIGEST_LENGTH; i++)
    {
        sprintf (&(PassMd5[i*2]), "%02x", Digest[i]);
    }
    return 0;
}

// FIXME: implement that
int DbAdmin::dumpInfo (int id)
{
    return -1;
//    return (manager->dumpInfo (id));
}


/**
 * add a new account and do necessary validity checks
 * @param newAccount useraccount structure containing requested info for
 *                   the account
 * @param baseDomain base domain for the domain (e.g. .ddts.net)
 */
bool DbAdmin::addAccount(userAccount *newAccount, const char *baseDomain)
{
    bool errorOccured = false;
    
    // first, we check for the validity of the provided fields

    if (manager->findUserAccountIdFromFqdn(newAccount->fqdn) != -1)
    {
        cout << "FQDN is already taken.<br>\n";
        errorOccured = true;
    }

    if (check->isValidDomain(newAccount->fqdn, baseDomain) == false)
    {
        cout << "FQDN is not valid.<br>\n";
        errorOccured = true;
    }

    if (check->isThirdLevelDomain(newAccount->fqdn) == false)
    {
        cout << "FQDN is not a third level domain.<br>\n";
        errorOccured = true;
    }

    if (check->isBannedDomain(newAccount->fqdn))
    {
        cout << "FQDN is banned or reserved.<br>\n";
        errorOccured = true;
    }
    
    if (check->name(newAccount->contactName) == false)
    {
        cout << "Invalid Name Field.<br>\n";
        errorOccured = true;
    }
    
    if (check->email(newAccount->contactEmail, baseDomain) == false)
    {
        cout << "Invalid Email Field.<br>\n";
        errorOccured = true;
    }

    if (errorOccured == true)
    {
        return !errorOccured;
    }

    // here we add the account
    if ((newAccount->userAccountId = manager->addUserAccount(newAccount)) 
        == -1)
    {
        return false;
    }

    log->debug ("%s added by %s", newAccount->fqdn, newAccount->contactName);

    return true;
}

/* why is that there ? might be deleted i think ... CHECKME */
bool DbAdmin::getAcctInfo(int id, userAccount *account)
{
    manager->getAcctInfo(id, account);

    if (account->hostStatus == STATIC)
    {
        struct in_addr ip;
        ip.s_addr = manager->fetchAcctAddr(id);
        if (ip.s_addr == 1)
        {
            return false;
        }
        
        strncpy(account->ipAddress, inet_ntoa(ip), 
                sizeof(account->ipAddress));
    }
    else
    {
        strncpy (account->ipAddress, "",
                 sizeof (account->ipAddress)); // FIXME
    }

    return true;
}

bool DbAdmin::updateAccount(userAccount *acct, const char *baseDomain)
{
    bool errorOccured = false;
    
    // first, we check for the validity of the provided fields
    if (check->name(acct->contactName) == false)
    {
        cout << "Invalid Name Field.<br>\n";
        errorOccured = true;
    }
    
    if (check->email(acct->contactEmail, baseDomain) == false)
    {
        cout << "Invalid Email Field.<br>\n";
        errorOccured = true;
    }

    if (errorOccured == true)
    {
        return false;
    }

    // update database
    if (manager->modUserAccount(acct->userAccountId, "contactName", 
                                acct->contactName) == -1)
    {
        cout << "Failed to update contact name.";
        return false;
    }

    if (manager->modUserAccount(acct->userAccountId, "contactEmail", 
                                acct->contactEmail) == -1)
    {
        cout << "Failed to update contact email.";
        return false;
    }
    
    if (manager->modUserAccount(acct->userAccountId, "arch", 
                                acct->arch) == -1)
    {
        cout << "Failed to update arch.";
        return false;
    }
    
    if (manager->modUserAccount(acct->userAccountId, "os", acct->os) == -1)
    {
        cout << "Failed to update OS.";
        return false;
    }

    if (acct->updatePassword[0] != '\0')
    {
        if (manager->modUserAccount(acct->userAccountId, "updatePassword",
                                    acct->updatePassword) == -1)
        {
            cout << "Failed to update update password.";
            return false;
        }
    }

    if (acct->adminPassword[0] != '\0')
    {
        if (manager->modUserAccount (acct->userAccountId, "adminPassword",
                                     acct->adminPassword) == -1)
        {
            cout << "Failed to update admin password.";
            return false;
        }
        cout << "Admin Password changed. Please re-login to the CGI.<br>\n";
    }

    char stat[2];
    unsigned long ip;
    ip = inet_addr(acct->ipAddress);
    
    sprintf (stat, "%d", acct->hostStatus);
    if (manager->modUserAccount(acct->userAccountId, "hostStatus", 
                                stat) == -1)
    {
        cout << "Failed to update hoststatus.";
        return false;
    }

    return true;
}

/* completely remove the account and all its dns records */
int DbAdmin::removeAccount (int id)
{
    manager->delUserAccount(id);
    return 0;
}


int DbAdmin::addDnsRecord(int id, const char *dname, DnsRecordType type, 
                          const char *data, const char *baseDomain)
{
    userAccount accountInfo;
    char l_dname[256];
    char l_data[256];

    /* i think getAcctInfo from DdtManager is enough, CHECKME */
    if (getAcctInfo(id, &accountInfo) == 0)
    {
        cout << "can't find info for user " << id << "<br>\n";
        return -1;
    }

    if (type == MX)
    {
        cout << "Checking MX's validity.<br>\n";
        strncpy(l_dname, accountInfo.fqdn, sizeof(l_dname) - 1);
        strncpy(l_data, data, sizeof(l_data) - 1);
    }
    else if (type == CNAME)
    {
        cout << "Checking CNAME's validity.<br>\n";
        strncpy(l_dname, dname, sizeof(l_dname) - 1);
        if (data[0] == '\0')
            strncpy(l_data, accountInfo.fqdn, sizeof(l_data) - 1);
        else
            strncpy(l_data, data, sizeof(l_data) - 1);
    }

    if (check->recordAdd(manager, id, l_dname, type, l_data, baseDomain)
        == false)
    {
        cout << "Bad syntax. Check Failed.<br>\n";
        return -1;
    }

    cout << "Syntax is good. Trying to add record.<br>\n";
#ifdef DDT_DEBUG
    cout << "id: " << id << " dname: " << l_dname << " type: " << type 
         << " data: " << l_data << "<br>\n";
#endif
    
    manager->addDnsRecord(id, l_dname, type, l_data);
    return 0;
}

int DbAdmin::delDnsRecord(int id, const char *dname, DnsRecordType type,
                          const char *data)
{
    userAccount accountInfo;
    char l_dname[256];
    char l_data[256];

    /* i think getAcctInfo from DdtManager is enough, CHECKME */
    if (getAcctInfo(id, &accountInfo) == 0)
    {
        cout << "can't find info for user " << id << "<br>\n";
        return -1;
    }

    if (type == MX)
    {
        cout << "Checking MX's validity.<br>\n";
        strncpy(l_dname, accountInfo.fqdn, sizeof(l_dname) - 1);
        strncpy(l_data, data, sizeof(l_data) - 1);
    }
    else if (type == CNAME)
    {
        cout << "Checking CNAME's validity.<br>\n";
        strncpy(l_dname, dname, sizeof(l_dname) - 1);
        if (data[0] == '\0')
            strncpy(l_data, accountInfo.fqdn, sizeof(l_data) - 1);
        else
            strncpy(l_data, data, sizeof(l_data) - 1);
    }

    // FIXME: check validity of requests
    //if (check->recordDel (manager, id, dname, type, data) == -1)
    //{
    //   cout << "Bad syntax. Check Failed.<br>\n";
    //    return -1;
    //}

    cout << "Syntax good. Trying to remove record.<br>\n";
/*
    cout << "id: " << id << " dname: " << l_dname << " type: " << type 
         << " data: " << l_data << "<br>\n";
*/
    manager->delDnsRecord(id, l_dname, type, l_data);
    return 0;
}
