/*
 * Copyright (c) 1995 - 2000 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <xfs/xfs_locl.h>
#include <xfs/xfs_deb.h>
#include <xfs/xfs_fs.h>
#include <xfs/xfs_message.h>
#include <xfs/xfs_msg_locl.h>

RCSID("$Id: xfs_message.c,v 1.9.2.1 2001/01/30 02:11:37 lha Exp $");

int
xfs_message_installroot(int fd,
			struct xfs_message_installroot *message,
			u_int size)
{
  int error = 0;

  XFSDEB(XDEBMSG, ("xfs_message_installroot\n"));

  if (xfs[fd].root != 0)
    {
      printf("XFS PANIC Warning: xfs_message_installroot again\n");
      error = EBUSY;
    }
  else
    {
      xfs[fd].root = new_xfs_node(&xfs[fd], &message->node); /* VN_HOLD's */
      XNODE_TO_VNODE(xfs[fd].root)->v_flag |= VROOT;
    }
  return error;
}

int
xfs_message_installnode(int fd,
			struct xfs_message_installnode *message,
			u_int size)
{
  int error = 0;
  struct xfs_node *n, *dp;

  XFSDEB(XDEBMSG, ("xfs_message_installnode\n"));

  dp = xfs_node_find(&xfs[fd], &message->parent_handle);
  if (dp)
    {
      n = new_xfs_node(&xfs[fd], &message->node); /* VN_HOLD's */
      dnlc_remove (XNODE_TO_VNODE(dp), message->name);
      xfs_dnlc_enter(XNODE_TO_VNODE(dp), message->name, XNODE_TO_VNODE(n));
      VN_RELE(XNODE_TO_VNODE(n));
    }
  else
    {
      printf("XFS PANIC Warning: xfs_message_install could not find parent\n");
      error = ENOENT;
    }
  return error;
}

int
xfs_message_installattr(int fd,
			struct xfs_message_installattr *message,
			u_int size)
{
  int error = 0;
  struct xfs_node *t;

  XFSDEB(XDEBMSG, ("xfs_message_installattr\n"));

  t = xfs_node_find(&xfs[fd], &message->node.handle);
  if (t != 0)
    {
      t->tokens = message->node.tokens;
      xfs_attr2vattr(&message->node.attr, &t->attr, 0);
      bcopy((caddr_t)message->node.id,
	    (caddr_t)t->id, sizeof(t->id));
      bcopy((caddr_t)message->node.rights,
	    (caddr_t)t->rights, sizeof(t->rights));
      t->anonrights = message->node.anonrights;
    }
  else
    {
      XFSDEB(XDEBMSG, ("xfs_message_installattr: no such node\n"));
    }
  return error;
}

int
xfs_message_installdata(int fd,
			struct xfs_message_installdata *message,
			u_int size)
{
  struct xfs_node *t;
  struct vnode *vp;
  int error = 0;

  XFSDEB(XDEBMSG, ("xfs_message_installdata\n"));

  t = xfs_node_find(&xfs[fd], &message->node.handle);
  if (t != 0)
    {
      XFSDEB(XDEBMSG, ("cache_name = '%s'\n", message->cache_name));

#if IRIX_64
      VOP_LOOKUP(curprocp->p_cdir, message->cache_name, &vp,
		 NULL, 0, NULL, curprocp->p_cred, error);
#else
      error = VOP_LOOKUP(u.u_cdir, message->cache_name, &vp,
			 NULL, 0, NULL, u.u_cred);
#endif
      if (error == 0)
	{
	  if (DATA_FROM_XNODE(t))
	    {
	      VN_RELE(DATA_FROM_XNODE(t));
	    }
	  ASSERT(vp != NULL);
	  DATA_FROM_XNODE(t) = vp; /* VOP_LOOKUP does an implicit VN_HOLD? */
	  t->tokens = message->node.tokens;
	  xfs_attr2vattr(&message->node.attr, &t->attr, 1);
	  if (XNODE_TO_VNODE(t)->v_type == VDIR)
	      dnlc_purge_vp(XNODE_TO_VNODE(t));
	  bcopy((caddr_t)message->node.id,
		(caddr_t)t->id, sizeof(t->id));
	  bcopy((caddr_t)message->node.rights,
		(caddr_t)t->rights, sizeof(t->rights));
	  t->anonrights = message->node.anonrights;
	}
      else
	printf("XFS PANIC Warning: xfs_message_installdata failed to lookup cache file = %s, error = %d\n", message->cache_name, error);
    }
  else
    {
      printf("XFS PANIC Warning: xfs_message_installdata didn't find node!\n");
      error = ENOENT;
    }
  return error;
}

int
xfs_message_invalidnode(int fd,
			struct xfs_message_invalidnode *message,
			u_int size)
{
  int error = 0;
  struct xfs_node *t;
  
  XFSDEB(XDEBMSG, ("xfs_message_invalidnode\n"));

  t = xfs_node_find(&xfs[fd], &message->handle);
  if (t != 0)
    {
      /* XXX Really need to put back dirty data first. */
      if (DATA_FROM_XNODE(t))
	{
	  VN_RELE(DATA_FROM_XNODE(t));
	  DATA_FROM_XNODE(t) = (struct vnode *) 0;
	}
      XFS_TOKEN_CLEAR(t, ~0,
		      XFS_OPEN_MASK | XFS_ATTR_MASK |
		      XFS_DATA_MASK | XFS_LOCK_MASK);
      dnlc_purge_vp (XNODE_TO_VNODE(t));
    }
  else
    {
#if 0
      printf("XFS PANIC Warning: xfs_message_invalidnode didn't find node!\n");
#endif
      error = ENOENT;
    }
  return error;
}

int
xfs_message_updatefid(int fd,
		      struct xfs_message_updatefid * message,
		      u_int size)
{
    int error = 0;
    struct xfs_node *t;

    XFSDEB(XDEBMSG, ("xfs_message_updatefid (%d,%d,%d,%d)\n",
		     message->old_handle.a,
		     message->old_handle.b,
		     message->old_handle.c,
		     message->old_handle.d));

    t = xfs_node_find (&xfs[fd], &message->old_handle);
    if (t != NULL) {
	t->handle = message->new_handle;
    } else {
	printf ("XFS PANIC WARNING! xfs_message_updatefid: no node!\n");
	error = ENOENT;
    }
    return error;
}

/*
 * Try to clean out nodes for the userland daemon
 */

static void
gc_vnode (struct vnode *vp)
{
    if (vp->v_count <= 1) {
	XFSDEB(XDEBMSG, ("xfs_message_gc: try\n"));
	dnlc_purge_vp(vp);
    } else {
	XFSDEB(XDEBMSG, ("xfs_message_gc: used (%d)\n", vp->v_count));
    }
}

int
xfs_message_gc_nodes(int fd,
		     struct xfs_message_gc_nodes *message,
		     u_int size)
{
    XFSDEB(XDEBMSG, ("xfs_message_gc\n"));

    if (message->len == 0) {
	struct xfs_node *xp, *next;

	for (xp = xfs[fd].nodes; xp != NULL; xp = next) {
	    next = xp->next;
	    gc_vnode (XNODE_TO_VNODE(xp));
	}
    } else {
	struct xfs_node *t;
	int i;

	for (i = 0; i < message->len; i++) {
	    t = xfs_node_find (&xfs[fd], &message->handle[i]);
	    if (t == NULL)
		continue;

	    gc_vnode(XNODE_TO_VNODE(t));
	}
    }

    return 0;
}

/*
 * Probe what version of xfs this support
 */

int
xfs_message_version(int fd,
		    struct xfs_message_version *message,
		    u_int size)
{
    struct xfs_message_wakeup msg;
    int ret;

    ret = XFS_VERSION;

    msg.header.opcode = XFS_MSG_WAKEUP;
    msg.sleepers_sequence_num = message->header.sequence_num;
    msg.error = ret;

    return xfs_message_send(fd, (struct xfs_message_header *) &msg, sizeof(msg));
}
