/*
 * Copyright (c) 1995 - 2000 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#define __NO_VERSION__

#include <xfs/xfs_locl.h>
#include <xfs/xfs_common.h>
#include <xfs/xfs_fs.h>
#include <xfs/xfs_deb.h>

RCSID("$Id: xfs_node.c,v 1.51.2.3 2001/09/06 23:40:45 ahltorp Exp $");

/*
 * Copy the attributes from `attr' into `inode', setting the fields
 * that weren't set in `attr' to reasonable defaults.
 */

void
xfs_attr2inode(const struct xfs_attr *attr, struct inode *inode,
	       int clear_node)
{
    if (clear_node) {
	inode->i_mode   = 0;
	inode->i_uid    = 0;
	inode->i_gid    = 0;
	inode->i_nlink  = 1;
	inode->i_size   = 0;
	inode->i_atime  = 0;
	inode->i_mtime  = 0;
	inode->i_ctime  = 0;
	inode->i_blocks = 0;
    }
    if (XA_VALID_MODE(attr))
	inode->i_mode   = attr->xa_mode;
    if (XA_VALID_UID(attr))
	inode->i_uid    = attr->xa_uid;
    if (XA_VALID_GID(attr))
	inode->i_gid    = attr->xa_gid;
    if (XA_VALID_NLINK(attr))
	inode->i_nlink  = attr->xa_nlink;
    if (XA_VALID_SIZE(attr)) {
	inode->i_size   = attr->xa_size;
    inode->i_blocks = inode->i_size >> I_BLOCKS_BITS; /* / I_BLOCKS_UNIT */
    }
    if (XA_VALID_ATIME(attr))
	inode->i_atime  = attr->xa_atime;
    if (XA_VALID_MTIME(attr))
	inode->i_mtime  = attr->xa_mtime;
    if (XA_VALID_CTIME(attr))
	inode->i_ctime  = attr->xa_ctime;
    if (XA_VALID_FILEID(attr))
	inode->i_ino = attr->xa_fileid;
}

/*
 * Allocate a new inode (of the file system identified by `sb') and
 * return it, associated with `newnode'.  Return the `inode' or NULL.
 * The reference count on `inode' is incremented.
 */

static struct inode *
xfs_iget(struct super_block *sb, struct xfs_msg_node *node,
	 struct xfs_node *newnode)
{
    struct inode *inode;
    struct xfs_attr *attr = &node->attr;

    XFSDEB(XDEBNODE, ("xfs_iget sb: %p node: %p newnode: %p\n", sb, node, newnode));
    if (sb == NULL) {
     	printk(KERN_EMERG "XFS Panic: xfs_iget: super block is NULL\n");
     	return NULL;
    }
    inode = get_empty_inode ();
    if (inode == NULL) {
    	printk(KERN_EMERG "XFS Panic: xfs_iget: get_empty_inode failed\n");
    	return NULL;
    }
    inode->i_sb  = sb;
    inode->i_dev = sb->s_dev;

    if (!XA_VALID_TYPE(attr)) {
	inode->i_op  = &xfs_dead_inode_operations;
#ifndef HAVE_STRUCT_INODE_OPERATIONS_DEFAULT_FILE_OPS
	inode->i_fop = &xfs_dead_operations;
#endif
    } else if (attr->xa_type == XFS_FILE_REG) {
	inode->i_op  = &xfs_file_inode_operations;
#ifndef HAVE_STRUCT_INODE_OPERATIONS_DEFAULT_FILE_OPS
	inode->i_fop = &xfs_file_operations;
#endif
#ifdef LINUX2_3
        inode->i_mapping->a_ops = &xfs_aops;
#endif
    } else if (attr->xa_type == XFS_FILE_DIR) {
	inode->i_op  = &xfs_dir_inode_operations;
#ifndef HAVE_STRUCT_INODE_OPERATIONS_DEFAULT_FILE_OPS
	inode->i_fop = &xfs_dir_operations;
#endif
#ifdef LINUX2_3
        inode->i_mapping->a_ops = &xfs_aops;
#endif
    } else if (attr->xa_type == XFS_FILE_LNK) {
	inode->i_op  = &xfs_link_inode_operations;
#ifndef HAVE_STRUCT_INODE_OPERATIONS_DEFAULT_FILE_OPS
	inode->i_fop = &xfs_link_operations;
#endif
#ifdef LINUX2_3
	inode->i_mapping->a_ops = &xfs_aops;
#endif
    } else {
	inode->i_op  = &xfs_dead_inode_operations;
#ifndef HAVE_STRUCT_INODE_OPERATIONS_DEFAULT_FILE_OPS
	inode->i_fop = &xfs_dead_operations;
#endif
    }
    
    inode->u.generic_ip = newnode;

    return inode;
}

/*
 * Find the node identified by `node->handle' belong to the filesystem
 * `xfsp' or create a new one.  The node is returned with incremented
 * reference count.  Always return a valid node.
 */

struct xfs_node *
new_xfs_node(struct xfs *xfsp, struct xfs_msg_node *node)
{
    struct xfs_node *result;
    
    XFSDEB(XDEBNODE, ("new_xfs_node %d.%d.%d.%d\n",
		      node->handle.a,
		      node->handle.b,
		      node->handle.c,
		      node->handle.d));
    
    /* Does not allow duplicates */
    result = xfs_node_find(xfsp, &node->handle);
    if (result == 0) {
	struct inode *v;
	
	result = xfs_alloc(sizeof(*result), XFS_MEM_XNODE);
	if (result == 0) {
	    printk(KERN_EMERG "xfs_alloc(%d) failed\n", (int)sizeof(*result));
	    panic("new_xfs_node: You Loose!");
	}
	
	memset(result, 0, sizeof(*result));
	
	/* Save reference on list */
	result->next = xfsp->nodes;
	xfsp->nodes = result;
	
	xfsp->nnodes++;
	
	result->anonrights = node->anonrights;
	result->handle = node->handle;
	result->flags = 0;
	result->tokens = 0;
	INIT_LIST_HEAD(&result->inactive_list);
	
	v=xfs_iget(XFS_TO_VFS(xfsp),node,result);
	result->vn = v;
	/* XXX - handle this case better */
    } else {
	/* Node is already cached */
	xfs_iref(XNODE_TO_VNODE(result));
    }
    
    /* Init other fields */
    result->attr = node->attr;
    xfs_attr2inode (&result->attr, result->vn, 1);
    result->tokens = node->tokens;
    memmove(result->id, node->id, sizeof(result->id));
    memmove(result->rights, node->rights, sizeof(result->rights));
    
    return result;
}

/*
 * Free the memory used by `node' and remove it from the list of nodes.
 */

void
free_xfs_node(struct xfs_node *node)
{
    if (node == NULL)
	return;

    clear_xfs_node (node);
    xfs_free(node, XFS_MEM_XNODE);
}

/*
 * remove everything about `node'
 */

void
clear_xfs_node(struct xfs_node *node)
{
    struct xfs *xfsp;
    struct xfs_node *t;
    
    XFSDEB(XDEBNODE, ("clear_xfs_node starting\n"));
    
    if (node == NULL)
	return;
    
    xfsp = XFS_FROM_XNODE(node);
    
    /* First remove from chain. */
    if (node == xfsp->nodes) {
	xfsp->nodes = node->next;
    } else {
	if (xfsp->nodes == NULL)
	    goto done;
	for (t = xfsp->nodes; t->next; t = t->next)
	    if (t->next == node) {
		t->next = t->next->next;
		goto done;
	    }
	printk(KERN_EMERG "XFS Panic: clear_xfs_node(0x%p) failed?\n", node);
	return;
    }
    
done:
    /* XXX Really need to put back dirty data first. */
    XFS_TOKEN_CLEAR(node, ~0,
		    XFS_OPEN_MASK | XFS_ATTR_MASK |
		    XFS_DATA_MASK | XFS_LOCK_MASK);
    if (DATA_FROM_XNODE(node)) {
	XFS_RESET_I_MAPPING(XNODE_TO_VNODE(node));
	dput(DATA_FROM_XNODE(node));
	DATA_FROM_XNODE(node) = NULL;
    }
    node->flags = 0;
    xfsp->nnodes--;
    XFSDEB(XDEBNODE, ("clear_xfs_node xnode: %p inode:%p\n",
		      node, XNODE_TO_VNODE(node)));
    XNODE_TO_VNODE(node)->u.generic_ip = NULL;
    
    XFSDEB(XDEBNODE, ("clear_xfs_node done\n"));
}

/*
 * Throw all nodes of the filesystem `xfsp'.
 */

void
free_all_xfs_nodes(struct xfs *xfsp)
{
    struct xfs_node *xnode;
    XFSDEB(XDEBNODE, ("free_all_xfs_nodes starting\n"));
    
    /* There might still be a few nodes out there, invalidate them */
    for (xnode = xfsp->nodes; xnode; xnode = xnode->next) {
	xfs_force_invalid_xnode(xnode);
    }
    
    XFSDEB(XDEBNODE, ("free_all_xfs_nodes done\n"));
}

/*
 * Return the node identifed by `handlep' in `xfsp' or NULL.
 */

struct xfs_node *
xfs_node_find(struct xfs *xfsp, xfs_handle *handlep)
{
    struct xfs_node *x;
    
    XFSDEB(XDEBNODE, ("xfs_node_find\n"));
    
    for(x = xfsp->nodes; x != NULL; x = x->next)
	if (xfs_handle_eq(&x->handle, handlep))
	    break;
    return x;
}

/*
 * Returns 1 if pag has any rights set in the node
 */

int
xfs_has_pag(const struct xfs_node *xn, xfs_pag_t pag)
{
    int i;
    
    for (i = 0; i < MAXRIGHTS; i++)
	if (xn->id[i] == pag)
	    return 1;
    
    return 0;
}

/*
 * Return the number of users of the node `xn'.
 */

int
xfs_node_users(struct xfs_node *xnode)
{
    struct inode *inode = XNODE_TO_VNODE(xnode);
    struct list_head *pos;
    int users = 0;
    
    list_for_each(pos, &inode->i_dentry) {
	struct dentry *dentry = list_entry(pos, struct dentry, d_alias);
	if (xfs_dcount(dentry))
	    users++;
    }
    return users;
}
