/*
 * Copyright (c) 1995 - 2000 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <xfs/xfs_locl.h>

RCSID("$Id: xfs_syscalls-wrap-freebsd.c,v 1.7 2000/10/02 23:32:40 lha Exp $");

/*
 * XFS system calls.
 */

#include <xfs/xfs_syscalls.h>
#include <xfs/xfs_message.h>
#include <xfs/xfs_fs.h>
#include <xfs/xfs_dev.h>
#include <xfs/xfs_node.h>
#include <xfs/xfs_deb.h>

#include <kafs.h>

#ifdef HAVE_STRUCT_PROC_P_RETVAL

static int
xfs_syscall(struct proc *proc, void *varg)
{
    int retval = 0;
    int ret;

    ret = xfspioctl(proc, varg, &retval);
    proc->p_retval[0] = retval;
    return ret;
}

#else

static int
xfs_syscall(struct proc *proc, void *varg, int *return_value)
{
    return xfspioctl(proc, varg, return_value);
}

#endif

struct sysent xfs_syscallent = {
    5,
    xfs_syscall
};

static struct sysent old_setgroups;

int (*old_setgroups_func)(struct proc *, void *);

#if KLD_MODULE && defined(SYSCALL_MODULE)

int
xfs_install_syscalls(void)
{
    old_setgroups = sysent[SYS_setgroups];
    old_setgroups_func = old_setgroups.sy_call;
    sysent[SYS_setgroups].sy_call = (sy_call_t *)xfs_setgroups;

    return 0;
}

int
xfs_uninstall_syscalls(void)
{
    sysent[SYS_setgroups] = old_setgroups;

    return 0;
}

int
xfs_stat_syscalls(void)
{
    return 0;
}

#else

/*
 * Where it's stored.
 */

int xfs_syscall_num;

static struct sysent old_syscallent;

/*
 *
 */

static int
try_install_syscall (int offset,
		     struct sysent new_sysent,
		     struct sysent *old_sysent)
{
    if (sysent[offset].sy_call == (sy_call_t *)lkmnosys) {
	*old_sysent = sysent[offset];
	sysent[offset] = new_sysent;
	return 0;
    }
    return EBUSY;
}

/*
 *
 */

static int
install_first_free_syscall (int *offset,
			    struct sysent new_sysent,
			    struct sysent *old_sysent)
{
    int i;

    for (i = 1; i < SYS_MAXSYSCALL; ++i)
	if (try_install_syscall (i, new_sysent, old_sysent) == 0) {
	    *offset = i;
	    return 0;
	}
    return ENFILE;
}


/*
 * Try AFS_SYSCALL first, if that fails, any free slot
 */

/* XXX */

#define AFS_SYSCALL 210

int
xfs_install_syscalls(void)
{
    int ret = ENOENT;

#ifdef AFS_SYSCALL
    if (ret != 0) {
	ret = try_install_syscall(AFS_SYSCALL,
				  xfs_syscallent,
				  &old_syscallent);
	if (ret == 0)
	    xfs_syscall_num = AFS_SYSCALL;
    }
#endif
    if (ret != 0)
	ret = install_first_free_syscall (&xfs_syscall_num,
					  xfs_syscallent,
					  &old_syscallent);
    if (ret == 0)
	XFSDEB(XDEBSYS, ("xfs_syscall installed at %d\n", xfs_syscall_num));
    else
	XFSDEB(XDEBSYS, ("failed installing xfs_syscall\n"));

    if (ret == 0) {
	old_setgroups = sysent[SYS_setgroups];
	old_setgroups_func = old_setgroups.sy_call;
	sysent[SYS_setgroups].sy_call = (sy_call_t *)xfs_setgroups;
    }
    return ret;
}

int
xfs_uninstall_syscalls(void)
{
    if (xfs_syscall_num)
	sysent[xfs_syscall_num] = old_syscallent;
    sysent[SYS_setgroups] = old_setgroups;
    return 0;
}

int
xfs_stat_syscalls(void)
{
    return 0;
}

#endif
