/*
 * Copyright (c) 1995 - 2000 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */


#include <xfs/xfs_locl.h>
#include <xfs/xfs_message.h>
#include <xfs/xfs_msg_locl.h>
#include <xfs/xfs_fs.h>
#include <xfs/xfs_dev.h>
#include <xfs/xfs_deb.h>

RCSID("$Id: xfs_dev-osf.c,v 1.9 2000/10/02 23:32:03 lha Exp $");

static int
xfs_devopen(dev_t dev, int flag, int format)
{
    XFSDEB(XDEBDEV, ("xfsopen dev = %d.%d, flag = %d, format = %d\n", 
		     major(dev), minor(dev), flag, format));
    return xfs_devopen_common(dev, u.u_procp);
}

static int
xfs_devclose(dev_t dev, int flag, int format)
{
    XFSDEB(XDEBDEV, ("xfsclose dev = %d.%d, flag = %d, format = %d\n", 
		     major(dev), minor(dev), flag, format));
    return xfs_devclose_common(dev, u.u_procp);
}

/*
 * Not used.
 */

static int
xfs_devioctl(dev_t dev, 
	     unsigned int cmd, caddr_t data, int flags)
{
    XFSDEB(XDEBDEV, ("xfs_devioctl dev = %d.%d, cmd = %lu, "
		     "data = %lx, flags = %x\n", 
		     major(dev), minor(dev), (unsigned long)cmd, data, flags));
    return EINVAL;
}

/*
 * Are there any messages on this filesystem?
 */

static int
xfs_devselect(dev_t dev, short *events, short *revents, int scanning)
{
    struct xfs_channel *chan = &xfs_channel[minor(dev)];
    if(*events & POLLNORM) {
	if(scanning) {
	    if (!xfs_emptyq(&chan->messageq))
		*revents |= POLLNORM; /* Something to read */
	    else
		select_enqueue(&chan->sel_q);
	} else {
	    select_dequeue(&chan->sel_q);
	}
    }
    if(*events & POLLOUT) {
	/* select for writing */
    }
    return 0;
}

void
xfs_select_wakeup(struct xfs_channel *chan)
{
    select_wakeup(&(chan)->sel_q);
}

/*
 * Install and uninstall device.
 */

#if 0
struct driver xfsdriver = {
    NULL, /* probe */
    NULL, /* slave */
    NULL, /* cattach */
    NULL, /* dattach */
    NULL, /* go */
    NULL, /* addr_list */
    NULL, /* dev_name */
    NULL, /* dev_list */
    "xfs", 
    NULL, /* ctlr_list */
    0,    /* xclu */
    0,    /* addr1_size */
    0,    /* addr1_atype */
    0,    /* addr2_size */
    0,    /* addr2_atype */
    NULL, /* ctlr_unattach */
    NULL, /* dev_unattach */
};
#endif

extern int nodev();

struct dsent xfs_dsent = {
    xfs_devopen,
    xfs_devclose,
    nodev, /* strategy */
    xfs_devread,
    xfs_devwrite,
    xfs_devioctl,
    nodev, /* dump */
    nodev, /* psize */
    nodev, /* stop */
    nodev, /* reset */
    xfs_devselect,
    nodev, /* mmap */
    nodev, /* segmap */
    NULL, /* ttys */
    DEV_FUNNEL,
    0, /* bflags */
    0  /* cflags */
};

int xfs_dev_major;

int
xfs_install_device()
{
    int i;

    xfs_dev_major = devsw_add("xfs", 0, 64, &xfs_dsent);
    if(xfs_dev_major < 0)
	xfs_dev_major = devsw_add("xfs", 0, -1, &xfs_dsent);
    if(xfs_dev_major < 0)
	return ENODEV;
    
    for (i = 0; i < NXFS; i++) {
	XFSDEB(XDEBDEV, ("before initq(messageq and sleepq)\n"));
	xfs_initq(&xfs_channel[i].messageq);
	xfs_initq(&xfs_channel[i].sleepq);
	xfs_channel[i].status = 0;
	queue_init(&xfs_channel[i].sel_q.links);
    }
    return 0;
}

int
xfs_uninstall_device()
{
    int i;
    dev_t dev;
    struct xfs_channel *chan;
    for (i = 0; i < NXFS; i++) {
	dev = makedev(xfs_dev_major, i);
	chan = &xfs_channel[minor(dev)];
	if (chan->status & CHANNEL_OPENED)
	    xfs_devclose(dev, 0, 0);
    }
    if(devsw_del("xfs", 0) < 0)
	return ESRCH;
    return 0;
}
