#############################################################################
# $Id: CMake.NCBIptb.cmake 568803 2018-08-10 16:42:57Z gouriano $
#############################################################################
#############################################################################
##
##  NCBI CMake wrapper
##    Author: Andrei Gourianov, gouriano@ncbi
##
##---------------------------------------------------------------------------
##  in CMakeLists.txt:
##    NCBI_add_subdirectory( list of subdirectories)
##    NCBI_add_library(      list of libraries)
##    NCBI_add_app(          list of apps)
##---------------------------------------------------------------------------
##  the following calls in CMakeLists.txt affect all projects in current dir and all subdirs
##    NCBI_headers( list of headers)
##    NCBI_disable_pch() / NCBI_enable_pch()
##    NCBI_requires( list of components)
##    NCBI_optional_components(  list of components)
##    NCBI_add_definitions(         list of compiler definitions)
##    NCBI_add_include_directories( list of directories)
##    NCBI_uses_toolkit_libraries(  list of libraries)
##    NCBI_uses_external_libraries( list of libraries)
##    NCBI_project_tags(list of tags)
##    
##---------------------------------------------------------------------------
##  in CMakeLists.xxx.[lib|app].txt - all calls affect current project only
##    NCBI_begin_lib(name) or NCBI_begin_app(name)
##
##      NCBI_sources(   list of source files)
##      NCBI_headers(   list of header files) - only relative paths and masks are allowed
##      NCBI_resources( list of resource files) - file extension is mandatory
##      NCBI_dataspecs( list of data specs - ASN, DTD, XSD etc) - file extension is mandatory
##
##      NCBI_requires( list of components)
##      NCBI_optional_components(  list of components)
##
##      NCBI_enable_pch() / NCBI_disable_pch() - no arguments
##      NCBI_disable_pch_for( list of source files)
##      NCBI_set_pch_header( header file)
##      NCBI_set_pch_define( macro to define)
##
##      NCBI_uses_toolkit_libraries(  list of libraries)
##      NCBI_uses_external_libraries( list of libraries)
##      NCBI_add_definitions(         list of compiler definitions)
##      NCBI_add_include_directories( list of directories)
##
##      NCBI_hosts_projects(list of parts) - used to assemble composite shared libraries
##
##      NCBI_project_tags(     list of tags)
##      NCBI_project_watchers( list of watchers)
##
##      Testing:
##      short form
##          NCBI_add_test( test command) - empty command means run the app with no arguments
##      long form
##          NCBI_begin_test(name) - name is optional
##              NCBI_set_test_command(command, maybe with arguments)
##           OR NCBI_set_test_arguments(arguments only)
##              NCBI_set_test_assets(list of assets) - required files and directories
##              NCBI_set_test_timeout(seconds)
##              NCBI_set_test_requires(list of components)
##          NCBI_end_test()
##
##    NCBI_end_lib(result) or NCBI_end_app(result) - argument 'result' is optional
##
#############################################################################
function(NCBI_add_root_subdirectory)

    set(NCBI_CURRENT_SOURCE_DIR ${CMAKE_CURRENT_SOURCE_DIR})
    set(NCBI_PTBMODE_COLLECT_DEPS OFF)
    if (NCBI_PTBCFG_ENABLE_COLLECTOR)
        message("Analyzing source tree...")
        set(NCBI_PTBMODE_COLLECT_DEPS ON)
        set_property(GLOBAL PROPERTY NCBI_PTBPROP_ALL_PROJECTS "")
        set_property(GLOBAL PROPERTY NCBI_PTBPROP_ALLOWED_PROJECTS "")

        NCBI_add_subdirectory(${ARGV})

        set(NCBI_PTBMODE_COLLECT_DEPS OFF)
        get_property(_allprojects     GLOBAL PROPERTY NCBI_PTBPROP_ALL_PROJECTS)
        get_property(_allowedprojects GLOBAL PROPERTY NCBI_PTBPROP_ALLOWED_PROJECTS)

        if(OFF)
            message("NCBI_PTBPROP_ALL_PROJECTS: ${_allprojects}")
            foreach(_prj IN LISTS _allprojects)
                get_property(_prjdeps GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_prj})
                message("NCBI_PTBPROP_DEPS_${_prj}: ${_prjdeps}")
            endforeach()
            message("NCBI_PTBPROP_ALLOWED_PROJECTS: ${_allowedprojects}")
        endif()

        if(NOT "${_allowedprojects}" STREQUAL "")
            message("Collecting projects...")
            foreach(_prj IN LISTS _allowedprojects)
                NCBI_internal_collect_dependencies(${_prj})
                get_property(_prjdeps GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_prj})
                set(NCBI_PTB_ALLOWED_PROJECTS ${NCBI_PTB_ALLOWED_PROJECTS} ${_prj} ${_prjdeps})
            endforeach()
            list(SORT NCBI_PTB_ALLOWED_PROJECTS)
            list(REMOVE_DUPLICATES NCBI_PTB_ALLOWED_PROJECTS)
#message("NCBI_PTB_ALLOWED_PROJECTS: ${NCBI_PTB_ALLOWED_PROJECTS}")
        elseif("${_allprojects}" STREQUAL "")
            message(FATAL_ERROR "List of projects is empty")
            return()
        endif()
        message("Configuring projects...")
    endif()

    NCBI_add_subdirectory(${ARGV})
endfunction()

#############################################################################
function(NCBI_add_subdirectory)
    if(NCBI_PTBMODE_PARTS)
        return()
    endif()
if(OFF)
# this requires that ALL subdirs use this new API
    set(_curdir ${NCBI_CURRENT_SOURCE_DIR})
    foreach(_sub IN LISTS ARGV)
        set(NCBI_CURRENT_SOURCE_DIR ${_curdir}/${_sub})
        if (EXISTS "${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.txt")
            if (NCBI_PTBMODE_COLLECT_DEPS)
                NCBI_internal_include("${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.txt")
            else()
                add_subdirectory(${_sub})
            endif()
        else()
            message(WARNING "ERROR: directory not found: ${NCBI_CURRENT_SOURCE_DIR}")
        endif()
    endforeach()
else()
  if(NCBI_PTBMODE_COLLECT_DEPS)
    set(_curdir ${NCBI_CURRENT_SOURCE_DIR})
    foreach(_sub IN LISTS ARGV)
      if (EXISTS "${_curdir}/${_sub}/CMakeLists.txt")
        set(NCBI_CURRENT_SOURCE_DIR ${_curdir}/${_sub})
        NCBI_internal_include("${_curdir}/${_sub}/CMakeLists.txt")
      else()
        message(WARNING "ERROR: directory not found: ${_curdir}/${_sub}")
      endif()
    endforeach()
  else()
    foreach(_sub IN LISTS ARGV)
      set(NCBI_CURRENT_SOURCE_DIR ${CMAKE_CURRENT_SOURCE_DIR}/${_sub})
      if (EXISTS "${CMAKE_CURRENT_SOURCE_DIR}/${_sub}")
        add_subdirectory(${_sub})
      else()
        message(WARNING "ERROR: directory not found: ${CMAKE_CURRENT_SOURCE_DIR}/${_sub}")
      endif()
    endforeach()
  endif()
endif()
endfunction()

#############################################################################
function(NCBI_add_library)
    if(NCBI_PTBMODE_PARTS)
        return()
    endif()
if(OFF)
# this requires that ALL subdirs use this new API
    foreach(_lib IN LISTS ARGV)
        if (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.lib.txt)
            NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.lib.txt)
        elseif (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.asn.txt)
            NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.asn.txt)
        else()
            message(WARNING "ERROR: file not found: ${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.lib.txt")
        endif()
    endforeach()
else()
  if(NCBI_PTBMODE_COLLECT_DEPS)
    foreach(_lib IN LISTS ARGV)
      if (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.lib.txt)
        NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.lib.txt)
      elseif (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.asn.txt)
        NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.asn.txt)
      else()
        message(WARNING "ERROR: file not found: ${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.lib.txt")
      endif()
    endforeach()
  else()
    set(NCBI_CURRENT_SOURCE_DIR ${CMAKE_CURRENT_SOURCE_DIR})
    foreach(_lib IN LISTS ARGV)
      if (EXISTS ${CMAKE_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.lib.txt)
        NCBI_internal_include(CMakeLists.${_lib}.lib.txt)
      elseif (EXISTS ${CMAKE_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.asn.txt)
        NCBI_internal_include(CMakeLists.${_lib}.asn.txt)
      else()
        message(WARNING "ERROR: file not found: ${CMAKE_CURRENT_SOURCE_DIR}/CMakeLists.${_lib}.lib.txt")
      endif()
    endforeach()
  endif()
endif()
endfunction()

#############################################################################
function(NCBI_add_app)
    if(NCBI_PTBMODE_PARTS)
        return()
    endif()
if(OFF)
# this requires that ALL subdirs use this new API
    foreach(_app IN LISTS ARGV)
        if (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_app}.app.txt)
            NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_app}.app.txt)
        else()
            message(WARNING "ERROR: file not found: ${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_app}.app.txt")
        endif()
    endforeach()
else()
  if(NCBI_PTBMODE_COLLECT_DEPS)
    foreach(_app IN LISTS ARGV)
      if (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_app}.app.txt)
        NCBI_internal_include(${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_app}.app.txt)
      else()
        message(WARNING "ERROR: file not found: ${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.${_app}.app.txt")
      endif()
    endforeach()
  else()
    set(NCBI_CURRENT_SOURCE_DIR ${CMAKE_CURRENT_SOURCE_DIR})
    foreach(_app IN LISTS ARGV)
      if (EXISTS ${CMAKE_CURRENT_SOURCE_DIR}/CMakeLists.${_app}.app.txt)
        NCBI_internal_include(CMakeLists.${_app}.app.txt)
      else()
        message(WARNING "ERROR: file not found: ${CMAKE_CURRENT_SOURCE_DIR}/CMakeLists.${_app}.app.txt")
      endif()
    endforeach()
  endif()
endif()
endfunction()

#############################################################################
macro(NCBI_begin_lib _name)
    if(NCBI_PTBMODE_PARTS)
        set(_libname ${_name}.part)
    else()
        set(_libname ${_name})
    endif()
    set(NCBI_PROJECT_lib ${_libname})
    if(NCBI_EXPERIMENTAL_CFG)
        set(NCBI_PROJECT ${_libname})
        set(NCBI_${NCBI_PROJECT}_OUTPUT ${_name})
    else()
        if(${_name} STREQUAL "general")
            set(NCBI_PROJECT ${_libname}-lib)
            set(NCBI_${NCBI_PROJECT}_OUTPUT ${_name})
        else()
            set(NCBI_PROJECT ${_libname})
            set(NCBI_${NCBI_PROJECT}_OUTPUT ${_name})
        endif()
    endif()
    if ("${ARGC}" GREATER "1")
        if(BUILD_SHARED_LIBS)
            set(NCBI_${NCBI_PROJECT}_TYPE ${ARGV1})
        else()
            set(NCBI_${NCBI_PROJECT}_TYPE STATIC)
        endif()
    else()
#CMake global flag
        if(BUILD_SHARED_LIBS)
            if(WIN32 OR XCODE)
                set(NCBI_${NCBI_PROJECT}_TYPE STATIC)
            else()
                set(NCBI_${NCBI_PROJECT}_TYPE SHARED)
            endif()
        else()
            set(NCBI_${NCBI_PROJECT}_TYPE STATIC)
        endif()
    endif()
    set(NCBI_PROJECT_PARTNAME ${_name})
    set(NCBI_PROJECT_ID ${_name}.${NCBI_${NCBI_PROJECT}_TYPE})
endmacro()

#############################################################################
macro(NCBI_end_lib)
    if(NOT DEFINED NCBI_PROJECT_lib)
        message(SEND_ERROR "${NCBI_CURRENT_SOURCE_DIR}/${NCBI_PROJECT}: Unexpected NCBI_end_lib call")
    endif()
    NCBI_internal_add_project()
    unset(NCBI_PROJECT)
endmacro()

#############################################################################
macro(NCBI_begin_app _name)
    if(NCBI_PTBMODE_PARTS)
        set(_appname ${_name}.part)
    else()
        set(_appname ${_name})
    endif()
    set(NCBI_PROJECT_app ${_appname})
    if(NCBI_EXPERIMENTAL_CFG)
        set(NCBI_PROJECT ${_appname})
        set(NCBI_${NCBI_PROJECT}_OUTPUT ${_name})
    else()
        set(NCBI_PROJECT ${_appname}-app)
        set(NCBI_${NCBI_PROJECT}_OUTPUT ${_name})
    endif()
    set(NCBI_${NCBI_PROJECT}_TYPE CONSOLEAPP)
    set(NCBI_PROJECT_PARTNAME ${_name})
    set(NCBI_PROJECT_ID ${_name}.${NCBI_${NCBI_PROJECT}_TYPE})
endmacro()

#############################################################################
macro(NCBI_end_app)
    if(NOT DEFINED NCBI_PROJECT_app)
        message(SEND_ERROR "${NCBI_CURRENT_SOURCE_DIR}/${NCBI_PROJECT}: Unexpected NCBI_end_app call")
    endif()
    NCBI_internal_add_project()
    unset(NCBI_PROJECT)
endmacro()

#############################################################################
macro(NCBI_sources)
    set(NCBI_${NCBI_PROJECT}_SOURCES ${NCBI_${NCBI_PROJECT}_SOURCES} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_headers)
    set(NCBI_${NCBI_PROJECT}_HEADERS ${NCBI_${NCBI_PROJECT}_HEADERS} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_resources)
    set(NCBI_${NCBI_PROJECT}_RESOURCES ${NCBI_${NCBI_PROJECT}_RESOURCES} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_dataspecs)
    set(NCBI_${NCBI_PROJECT}_DATASPEC ${NCBI_${NCBI_PROJECT}_DATASPEC} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_hosts_projects)
    set(NCBI_${NCBI_PROJECT}_PARTS ${NCBI_${NCBI_PROJECT}_PARTS} "${ARGV}")
endmacro()

macro(NCBI_hosts_virtual_projects)
    set(NCBI_${NCBI_PROJECT}_VIRTPARTS ${NCBI_${NCBI_PROJECT}_VIRTPARTS} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_requires)
    set(NCBI_${NCBI_PROJECT}_REQUIRES ${NCBI_${NCBI_PROJECT}_REQUIRES} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_optional_components)
    set(NCBI_${NCBI_PROJECT}_COMPONENTS ${NCBI_${NCBI_PROJECT}_COMPONENTS} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_enable_pch)
    set(NCBI_${NCBI_PROJECT}_USEPCH ON)
endmacro()
macro(NCBI_disable_pch)
    set(NCBI_${NCBI_PROJECT}_USEPCH OFF)
endmacro()

#############################################################################
macro(NCBI_disable_pch_for)
    set(NCBI_${NCBI_PROJECT}_NOPCH ${NCBI_${NCBI_PROJECT}_NOPCH} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_set_pch_header)
    set(NCBI_${NCBI_PROJECT}_PCH "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_set_pch_define)
    set(NCBI_${NCBI_PROJECT}_PCH_DEFINE "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_uses_toolkit_libraries)
    set(NCBI_${NCBI_PROJECT}_NCBILIB ${NCBI_${NCBI_PROJECT}_NCBILIB} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_uses_external_libraries)
    set(NCBI_${NCBI_PROJECT}_EXTLIB ${NCBI_${NCBI_PROJECT}_EXTLIB} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_add_include_directories)
    set(NCBI_${NCBI_PROJECT}_INCLUDES ${NCBI_${NCBI_PROJECT}_INCLUDES} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_add_definitions)
    set(NCBI_${NCBI_PROJECT}_DEFINES ${NCBI_${NCBI_PROJECT}_DEFINES} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_project_tags)
    set(NCBI_${NCBI_PROJECT}_PROJTAG ${NCBI_${NCBI_PROJECT}_PROJTAG} "${ARGV}")
endmacro()

#############################################################################
macro(NCBI_project_watchers)
    set(NCBI_${NCBI_PROJECT}_WATCHER ${NCBI_${NCBI_PROJECT}_WATCHER} "${ARGV}")
endmacro()

#############################################################################
#############################################################################
#############################################################################
macro(NCBI_begin_test)
    if(DEFINED NCBI_TEST)
        message(SEND_ERROR "${NCBI_CURRENT_SOURCE_DIR}/${NCBI_PROJECT}: Unexpected NCBI_begin_test call")
    endif()
    if (DEFINED NCBITEST_${NCBI_PROJECT}_NUM)
        math(EXPR NCBITEST_${NCBI_PROJECT}_NUM "${NCBITEST_${NCBI_PROJECT}_NUM} + 1")
    endif()
    if ("${ARGC}" GREATER "0")
        set(_testname "${ARGV}")
        if ( "${_testname}" STREQUAL "")
            set(_testname "${NCBI_PROJECT}${NCBITEST_${NCBI_PROJECT}_NUM}")
        endif()
    else()
        set(_testname "${NCBI_PROJECT}${NCBITEST_${NCBI_PROJECT}_NUM}")
    endif()
    set(NCBI_TEST ${_testname})
    if (NOT DEFINED NCBITEST_${NCBI_PROJECT}_NUM)
        set(NCBITEST_${NCBI_PROJECT}_NUM "1")
    endif()
endmacro()

##############################################################################
macro(NCBI_set_test_command)
    set( _args ${ARGV})
    list(GET _args 0 _cmd)
    list(REMOVE_AT _args 0) 
    if ( "${_cmd}" STREQUAL "${NCBI_PROJECT}")
        set(_cmd "${NCBI_${NCBI_PROJECT}_OUTPUT}")
    endif()
    if (NOT NCBI_EXPERIMENTAL_CFG)
        set(_cmd "${EXECUTABLE_OUTPUT_PATH}/${_cmd}")
    endif()
    set(NCBITEST_${NCBI_TEST}_CMD "${_cmd}")
    set(NCBITEST_${NCBI_TEST}_ARG "${_args}")
endmacro()

##############################################################################
macro(NCBI_set_test_arguments)
    set(NCBITEST_${NCBI_TEST}_ARG "${ARGV}")
endmacro()

##############################################################################
macro(NCBI_set_test_assets)
    set(NCBITEST_${NCBI_TEST}_ASSETS "${ARGV}")
endmacro()

##############################################################################
macro(NCBI_set_test_timeout)
    set(NCBITEST_${NCBI_TEST}_TIMEOUT "${ARGV}")
endmacro()

##############################################################################
macro(NCBI_set_test_requires)
    set(NCBITEST_${NCBI_TEST}_REQUIRES "${ARGV}")
endmacro()

##############################################################################
macro(NCBI_end_test)
    set(NCBI_ALLTESTS ${NCBI_ALLTESTS} ${NCBI_TEST})
    unset(NCBI_TEST)
endmacro()

##############################################################################
macro(NCBI_add_test)
    if ("${ARGC}" GREATER "0")
        set(_cmd "${ARGV}")
        if ( "${_cmd}" STREQUAL "")
            set(_cmd "${NCBI_PROJECT}")
        endif()
    else()
        set(_cmd "${NCBI_PROJECT}")
    endif()
    NCBI_begin_test()
    NCBI_set_test_command(${_cmd})
    NCBI_end_test()
endmacro()


##############################################################################
##############################################################################
##############################################################################

function(NCBI_internal_collect_dependencies _project)
    get_property(_prjdeps GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_project} SET)
    if (NOT _prjdeps)
        message(WARNING "ERROR: project ${_project} not found")
        return()
    endif()
    get_property(_prjdeps GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_project})
    foreach( _value IN LISTS _prjdeps)
        NCBI_internal_recur_collect_dependencies( ${_project} ${_value})
    endforeach()
    get_property(_prjdeps GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_project})
    foreach( _value IN LISTS _prjdeps)
        get_property(_host GLOBAL PROPERTY NCBI_PTBPROP_HOST_${_value})
        if ("${_host}" STREQUAL "")
            list(APPEND _deps ${_value})
        else()
            list(APPEND _deps ${_host})
        endif()
    endforeach()
    if (NOT "${_deps}" STREQUAL "")
        list(REMOVE_DUPLICATES _deps)
    endif()
    set_property(GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_project} "${_deps}")
endfunction()

##############################################################################
function(NCBI_internal_recur_collect_dependencies _project _dep)
    get_property(_prjdeps GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_project})
    get_property(_depdeps GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_dep} SET)
    if (NOT _depdeps)
        message(WARNING "ERROR: project ${_dep} not found")
    endif()
    get_property(_depdeps GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_dep})
    set_property(GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${_project} ${_prjdeps} ${_depdeps})
    foreach( _value IN LISTS _depdeps)
        list(FIND _prjdeps ${_value} _found)
        if (${_found} LESS "0")
            NCBI_internal_recur_collect_dependencies( ${_project} ${_value})
        endif()
    endforeach()
endfunction()

##############################################################################
function(NCBI_internal_include)
    include(${ARGV0})
endfunction()

##############################################################################
function(NCBI_internal_collect_sources)
    set(_dir ${NCBI_CURRENT_SOURCE_DIR})
    if(NCBI_PTBMODE_PARTS)
        set(_prefix "Hosted Libraries\\${NCBI_PROJECT_PARTNAME}\\")
    else()
        set(_prefix "")
    endif()

    foreach(_file IN LISTS NCBI_${NCBI_PROJECT}_NOPCH)
        if(EXISTS ${_dir}/${_file}.cpp)
            list(APPEND _nopch "${_dir}/${_file}.cpp")
        elseif(EXISTS ${_dir}/${_file}.c)
            list(APPEND _nopch   "${_dir}/${_file}.c")
        elseif(EXISTS ${_dir}/${_file})
            list(APPEND _nopch   "${_dir}/${_file}")
        else()
            list(APPEND _nopch   "${_file}")
        endif()
    endforeach()

    foreach(_file IN LISTS NCBI_${NCBI_PROJECT}_SOURCES)
        if(EXISTS ${_dir}/${_file}.cpp)
            list(APPEND _sources "${_dir}/${_file}.cpp")
        elseif(EXISTS ${_dir}/${_file}.c)
            list(APPEND _sources "${_dir}/${_file}.c")
            list(APPEND _nopch   "${_dir}/${_file}.c")
        else()
            get_filename_component(_ext ${_file} EXT)
            if(EXISTS ${_file} OR IS_ABSOLUTE ${_file})
                list(APPEND _sources "${_file}")
                if("${_ext}" STREQUAL ".c")
                    list(APPEND _nopch "${_file}")
                endif()
            else()
                list(APPEND _sources "${_dir}/${_file}")
                if("${_ext}" STREQUAL ".c")
                    list(APPEND _nopch "${_dir}/${_file}")
                endif()
            endif()
        endif()
    endforeach()

    if (NOT NCBI_PTBMODE_PARTS AND DEFINED NCBI_DEFAULT_DLLENTRY)
        if (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "SHARED")
            list(APPEND _sources ${NCBI_DEFAULT_DLLENTRY})
        endif()
    endif()

    if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
        message("create source group  ${_prefix}Source Files")
        message("${_sources}")
    endif()
    source_group("${_prefix}Source Files"   FILES ${_sources})
    set(NCBITMP_PROJECT_SOURCES ${NCBITMP_PROJECT_SOURCES} ${_sources} PARENT_SCOPE)
    set(NCBITMP_PROJECT_NOPCH   ${NCBITMP_PROJECT_NOPCH}   ${_nopch}   PARENT_SCOPE)
endfunction()

##############################################################################
function(NCBI_internal_collect_headers)

    file(RELATIVE_PATH _rel "${NCBI_SRC_ROOT}" "${NCBI_CURRENT_SOURCE_DIR}")
    set(_inc_dir ${NCBI_INC_ROOT}/${_rel})
    if(NCBI_PTBMODE_PARTS)
        set(_prefix "Hosted Libraries\\${NCBI_PROJECT_PARTNAME}\\")
    else()
        set(_prefix "")
    endif()
    if (DEFINED NCBI_${NCBI_PROJECT}_HEADERS)
        set(_headers ${NCBI_${NCBI_PROJECT}_HEADERS})
    elseif(DEFINED NCBI__HEADERS)
        set(_headers ${NCBI__HEADERS})
    else()
        set(_headers *.h* *impl/*.h* *.inl *impl/*.inl)
    endif()

    if (MSVC OR XCODE)
        set(_priv "")
        set(_pub "")
        set(_inl "")
        if ( NOT "${_headers}" STREQUAL "")
            foreach(_mask ${_headers})
                file(GLOB _files LIST_DIRECTORIES false "${NCBI_CURRENT_SOURCE_DIR}/${_mask}")
                set(_priv ${_priv} ${_files})
                if (${_mask} MATCHES ".inl$")
                    file(GLOB _files LIST_DIRECTORIES false "${_inc_dir}/${_mask}")
                    set(_inl ${_inl} ${_files})
                else()
                    file(GLOB _files LIST_DIRECTORIES false "${_inc_dir}/${_mask}")
                    set(_pub ${_pub} ${_files})
                endif()
            endforeach()

            if ( NOT "${_priv}" STREQUAL "")
                source_group("${_prefix}Private Headers" FILES ${_priv})
            endif()
            if ( NOT "${_pub}" STREQUAL "")
                source_group("${_prefix}Header Files"    FILES ${_pub})
            endif()
            if ( NOT "${_inl}" STREQUAL "")
                source_group("${_prefix}Inline Files"    FILES ${_inl})
            endif()
        endif()
        set(NCBITMP_PROJECT_HEADERS ${NCBITMP_PROJECT_HEADERS} ${_priv} ${_pub} ${_inl} PARENT_SCOPE)
    else()
        set(NCBITMP_PROJECT_HEADERS "" PARENT_SCOPE)
    endif()
endfunction()

##############################################################################
function(NCBI_internal_add_resources) 
    if (MSVC)
        if (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "CONSOLEAPP")
            if (DEFINED NCBI_${NCBI_PROJECT}_RESOURCES)
                set(_res ${NCBI_${NCBI_PROJECT}_RESOURCES})
            elseif (DEFINED NCBI__RESOURCES)
                set(_res ${NCBI__RESOURCES})
            else()
                set(_res ${NCBI_DEFAULT_RESOURCES})
            endif()
        else()
            if (DEFINED NCBI_${NCBI_PROJECT}_RESOURCES)
                set(_res ${NCBI_${NCBI_PROJECT}_RESOURCES})
            endif()
        endif()

        if(NCBI_PTBMODE_PARTS)
            set(_prefix "Hosted Libraries\\${NCBI_PROJECT_PARTNAME}\\")
        else()
            set(_prefix "")
        endif()
        if ( NOT "${_res}" STREQUAL "")
            source_group("${_prefix}Resource Files" FILES ${_res})
        endif()
        set(NCBITMP_PROJECT_RESOURCES ${_res} PARENT_SCOPE)
    else()
        set(NCBITMP_PROJECT_RESOURCES "" PARENT_SCOPE)
    endif()
endfunction()

##############################################################################
function(NCBI_internal_collect_dataspec) 
    if (NOT DEFINED NCBI_${NCBI_PROJECT}_DATASPEC)
        return()
    endif()

    if(NCBI_PTBMODE_PARTS)
        set(_prefix "Hosted Libraries\\${NCBI_PROJECT_PARTNAME}\\")
    else()
        set(_prefix "")
    endif()

    foreach(_dataspec IN LISTS NCBI_${NCBI_PROJECT}_DATASPEC)
        if (EXISTS ${NCBI_CURRENT_SOURCE_DIR}/${_dataspec})
            get_filename_component(_basename ${_dataspec} NAME_WE)
            get_filename_component(_path ${NCBI_CURRENT_SOURCE_DIR}/${_dataspec} DIRECTORY)
            set(_specfiles ${_specfiles}  ${NCBI_CURRENT_SOURCE_DIR}/${_dataspec})
            set_source_files_properties(${_path}/${_basename}__.cpp ${_path}/${_basename}___.cpp PROPERTIES GENERATED TRUE)
            set(_srcfiles ${_srcfiles}  ${_path}/${_basename}__.cpp ${_path}/${_basename}___.cpp)
        else()
            message(WARNING "ERROR: file not found: ${NCBI_CURRENT_SOURCE_DIR}/${_dataspec}")
        endif()
    endforeach()

    source_group("${_prefix}Source Files"   FILES ${_srcfiles})
    set(NCBITMP_PROJECT_SOURCES ${NCBITMP_PROJECT_SOURCES} ${_srcfiles} PARENT_SCOPE)
    source_group("${_prefix}DataSpec Files" FILES ${_specfiles})
    set(NCBITMP_PROJECT_DATASPEC ${NCBITMP_PROJECT_DATASPEC} ${_specfiles} PARENT_SCOPE)
endfunction()

##############################################################################
function(NCBI_internal_add_dataspec) 
    if (NOT DEFINED NCBITMP_PROJECT_DATASPEC)
        return()
    endif()

    foreach(_dataspec IN LISTS NCBITMP_PROJECT_DATASPEC)
        get_filename_component(_basename ${_dataspec} NAME_WE)
        get_filename_component(_ext ${_dataspec} EXT)
        set(_filepath ${_dataspec})
        get_filename_component(_path ${_filepath} DIRECTORY)
        file(RELATIVE_PATH _relpath ${NCBI_SRC_ROOT} ${_path})
        set(_module_imports "")
        set(_imports "")

        if(EXISTS "${_path}/${_basename}.module")
            FILE(READ "${_path}/${_basename}.module" _module_contents)
            STRING(REGEX MATCH "MODULE_IMPORT *=[^\n]*[^ \n]" _tmp "${_module_contents}")
            STRING(REGEX REPLACE "MODULE_IMPORT *= *" "" _tmp "${_tmp}")
            STRING(REGEX REPLACE "  *$" "" _imp_list "${_tmp}")
            STRING(REGEX REPLACE " " ";" _imp_list "${_imp_list}")

            foreach(_module IN LISTS _imp_list)
                set(_module_imports "${_module_imports} ${_module}${_ext}")
            endforeach()
            if (NOT "${_module_imports}" STREQUAL "")
                set(_imports -M ${_module_imports})
            endif()
        endif()

        set(_oc ${_basename})
        if (NOT "${NCBI_DEFAULT_PCH}" STREQUAL "")
            set(_pch -pch ${NCBI_DEFAULT_PCH})
        endif()
        set(_od ${_path}/${_basename}.def)
        set(_oex -oex " ")
        set(_cmd ${NCBI_DATATOOL} ${_oex} ${_pch} -m ${_filepath} -oA -oc ${_oc} -od ${_od} -odi -ocvs -or ${_relpath} -oR ${top_src_dir} ${_imports})
        add_custom_command(
            OUTPUT ${_path}/${_basename}__.cpp ${_path}/${_basename}___.cpp
            COMMAND ${_cmd} VERBATIM
            WORKING_DIRECTORY ${top_src_dir}
            COMMENT "Generate C++ classes from ${_filepath}"
            DEPENDS ${NCBI_DATATOOL}
            VERBATIM
        )
    endforeach()
endfunction()

##############################################################################
function(NCBI_internal_define_precompiled_header_usage)

    if (DEFINED NCBI_${NCBI_PROJECT}_USEPCH)
        set(_usepch ${NCBI_${NCBI_PROJECT}_USEPCH})
    elseif (DEFINED NCBI__USEPCH)
        set(_usepch ${NCBI__USEPCH})
    else()
        set(_usepch ${NCBI_DEFAULT_USEPCH})
    endif()

    if (DEFINED NCBI_${NCBI_PROJECT}_PCH)
        set(_pch ${NCBI_${NCBI_PROJECT}_PCH})
    elseif (DEFINED NCBI__PCH)
        set(_pch ${NCBI__PCH})
    else()
        set(_pch ${NCBI_DEFAULT_PCH})
    endif()

    if (DEFINED NCBI_${NCBI_PROJECT}_PCH_DEFINE)
        set(_pchdef ${NCBI_${NCBI_PROJECT}_PCH_DEFINE})
    elseif (DEFINED NCBI__PCH_DEFINE)
        set(_pchdef ${NCBI__PCH_DEFINE})
    else()
        set(_pchdef ${NCBI_DEFAULT_PCH_DEFINE})
    endif()

if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
    message("_usepch ${_usepch}:  prj=${NCBI_${NCBI_PROJECT}_USEPCH} dir=${NCBI__USEPCH} def=${NCBI_DEFAULT_USEPCH}")
    message("_pch ${_pch}:  prj=${NCBI_${NCBI_PROJECT}_PCH} dir=${NCBI__PCH} def=${NCBI_DEFAULT_PCH}")
    message("_pchdef ${_pchdef}")
endif()

    if (NOT DEFINED NCBITMP_PROJECT_SOURCES)
        set(_usepch OFF)
    endif()

    if (MSVC)
        if (_usepch)
            set_target_properties(${NCBI_PROJECT} PROPERTIES COMPILE_FLAGS "/Yu${_pch}")
            set(_files ${NCBITMP_PROJECT_SOURCES})
            if (DEFINED NCBITMP_PROJECT_NOPCH)
                foreach(_file ${NCBITMP_PROJECT_NOPCH})
                    list(FIND _files ${_file} _found)
                    if (${_found} LESS "0")
                        list(FIND _files ${_file}.cpp _found)
                        if (${_found} LESS "0")
                            list(FIND _files ${_file}.c _found)
                            if (${_found} LESS "0")
                                continue()
                            endif()
                            set(_file ${_file}.c)
                        else()
                            set(_file ${_file}.cpp)
                        endif()
                    endif()
                    list(REMOVE_ITEM _files ${_file})
                    set_source_files_properties(${_file} PROPERTIES COMPILE_FLAGS "/Y-")
                endforeach()
            endif()

            list(GET _files 0 _pchfile)
            get_filename_component(_pchname ${_pchfile} NAME)
            foreach(_file ${_files})
                get_filename_component(_name ${_file} NAME)
                if (${_name} STRLESS ${_pchname})
                    set(_pchname ${_name})
                    set(_pchfile ${_file})
                endif()
                set_source_files_properties(${_file} PROPERTIES COMPILE_DEFINITIONS ${_pchdef})
            endforeach()
            set_source_files_properties(${_pchfile} PROPERTIES COMPILE_FLAGS "/Yc${_pch}")
        endif(_usepch)
    endif (MSVC)
endfunction()

##############################################################################
macro(NCBI_internal_process_project_requires)
    set(NCBITMP_REQUIRE_NOTFOUND "")
    if(NCBI_PTBMODE_PARTS)
        set(_all ${NCBI__REQUIRES} ${NCBI_${NCBI_PROJECT}_REQUIRES})
    else()
        set(_all ${NCBI__REQUIRES} ${NCBI_${NCBI_PROJECT}_REQUIRES} ${NCBITMP_PROJECT_REQUIRES})
    endif()
    if (NOT "${_all}" STREQUAL "")
        list(REMOVE_DUPLICATES _all)
    endif()

    foreach(_req IN LISTS _all)
        string(SUBSTRING ${_req} 0 1 _sign)
        if ("${_sign}" STREQUAL "-")
            string(SUBSTRING ${_req} 1 -1 _value)
            set(_negate ON)
        else()
            set(_value ${_req})
            set(_negate OFF)
        endif()
        if (NCBI_REQUIRE_${_value}_FOUND OR NCBI_COMPONENT_${_value}_FOUND)
            if (_negate)
                set(NCBITMP_REQUIRE_NOTFOUND ${NCBITMP_REQUIRE_NOTFOUND} ${_req})
            else()
                set(NCBITMP_INCLUDES ${NCBITMP_INCLUDES} ${NCBI_COMPONENT_${_value}_INCLUDE})
                set(NCBITMP_DEFINES  ${NCBITMP_DEFINES}  ${NCBI_COMPONENT_${_value}_DEFINES})
                set(NCBITMP_EXTLIB   ${NCBITMP_EXTLIB}   ${NCBI_COMPONENT_${_value}_LIBS})
                foreach(_lib IN LISTS NCBI_COMPONENT_${_value}_NCBILIB)
                    if(NOT ${_lib} STREQUAL ${NCBI_${NCBI_PROJECT}_OUTPUT})
                        set(NCBITMP_NCBILIB  ${NCBITMP_NCBILIB}  ${_lib})
                    endif()
                endforeach()
            endif()
        else()
            if (_negate)
# no-no, this is a mistake
#        set(NCBITMP_INCLUDES ${NCBITMP_INCLUDES} ${NCBI_COMPONENT_${_value}_INCLUDE})
#        set(NCBITMP_DEFINES  ${NCBITMP_DEFINES}  ${NCBI_COMPONENT_${_value}_DEFINES})
#        set(NCBITMP_EXTLIB     ${NCBITMP_EXTLIB}     ${NCBI_COMPONENT_${_value}_LIBS})
            else()
                set(NCBITMP_REQUIRE_NOTFOUND ${NCBITMP_REQUIRE_NOTFOUND} ${_req})
            endif()
        endif()     
    endforeach()
endmacro()

##############################################################################
macro(NCBI_internal_process_project_components)
    set(NCBITMP_COMPONENT_NOTFOUND "")
    if(NCBI_PTBMODE_PARTS)
        set(_all ${NCBI__COMPONENTS} ${NCBI_${NCBI_PROJECT}_COMPONENTS})
    else()
        set(_all ${NCBI__COMPONENTS} ${NCBI_${NCBI_PROJECT}_COMPONENTS} ${NCBITMP_PROJECT_COMPONENTS})
    endif()
    if (NOT "${_all}" STREQUAL "")
        list(REMOVE_DUPLICATES _all)
    endif()

    foreach(_value IN LISTS _all)
        if (NCBI_REQUIRE_${_value}_FOUND OR NCBI_COMPONENT_${_value}_FOUND)
            set(NCBITMP_INCLUDES ${NCBITMP_INCLUDES} ${NCBI_COMPONENT_${_value}_INCLUDE})
            set(NCBITMP_DEFINES  ${NCBITMP_DEFINES}  ${NCBI_COMPONENT_${_value}_DEFINES})
            set(NCBITMP_EXTLIB   ${NCBITMP_EXTLIB}   ${NCBI_COMPONENT_${_value}_LIBS})
            foreach(_lib IN LISTS NCBI_COMPONENT_${_value}_NCBILIB)
                if(NOT ${_lib} STREQUAL ${NCBI_${NCBI_PROJECT}_OUTPUT})
                    set(NCBITMP_NCBILIB  ${NCBITMP_NCBILIB}  ${_lib})
                endif()
            endforeach()
        else()
            set(NCBITMP_COMPONENT_NOTFOUND ${NCBITMP_COMPONENT_NOTFOUND} ${_value})
        endif()
    endforeach()
endmacro()

##############################################################################
function(NCBI_internal_collect_parts)
    set(NCBI_PTBMODE_PARTS ON)
    set(_hostproject ${NCBI_PROJECT})

    foreach(_part IN LISTS NCBI_${_hostproject}_PARTS)
        set(_filepath ${NCBI_SRC_ROOT}/${_part})
        get_filename_component(_path ${_filepath} DIRECTORY)
        get_filename_component(_lib ${_filepath} NAME)

        unset(NCBI_PROJECT)
        unset(NCBI_PROJECT_ID)
        unset(NCBI__USEPCH)
        unset(NCBI__REQUIRES)
        unset(NCBI__COMPONENTS)
        unset(NCBI__INCLUDES)
        unset(NCBI__DEFINES)
        unset(NCBI__NCBILIB)
        unset(NCBI__EXTLIB)

        file(RELATIVE_PATH _relpath ${NCBI_SRC_ROOT} ${_path})
        string(REPLACE "/" ";" _dirlist ${_relpath})
        set(NCBI_CURRENT_SOURCE_DIR ${NCBI_SRC_ROOT})
        foreach(_dir ${_dirlist})
            set(NCBI_CURRENT_SOURCE_DIR ${NCBI_CURRENT_SOURCE_DIR}/${_dir})
            include(${NCBI_CURRENT_SOURCE_DIR}/CMakeLists.txt)
        endforeach()
        set(NCBI_CURRENT_SOURCE_DIR ${_path})

        if (EXISTS ${_path}/CMakeLists.${_lib}.txt)
            include(${_path}/CMakeLists.${_lib}.txt)
        elseif (EXISTS ${_path}/CMakeLists.${_lib}.lib.txt)
            include(${_path}/CMakeLists.${_lib}.lib.txt)
        elseif (EXISTS ${_path}/CMakeLists.${_lib}.asn.txt)
            include(${_path}/CMakeLists.${_lib}.asn.txt)
        else()
            message(WARNING "ERROR: project part not found: ${NCBI_SRC_ROOT}/${_part}")
        endif()
    endforeach()

    set(NCBITMP_PROJECT_PART_IDS    ${NCBITMP_PROJECT_PART_IDS}   PARENT_SCOPE)
    set(NCBITMP_PROJECT_PARTS       ${NCBITMP_PROJECT_PARTS}      PARENT_SCOPE)
    set(NCBITMP_PROJECT_REQUIRES    ${NCBITMP_PROJECT_REQUIRES}   PARENT_SCOPE)
    set(NCBITMP_PROJECT_COMPONENTS  ${NCBITMP_PROJECT_COMPONENTS} PARENT_SCOPE)
    set(NCBITMP_PROJECT_INCLUDES    ${NCBITMP_PROJECT_INCLUDES}   PARENT_SCOPE)
    set(NCBITMP_PROJECT_DEFINES     ${NCBITMP_PROJECT_DEFINES}    PARENT_SCOPE)
    set(NCBITMP_PROJECT_NCBILIB     ${NCBITMP_PROJECT_NCBILIB}    PARENT_SCOPE)
    set(NCBITMP_PROJECT_EXTLIB      ${NCBITMP_PROJECT_EXTLIB}     PARENT_SCOPE)
    set(NCBITMP_PROJECT_SOURCES     ${NCBITMP_PROJECT_SOURCES}    PARENT_SCOPE)
    set(NCBITMP_PROJECT_NOPCH       ${NCBITMP_PROJECT_NOPCH}      PARENT_SCOPE)
    set(NCBITMP_PROJECT_DATASPEC    ${NCBITMP_PROJECT_DATASPEC}   PARENT_SCOPE)
    set(NCBITMP_PROJECT_HEADERS     ${NCBITMP_PROJECT_HEADERS}    PARENT_SCOPE)
endfunction()

##############################################################################
macro(NCBI_internal_process_parts)

    set(NCBITMP_PROJECT_PART_IDS "")
    set(NCBITMP_PROJECT_PARTS "")
    NCBI_internal_collect_parts()

    if (NCBI_PTBMODE_COLLECT_DEPS)
#set_property(GLOBAL PROPERTY NCBI_PTBPROP_PARTS_${NCBI_PROJECT_ID} ${NCBITMP_PROJECT_PART_IDS})
        foreach(_part IN LISTS NCBITMP_PROJECT_PART_IDS)
            set_property(GLOBAL PROPERTY NCBI_PTBPROP_HOSTID_${_part} ${NCBI_PROJECT_ID})
        endforeach()
        foreach(_part IN LISTS NCBITMP_PROJECT_PARTS NCBI_${NCBI_PROJECT}_VIRTPARTS)
            if (NOT "${_part}" STREQUAL ${NCBI_PROJECT})
                set_property(GLOBAL PROPERTY NCBI_PTBPROP_HOST_${_part} ${NCBI_PROJECT})
            endif()
        endforeach()
    endif()
endmacro()

##############################################################################
function(NCBI_internal_verify_libs)
    set(_ncbilib ${NCBITMP_NCBILIB})
    if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
        message("NCBI_internal_verify_libs ${NCBI_PROJECT_ID}:  on enter = ${_ncbilib}")
    endif()
    set(_value "")
    if ( NOT "${_ncbilib}" STREQUAL "")
        list(REMOVE_DUPLICATES _ncbilib)
        foreach(_prj IN LISTS _ncbilib)
            get_property(_host GLOBAL PROPERTY NCBI_PTBPROP_HOST_${_prj})
            if ("${_host}" STREQUAL "")
                set(_value ${_value} ${_prj})
            else()
                set(_value ${_value} ${_host})
            endif()
        endforeach()
    endif()
    if ( NOT "${_value}" STREQUAL "")
        list(REMOVE_DUPLICATES _value)
        list(REMOVE_ITEM _value ${NCBI_PROJECT})
    endif()

    set(_tk_libs "${_value}")
    set(_value "")
    foreach(_tk_lib IN LISTS _tk_libs)
        if(${_tk_lib} STREQUAL "general")
            if(NCBI_EXPERIMENTAL_CFG)
                set(_value ${_value} \$<1:general>)
            else()
                set(_value ${_value} general-lib)
            endif()
        else()
            set(_value ${_value} ${_tk_lib})
        endif()
    endforeach()

    if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
        message("NCBI_internal_verify_libs ${NCBI_PROJECT_ID}:  on exit = ${_value}")
    endif()
    set(NCBITMP_NCBILIB ${_value} PARENT_SCOPE)
endfunction()

##############################################################################
function(NCBI_internal_process_project_filters _result)
    if(DEFINED NCBI_PTBCFG_PROJECT_LIST)
        foreach(_dir IN LISTS NCBI_PTBCFG_PROJECT_LIST)
            if(${NCBI_CURRENT_SOURCE_DIR} MATCHES ${NCBI_SRC_ROOT}/${_dir})
                set(${_result} TRUE PARENT_SCOPE)
                return()
            endif()
        endforeach()
    endif()
    set(${_result} FALSE PARENT_SCOPE)
endfunction()

##############################################################################
macro(NCBI_internal_process_test_requires _test)
    set(NCBITEST_REQUIRE_NOTFOUND "")
    set(_all ${NCBITEST__REQUIRES} ${NCBITEST_${_test}_REQUIRES})
    if (NOT "${_all}" STREQUAL "")
        list(REMOVE_DUPLICATES _all)
    endif()

    foreach(_req IN LISTS _all)
        string(SUBSTRING ${_req} 0 1 _sign)
        if ("${_sign}" STREQUAL "-")
            string(SUBSTRING ${_req} 1 -1 _value)
            set(_negate ON)
        else()
            set(_value ${_req})
            set(_negate OFF)
        endif()
        if (NCBI_REQUIRE_${_value}_FOUND OR NCBI_COMPONENT_${_value}_FOUND)
            if (_negate)
                set(NCBITEST_REQUIRE_NOTFOUND ${NCBITEST_REQUIRE_NOTFOUND} ${_req})
            endif()
        else()
            if (NOT _negate)
                set(NCBITEST_REQUIRE_NOTFOUND ${NCBITEST_REQUIRE_NOTFOUND} ${_req})
            endif()
        endif()     
    endforeach()
endmacro()

##############################################################################
function(NCBI_internal_add_test _test)
    if( NOT DEFINED NCBITEST_${_test}_CMD)
        set(NCBITEST_${_test}_CMD ${NCBI_${NCBI_PROJECT}_OUTPUT})
    endif()
    get_filename_component(_ext ${NCBITEST_${_test}_CMD} EXT)
    if("${_ext}" STREQUAL ".sh")
        set(NCBITEST_${_test}_REQUIRES ${NCBITEST_${_test}_REQUIRES} -MSWin)
        set(NCBITEST_${_test}_ASSETS ${NCBITEST__ASSETS} ${NCBITEST_${_test}_ASSETS} ${NCBITEST_${_test}_CMD})
    endif()

    NCBI_internal_process_test_requires(${_test})
    if ( NOT "${NCBITEST_REQUIRE_NOTFOUND}" STREQUAL "")
        message("${NCBI_CURRENT_SOURCE_DIR}/${NCBI_PROJECT}: Test ${_test} is excluded because of unmet requirements: ${NCBITEST_REQUIRE_NOTFOUND}")
        return()
    endif()

    if (DEFINED NCBITEST_${_test}_ASSETS)
        set(_assets ${NCBITEST_${_test}_ASSETS})
    elseif(DEFINED NCBITEST__ASSETS)
        set(_assets ${NCBITEST__ASSETS})
    endif()
    if (DEFINED NCBITEST_${_test}_TIMEOUT)
        set(_timeout ${NCBITEST_${_test}_TIMEOUT})
    elseif(DEFINED NCBITEST__TIMEOUT)
        set(_timeout ${NCBITEST__TIMEOUT})
    else()
        set(_timeout 86400)
    endif()
    string(REPLACE ";" " " _args    "${NCBITEST_${_test}_ARG}")
    string(REPLACE ";" " " _assets   "${_assets}")

    add_test(NAME ${_test} COMMAND ${CMAKE_COMMAND}
        -DNCBITEST_NAME=${_test}
        -DNCBITEST_CONFIG=$<CONFIG>
        -DNCBITEST_COMMAND=${NCBITEST_${_test}_CMD}
        -DNCBITEST_ARGS=${_args}
        -DNCBITEST_TIMEOUT=${_timeout}
        -DNCBITEST_BINDIR=${CMAKE_RUNTIME_OUTPUT_DIRECTORY}
        -DNCBITEST_SOURCEDIR=${NCBI_CURRENT_SOURCE_DIR}
        -DNCBITEST_ASSETS=${_assets}
        -P "${NCBITEST_DRIVER}")
endfunction()

##############################################################################
function(NCBI_internal_add_project)

    if (NOT NCBI_PTBMODE_PARTS AND NOT NCBI_PTBMODE_COLLECT_DEPS AND NCBI_PTBCFG_ENABLE_COLLECTOR)
        if(DEFINED NCBI_PTB_ALLOWED_PROJECTS)
            list(FIND NCBI_PTB_ALLOWED_PROJECTS ${NCBI_PROJECT} _found)
            if (${_found} LESS "0")
                if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
                    message("${NCBI_CURRENT_SOURCE_DIR}/${NCBI_PROJECT} is excluded by user's request")
                endif()
                if ("${ARGC}" GREATER "0")
                    set(${ARGV0} FALSE PARENT_SCOPE)
                endif()
                return()
            endif()
        endif()
        get_property(_hosted GLOBAL PROPERTY NCBI_PTBPROP_HOSTID_${NCBI_PROJECT_ID})
        if ("${_hosted}" STREQUAL "")
            get_property(_hosted GLOBAL PROPERTY NCBI_PTBPROP_HOST_${NCBI_PROJECT})
        endif()

        if (NOT "${_hosted}" STREQUAL "")
            if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
                message("${NCBI_CURRENT_SOURCE_DIR}/${NCBI_PROJECT_ID} is excluded because it is part of ${_hosted}")
            endif()
            if ("${ARGC}" GREATER "0")
                set(${ARGV0} FALSE PARENT_SCOPE)
            endif()
            return()
        endif()
    endif()

if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
message("-----------------------------------")
message("NCBI_PROJECT = ${NCBI_PROJECT}")
message("  TYPE = ${NCBI_${NCBI_PROJECT}_TYPE}")
message("  SOURCES = ${NCBI_${NCBI_PROJECT}_SOURCES}")
message("  RESOURCES = ${NCBI_${NCBI_PROJECT}_RESOURCES}")
message("  HEADERS = ${NCBI_${NCBI_PROJECT}_HEADERS}")
message("  REQUIRES = ${NCBI_${NCBI_PROJECT}_REQUIRES}")
message("  COMPONENTS = ${NCBI_${NCBI_PROJECT}_COMPONENTS}")
message("  NOPCH = ${NCBI_${NCBI_PROJECT}_NOPCH}")
message("  NCBILIB = ${NCBI_${NCBI_PROJECT}_NCBILIB}")
message("  EXTLIB = ${NCBI_${NCBI_PROJECT}_EXTLIB}")
message("  INCLUDES = ${NCBI_${NCBI_PROJECT}_INCLUDES}")
message("  DEFINES = ${NCBI_${NCBI_PROJECT}_DEFINES}")
endif()

    if (DEFINED NCBI_${NCBI_PROJECT}_PARTS)
        NCBI_internal_process_parts()
    endif()

    set(NCBITMP_INCLUDES ${NCBITMP_PROJECT_INCLUDES} ${NCBI__INCLUDES} ${NCBI_${NCBI_PROJECT}_INCLUDES})
    set(NCBITMP_DEFINES  ${NCBITMP_PROJECT_DEFINES}  ${NCBI__DEFINES}  ${NCBI_${NCBI_PROJECT}_DEFINES})
    set(NCBITMP_NCBILIB  ${NCBITMP_PROJECT_NCBILIB}  ${NCBI__NCBILIB}  ${NCBI_${NCBI_PROJECT}_NCBILIB})
    set(NCBITMP_EXTLIB   ${NCBITMP_PROJECT_EXTLIB}   ${NCBI__EXTLIB}   ${NCBI_${NCBI_PROJECT}_EXTLIB})

    NCBI_internal_process_project_requires()
    if ( NOT "${NCBITMP_REQUIRE_NOTFOUND}" STREQUAL "" AND NOT NCBI_PTBMODE_COLLECT_DEPS)
        message("${NCBI_CURRENT_SOURCE_DIR}/${NCBI_PROJECT} is excluded because of unmet requirements: ${NCBITMP_REQUIRE_NOTFOUND}")
        if (NOT NCBI_PTBMODE_PARTS)
            if ("${ARGC}" GREATER "0")
                set(${ARGV0} FALSE PARENT_SCOPE)
            endif()
            return()
        endif()
    endif()

    NCBI_internal_process_project_components()
    if ( NOT "${NCBITMP_COMPONENT_NOTFOUND}" STREQUAL "" AND NOT NCBI_PTBMODE_COLLECT_DEPS)
        message("${NCBI_CURRENT_SOURCE_DIR}/${NCBI_PROJECT}: cannot find optional component: ${NCBITMP_COMPONENT_NOTFOUND}")
    endif()
    if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
        message("NCBITMP_INCLUDES = ${NCBITMP_INCLUDES}")
        message("NCBITMP_DEFINES = ${NCBITMP_DEFINES}")
        message("NCBITMP_NCBILIB = ${NCBITMP_NCBILIB}")
        message("NCBITMP_EXTLIB = ${NCBITMP_EXTLIB}")
    endif()

    if (NCBI_PTBMODE_COLLECT_DEPS)
        if (NCBI_PTBMODE_PARTS)
            set(NCBITMP_PROJECT_PART_IDS  ${NCBITMP_PROJECT_PART_IDS}  ${NCBI_PROJECT_ID}       PARENT_SCOPE )
            set(NCBITMP_PROJECT_PARTS     ${NCBITMP_PROJECT_PARTS}     ${NCBI_PROJECT_PARTNAME} PARENT_SCOPE )
        endif()
        set_property(GLOBAL PROPERTY NCBI_PTBPROP_DEPS_${NCBI_PROJECT} "${NCBITMP_NCBILIB}")
        if(NOT NCBI_PTBMODE_PARTS)
            get_property(_allprojects GLOBAL PROPERTY NCBI_PTBPROP_ALL_PROJECTS)
            set(_allprojects ${_allprojects} ${NCBI_PROJECT})
            set_property(GLOBAL PROPERTY NCBI_PTBPROP_ALL_PROJECTS ${_allprojects})

            NCBI_internal_process_project_filters(_allowed)
            if (_allowed)
                get_property(_allowedprojects GLOBAL PROPERTY NCBI_PTBPROP_ALLOWED_PROJECTS)
                set(_allowedprojects ${_allowedprojects} ${NCBI_PROJECT})
                set_property(GLOBAL PROPERTY NCBI_PTBPROP_ALLOWED_PROJECTS ${_allowedprojects})
            endif()
        endif()
        if ("${ARGC}" GREATER "0")
            set(${ARGV0} FALSE PARENT_SCOPE)
        endif()
        return()
    endif()

#message("processing ${NCBI_PROJECT_ID}")
    NCBI_internal_collect_sources() 
    NCBI_internal_collect_dataspec()
    NCBI_internal_collect_headers() 
    if (NCBI_PTBMODE_PARTS)
        set(NCBITMP_PROJECT_REQUIRES    ${NCBITMP_PROJECT_REQUIRES}   ${NCBI__REQUIRES}   ${NCBI_${NCBI_PROJECT}_REQUIRES}    PARENT_SCOPE)
        set(NCBITMP_PROJECT_COMPONENTS  ${NCBITMP_PROJECT_COMPONENTS} ${NCBI__COMPONENTS} ${NCBI_${NCBI_PROJECT}_COMPONENTS}  PARENT_SCOPE)
        set(NCBITMP_PROJECT_INCLUDES    ${NCBITMP_PROJECT_INCLUDES}   ${NCBI__INCLUDES}   ${NCBI_${NCBI_PROJECT}_INCLUDES}    PARENT_SCOPE)
        set(NCBITMP_PROJECT_DEFINES     ${NCBITMP_PROJECT_DEFINES}    ${NCBI__DEFINES}    ${NCBI_${NCBI_PROJECT}_DEFINES}     PARENT_SCOPE)
        set(NCBITMP_PROJECT_NCBILIB     ${NCBITMP_PROJECT_NCBILIB}    ${NCBI__NCBILIB}    ${NCBI_${NCBI_PROJECT}_NCBILIB}     PARENT_SCOPE)
        set(NCBITMP_PROJECT_EXTLIB      ${NCBITMP_PROJECT_EXTLIB}     ${NCBI__EXTLIB}     ${NCBI_${NCBI_PROJECT}_EXTLIB}      PARENT_SCOPE)
        set(NCBITMP_PROJECT_SOURCES     ${NCBITMP_PROJECT_SOURCES}    PARENT_SCOPE)
        set(NCBITMP_PROJECT_NOPCH       ${NCBITMP_PROJECT_NOPCH}      PARENT_SCOPE)
        set(NCBITMP_PROJECT_DATASPEC    ${NCBITMP_PROJECT_DATASPEC}   PARENT_SCOPE)
        set(NCBITMP_PROJECT_HEADERS     ${NCBITMP_PROJECT_HEADERS}    PARENT_SCOPE)
        if ("${ARGC}" GREATER "0")
            set(${ARGV0} FALSE PARENT_SCOPE)
        endif()
        return()
    endif()
    NCBI_internal_add_resources()
    NCBI_internal_add_dataspec()
    NCBI_internal_verify_libs()

    if (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "STATIC")

        set(NCBITMP_DEFINES  ${NCBITMP_DEFINES} "_LIB")
#message("add static library(${NCBI_PROJECT} STATIC ${NCBITMP_PROJECT_SOURCES} ${NCBITMP_PROJECT_HEADERS} ${NCBITMP_PROJECT_RESOURCES} ${NCBITMP_PROJECT_DATASPEC})")
#message("add static library ${NCBI_PROJECT}")
        add_library(${NCBI_PROJECT} STATIC ${NCBITMP_PROJECT_SOURCES} ${NCBITMP_PROJECT_HEADERS} ${NCBITMP_PROJECT_RESOURCES} ${NCBITMP_PROJECT_DATASPEC})
        set(_suffix ${CMAKE_STATIC_LIBRARY_SUFFIX})

    elseif (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "SHARED")

        set(NCBITMP_DEFINES  ${NCBITMP_DEFINES} "_USRDLL")
#message("add shared library(${NCBI_PROJECT} SHARED ${NCBITMP_PROJECT_SOURCES} ${NCBITMP_PROJECT_HEADERS} ${NCBITMP_PROJECT_RESOURCES} ${NCBITMP_PROJECT_DATASPEC})")
#message("add shared library ${NCBI_PROJECT}")
        add_library(${NCBI_PROJECT} SHARED ${NCBITMP_PROJECT_SOURCES} ${NCBITMP_PROJECT_HEADERS} ${NCBITMP_PROJECT_RESOURCES} ${NCBITMP_PROJECT_DATASPEC})
        set(_suffix ${CMAKE_SHARED_LIBRARY_SUFFIX})

    elseif (${NCBI_${NCBI_PROJECT}_TYPE} STREQUAL "CONSOLEAPP")

        if(NCBI_EXPERIMENTAL_CFG)
            set(NCBITMP_DEFINES  ${NCBITMP_DEFINES} "_CONSOLE")
#message("add_executable(${NCBI_PROJECT} ${NCBITMP_PROJECT_SOURCES} ${NCBITMP_PROJECT_HEADERS} ${NCBITMP_PROJECT_RESOURCES} ${NCBITMP_PROJECT_DATASPEC})")
#message("add executable ${NCBI_PROJECT}")
            add_executable(${NCBI_PROJECT} ${NCBITMP_PROJECT_SOURCES} ${NCBITMP_PROJECT_HEADERS} ${NCBITMP_PROJECT_RESOURCES} ${NCBITMP_PROJECT_DATASPEC})
            set(_suffix ${CMAKE_EXECUTABLE_SUFFIX})
        else()
#message("add_executable(${NCBI_PROJECT} ${NCBITMP_PROJECT_SOURCES})")
            add_executable(${NCBI_PROJECT} ${NCBITMP_PROJECT_SOURCES})
        endif()

    else()

        message("${NCBI_CURRENT_SOURCE_DIR}/${NCBI_PROJECT} unsupported project type ${NCBI_${NCBI_PROJECT}_TYPE}")
        if ("${ARGC}" GREATER "0")
            set(${ARGV0} FALSE PARENT_SCOPE)
        endif()
        return()

    endif()
    if (DEFINED NCBI_${NCBI_PROJECT}_OUTPUT)
        set_target_properties(${NCBI_PROJECT} PROPERTIES OUTPUT_NAME ${NCBI_${NCBI_PROJECT}_OUTPUT})
    endif()

if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
message("${NCBI_CURRENT_SOURCE_DIR}/${NCBI_PROJECT_ID} added")
message("NCBITMP_PROJECT_SOURCES ${NCBITMP_PROJECT_SOURCES}")
message("NCBITMP_PROJECT_HEADERS ${NCBITMP_PROJECT_HEADERS}")
message("NCBITMP_PROJECT_RESOURCES ${NCBITMP_PROJECT_RESOURCES}")
endif()

    target_include_directories(${NCBI_PROJECT} PRIVATE ${NCBITMP_INCLUDES})
    target_compile_definitions(${NCBI_PROJECT} PRIVATE ${NCBITMP_DEFINES})
#message("target_link_libraries: ${NCBI_PROJECT}     ${NCBITMP_NCBILIB} ${NCBITMP_EXTLIB}")
    target_link_libraries(     ${NCBI_PROJECT}         ${NCBITMP_NCBILIB} ${NCBITMP_EXTLIB})

    if (DEFINED _suffix)
        set_target_properties( ${NCBI_PROJECT} PROPERTIES PROJECT_LABEL ${NCBI_PROJECT}${_suffix})
    endif()
    NCBI_internal_define_precompiled_header_usage()

    if (DEFINED NCBI_ALLTESTS)
        foreach(_test IN LISTS NCBI_ALLTESTS)
            NCBI_internal_add_test(${_test})
        endforeach()
    endif()

if(NCBI_VERBOSE_ALLPROJECTS OR NCBI_VERBOSE_PROJECT_${NCBI_PROJECT})
message("-----------------------------------")
endif()

    if ("${ARGC}" GREATER "0")
        set(${ARGV0} TRUE PARENT_SCOPE)
    endif()
endfunction()
