#include <assert.h>
#include <stdlib.h>
#include <stdio.h>

#include "lib/util.h"
#include "state.h"

using std::map;
using std::set;
using std::string;

inotify_state::inotify_state(const char* base_path)
	: inotify_fd(inotify_init()), base_path(base_path)
{
	die_if(inotify_fd < 0, "inotify_init");
}

inotify_state::~inotify_state()
{
	for (map<int, watcher*>::iterator it = watchers.begin();
		 it != watchers.end();
		 ++it)
		delete it->second;
	(void) close(inotify_fd);
}

static const char* multiinode_1 = "Encountered the same inode multiple times";
static const char* multiinode_2 = "Either there are multiple paths to the inode (not supported) or the underlying filesystem is quickly changing (restart)";

void inotify_state::add_imap_watcher(int wd, const string& imap_name, watcher* w)
{
	if (watchers.find(wd) != watchers.end())
	{
		fprintf(stderr, "%s, while starting to watch IMAP folder \"%s\". %s.\n", multiinode_1, imap_name.c_str(), multiinode_2);
		exit(1);	
	}

	watchers[wd] = w;
	new_wds.insert(wd);

	// allow multiple calls with the same 'imap_name' to allow multiple wds
	watcher* existing_imap = get_imap_watcher(imap_name);
	if (!existing_imap)
		imap2watcher[imap_name] = w;
	else
		assert(existing_imap == w);
}

void inotify_state::remove_imap_watcher(int wd, const string& imap_name)
{
	map<int, watcher*>::iterator itw = watchers.find(wd);
	assert(itw != watchers.end());
	(void) watchers.erase(itw);

	// allow multiple calls with the same 'imap_name' to allow multiple wds
	map<string, watcher*>::iterator itn = imap2watcher.find(imap_name);
	if (itn != imap2watcher.end())
		(void) imap2watcher.erase(itn);
}


watcher* inotify_state::get_watcher_by_wd(int wd)
{
	map<int, watcher*>::iterator it = watchers.find(wd);
	if (it == watchers.end())
		return NULL;
	return it->second;
}

watcher* inotify_state::get_imap_watcher(const string& imap_name)
{
	map<string, watcher*>::iterator it = imap2watcher.find(imap_name);
	if (it == imap2watcher.end())
		return NULL;
	return it->second;
}


bool inotify_state::is_new_wd(int wd) const
{
	return new_wds.find(wd) != new_wds.end();
}

void inotify_state::reset_new_wds()
{
	new_wds.clear();
}
