/****************************************************************************/
/* Log forwarder implementation - Forwarder executable                      */
/*                                                                          */
/*  Author(s):                                                              */
/*    - Gael Le Mahec   (gael.le.mahec@ens-lyon.fr)                         */
/*                                                                          */
/*  This file is part of DIET .                                             */
/*                                                                          */
/*  Copyright (C) 2000-2003 ENS Lyon, LIFC, INSA, INRIA and SysFera (2000)  */
/*                                                                          */
/*  - Frederic.Desprez@ens-lyon.fr (Project Manager)                        */
/*  - Eddy.Caron@ens-lyon.fr (Technical Manager)                            */
/*  - Tech@sysfera.com (Maintainer and Technical Support)                   */
/*                                                                          */
/*  This software is a computer program whose purpose is to provide an      */
/*  distributed logging services.                                           */
/*                                                                          */
/*                                                                          */
/*  This software is governed by the CeCILL license under French law and    */
/*  abiding by the rules of distribution of free software.  You can  use,   */
/*  modify and/ or redistribute the software under the terms of the CeCILL  */
/*  license as circulated by CEA, CNRS and INRIA at the following URL       */
/*  "http://www.cecill.info".                                               */
/*                                                                          */
/*  As a counterpart to the access to the source code and  rights to copy,  */
/*  modify and redistribute granted by the license, users are provided      */
/*  only with a limited warranty  and the software's author,  the holder    */
/*  of the economic rights,  and the successive licensors  have only        */
/*  limited liability.                                                      */
/*                                                                          */
/*  In this respect, the user's attention is drawn to the risks             */
/*  associated with loading,  using,  modifying and/or developing or        */
/*  reproducing the software by the user in light of its specific status    */
/*  of free software, that may mean  that it is complicated to              */
/*  manipulate, and  that  also therefore means  that it is reserved for    */
/*  developers and experienced professionals having in-depth computer       */
/*  knowledge. Users are therefore encouraged to load and test the          */
/*  software's suitability as regards their requirements in conditions      */
/*  enabling the security of their systems and/or data to be ensured and,   */
/*  more generally, to use and operate it in the same conditions as         */
/*  regards security.                                                       */
/*                                                                          */
/*  The fact that you are presently reading this means that you have had    */
/*  knowledge of the CeCILL license and that you accept its terms.          */
/*                                                                          */
/****************************************************************************/
/* $Id$
 * $Log$
 * Revision 1.4  2011/05/13 08:17:51  bdepardo
 * Update ORB manager with changes made in DIET ORB manager.
 *
 * Revision 1.3  2011/05/11 16:27:25  bdepardo
 * Updated forwarder files: bug correction and optimization coming from DIET
 *
 * Revision 1.2  2010/12/03 12:40:25  kcoulomb
 * MAJ log to use forwarders
 *
 * Revision 1.1  2010/11/10 02:56:32  kcoulomb
 * Add missing file
 *
 * Revision 1.4  2010/07/27 16:16:48  glemahec
 * Forwarders robustness
 *
 * Revision 1.3  2010/07/14 23:45:30  bdepardo
 * Header corrections
 *
 * Revision 1.2  2010/07/13 15:24:13  glemahec
 * Warnings corrections and some robustness improvements
 *
 * Revision 1.1  2010/07/12 16:11:04  glemahec
 * DIET 2.5 beta 1 - New ORB manager; logForwarder application
 ****************************************************************************/

#include "LogForwarder.hh"
#include "LogORBMgr.hh"
#include "SSHTunnel.hh"
#include "Options.hh"

#include "logFwdr.hh"

#include <iostream>
#include <cstdlib>
#include <string>
#include <cstring>
#include <sstream>
#include <fstream>
#include <algorithm>

#include <omniORB4/CORBA.h>

#include <unistd.h>	// For sleep function

using namespace std;

int main(int argc, char* argv[], char* envp[]) {
  /* Forwarder configuration. */
  FwrdConfig cfg(argv[0]);
	
  Options opt(&cfg, argc, argv, envp);
  /* Mandatory parameter. */
  opt.setOptCallback("--name", name);
	
  /* Mandatory when creating tunnels. */
  opt.setOptCallback("--peer-name", peer_name);
  opt.setOptCallback("--ssh-host", ssh_host);
  /* Optionnal, set to "localhost" by default. */
  opt.setOptCallback("--remote-host", remote_host);
  /* Optionnal, we try to determine it automatically. */
  opt.setOptCallback("--remote-port", remote_port_from);

  /* Optionnal - default values are set to port 22,
   * current user login and $HOME/.ssh/id_[rsa|dsa].
   */
  opt.setOptCallback("--ssh-port", ssh_port);
  opt.setOptCallback("--ssh-login", ssh_login);
  opt.setOptCallback("--ssh-key", key_path);
	
  /* Optionnal parameters/flags. */
  opt.setOptCallback("--retry", nb_retry);
  opt.setOptCallback("--peer-ior", peer_ior);
  opt.setFlagCallback('C', create);
  //opt.setFlagCallback('f', create_from);
	
  opt.setOptCallback("--net-config", net_config);

  opt.processOptions();
	
  if (cfg.getName()=="") {
    std::ostringstream name;
    char host[256];
		
    gethostname(host, 256);
    host[255]='\0';
		
    std::transform(host, host+strlen(host), host, change);
    name << "Forwarder-" << host << "-" << getpid();
    cerr << "Missing parameter: name (use --name to fix it)" << endl;
    cerr << "Use default name: " << name.str() << endl;
    cfg.setName(name.str());
  }
  if (cfg.getCfgPath()=="") {
    cerr << "Missing parameter: net-config (use --net-config <file> to fix it)" << endl;
    return EXIT_FAILURE;
  }
	  
  if (cfg.createFrom()) {
    if (cfg.getPeerName()==""       ||
	cfg.getSshHost()=="") {
      cerr << "Missing parameter(s) to create tunnel.";
      cerr << " Mandatory parameters:" << endl;
      cerr << '\t' << "- Peer name (--peer-name <name>)" << endl;
      cerr << '\t' << "- SSH host (--ssh-host <host>)" << endl;
      return EXIT_FAILURE;
    }
  }
  
	
  SSHTunnel tunnel;
  LogForwarder* forwarder;
  try {
    forwarder = new LogForwarder(cfg.getName(), cfg.getCfgPath());
  } catch (exception &e) {
    cerr << "Error: " << e.what() << endl;
    return EXIT_FAILURE;
  }
  LogORBMgr::init(argc, argv);
  LogORBMgr* mgr = LogORBMgr::getMgr();
  string ior;
  int count = 0;
	
  mgr->activate(forwarder);
  do {
    try {
      mgr->bind(LOGFWRDCTXT, cfg.getName(), forwarder->_this(), true);
      break;
    } catch (CORBA::TRANSIENT& err) {
      cerr << "Error when binding the forwarder " << cfg.getName() << endl;
      if (count++<cfg.getNbRetry()) {
	sleep(5);
	continue;
      }
      mgr->deactivate(forwarder);
      return EXIT_FAILURE;
    }
  } while (true);
	
  /* Write the IOR to a file on /tmp. */
  ior = mgr->getIOR(forwarder->_this());
  string iorFilename("/tmp/LOG-forwarder-ior-");
  iorFilename+=cfg.getName()+".tmp";
  ofstream of(iorFilename.c_str(), ios_base::trunc);
	
  if (!of.is_open()) {
    cerr << "Warning: cannot open file " << iorFilename
	 << " to store the IOR" << endl;
  } else {	
    cout << "Write IOR to " << iorFilename << endl;
    if (cfg.createFrom()) { // Creating tunnel(s)
      istringstream is(cfg.getRemotePortFrom());
      int port;
			
      is >> port;
      of << LogORBMgr::convertIOR(ior, cfg.getRemoteHost(), port);
    }	else {// Waiting for connexion.
      of << ior;
    }
    of << std::endl;
    of << freeTCPport(); // also write a free port
    of.close();
  }
  cout << "Forwarder: " << ior << endl;
	
	
  tunnel.setSshHost(cfg.getSshHost());
  tunnel.setRemoteHost(cfg.getRemoteHost());
		
  tunnel.setSshPath(cfg.getSshPath());
  tunnel.setSshPort(cfg.getSshPort());
  tunnel.setSshLogin(cfg.getSshLogin());
  tunnel.setSshKeyPath(cfg.getSshKeyPath());
	
  /* Manage the peer IOR. */
  if (cfg.getPeerIOR()=="" && cfg.createFrom()) {
    /* Try to retrieve the peer IOR. */
    SSHCopy copy(cfg.getSshHost(),
		 "/tmp/LOG-forwarder-ior-"+cfg.getPeerName()+".tmp",
		 "/tmp/LOG-forwarder-ior-"+cfg.getPeerName()+".tmp");
    copy.setSshPath("/usr/bin/scp");
    copy.setSshPort(cfg.getSshPort());
    copy.setSshLogin(cfg.getSshLogin());
    copy.setSshKeyPath(cfg.getSshKeyPath());
    try {
      if (copy.getFile()) {
        std::cout << "Got remote IOR file" << std::endl;
        cfg.setPeerIOR("/tmp/LOG-forwarder-ior-"+cfg.getPeerName()+".tmp");
      }
    } catch (...) {
      std::cout << "Got an exception while retrieving IOR file" << std::endl;
    }
  }
  if (cfg.getPeerIOR()!="" && cfg.getPeerIOR().find("IOR:")!=0) {
    /* Extract the IOR from a file. */
    ifstream file(cfg.getPeerIOR().c_str());
    string peerIOR;
    string peerPort;
    if (!file.is_open()) {
      cerr << "Error: Invalid peer-ior parameter" << endl;
      return EXIT_FAILURE;
    }
    file >> peerIOR;
    cfg.setPeerIOR(peerIOR);
    if (!file.eof() && cfg.getRemotePortFrom() == "") {
      file >> peerPort;
      cfg.setRemotePortFrom(peerPort);
    }
  }
	
  if (cfg.getPeerIOR() != "") {
    tunnel.setRemotePortTo(LogORBMgr::getPort(cfg.getPeerIOR()));
  } else {
    tunnel.setRemotePortTo(cfg.getRemotePortTo());
  }
  if (cfg.getRemoteHost() == "") {
    if (cfg.getPeerIOR() != "") {
      tunnel.setRemoteHost(LogORBMgr::getHost(cfg.getPeerIOR()));
    } else {
      tunnel.setRemoteHost("127.0.0.1");
    }
  } else {
    tunnel.setRemoteHost(cfg.getRemoteHost());
  }

  tunnel.setRemotePortFrom(cfg.getRemotePortFrom());
  //	tunnel.setLocalPortFrom(cfg.getLocalPortFrom());
  if (cfg.createFrom()) {
    if (cfg.getRemotePortFrom() == "") {
      cerr << "Failed to automatically determine a remote free port." << endl;
      cerr << " You need to specify the remote port:" << endl;
      cerr << '\t' << "- Remote port (--remote-port <port>)" << endl;
      return EXIT_FAILURE;
    }
  }

	
  tunnel.setLocalPortTo(LogORBMgr::getPort(ior));

  tunnel.createTunnelTo(cfg.createTo());
  tunnel.createTunnelFrom(cfg.createFrom());
	
  tunnel.open();

  /* Try to find the peer. */
  bool canLaunch = true;
  if (cfg.getPeerIOR()!="") {
    try {
      if (connectPeer(ior, cfg.getPeerIOR(), "localhost", tunnel.getRemoteHost(),
                      tunnel.getLocalPortFrom(), tunnel.getRemotePortFrom(), forwarder, mgr)) {
        /* In this case it seems that there is a problem with the alias 'localhost', thus we
         * try to use 127.0.0.1
         */
        if (tunnel.getRemoteHost() == "localhost") {
          tunnel.setRemoteHost("127.0.0.1");
        }
        if (connectPeer(ior, cfg.getPeerIOR(), "127.0.0.1", tunnel.getRemoteHost(),
                        tunnel.getLocalPortFrom(), tunnel.getRemotePortFrom(), forwarder, mgr)) {
          cout << "Unable to contact remote peer. Waiting for connection..." << endl;
        }
      }
    } catch (...) {
      cerr << "Error while connecting to remote peer" << endl;
      canLaunch = false;
    }
  }

  if (canLaunch) {
    mgr->wait();
  }

  std::cout << "Forwarder is now terminated" << std::endl;
	
  return EXIT_SUCCESS;
}

int
connectPeer(const std::string &ior, const std::string &peerIOR,
            const std::string &newHost, const std::string &remoteHost,
            int localPortFrom, int remotePortFrom, LogForwarder *forwarder, LogORBMgr* mgr) {
  
  std::string newPeerIOR = LogORBMgr::convertIOR(peerIOR, newHost, localPortFrom);

  CorbaLogForwarder_var peer;
		
  peer = mgr->resolve<CorbaLogForwarder, CorbaLogForwarder_var>(newPeerIOR);
		
  try {
    peer->connectPeer(ior.c_str(), remoteHost.c_str(), remotePortFrom);
    forwarder->setPeer(peer);
  } catch (CORBA::TRANSIENT& err) {
    cout << "Unable to contact remote peer using '" << newHost <<"' as a \"new remote host\"" << endl;
    return 1;
  }

  cout << "Contacted remote peer using '" << newHost << "' as new remote host" << endl;
  return 0;
}


void name(const string& name, Configuration* cfg) {
  static_cast<FwrdConfig*>(cfg)->setName(name);
}

void peer_name(const string& name, Configuration* cfg) {
  static_cast<FwrdConfig*>(cfg)->setPeerName(name);
}

void peer_ior(const string& ior, Configuration* cfg) {
  static_cast<FwrdConfig*>(cfg)->setPeerIOR(ior);
}

void net_config(const string& path, Configuration* cfg) {
  static_cast<FwrdConfig*>(cfg)->setCfgPath(path);
}

void ssh_host(const string& host, Configuration* cfg) {
  static_cast<FwrdConfig*>(cfg)->setSshHost(host);
}

void remote_host(const string& host, Configuration* cfg) {
  static_cast<FwrdConfig*>(cfg)->setRemoteHost(host);
}

void remote_port_to(const string& port, Configuration* cfg) {
  static_cast<FwrdConfig*>(cfg)->setRemotePortTo(port);
}

void remote_port_from(const string& port, Configuration* cfg) {
  static_cast<FwrdConfig*>(cfg)->setRemotePortFrom(port);
}

void local_port_from(const string& port, Configuration* cfg) {
  static_cast<FwrdConfig*>(cfg)->setLocalPortFrom(port);
}

void ssh_path(const string& path, Configuration* cfg) {
  static_cast<FwrdConfig*>(cfg)->setSshPath(path);
}

void ssh_port(const string& port, Configuration* cfg) {
  static_cast<FwrdConfig*>(cfg)->setSshPort(port);
}

void ssh_login(const string& login, Configuration* cfg) {
  static_cast<FwrdConfig*>(cfg)->setSshLogin(login);
}

void key_path(const string& path, Configuration* cfg) {
  static_cast<FwrdConfig*>(cfg)->setSshKeyPath(path);
}

void create(const string& create, Configuration* cfg) {
  (void) create;
  static_cast<FwrdConfig*>(cfg)->createTo(true);
  static_cast<FwrdConfig*>(cfg)->createFrom(true);
}

void nb_retry(const string& nb, Configuration* cfg) {
  istringstream is(nb);
  int n;
  is >> n;
  static_cast<FwrdConfig*>(cfg)->setNbRetry(n);
}
/* Fwdr configuration implementation. */
FwrdConfig::FwrdConfig(const string& pgName) : Configuration(pgName)
{
  createTunnelTo = false;
  createTunnelFrom = false;
  nbRetry = 3;
}

const string& FwrdConfig::getName() const {
  return name;
}
const string& FwrdConfig::getPeerName() const {
  return peerName;
}
const string& FwrdConfig::getPeerIOR() const {
  return peerIOR;
}

const string& FwrdConfig::getSshHost() const {
  return sshHost;
}
const string& FwrdConfig::getRemoteHost() const {
  return remoteHost;
}
const string& FwrdConfig::getRemotePortTo() const {
  return remotePortTo;
}
const string& FwrdConfig::getRemotePortFrom() const {
  return remotePortFrom;
}
const string& FwrdConfig::getLocalPortFrom() const {
  return localPortFrom;
}
bool  FwrdConfig::createTo() const {
  return createTunnelTo;
}
bool  FwrdConfig::createFrom() const {
  return createTunnelFrom;
}
const string& FwrdConfig::getSshPath() const {
  return sshPath;
}
const string& FwrdConfig::getSshPort() const {
  return sshPort;
}
const string& FwrdConfig::getSshLogin() const {
  return sshLogin;
}
const string& FwrdConfig::getSshKeyPath() const {
  return sshKeyPath;
}
int  FwrdConfig::getNbRetry() const {
  return nbRetry;
}
const string& FwrdConfig::getCfgPath() const {
  return cfgPath;
}

void FwrdConfig::setName(const string& name) {
  this->name = name;
}
void FwrdConfig::setPeerName(const string& name) {
  this->peerName = name;
}
void FwrdConfig::setPeerIOR(const string& ior) {
  this->peerIOR = ior;
}

void FwrdConfig::setSshHost(const string& host) {
  this->sshHost = host;
}
void FwrdConfig::setRemoteHost(const string& host) {
  this->remoteHost = host;
}
void FwrdConfig::setRemotePortTo(const string& port) {
  this->remotePortTo = port;
}
void FwrdConfig::setRemotePortFrom(const string& port) {
  this->remotePortFrom = port;
}
void FwrdConfig::setLocalPortFrom(const string& port) {
  this->localPortFrom = port;
}
void FwrdConfig::createTo(bool create) {
  this->createTunnelTo = create;
}
void FwrdConfig::createFrom(bool create) {
  this->createTunnelFrom = create;
}
void FwrdConfig::setSshPath(const string& path) {
  this->sshPath = path;
}
void FwrdConfig::setSshPort(const string& port) {
  this->sshPort = port;
}
void FwrdConfig::setSshLogin(const string& login) {
  this->sshLogin = login;
}
void FwrdConfig::setSshKeyPath(const string& path) {
  this->sshKeyPath = path;
}
void FwrdConfig::setNbRetry(const int nb) {
  this->nbRetry = nb;
}
void FwrdConfig::setCfgPath(const string& path) {
  this->cfgPath = path;
}

int change(int c) {
  if (c=='.') return '-';
  return c;
}

