/*
 *  Copyright 1994-2021 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include <sys/utsname.h>
#include <ulfius.h>
#include "context.h"
#include "defaults.h"
#include "settings.h"


#define STATS_URL "https://stats.biniou.net/lebiniou"


extern uint64_t frames;
static const char *cfg_types[CFG_NB] = { "default", "custom" };
extern uint8_t json_settings_type;


#if GLIB_CHECK_VERSION(2, 64, 0)
static void
add_os_info(json_t *os_info, const char *name, const char *key)
{
  gchar *info = g_get_os_info(key);

  if (NULL != info) {
    json_object_set_new(os_info, name, json_string(info));
    g_free(info);
  }
}
#endif


void Context_statistics(const Context_t *ctx)
{
  json_t *stats = json_object();

#ifdef DEBUG
  json_object_set_new(stats, "debug", json_true());
#else
  json_object_set_new(stats, "debug", json_false());
#endif
#ifdef FIXED
  json_object_set_new(stats, "fixed", json_true());
#else
  json_object_set_new(stats, "fixed", json_false());
#endif
  json_object_set_new(stats, "version", json_string(LEBINIOU_VERSION));
  json_object_set_new(stats, "elapsed", json_integer((uint64_t)Timer_elapsed(ctx->timer)));

  if (NULL != ctx->input_plugin) {
    json_object_set_new(stats, "input", json_string(ctx->input_plugin->name));
  } else {
    json_object_set_new(stats, "input", json_string("NULL"));
  }

  for (GSList *outputs = ctx->outputs; NULL != outputs; outputs = g_slist_next(outputs)) {
    Plugin_t *output = (Plugin_t *)outputs->data;
    char *o = g_strdup_printf("output.%s", output->name);

    json_object_set_new(stats, o, json_true());
    xfree(o);
  }

  json_object_set_new(stats, "width", json_integer(WIDTH));
  json_object_set_new(stats, "height", json_integer(HEIGHT));
  
  json_object_set_new(stats, "frames", json_integer(frames));
  json_t *commands = json_object();
  json_object_set_new(commands, "sdl2", json_integer(ctx->commands[BC_SDL2]));
  json_object_set_new(commands, "rest", json_integer(ctx->commands[BC_REST]));
  json_object_set_new(commands, "web", json_integer(ctx->commands[BC_WEB]));
  json_object_set_new(stats, "commands", commands);

  struct utsname _uname;
  if (!uname(&_uname)) {
    json_t *uname = json_object();
    json_object_set_new(uname, "sysname", json_string(_uname.sysname));
    json_object_set_new(uname, "nodename", json_string(_uname.nodename));
    json_object_set_new(uname, "release", json_string(_uname.release));
    json_object_set_new(uname, "kernel", json_string(_uname.version));
    json_object_set_new(uname, "machine", json_string(_uname.machine));
    json_object_set_new(stats, "uname", uname);
  } else {
    xperror("uname");
  }

  json_object_set_new(stats, "flatpak",
#ifdef FLATPAK
                      json_true()
#else
                      json_false()
#endif
                      );

  json_object_set_new(stats, "json_settings", json_string(cfg_types[json_settings_type]));

  // OS info
#if GLIB_CHECK_VERSION(2, 64, 0)
  json_t *os_info = json_object();
  add_os_info(os_info, "os_name", G_OS_INFO_KEY_NAME);
  add_os_info(os_info, "os_pretty_name", G_OS_INFO_KEY_PRETTY_NAME);
  add_os_info(os_info, "os_version", G_OS_INFO_KEY_VERSION);
  add_os_info(os_info, "os_version_codename", G_OS_INFO_KEY_VERSION_CODENAME);
  add_os_info(os_info, "os_version_id", G_OS_INFO_KEY_VERSION_ID);
  add_os_info(os_info, "os_id", G_OS_INFO_KEY_ID);
  add_os_info(os_info, "os_home_url", G_OS_INFO_KEY_HOME_URL);
  json_object_set_new(stats, "os_info", os_info);
#endif
  
  DEBUG_JSON("stats", stats);

  bulfius_post("/statistics", stats);
  json_decref(stats);
}
