/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \page lcmaps_poolaccount.mod poolaccount plugin

    \section poolaccountsection1 SYNOPSIS
    \b lcmaps_poolaccount.mod
        [-gridmapfile|-GRIDMAPFILE|-gridmap|-GRIDMAP \<location gridmapfile\>]
        [-gridmapdir|-GRIDMAPDIR \<location gridmapdir\>]
        [-override_inconsistency|-OVERRIDE_INCONSISTENCY]
        [-max_mappings_per_credential \<max nr of mappings\>]
 
    \section poolaccountsection2 DESCRIPTION
 
    This plugin is a Acquisition Plugin and will provide the LCMAPS system with Pool Account
    information.
    To do this it needs to look up the Distinguished Name (DN) from a user's certificate in the gridmapfile.
    If this DN is found in the gridmapfile the plugin now knows to which pool of local system accounts
    the user wil be mapped.
    The poolname (starting with a dot ('.') instead of an alphanumeric character)
    will be converted into the an account from a list of local accounts.
    This list is located in the \e gridmapdir and is made out of filenames.
    These filenames correspond to the system poolaccount names.
    (E.g. if a DN corresponds to \c .test in the gridmapfile, it will be mapped to 
    \c test001, \c test002, etc., which names can be found in the gridmapdir)

    If there is no pool account assigned to the user yet, the plugin
    will get a directory listing of the gridmapdir.
    This list will contain usernames corresponding to system accounts specially designated for pool accounting.
    If the plugin resolved the mapping of a certain pool name, let's say '.test',
    the plugin will look into the directory list and will find the first available file in the list
    corresponding with 'test' (e.g. 'test001') by checking the number of links to its i-node.
    If this number is 1, this account is still available.
    To lease this account a second hard link is created, named after the URL-encoded, decapitalized DN.

    When a user returns to this site the plugin will look for the DN of
    the user (URL encoded) in this directory. If found, the corresponding poolaccount will be assigned
    to the user.

    The plugin will resolve the UID, GID and all the secondary GIDs belonging to the poolaccount.
    When this has been done and there weren't any problems detected,
    the plugin will add this information to a datastructure in the Plugin Manager.
    The plugin will finish its run with a LCMAPS_MOD_SUCCESS.
    This result will be reported to the Plugin Manager which started this plugin and
    it will forward this result to the Evaluation Manager, which will take appropriate actions
    for the next plugin to run.
    Normally this plugin would be followed by an Enforcement plugin that can apply these gathered
    credentials in a way that is appropriate to a system administration's needs.
 
    \section poolaccountoptions OPTIONS
    \subsection poolaccountoptie1 -GRIDMAPFILE \<gridmapfile\>
        See \ref poolaccountoptie4 "-gridmap"
 
    \subsection poolaccountoptie2 -gridmapfile \<gridmapfile\>
        See \ref poolaccountoptie4 "-gridmap"
 
    \subsection poolaccountoptie3 -GRIDMAP \<gridmapfile\>
        See \ref poolaccountoptie4 "-gridmap"
 
    \subsection poolaccountoptie4 -gridmap \<gridmapfile\>
        If this option is set, it will override the default path of the gridmapfile.
        It is advised to use an absolute path to the gridmapfile to avoid usage of the wrong file(path).

    \subsection poolaccountoptie5 -GRIDMAPDIR \<gridmapdir\>
        See \ref poolaccountoptie6 "-gridmapdir"
 
    \subsection poolaccountoptie6 -gridmapdir \<gridmapdir\>
        If this option is set, it will override the default path to the gridmapdir.
        It is advised to use an absolute path to the gridmapdir to avoid usage of the wrong path.

    \subsection poolaccountoptie7 -OVERRIDE_INCONSISTENCY
        See \ref poolaccountoptie8 "-override_inconsistency"

    \subsection poolaccountoptie8 -override_inconsistency
        Moving a user from one pool to another (because of a VO change)
        should only be done by changing the gridmapfile indicating the new pool for this user.
        If a user has already been mapped previously to a poolaccount, there is a link present
        between this poolaccount and his DN.
        In the good old days prior to LCMAPS, a 'pool change' would still result in a mapping to
        the old pool account, neglecting the administrative changes in the gridmapfile.
        LCMAPS corrects this behaviour:
        By default the poolaccount plugin will \e fail if the pool designated by the gridmapfile
        doesn't match the previously mapped poolaccount leasename.
        If the site doesn't want a failure on this inconsistency it can turn on this parameter.
        When the inconsistency is detected the plugin will automatically unlink the previous mapping
        and will proceed by making a \e new lease from the new pool.

    \subsection poolaccountoptie9 -max_mappings_per_credential \<max nr of mappings\>
        This value indicates the maximum number of accounts a user, or more specifically
        a set of credentials (=DN + FQANS), can be mapped to. Normally this number is 1.
        But if each job should run under its own account the number should be increased.
        The leasename (or poolindex) in this case looks like:
                url_encoded(<DN>):mapcount=<mapnumber>)

    \subsection poolaccountoptie10 -no_wildcard
        If this option is set, wildcards cannot be used in the grid-mapfile (on by default)

    \subsection poolaccountoptie11 -strict_poolprefix_match [yes|no]. Default is 'yes'.
        If this is set to 'yes', a line in the gridmapfile like
        <DN> .pool
        will result in accounts matching the regexp 'pool[0-9]+'.
        Otherwise it will be allowed to match 'pool.*' (legacy behaviour).

\section poolaccountReturnvalue RETURN VALUES
        \li LCMAPS_MOD_SUCCESS : Success
        \li LCMAPS_MOD_FAIL    : Failure
 
 
\section poolaccountErrors ERRORS
        See bugzilla for known errors (http://marianne.in2p3.fr/datagrid/bugzilla/)
 
\section poolaccountSeeAlso SEE ALSO
        \ref lcmaps_localaccount.mod "lcmaps_localaccount.mod",
        \ref lcmaps_posix_enf.mod "lcmaps_posix_enf.mod",
        \ref lcmaps_ldap_enf.mod "lcmaps_ldap_enf.mod",
        \ref lcmaps_voms.mod "lcmaps_voms.mod",
        \ref lcmaps_voms_poolaccount.mod "lcmaps_voms_poolaccount.mod",
        \ref lcmaps_voms_poolgroup.mod "lcmaps_voms_poolgroup.mod",
        \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod"
*/

/*!
    \file   lcmaps_poolaccount.c
    \brief  Interface to the LCMAPS plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This file contains the code of the poolaccount plugin
    -# plugin_initialize()
    -# plugin_run()
    -# plugin_terminate()
    -# plugin_introspect()
*/

/* Define both _XOPEN_SOURCE for strdup and _GNU_SOURCE, this makes Solaris
 * happier */
#define _XOPEN_SOURCE	500
/* Try to provide RTLD_DEFAULT */
/* MacOS needs no POSIX or _DARWIN_C_SOURCE to have RTLD_DEFAULT */
#ifdef __APPLE__
# define _DARWIN_C_SOURCE
#else
# define _GNU_SOURCE
#endif
#include <dlfcn.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <ctype.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>

#include "lcmaps_plugins_basic_config.h"

#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>
#include <lcmaps/lcmaps_cred_data.h>

#if defined(HAVE_LCMAPS_LCMAPS_PLUGIN_PROTOTYPES_H)
#   include <lcmaps/lcmaps_plugin_prototypes.h>
#else
#   include "lcmaps_plugin_prototypes.h"
#endif

#include "lcmaps_gridmapdir.h"


/************************************************************************
 * defines
 ************************************************************************/

#define PLUGIN_PREFIX	"lcmaps_poolaccount"

#define PLUGIN_RUN	0   /* full run mode */
#define PLUGIN_VERIFY	1   /* verify-only mode */


#define LCMAPS_DEFAULT_MAPPING_MAX 1
#define LCMAPS_ABSOLUTE_MAPPING_MAX 9999

/************************************************************************
 * global variables
 ************************************************************************/

static char *gridmapfile            = NULL; /* filename of gridmapfile */
static char *gridmapdir             = NULL; /* dirname of gridmapdir */
static int  override_inconsistency  = 0;
static int  use_wildcard            = 1; /* default use wildcards */
static int  mapping_max             = LCMAPS_DEFAULT_MAPPING_MAX;
static int  strict_poolprefix_match = 1; /* By default strict matching */


/************************************************************************
 * private prototypes
 ************************************************************************/

/* called by plugin_run() and plugin_verify() */
static int plugin_run_or_verify(int argc, lcmaps_argument_t *argv,
				int lcmaps_mode);


/************************************************************************
 * public functions
 ************************************************************************/

/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
int plugin_initialize(int argc, char **argv) {
    const char * logstr = PLUGIN_PREFIX"-plugin_initialize()";
    int i;
    long long_value;
    char *endptr=NULL;

    /* Log commandline parameters on debug */
    lcmaps_log(LOG_DEBUG,"%s: passed arguments:\n",logstr);
    for (i=0; i < argc; i++)
	lcmaps_log(LOG_DEBUG,"%s: arg %d is %s\n", logstr, i, argv[i]);

    /* Parse arguments, argv[0] = name of plugin, so start with i = 1 */
    for (i = 1; i < argc; i++) {
	/* check grid-mapfile option (or similar variations) */
        if ( strcmp(argv[i], "-gridmapfile") == 0 ||
             strcmp(argv[i], "-GRIDMAPFILE") == 0 ||
             strcmp(argv[i], "-gridmap") == 0 ||
             strcmp(argv[i], "-GRIDMAP") == 0 )
	{
	    /* check valid filename argument */
            if (argv[i + 1] == NULL || argv[i + 1][0]=='\0') {
		lcmaps_log(LOG_ERR,
		    "%s: option %s needs to be followed by valid filename\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* free existing one and copy new one */
	    free(gridmapfile); gridmapfile=NULL;
	    if (argv[i+1][0]=='/')  { /* absolute path */
		if ( (gridmapfile = strdup(argv[i + 1])) == NULL)	{
		    lcmaps_log(LOG_ERR, "%s: out of memory\n", logstr);
		    goto fail_init_module;
		}
	    } else { /* relative path */
		if (lcmaps_get_prefixed_file(argv[i + 1], &gridmapfile)==-1)
		    goto fail_init_module;
	    }
	    /* log the name and increase arg counter */
	    lcmaps_log(LOG_DEBUG, "%s: Using grid-mapfile \"%s\".\n",
		    logstr, gridmapfile);
            i++;
        }
	/* check gridmapdir option (or similar variations) */
        else if ( strcmp(argv[i], "-gridmapdir") == 0 ||
                  strcmp(argv[i], "-GRIDMAPDIR") == 0 )
        {
	    /* check valid dirname argument */
            if (argv[i+1]==NULL || argv[i+1][0]!='/')
	    {
		lcmaps_log(LOG_ERR,
		    "%s: option %s needs to be followed by "
		    "a valid absolute directory.\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* free existing one and copy new one */
	    free(gridmapdir); gridmapdir=NULL;
	    if (argv[i+1][0]=='/')  { /* absolute path */
		if ( (gridmapdir = strdup(argv[i + 1])) == NULL)	{
		    lcmaps_log(LOG_ERR, "%s: out of memory\n", logstr);
		    goto fail_init_module;
		}
	    } else { /* relative path */
		if (lcmaps_get_prefixed_file(argv[i + 1], &gridmapdir)==-1)
		    goto fail_init_module;
	    }
	    /* log the name and increase arg counter */
	    lcmaps_log(LOG_DEBUG, "%s: Using gridmapdir \"%s\".\n",
		    logstr, gridmapdir);
            i++;
        }
	/* check max_mappings_per_credential option */
        else if ( strcmp(argv[i], "-max_mappings_per_credential") == 0 )
        {
	    /* check valid argument */
	    if (argv[i + 1] == NULL || argv[i + 1][0]=='\0') {
		lcmaps_log(LOG_ERR,
		    "%s: option %s needs to be followed by a valid number\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }

	    /* try to convert argument to integer in right range */
	    errno=0;
	    long_value=strtol(argv[i + 1],&endptr,10);
	    /* endptr will either point to first non-valid char, or end of valid
	     * string unless argv[i+1]=="", which we already checked above */
	    if (errno!=0 || *endptr!='\0' ||
		long_value < (long)MAPPING_MIN ||
		long_value > (long)LCMAPS_ABSOLUTE_MAPPING_MAX)
	    {
		lcmaps_log(LOG_ERR,
		    "%s: Illegal value for \"-max_mappings_per_credential\" "
		    "(%s): Should be in the range: %d-%d (failure)\n",
		    logstr, argv[i + 1],
		    MAPPING_MIN, LCMAPS_ABSOLUTE_MAPPING_MAX);
		goto fail_init_module;
	    }
	    mapping_max = (int)long_value;
            i++;
        }
	/* check strict_poolprefix_match option */
        else if ( strcmp(argv[i], "-strict_poolprefix_match") == 0 )
        {
	    /* check valid argument */
	    if (argv[i + 1] == NULL || argv[i + 1][0]=='\0') {
		lcmaps_log(LOG_ERR,
		    "%s: option %s needs to be followed by \"yes\" or \"no\"\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }

	    /* parse argument */
	    if (strcmp(argv[i+1],"yes") == 0)
		strict_poolprefix_match = 1;
	    else if (strcmp(argv[i+1],"no") == 0)
		strict_poolprefix_match = 0;
	    else {
		lcmaps_log(LOG_ERR,"%s: use \"yes\" or \"no\" for option %s\n",
			logstr, argv[i]);
		goto fail_init_module;
            }
            i++;
        }
	/* check override_inconsistency (or similar variations) */
        else if ( strcmp(argv[i], "-override_inconsistency") == 0 ||
                  strcmp(argv[i], "-OVERRIDE_INCONSISTENCY") == 0 )
        {
            override_inconsistency = 1;
        }
	/* check disablewildcard (or similar variations) */
        else if ( strcmp(argv[i], "-no_wildcard") == 0 ||
	          strcmp(argv[i], "-disablewildcard") == 0 ||
                  strcmp(argv[i], "--disablewildcard") == 0 ||
                  strcmp(argv[i], "-disablewildcards") == 0 ||
                  strcmp(argv[i], "--disablewildcards") == 0 )
        {
            use_wildcard = 0;
        }
	/* any other argument is an error */
	else
	{
            lcmaps_log(LOG_ERR,
		    "%s: Unknown argument for plugin: %s (failure)\n",
		    logstr, argv[i]);
            goto fail_init_module;
        }
    }

    /* Check we have a grid-mapfile */
    if (gridmapfile==NULL || gridmapfile[0]=='\0')  {
	free(gridmapfile); gridmapfile=NULL;
        lcmaps_log(LOG_INFO,
	    "%s: No grid-mapfile was provided, will use default.\n", logstr);
    }

    /* Check we have a gridmapdir */
    if (gridmapdir==NULL || gridmapdir[0]=='\0')    {
	free(gridmapdir); gridmapdir=NULL;
        lcmaps_log(LOG_INFO,
	    "%s: No gridmapdir was provided, will try environment variable.\n",
	    logstr);
    }

    return LCMAPS_MOD_SUCCESS;

fail_init_module:
    free(gridmapfile);
    gridmapfile = NULL;
    free(gridmapdir);
    gridmapdir = NULL;

    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_introspect(int *argc, lcmaps_argument_t **argv) {
    const char * logstr = PLUGIN_PREFIX"-plugin_introspect()";
    static lcmaps_argument_t argList[] = {
	{"user_dn"	,"char *", 0,NULL},
	{"mapcounter"	,"int"   , 1,NULL},
	{"requested_uid","uid_t" , 0,NULL},
	{NULL           ,NULL    ,-1,NULL},
	{NULL           ,NULL    ,-1,NULL}
    };

    /* Get the version of LCMAPS being used: we need at least 1.5.8 to be able
     * to demand "requested_username" in the argList */
    int major=0,minor=0,patch=0;
    /* Most UNIX now support RTLD_DEFAULT (POSIX reserved) */
#ifdef RTLD_DEFAULT
    int (*lcmaps_major)(void),(*lcmaps_minor)(void),(*lcmaps_patch)(void);
    dlerror();
    lcmaps_major=dlsym(RTLD_DEFAULT,"lcmaps_get_major_version");
    lcmaps_minor=dlsym(RTLD_DEFAULT,"lcmaps_get_minor_version");
    lcmaps_patch=dlsym(RTLD_DEFAULT,"lcmaps_get_patch_version");
    if (dlerror()==NULL)    {
	major=lcmaps_major();
	minor=lcmaps_minor();
	patch=lcmaps_patch();
    }
#else
    /* No RTLD_DEFAULT, just hope the symbol exists in LCMAPS */
    major=lcmaps_get_major_version();
    minor=lcmaps_get_minor_version();
    patch=lcmaps_get_patch_version();
#endif

    /* Too old when older than 1.5.8 */
    if (major<1 || (major==1 && (minor<5 || (minor==5 && patch<8))))	{
	lcmaps_log(LOG_DEBUG,
	    "%s: Old LCMAPS found (%d.%d.%d), not using requested_username\n",
	    logstr,major,minor,patch);
    } else {
	lcmaps_log(LOG_DEBUG,
	    "%s LCMAPS (%d.%d.%d) supports using requested_username\n",
	    logstr,major,minor,patch);
	argList[3].argName="requested_username";
	argList[3].argType="char *";
	argList[3].argInOut=1;
	argList[3].value=NULL;
    }

    lcmaps_log(LOG_DEBUG,"%s: introspecting\n", logstr);

    *argv = argList;
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log(LOG_DEBUG,"%s: address first argument: %p\n",
	    logstr, (void*)argList);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_run(int argc, lcmaps_argument_t *argv) {
    return plugin_run_or_verify(argc, argv, PLUGIN_RUN);
}

/******************************************************************************
Function:   plugin_verify
Description:
    Verify if user is entitled to use local credentials based on his grid
    credentials. This means that the site should already have been set up
    by, e.g., LCMAPS in a previous run. This method will not try to setup
    account leases, modify (distributed) passwd/group files, etc. etc.
    The outcome should be identical to that of plugin_run().

    Policy: This method will not fail if the uid found is not the
            uid requested for the user, but it will issue a warning.
            The full check is deferred to the stage after which all plugins have run.

Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_verify(int argc, lcmaps_argument_t *argv) {
    return plugin_run_or_verify(argc, argv, PLUGIN_VERIFY);
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_terminate(void) {
    const char * logstr = PLUGIN_PREFIX"-plugin_terminate()";

    lcmaps_log(LOG_DEBUG,"%s: terminating\n", logstr);

    free(gridmapfile);
    gridmapfile=NULL;
    free(gridmapdir);
    gridmapdir=NULL;

    return LCMAPS_MOD_SUCCESS;
}


/************************************************************************
 * private functions
 ************************************************************************/

/**
 * Actual run/verify function. Called by both plugin_run and plugin_verify
 * with different lcmaps_mode.
 */
static int plugin_run_or_verify(int argc, lcmaps_argument_t *argv,
				int lcmaps_mode) {
    const char *        logstr       = NULL;
    const char *        logmapfile   = gridmapfile ? gridmapfile
						   : "default grid-mapfile";
    void *              value        = NULL;
    char **             user_dn_list = NULL;
    char *              dn           = NULL;
    int                 dn_cnt       = 0;
    int                 mapcounter   = -1;
    char *              req_username = NULL;
    int                 req_username_needs_free=0;
    uid_t               req_uid      = (uid_t)(-1);
    struct passwd *     pw           = NULL;
    unsigned short      options      = 0;
    int                 imap         = 0;
    int                 rc           = 0;
    char *              username     = NULL;
    char *              encoded_dn   = NULL;
    int                 found_mapping= 0;
    struct passwd       *user_info   = NULL;
    int                 cnt_sec_gid  = 0;
    gid_t *             sec_gid      = NULL;
    int                 i            = 0;

    /* Set suitable logstr */
    if (lcmaps_mode == PLUGIN_RUN)
        logstr = PLUGIN_PREFIX"-plugin_run()";
    else if (lcmaps_mode == PLUGIN_VERIFY)
        logstr = PLUGIN_PREFIX"-plugin_verify()";
    else {
        lcmaps_log(LOG_ERR, PLUGIN_PREFIX"-plugin_run_or_verify(): "
		"attempt to run plugin in invalid mode: %d\n", lcmaps_mode);
        goto fail_plugin;
    }

    /* Try to get DN from LCMAPS */
    /* First try to obtain DN from the credential data (i.e. stored by other
     * plugins */
    user_dn_list = getCredentialData(DN, &dn_cnt);
    if (dn_cnt>0)   {
	/* Log already registered DNs */
	for (i=0; i<dn_cnt; i++)
	    lcmaps_log(LOG_DEBUG,"%s: found registered DN[%d/%d]: %s\n",
		    logstr, i+1, dn_cnt, user_dn_list[i]);
	dn=user_dn_list[0];
    } else {
	/* No DNs are registered, use the introspect/run arguments */
	value=lcmaps_getArgValue("user_dn", "char *", argc, argv);
	if (value == NULL || (dn = *(char **)value) == NULL ) {
	    lcmaps_log(LOG_WARNING,"%s: could not get value of dn !\n", logstr);
	    return -1;
	}
	
	/* push it to the end-result registry */
	lcmaps_log(LOG_DEBUG, "%s: Adding DN to LCMAPS framework: %s\n",
		logstr, dn);
	addCredentialData(DN, &dn);
    }

    /* Log the found DN */
    lcmaps_log(LOG_DEBUG,"%s: found dn: %s\n", logstr, dn);

    /* Get value of mapcounter */
    if ( ( value = lcmaps_getArgValue("mapcounter", "int", argc, argv) ) ) {
	mapcounter = *(int *)value;
        lcmaps_log(LOG_DEBUG,"%s: mapcounter: %d\n", logstr, mapcounter);
    } else {
	/* normal behaviour: we don't use mapcounter */
        lcmaps_log(LOG_DEBUG,"%s: mapcounter is not set\n", logstr);
    }

    /* In verify mode, requested account is typically in requested_uid, in run
     * mode, it can be provided as requested_username. Let's just see if either
     * is set. */

    /* Get requested_username value */
    value = lcmaps_getArgValue("requested_username", "char *", argc, argv);
    if ( value != NULL && *(char **)value !=NULL ) {
	/* copy and log resulting account name */
	req_username=*(char **)value;
	lcmaps_log(LOG_DEBUG,"%s: the requested user is %s\n",
		logstr, req_username);
    } else { /* No (valid) requested_username, try requested_uid */
	/* Get requested_uid value */
	value = lcmaps_getArgValue("requested_uid", "uid_t", argc, argv);
	if ( value!=NULL && *(int *)value != -1)  { /* undefined value -> -1 */
	    req_uid = *(uid_t *)value;
	    /* Basic sanity check */
	    if (req_uid == 0) {
		lcmaps_log(LOG_ERR,
			"%s: illegal request for uid == 0 (failure)\n", logstr);
		goto fail_plugin;
	    }
	    /* Get passwd info */
	    if ( (pw = getpwuid(req_uid )) == NULL ||
		 pw->pw_name == NULL )
	    {
		lcmaps_log(LOG_WARNING,
			"%s: the requested uid %d is illegal.\n",
			logstr, (int) req_uid);
		goto fail_plugin;
	    }
	    /* copy resulting account name */
	    if ( (req_username=strdup(pw->pw_name))==NULL ) {
		lcmaps_log(LOG_ERR,"%s: out of memory\n",logstr);
		goto fail_plugin;
	    }
	    /* keep track whether it needs free */
	    req_username_needs_free=1;
	    /* Log the result */
	    lcmaps_log(LOG_DEBUG,"%s: the requested user is %s(%d)\n",
		    logstr, req_username, (int)req_uid);
	} else if (lcmaps_mode == PLUGIN_VERIFY)    {
	    lcmaps_log(LOG_WARNING,
		    "%s: both requested_username and requested_uid are unset, "
		    "need at least one in PLUGIN_VERIFY mode\n", logstr);
	    goto fail_plugin;
	}
    }

    /* Set the matching options */

    /* grid-mapfile must exist */
    options = REQUIRE_MAPFILE;

    /* check if we should use wildcards ... */
    if (use_wildcard)
	options = options|MATCH_WILD_CHARS;

    /* if override_consistency is set add this to the options so it will
     * take effect */
    if (override_inconsistency)
        options = options|OVERRIDE_INCONSISTANCY;

    /* Do not create new leases in verification mode */
    if (lcmaps_mode == PLUGIN_VERIFY)
        options = options|ONLY_USE_EXISTING_LEASE;

    /* if strict_poolprefix_match is set add this to the matchin_type so it will
     * take effect */
    if (strict_poolprefix_match)
        options = options|MATCH_STRICT_PREFIX_NUM;

    /* Try to find the dn in the gridmapfile */
    lcmaps_log(LOG_DEBUG,"%s: using leasename: %s\n", logstr,dn);

    /* For Verify mode and unset mapcounter, we first try to loop through
     * all available map counters */
    if (lcmaps_mode == PLUGIN_VERIFY && mapcounter < MAPPING_MIN)   {
	/* No particular mapcounter specified: try them all */
	for (imap = 1; imap <= mapping_max; imap++) {
	    rc = lcmaps_gridmapdir(gridmapfile, dn, gridmapdir,
				   mapping_max, imap,
				   dn, req_username, options,
				   &username, &encoded_dn);
	
	    /* parse return value of lcmaps_gridmapdir */
	    if (rc==-1)
		/* error */
		goto fail_plugin;
	    if (rc==0)	{
		/* no match: log and try next */
		lcmaps_log(LOG_DEBUG,
		    "%s: No match for the requested poolaccount for \"%s\""
		    ", mapcount=%d\n", logstr, dn, imap);
		continue; /* next mapcounter */
	    }

	    /* match */
	    found_mapping = 1;
	    break;  /* out of mapcounter loop */
	}
	/* If we didn't found it, log and try mapcounter-less */
	if (!found_mapping)
	    lcmaps_log(LOG_DEBUG,
		"%s: No mapcounter-based match found for \"%s\", "
		"trying mapcounter-less mapping...\n", logstr, dn);
    }

    /* Check if we already found it */
    if (found_mapping == 0)	{
	rc = lcmaps_gridmapdir(gridmapfile, dn, gridmapdir,
			       mapping_max, mapcounter,
			       dn, req_username, options,
			       &username, &encoded_dn);

	/* parse return value of lcmaps_gridmapdir */
	if (rc==-1)
	    /* error */
	    goto fail_plugin;
	if (rc==0)	{
	    /* no match, this should be at most a NOTICE */
	    if (mapcounter>0)	{
		if (req_username)
		    lcmaps_log(LOG_NOTICE,
			"%s: No match for requested poolaccount %s for user "
			"\"%s\", mapcount=%d, in %s and %s\n", logstr,
			req_username, dn, mapcounter, logmapfile, gridmapdir);
		else /* no req_username */
		    lcmaps_log(LOG_NOTICE,
			"%s: No match for a poolaccount for user "
			"\"%s\", mapcount=%d, in %s and %s\n", logstr,
			dn, mapcounter, logmapfile, gridmapdir);
	    } else { /* no mapcounter */
		if (req_username)
		    lcmaps_log(LOG_NOTICE,
			"%s: No match for requested poolaccount %s for user "
			"\"%s\", in %s and %s\n", logstr,
			req_username, dn, logmapfile, gridmapdir);
		else /* no req_username */
		    lcmaps_log(LOG_NOTICE,
			"%s: No match for a poolaccount for user "
			"\"%s\", in %s and %s\n", logstr,
			dn, logmapfile, gridmapdir);
	    }
	    goto fail_plugin;
	}

	/* match */
    }

    /* found match: log */
    lcmaps_log(LOG_DEBUG,"%s: found %susername: %s\n",
	    logstr, req_username ? "requested " : "", username);

    /* Convert username to uid, pgid and sgids */

    /* Get account info for found username */
    if ( (user_info = getpwnam(username)) == NULL ) {
	lcmaps_log(LOG_WARNING,
		"%s: no user account found with the name \"%s\"\n",
		logstr, username);
	goto fail_plugin;
    }

    /* Log resulting account */
    lcmaps_log(LOG_DEBUG,"%s: username : %s\n", logstr, user_info->pw_name);
    lcmaps_log(LOG_DEBUG,"%s: user_id  : %d\n", logstr, user_info->pw_uid);
    lcmaps_log(LOG_DEBUG,"%s: group_id : %d\n", logstr, user_info->pw_gid);
    lcmaps_log(LOG_DEBUG,"%s: home dir : %s\n", logstr, user_info->pw_dir);

    /* Add this credential data to the credential data repository in the plugin
     * manager */
    addCredentialData(UID, &(user_info->pw_uid));
    addCredentialData(PRI_GID, &(user_info->pw_gid));

    /* Retrieve secondary group id's */
    if (lcmaps_get_gidlist(username, &cnt_sec_gid, &sec_gid)==0) {
	for (i = 0; i < cnt_sec_gid; i++)
	    addCredentialData(SEC_GID, &(sec_gid[i]));
	free(sec_gid);
    }

    /* Added for the POOL_INDEX request for the DAS */
    if (encoded_dn)
	addCredentialData(POOL_INDEX, &encoded_dn);

    /* success */
    if (req_username_needs_free)
	free(req_username);
    free(username);
    free(encoded_dn);

    lcmaps_log(LOG_INFO,"%s: poolaccount plugin succeeded\n", logstr);

    return LCMAPS_MOD_SUCCESS;

fail_plugin:
    if (req_username_needs_free)
	free(req_username);
    free(username);
    free(encoded_dn);

    lcmaps_log(LOG_INFO,"%s: poolaccount plugin failed\n", logstr);

    return LCMAPS_MOD_FAIL;
}
