/*
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "vt102_keydefs.h"
#include "vt102_keyboard.h"

struct cpssp {
	/* Config */
	int layout; /* which layout to use, 0 is EN, 1 is DE */

	/* Ports */
	struct sig_integer *port_keyboard;

	/* Signals */
	/* None... */

	/* State */
	char shift;
	char ctrl;
	char alt;

	/* Processes */
};

/*
 * the following table is stolen from the ascii gui (vm_console.h)
 * it is used to get key"values" from keycodes by reverse-lookup
 */
static const struct keyboard {
	char		shift;
	char		ctrl;
	char		alt;
	char		key;
} kbd[2][128] = {		/* 0 is EN, 1 is DE */
    {			/* english keyboard layout */
	{ 0, 0, 0,  16 },	/* don't know what to return... */
	{ 0, 1, 0,  30 },
	{ 0, 1, 0,  48 },
	{ 0, 1, 0,  46 },	/* ^C */
	{ 0, 1, 0,  32 },
	{ 0, 1, 0,  18 },
	{ 0, 1, 0,  33 },
	{ 0, 1, 0,  34 },
	{ 0, 1, 0,  35 },	/* ^H */
	{ 0, 0, 0,  15 },	/* ^I == TAB */
	{ 0, 1, 0,  36 },
	{ 0, 1, 0,  37 },
	{ 0, 1, 0,  38 },
	{ 0, 0, 0,  28 },
	{ 0, 1, 0,  49 },
	{ 0, 1, 0,  24 },	/* ^O */
	{ 0, 1, 0,  25 },
	{ 0, 1, 0,  16 },
	{ 0, 1, 0,  19 },
	{ 0, 1, 0,  31 },
	{ 0, 1, 0,  20 },
	{ 0, 1, 0,  22 },	/* ^U */
	{ 0, 1, 0,  47 },
	{ 0, 1, 0,  17 },
	{ 0, 1, 0,  45 },
	{ 0, 1, 0,  21 },
	{ 0, 1, 0,  44 },
	{ 0, 0, 0,   1 },	/* ESC */
	{ 0, 1, 0,  43 },
	{ 0, 1, 0,  27 },
	{ 1, 1, 0,   7 },
	{ 1, 1, 0,  12 },	/* ^_ == ASCII 31 */

	{ 0, 0, 0,  57 },	/* space */
	{ 1, 0, 0,   2 },	/* ! */
	{ 1, 0, 0,  40 },	/* " */
	{ 1, 0, 0,   4 },	/* # */
	{ 1, 0, 0,   5 },	/* $ */
	{ 1, 0, 0,   6 },	/* % */
	{ 1, 0, 0,   8 },	/* & */
	{ 0, 0, 0,  40 },	/* ' */
	{ 1, 0, 0,  10 },	/* ( */
	{ 1, 0, 0,  11 },	/* ) */
	{ 1, 0, 0,   9 },	/* * */
	{ 1, 0, 0,  13 },	/* + */
	{ 0, 0, 0,  51 },	/* , */
	{ 0, 0, 0,  12 },	/* - */
	{ 0, 0, 0,  52 },	/* . */
	{ 0, 0, 0,  53 },	/* / */
	{ 0, 0, 0,  11 },	/* 0 */
	{ 0, 0, 0,   2 },	/* 1 */
	{ 0, 0, 0,   3 },	/* 2 */
	{ 0, 0, 0,   4 },	/* 3 */
	{ 0, 0, 0,   5 },	/* 4 */
	{ 0, 0, 0,   6 },	/* 5 */
	{ 0, 0, 0,   7 },	/* 6 */
	{ 0, 0, 0,   8 },	/* 7 */
	{ 0, 0, 0,   9 },	/* 8 */
	{ 0, 0, 0,  10 },	/* 9 */
	{ 1, 0, 0,  39 },	/* : */
	{ 0, 0, 0,  39 },	/* ; */
	{ 1, 0, 0,  51 },	/* < */
	{ 0, 0, 0,  13 },	/* = */
	{ 1, 0, 0,  52 },	/* > */
	{ 1, 0, 0,  53 },	/* ? */

	{ 1, 0, 0,   3 },	/* @ */
	{ 1, 0, 0,  30 },	/* A */
	{ 1, 0, 0,  48 },	/* B */
	{ 1, 0, 0,  46 },	/* C */
	{ 1, 0, 0,  32 },	/* D */
	{ 1, 0, 0,  18 },	/* E */
	{ 1, 0, 0,  33 },	/* F */
	{ 1, 0, 0,  34 },	/* G */
	{ 1, 0, 0,  35 },	/* H */
	{ 1, 0, 0,  23 },	/* I */
	{ 1, 0, 0,  36 },	/* J */
	{ 1, 0, 0,  37 },	/* K */
	{ 1, 0, 0,  38 },	/* L */
	{ 1, 0, 0,  50 },	/* M */
	{ 1, 0, 0,  49 },	/* N */
	{ 1, 0, 0,  24 },	/* O */
	{ 1, 0, 0,  25 },	/* P */
	{ 1, 0, 0,  16 },	/* Q */
	{ 1, 0, 0,  19 },	/* R */
	{ 1, 0, 0,  31 },	/* S */
	{ 1, 0, 0,  20 },	/* T */
	{ 1, 0, 0,  22 },	/* U */
	{ 1, 0, 0,  47 },	/* V */
	{ 1, 0, 0,  17 },	/* W */
	{ 1, 0, 0,  45 },	/* X */
	{ 1, 0, 0,  21 },	/* Y */
	{ 1, 0, 0,  44 },	/* Z */
	{ 0, 0, 0,  26 },	/* [ */
	{ 0, 0, 0,  43 },	/* \ */
	{ 0, 0, 0,  27 },	/* ] */
	{ 1, 0, 0,   7 },	/* ^ */
	{ 1, 0, 0,  12 },	/* _ */

	{ 0, 0, 0,  41 },	/* ` */
	{ 0, 0, 0,  30 },	/* a */
	{ 0, 0, 0,  48 },	/* b */
	{ 0, 0, 0,  46 },	/* c */
	{ 0, 0, 0,  32 },	/* d */
	{ 0, 0, 0,  18 },	/* e */
	{ 0, 0, 0,  33 },	/* f */
	{ 0, 0, 0,  34 },	/* g */
	{ 0, 0, 0,  35 },	/* h */
	{ 0, 0, 0,  23 },	/* i */
	{ 0, 0, 0,  36 },	/* j */
	{ 0, 0, 0,  37 },	/* k */
	{ 0, 0, 0,  38 },	/* l */
	{ 0, 0, 0,  50 },	/* m */
	{ 0, 0, 0,  49 },	/* n */
	{ 0, 0, 0,  24 },	/* o */
	{ 0, 0, 0,  25 },	/* p */
	{ 0, 0, 0,  16 },	/* q */
	{ 0, 0, 0,  19 },	/* r */
	{ 0, 0, 0,  31 },	/* s */
	{ 0, 0, 0,  20 },	/* t */
	{ 0, 0, 0,  22 },	/* u */
	{ 0, 0, 0,  47 },	/* v */
	{ 0, 0, 0,  17 },	/* w */
	{ 0, 0, 0,  45 },	/* x */
	{ 0, 0, 0,  21 },	/* y */
	{ 0, 0, 0,  44 },	/* z */
	{ 1, 0, 0,  26 },	/* { */
	{ 1, 0, 0,  43 },	/* | */
	{ 1, 0, 0,  27 },	/* } */
	{ 1, 0, 0,  41 },	/* ~ */
	{ 0, 0, 0,  14 }	/* DEL */
    },
    {			/* GERMAN keyboard layout */
	{ 0, 1, 1,  16 },
	{ 0, 1, 0,  30 },
	{ 0, 1, 0,  48 },
	{ 0, 1, 0,  46 },	/* ^C */
	{ 0, 1, 0,  32 },
	{ 0, 1, 0,  18 },
	{ 0, 1, 0,  33 },
	{ 0, 1, 0,  34 },
	{ 0, 1, 0,  35 },	/* ^H */
	{ 0, 1, 0,  15 },	/* ^I == TAB */
	{ 0, 1, 0,  36 },
	{ 0, 1, 0,  37 },
	{ 0, 1, 0,  38 },
	{ 0, 0, 0,  28 },
	{ 0, 1, 0,  49 },
	{ 0, 1, 0,  24 },	/* ^O */
	{ 0, 1, 0,  25 },
	{ 0, 1, 0,  16 },
	{ 0, 1, 0,  19 },
	{ 0, 1, 0,  31 },
	{ 0, 1, 0,  20 },
	{ 0, 1, 0,  22 },	/* ^U */
	{ 0, 1, 0,  47 },
	{ 0, 1, 0,  17 },
	{ 0, 1, 0,  45 },
	{ 0, 1, 0,  21 },
	{ 0, 1, 0,  44 },
	{ 0, 0, 0,   1 },	/* ESC */
	{ 0, 1, 1,  12 },
	{ 0, 1, 1,  10 },
	{ 0, 1, 0,  41 },
	{ 1, 1, 0,  53 },	/* ^_ == ASCII 31 */

	{ 0, 0, 0,  57 },	/* space */
	{ 1, 0, 0,   2 },	/* ! */
	{ 1, 0, 0,   3 },	/* " */
	{ 1, 0, 0,  43 },	/* # */
	{ 1, 0, 0,   5 },	/* $ */
	{ 1, 0, 0,   6 },	/* % */
	{ 1, 0, 0,   7 },	/* & */
	{ 0, 0, 0,  13 },	/* ' */
	{ 1, 0, 0,   9 },	/* ( */
	{ 1, 0, 0,  10 },	/* ) */
	{ 1, 0, 0,  27 },	/* * */
	{ 0, 0, 0,  27 },	/* + */
	{ 0, 0, 0,  51 },	/* , */
	{ 0, 0, 0,  53 },	/* - */
	{ 0, 0, 0,  52 },	/* . */
	{ 1, 0, 0,   8 },	/* / */
	{ 0, 0, 0,  11 },	/* 0 */
	{ 0, 0, 0,   2 },	/* 1 */
	{ 0, 0, 0,   3 },	/* 2 */
	{ 0, 0, 0,   4 },	/* 3 */
	{ 0, 0, 0,   5 },	/* 4 */
	{ 0, 0, 0,   6 },	/* 5 */
	{ 0, 0, 0,   7 },	/* 6 */
	{ 0, 0, 0,   8 },	/* 7 */
	{ 0, 0, 0,   9 },	/* 8 */
	{ 0, 0, 0,  10 },	/* 9 */
	{ 1, 0, 0,  51 },	/* : */
	{ 1, 0, 0,  52 },	/* ; */
	{ 0, 0, 0,  86 },	/* < */
	{ 1, 0, 0,  11 },	/* = */
	{ 1, 0, 0,  86 },	/* > */
	{ 1, 0, 0,  12 },	/* ? */

	{ 0, 0, 1,  16 },	/* @ */
	{ 1, 0, 0,  30 },	/* A */
	{ 1, 0, 0,  48 },	/* B */
	{ 1, 0, 0,  46 },	/* C */
	{ 1, 0, 0,  32 },	/* D */
	{ 1, 0, 0,  18 },	/* E */
	{ 1, 0, 0,  33 },	/* F */
	{ 1, 0, 0,  34 },	/* G */
	{ 1, 0, 0,  35 },	/* H */
	{ 1, 0, 0,  23 },	/* I */
	{ 1, 0, 0,  36 },	/* J */
	{ 1, 0, 0,  37 },	/* K */
	{ 1, 0, 0,  38 },	/* L */
	{ 1, 0, 0,  50 },	/* M */
	{ 1, 0, 0,  49 },	/* N */
	{ 1, 0, 0,  24 },	/* O */
	{ 1, 0, 0,  25 },	/* P */
	{ 1, 0, 0,  16 },	/* Q */
	{ 1, 0, 0,  19 },	/* R */
	{ 1, 0, 0,  31 },	/* S */
	{ 1, 0, 0,  20 },	/* T */
	{ 1, 0, 0,  22 },	/* U */
	{ 1, 0, 0,  47 },	/* V */
	{ 1, 0, 0,  17 },	/* W */
	{ 1, 0, 0,  45 },	/* X */
	{ 1, 0, 0,  21 },	/* Y */
	{ 1, 0, 0,  44 },	/* Z */
	{ 0, 0, 1,   9 },	/* [ */
	{ 0, 0, 1,  12 },	/* \ */
	{ 0, 0, 1,  10 },	/* ] */
	{ 0, 0, 0,  41 },	/* ^ */
	{ 1, 0, 0,  53 },	/* _ */

	{ 1, 0, 0,  16 },	/* ` */
	{ 0, 0, 0,  30 },	/* a */
	{ 0, 0, 0,  48 },	/* b */
	{ 0, 0, 0,  46 },	/* c */
	{ 0, 0, 0,  32 },	/* d */
	{ 0, 0, 0,  18 },	/* e */
	{ 0, 0, 0,  33 },	/* f */
	{ 0, 0, 0,  34 },	/* g */
	{ 0, 0, 0,  35 },	/* h */
	{ 0, 0, 0,  23 },	/* i */
	{ 0, 0, 0,  36 },	/* j */
	{ 0, 0, 0,  37 },	/* k */
	{ 0, 0, 0,  38 },	/* l */
	{ 0, 0, 0,  50 },	/* m */
	{ 0, 0, 0,  49 },	/* n */
	{ 0, 0, 0,  24 },	/* o */
	{ 0, 0, 0,  25 },	/* p */
	{ 0, 0, 0,  16 },	/* q */
	{ 0, 0, 0,  19 },	/* r */
	{ 0, 0, 0,  31 },	/* s */
	{ 0, 0, 0,  20 },	/* t */
	{ 0, 0, 0,  22 },	/* u */
	{ 0, 0, 0,  47 },	/* v */
	{ 0, 0, 0,  17 },	/* w */
	{ 0, 0, 0,  45 },	/* x */
	{ 0, 0, 0,  21 },	/* y */
	{ 0, 0, 0,  44 },	/* z */
	{ 0, 0, 1,   8 },	/* { */
	{ 0, 0, 1,  86 },	/* | */
	{ 0, 0, 1,  11 },	/* } */
	{ 0, 0, 1,  27 },	/* ~ */
	{ 1, 0, 0,  53 }	/* DEL */
    }
};

static unsigned char
lookup_value(struct cpssp *cpssp, char s, char c, char a, unsigned char k)
{
	unsigned char i;
	assert(k < 128);
	/* skip the first (bogus) entry: i = 1 */
	for (i = 1; i < 128; i++) {
		if (kbd[cpssp->layout][i].key == k
		 && kbd[cpssp->layout][i].shift == s
		 && kbd[cpssp->layout][i].ctrl == c
		 && kbd[cpssp->layout][i].alt == a) {
			/* found an entry */
			return i;
		}
	}
	/* if this is reached, no entry was found */
	return 0;
}

/*
 * the following table maps "special keys" like cursor keys and the keys on
 * the auxiliary keypad to special keyvalues (defined in keydefs.h)
 *
 * TODO: usual PC keyboards lack some of the VT102's keys, so we have to find
 * 	 good substitutes.
 * TODO: find out if keycodes of those "special keys" are dependant on the
 * 	 keyboard layout (de, en, ...)
 */
static const struct keyboard_special {
	unsigned char keycode;
	unsigned char value;
} kbd_sp[] = {
	{ 99, KEY_SETUP }, /* "Print Screen/SysRq" */
#ifdef VT_DEBUG
	{ 70, KEY_DEBUG }, /* scroll lock */
#endif
	{ 119, KEY_BREAK }, /* "Pause/Break" */
	{ 103, KEY_CURSOR_UP },
	{ 108, KEY_CURSOR_DOWN },
	{ 105, KEY_CURSOR_LEFT },
	{ 106, KEY_CURSOR_RIGHT },
	{ 28, KEY_RETURN },
	{ 59, KEY_PF1 }, /* "F1" - "F4" */
	{ 60, KEY_PF2 },
	{ 61, KEY_PF3 },
	{ 62, KEY_PF4 },
	{ 96, KEY_ENTER }, /* "Enter" */
	{ 82, KEY_NUMPAD_0 },
	{ 79, KEY_NUMPAD_1 },
	{ 80, KEY_NUMPAD_2 },
	{ 81, KEY_NUMPAD_3 },
	{ 75, KEY_NUMPAD_4 },
	{ 76, KEY_NUMPAD_5 },
	{ 77, KEY_NUMPAD_6 },
	{ 71, KEY_NUMPAD_7 },
	{ 72, KEY_NUMPAD_8 },
	{ 73, KEY_NUMPAD_9 },
	{ 74, KEY_NUMPAD_MINUS },
	{ 78, KEY_NUMPAD_COMMA }, /* "+" key on numpad */
	{ 83, KEY_NUMPAD_PERIOD },
	{ 111, KEY_DELETE },
	{ 69, KEY_NUMLOCK }
};

static unsigned char
lookup_special(unsigned char keycode, unsigned char notfound)
{
	int i, numentries;
	numentries = sizeof(kbd_sp) / sizeof(struct keyboard_special);
	for (i = 0; i < numentries; i++) {
		if (kbd_sp[i].keycode == keycode) {
			/* keycode belongs to a special key */
			return kbd_sp[i].value;
		}
	}
	/* keycode doesn't belong to special key */
	return notfound;
}

/* handle keypress */
static void
vt102_keyboard_event(struct cpssp *cpssp, unsigned char keycode, char press)
{
	unsigned char key;

	assert(keycode < 128);
	/* TODO :) */
	switch (keycode) {
	case 42: /* left shift */
	case 54: /* right shift */
		cpssp->shift = press;
		return;
	case 29: /* left ctrl */
	case 97: /* right ctrl */
		cpssp->ctrl = press;
		return;
	case 56: /* left alt */
	case 100: /* right alt */
		cpssp->alt = press;
		return;
	}
	/* no shift, ctrl or alt */
	if (press == 0) {
		return;
	}
	/* get the value: */
	/* first, lookup in the "standard keys" table */
	key = lookup_value(cpssp, cpssp->shift, cpssp->ctrl, cpssp->alt,
			keycode);
	/* then, test for special keys */
	key = lookup_special(keycode, key);
	if (key == 0) {
		return;
	}

	sig_integer_set(cpssp->port_keyboard, cpssp, key);
}

#define func(n) \
static void \
vt102_keyboard_key ## n ## _set(void *_cpssp, unsigned int pressed) \
{ \
	struct cpssp *cpssp = (struct cpssp *) _cpssp; \
	\
	vt102_keyboard_event(cpssp, n, pressed); \
}

func(0) func(1) func(2) func(3)
func(4) func(5) func(6) func(7)
func(8) func(9) func(10) func(11)
func(12) func(13) func(14) func(15)
func(16) func(17) func(18) func(19)
func(20) func(21) func(22) func(23)
func(24) func(25) func(26) func(27)
func(28) func(29) func(30) func(31)
func(32) func(33) func(34) func(35)
func(36) func(37) func(38) func(39)
func(40) func(41) func(42) func(43)
func(44) func(45) func(46) func(47)
func(48) func(49) func(50) func(51)
func(52) func(53) func(54) func(55)
func(56) func(57) func(58) func(59)
func(60) func(61) func(62) func(63)
func(64) func(65) func(66) func(67)
func(68) func(69) func(70) func(71)
func(72) func(73) func(74) func(75)
func(76) func(77) func(78) func(79)
func(80) func(81) func(82) func(83)
func(84) func(85) func(86) func(87)
func(88) func(89) func(90) func(91)
func(92) func(93) func(94) func(95)
func(96) func(97) func(98) func(99)
func(100) func(101) func(102) func(103)
func(104) func(105) func(106) func(107)
func(108) func(109) func(110) func(111)
func(112) func(113) func(114) func(115)
func(116) func(117) func(118) func(119)
func(120) func(121) func(122) func(123)
func(124) func(125) func(126) func(127)

#undef func

void *
vt102_keyboard_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_integer *port_keyboard,
	struct sig_boolean *port_key0,
	struct sig_boolean *port_key1,
	struct sig_boolean *port_key2,
	struct sig_boolean *port_key3,
	struct sig_boolean *port_key4,
	struct sig_boolean *port_key5,
	struct sig_boolean *port_key6,
	struct sig_boolean *port_key7,
	struct sig_boolean *port_key8,
	struct sig_boolean *port_key9,
	struct sig_boolean *port_key10,
	struct sig_boolean *port_key11,
	struct sig_boolean *port_key12,
	struct sig_boolean *port_key13,
	struct sig_boolean *port_key14,
	struct sig_boolean *port_key15,
	struct sig_boolean *port_key16,
	struct sig_boolean *port_key17,
	struct sig_boolean *port_key18,
	struct sig_boolean *port_key19,
	struct sig_boolean *port_key20,
	struct sig_boolean *port_key21,
	struct sig_boolean *port_key22,
	struct sig_boolean *port_key23,
	struct sig_boolean *port_key24,
	struct sig_boolean *port_key25,
	struct sig_boolean *port_key26,
	struct sig_boolean *port_key27,
	struct sig_boolean *port_key28,
	struct sig_boolean *port_key29,
	struct sig_boolean *port_key30,
	struct sig_boolean *port_key31,
	struct sig_boolean *port_key32,
	struct sig_boolean *port_key33,
	struct sig_boolean *port_key34,
	struct sig_boolean *port_key35,
	struct sig_boolean *port_key36,
	struct sig_boolean *port_key37,
	struct sig_boolean *port_key38,
	struct sig_boolean *port_key39,
	struct sig_boolean *port_key40,
	struct sig_boolean *port_key41,
	struct sig_boolean *port_key42,
	struct sig_boolean *port_key43,
	struct sig_boolean *port_key44,
	struct sig_boolean *port_key45,
	struct sig_boolean *port_key46,
	struct sig_boolean *port_key47,
	struct sig_boolean *port_key48,
	struct sig_boolean *port_key49,
	struct sig_boolean *port_key50,
	struct sig_boolean *port_key51,
	struct sig_boolean *port_key52,
	struct sig_boolean *port_key53,
	struct sig_boolean *port_key54,
	struct sig_boolean *port_key55,
	struct sig_boolean *port_key56,
	struct sig_boolean *port_key57,
	struct sig_boolean *port_key58,
	struct sig_boolean *port_key59,
	struct sig_boolean *port_key60,
	struct sig_boolean *port_key61,
	struct sig_boolean *port_key62,
	struct sig_boolean *port_key63,
	struct sig_boolean *port_key64,
	struct sig_boolean *port_key65,
	struct sig_boolean *port_key66,
	struct sig_boolean *port_key67,
	struct sig_boolean *port_key68,
	struct sig_boolean *port_key69,
	struct sig_boolean *port_key70,
	struct sig_boolean *port_key71,
	struct sig_boolean *port_key72,
	struct sig_boolean *port_key73,
	struct sig_boolean *port_key74,
	struct sig_boolean *port_key75,
	struct sig_boolean *port_key76,
	struct sig_boolean *port_key77,
	struct sig_boolean *port_key78,
	struct sig_boolean *port_key79,
	struct sig_boolean *port_key80,
	struct sig_boolean *port_key81,
	struct sig_boolean *port_key82,
	struct sig_boolean *port_key83,
	struct sig_boolean *port_key84,
	struct sig_boolean *port_key85,
	struct sig_boolean *port_key86,
	struct sig_boolean *port_key87,
	struct sig_boolean *port_key88,
	struct sig_boolean *port_key89,
	struct sig_boolean *port_key90,
	struct sig_boolean *port_key91,
	struct sig_boolean *port_key92,
	struct sig_boolean *port_key93,
	struct sig_boolean *port_key94,
	struct sig_boolean *port_key95,
	struct sig_boolean *port_key96,
	struct sig_boolean *port_key97,
	struct sig_boolean *port_key98,
	struct sig_boolean *port_key99,
	struct sig_boolean *port_key100,
	struct sig_boolean *port_key101,
	struct sig_boolean *port_key102,
	struct sig_boolean *port_key103,
	struct sig_boolean *port_key104,
	struct sig_boolean *port_key105,
	struct sig_boolean *port_key106,
	struct sig_boolean *port_key107,
	struct sig_boolean *port_key108,
	struct sig_boolean *port_key109,
	struct sig_boolean *port_key110,
	struct sig_boolean *port_key111,
	struct sig_boolean *port_key112,
	struct sig_boolean *port_key113,
	struct sig_boolean *port_key114,
	struct sig_boolean *port_key115,
	struct sig_boolean *port_key116,
	struct sig_boolean *port_key117,
	struct sig_boolean *port_key118,
	struct sig_boolean *port_key119,
	struct sig_boolean *port_key120,
	struct sig_boolean *port_key121,
	struct sig_boolean *port_key122,
	struct sig_boolean *port_key123,
	struct sig_boolean *port_key124,
	struct sig_boolean *port_key125,
	struct sig_boolean *port_key126,
	struct sig_boolean *port_key127
)
{
#define func(nr) \
	static const struct sig_boolean_funcs key ## nr ## _funcs = { \
		.set = vt102_keyboard_key ## nr ## _set, \
	}
	func(0); func(1); func(2); func(3);
	func(4); func(5); func(6); func(7);
	func(8); func(9); func(10); func(11);
	func(12); func(13); func(14); func(15);
	func(16); func(17); func(18); func(19);
	func(20); func(21); func(22); func(23);
	func(24); func(25); func(26); func(27);
	func(28); func(29); func(30); func(31);
	func(32); func(33); func(34); func(35);
	func(36); func(37); func(38); func(39);
	func(40); func(41); func(42); func(43);
	func(44); func(45); func(46); func(47);
	func(48); func(49); func(50); func(51);
	func(52); func(53); func(54); func(55);
	func(56); func(57); func(58); func(59);
	func(60); func(61); func(62); func(63);
	func(64); func(65); func(66); func(67);
	func(68); func(69); func(70); func(71);
	func(72); func(73); func(74); func(75);
	func(76); func(77); func(78); func(79);
	func(80); func(81); func(82); func(83);
	func(84); func(85); func(86); func(87);
	func(88); func(89); func(90); func(91);
	func(92); func(93); func(94); func(95);
	func(96); func(97); func(98); func(99);
	func(100); func(101); func(102); func(103);
	func(104); func(105); func(106); func(107);
	func(108); func(109); func(110); func(111);
	func(112); func(113); func(114); func(115);
	func(116); func(117); func(118); func(119);
	func(120); func(121); func(122); func(123);
	func(124); func(125); func(126); func(127);

#undef func

	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->layout = 0; /* FIXME */

	cpssp->shift = 0;
	cpssp->ctrl = 0;
	cpssp->alt = 0;

	cpssp->port_keyboard = port_keyboard;

	/* Out */
	sig_integer_connect_out(port_keyboard, cpssp, 0);

	/* In */
#define func(nr) \
	sig_boolean_connect_in(port_key ## nr, cpssp, &key ## nr ## _funcs)

	func(0); func(1); func(2); func(3);
	func(4); func(5); func(6); func(7);
	func(8); func(9); func(10); func(11);
	func(12); func(13); func(14); func(15);
	func(16); func(17); func(18); func(19);
	func(20); func(21); func(22); func(23);
	func(24); func(25); func(26); func(27);
	func(28); func(29); func(30); func(31);
	func(32); func(33); func(34); func(35);
	func(36); func(37); func(38); func(39);
	func(40); func(41); func(42); func(43);
	func(44); func(45); func(46); func(47);
	func(48); func(49); func(50); func(51);
	func(52); func(53); func(54); func(55);
	func(56); func(57); func(58); func(59);
	func(60); func(61); func(62); func(63);
	func(64); func(65); func(66); func(67);
	func(68); func(69); func(70); func(71);
	func(72); func(73); func(74); func(75);
	func(76); func(77); func(78); func(79);
	func(80); func(81); func(82); func(83);
	func(84); func(85); func(86); func(87);
	func(88); func(89); func(90); func(91);
	func(92); func(93); func(94); func(95);
	func(96); func(97); func(98); func(99);
	func(100); func(101); func(102); func(103);
	func(104); func(105); func(106); func(107);
	func(108); func(109); func(110); func(111);
	func(112); func(113); func(114); func(115);
	func(116); func(117); func(118); func(119);
	func(120); func(121); func(122); func(123);
	func(124); func(125); func(126); func(127);

#undef func

	return cpssp;
}

void
vt102_keyboard_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
vt102_keyboard_suspend(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
vt102_keyboard_resume(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
