/*
 * Copyright (C) 2007-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* CONFIG */
#define DEBUG_DISASSEMBLE		0

#define DEBUG_CONTROL_FLOW		0
#define DEBUG_CONTROL_FLOW_REGS		0
#define DEBUG_CONTROL_FLOW_CREGS	0
#define DEBUG_CONTROL_FLOW_SREGS	0
#define DEBUG_CONTROL_FLOW_FLAGS	0

#define EXPERIMENTAL_RESET		1

#ifdef INCLUDE

#endif /* INCLUDE */
#ifdef STATE

#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
#define NUM_GP_REGS		16
#else
#define NUM_GP_REGS		8
#endif

struct {
	/* Signal States */
	unsigned int state_error;
	unsigned int state_ignne;
	int state_interrupt_pending;
	int state_nmi_pending;
	int state_smi_pending;
	int state_n_reset;
	uint8_t fpu_error;
#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
	int state_n_init;
#endif

	/* State */
	int state_event;
	uint8_t state_halt;
	uint8_t int_delay;

	/* Registers */
	udata_t STATE_gp_regs[NUM_GP_REGS];
	udata_t STATE_eip;
	uint64_t tsc;

	uint16_t STATE_selector[8];

	/* SMM State */
	uint8_t smm;
	uint32_t smbase;

	/* Control Registers */
	uint8_t STATE_cr0_pe;	/* PE: Bit 0 */
	uint8_t STATE_cr0_mp;	/* MP: Bit 1 */
	uint8_t STATE_cr0_em;	/* EM: Bit 2 */
	uint8_t STATE_cr0_ts;	/* TS: Bit 3 */
	uint8_t STATE_cr0_et;	/* ET: Bit 4 */
	uint8_t STATE_cr0_ne;	/* NE: Bit 5 */
#if 80486 <= CONFIG_CPU
	uint8_t STATE_cr0_wp;	/* WP: Bit 16 */
#endif
#if 80486 <= CONFIG_CPU
	uint8_t STATE_cr0_am;	/* AM: Bit 18 */
#endif
	uint8_t STATE_cr0_nw;	/* NW: Bit 29 */
	uint8_t STATE_cr0_cd;	/* CD: Bit 30 */
	uint8_t STATE_cr0_pg;	/* PG: Bit 31 */

	udata_t STATE_cr2;

	paddr_t STATE_cr3;

#if 80586 <= CONFIG_CPU
#if defined CONFIG_CPU_TSC_SUPPORT && CONFIG_CPU_TSC_SUPPORT
	uint8_t STATE_cr4_tsd;	/* Bit 2 */
#endif
#if defined CONFIG_CPU_DE_SUPPORT && CONFIG_CPU_DE_SUPPORT
	uint8_t STATE_cr4_de;	/* Bit 3 */
#endif
#if defined CONFIG_CPU_PSE_SUPPORT && CONFIG_CPU_PSE_SUPPORT
	uint8_t STATE_cr4_pse;	/* Bit 4 */
#endif
#if defined CONFIG_CPU_PAE_SUPPORT && CONFIG_CPU_PAE_SUPPORT
	uint8_t STATE_cr4_pae;	/* Bit 5 */
#endif
#if defined CONFIG_CPU_MCA_SUPPORT && CONFIG_CPU_MCA_SUPPORT
	uint8_t STATE_cr4_mce;	/* Bit 6 */
#endif
#if defined CONFIG_CPU_PGE_SUPPORT && CONFIG_CPU_PGE_SUPPORT
	uint8_t STATE_cr4_pge;	/* Bit 7 */
#endif
#endif

		/* dr */
	struct {
		uint32_t addr;
		uint8_t len;
		uint8_t type;
		uint8_t detected;
		uint8_t local_enabled;
		uint8_t global_enabled;
	} STATE_dr[4];
	uint8_t STATE_dr_bs;

		/* eflags */
	uint8_t STATE_e_cf;
	uint8_t STATE_e_pf;
	uint8_t STATE_e_af;
	uint8_t STATE_e_zf;
	uint8_t STATE_e_sf;
	uint8_t STATE_e_tf;
	uint8_t STATE_e_tf_delayed;
	uint8_t STATE_e_if;
	uint8_t STATE_e_df;
	uint8_t STATE_e_of;
	uint8_t STATE_e_iopl;
	uint8_t STATE_e_nt;
	uint8_t STATE_e_vm;
#if 80486 <= CONFIG_CPU
	uint8_t STATE_e_ac;
#endif
#if 80586 <= CONFIG_CPU
	uint8_t STATE_e_id;
#endif

	int8_t direction;

#if defined CONFIG_CPU_SEP_SUPPORT && CONFIG_CPU_SEP_SUPPORT
	uint32_t STATE_sysenter_cs_msr;
	uint32_t STATE_sysenter_eip_msr;
	uint32_t STATE_sysenter_esp_msr;
#endif

#if defined CONFIG_CPU_MCA_SUPPORT && CONFIG_CPU_MCA_SUPPORT
	uint64_t mca_mcg;
	uint64_t mca_mcX_ctl[5];
#endif

#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	uint8_t STATE_lm_enabled_msr;
	uint8_t STATE_lm_active_msr;
#endif
#if defined CONFIG_CPU_NX_SUPPORT && CONFIG_CPU_NX_SUPPORT
	uint8_t STATE_efer_nxe;
#endif

		/* exception */
	uint16_t errc;
	uint8_t exception_pending;
	uint32_t exception_num;
	uint16_t error_code;

#if 80686 <= CONFIG_CPU
	uint32_t update_signature;
#endif

	uint16_t fpu_opcode;
	uint32_t fpu_eip;
	uint16_t fpu_cs;
	uint32_t fpu_addr;
	uint16_t fpu_ds;
} NAME;

#endif /* STATE */
#ifdef EXPORT

#if 80486 <= CONFIG_CPU
/*forward*/ static int
NAME_(cr0_wp_get)(struct cpssp *cpssp);
#endif
/*forward*/ static int
NAME_(cr0_pg_get)(struct cpssp *cpssp);
/*forward*/ static paddr_t
NAME_(cr3_get)(struct cpssp *cpssp);
#if defined CONFIG_CPU_PGE_SUPPORT && CONFIG_CPU_PGE_SUPPORT
/*forward*/ static int
NAME_(cr4_pge_get)(struct cpssp *cpssp);
#endif
#if defined CONFIG_CPU_PSE_SUPPORT && CONFIG_CPU_PSE_SUPPORT
/*forward*/ static int
NAME_(cr4_pse_get)(struct cpssp *cpssp);
#endif
#if defined CONFIG_CPU_PAE_SUPPORT && CONFIG_CPU_PAE_SUPPORT
/*forward*/ static int
NAME_(cr4_pae_get)(struct cpssp *cpssp);
#endif
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
/*forward*/ static int
NAME_(efer_lme_get)(struct cpssp *cpssp);
#endif
#if defined CONFIG_CPU_NX_SUPPORT && CONFIG_CPU_NX_SUPPORT
/*forward*/ static int
NAME_(efer_nxe_get)(struct cpssp *cpssp);
#endif
#if 80486 <= CONFIG_CPU
/*forward*/ static uint8_t
NAME_(align_ac_get)(struct cpssp *cpssp);
#endif
/*forward*/ static void
NAME_(align_cr2_set)(struct cpssp *cpssp, uint32_t val);
#if CONFIG_FPU
#if 80686 <= CONFIG_CPU
/*forward*/ static void
NAME_(fpu_set_eflags)(struct cpssp *cpssp, uint8_t eflags);
#endif
#if defined CONFIG_CPU_CMOV_SUPPORT && CONFIG_CPU_CMOV_SUPPORT
/*forward*/ static uint8_t
NAME_(fpu_get_eflags)(struct cpssp *cpssp);
#endif
/*forward*/ static void
NAME_(n_ignne_set)(struct cpssp *cpssp, unsigned int val);
#if ! CONFIG_INSPECTION
/*forward*/ static void
NAME_(n_ferr_set)(struct cpssp *cpssp, unsigned int val);
#endif /* ! CONFIG_INSPECTION) */
#endif
/*forward*/ static void
NAME_(n_error_set)(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
NAME_(irq_set)(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
NAME_(nmi_set)(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
NAME_(smi_set)(struct cpssp *cpssp, unsigned int val);
#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
/*forward*/ static void
NAME_(startup_by_apic)(struct cpssp *cpssp, uint8_t vec);
/*forward*/ static void
NAME_(init_by_apic)(struct cpssp *cpssp);
#endif /* defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT */
/*forward*/ static void
NAME_(n_reset_set)(struct cpssp *cpssp, unsigned int val);
#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
/*forward*/ static void
NAME_(n_init_set)(struct cpssp *cpssp, unsigned int val);
#endif
/*forward*/ static void
NAME_(power_set)(struct cpssp *cpssp, unsigned int val);
#if ! CONFIG_INSPECTION
/*forward*/ static void
NAME_(create)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(destroy)(struct cpssp *cpssp);
#endif /* ! CONFIG_INSPECTION */

#endif /* EXPORT */
#ifdef BEHAVIOR

enum {
	NAME_(EXC_NONE) = 0, /* FIXME */

	NAME_(EXC_DE) = 0,
	NAME_(EXC_DB) = 1,
	NAME_(EXC_NMI) = 2,
	NAME_(EXC_BP) = 3,
	NAME_(EXC_OF) = 4,
	NAME_(EXC_BR) = 5,
	NAME_(EXC_UD) = 6,
	NAME_(EXC_NM) = 7,
	NAME_(EXC_DF) = 8,
	NAME_(EXC_COPROC) = 9,
	NAME_(EXC_TS) = 10,
	NAME_(EXC_NP) = 11,
	NAME_(EXC_SS) = 12,
	NAME_(EXC_GP) = 13,
	NAME_(EXC_PF) = 14,
//	NAME_(EXC_??) = 15,
	NAME_(EXC_MF) = 16,
#if 80486 <= CONFIG_CPU
	NAME_(EXC_AC) = 17,
#endif
#if defined CONFIG_CPU_MCA_SUPPORT && CONFIG_CPU_MCA_SUPPORT
	NAME_(EXC_MC) = 18,
#endif
#if defined CONFIG_CPU_SSE_SUPPORT && CONFIG_CPU_SSE_SUPPORT
	NAME_(EXC_XF) = 19,
#endif
};

enum {
	NAME_(SEG_ES) = 0x0,
	NAME_(SEG_CS) = 0x1,
	NAME_(SEG_SS) = 0x2,
	NAME_(SEG_DS) = 0x3,
	NAME_(SEG_FS) = 0x4,
	NAME_(SEG_GS) = 0x5,
	NAME_(SEG_LDT) = 0x6,
	NAME_(SEG_TR) = 0x7,
	NAME_(SEG_GDT) = 0x8,
	NAME_(SEG_IDT) = 0x9,
	NAME_(SEG_CODE) = 0xa,
	NAME_(SEG_DATA) = 0xb,
};

enum {
	NAME_(E_AX) = 0x0,
	NAME_(E_CX) = 0x1,
	NAME_(E_DX) = 0x2,
	NAME_(E_BX) = 0x3,
	NAME_(E_SP) = 0x4,
	NAME_(E_BP) = 0x5,
	NAME_(E_SI) = 0x6,
	NAME_(E_DI) = 0x7,
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	NAME_(R8) = 0x8,
	NAME_(R9) = 0x9,
	NAME_(R10) = 0xa,
	NAME_(R11) = 0xb,
	NAME_(R12) = 0xc,
	NAME_(R13) = 0xd,
	NAME_(R14) = 0xe,
	NAME_(R15) = 0xf,
#endif
};

enum {
	NAME_(AL) = NAME_(E_AX),
	NAME_(CL) = NAME_(E_CX),
	NAME_(DL) = NAME_(E_DX),
	NAME_(BL) = NAME_(E_BX),
	NAME_(AH) = NAME_(E_AX) + 4,
	NAME_(CH) = NAME_(E_CX) + 4,
	NAME_(DH) = NAME_(E_DX) + 4,
	NAME_(BH) = NAME_(E_BX) + 4,
};

struct NAME_(cpu) {
	/* tmp vars */
	        /* working copy */
	udata_t eip;
	udata_t esp; // TODO FIXME XXX IMPORTANT fix esp in funcs XXX

	        /* mode */
	uint8_t pre_op;
	uint8_t pre_addr;
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	uint8_t pre_rex;
#endif
	uint8_t pre_repe_repne;

	        /* addr */
	uint8_t reg_opex;
	uint8_t rm;
	uint8_t mod;
	uint8_t seg[2];
	uint8_t seg_override;
	udata_t addr;

#if ! EXPERIMENTAL_RESET
	        /* exception */
	jmp_buf exception_buf;
#endif
};

#if 80486 <= CONFIG_CPU
static int
NAME_(cr0_wp_get)(struct cpssp *cpssp)
{
	return STATE_GET(cr0_wp);
}
#endif /* 80486 <= CONFIG_CPU */

static int
NAME_(cr0_pg_get)(struct cpssp *cpssp)
{
	return STATE_GET(cr0_pg);
}

static paddr_t
NAME_(cr3_get)(struct cpssp *cpssp)
{
	return STATE_GET(cr3);
}

#if defined CONFIG_CPU_PGE_SUPPORT && CONFIG_CPU_PGE_SUPPORT
static int
NAME_(cr4_pge_get)(struct cpssp *cpssp)
{
	return STATE_GET(cr4_pge);
}
#endif

#if defined CONFIG_CPU_PSE_SUPPORT && CONFIG_CPU_PSE_SUPPORT
static int
NAME_(cr4_pse_get)(struct cpssp *cpssp)
{
	return STATE_GET(cr4_pse);
}
#endif

#if defined CONFIG_CPU_PAE_SUPPORT && CONFIG_CPU_PAE_SUPPORT
static int
NAME_(cr4_pae_get)(struct cpssp *cpssp)
{
	return STATE_GET(cr4_pae);
}
#endif

#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
static int
NAME_(efer_lme_get)(struct cpssp *cpssp)
{
	return STATE_GET(lm_active_msr);
}
#endif

#if defined CONFIG_CPU_NX_SUPPORT && CONFIG_CPU_NX_SUPPORT
static int
NAME_(efer_nxe_get)(struct cpssp *cpssp)
{
	return STATE_GET(efer_nxe);
}
#endif

/* ################ SEG FUNCS ############################################## */

static int
NAME_(seg_present)(struct cpssp *cpssp, uint8_t seg)
{
	seg_descr_t descr;

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, seg);

	return (descr.entry[0] >> 47) & 0x1;
}

static uint32_t
NAME_(seg_ar)(struct cpssp *cpssp, uint8_t seg)
{
	seg_descr_t descr;

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, seg);

	return (descr.entry[0] >> 32) & 0x00f0ff00;
}

#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
static uint8_t
NAME_(seg_l)(struct cpssp *cpssp, uint8_t seg)
{
	seg_descr_t descr;

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, seg);

	return (descr.entry[0] >> 53) & 0x1;
}

static uint8_t
NAME_(seg_istn)(struct cpssp *cpssp, uint8_t seg)
{
	seg_descr_t descr;

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, seg);

	return (descr.entry[0] >> 32) & 0x7;
}
#endif

static uint8_t
NAME_(seg_db)(struct cpssp *cpssp, uint8_t seg)
{
	seg_descr_t descr;

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, seg);

	return (descr.entry[0] >> 54) & 0x1;
}

static uint8_t
NAME_(seg_dpl)(struct cpssp *cpssp, uint8_t seg)
{
	seg_descr_t descr;

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, seg);

	return (descr.entry[0] >> 45) & 0x3;
}

static uint8_t
NAME_(seg_type)(struct cpssp *cpssp, uint8_t seg)
{
	seg_descr_t descr;

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, seg);

	return (descr.entry[0] >> 40) & 0x1f;
}

static uint32_t
NAME_(seg_limit)(struct cpssp *cpssp, uint8_t seg)
{
	seg_descr_t descr;
	uint32_t limit;
	uint32_t granularity;

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, seg);

	limit = (((descr.entry[0] >> 48) & 0xf) << 16) | ((descr.entry[0] & 0xffff) << 0);
	granularity = (descr.entry[0] >> 55) & 0x1;

	return (limit << (12 * granularity)) | (0xfff * granularity);
}

static uint32_t
NAME_(seg_base)(struct cpssp *cpssp, uint8_t seg)
{
	seg_descr_t descr;

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, seg);

	return (((descr.entry[0] >> 56) & 0xff) << 24)
		| (((descr.entry[0] >> 32) & 0xff) << 16)
		| (((descr.entry[0] >> 16) & 0xffff) << 0);
}

static udata_t
NAME_(seg_offset)(struct cpssp *cpssp, uint8_t seg)
{
	seg_descr_t descr;
	udata_t offset;

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, seg);
	offset = 0
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
		| ((descr.entry[1] & 0xffffffff) << 32)
#endif
		| ((descr.entry[0] >> 32) & 0xffff0000) | (descr.entry[0] & 0x0000ffff);

	return offset;
}

static uint16_t
NAME_(seg_selector)(struct cpssp *cpssp, uint8_t seg)
{
	seg_descr_t descr;

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, seg);

	return (descr.entry[0] >> 16) & 0xffff;
}

static uint8_t
NAME_(seg_param)(struct cpssp *cpssp, uint8_t seg)
{
	seg_descr_t descr;

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, seg);

	return (descr.entry[0] >> 32) & 0x1f;
}

/* ################ EXCEPTIONS ############################################# */

#if EXPERIMENTAL_RESET
/* If our instruction modifies memory in an unpredictable fashion, add "memory" to the list of clobbered registers.
 * This will cause GCC to not keep memory values cached in registers across the assembler instruction. We also have
 * to add the volatile keyword if the memory affected is not listed in the inputs or outputs of the asm.
 * */
static void
NAME_(reset_component)(struct cpssp *cpssp) {
#if defined(__i386__)
	uint32_t *sp;
	sp = cpssp->process.tos;

	*--sp = (uint32_t) cpssp;
	*--sp = 0; /* Return Address */

	*--sp = (uint32_t) cpssp->process.f; /* eip */
	asm volatile (
		"movl %0, %%esp\n\t"
		"ret\n\t"
		:: "r" (sp)
		: "memory"
	);

#elif defined(__x86_64__)
	uint64_t *sp;

	sp = cpssp->process.tos;

	*--sp = 0; /* Return Address */

	*--sp = (uint64_t) cpssp->process.f; /* rip */
	asm volatile (
		"movq %1, %%rsp\n\t"
		"ret\n\t"
		:: "D" (cpssp), "r" (sp)
		: "memory"
	);
#else
#error Unknown CPU
#endif
}
#endif

static void
NAME_(store_reg32)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg_num, uint32_t val);

static void __attribute__((__noreturn__))
NAME_(exception)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t num, uint8_t errc, uint16_t error_code)
{
	if (cpssp->NAME.exception_pending != 0) {
		fprintf(stderr, "Double fault at %08x (%04x:%08x) (TSC: %llu)\n",
			NAME_(seg_base)(cpssp, NAME_(SEG_CS)) + STATE_GET(eip),
			STATE_GET(selector[NAME_(SEG_CS)]), STATE_GET(eip),
			cpssp->process.inst_cnt);
		fprintf(stderr, "First: 0x%02x (0x%04x), second: 0x%02x (0x%04x)\n",
				cpssp->NAME.exception_num, cpssp->NAME.error_code,
				num, error_code);
		fprintf(stderr, "Not implemented...\n");
		assert(0);
	}
	cpssp->NAME.exception_pending = 1;
	cpssp->NAME.exception_num = num;
	cpssp->NAME.errc = errc;
	cpssp->NAME.error_code = error_code;
	cpssp->NAME.state_event = 1;
	NAME_(store_reg32)(cpssp, cpu, NAME_(E_SP), cpu->esp);
	cpssp->process.inst_cnt++;
#if EXPERIMENTAL_RESET
	NAME_(reset_component)(cpssp);
#else
	longjmp(cpu->exception_buf, 1);
#endif
	/* make gcc happy */
	assert(0);
}

static void
NAME_(DE)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	fprintf(stderr, "WARNING: Divide error at %08x (%04x:%08x) (TSC: %llu)\n",
			NAME_(seg_base)(cpssp, NAME_(SEG_CS)) + STATE_GET(eip),
			STATE_GET(selector[NAME_(SEG_CS)]), STATE_GET(eip),
			cpssp->process.inst_cnt);
	NAME_(exception)(cpssp, cpu, NAME_(EXC_DE), 0, 0);
}

#if 0 /* Not used. */
static void
NAME_(DB)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	NAME_(exception)(cpssp, cpu, NAME_(EXC_DB), 0, 0);
}
#endif

/* 2: NMI */

#if 0 /* Not used. */
static void
NAME_(BP)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	NAME_(exception)(cpssp, cpu, NAME_(EXC_BP), 0, 0);
}
#endif

#if 0 /* Not used. */
static void
NAME_(OF)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	NAME_(exception)(cpssp, cpu, NAME_(EXC_OF), 0, 0);
}
#endif

static void
NAME_(BR)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	fprintf(stderr, "WARNING: Bound range error at %08x (%04x:%08x) (TSC: %llu)\n",
			NAME_(seg_base)(cpssp, NAME_(SEG_CS)) + STATE_GET(eip),
			STATE_GET(selector[NAME_(SEG_CS)]), STATE_GET(eip),
			cpssp->process.inst_cnt);
	NAME_(exception)(cpssp, cpu, NAME_(EXC_BR), 0, 0);
}

static void
NAME_(UD)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t opcode)
{
	if (opcode <= 0xff) {
		fprintf(stderr, "WARNING: Unknown opcode %02x at %08x (%04x:%08x) (TSC: %llu)\n",
				opcode,
				NAME_(seg_base)(cpssp, NAME_(SEG_CS)) + STATE_GET(eip),
				STATE_GET(selector[NAME_(SEG_CS)]), STATE_GET(eip),
				cpssp->process.inst_cnt);
	} else {
		fprintf(stderr, "WARNING: Unknown opcode %02x %02x at %08x (%04x:%08x) (TSC: %llu)\n",
				(opcode >> 8) & 0xff, (opcode >> 0) & 0xff,
				NAME_(seg_base)(cpssp, NAME_(SEG_CS)) + STATE_GET(eip),
				STATE_GET(selector[NAME_(SEG_CS)]), STATE_GET(eip),
				cpssp->process.inst_cnt);
	}
	NAME_(exception)(cpssp, cpu, NAME_(EXC_UD), 0, 0);
}

static void
NAME_(NM)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	NAME_(exception)(cpssp, cpu, NAME_(EXC_NM), 0, 0);
}

#if 0 /* Not used. */
static void
NAME_(DF)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	NAME_(exception)(cpssp, cpu, NAME_(EXC_DF), 0, 0);
}
#endif

/* 9: Coprocessor Segment Overrun */

static void
NAME_(TSe)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t a)
{
	fprintf(stderr, "WARNING: Task state error %04x at %08x (%04x:%08x) (TSC: %llu)\n",
			a,
			NAME_(seg_base)(cpssp, NAME_(SEG_CS)) + STATE_GET(eip),
			STATE_GET(selector[NAME_(SEG_CS)]), STATE_GET(eip),
			cpssp->process.inst_cnt);
	NAME_(exception)(cpssp, cpu, NAME_(EXC_TS), 1, a);
}

static void
NAME_(NPe)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t a)
{
	fprintf(stderr, "WARNING: Not present error %04x at %08x (%04x:%08x) (TSC: %llu)\n",
			a,
			NAME_(seg_base)(cpssp, NAME_(SEG_CS)) + STATE_GET(eip),
			STATE_GET(selector[NAME_(SEG_CS)]), STATE_GET(eip),
			cpssp->process.inst_cnt);
	NAME_(exception)(cpssp, cpu, NAME_(EXC_NP), 1, a);
}

static void
NAME_(SSe)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t a)
{
	fprintf(stderr, "WARNING: Stack segment error %04x at %08x (%04x:%08x) (TSC: %llu)\n",
			a,
			NAME_(seg_base)(cpssp, NAME_(SEG_CS)) + STATE_GET(eip),
			STATE_GET(selector[NAME_(SEG_CS)]), STATE_GET(eip),
			cpssp->process.inst_cnt);
	NAME_(exception)(cpssp, cpu, NAME_(EXC_SS), 1, a);
}

static void
NAME_(GPe)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t a)
{
	fprintf(stderr, "WARNING: General protection error %04x at %08x (%04x:%08x) (TSC: %llu)\n",
			a,
			NAME_(seg_base)(cpssp, NAME_(SEG_CS)) + STATE_GET(eip),
			STATE_GET(selector[NAME_(SEG_CS)]), STATE_GET(eip),
			cpssp->process.inst_cnt);
	NAME_(exception)(cpssp, cpu, NAME_(EXC_GP), 1, a);
}

static void
NAME_(PFe)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t a)
{
#if 0 /* Happens too often. */
	fprintf(stderr, "WARNING: Page fault error %04x at %08x (%04x:%08x) (TSC: %llu)\n",
			a,
			NAME_(seg_base)(cpssp, NAME_(SEG_CS)) + STATE_GET(eip),
			STATE_GET(selector[NAME_(SEG_CS)]), STATE_GET(eip),
			cpssp->process.inst_cnt);
#endif
	NAME_(exception)(cpssp, cpu, NAME_(EXC_PF), 1, a);
}

/* 15: Intel Reserved */

static void
NAME_(MF)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	fprintf(stderr, "WARNING: Math error at %08x (%04x:%08x) (TSC: %llu)\n",
			NAME_(seg_base)(cpssp, NAME_(SEG_CS)) + STATE_GET(eip),
			STATE_GET(selector[NAME_(SEG_CS)]), STATE_GET(eip),
			cpssp->process.inst_cnt);
	NAME_(exception)(cpssp, cpu, NAME_(EXC_MF), 0, 0);
}

#if 80486 <= CONFIG_CPU
static void
NAME_(AC)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	NAME_(exception)(cpssp, cpu, NAME_(EXC_AC), 1, 0);
}
#endif

static void
NAME_(mem_err)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int err, uint16_t errc)
{
	switch (err) {
	case NAME_(EXC_GP):
		NAME_(GPe)(cpssp, cpu, errc);
		/*NOTREACHED*/
	case NAME_(EXC_PF):
		NAME_(PFe)(cpssp, cpu, errc);
		/*NOTREACHED*/
#if 80486 <= CONFIG_CPU
	case NAME_(EXC_AC):
		NAME_(AC)(cpssp, cpu);
		/*NOTREACHED*/
#endif
	default:
		assert(0); /* Mustn't happen. */
	}
}

static void
NAME_(mem_err2)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int err0, uint16_t errc0, int err1, uint16_t errc1)
{
	if (err0 == NAME_(EXC_GP)) {
		err0 = err1;
		errc0 = errc1;
	}

	switch (err0) {
	case NAME_(EXC_GP):
		NAME_(GPe)(cpssp, cpu, errc0);
		/*NOTREACHED*/
	case NAME_(EXC_PF):
		NAME_(PFe)(cpssp, cpu, errc0);
		/*NOTREACHED*/
#if 80486 <= CONFIG_CPU
	case NAME_(EXC_AC):
		NAME_(AC)(cpssp, cpu);
		/*NOTREACHED*/
#endif
	default:
		assert(0); /* Mustn't happen. */
	}
}

/* ################ MISC ############################################# */

static uint8_t
NAME_(load_reg8)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg_num)
{
	uint8_t bit;
	uint8_t reg;
	uint8_t res;
	reg = reg_num & 0x3;
	bit = (reg_num & 0x4) << 1;
	res = (STATE_GET(gp_regs[reg]) >> bit);
	return res;
}

static void
NAME_(store_reg8)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg_num, uint8_t val)
{
	uint8_t bit;
	uint8_t reg;
	uint32_t tmp;
	uint32_t mask;
	uint32_t val32;

	reg = reg_num & 0x3;
	bit = (reg_num & 0x4) << 1;
	tmp = STATE_GET(gp_regs[reg]);
	mask = ~(0xff << bit);
	tmp &= mask;
	val32 = val;
	tmp |= (val32 << bit);
	STATE_SET(gp_regs[reg], tmp);
}

static uint16_t
NAME_(load_reg16)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg_num)
{
	return STATE_GET(gp_regs[reg_num]);
}

static void
NAME_(store_reg16)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg_num, uint16_t val)
{
	uint32_t tmp;

	tmp = STATE_GET(gp_regs[reg_num]);
	tmp &= 0xffff0000;
	tmp |= val;
	STATE_SET(gp_regs[reg_num], tmp);
}

static uint32_t
NAME_(load_reg32)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg_num)
{
	return STATE_GET(gp_regs[reg_num]);
}

static void
NAME_(store_reg32)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg_num, uint32_t val)
{
	STATE_SET(gp_regs[reg_num], val);
}

#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
static uint64_t
NAME_(load_reg64)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg_num)
{
	return STATE_GET(gp_regs[reg_num]);
}

static void
NAME_(store_reg64)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg_num, uint64_t val)
{
	STATE_SET(gp_regs[reg_num], val);
}
#endif

static udata_t
NAME_(load_reg)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg_num, int op_mode)
{
	switch (op_mode) {
	case 0:
		return NAME_(load_reg8)(cpssp, cpu, reg_num);
	case 1:
		return NAME_(load_reg16)(cpssp, cpu, reg_num);
	case 2:
		return NAME_(load_reg32)(cpssp, cpu, reg_num);
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	case 3:
		return NAME_(load_reg64)(cpssp, cpu, reg_num);
#endif
	default:
		assert(0); /* Mustn't happen. */
	}
}

static void
NAME_(store_reg)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg_num, udata_t val, int op_mode)
{
	switch (op_mode) {
	case 0:
		NAME_(store_reg8)(cpssp, cpu, reg_num, val);
		break;
	case 1:
		NAME_(store_reg16)(cpssp, cpu, reg_num, val);
		break;
	case 2:
		NAME_(store_reg32)(cpssp, cpu, reg_num, val);
		break;
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	case 3:
		NAME_(store_reg64)(cpssp, cpu, reg_num, val);
		break;
#endif
	default:
		assert(0); /* Mustn't happen. */
	}
}

static void
NAME_(set_cr)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg, uint32_t val)
{
	switch (reg) {
	case 0x0:
		if (((val >> 31) & 0x1) && ! (val & 0x1)) {
			NAME_(GPe)(cpssp, cpu, 0);
		}
#if (defined CONFIG_CPU_PAE_SUPPORT && CONFIG_CPU_PAE_SUPPORT) \
	&& (defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT)
		if (((val >> 31) & 0x1) && STATE_GET(lm_enabled_msr)) {
			if (! STATE_GET(cr4_pae)) {
				NAME_(GPe)(cpssp, cpu, 0);
			}
			STATE_SET(lm_active_msr, 1);
		} else {
			STATE_SET(lm_active_msr, 0);
		}
#endif
		if (((val >> 31) & 1) != STATE_GET(cr0_pg)
		 || ((val >> 0) & 1) != STATE_GET(cr0_pe)) {
			DEFAULT_NAME_(mmu_tlb_flush)(cpssp, 1);
		}

		STATE_SET(cr0_pg, (val >> 31) & 0x1);
		STATE_SET(cr0_cd, (val >> 30) & 0x1);
		STATE_SET(cr0_nw, (val >> 29) & 0x1);
#if 80486 <= CONFIG_CPU
		STATE_SET(cr0_wp, (val >> 16) & 0x1);
#endif
#if 80486 <= CONFIG_CPU
		STATE_SET(cr0_am, (val >> 18) & 0x1);
#endif
		STATE_SET(cr0_ne, (val >> 5) & 0x1);
		STATE_SET(cr0_et, (val >> 4) & 0x1);
		STATE_SET(cr0_ts, (val >> 3) & 0x1);
		STATE_SET(cr0_em, (val >> 2) & 0x1);
		STATE_SET(cr0_mp, (val >> 1) & 0x1);
		STATE_SET(cr0_pe, (val >> 0) & 0x1);
		break;

	case 0x2:
		STATE_SET(cr2, val);
		break;

	case 0x3:
		DEFAULT_NAME_(mmu_tlb_flush)(cpssp, 0);
		STATE_SET(cr3, val);
		break;

#if 80586 <= CONFIG_CPU
	case 0x4:
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
		if (STATE_GET(lm_active_msr) && !((val >> 5) & 1)) {
			NAME_(GPe)(cpssp, cpu, 0);
		}
#endif
		if (0
#if defined CONFIG_CPU_PGE_SUPPORT && CONFIG_CPU_PGE_SUPPORT
			|| ((val >> 7) & 1) != STATE_GET(cr4_pge)
#endif
#if defined CONFIG_CPU_PAE_SUPPORT && CONFIG_CPU_PAE_SUPPORT
			|| ((val >> 5) & 1) != STATE_GET(cr4_pae)
#endif
#if defined CONFIG_CPU_PSE_SUPPORT && CONFIG_CPU_PSE_SUPPORT
			|| ((val >> 4) & 1) != STATE_GET(cr4_pse)
#endif
				) {
			DEFAULT_NAME_(mmu_tlb_flush)(cpssp, 1);
		}

#if defined CONFIG_CPU_PGE_SUPPORT && CONFIG_CPU_PGE_SUPPORT
		STATE_SET(cr4_pge, (val >> 7) & 1);
#endif
#if defined CONFIG_CPU_MCA_SUPPORT && CONFIG_CPU_MCA_SUPPORT
		STATE_SET(cr4_mce, (val >> 6) & 1);
#endif
#if defined CONFIG_CPU_PAE_SUPPORT && CONFIG_CPU_PAE_SUPPORT
		STATE_SET(cr4_pae, (val >> 5) & 1);
#endif
#if defined CONFIG_CPU_PSE_SUPPORT && CONFIG_CPU_PSE_SUPPORT
		STATE_SET(cr4_pse, (val >> 4) & 1);
#endif
#if defined CONFIG_CPU_DE_SUPPORT && CONFIG_CPU_DE_SUPPORT
		STATE_SET(cr4_de, (val >> 3) & 1);
#endif
#if defined CONFIG_CPU_TSC_SUPPORT && CONFIG_CPU_TSC_SUPPORT
		STATE_SET(cr4_tsd, (val >> 2) & 1);
#endif
		break;
#endif

#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
	case 0x8:
		DEFAULT_NAME_(apic_tpr_set)(cpssp, val);
		break;
#endif

	default:
		fprintf(stderr, "WARNING: %s reg=%d\n", __FUNCTION__, reg);
		break;
	}
}

static uint32_t
NAME_(get_cr)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg)
{
	uint32_t val;

	switch (reg) {
	case 0x0:
		val = (STATE_GET(cr0_pg) << 31)
			| (STATE_GET(cr0_cd) << 30)
			| (STATE_GET(cr0_nw) << 29)
#if 80486 <= CONFIG_CPU
			| (STATE_GET(cr0_am) << 18)
#endif
#if 80486 <= CONFIG_CPU
			| (STATE_GET(cr0_wp) << 16)
#endif
			| (STATE_GET(cr0_ne) << 5)
			| (STATE_GET(cr0_et) << 4)
			| (STATE_GET(cr0_ts) << 3)
			| (STATE_GET(cr0_em) << 2)
			| (STATE_GET(cr0_mp) << 1)
			| (STATE_GET(cr0_pe) << 0);
		break;

	case 0x2:
		val = STATE_GET(cr2);
		break;

	case 0x3:
		val = STATE_GET(cr3);
		break;

#if 80586 <= CONFIG_CPU
	case 0x4:
		val = 0;
#if defined CONFIG_CPU_PGE_SUPPORT && CONFIG_CPU_PGE_SUPPORT
		val |= STATE_GET(cr4_pge) << 7;
#endif
#if defined CONFIG_CPU_MCA_SUPPORT && CONFIG_CPU_MCA_SUPPORT
		val |= STATE_GET(cr4_mce) << 6;
#endif
#if defined CONFIG_CPU_PAE_SUPPORT && CONFIG_CPU_PAE_SUPPORT
		val |= STATE_GET(cr4_pae) << 5;
#endif
#if defined CONFIG_CPU_PSE_SUPPORT && CONFIG_CPU_PSE_SUPPORT
		val |= STATE_GET(cr4_pse) << 4;
#endif
#if defined CONFIG_CPU_DE_SUPPORT && CONFIG_CPU_DE_SUPPORT
		val |= STATE_GET(cr4_de) << 3;
#endif
#if defined CONFIG_CPU_TSC_SUPPORT && CONFIG_CPU_TSC_SUPPORT
		val |= STATE_GET(cr4_tsd) << 2;
#endif
		break;
#endif

#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
	case 0x8:
		val = DEFAULT_NAME_(apic_tpr_get)(cpssp);
		break;
#endif

	default:
		fprintf(stderr, "WARNING: %s reg=%d\n", __FUNCTION__, reg);
		val = 0;
		break;
	}

	return val;
}

static uint16_t
NAME_(smsw)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	return (uint16_t) NAME_(get_cr)(cpssp, cpu, 0);
}

static void
NAME_(lmsw)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t val)
{
	uint32_t tmp;

	tmp = NAME_(get_cr)(cpssp, cpu, 0);
	tmp &= 0xfffffff1; /* Protected Mode cannot be switched off. */
	tmp |= val & 0xf;
	NAME_(set_cr)(cpssp, cpu, 0, tmp);
}

static void
NAME_(align_cr2_set)(struct cpssp *cpssp, uint32_t val)
{
	STATE_SET(cr2, val);
}

static void
NAME_(set_dr)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg, uint32_t val)
{
	switch (reg) {
	case 0:
	case 1:
	case 2:
	case 3:
		if (val != 0) {
			fprintf(stderr, "WARNING: Writing 0x%08x to debug register %d.\n", val, reg);
			fprintf(stderr, "WARNING: Not implemented, yet!\n");
		}

		STATE_SET(dr[reg].addr, val);
		break;

	case 4:
	case 5:
		assert(0); /* Mustn't happen. */
		break;

	case 6:
		if ((val & 0x0000e00f) != 0) {
			fprintf(stderr, "WARNING: Writing 0x%08x to debug register %d.\n", val, reg);
			fprintf(stderr, "WARNING: Not implemented, yet!\n");
		}

		/* More... - FIXME */
		STATE_SET(dr_bs, (val >> 14) & 1);
		/* More... - FIXME */
		STATE_SET(dr[3].detected, (val >> 3) & 1);
		STATE_SET(dr[2].detected, (val >> 2) & 1);
		STATE_SET(dr[1].detected, (val >> 1) & 1);
		STATE_SET(dr[0].detected, (val >> 0) & 1);
		break;

	case 7:
		if ((val & 0xffff23ff) != 0) {
			fprintf(stderr, "WARNING: Writing 0x%08x to debug register %d.\n", val, reg);
			fprintf(stderr, "WARNING: Not implemented, yet!\n");
		}

		STATE_SET(dr[3].type, (val >> 28) & 0x3);
		STATE_SET(dr[3].len, (val >> 30) & 0x3);
		STATE_SET(dr[2].type, (val >> 24) & 0x3);
		STATE_SET(dr[2].len, (val >> 26) & 0x3);
		STATE_SET(dr[1].type, (val >> 20) & 0x3);
		STATE_SET(dr[1].len, (val >> 22) & 0x3);
		STATE_SET(dr[0].type, (val >> 16) & 0x3);
		STATE_SET(dr[0].len, (val >> 18) & 0x3);
		/* More ... - FIXME */
		STATE_SET(dr[3].local_enabled, (val >> 6) & 1);
		STATE_SET(dr[3].global_enabled, (val >> 7) & 1);
		STATE_SET(dr[2].local_enabled, (val >> 4) & 1);
		STATE_SET(dr[2].global_enabled, (val >> 5) & 1);
		STATE_SET(dr[1].local_enabled, (val >> 2) & 1);
		STATE_SET(dr[1].global_enabled, (val >> 3) & 1);
		STATE_SET(dr[0].local_enabled, (val >> 0) & 1);
		STATE_SET(dr[0].global_enabled, (val >> 1) & 1);
		break;

	default:
		assert(0); /* Mustn't happen. */
		break;
	}
}

static uint32_t
NAME_(get_dr)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg)
{
	uint32_t val;

	switch (reg) {
	case 0:
	case 1:
	case 2:
	case 3:
		val = STATE_GET(dr[reg].addr);
		break;

	case 4:
	case 5:
		assert(0); /* Mustn't happen. */
		break;

	case 6:
		/* More ... - FIXME */
		val = (STATE_GET(dr_bs) << 14)
		/* More ... - FIXME */
			| (STATE_GET(dr[3].detected) << 3)
			| (STATE_GET(dr[0].detected) << 2)
			| (STATE_GET(dr[0].detected) << 1)
			| (STATE_GET(dr[0].detected) << 0);
		break;

	case 7:
		val = (STATE_GET(dr[3].len) << 30)
			| (STATE_GET(dr[3].type) << 28)
			| (STATE_GET(dr[2].len) << 26)
			| (STATE_GET(dr[2].type) << 24)
			| (STATE_GET(dr[1].len) << 22)
			| (STATE_GET(dr[1].type) << 20)
			| (STATE_GET(dr[0].len) << 18)
			| (STATE_GET(dr[0].type) << 16)
			/* More... - FIXME */
			| (STATE_GET(dr[3].global_enabled) << 7)
			| (STATE_GET(dr[3].local_enabled) << 6)
			| (STATE_GET(dr[2].global_enabled) << 5)
			| (STATE_GET(dr[2].local_enabled) << 4)
			| (STATE_GET(dr[1].global_enabled) << 3)
			| (STATE_GET(dr[1].local_enabled) << 2)
			| (STATE_GET(dr[0].global_enabled) << 1)
			| (STATE_GET(dr[0].local_enabled) << 0);
		break;

	default:
		assert(0); /* Mustn't happen. */
		break;
	}

	return val;
}

static uint8_t
NAME_(smx8)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint32_t addr,
	uint8_t seg,
	uint8_t usermode
)
{
	uint8_t val;
	uint16_t err;
	int ret;

	ret = DEFAULT_NAME_(seg_mx8)(cpssp, addr, seg, &val, usermode, &err);
	if (ret) {
		NAME_(mem_err)(cpssp, cpu, ret, err);
	}

	return val;
}

static uint8_t
NAME_(smr8)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint32_t addr,
	uint8_t seg,
	int w_test,
	uint8_t usermode
)
{
	uint8_t val;
	uint16_t err;
	int ret;

	ret = DEFAULT_NAME_(seg_mr8)(cpssp, addr, seg, w_test, &val, usermode, &err);
	if (ret) {
		NAME_(mem_err)(cpssp, cpu, ret, err);
	}

	return val;
}

static void
NAME_(smw8)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint32_t addr,
	uint8_t seg,
	uint8_t val,
	uint8_t usermode
)
{
	uint16_t err;
	int ret;

	ret = DEFAULT_NAME_(seg_mw8)(cpssp, addr, seg, val, usermode, &err);
	if (ret) {
		NAME_(mem_err)(cpssp, cpu, ret, err);
	}
}

static uint16_t
NAME_(smx16)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint32_t addr,
	uint8_t seg,
	uint8_t usermode
)
{
	uint16_t val;
	uint16_t err;
	int ret;

	ret = DEFAULT_NAME_(seg_mx16)(cpssp, addr, seg, &val, usermode, &err);
	if (ret) {
		NAME_(mem_err)(cpssp, cpu, ret, err);
	}

	return val;
}

static uint16_t
NAME_(smr16)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint32_t addr,
	uint8_t seg,
	int w_test,
	uint8_t usermode
)
{
	uint16_t val;
	uint16_t err;
	int ret;

	ret = DEFAULT_NAME_(seg_mr16)(cpssp, addr, seg, w_test, &val, usermode, &err);
	if (ret) {
		NAME_(mem_err)(cpssp, cpu, ret, err);
	}

	return val;
}

static void
NAME_(smw16)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint32_t addr,
	uint8_t seg,
	uint16_t val,
	uint8_t usermode
)
{
	uint16_t err;
	int ret;

	ret = DEFAULT_NAME_(seg_mw16)(cpssp, addr, seg, val, usermode, &err);
	if (ret) {
		NAME_(mem_err)(cpssp, cpu, ret, err);
	}
}

static uint32_t
NAME_(smx32)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint32_t addr,
	uint8_t seg,
	uint8_t usermode
)
{
	uint32_t val;
	uint16_t err;
	int ret;

	ret = DEFAULT_NAME_(seg_mx32)(cpssp, addr, seg, &val, usermode, &err);
	if (ret) {
		NAME_(mem_err)(cpssp, cpu, ret, err);
	}

	return val;
}

static uint32_t
NAME_(smr32)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint32_t addr,
	uint8_t seg,
	int w_test,
	uint8_t usermode
)
{
	uint32_t val;
	uint16_t err;
	int ret;

	ret = DEFAULT_NAME_(seg_mr32)(cpssp, addr, seg, w_test, &val, usermode, &err);
	if (ret) {
		NAME_(mem_err)(cpssp, cpu, ret, err);
	}

	return val;
}

static void
NAME_(smw32)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint32_t addr,
	uint8_t seg,
	uint32_t val,
	uint8_t usermode
)
{
	uint16_t err;
	int ret;

	ret = DEFAULT_NAME_(seg_mw32)(cpssp, addr, seg, val, usermode, &err);
	if (ret) {
		NAME_(mem_err)(cpssp, cpu, ret, err);
	}
}

#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
static uint64_t
NAME_(smx64)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint32_t addr,
	uint8_t seg,
	uint8_t usermode
)
{
	uint64_t val;
	uint16_t err;
	int ret;

	ret = DEFAULT_NAME_(seg_mx64)(cpssp, addr, seg, &val, usermode, &err);
	if (ret) {
		NAME_(mem_err)(cpssp, cpu, ret, err);
	}

	return val;
}

static uint64_t
NAME_(smr64)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint32_t addr,
	uint8_t seg,
	int w_test,
	uint8_t usermode
)
{
	uint64_t val;
	uint16_t err;
	int ret;

	ret = DEFAULT_NAME_(seg_mr64)(cpssp, addr, seg, w_test, &val, usermode, &err);
	if (ret) {
		NAME_(mem_err)(cpssp, cpu, ret, err);
	}

	return val;
}

static void
NAME_(smw64)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint32_t addr,
	uint8_t seg,
	uint64_t val,
	uint8_t usermode
)
{
	uint16_t err;
	int ret;

	ret = DEFAULT_NAME_(seg_mw64)(cpssp, addr, seg, val, usermode, &err);
	if (ret) {
		NAME_(mem_err)(cpssp, cpu, ret, err);
	}
}
#endif

static udata_t
NAME_(smr)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint32_t addr,
	uint8_t seg,
	int w_test,
	uint8_t usermode,
	int op_mode
)
{
	switch (op_mode) {
	case 0:
		return NAME_(smr8)(cpssp, cpu, addr, seg, w_test, usermode);
	case 1:
		return NAME_(smr16)(cpssp, cpu, addr, seg, w_test, usermode);
	case 2:
		return NAME_(smr32)(cpssp, cpu, addr, seg, w_test, usermode);
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	case 3:
		return NAME_(smr64)(cpssp, cpu, addr, seg, w_test, usermode);
#endif
	default:
		assert(0); /* Mustn't happen. */
	}
}

static void
NAME_(smw)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint32_t addr,
	uint8_t seg,
	udata_t val,
	uint8_t usermode,
	int op_mode
)
{
	switch (op_mode) {
	case 0:
		NAME_(smw8)(cpssp, cpu, addr, seg, val, usermode);
		break;
	case 1:
		NAME_(smw16)(cpssp, cpu, addr, seg, val, usermode);
		break;
	case 2:
		NAME_(smw32)(cpssp, cpu, addr, seg, val, usermode);
		break;
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	case 3:
		NAME_(smw64)(cpssp, cpu, addr, seg, val, usermode);
		break;
#endif
	default:
		assert(0); /* Mustn't happen. */
	}
}

static uint16_t
NAME_(mr16)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	udata_t addr,
	uint8_t seg,
	int w_test,
	uint8_t usermode,
	uint8_t exception,
	uint16_t error_code
)
{
	uint16_t val;
	uint16_t err;
	int ret;

	ret = DEFAULT_NAME_(seg_mr16)(cpssp, addr, seg, w_test, &val, usermode, &err);
	if (ret) {
		if (ret == NAME_(EXC_GP)) {
			ret = exception;
			err = error_code;
		}
		NAME_(mem_err)(cpssp, cpu, ret, err);
	}

	return val;
}

static uint32_t
NAME_(mr32)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	udata_t addr,
	uint8_t seg,
	int w_test,
	uint8_t usermode,
	uint8_t exception,
	uint16_t error_code
)
{
	uint32_t val;
	uint16_t err;
	int ret;

	ret = DEFAULT_NAME_(seg_mr32)(cpssp, addr, seg, w_test, &val, usermode, &err);
	if (ret) {
		if (ret == NAME_(EXC_GP)) {
			ret = exception;
			err = error_code;
		}
		NAME_(mem_err)(cpssp, cpu, ret, err);
	}

	return val;
}

#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
static uint64_t
NAME_(mr64)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	udata_t addr,
	uint8_t seg,
	int w_test,
	uint8_t usermode,
	uint8_t exception,
	uint16_t error_code
)
{
	uint64_t val;
	uint16_t err;
	int ret;

	ret = DEFAULT_NAME_(seg_mr64)(cpssp, addr, seg, w_test, &val, usermode, &err);
	if (ret) {
		if (ret == NAME_(EXC_GP)) {
			ret = exception;
			err = error_code;
		}
		NAME_(mem_err)(cpssp, cpu, ret, err);
	}

	return val;
}
#endif

/* ################ SEG FUNCS ############################################## */

static void
NAME_(fake_entry)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint8_t seg,
	uint8_t type,
	uint32_t base,
	uint32_t limit,
	uint8_t db,
	uint8_t dpl
)
{
	seg_descr_t descr;
	uint32_t entry[2];
	int granularity;

	if (0xfffff < limit) {
		assert((limit & 0xfff) == 0xfff);
		limit >>= 12;
		granularity = 1;
	} else {
		granularity = 0;
	}

	entry[0] = (((base >> 0) & 0xffff) << 16) /* Bit 15-0 Base */
		| (((limit >> 0) & 0xffff) << 0); /* Bit 15-0 Limit */
	entry[1] = (((base >> 24) & 0xff) << 24) /* Bit 31-24 Base */
		| (granularity << 23) /* Granularity */
		| (db << 22) /* D/B */
		| (0b0 << 21) /* Reserved */
		| (0b0 << 20) /* Available */
		| (((limit >> 16) & 0xf) << 16) /* Bit 19-16 Limit */
		| (0b1 << 15) /* Present */
		| (dpl << 13) /* DPL */
		| (type << 8) /* Type */
		| (((base >> 16) & 0xff) << 0); /* Bit 23-16 Base */

	descr.entry[0] = ((uint64_t) entry[1] << 32) | ((uint64_t) entry[0] << 0);
	DEFAULT_NAME_(seg_descr_set)(cpssp, seg, descr);
}

static int
NAME_(read_entry)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint8_t seg,
	uint16_t selector,
	int isint,
	int protected,
	int vm86,
	int code,
	uint16_t *errp
)
{
	uint32_t e1, e0;
	seg_descr_t e;
	int ret;

	if (! protected || (! isint && vm86)) {
		/*
		 * Real-Address Mode
		 */
		if (isint) {
			/*
			 * Fake interrupt descriptor entry.
			 */
			ret = DEFAULT_NAME_(seg_mr32)(cpssp, selector >> 1, NAME_(SEG_IDT), 0, &e0, 0, errp);

			e1 = (0x0000 << 16) /* High bits %eip */
				| (1 << 15) /* Present bit */
				| (0 << 13) /* DPL */
				| (0 << 12) /* System descriptor */
				| (0x6 << 8) /* 16-bit Interrupt Gate */
				| (0x00 << 0); /* Reserved */

		} else {
			/*
			 * Fake code/data descriptor entry.
			 */
			uint32_t base;
			uint32_t limit;
			int granularity;

			base = selector << 4;
			limit = 0xffff;
			granularity = 0;

			e0 = (((base >> 0) & 0xffff) << 16) /* Bit 15-0 Base */
				| (((limit >> 0) & 0xffff) << 0); /* Bit 15-0 Limit */
			e1 = (((base >> 24) & 0xff) << 24) /* Bit 31-24 Base */
				| (granularity << 23) /* Granularity */
				| (0b0 << 22) /* D/B */
				| (0b0 << 21) /* Reserved */
				| (0b0 << 20) /* Available */
				| (((limit >> 16) & 0xf) << 16) /* Bit 19-16 Limit */
				| (0b1 << 15) /* Present */
				| (vm86 << 14) | (vm86 << 13) /* DPL */
				| (0b1 << 12) /* Code/Data */
				| (((code << 3) | 0b0011) << 8) /* Type */
				| (((base >> 16) & 0xff) << 0); /* Bit 23-16 Base */

			ret = NAME_(EXC_NONE);
		}

		e.entry[0] = ((uint64_t) e1 << 32) | ((uint64_t) e0 << 0);

	} else {
		/*
		 * Protected Mode
		 */
		uint8_t table;
		uint16_t offset;

		if (! isint
		 && (selector & 0xfffc) == 0) {
			/* Null Selector */
			e.entry[0] = 0x0000000000000000ULL;
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
			if (STATE_GET(lm_active_msr)) {
				e.entry[1] = 0x0000000000000000ULL;
			}
#endif

			ret = NAME_(EXC_NONE);

		} else {
			/* Read entry. */
			if (isint) {
				table = NAME_(SEG_IDT);
			} else if (selector & 0x4) {
				table = NAME_(SEG_LDT);
			} else {
				table = NAME_(SEG_GDT);
			}

			offset = selector & ~0x7;

			e.entry[0] = 0; /* Make gcc happy. */
			ret = DEFAULT_NAME_(seg_mr64)(cpssp, offset, table, 0, &e.entry[0], 0, errp);
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
			/* XXX ugly, but we can only load entry[1] if it's a system segment
			 * or we get GPs because limit check fails
			 */
			if (STATE_GET(lm_active_msr) && !(e.entry[0] & (1ULL << (12 + 32)))) {
				e.entry[1] = 0; /* Make gcc happy. */
				ret = DEFAULT_NAME_(seg_mr64)(cpssp, offset + 8, table, 0, &e.entry[1], 0, errp);
			}
#endif
		}
	}

	DEFAULT_NAME_(seg_descr_set)(cpssp, seg, e);

	return ret;
}

static void
NAME_(mark_accessed)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint16_t selector,
	uint8_t seg
)
{
	seg_descr_t descr;
	uint8_t type;

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, seg);

	type = (descr.entry[0] >> 40) & 0x1f;
	if ((type & 0x11) == 0x10) {
		/* Real segment descriptor. */
		uint8_t table;

		/* Set accessed. */
		descr.entry[0] |= (uint64_t) 1 << 40;

		if (selector & 0x4) {
			table = NAME_(SEG_LDT);
		} else {
			table = NAME_(SEG_GDT);
		}
		NAME_(smw32)(cpssp, cpu, (selector & ~0x7) + 4, table, descr.entry[0] >> 32, 0);
	}

	DEFAULT_NAME_(seg_descr_set)(cpssp, seg, descr);
}

static void
NAME_(mark_busy)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint16_t selector,
	uint8_t seg
)
{
	seg_descr_t descr;
	uint32_t entry1;
	uint8_t table;

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, seg);

	entry1 = descr.entry[0] >> 32;

	/* Set busy. */
	entry1 |= 1 << 9;

	if (selector & 0x4) {
		table = NAME_(SEG_LDT);
	} else {
		table = NAME_(SEG_GDT);
	}
	NAME_(smw32)(cpssp, cpu, (selector & ~0x7) + 4, table, entry1, 0);

	DEFAULT_NAME_(seg_descr_set)(cpssp, seg, descr);
}

static uint8_t
NAME_(cpl)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	return NAME_(seg_dpl)(cpssp, NAME_(SEG_CS));
}

static int
NAME_(usermode)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	return NAME_(cpl)(cpssp, cpu) == 3;
}

static void
NAME_(check_seg_over)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	cpu->seg[0] = cpu->seg[cpu->seg_override];
}

static void
NAME_(io_check)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint32_t port, uint8_t bytes)
{
	uint16_t offset;
	uint16_t bits;

	if (NAME_(cpl)(cpssp, cpu) <= STATE_GET(e_iopl)) {
		return;
	}

	if ((NAME_(seg_type)(cpssp, NAME_(SEG_TR)) & 0x1d) != 0x09) {
		fprintf(stderr, "WARNING: io_check: no 32-bit TSS.\n");
		NAME_(GPe)(cpssp, cpu, 0);
	}

	offset = NAME_(smr16)(cpssp, cpu, 0x66, NAME_(SEG_TR), 0, 0);
	offset += port / 8;

	bits = NAME_(smr16)(cpssp, cpu, offset, NAME_(SEG_TR), 0, 0);

	bits &= (1 << bytes) - 1;
	if (bits) {
		fprintf(stderr, "WARNING: io_check: bits set.\n");
		NAME_(GPe)(cpssp, cpu, 0);
	}
}

#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
static int
NAME_(check_eip_canonical)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t eip)
{
	int ret;
	udata_t mask;

	mask = ~((1UL << CONFIG_CPU_VIRT_BITS) - 1);
	eip &= mask;
	ret = (eip == 0) | (eip == mask);

	return ret;
}
#endif

static void
NAME_(out)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint32_t port, uint32_t val, int op_mode)
{
	if (op_mode == 0 && (uint16_t) port == 0xffff) {
		fprintf(stderr, "%c", (char) val);
		return;
	}

	NAME_(io_check)(cpssp, cpu, port, 1 << op_mode);

	switch (op_mode) {
	case 0:
		DEFAULT_NAME_(uiowb)(cpssp, port, val);
		break;
	case 1:
		DEFAULT_NAME_(uioww)(cpssp, port, val);
		break;
	case 2:
		DEFAULT_NAME_(uiowd)(cpssp, port, val);
		break;
	default:
		assert(0); /* Mustn't happen. */
	}
}

static uint32_t
NAME_(in)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint32_t port, int op_mode)
{
	uint8_t tmp8;
	uint16_t tmp16;
	uint32_t tmp32;

	NAME_(io_check)(cpssp, cpu, port, 1 << op_mode);

	switch (op_mode) {
	case 0:
		DEFAULT_NAME_(uiorb)(cpssp, port, &tmp8);
		return (uint32_t) tmp8;
	case 1:
		DEFAULT_NAME_(uiorw)(cpssp, port, &tmp16);
		return (uint32_t) tmp16;
	case 2:
		DEFAULT_NAME_(uiord)(cpssp, port, &tmp32);
		return (uint32_t) tmp32;
	default:
		assert(0); /* Mustn't happen. */
	}
}

/* ################ OPERATION MODE AND CPU CONTROL ######################### */

static int
NAME_(get_lm_mode)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	return NAME_(seg_l)(cpssp, NAME_(SEG_CS)) & STATE_GET(lm_active_msr);
#endif
	return 0;
}

static int
NAME_(get_rex_w)(struct NAME_(cpu) *cpu)
{
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	return (cpu->pre_rex >> 3) & 1;
#endif
	return 0;
}

/* *-mode:
 * 0	 8bit
 * 1	16bit
 * 2	32bit
 * 3	64bit
 */

static int
NAME_(get_addr_mode)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	uint8_t lm;
	uint8_t cs;
	uint8_t pre;

	lm = NAME_(get_lm_mode)(cpssp, cpu);
	cs = NAME_(seg_db)(cpssp, NAME_(SEG_CS));
	pre = cpu->pre_addr;
	return 0
		| ((lm | (cs ^ pre)) << 1)
		| ((!(cs ^ pre)) << 0)
		;
}

static int
NAME_(get_stack_mode)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	uint8_t lm;
	uint8_t ss;

	lm = NAME_(get_lm_mode)(cpssp, cpu);
	ss = NAME_(seg_db)(cpssp, NAME_(SEG_SS));

	return 0
		| ((lm | ss) << 1)
		| ((lm | !ss) << 0)
		;
}

static uint8_t
NAME_(op_mode_map)[] =
{
	 0,  0,  0,  0,
	 0,  0,  0,  0,
	 0,  0,  0,  0,
	 0,  0,  0,  0,
	 1,  2,  2,  1,
	-1, -1, -1, -1,
	 2,  1, -1, -1,
	 3,  3, -1, -1
};

static int
NAME_(get_op_mode2)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t opcode)
{
	uint8_t lm;
	uint8_t cs;
	uint8_t pre;
	uint8_t rex_w;

	lm = NAME_(get_lm_mode)(cpssp, cpu);
	cs = NAME_(seg_db)(cpssp, NAME_(SEG_CS));
	pre = cpu->pre_op;
	rex_w = NAME_(get_rex_w)(cpu);

	return NAME_(op_mode_map)[0
		| ((opcode & 1) << 4)
		| (lm << 3)
		| (rex_w << 2)
		| (cs << 1)
		| (pre << 0)]
		;
}

static int
NAME_(get_op_mode1)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	return NAME_(get_op_mode2)(cpssp, cpu, 1);
}

static int
NAME_(get_op_mode_ip_sp)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	uint8_t lm;
	uint8_t cs;
	uint8_t pre;

	lm = NAME_(get_lm_mode)(cpssp, cpu);
	cs = NAME_(seg_db)(cpssp, NAME_(SEG_CS));
	pre = cpu->pre_op;
	return 0
		| ((lm ^ (cs ^ pre)) << 1)
		| ((lm | !(cs ^ pre)) << 0)
		;
}

static void
NAME_(change_df)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint32_t val)
{
	STATE_SET(e_df, val);
	cpssp->NAME.direction = 1 - (2 * val);
}

/*
 * EFLAGS
 * ||31      22|  21|   20|   19|  18|  17|  16||
 * ||[RESERVED]| ID | VIP | VIF | AC | VM | RF ||
 *
 * || 15|  14|13  12|  11|  10|   9|   8|   7|   6|  5|   4|  3|   2|  1|   0||
 * || 0 | NT | IOPL | OF | DF | IF | TF | SF | ZF | 0 | AF | 0 | PF | 1 | CF ||
 */
static uint32_t
NAME_(get_eflags)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	uint32_t eflags;

	eflags = 0;
	eflags |= STATE_GET(e_cf) << 0;
	eflags |= 1 << 1;
	eflags |= STATE_GET(e_pf) << 2;
	eflags |= 0 << 3;
	eflags |= STATE_GET(e_af) << 4;
	eflags |= STATE_GET(e_zf) << 6;
	eflags |= STATE_GET(e_sf) << 7;
	eflags |= STATE_GET(e_tf) << 8;
	eflags |= STATE_GET(e_if) << 9;
	eflags |= STATE_GET(e_df) << 10;
	eflags |= STATE_GET(e_of) << 11;
	eflags |= STATE_GET(e_iopl) << 12;
	eflags |= STATE_GET(e_nt) << 14;
	eflags |= STATE_GET(e_vm) << 17;
#if 80486 <= CONFIG_CPU
	eflags |= STATE_GET(e_ac) << 18;
#endif
#if 80586 <= CONFIG_CPU
	eflags |= STATE_GET(e_id) << 21;
#endif

	return eflags;
}

static void
NAME_(set_eflags8)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint32_t eflags)
{
	STATE_SET(e_cf, (eflags >> 0) & 0x1);
	/* Bit 1: Reserved */
	STATE_SET(e_pf, (eflags >> 2) & 0x1);
	/* Bit 3: Reserved */
	STATE_SET(e_af, (eflags >> 4) & 0x1);
	/* Bit 5: Reserved */
	STATE_SET(e_zf, (eflags >> 6) & 0x1);
	STATE_SET(e_sf, (eflags >> 7) & 0x1);
}

#if 80686 <= CONFIG_CPU
static void
NAME_(fpu_set_eflags)(struct cpssp *cpssp, uint8_t eflags)
{
	STATE_SET(e_cf, (eflags >> 0) & 0x1);
	/* Bit 1: Reserved */
	STATE_SET(e_pf, (eflags >> 2) & 0x1);
	/* Bit 3: Reserved */
	/* STATE_SET(e_af, (eflags >> 4) & 0x1); - not set. */
	/* Bit 5: Reserved */
	STATE_SET(e_zf, (eflags >> 6) & 0x1);
	/* STATE_SET(e_sf, (eflags >> 7) & 0x1); - not set. */
}
#endif /* 80686 <= CONFIG_CPU */

#if defined CONFIG_CPU_CMOV_SUPPORT && CONFIG_CPU_CMOV_SUPPORT
static uint8_t
NAME_(fpu_get_eflags)(struct cpssp *cpssp)
{
	uint8_t eflags;

	eflags = 0;
	eflags |= STATE_GET(e_cf) << 0;
	/* eflags |= 1 << 1; - not used. */
	eflags |= STATE_GET(e_pf) << 2;
	/* eflags |= 0 << 3; - not used. */
	/* eflags |= STATE_GET(e_af) << 4; - not used. */
	eflags |= STATE_GET(e_zf) << 6;
	/* eflags |= STATE_GET(e_sf) << 7; - not used. */

	return eflags;
}
#endif /* defined CONFIG_CPU_CMOV_SUPPORT && CONFIG_CPU_CMOV_SUPPORT */

static void
NAME_(set_eflags_common)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	udata_t eflags,
	int op_mode
)
{
	NAME_(set_eflags8)(cpssp, cpu, eflags);

	STATE_SET(e_tf, (eflags >> 8) & 0x1);
	/* Bit 9: IF not changed. */
	NAME_(change_df)(cpssp, cpu, (eflags >> 10) & 0x1);
	STATE_SET(e_of, (eflags >> 11) & 0x1);
	/* Bit 13-12: IOPL not changed */
	STATE_SET(e_nt, (eflags >> 14) & 0x1);
	/* Bit 15: Reserved */

	if (likely(op_mode > 1)) {
		/* STATE_SET(e_rf, (eflags >> 16) & 0x1); FIXME */
		/* Bit 17: VM not changed */
#if 80486 <= CONFIG_CPU
		STATE_SET(e_ac, (eflags >> 18) & 0x1);
#endif
		/* Bit 19: VIF not changed */
		/* Bit 20: VIP not changed */
#if 80586 <= CONFIG_CPU
		STATE_SET(e_id, (eflags >> 21) & 0x1);
#endif
		/* Bit 31-22: Reserved */
	}
}

static void
NAME_(set_eflags_vm)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint32_t eflags
)
{
	STATE_SET(e_vm, (eflags >> 17) & 0x1);
}

static void
NAME_(set_eflags_iopl)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint32_t eflags
)
{
	STATE_SET(e_iopl, (eflags >> 12) & 0x3);
}

static void
NAME_(set_eflags_if)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint32_t eflags
)
{
	STATE_SET(e_if, (eflags >> 9) & 0x1);
	cpssp->NAME.state_event = 1; /* Re-check interrupts. */
}

static void
NAME_(set_eflags_all)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	udata_t eflags,
	int op_mode
)
{
	NAME_(set_eflags_common)(cpssp, cpu, eflags, op_mode);
	NAME_(set_eflags_if)(cpssp, cpu, eflags);
	NAME_(set_eflags_iopl)(cpssp, cpu, eflags);
	NAME_(set_eflags_vm)(cpssp, cpu, eflags);
}

static udata_t
NAME_(get_eflags_pushf)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	udata_t eflags;

	eflags = NAME_(get_eflags)(cpssp, cpu);

	eflags &= ~(1 << 16); /* RF is not reported. */
	eflags &= ~(1 << 17); /* VM is not reported. */

	return eflags;
}

static void
NAME_(set_eflags_popf)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	udata_t eflags,
	int op_mode
)
{
	NAME_(set_eflags_common)(cpssp, cpu, eflags, op_mode);
	if (NAME_(cpl)(cpssp, cpu) == 0) {
		NAME_(set_eflags_iopl)(cpssp, cpu, eflags);
	}
	if (NAME_(cpl)(cpssp, cpu) <= STATE_GET(e_iopl)) {
		NAME_(set_eflags_if)(cpssp, cpu, eflags);
	}
}

#if 80486 <= CONFIG_CPU
static uint8_t
NAME_(align_ac_get)(struct cpssp *cpssp)
{
	return STATE_GET(e_ac) & STATE_GET(cr0_am);
}
#endif

/* ################ SIGN EXTENSION ######################################### */

/*
 * 0	1 byte
 * 1	2 byte
 * 2	4 byte
 * 3	8 byte
 */

static udata_t
NAME_(sign_ext)(udata_t val, uint8_t from, uint8_t to)
{
	udata_t uval;
	data_t sval;

	sval = val << ((8 * sizeof(udata_t)) - (8 << from));
	uval = sval >> ((8 << to) - (8 << from));
	uval = uval >> ((8 * sizeof(udata_t)) - (8 << to));

	return uval;
}

static udata_t
NAME_(fix_size)(udata_t val, uint8_t to)
{
	val <<= (8 * sizeof(udata_t)) - (8 << to);
	val >>= (8 * sizeof(udata_t)) - (8 << to);

	return val;
}

/* ################ LOAD AND STORE OPERANDS ################################ */

static uint32_t
NAME_(load_sreg)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t seg)
{
	return STATE_GET(selector[seg]);
}

static void
NAME_(store_sreg)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t seg, uint16_t selector)
{
	STATE_SET(selector[seg], selector);
}

static void
NAME_(store_segment)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t seg, uint16_t selector)
{
	int err;
	uint16_t errc;
	uint8_t type;

	err = NAME_(read_entry)(cpssp, cpu, NAME_(SEG_DATA), selector,
			0, STATE_GET(cr0_pe), STATE_GET(e_vm),
			seg == NAME_(SEG_CS), &errc);
	if (err != NAME_(EXC_NONE)) {
		NAME_(mem_err2)(cpssp, cpu, err, errc, NAME_(EXC_GP), selector & 0xfffc);
	}

	type = NAME_(seg_type)(cpssp, NAME_(SEG_DATA));
	switch (seg) {
	case NAME_(SEG_CS):
		if ((type & 0x18) != 0x18) {
			/* No code descriptor. */
			fprintf(stderr, "WARNING: No code segment.\n");
			NAME_(GPe)(cpssp, cpu, selector & 0xfffc);
		}
		break;
	case NAME_(SEG_DS):
	case NAME_(SEG_ES):
	case NAME_(SEG_FS):
	case NAME_(SEG_GS):
		if (type != 0x00 /* No null descriptor. */
		 && (type & 0x18) != 0x10 /* No data segment descriptor. */
		 && (type & 0x1a) != 0x1a /* No readable code segment descriptor. */) {
			/* Wrong data descriptor. */
			fprintf(stderr, "WARNING: Bad data segment.\n");
			NAME_(GPe)(cpssp, cpu, selector & 0xfffc);
		}
		break;
	case NAME_(SEG_SS):
		if ((type & 0x1a) != 0x12) {
			/* No writable data segment descriptor. */
			fprintf(stderr, "WARNING: Bad stack segment.\n");
			NAME_(SSe)(cpssp, cpu, selector & 0xfffc);
		}
		break;
	default:
		assert(0); /* Mustn't happen. */
	}

	if (type != 0x00
	 && ! NAME_(seg_present)(cpssp, NAME_(SEG_DATA))) {
		/* Not present. */
		if (seg == NAME_(SEG_SS)) {
			NAME_(SSe)(cpssp, cpu, selector & 0xfffc);
		} else {
			NAME_(NPe)(cpssp, cpu, selector & 0xfffc);
		}
	}

	if (! STATE_GET(cr0_pe)
	 || STATE_GET(e_vm)) {
		/* Keep old B/D and G flags. */
		seg_descr_t old_seg;
		seg_descr_t new_seg;

		old_seg = DEFAULT_NAME_(seg_descr_get)(cpssp, seg);
		new_seg = DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_DATA));
		new_seg.entry[0] &= ~((1ULL << 55) | (1ULL << 54));
		new_seg.entry[0] |= old_seg.entry[0] & ((1ULL << 55) | (1ULL << 54));
		DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_DATA), new_seg);

	}

	NAME_(store_sreg)(cpssp, cpu, seg, selector);
	NAME_(mark_accessed)(cpssp, cpu, selector, NAME_(SEG_DATA));
	DEFAULT_NAME_(seg_descr_set)(cpssp, seg,
			DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_DATA)));
}

#if defined CONFIG_CPU_CX8_SUPPORT && CONFIG_CPU_CX8_SUPPORT
static uint64_t
NAME_(load_reg_cx8)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg_num)
{
	uint32_t h, l;

	switch (reg_num) {
	case NAME_(E_AX):
		h = NAME_(load_reg32)(cpssp, cpu, NAME_(E_DX));
		l = NAME_(load_reg32)(cpssp, cpu, NAME_(E_AX));
		break;
	case NAME_(E_BX):
		h = NAME_(load_reg32)(cpssp, cpu, NAME_(E_CX));
		l = NAME_(load_reg32)(cpssp, cpu, NAME_(E_BX));
		break;
	default:
		assert(0); /* Mustn't happen. */
	}

	return ((uint64_t) h << 32) | ((uint64_t) l << 0);
}
#endif /* defined CONFIG_CPU_CX8_SUPPORT && CONFIG_CPU_CX8_SUPPORT */

static uint8_t
NAME_(load_mem8)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int w_test)
{
	return NAME_(smr8)(cpssp, cpu, cpu->addr, cpu->seg[0],
			w_test, NAME_(usermode)(cpssp, cpu));
}

static uint16_t
NAME_(load_mem16)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int w_test)
{
	return NAME_(smr16)(cpssp, cpu, cpu->addr, cpu->seg[0],
			w_test, NAME_(usermode)(cpssp, cpu));
}

static uint32_t
NAME_(load_mem32)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int w_test)
{
	return NAME_(smr32)(cpssp, cpu, cpu->addr, cpu->seg[0],
			w_test, NAME_(usermode)(cpssp, cpu));
}

static udata_t
NAME_(load_mem)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int op_mode, int w_test)
{
	return NAME_(smr)(cpssp, cpu, cpu->addr, cpu->seg[0],
			w_test, NAME_(usermode)(cpssp, cpu), op_mode);
}

#if defined CONFIG_CPU_CX8_SUPPORT && CONFIG_CPU_CX8_SUPPORT
static uint64_t
NAME_(load_mem_cx8)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int w_test)
{
	uint32_t h, l;

	l = NAME_(smr32)(cpssp, cpu, cpu->addr + 0, cpu->seg[0],
			w_test, NAME_(usermode)(cpssp, cpu));
	h = NAME_(smr32)(cpssp, cpu, cpu->addr + 4, cpu->seg[0],
			w_test, NAME_(usermode)(cpssp, cpu));

	return ((uint64_t) h << 32) | ((uint64_t) l << 0);
}
#endif /* defined CONFIG_CPU_CX8_SUPPORT && CONFIG_CPU_CX8_SUPPORT */

static udata_t
NAME_(load_mem_fpointer)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t *res2, int op_mode)
{
	udata_t res;

	res = NAME_(smr)(cpssp, cpu, cpu->addr + 0, cpu->seg[0],
			0, NAME_(usermode)(cpssp, cpu), op_mode);
	*res2 = NAME_(smr16)(cpssp, cpu, cpu->addr + (1 << op_mode), cpu->seg[0],
			0, NAME_(usermode)(cpssp, cpu));

	return res;
}

static void
NAME_(load_limit_base)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint64_t *descrp, int op_mode)
{
	uint32_t entry[2];
	uint32_t base;
	uint16_t limit;

	limit = NAME_(smr16)(cpssp, cpu, cpu->addr + 0, cpu->seg[0],
			0, NAME_(usermode)(cpssp, cpu));
	if (1 == op_mode) {
		base = NAME_(smr32)(cpssp, cpu, cpu->addr + 2, cpu->seg[0],
				0, NAME_(usermode)(cpssp, cpu));
		base &= 0x00ffffff;
	} else {
		base = NAME_(smr)(cpssp, cpu, cpu->addr + 2, cpu->seg[0],
				0, NAME_(usermode)(cpssp, cpu), op_mode);
	}

	/* Fake entry. */
	entry[0] = (((base >> 0) & 0xffff) << 16) /* Bit 15-0 Base */
		| (((limit >> 0) & 0xffff) << 0); /* Bit 15-0 Limit */
	entry[1] = (((base >> 24) & 0xff) << 24) /* Bit 32-24 Base */
		| (0b0 << 23) /* Granularity */
		| (0b0 << 22) /* D/B */
		| (0b0 << 21) /* Reserved */
		| (0b0 << 20) /* Available */
		| (0x0 << 16) /* Bit 19-16 Limit */
		| (0b1 << 15) /* Present */
		| (0x0 << 13) /* DPL */
		| (0b0 << 12) /* System */
		| (0x0 << 8) /* Type (must be 0!) */
		| (((base >> 16) & 0xff) << 0); /* Bit 23-16 Base */
	/* TODO 64 bit: load full base */

	*descrp = ((uint64_t) entry[1] << 32) | ((uint64_t) entry[0] << 0);
}

#if defined CONFIG_CPU_CX8_SUPPORT && CONFIG_CPU_CX8_SUPPORT
static void
NAME_(store_reg_cx8)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg_num, uint64_t val)
{
	switch (reg_num) {
	case NAME_(E_AX):
		NAME_(store_reg32)(cpssp, cpu, NAME_(E_DX), (uint32_t) (val >> 32));
		NAME_(store_reg32)(cpssp, cpu, NAME_(E_AX), (uint32_t) (val >>  0));
		break;
	case NAME_(E_BX):
		NAME_(store_reg32)(cpssp, cpu, NAME_(E_CX), (uint32_t) (val >> 32));
		NAME_(store_reg32)(cpssp, cpu, NAME_(E_BX), (uint32_t) (val >>  0));
		break;
	default:
		assert(0); /* Mustn't happen. */
	}
}
#endif /* defined CONFIG_CPU_CX8_SUPPORT && CONFIG_CPU_CX8_SUPPORT */

static void
NAME_(store_mem8)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t val)
{
	NAME_(smw8)(cpssp, cpu, cpu->addr, cpu->seg[0], val,
			NAME_(usermode)(cpssp, cpu));
}

static void
NAME_(store_mem16)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t val)
{
	NAME_(smw16)(cpssp, cpu, cpu->addr, cpu->seg[0], val,
			NAME_(usermode)(cpssp, cpu));
}

static void
NAME_(store_mem32)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint32_t val)
{
	NAME_(smw32)(cpssp, cpu, cpu->addr, cpu->seg[0], val,
			NAME_(usermode)(cpssp, cpu));
}

static void
NAME_(store_mem)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t val, int op_mode)
{
	NAME_(smw)(cpssp, cpu, cpu->addr, cpu->seg[0], val,
			NAME_(usermode)(cpssp, cpu), op_mode);
}

#if defined CONFIG_CPU_CX8_SUPPORT && CONFIG_CPU_CX8_SUPPORT
static void
NAME_(store_mem_cx8)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint64_t val)
{
	NAME_(smw32)(cpssp, cpu, cpu->addr + 0, cpu->seg[0],
			(uint32_t) (val >> 0), NAME_(usermode)(cpssp, cpu));
	NAME_(smw32)(cpssp, cpu, cpu->addr + 4, cpu->seg[0],
			(uint32_t) (val >> 32), NAME_(usermode)(cpssp, cpu));
}
#endif /* defined CONFIG_CPU_CX8_SUPPORT && CONFIG_CPU_CX8_SUPPORT */

static void
NAME_(store_limit_base)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint64_t descr, int op_mode)
{
	uint16_t limit;
	udata_t base;

	limit = descr & 0xffff;
	base = (((descr >> 56) & 0xff) << 24)
		| (((descr >> 32) & 0xff) << 16)
		| (((descr >> 16) & 0xffff) << 0);
	/* TODO 64 bit: store full base */

	if (1 == op_mode) {
#if CONFIG_CPU <= 80286
		base |= 0xff000000;
#else
		base &= 0x00ffffff;
#endif
		op_mode = 2;
	}

	NAME_(smw16)(cpssp, cpu, cpu->addr + 0, cpu->seg[0], limit,
			NAME_(usermode)(cpssp, cpu));
	NAME_(smw)(cpssp, cpu, cpu->addr + 2, cpu->seg[0], base,
			NAME_(usermode)(cpssp, cpu), op_mode);
}

static uint8_t
NAME_(load_rm8)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int wflag)
{
	uint32_t val;

	if (0x3 == cpu->mod) {
		val = NAME_(load_reg8)(cpssp, cpu, cpu->rm);
	} else {
		val = NAME_(load_mem8)(cpssp, cpu, wflag);
	}

	return val;
}

static uint16_t
NAME_(load_rm16)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int wflag)
{
	uint32_t val;

	if (0x3 == cpu->mod) {
		val = NAME_(load_reg16)(cpssp, cpu, cpu->rm);
	} else {
		val = NAME_(load_mem16)(cpssp, cpu, wflag);
	}

	return val;
}

static udata_t
NAME_(load_rm)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int op_mode, int wflag)
{
	uint32_t val;

	if (0x3 == cpu->mod) {
		val = NAME_(load_reg)(cpssp, cpu, cpu->rm, op_mode);
	} else {
		val = NAME_(load_mem)(cpssp, cpu, op_mode, wflag);
	}

	return val;
}

static void
NAME_(store_rm8)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t val)
{
	if (0x3 == cpu->mod) {
		NAME_(store_reg8)(cpssp, cpu, cpu->rm, val);
	} else {
		NAME_(store_mem8)(cpssp, cpu, val);
	}
}

static void
NAME_(store_rm16)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t val)
{
	if (0x3 == cpu->mod) {
		NAME_(store_reg16)(cpssp, cpu, cpu->rm, val);
	} else {
		NAME_(store_mem16)(cpssp, cpu, val);
	}
}

static void
NAME_(store_rm)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t val, int op_mode)
{
	if (0x3 == cpu->mod) {
		NAME_(store_reg)(cpssp, cpu, cpu->rm, val, op_mode);
	} else {
		NAME_(store_mem)(cpssp, cpu, val, op_mode);
	}
}

#if DEBUG_DISASSEMBLE
static uint32_t NAME_(last_eip) = 0;
#endif

static void
NAME_(fetch_start)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
#if DEBUG_DISASSEMBLE
	if (2 <= DEBUG_DISASSEMBLE + loglevel) {
		if (NAME_(last_eip) != cpu->eip) {
			fprintf(stderr, "0: Executing at %08lx (%08lx:%08lx)\n",
					NAME_(seg_base)(cpssp, NAME_(SEG_CS)) + STATE_GET(eip),
					NAME_(seg_base)(cpssp, NAME_(SEG_CS)),
					STATE_GET(eip));
		}
		fprintf(stderr, "DIS %08x",
				NAME_(seg_base)(cpssp, NAME_(SEG_CS)) + STATE_GET(eip));
	}
#endif /* DEBUG_DISASSEMBLE */
}

static void
NAME_(fetch_end)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
#if DEBUG_DISASSEMBLE
	if (2 <= DEBUG_DISASSEMBLE + loglevel) {
		fprintf(stderr, "\n");
	}
#endif
}

static uint8_t
NAME_(fetch8)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	uint32_t ret;

	ret = NAME_(smx8)(cpssp, cpu, cpu->eip, NAME_(SEG_CS), NAME_(usermode)(cpssp, cpu));
	cpu->eip++;

#if DEBUG_DISASSEMBLE
	if (2 <= DEBUG_DISASSEMBLE + loglevel) {
		fprintf(stderr, " %02x", ret);
		NAME_(last_eip) = cpu->eip;
	}
#endif

	return ret;
}

static uint16_t
NAME_(fetch16)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	uint32_t ret;

	ret = NAME_(smx16)(cpssp, cpu, cpu->eip, NAME_(SEG_CS), NAME_(usermode)(cpssp, cpu));
	cpu->eip += 2;

#if DEBUG_DISASSEMBLE
	if (2 <= DEBUG_DISASSEMBLE + loglevel) {
		fprintf(stderr, " %04x", ret);
		NAME_(last_eip) = cpu->eip;
	}
#endif

	return ret;
}

static uint32_t
NAME_(fetch32)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	uint32_t ret;

	ret = NAME_(smx32)(cpssp, cpu, cpu->eip, NAME_(SEG_CS), NAME_(usermode)(cpssp, cpu));
	cpu->eip += 4;

#if DEBUG_DISASSEMBLE
	if (2 <= DEBUG_DISASSEMBLE + loglevel) {
		fprintf(stderr, " %08x", ret);
		NAME_(last_eip) = cpu->eip;
	}
#endif

	return ret;
}

#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
static uint64_t
NAME_(fetch64)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	uint64_t ret;

	ret = NAME_(smx64)(cpssp, cpu, cpu->eip, NAME_(SEG_CS), NAME_(usermode)(cpssp, cpu));
	cpu->eip += 8;

#if DEBUG_DISASSEMBLE
	if (2 <= DEBUG_DISASSEMBLE + loglevel) {
		fprintf(stderr, " %08x", ret);
		NAME_(last_eip) = cpu->eip;
	}
#endif

	return ret;
}
#endif

static udata_t
NAME_(fetch)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int op_mode)
{
	switch (op_mode) {
	case 0:
		return NAME_(fetch8)(cpssp, cpu);
	case 1:
		return NAME_(fetch16)(cpssp, cpu);
	case 2:
		return NAME_(fetch32)(cpssp, cpu);
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	case 3:
		return NAME_(fetch64)(cpssp, cpu);
#endif
	default:
		assert(0); /* Mustn't happen. */
	}
}

/* ################ MSR ############################################## */

#if defined CONFIG_CPU_SEP_SUPPORT && CONFIG_CPU_SEP_SUPPORT

static int
NAME_(sep_wrmsr)(struct cpssp *cpssp, uint32_t ecx, uint64_t val)
{
	switch (ecx) {
	case 0x174: /* SYSENTER_CS_MSR */
		STATE_SET(sysenter_cs_msr, (uint32_t) val);
		return 1;
	case 0x175: /* SYSENTER_ESP_MSR */
		STATE_SET(sysenter_esp_msr, (uint32_t) val);
		return 1;
	case 0x176: /* SYSENTER_EIP_MSR */
		STATE_SET(sysenter_eip_msr, (uint32_t) val);
		return 1;
	default:
		return 0;
	}
}

static int
NAME_(sep_rdmsr)(struct cpssp *cpssp, uint32_t ecx, uint64_t *valp)
{
	switch (ecx) {
	case 0x174: /* SYSENTER_CS_MSR */
		*valp = (uint64_t) STATE_GET(sysenter_cs_msr);
		return 1;
	case 0x175: /* SYSENTER_ESP_MSR */
		*valp = (uint64_t) STATE_GET(sysenter_esp_msr);
		return 1;
	case 0x176: /* SYSENTER_EIP_MSR */
		*valp = (uint64_t) STATE_GET(sysenter_eip_msr);
		return 1;
	default:
		return 0;
	}
}

static void
NAME_(sysenter)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	STATE_SET(e_vm, 0);
	STATE_SET(e_if, 0);
	/* STATE_SET(e_rf, 0); - FIXME */

	/* Set %cs segment to fixed value. */
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_CS), STATE_GET(sysenter_cs_msr) + 0);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_CS), 0x1b, 0x00000000, 0xffffffff, 1, 0);

	/* Set %ss segment to fixed value. */
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_SS), STATE_GET(sysenter_cs_msr) + 8);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_SS), 0x13, 0x00000000, 0xffffffff, 1, 0);

	/* Set %esp. */
	NAME_(store_reg32)(cpssp, cpu, NAME_(E_SP), STATE_GET(sysenter_esp_msr));
	/* Set %eip. */
	cpu->eip = STATE_GET(sysenter_eip_msr);
}

static void
NAME_(sysexit)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	/* Set %cs segment to fixed value. */
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_CS), (STATE_GET(sysenter_cs_msr) + 16) | 0x3);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_CS), 0x1b, 0x00000000, 0xffffffff, 1, 3);

	/* Set %ss segment to fixed value. */
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_SS), (STATE_GET(sysenter_cs_msr) + 24) | 0x3);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_SS), 0x13, 0x00000000, 0xffffffff, 1, 3);

	/* Set %esp. */
	NAME_(store_reg32)(cpssp, cpu, NAME_(E_SP),
			NAME_(load_reg32)(cpssp, cpu, NAME_(E_CX)));
	/* Set %eip. */
	cpu->eip = NAME_(load_reg32)(cpssp, cpu, NAME_(E_DX));
}

#endif /* defined CONFIG_CPU_SEP_SUPPORT && CONFIG_CPU_SEP_SUPPORT */

#if defined CONFIG_CPU_TSC_SUPPORT && CONFIG_CPU_TSC_SUPPORT

static uint64_t
NAME_(_rdtsc)(struct cpssp *cpssp)
{
	uint64_t val;

#if 0
	val = cpssp->process.inst_cnt;
#else
	{
		uint64_t t;
		uint64_t x;
		uint64_t y;

		/* Beware of overflows! */
		/* Obey offset (see below)! FIXME */
		t = time_virt();
		x = t / TIME_HZ;
		y = t % TIME_HZ;
		val = (x * CONFIG_HZ) + (y * CONFIG_HZ) / TIME_HZ;
	}
#endif

	return val;
}

static int
NAME_(tsc_wrmsr)(struct cpssp *cpssp, uint32_t ecx, uint64_t val)
{
	switch (ecx) {
	case 0x10: /* TSC, IA32_TIME_STAMP_COUNTER */
		/* Set tsc. */
		/* FIXME */
		return 1;
	default:
		return 0;
	}
}

static int
NAME_(tsc_rdmsr)(struct cpssp *cpssp, uint32_t ecx, uint64_t *valp)
{
	switch (ecx) {
	case 0x10: /* TSC, IA32_TIME_STAMP_COUNTER */
		*valp = NAME_(_rdtsc)(cpssp);
		return 1;
	default:
		return 0;
	}
}

static void
NAME_(rdtsc)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint32_t *eaxp, uint32_t *edxp)
{
	uint64_t val;

	val = NAME_(_rdtsc)(cpssp);

	*edxp = (val >> 32) & 0xffffffff;
	*eaxp = (val >>  0) & 0xffffffff;
}

#endif /* defined CONFIG_CPU_TSC_SUPPORT && CONFIG_CPU_TSC_SUPPORT */

#if defined CONFIG_CPU_MCA_SUPPORT && CONFIG_CPU_MCA_SUPPORT

static int
NAME_(mca_wrmsr)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint32_t ecx, uint64_t val)
{
	switch (ecx) {
	case 0x179: /* MSR_IA32_MCG_CAP */
		/* Read-only. */
		assert(0); /* FIXME */
		return 1;

	case 0x17a: /* MSR_IA32_MCG_STATUS */
		/* Read-only. */
		assert(val == 0); /* FIXME */
		return 1;

	case 0x17b: /* MSR_IA32_MCG_CTL */
		/* Just remember setting. */
		cpssp->NAME.mca_mcg = val;
		return 1;

	case 0x400: /* MSR_IA32_MC0_CTL */
	case 0x404: /* MSR_IA32_MC1_CTL */
	case 0x408: /* MSR_IA32_MC2_CTL */
	case 0x40c: /* MSR_IA32_MC3_CTL */
	case 0x410: /* MSR_IA32_MC4_CTL */
		/* Just remember setting. */
		cpssp->NAME.mca_mcX_ctl[(ecx - 0x400) / 4] = val;
		return 1;

	case 0x401: /* MSR_IA32_MC0_STATUS */
	case 0x405: /* MSR_IA32_MC1_STATUS */
	case 0x409: /* MSR_IA32_MC2_STATUS */
	case 0x40d: /* MSR_IA32_MC3_STATUS */
	case 0x411: /* MSR_IA32_MC4_STATUS */
		/* Software might write '0' to register to clear. */
		/* Writing 1s will cause a general protection fault. */
		if (val != 0) {
			NAME_(GPe)(cpssp, cpu, 0);
		}
		/* FIXME */
		return 1;

	case 0x402: /* MSR_IA32_MC0_ADDR */
	case 0x406: /* MSR_IA32_MC1_ADDR */
	case 0x40a: /* MSR_IA32_MC2_ADDR */
	case 0x40e: /* MSR_IA32_MC3_ADDR */
	case 0x412: /* MSR_IA32_MC4_ADDR */
		/* Read-only. */
		assert(0); /* FIXME */
		return 1;

	case 0x403: /* MSR_IA32_MC0_MISC */
	case 0x407: /* MSR_IA32_MC1_MISC */
	case 0x40b: /* MSR_IA32_MC2_MISC */
	case 0x40f: /* MSR_IA32_MC3_MISC */
	case 0x413: /* MSR_IA32_MC4_MISC */
		/* Register not implented. */
		assert(0); /* FIXME */
		return 1;

	default:
		return 0;
	}
}

static int
NAME_(mca_rdmsr)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint32_t ecx, uint64_t *valp)
{
	switch (ecx) {
	case 0x179: /* MSR_IA32_MCG_CAP */
		*valp = (1 << 8) /* MCG_CTL available */
			| (5 << 0); /* 5 error-reporting banks */
		return 1;

	case 0x17a: /* MSR_IA32_MCG_STATUS */
		*valp = (0 << 2) /* No machine check in progress */
			| (0 << 1) /* No error IP valid */
			| (0 << 0); /* No restart IP valid */
		return 1;

	case 0x17b: /* MSR_IA32_MCG_CTL */
		/* Just report setting. */
		*valp = cpssp->NAME.mca_mcg;
		return 1;

	case 0x400: /* MSR_IA32_MC0_CTL */
	case 0x404: /* MSR_IA32_MC1_CTL */
	case 0x408: /* MSR_IA32_MC2_CTL */
	case 0x40c: /* MSR_IA32_MC3_CTL */
	case 0x410: /* MSR_IA32_MC4_CTL */
		/* Just report setting. */
		*valp = cpssp->NAME.mca_mcX_ctl[(ecx - 0x400) / 4];
		return 1;

	case 0x401: /* MSR_IA32_MC0_STATUS */
	case 0x405: /* MSR_IA32_MC1_STATUS */
	case 0x409: /* MSR_IA32_MC2_STATUS */
	case 0x40d: /* MSR_IA32_MC3_STATUS */
	case 0x411: /* MSR_IA32_MC4_STATUS */
		*valp = 0; /* No error pending. */
		return 1;

	case 0x402: /* MSR_IA32_MC0_ADDR */
	case 0x406: /* MSR_IA32_MC1_ADDR */
	case 0x40a: /* MSR_IA32_MC2_ADDR */
	case 0x40e: /* MSR_IA32_MC3_ADDR */
	case 0x412: /* MSR_IA32_MC4_ADDR */
		*valp = 0; /* No error pending. */
		return 1;

	case 0x403: /* MSR_IA32_MC0_MISC */
	case 0x407: /* MSR_IA32_MC1_MISC */
	case 0x40b: /* MSR_IA32_MC2_MISC */
	case 0x40f: /* MSR_IA32_MC3_MISC */
	case 0x413: /* MSR_IA32_MC4_MISC */
		*valp = 0; /* Register not implented. */
		return 1;

	default:
		return 0;
	}
}

#endif /* defined CONFIG_CPU_MCA_SUPPORT && CONFIG_CPU_MCA_SUPPORT */

#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT

static int
NAME_(lm_wrmsr)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint32_t ecx, uint64_t val)
{
	switch (ecx) {
	case 0xc0000080:
		if (STATE_GET(lm_active_msr) != ((val >> 10) & 0x1)) {
			NAME_(GPe)(cpssp, cpu, 0);
		}
		if (STATE_GET(cr0_pg) && (STATE_GET(lm_enabled_msr) != ((val >> 8) & 0x1))) {
			NAME_(GPe)(cpssp, cpu, 0);
		}
		STATE_SET(lm_enabled_msr,  (val >> 8) & 0x1);
#if defined CONFIG_CPU_NX_SUPPORT && CONFIG_CPU_NX_SUPPORT
		STATE_SET(efer_nxe, (val >> 11) & 0x1);
#endif
		return 1;
	default:
		return 0;
	}
}

static int
NAME_(lm_rdmsr)(struct cpssp *cpssp, uint32_t ecx, uint64_t *valp)
{
	switch (ecx) {
	case 0xc0000080:
		*valp = 0;
		*valp |= STATE_GET(lm_enabled_msr) << 8;
		*valp |= STATE_GET(lm_active_msr) << 10;
#if defined CONFIG_CPU_NX_SUPPORT && CONFIG_CPU_NX_SUPPORT
		*valp |= STATE_GET(efer_nxe) << 11;
#endif
		return 1;
	default:
		return 0;
	}
}

#endif /* defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT */

#if defined CONFIG_CPU_MSR_SUPPORT && CONFIG_CPU_MSR_SUPPORT

static void
NAME_(wrmsr)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint32_t ecx, uint32_t eax, uint32_t edx)
{
	uint64_t val;

	val = ((uint64_t) edx << 32) | ((uint64_t) eax << 0);

	switch (ecx) {
#if 80686 <= CONFIG_CPU
	case 0x79:
		/* BIOS update trigger. */
		/* We don't load any microcode updates... */
		break;

	case 0x8b:
		cpssp->NAME.update_signature = (uint32_t) (val >> 32);
		break;
#endif

	default:
		if (1
#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
		 && ! DEFAULT_NAME_(apic_wrmsr)(cpssp, ecx, val)
#endif
#if defined CONFIG_CPU_MTRR_SUPPORT && CONFIG_CPU_MTRR_SUPPORT
		 && ! DEFAULT_NAME_(mtrr_wrmsr)(cpssp, ecx, val)
#endif
#if defined CONFIG_CPU_SEP_SUPPORT && CONFIG_CPU_SEP_SUPPORT
		 && ! NAME_(sep_wrmsr)(cpssp, ecx, val)
#endif
#if defined CONFIG_CPU_TSC_SUPPORT && CONFIG_CPU_TSC_SUPPORT
		 && ! NAME_(tsc_wrmsr)(cpssp, ecx, val)
#endif
#if defined CONFIG_CPU_MCA_SUPPORT && CONFIG_CPU_MCA_SUPPORT
		 && ! NAME_(mca_wrmsr)(cpssp, cpu, ecx, val)
#endif
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
		 && ! NAME_(lm_wrmsr)(cpssp, cpu, ecx, val)
#endif
				) {
			fprintf(stderr, "%s: ecx=%08x\n", __FUNCTION__, ecx);
		}
		break;
	}
}

static void
NAME_(rdmsr)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint32_t ecx, uint32_t *eaxp, uint32_t *edxp)
{
	uint64_t val;

	switch (ecx) {
#if 80686 <= CONFIG_CPU
	case 0x79:
		/* BIOS update trigger. */
		/* We don't load any microcode updates... */
		break;

	case 0x8b:
		/* BIOS update signature */
		val = (uint64_t) cpssp->NAME.update_signature << 32;
		break;
#endif

	default:
		if (1
#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
		 && ! DEFAULT_NAME_(apic_rdmsr)(cpssp, ecx, &val)
#endif
#if defined CONFIG_CPU_MTRR_SUPPORT && CONFIG_CPU_MTRR_SUPPORT
		 && ! DEFAULT_NAME_(mtrr_rdmsr)(cpssp, ecx, &val)
#endif
#if defined CONFIG_CPU_SEP_SUPPORT && CONFIG_CPU_SEP_SUPPORT
		 && ! NAME_(sep_rdmsr)(cpssp, ecx, &val)
#endif
#if defined CONFIG_CPU_TSC_SUPPORT && CONFIG_CPU_TSC_SUPPORT
		 && ! NAME_(tsc_rdmsr)(cpssp, ecx, &val)
#endif
#if defined CONFIG_CPU_MCA_SUPPORT && CONFIG_CPU_MCA_SUPPORT
		 && ! NAME_(mca_rdmsr)(cpssp, cpu, ecx, &val)
#endif
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
		 && ! NAME_(lm_rdmsr)(cpssp, ecx, &val)
#endif
				) {
			fprintf(stderr, "%s: ecx=%08x\n", __FUNCTION__, ecx);
			val = 0;
		}
		break;
	}

	*edxp = (val >> 32) & 0xffffffff;
	*eaxp = (val >>  0) & 0xffffffff;
}

#endif /* defined CONFIG_CPU_MSR_SUPPORT && CONFIG_CPU_MSR_SUPPORT */

static void
NAME_(push2)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	int seg,
	udata_t *spp,
	udata_t val,
	int st_mode,
	int op_mode,
	int usermode
)
{
	udata_t addr;

	addr = *spp;
	addr -= 1 << op_mode;
	addr = NAME_(fix_size)(addr, st_mode);

	NAME_(smw)(cpssp, cpu, addr, seg, val, usermode, op_mode);

	*spp = addr;
}

static udata_t
NAME_(pop2)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	int seg,
	udata_t *spp,
	int st_mode,
	int op_mode,
	int usermode
)
{
	udata_t addr;
	udata_t ret;

	addr = *spp;

	ret = NAME_(smr)(cpssp, cpu, addr, seg, 0, usermode, op_mode);

	addr += 1 << op_mode;
	addr = NAME_(fix_size)(addr, st_mode);

	*spp = addr;

	return ret;
}

static void
NAME_(push)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t val, int op_mode, uint8_t usermode)
{
	int st_mode;
	udata_t sp;

	st_mode = NAME_(get_stack_mode)(cpssp, cpu);

	sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);

	NAME_(push2)(cpssp, cpu, NAME_(SEG_SS), &sp, val, st_mode, op_mode, usermode);

	NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);
}

static udata_t
NAME_(pop)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int op_mode, uint8_t usermode)
{
	int st_mode;
	udata_t sp;
	udata_t ret;

	st_mode = NAME_(get_stack_mode)(cpssp, cpu);

	sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);

	ret = NAME_(pop2)(cpssp, cpu, NAME_(SEG_SS), &sp, st_mode, op_mode, usermode);

	NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);

	return ret;
}

static void
NAME_(pusha)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int st_mode, int op_mode, uint8_t usermode)
{
	udata_t sp;
	udata_t eax, ecx, edx, ebx, esp, ebp, esi, edi;

	eax = NAME_(load_reg)(cpssp, cpu, NAME_(E_AX), op_mode);
	ecx = NAME_(load_reg)(cpssp, cpu, NAME_(E_CX), op_mode);
	edx = NAME_(load_reg)(cpssp, cpu, NAME_(E_DX), op_mode);
	ebx = NAME_(load_reg)(cpssp, cpu, NAME_(E_BX), op_mode);
	esp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), op_mode);
	ebp = NAME_(load_reg)(cpssp, cpu, NAME_(E_BP), op_mode);
	esi = NAME_(load_reg)(cpssp, cpu, NAME_(E_SI), op_mode);
	edi = NAME_(load_reg)(cpssp, cpu, NAME_(E_DI), op_mode);

	sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);

	NAME_(push2)(cpssp, cpu, NAME_(SEG_SS), &sp, eax, st_mode, op_mode, usermode);
	NAME_(push2)(cpssp, cpu, NAME_(SEG_SS), &sp, ecx, st_mode, op_mode, usermode);
	NAME_(push2)(cpssp, cpu, NAME_(SEG_SS), &sp, edx, st_mode, op_mode, usermode);
	NAME_(push2)(cpssp, cpu, NAME_(SEG_SS), &sp, ebx, st_mode, op_mode, usermode);
	NAME_(push2)(cpssp, cpu, NAME_(SEG_SS), &sp, esp, st_mode, op_mode, usermode);
	NAME_(push2)(cpssp, cpu, NAME_(SEG_SS), &sp, ebp, st_mode, op_mode, usermode);
	NAME_(push2)(cpssp, cpu, NAME_(SEG_SS), &sp, esi, st_mode, op_mode, usermode);
	NAME_(push2)(cpssp, cpu, NAME_(SEG_SS), &sp, edi, st_mode, op_mode, usermode);

	NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);
}

static void
NAME_(popa)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int st_mode, int op_mode, uint8_t usermode)
{
	udata_t sp;
	udata_t eax, ecx, edx, ebx, esp, ebp, esi, edi;

	sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);

	edi = NAME_(pop2)(cpssp, cpu, NAME_(SEG_SS), &sp, st_mode, op_mode, usermode);
	esi = NAME_(pop2)(cpssp, cpu, NAME_(SEG_SS), &sp, st_mode, op_mode, usermode);
	ebp = NAME_(pop2)(cpssp, cpu, NAME_(SEG_SS), &sp, st_mode, op_mode, usermode);
	esp = NAME_(pop2)(cpssp, cpu, NAME_(SEG_SS), &sp, st_mode, op_mode, usermode);
	ebx = NAME_(pop2)(cpssp, cpu, NAME_(SEG_SS), &sp, st_mode, op_mode, usermode);
	edx = NAME_(pop2)(cpssp, cpu, NAME_(SEG_SS), &sp, st_mode, op_mode, usermode);
	ecx = NAME_(pop2)(cpssp, cpu, NAME_(SEG_SS), &sp, st_mode, op_mode, usermode);
	eax = NAME_(pop2)(cpssp, cpu, NAME_(SEG_SS), &sp, st_mode, op_mode, usermode);

	NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);

	NAME_(store_reg)(cpssp, cpu, NAME_(E_DI), edi, op_mode);
	NAME_(store_reg)(cpssp, cpu, NAME_(E_SI), esi, op_mode);
	NAME_(store_reg)(cpssp, cpu, NAME_(E_BP), ebp, op_mode);
	NAME_(store_reg)(cpssp, cpu, NAME_(E_BX), ebx, op_mode);
	NAME_(store_reg)(cpssp, cpu, NAME_(E_DX), edx, op_mode);
	NAME_(store_reg)(cpssp, cpu, NAME_(E_CX), ecx, op_mode);
	NAME_(store_reg)(cpssp, cpu, NAME_(E_AX), eax, op_mode);
}

static void
NAME_(enter)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t size, uint8_t level)
{
	int st_mode;
	int op_mode;
	udata_t sp;
	udata_t bp;

	level &= 0x1f;

	st_mode = NAME_(get_stack_mode)(cpssp, cpu);
	op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);

	sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);

	/* Push %ebp */
	bp = NAME_(load_reg)(cpssp, cpu, NAME_(E_BP), st_mode);
	NAME_(push2)(cpssp, cpu, NAME_(SEG_SS), &sp, bp, st_mode, op_mode, NAME_(usermode)(cpssp, cpu));

	if (level) {
		assert(0); /* FIXME */
	}

	/* %ebp = %esp */
	NAME_(store_reg)(cpssp, cpu, NAME_(E_BP), sp, st_mode);

	/* %esp -= size */
	sp -= size;
	if (! (st_mode - 1)) { // FIXME st_mode
		sp &= 0x0000ffff;
	}

	NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);
}

static void
NAME_(leave)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	int st_mode;
	int op_mode;
	udata_t sp;
	udata_t bp;

	st_mode = NAME_(get_stack_mode)(cpssp, cpu);
	op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);

	/* %esp = %ebp */
	sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_BP), st_mode);

	/* Pop %ebp */
	bp = NAME_(pop2)(cpssp, cpu, NAME_(SEG_SS), &sp, st_mode, op_mode, NAME_(usermode)(cpssp, cpu));
	NAME_(store_reg)(cpssp, cpu, NAME_(E_BP), bp, st_mode);

	NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);
}

/* ################ FLAGS ################################################## */

/*
 * Set if the least-significant byte of the result
 * contains an even number of 1 bits; cleared otherwise.
 */
static void
NAME_(det_pf)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint32_t res)
{
	uint8_t lsb = res;
	lsb ^= (lsb >> 4);
	lsb ^= (lsb >> 2);
	lsb ^= (lsb >> 1);
	lsb = (~lsb & 0x01);
	STATE_SET(e_pf, lsb);
}

/*
 * Set equal to the most-significant bit of the result, which is the sign bit of a
 * signed integer. (0 indicates a positive value and 1 indicates a negative value.)
 */
static void
NAME_(det_sf_s)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t op_mode, udata_t res)
{
	STATE_SET(e_sf, (res >> ((8 << op_mode) - 1)) & 0x1);
}

/*
 * Set if the result is zero; cleared otherwise.
 */
static void
NAME_(det_zf_s)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t op_mode, udata_t res)
{
	res <<= (8 * sizeof(udata_t)) - (8 << op_mode);

	STATE_SET(e_zf, res == 0);
}

static void
NAME_(det_zf)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t res)
{
	STATE_SET(e_zf, res == 0);
}

/* ################ INSTRUCTIONS ########################################### */

static udata_t
NAME_(and)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t op_mode)
{
	udata_t res;

	res = op1 & op2;

	STATE_SET(e_af, 0); /* Undefined */
	STATE_SET(e_cf, 0);
	STATE_SET(e_of, 0);
	NAME_(det_pf)(cpssp, cpu, res);
	NAME_(det_zf)(cpssp, cpu, res);
	NAME_(det_sf_s)(cpssp, cpu, op_mode, res);

	return res;
}

static udata_t
NAME_(or)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t op_mode)
{
	udata_t res;

	res = op1 | op2;

	STATE_SET(e_af, 0); /* Undefined */
	STATE_SET(e_cf, 0);
	STATE_SET(e_of, 0);
	NAME_(det_pf)(cpssp, cpu, res);
	NAME_(det_zf)(cpssp, cpu, res);
	NAME_(det_sf_s)(cpssp, cpu, op_mode, res);

	return res;
}

static udata_t
NAME_(xor)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t op_mode)
{
	udata_t res;

	res = op1 ^ op2;

	STATE_SET(e_af, 0); /* Undefined */
	STATE_SET(e_cf, 0);
	STATE_SET(e_of, 0);
	NAME_(det_pf)(cpssp, cpu, res);
	NAME_(det_zf)(cpssp, cpu, res);
	NAME_(det_sf_s)(cpssp, cpu, op_mode, res);

	return res;
}

static udata_t
NAME_(add)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t op_mode)
{
	udata_t res;
	udata_t ci;

	res = op1 + op2;
	res = NAME_(fix_size)(res, op_mode);

	ci = res ^ op1 ^ op2;
	STATE_SET(e_cf, res < op1);
	STATE_SET(e_af, (ci >> 4) & 0x1);
	STATE_SET(e_of, (STATE_GET(e_cf) ^ (ci >> ((8 << op_mode) - 1))) & 0x1);
	NAME_(det_pf)(cpssp, cpu, res);
	NAME_(det_zf)(cpssp, cpu, res);
	NAME_(det_sf_s)(cpssp, cpu, op_mode, res);

	return res;
}

static udata_t
NAME_(adc)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t op_mode)
{
	udata_t res;
	udata_t ci;

	res = op1 + op2 + STATE_GET(e_cf);
	res = NAME_(fix_size)(res, op_mode);

	ci = res ^ op1 ^ op2;
	// TODO best way to do?
	if (STATE_GET(e_cf)) {
		STATE_SET(e_cf, (res <= op1) | (res <= op2));
	} else {
		STATE_SET(e_cf, (res < op1) | (res < op2));
	}
	STATE_SET(e_af, (ci >> 4) & 0x1);
	STATE_SET(e_of, (STATE_GET(e_cf) ^ (ci >> ((8 << op_mode) - 1))) & 0x1);
	NAME_(det_pf)(cpssp, cpu, res);
	NAME_(det_zf)(cpssp, cpu, res);
	NAME_(det_sf_s)(cpssp, cpu, op_mode, res);

	return res;
}


static udata_t
NAME_(sub)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t op_mode)
{
	udata_t res;
	udata_t ci;

	res = op1 - op2;
	res = NAME_(fix_size)(res, op_mode);

	ci = res ^ op1 ^ op2;
	STATE_SET(e_af, (ci >> 4) & 0x1);
	STATE_SET(e_cf, op1 < op2);
	STATE_SET(e_of, (STATE_GET(e_cf) ^ (ci >> ((8 << op_mode) - 1))) & 0x1);
	NAME_(det_pf)(cpssp, cpu, res);
	NAME_(det_zf)(cpssp, cpu, res);
	NAME_(det_sf_s)(cpssp, cpu, op_mode, res);

	return res;
}

static udata_t
NAME_(sbb)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t op_mode)
{
	udata_t res;
	udata_t ci;

	res = op1 - op2 - STATE_GET(e_cf);
	res = NAME_(fix_size)(res, op_mode);

	ci = res ^ op1 ^ op2;
	// TODO best way to do?
	STATE_SET(e_af, (ci >> 4) & 0x1);
	STATE_SET(e_cf, (op1 < op2) | ((op1 == op2) & STATE_GET(e_cf)));
	STATE_SET(e_of, (STATE_GET(e_cf) ^ (ci >> ((8 << op_mode) - 1))) & 0x1);
	NAME_(det_pf)(cpssp, cpu, res);
	NAME_(det_zf)(cpssp, cpu, res);
	NAME_(det_sf_s)(cpssp, cpu, op_mode, res);

	return res;
}

static void
NAME_(cmp)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t op_mode)
{
	(void) NAME_(sub)(cpssp, cpu, op1, op2, op_mode);
}

static void
NAME_(test)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t op_mode)
{
	(void) NAME_(and)(cpssp, cpu, op1, op2, op_mode);
}

static udata_t
NAME_(neg)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, uint8_t op_mode)
{
	udata_t res;
	udata_t ci;

	res = -op1;

	// TODO correct if sub works
	// TODO is that correct?
	ci = (res ^ op1);
	STATE_SET(e_af, (ci >> 4) & 0x1);
	STATE_SET(e_cf, (op1 != 0));
	STATE_SET(e_of, (STATE_GET(e_cf) ^ (ci >> ((8 << op_mode) - 1))) & 0x1);
	NAME_(det_pf)(cpssp, cpu, res);
	NAME_(det_zf)(cpssp, cpu, res);
	NAME_(det_sf_s)(cpssp, cpu, op_mode, res);

	return res;
}

static udata_t
NAME_(not)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1)
{
	udata_t res;

	res = ~op1;

	/* AF: not affected. */
	/* CF: not affected. */
	/* OF: not affected. */
	/* PF: not affected. */
	/* SF: not affected. */
	/* ZF: not affected. */

	return res;
}

static udata_t
NAME_(inc_dec)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, uint8_t op_mode, uint8_t op)
{
	udata_t res;
	udata_t ci;

	if (op) {
		res = op1 - 1;

		ci = res ^ op1;
		STATE_SET(e_of, ((op1 & ci) >> ((8 << op_mode) - 1)) & 0x1);
	} else {
		res = op1 + 1;

		ci = res ^ op1;
		STATE_SET(e_of, ((~op1 & ci) >> ((8 << op_mode) - 1)) & 0x1);
	}

	STATE_SET(e_af, (ci >> 4) & 0x1);
	/* CF: not affected. */
	NAME_(det_pf)(cpssp, cpu, res);
	NAME_(det_sf_s)(cpssp, cpu, op_mode, res);
	NAME_(det_zf_s)(cpssp, cpu, op_mode, res);

	return res;
}

#if 80486 <= CONFIG_CPU
static uint32_t
NAME_(bswap)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1)
{
	uint32_t res;

	res = (((op1 >> 24) & 0xff) << 0)
		| (((op1 >> 16) & 0xff) << 8)
		| (((op1 >> 8) & 0xff) << 16)
		| (((op1 >> 0) & 0xff) << 24);

	/* AF: not affected. */
	/* CF: not affected. */
	/* OF: not affected. */
	/* PF: not affected. */
	/* SF: not affected. */
	/* ZF: not affected. */

	return res;
}
#endif /* 80486 <= CONFIG_CPU */

// TODO simplify
static uint32_t
NAME_(sal_shl)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t op_mode)
{
	uint32_t res;
	uint8_t op_size;

	op_size = 8 << op_mode;

	op2 &= 0x1f;
	res = op1 << op2;

	if (! op2) {
		return res;
	}

	STATE_SET(e_af, 0); /* Undefined */
	STATE_SET(e_cf, (op1 >> (op_size - op2)) & 0x1);
	STATE_SET(e_of, ((op1 >> (op_size - op2)) ^ (res >> (op_size - 1))) & 0x1);
	NAME_(det_pf)(cpssp, cpu, res);
	NAME_(det_sf_s)(cpssp, cpu, op_mode, res);
	NAME_(det_zf_s)(cpssp, cpu, op_mode, res);

	return res;
}

static uint32_t
NAME_(sar)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t op_mode)
{
	uint32_t res;
	uint8_t op_size;

	op_size = 8 << op_mode;


	op1 <<= 32 - op_size;
	op1 = ((int32_t) op1) >> (32 - op_size);
	op2 &= 0x1f;
	res = ((int32_t) op1) >> op2;

	if (! op2) {
		return res;
	}

	STATE_SET(e_af, 0); /* Undefined */
	STATE_SET(e_cf, ((int32_t) op1 >> (op2 - 1)) & 1);
	STATE_SET(e_of, 0);
	NAME_(det_pf)(cpssp, cpu, res);
	NAME_(det_sf_s)(cpssp, cpu, op_mode, res);
	NAME_(det_zf_s)(cpssp, cpu, op_mode, res);

	return res;
}

static uint32_t
NAME_(shr)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t op_mode)
{
	uint32_t res;
	uint8_t op_size;

	op_size = 8 << op_mode;


	op2 &= 0x1f;
	res = op1 >> op2;

	if (! op2) {
		return res;
	}

	STATE_SET(e_af, 0); /* Undefined */
	STATE_SET(e_cf, (op1 >> (op2 - 1)) & 0x1);
	STATE_SET(e_of, (((op1 >> (op2 - 1)) ^ res) >> (op_size - 1)) & 0x1);
	NAME_(det_pf)(cpssp, cpu, res);
	NAME_(det_sf_s)(cpssp, cpu, op_mode, res);
	NAME_(det_zf_s)(cpssp, cpu, op_mode, res);

	return res;
}

static udata_t
NAME_(shld)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, udata_t op3, uint8_t op_mode)
{
	udata_t res;
	uint8_t op_size;

	op_size = 8 << op_mode;


	op3 &= (0x3f >> (op_size != 64));

	if (op3 != 0) {
		if (op3 <= op_size) {
			res = op1 << op3;
			res |= (op2 >> (op_size - op3));
			STATE_SET(e_cf, (op1 >> (op_size - op3)) & 0x1);
		} else {
			res = op2 << (op3 - 16); /* FIXME is this still correct in lm */
			res |= op1 >> (32 - op3);
			STATE_SET(e_cf, (op2 >> (op_size + 16 - op3)) & 0x1);
		}

		STATE_SET(e_af, 0); /* Undefined */
		STATE_SET(e_of, ((res >> (op_size - 1)) ^ (op1 >> (op_size - op3))) & 0x1); /* Undefined if op3 != 1 */
		NAME_(det_pf)(cpssp, cpu, res);
		NAME_(det_sf_s)(cpssp, cpu, op_mode, res);
		NAME_(det_zf_s)(cpssp, cpu, op_mode, res);

	} else {
		res = op1;

		/* AF: not affected. */
		/* CF: not affected. */
		/* OF: not affected. */
		/* PF: not affected. */
		/* SF: not affected. */
		/* ZF: not affected. */
	}

	return res;
}

static udata_t
NAME_(shrd)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, udata_t op3, uint8_t op_mode)
{
	udata_t res;
	uint8_t op_size;

	op_size = 8 << op_mode;


	op3 &= (0x3f >> (op_size != 64));

	if (op3 != 0) {
		if (op3 <= op_size) {
			res = op1 >> op3;
			res |= op2 << (op_size - op3);
			STATE_SET(e_cf, (op1 >> (op3 - 1)) & 0x1);
		} else {
			res = op2 >> (op3 - 16); /* FIXME is this still correct in lm */
			res |= op1 << (32 - op3);
			STATE_SET(e_cf, (op2 >> (op3 - 16 - 1)) & 0x1);
		}

		STATE_SET(e_af, 0); /* Undefined */
		STATE_SET(e_of, (((op1 >> (op3 - 1)) ^ res) >> (op_size - 1)) & 0x1); /* Undefined if op3 != 1 */
		NAME_(det_pf)(cpssp, cpu, res);
		NAME_(det_sf_s)(cpssp, cpu, op_mode, res);
		NAME_(det_zf_s)(cpssp, cpu, op_mode, res);

	} else {
		res = op1;

		/* AF: not affected. */
		/* CF: not affected. */
		/* OF: not affected. */
		/* PF: not affected. */
		/* SF: not affected. */
		/* ZF: not affected. */
	}

	return res;
}

static uint32_t
NAME_(rol)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t op_mode)
{
	uint32_t count;
	uint32_t res;
	uint8_t flags[2];
	uint8_t op_size;

	op_size = 8 << op_mode;


	op2 &= 0x1f;
	count = op2 & (op_size - 1);

	res = op1 << count;
	res |= op1 >> (op_size - count);

	/* AF: not affected. */
	flags[0] = STATE_GET(e_cf);
	flags[1] = res & 0x1;
	STATE_SET(e_cf, flags[op2 != 0]);
	flags[0] = STATE_GET(e_of);
	flags[1] = ((res ^ op1) >> (op_size - 1)) & 0x1;
	STATE_SET(e_of, flags[op2 != 0]);
	/* PF: not affected. */
	/* SF: not affected. */
	/* ZF: not affected. */

	return res;
}

static uint32_t
NAME_(ror)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t op_mode)
{
	uint8_t count;
	uint32_t res;
	uint8_t flags[2];
	uint8_t op_size;

	op_size = 8 << op_mode;


	op2 &= 0x1f;
	count = op2 & (op_size - 1);

	res = op1 >> count;
	res |= op1 << (op_size - count);

	/* AF: not affected. */
	flags[0] = STATE_GET(e_cf);
	flags[1] = (res >> (op_size - 1)) & 0x1;
	STATE_SET(e_cf, flags[op2 != 0]);
	flags[0] = STATE_GET(e_of);
	flags[1] = ((res ^ op1) >> (op_size - 1)) & 0x1;
	STATE_SET(e_of, flags[op2 != 0]);
	/* PF: not affected. */
	/* SF: not affected. */
	/* ZF: not affected. */

	return res;
}

static uint32_t
NAME_(rcl)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t op_mode)
{
	uint32_t res;
	uint8_t op_size;

	op_size = 8 << op_mode;


	op2 &= 0x1f;
	op2 %= (op_size + 1);

	if (op2 != 0) {
		res = op1 << op2;
		res |= STATE_GET(e_cf) << (op2 - 1);
		if (1 < op2) {
			res |= op1 >> (op_size - op2 + 1);
		}

		/* AF: not affected. */
		STATE_SET(e_cf, (op1 >> (op_size - op2)) & 0x1);
		STATE_SET(e_of, ((res >> (op_size - 1)) & 0x1) ^ STATE_GET(e_cf));
		/* PF: not affected. */
		/* SF: not affected. */
		/* ZF: not affected. */

	} else {
		res = op1;
		/* AF: not affected. */
		/* CF: not affected. */
		/* OF: not affected. */
		/* PF: not affected. */
		/* SF: not affected. */
		/* ZF: not affected. */
	}

	return res;
}

static uint32_t
NAME_(rcr)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t op_mode)
{
	uint32_t res;
	uint8_t op_size;

	op_size = 8 << op_mode;


	op2 &= 0x1f;
	op2 %= (op_size + 1);

	if (op2 != 0) {
		res = op1 >> op2;
		res |= STATE_GET(e_cf) << (op_size - op2);
		if (1 < op2) {
			res |= op1 << (op_size - op2 + 1);
		}

		/* AF: not affected. */
		STATE_SET(e_cf, (op1 >> (op2 - 1)) & 0x1);
		STATE_SET(e_of, ((op1 >> (op_size - 1)) & 1) ^ ((res >> (op_size - op2)) & 0x1));
		/* PF: not affected. */
		/* SF: not affected. */
		/* ZF: not affected. */

	} else {
		res = op1;

		/* AF: not affected. */
		/* CF: not affected. */
		/* OF: not affected. */
		/* PF: not affected. */
		/* SF: not affected. */
		/* ZF: not affected. */
	}

	return res;
}

static udata_t
NAME_(cwd)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, int op_mode)
{
	data_t tmp;

	tmp = op1;
	tmp <<= ((8 * sizeof(udata_t)) - (8 << op_mode));
	tmp >>= ((8 * sizeof(udata_t)) - 1);
	return tmp;
}

static udata_t
NAME_(mul)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, udata_t *res2, int op_mode)
{
	uddata_t res;
	int flag;

	res = (uddata_t) op1 * (uddata_t) op2;

	flag = res != NAME_(fix_size)(res, op_mode);
	STATE_SET(e_af, 0); /* Undefined */
	STATE_SET(e_cf, flag);
	STATE_SET(e_of, flag);
	NAME_(det_pf)(cpssp, cpu, res); /* Undefined */
	NAME_(det_sf_s)(cpssp, cpu, op_mode, res); /* Undefined */
	NAME_(det_zf_s)(cpssp, cpu, op_mode, res); /* Undefined */

	*res2 = res >> (8 << op_mode);
	return res;
}

static udata_t
NAME_(imul)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, udata_t *res2, int op_mode)
{
	ddata_t res;
	ddata_t res3;
	int flag;

	res = ((ddata_t) ((data_t) op1)) * ((ddata_t) ((data_t) op2));

	res3 = res;
	res3 <<= ((8 * sizeof(ddata_t)) - (8 << op_mode));
	res3 >>= ((8 * sizeof(ddata_t)) - (8 << op_mode));
	flag = res != res3;
	STATE_SET(e_af, 0); /* Undefined */
	STATE_SET(e_cf, flag);
	STATE_SET(e_of, flag);
	NAME_(det_pf)(cpssp, cpu, res); /* Undefined */
	NAME_(det_sf_s)(cpssp, cpu, op_mode, res); /* Undefined */
	NAME_(det_zf_s)(cpssp, cpu, op_mode, res); /* Undefined */

	*res2 = res >> (8 << op_mode);
	return res;
}

static udata_t
NAME_(div)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, udata_t op3, udata_t *res2, int op_mode)
{
	uddata_t res;
	uddata_t tmp;

	if (! op3) {
		NAME_(DE)(cpssp, cpu);
	}

	tmp = op1;
	tmp <<= (8 << op_mode);
	tmp |= op2;
	res = tmp / op3;
	*res2 = tmp % op3;

	tmp = 0xffffffffffffffffL;
	tmp >>= (64 - (8 << op_mode));
	if (tmp < res) {
		NAME_(DE)(cpssp, cpu);
	}

	STATE_SET(e_af, 0); /* Undefined */
	STATE_SET(e_cf, 0); /* Undefined */
	STATE_SET(e_of, 0); /* Undefined */
	STATE_SET(e_pf, 0); /* Undefined */
	STATE_SET(e_sf, 0); /* Undefined */
	STATE_SET(e_zf, 0); /* Undefined */

	return res;
}

static udata_t
NAME_(idiv)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, udata_t op3, udata_t *res2, int op_mode)
{
	ddata_t resd;
	ddata_t resm;
	ddata_t tmp;
	int err;

	if (op3 == 0) {
		NAME_(DE)(cpssp, cpu);
	}

	tmp = (ddata_t) (data_t) op1;
	tmp <<= (8 << op_mode);
	tmp |= op2 & (0xffffffffffffffffL >> (64 - (8 << op_mode)));
	resd = tmp / (data_t) op3;
	resm = tmp % (data_t) op3;
	tmp = resd;
	tmp <<= (8 * sizeof(ddata_t)) - (8 << op_mode);
	tmp >>= (8 * sizeof(ddata_t)) - (8 << op_mode);
	err = resd != tmp;

	if (err) {
		NAME_(DE)(cpssp, cpu);
	}

#if 0 /* Should set flags to 0. qemu forgets to do it... */
	STATE_SET(e_af, 0); /* Undefined */
	STATE_SET(e_cf, 0); /* Undefined */
	STATE_SET(e_of, 0); /* Undefined */
	STATE_SET(e_pf, 0); /* Undefined */
	STATE_SET(e_sf, 0); /* Undefined */
	STATE_SET(e_zf, 0); /* Undefined */
#endif

	*res2 = resm;
	return resd;
}

static uint8_t
NAME_(daa)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t al)
{
	if (0x09 < (al & 0x0f) || STATE_GET(e_af)) {
		al += 0x06;
		STATE_SET(e_cf, STATE_GET(e_cf) | (al < 6));
		STATE_SET(e_af, 1);
	} else {
		STATE_SET(e_af, 0);
	}
	if (0x90 < (al & 0xf0) || STATE_GET(e_cf)) {
		al += 0x60;
		STATE_SET(e_cf, 1);
	} else {
		STATE_SET(e_cf, 0);
	}
	STATE_SET(e_of, 0); /* Undefined */
	NAME_(det_pf)(cpssp, cpu, al);
	NAME_(det_sf_s)(cpssp, cpu, 0, al);
	NAME_(det_zf_s)(cpssp, cpu, 0, al);

	return al;
}

static uint8_t
NAME_(das)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t al)
{
	uint8_t af;
	uint8_t cf;
	uint8_t al_save;

	af = STATE_GET(e_af);
	cf = STATE_GET(e_cf);

	STATE_SET(e_af, 0);
	STATE_SET(e_cf, 0);
	al_save = al;

	if (0x09 < (al & 0x0f) || af) {
		STATE_SET(e_af, 1);
		if (al < 0x06 || cf) {
			STATE_SET(e_cf, 1);
		}
		al -= 0x06;
	}
	if (0x99 < al_save || cf) {
		al -= 0x60;
		STATE_SET(e_cf, 1);
	}
	STATE_SET(e_of, 0); /* Undefined */
	NAME_(det_pf)(cpssp, cpu, al);
	NAME_(det_sf_s)(cpssp, cpu, 0, al);
	NAME_(det_zf_s)(cpssp, cpu, 0, al);

	return al;
}

static uint16_t
NAME_(aaa)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t op1)
{
	uint8_t al;
	uint8_t ah;

	al = (op1 >> 0) & 0xff;
	ah = (op1 >> 8) & 0xff;

	if (9 < al || STATE_GET(e_af)) {
		al += 6;
		ah += 1;
		STATE_SET(e_af, 1);
		STATE_SET(e_cf, 1);
	} else {
		STATE_SET(e_af, 0);
		STATE_SET(e_cf, 0);
	}
	STATE_SET(e_of, 0); /* Undefined */
	STATE_SET(e_pf, 0); /* Undefined */
	STATE_SET(e_sf, 0); /* Undefined */
	STATE_SET(e_zf, 0); /* Undefined */
	al &= 0x0f;

	return (ah << 8) | (al << 0);
}

static uint16_t
NAME_(aas)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t op1)
{
	uint8_t al;
	uint8_t ah;

	al = (op1 >> 0) & 0xff;
	ah = (op1 >> 8) & 0xff;

	if (9 < al || STATE_GET(e_af)) {
		al -= 6;
		ah -= 1;
		STATE_SET(e_af, 1);
		STATE_SET(e_cf, 1);
	} else {
		STATE_SET(e_af, 0);
		STATE_SET(e_cf, 0);
	}
	STATE_SET(e_of, 0); /* Undefined */
	STATE_SET(e_pf, 0); /* Undefined */
	STATE_SET(e_sf, 0); /* Undefined */
	STATE_SET(e_zf, 0); /* Undefined */
	al &= 0x0f;

	return (ah << 8) | (al << 0);
}

static uint16_t
NAME_(aam)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t op1, uint8_t op2)
{
	uint8_t al;
	uint8_t ah;

	if (op2 == 0) {
		NAME_(DE)(cpssp, cpu);
	}

	al = (op1 >> 0) & 0xff;
	ah = (op1 >> 8) & 0xff;

	ah = al / op2;
	al = al % op2;
	STATE_SET(e_af, 0); /* Undefined */
	STATE_SET(e_cf, 0); /* Undefined */
	STATE_SET(e_of, 0); /* Undefined */
	NAME_(det_pf)(cpssp, cpu, al);
	NAME_(det_sf_s)(cpssp, cpu, 0, al);
	NAME_(det_zf_s)(cpssp, cpu, 0, al);

	return (ah << 8) | (al << 0);
}

static uint16_t
NAME_(aad)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t op1, uint8_t op2)
{
	uint8_t al;
	uint8_t ah;

	al = (op1 >> 0) & 0xff;
	ah = (op1 >> 8) & 0xff;

	al = ah * op2 + al;
	ah = 0;
	STATE_SET(e_af, 0); /* Undefined */
	STATE_SET(e_cf, 0); /* Undefined */
	STATE_SET(e_of, 0); /* Undefined */
	NAME_(det_pf)(cpssp, cpu, al);
	NAME_(det_sf_s)(cpssp, cpu, 0, al);
	NAME_(det_zf_s)(cpssp, cpu, 0, al);

	return (ah << 8) | (al << 0);
}

static uint32_t
NAME_(alu_bit)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t op_mode)
{
	uint32_t res;

	switch (cpu->reg_opex) {
	case 0x0:
		res = NAME_(rol)(cpssp, cpu, op1, op2, op_mode);
		break;
	case 0x1:
		res = NAME_(ror)(cpssp, cpu, op1, op2, op_mode);
		break;
	case 0x2:
		res = NAME_(rcl)(cpssp, cpu, op1, op2, op_mode);
		break;
	case 0x3:
		res = NAME_(rcr)(cpssp, cpu, op1, op2, op_mode);
		break;
	case 0x4:
		res = NAME_(sal_shl)(cpssp, cpu, op1, op2, op_mode);
		break;
	case 0x5:
		res = NAME_(shr)(cpssp, cpu, op1, op2, op_mode);
		break;
	case 0x6:
		assert(0); /* Mustn't happen. */
		break;
	case 0x7:
		res = NAME_(sar)(cpssp, cpu, op1, op2, op_mode);
		break;
	default:
		assert(0); /* Mustn't happen. */
		break;
	}

	return res;
}

static udata_t
NAME_(alu_arith)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t operation, uint8_t op_mode)
{
	udata_t res;

	switch (operation) {
	case 0x0:
		res = NAME_(add)(cpssp, cpu, op1, op2, op_mode);
		break;
	case 0x1:
		res = NAME_(or)(cpssp, cpu, op1, op2, op_mode);
		break;
	case 0x2:
		res = NAME_(adc)(cpssp, cpu, op1, op2, op_mode);
		break;
	case 0x3:
		res = NAME_(sbb)(cpssp, cpu, op1, op2, op_mode);
		break;
	case 0x4:
		res = NAME_(and)(cpssp, cpu, op1, op2, op_mode);
		break;
	case 0x5:
		res = NAME_(sub)(cpssp, cpu, op1, op2, op_mode);
		break;
	case 0x6:
		res = NAME_(xor)(cpssp, cpu, op1, op2, op_mode);
		break;
	case 0x7:
		NAME_(cmp)(cpssp, cpu, op1, op2, op_mode);
		res = 0; /* Not used. */
		break;
	default:
		assert(0); /* Mustn't happen. */
		break;
	}

	return res;
}

static void
NAME_(jmp)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t offset, uint8_t op_mode)
{
	// TODO check for seg_limit
	cpu->eip += offset;
	cpu->eip = NAME_(fix_size)(cpu->eip, op_mode);
}

static void
NAME_(retn)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint32_t val)
{
	int st_mode;
	int op_mode;
	udata_t sp;

	st_mode = NAME_(get_stack_mode)(cpssp, cpu);
	op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
	sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);

	/* Pop %eip. */
	cpu->eip = NAME_(pop2)(cpssp, cpu, NAME_(SEG_SS), &sp, st_mode, op_mode, NAME_(usermode)(cpssp, cpu));

	/* Release parameters from stack. */
	sp += val;

	NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);
}

static void
NAME_(jcc)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t offset, uint32_t cond, uint8_t op_mode)
{
	if (cond) {
		NAME_(jmp)(cpssp, cpu, offset, op_mode);
	}
}

static void
NAME_(loopcc)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t offset, uint32_t cond, uint32_t not)
{
	uint8_t addr_mode;
	udata_t cx;

	addr_mode = NAME_(get_addr_mode)(cpssp, cpu);

	cx = NAME_(load_reg)(cpssp, cpu, NAME_(E_CX), addr_mode);
	cx--;
	cx = NAME_(fix_size)(cx, addr_mode);

	NAME_(store_reg)(cpssp, cpu, NAME_(E_CX), cx, addr_mode);
	if (cx != 0 && (cond ^ not)) {
		NAME_(jmp)(cpssp, cpu, offset, NAME_(get_op_mode_ip_sp)(cpssp, cpu)); /* FIXME op_mode or addr_mode, too? */
	}
}

static uint32_t
NAME_(test_ecx)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t ecx)
{
	return ecx == 0;
}

static uint32_t
NAME_(test_pf)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	return STATE_GET(e_pf);
}

static uint32_t
NAME_(test_of)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	return STATE_GET(e_of);
}

static uint32_t
NAME_(test_sf)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	return STATE_GET(e_sf);
}

static uint32_t
NAME_(test_zf)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	return STATE_GET(e_zf);
}

static uint32_t
NAME_(test_cf)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	return STATE_GET(e_cf);
}

static uint32_t
NAME_(test_na_be)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	return STATE_GET(e_cf) | STATE_GET(e_zf);
}

static uint32_t
NAME_(test_ng_le)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	return STATE_GET(e_zf) | (STATE_GET(e_sf) ^ STATE_GET(e_of));
}

static uint32_t
NAME_(test_l_nge)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	return STATE_GET(e_sf) ^ STATE_GET(e_of);
}

static uint32_t
NAME_(test_cc)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t cc)
{
	uint32_t res;

	switch (cc) {
	case 0x0:
	case 0x1:
		res = NAME_(test_of)(cpssp, cpu);
		break;
	case 0x2:
	case 0x3:
		res = NAME_(test_cf)(cpssp, cpu);
		break;
	case 0x4:
	case 0x5:
		res = NAME_(test_zf)(cpssp, cpu);
		break;
	case 0x6:
	case 0x7:
		res = NAME_(test_na_be)(cpssp, cpu);
		break;
	case 0x8:
	case 0x9:
		res = NAME_(test_sf)(cpssp, cpu);
		break;
	case 0xa:
	case 0xb:
		res = NAME_(test_pf)(cpssp, cpu);
		break;
	case 0xc:
	case 0xd:
		res = NAME_(test_l_nge)(cpssp, cpu);
		break;
	case 0xe:
	case 0xf:
		res = NAME_(test_ng_le)(cpssp, cpu);
		break;
	default:
		assert(0); /* Mustn't happen. */
	}
	res ^= cc & 0x1;

	return res;
}

static void
NAME_(lidt)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint64_t descr)
{
	seg_descr_t entry;

	entry.entry[0] = descr;
	DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_IDT), entry);
}

static void
NAME_(sidt)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint64_t *descr)
{
	seg_descr_t entry;

	entry = DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_IDT));
	*descr = entry.entry[0];
}

static void
NAME_(lgdt)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint64_t descr)
{
	seg_descr_t entry;

	entry.entry[0] = descr;
	DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_GDT), entry);
}

static void
NAME_(sgdt)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint64_t *descr)
{
	seg_descr_t entry;

	entry = DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_GDT));
	*descr = entry.entry[0];
}

static void
NAME_(lldt)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t selector)
{
	int err;
	uint16_t errc;
	uint8_t type;

	if (selector & 4) {
		fprintf(stderr, "WARNING: LDT selector points to LDT.\n");
		NAME_(GPe)(cpssp, cpu, selector & 0xfffc);
	}

	err = NAME_(read_entry)(cpssp, cpu, NAME_(SEG_DATA), selector, 0, STATE_GET(cr0_pe), STATE_GET(e_vm), 0, &errc);
	if (err != NAME_(EXC_NONE)) {
		NAME_(mem_err2)(cpssp, cpu, err, errc, NAME_(EXC_GP), selector & 0xfffc);
	}

	type = NAME_(seg_type)(cpssp, NAME_(SEG_DATA));
	if (type != 0x00
	 && (type & 0x1f) != 0x02) {
		/* No LDT. */
		fprintf(stderr, "WARNING: No LDT.\n");
		NAME_(GPe)(cpssp, cpu, selector & 0xfffc);
	}

	if (type != 0x00
	 && ! NAME_(seg_present)(cpssp, NAME_(SEG_DATA))) {
		/* Not present. */
		NAME_(NPe)(cpssp, cpu, selector & 0xfffc);
	}

	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_LDT), selector);
	DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_LDT), DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_DATA)));
}

static void
NAME_(ltr)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t selector)
{
	int err;
	uint16_t errc;
	uint8_t type;

	if (selector & 4) {
		fprintf(stderr, "WARNING: TR selector points to LDT.\n");
		NAME_(GPe)(cpssp, cpu, selector & 0xfffc);
	}

	err = NAME_(read_entry)(cpssp, cpu, NAME_(SEG_DATA), selector, 0, STATE_GET(cr0_pe), STATE_GET(e_vm), 0, &errc);
	if (err != NAME_(EXC_NONE)) {
		NAME_(mem_err2)(cpssp, cpu, err, errc, NAME_(EXC_GP), selector & 0xfffc);
	}

	type = NAME_(seg_type)(cpssp, NAME_(SEG_DATA));
	if ((type & 0x17) != 0x01) {
		/* No TSS or TSS busy. */
		fprintf(stderr, "WARNING: No TSS or TSS busy.\n");
		NAME_(GPe)(cpssp, cpu, selector & 0xfffc);
	}

	if (! NAME_(seg_present)(cpssp, NAME_(SEG_DATA))) {
		/* Not present. */
		/* Cannot happen? FIXME */
		NAME_(NPe)(cpssp, cpu, selector & 0xfffc);
	}

	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_TR), selector);
	NAME_(mark_busy)(cpssp, cpu, selector, NAME_(SEG_DATA));
	DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_TR),
			DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_DATA)));
}

static udata_t
NAME_(rep_test)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	int addr_mode;
	udata_t tmp;

	addr_mode = NAME_(get_addr_mode)(cpssp, cpu);

	tmp = NAME_(load_reg)(cpssp, cpu, NAME_(E_CX), addr_mode);

	return tmp;
}

static void
NAME_(rep)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	cpu->eip = STATE_GET(eip);
}

static void
NAME_(rep_dec)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	int addr_mode;
	udata_t tmp;

	addr_mode = NAME_(get_addr_mode)(cpssp, cpu);

	tmp = NAME_(load_reg)(cpssp, cpu, NAME_(E_CX), addr_mode);
	tmp--;
	NAME_(store_reg)(cpssp, cpu, NAME_(E_CX), tmp, addr_mode);
}

static void
NAME_(repn_z)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	int sel;

	/* repne/repnz  == 0x1 */
	/* rep/repe/repz == 0x2 */
	sel = cpu->pre_repe_repne & 0x1;
	sel ^= STATE_GET(e_zf);

	if (sel) {
		cpu->eip = STATE_GET(eip);
	}
}

static void
NAME_(movs)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int op_mode)
{
	int addr_mode;
	udata_t next;
	udata_t addr_src;
	udata_t addr_dst;
	udata_t val;

	addr_mode = NAME_(get_addr_mode)(cpssp, cpu);
	next = (1 << op_mode) * cpssp->NAME.direction;

	addr_src = NAME_(load_reg)(cpssp, cpu, NAME_(E_SI), addr_mode);
	addr_dst = NAME_(load_reg)(cpssp, cpu, NAME_(E_DI), addr_mode);
	val = NAME_(smr)(cpssp, cpu, addr_src, cpu->seg[0], 0, NAME_(usermode)(cpssp, cpu), op_mode);
	NAME_(smw)(cpssp, cpu, addr_dst, NAME_(SEG_ES), val, NAME_(usermode)(cpssp, cpu), op_mode);
	addr_src += next;
	addr_dst += next;
	NAME_(store_reg)(cpssp, cpu, NAME_(E_SI), addr_src, addr_mode);
	NAME_(store_reg)(cpssp, cpu, NAME_(E_DI), addr_dst, addr_mode);
}

static void
NAME_(cmps)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int op_mode)
{
	int addr_mode;
	udata_t next;
	udata_t addr_src;
	udata_t addr_dst;
	udata_t op1;
	udata_t op2;

	addr_mode = NAME_(get_addr_mode)(cpssp, cpu);
	next = (1 << op_mode) * cpssp->NAME.direction;

	addr_src = NAME_(load_reg)(cpssp, cpu, NAME_(E_SI), addr_mode);
	addr_dst = NAME_(load_reg)(cpssp, cpu, NAME_(E_DI), addr_mode);
	op1 = NAME_(smr)(cpssp, cpu, addr_src, cpu->seg[0], 0, NAME_(usermode)(cpssp, cpu), op_mode);
	op2 = NAME_(smr)(cpssp, cpu, addr_dst, NAME_(SEG_ES), 0, NAME_(usermode)(cpssp, cpu), op_mode);
	NAME_(cmp)(cpssp, cpu, op1, op2, op_mode);
	addr_src += next;
	addr_dst += next;
	NAME_(store_reg)(cpssp, cpu, NAME_(E_SI), addr_src, addr_mode);
	NAME_(store_reg)(cpssp, cpu, NAME_(E_DI), addr_dst, addr_mode);
}

static void
NAME_(stos)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int op_mode)
{
	int addr_mode;
	udata_t next;
	udata_t addr;
	udata_t val;

	addr_mode = NAME_(get_addr_mode)(cpssp, cpu);
	next = (1 << op_mode) * cpssp->NAME.direction;

	addr = NAME_(load_reg)(cpssp, cpu, NAME_(E_DI), addr_mode);
	val = NAME_(load_reg)(cpssp, cpu, NAME_(E_AX), op_mode);
	NAME_(smw)(cpssp, cpu, addr, NAME_(SEG_ES), val, NAME_(usermode)(cpssp, cpu), op_mode);
	addr += next;
	NAME_(store_reg)(cpssp, cpu, NAME_(E_DI), addr, addr_mode);
}

static void
NAME_(scas)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int op_mode)
{
	int addr_mode;
	udata_t next;
	udata_t addr;
	udata_t op1;
	udata_t op2;

	addr_mode = NAME_(get_addr_mode)(cpssp, cpu);
	next = (1 << op_mode) * cpssp->NAME.direction;

	addr = NAME_(load_reg)(cpssp, cpu, NAME_(E_DI), addr_mode);
	op1 = NAME_(load_reg)(cpssp, cpu, NAME_(E_AX), op_mode);
	op2 = NAME_(smr)(cpssp, cpu, addr, NAME_(SEG_ES), 0, NAME_(usermode)(cpssp, cpu), op_mode);
	NAME_(cmp)(cpssp, cpu, op1, op2, op_mode);
	addr += next;
	NAME_(store_reg)(cpssp, cpu, NAME_(E_DI), addr, addr_mode);
}

static void
NAME_(lods)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int op_mode)
{
	int addr_mode;
	udata_t next;
	udata_t addr;
	udata_t val;

	addr_mode = NAME_(get_addr_mode)(cpssp, cpu);
	next = (1 << op_mode) * cpssp->NAME.direction;

	addr = NAME_(load_reg)(cpssp, cpu, NAME_(E_SI), addr_mode);
	val = NAME_(smr)(cpssp, cpu, addr, cpu->seg[0], 0, NAME_(usermode)(cpssp, cpu), op_mode);
	NAME_(store_reg)(cpssp, cpu, NAME_(E_AX), val, op_mode);
	addr += next;
	NAME_(store_reg)(cpssp, cpu, NAME_(E_SI), addr, addr_mode);
}

static void
NAME_(ins)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int op_mode)
{
	int addr_mode;
	udata_t next;
	udata_t addr;
	udata_t port;
	udata_t val;

	addr_mode = NAME_(get_addr_mode)(cpssp, cpu);
	next = (1 << op_mode) * cpssp->NAME.direction;

	addr = NAME_(load_reg)(cpssp, cpu, NAME_(E_DI), addr_mode);
	(void) NAME_(smr)(cpssp, cpu, addr, NAME_(SEG_ES), 2, NAME_(usermode)(cpssp, cpu), op_mode);
	port = NAME_(load_reg16)(cpssp, cpu, NAME_(E_DX));
	val = NAME_(in)(cpssp, cpu, port, op_mode);
	NAME_(smw)(cpssp, cpu, addr, NAME_(SEG_ES), val, NAME_(usermode)(cpssp, cpu), op_mode);
	addr += next;
	NAME_(store_reg)(cpssp, cpu, NAME_(E_DI), addr, addr_mode);
}

static void
NAME_(outs)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int op_mode)
{
	int addr_mode;
	udata_t next;
	udata_t addr;
	udata_t port;
	udata_t val;

	addr_mode = NAME_(get_addr_mode)(cpssp, cpu);
	next = (1 << op_mode) * cpssp->NAME.direction;

	addr = NAME_(load_reg)(cpssp, cpu, NAME_(E_SI), addr_mode);
	port = NAME_(load_reg16)(cpssp, cpu, NAME_(E_DX));
	val = NAME_(smr)(cpssp, cpu, addr, cpu->seg[0], 0, NAME_(usermode)(cpssp, cpu), op_mode);
	NAME_(out)(cpssp, cpu, port, val, op_mode);
	addr += next;
	NAME_(store_reg)(cpssp, cpu, NAME_(E_SI), addr, addr_mode);
}

static void
NAME_(calltss)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint16_t tss_selector,
	uint32_t eip,
	int isint,
	int iscall
)
{
	int op_mode;
	uint32_t off;
	uint32_t val;
	int i;

	/*
	 * Save state of current task.
	 */
	if ((NAME_(seg_type)(cpssp, NAME_(SEG_TR)) & 0x18) == 0x00) {
		/* 16-bit TSS */
		op_mode = 1; /* FIXME 64bit mode */

	} else {
		/* 32-bit TSS */
		op_mode = 2; /* FIXME 64bit mode */
	}

	off = 0;

	/* Previous Task */
	if (iscall) {
		val = STATE_GET(selector[NAME_(SEG_TR)]);
		NAME_(smw)(cpssp, cpu, off, NAME_(SEG_TR), val, 0, op_mode);
	}
	off += 1 << op_mode;

	/* %ss0/%esp0, %ss1/%esp1, %ss2/%esp2 */
	/* Static fields. */
	off += 1 << op_mode;
	off += 1 << op_mode;
	off += 1 << op_mode;
	off += 1 << op_mode;
	off += 1 << op_mode;
	off += 1 << op_mode;

	if (op_mode == 2) {
		/* %cr3 */
		/* Static field. */
		off += 1 << op_mode;
	}

	/* %eip */
	val = cpu->eip;
	NAME_(smw)(cpssp, cpu, off, NAME_(SEG_TR), val, 0, op_mode);
	off += 1 << op_mode;

	/* %eflags */
	val = NAME_(get_eflags)(cpssp, cpu);
	NAME_(smw)(cpssp, cpu, off, NAME_(SEG_TR), val, 0, op_mode);
	off += 1 << op_mode;

	/* %eax, %ecx, %edx, %ebx, %esp, %ebp, %esi, %edi */
	for (i = 0; i < 8; i++) {
		val = NAME_(load_reg32)(cpssp, cpu, i);
		NAME_(smw)(cpssp, cpu, off, NAME_(SEG_TR), val, 0, op_mode);
		off += 1 << op_mode;
	}

	/* %es, %cs, %ss, %ds */
	for (i = 0; i < 4; i++) {
		val = STATE_GET(selector[i]);
		NAME_(smw16)(cpssp, cpu, off, NAME_(SEG_TR), val, 0);
		off += 1 << op_mode;
	}
	if (op_mode == 2) {
		/* %fs, %gs */
		for (i = 4; i < 6; i++) {
			val = STATE_GET(selector[i]);
			NAME_(smw16)(cpssp, cpu, off, NAME_(SEG_TR), val, 0);
			off += 1 << op_mode;
		}
	}

	/* %ldt */
	/* Static field. */
	off += 1 << op_mode;

	if (op_mode == 2) {
		/* I/O Map, T-bit */
		/* Static field. */
		off += 1 << op_mode;
	}

	if (op_mode == 1) {
		assert(off == 42);
	} else {
		assert(off == 104);
	}

	/*** Point of no return. ***/

	/*
	 * Switch task segment.
	 */
	if (! iscall) {
		/* Reset busy flag of old task. */
		uint32_t entry;

		entry = NAME_(smr32)(cpssp, cpu, (STATE_GET(selector[NAME_(SEG_TR)]) & ~0x7) + 4, NAME_(SEG_GDT), 1, 0);
		entry &= ~(1 << 9);
		NAME_(smw32)(cpssp, cpu, (STATE_GET(selector[NAME_(SEG_TR)]) & ~0x7) + 4, NAME_(SEG_GDT), entry, 0);
	}
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_TR), tss_selector);
	NAME_(mark_busy)(cpssp, cpu, tss_selector, NAME_(SEG_CODE));
	DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_TR),
			DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_CODE)));

	STATE_SET(cr0_ts, 1);

	/*
	 * Load state of next task.
	 */
	if ((NAME_(seg_type)(cpssp, NAME_(SEG_TR)) & 0x18) == 0x00) {
		/* 16-bit TSS */
		op_mode = 1;

	} else {
		/* 32-bit TSS */
		op_mode = 2;
	}

	off = 0;

	/* Previous Task Link */
	off += 1 << op_mode;

	/* %ss0/%esp0, %ss1/%esp1, %ss2/%esp2 */
	off += 1 << op_mode;
	off += 1 << op_mode;
	off += 1 << op_mode;
	off += 1 << op_mode;
	off += 1 << op_mode;
	off += 1 << op_mode;

	if (op_mode == 2) {
		/* %cr3 */
		val = NAME_(smr32)(cpssp, cpu, off, NAME_(SEG_TR), 0, 0);
		off += 1 << op_mode;
		NAME_(set_cr)(cpssp, cpu, 3, val);
	}

	/* %eip */
	val = NAME_(smr)(cpssp, cpu, off, NAME_(SEG_TR), 0, 0, op_mode);
	off += 1 << op_mode;
	cpu->eip = val;

	/* %eflags */
	val = NAME_(smr)(cpssp, cpu, off, NAME_(SEG_TR), 0, 0, op_mode);
	off += 1 << op_mode;
	NAME_(set_eflags_all)(cpssp, cpu, val, op_mode);

	/* %eax, %ecx, %edx, %ebx, %esp, %ebp, %esi, %edi */
	for (i = 0; i < 8; i++) {
		val = NAME_(smr)(cpssp, cpu, off, NAME_(SEG_TR), 0, 0, op_mode);
		off += 1 << op_mode;
		NAME_(store_reg32)(cpssp, cpu, i, val);
	}

	/* %es, %cs, %ss, %ds */
	for (i = 0; i < 4; i++) {
		val = NAME_(smr16)(cpssp, cpu, off, NAME_(SEG_TR), 0, 0);
		off += 1 << op_mode;
		STATE_SET(selector[i], val);
	}
	/* %fs, %gs */
	for (i = 4; i < 6; i++) {
		if (op_mode == 2) {
			val = NAME_(smr16)(cpssp, cpu, off, NAME_(SEG_TR), 0, 0);
			off += 1 << op_mode;
		} else {
			val = 0x0000;
		}
		STATE_SET(selector[i], val);
	}

	/* %ldt */
	val = NAME_(smr16)(cpssp, cpu, off, NAME_(SEG_TR), 0, 0);
	off += 1 << op_mode;
	STATE_SET(selector[NAME_(SEG_LDT)], val);

	/* I/O Map, T-bit */
	off += 1 << op_mode;

	if (op_mode == 1) {
		assert(off == 42);
	} else {
		assert(off == 104);
	}

	/* Load %ldt. Must be first! */
	val = STATE_GET(selector[NAME_(SEG_LDT)]);
	NAME_(lldt)(cpssp, cpu, val);

	/* Load %es, %cs, %ss, %ds, %fs, %gs. */
	for (i = 0; i < 6; i++) {
		val = STATE_GET(selector[i]);
		NAME_(store_segment)(cpssp, cpu, i, val);
	}

	/* Correct? FIXME */
	if (cpssp->NAME.errc) {
		NAME_(push)(cpssp, cpu, cpssp->NAME.error_code, 1 + 1, 0); // FIXME op_mode
	}
}

static void
NAME_(rettss)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int op_mode, int isint, uint8_t params)
{
	assert(0);
}

static void
NAME_(callseg)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	uint8_t op_mode,
	uint16_t cs_selector,
	udata_t eip,
	int isint,
	int iscall,
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	int istn,
#endif
	int param
)
{
	int newstack;
	uint16_t ss;
	udata_t sp;
	int st_mode;
	uint16_t ss_old;
	udata_t sp_old;
	int st_mode_old;

	if (! (NAME_(seg_type)(cpssp, NAME_(SEG_CODE)) & 0x04)
	 && NAME_(seg_dpl)(cpssp, NAME_(SEG_CODE)) < NAME_(cpl)(cpssp, cpu)) {
		/* Inter-Privilege-Level */
		newstack = 1;
	} else {
		newstack = 0;
	}

	if (newstack) {
		uint32_t stack_addr;
		int err;
		uint16_t errc;
		uint8_t tss_type;

		tss_type = (NAME_(seg_type)(cpssp, NAME_(SEG_TR)) >> 3) & 0x1;
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
		tss_type |= STATE_GET(lm_active_msr) << 1;
#endif
		switch (tss_type) {
		case 0:
			/* 16-bit TSS */
			stack_addr = (NAME_(seg_dpl)(cpssp, NAME_(SEG_CODE)) << 2) + 2;
			sp = NAME_(mr16)(cpssp, cpu, stack_addr, NAME_(SEG_TR), 0, 0,
					NAME_(EXC_TS), STATE_GET(selector[NAME_(SEG_TR)]));
			ss = NAME_(mr16)(cpssp, cpu, stack_addr + 2, NAME_(SEG_TR), 0, 0,
					NAME_(EXC_TS), STATE_GET(selector[NAME_(SEG_TR)]));
			break;
		case 1:
			/* 32-bit TSS */
			stack_addr = (NAME_(seg_dpl)(cpssp, NAME_(SEG_CODE)) << 3) + 4;
			sp = NAME_(mr32)(cpssp, cpu, stack_addr, NAME_(SEG_TR), 0, 0,
					NAME_(EXC_TS), STATE_GET(selector[NAME_(SEG_TR)]));
			ss = NAME_(mr16)(cpssp, cpu, stack_addr + 4, NAME_(SEG_TR), 0, 0,
					NAME_(EXC_TS), STATE_GET(selector[NAME_(SEG_TR)]));
			break;
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
		case 2:
			/* RESERVED */
			/* FIXME */
			//NAME_(TSe)(cpssp, cpu, DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_TR)));
			abort();
			break;
		case 3:
			/* 64-bit TSS */
			printf("test\n");
			if (istn) {
				stack_addr = (istn << 3) + 28;
			} else {
				stack_addr = (NAME_(seg_dpl)(cpssp, NAME_(SEG_CODE)) << 3) + 4;
			}
			sp = NAME_(mr64)(cpssp, cpu, stack_addr, NAME_(SEG_TR), 0, 0,
					NAME_(EXC_TS), STATE_GET(selector[NAME_(SEG_TR)]));
			ss = 0 | NAME_(seg_dpl)(cpssp, NAME_(SEG_CODE));
			break;
#endif
		default:
			/* can not happen */
			assert(0);
		}

		/* Read stack segment descriptor. */
		err = NAME_(read_entry)(cpssp, cpu, NAME_(SEG_DATA), ss, 0, 1, 0, 0, &errc);
		if (err != NAME_(EXC_NONE)) {
			NAME_(mem_err2)(cpssp, cpu, err, errc, NAME_(EXC_TS), ss & 0xfffc);
		}

		/* Check stack segment descriptor. */
		if ((NAME_(seg_type)(cpssp, NAME_(SEG_DATA)) & 0x1a) != 0x12) {
			/* Not a writable data segment. */
			fprintf(stderr, "WARNING: Not a writable stack segment.\n");
			NAME_(TSe)(cpssp, cpu, ss & 0xfffc);
		}
		if ((ss & 0x3) != NAME_(seg_dpl)(cpssp, NAME_(SEG_DATA))
		 || NAME_(seg_dpl)(cpssp, NAME_(SEG_DATA)) != NAME_(seg_dpl)(cpssp, NAME_(SEG_CODE))) {
			/* Protection fault. */
			fprintf(stderr, "WARNING: Stack segment with bad privilege level.\n");
			NAME_(TSe)(cpssp, cpu, ss & 0xfffc);
		}
		if (! NAME_(seg_present)(cpssp, NAME_(SEG_DATA))) {
			/* Not present. */
			fprintf(stderr, "WARNING: Stack segment not present.\n");
			NAME_(SSe)(cpssp, cpu, ss & 0xfffc);
		}
		st_mode = 1 + NAME_(seg_db)(cpssp, NAME_(SEG_DATA)); /* FIXME 64bit mode*/

	} else {
		ss = STATE_GET(selector[NAME_(SEG_SS)]);
		DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_DATA),
				DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_SS)));
		st_mode = 1 + NAME_(seg_db)(cpssp, NAME_(SEG_DATA)); /* FIXME 64bit mode */
		sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);
	}

	if (newstack
	 && isint
	 && STATE_GET(e_vm)) {
		uint16_t errc;

		NAME_(push2)(cpssp, cpu, NAME_(SEG_DATA), &sp, STATE_GET(selector[NAME_(SEG_GS)]), st_mode, op_mode, 0);
		NAME_(push2)(cpssp, cpu, NAME_(SEG_DATA), &sp, STATE_GET(selector[NAME_(SEG_FS)]), st_mode, op_mode, 0);
		NAME_(push2)(cpssp, cpu, NAME_(SEG_DATA), &sp, STATE_GET(selector[NAME_(SEG_DS)]), st_mode, op_mode, 0);
		NAME_(push2)(cpssp, cpu, NAME_(SEG_DATA), &sp, STATE_GET(selector[NAME_(SEG_ES)]), st_mode, op_mode, 0);

		NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_ES), 0x0000);
		(void) NAME_(read_entry)(cpssp, cpu, NAME_(SEG_ES), 0x0000, 0, 1, 0, 0, &errc);

		NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_DS), 0x0000);
		(void) NAME_(read_entry)(cpssp, cpu, NAME_(SEG_DS), 0x0000, 0, 1, 0, 0, &errc);

		NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_FS), 0x0000);
		(void) NAME_(read_entry)(cpssp, cpu, NAME_(SEG_FS), 0x0000, 0, 1, 0, 0, &errc);

		NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_GS), 0x0000);
		(void) NAME_(read_entry)(cpssp, cpu, NAME_(SEG_GS), 0x0000, 0, 1, 0, 0, &errc);
	}

	ss_old = STATE_GET(selector[NAME_(SEG_SS)]);
	st_mode_old = NAME_(seg_db)(cpssp, NAME_(SEG_SS));
	sp_old = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode_old + 1);

	if (newstack
	 && iscall) {
		NAME_(push2)(cpssp, cpu, NAME_(SEG_DATA), &sp, ss_old, st_mode, op_mode, 0);
		NAME_(push2)(cpssp, cpu, NAME_(SEG_DATA), &sp, sp_old, st_mode, op_mode, 0);
	}

	if (isint) {
		uint32_t eflags_old;

		eflags_old = NAME_(get_eflags)(cpssp, cpu);

		NAME_(push2)(cpssp, cpu, NAME_(SEG_DATA), &sp, eflags_old, st_mode, op_mode, 0);
	}

	if (newstack
	 && param != 0) {
		int i;

		fprintf(stderr, "%d params!\n", param);
		for (i = param - 1; 0 <= i; i--) {
			uint32_t val;

			val = NAME_(smr)(cpssp, cpu, sp_old + i * (1 << op_mode), NAME_(SEG_SS), 0, 0, op_mode);
			NAME_(push2)(cpssp, cpu, NAME_(SEG_DATA), &sp, val, st_mode, op_mode, 0);
		}
	}

	if (iscall) {
		uint16_t cs_selector_old;
		uint32_t eip_old;

		eip_old = cpu->eip;
		cs_selector_old = STATE_GET(selector[NAME_(SEG_CS)]);

		NAME_(push2)(cpssp, cpu, NAME_(SEG_DATA), &sp, cs_selector_old, st_mode, op_mode, 0);
		NAME_(push2)(cpssp, cpu, NAME_(SEG_DATA), &sp, eip_old, st_mode, op_mode, 0);
	}
	if (cpssp->NAME.errc) {
		NAME_(push2)(cpssp, cpu, NAME_(SEG_DATA), &sp, cpssp->NAME.error_code, st_mode, op_mode, 0);
	}

	/* Set %ss:%esp. */
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_SS), ss);
	NAME_(mark_accessed)(cpssp, cpu, ss, NAME_(SEG_DATA));
	DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_SS), DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_DATA)));
	NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);

	/* Set %cs:%eip. */
	cpu->eip = eip;
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_CS), cs_selector);
	NAME_(mark_accessed)(cpssp, cpu, cs_selector, NAME_(SEG_CODE));
	DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_CS), DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_CODE)));
}

static void
NAME_(retseg)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int op_mode, int isint, uint8_t params)
{
	int err;
	uint16_t errc;
	uint16_t ss_selector;
	uint16_t cs_selector;
	uint32_t eflags;
	uint32_t eip;
	uint32_t esp;
	uint8_t rpl;

	eip = NAME_(pop)(cpssp, cpu, op_mode, NAME_(usermode)(cpssp, cpu)); /* FIXME st_mode in 64bit mode -> pop2 */
	cs_selector = NAME_(pop)(cpssp, cpu, op_mode, NAME_(usermode)(cpssp, cpu));
	if (isint) {
		eflags = NAME_(pop)(cpssp, cpu, op_mode, NAME_(usermode)(cpssp, cpu));
	} else {
		eflags = 0; /* Not used. */
	}

	err = NAME_(read_entry)(cpssp, cpu, NAME_(SEG_CODE), cs_selector,
			0, STATE_GET(cr0_pe), STATE_GET(e_vm) | ((eflags >> 17) & 1), 1, &errc);
	if (err != NAME_(EXC_NONE)) {
		NAME_(mem_err2)(cpssp, cpu, err, errc, NAME_(EXC_GP), cs_selector & 0xfffc);
	}

	if (! STATE_GET(cr0_pe)) {
		/* Real-mode */
		rpl = 0;
	} else if (STATE_GET(e_vm)
		|| ((eflags >> 17) & 1)) {
		/* Virtual-8086 Mode */
		rpl = 3;
	} else {
		/* Protected Mode */
		rpl = cs_selector & 3;
	}

	if ((NAME_(seg_type)(cpssp, NAME_(SEG_CODE)) & 0x18) != 0x18) {
		/* No code segment. */
		NAME_(GPe)(cpssp, cpu, cs_selector & 0xfffc);
	}
	if (rpl < NAME_(cpl)(cpssp, cpu)) {
		/* Protection fault. */
		NAME_(GPe)(cpssp, cpu, cs_selector & 0xfffc);
	}
	if (NAME_(seg_type)(cpssp, NAME_(SEG_CODE)) & 0x4) {
		/* Conforming code segment. */
		if (rpl < NAME_(seg_dpl)(cpssp, NAME_(SEG_CODE))) {
			/* Protection fault. */
			NAME_(GPe)(cpssp, cpu, cs_selector & 0xfffc);
		}
	} else {
		/* Non-conforming code segment. */
		if (rpl != NAME_(seg_dpl)(cpssp, NAME_(SEG_CODE))) {
			/* Protection fault. */
			NAME_(GPe)(cpssp, cpu, cs_selector & 0xfffc);
		}
	}
	if (! NAME_(seg_present)(cpssp, NAME_(SEG_CODE))) {
		/* Not present. */
		NAME_(NPe)(cpssp, cpu, cs_selector & 0xfffc);
	}

	if (NAME_(cpl)(cpssp, cpu) < rpl) {
		/*
		 * Return-To-Outer-Privilege-Level
		 */
		uint16_t sel_list[4];
		static const uint8_t seg_list[4] = {
			NAME_(SEG_ES),
			NAME_(SEG_DS),
			NAME_(SEG_FS),
			NAME_(SEG_GS),
		};
		int i;

		esp = NAME_(pop)(cpssp, cpu, op_mode, NAME_(usermode)(cpssp, cpu));
		ss_selector = NAME_(pop)(cpssp, cpu, op_mode, NAME_(usermode)(cpssp, cpu));

		err = NAME_(read_entry)(cpssp, cpu, NAME_(SEG_DATA), ss_selector,
				0, STATE_GET(cr0_pe), STATE_GET(e_vm) | ((eflags >> 17) & 1), 0, &errc);
		if (err != NAME_(EXC_NONE)) {
			NAME_(mem_err2)(cpssp, cpu, err, errc, NAME_(EXC_SS), ss_selector & 0xfffc);
		}

		if (unlikely((NAME_(seg_type)(cpssp, NAME_(SEG_DATA)) & 0x1a) != 0x12)) {
			fprintf(stderr, "WARNING: Bad stack segment.\n");
			NAME_(SSe)(cpssp, cpu, ss_selector & 0xfffc);
		}
#if 0 /* Problem with vm86 mode - FIXME */
		if (unlikely((ss_selector & 0x3) != (cs_selector & 0x3))) {
			fprintf(stderr, "WARNING: Stack segment with different dpl.\n");
			NAME_(GPe)(cpssp, cpu, ss_selector & 0xfffc);
		}
		if (unlikely(NAME_(seg_dpl)(cpssp, NAME_(SEG_DATA)) != (cs_selector & 0x3))) {
			fprintf(stderr, "WARNING: Stack segment with different dpl.\n");
			NAME_(SSe)(cpssp, cpu, ss_selector & 0xfffc);
		}
#endif /* Problem with vm86 mode - FIXME */
		if (unlikely(! NAME_(seg_present)(cpssp, NAME_(SEG_DATA)))) {
			fprintf(stderr, "WARNING: Stack segment not present.\n");
			NAME_(SSe)(cpssp, cpu, ss_selector & 0xfffc);
		}

		/* Pop/nullify selectors for %es, %ds, %fs, %gs. */
		for (i = 0; i < 4; i++) {
			if ((eflags >> 17) & 1) {
				sel_list[i] = NAME_(pop)(cpssp, cpu, 1 + 1, 0); // FIXME op_mode
			} else {
				sel_list[i] = 0x0000;
			}
		}

		/*** Point of no return. ***/

		/* Load/fix %es, %ds, %fs, %gs. */
		for (i = 0; i < 4; i++) {
			int flag;

			if ((eflags >> 17) & 1) {
				/* Return to vm86 mode. */
				flag = 1;

			} else {
				/* Return to protected mode. */
				uint8_t seg;
				uint16_t sel;

				seg = seg_list[i];
				sel = STATE_GET(selector[seg]);

				flag = (sel & 0xfffc)
					&& (((NAME_(seg_type)(cpssp, seg) & 0x18) == 0x10)
					 || ((NAME_(seg_type)(cpssp, seg) & 0x1c) == 0x18))
					&& NAME_(seg_dpl)(cpssp, seg) < NAME_(seg_dpl)(cpssp, NAME_(SEG_CODE));
			}
			if (flag) {
				NAME_(store_sreg)(cpssp, cpu, seg_list[i], sel_list[i]);
				(void) NAME_(read_entry)(cpssp, cpu, seg_list[i], sel_list[i], 0,
						STATE_GET(cr0_pe), STATE_GET(e_vm) | ((eflags >> 17) & 1), 0, &errc);
			}
		}

		/* Load %ss/%esp. */
		NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), esp, op_mode);
		NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_SS), ss_selector);
		NAME_(mark_accessed)(cpssp, cpu, ss_selector, NAME_(SEG_DATA));
		DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_SS),
				DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_DATA)));
	}

	/* Load %eflags. */
	if (isint) {
		/* CF, PF, AF, ZF, SF, TF, DF, OF, NT, RF, AC, ID */
		NAME_(set_eflags_common)(cpssp, cpu, eflags, op_mode);

		if (! STATE_GET(cr0_pe)) {
			/* REAL-ADDRESS-MODE */
			/* VM, VIP, VIF not modified. */
			NAME_(set_eflags_iopl)(cpssp, cpu, eflags);
			NAME_(set_eflags_if)(cpssp, cpu, eflags);

		} else if (STATE_GET(e_vm)) {
			/* RETURN-FROM-VIRTUAL-8086-MODE */
			/* VM, VIP, VIF not modified. */
			/* IOPL not modified. */
			NAME_(set_eflags_if)(cpssp, cpu, eflags);

		} else if ((eflags >> 17) & 1) {
			/* RETURN-TO-VIRTUAL-8086-MODE */
			NAME_(set_eflags_vm)(cpssp, cpu, eflags);
			NAME_(set_eflags_iopl)(cpssp, cpu, eflags);
			NAME_(set_eflags_if)(cpssp, cpu, eflags);

		} else {
			/* RETURN-TO-OUTER-PRIVILEGE-LEVEL */
			/* RETURN-TO-SAME-PRIVILEGE-LEVEL */
			if (NAME_(cpl)(cpssp, cpu) == 0) {
				NAME_(set_eflags_vm)(cpssp, cpu, eflags);
				NAME_(set_eflags_iopl)(cpssp, cpu, eflags);
			}
			if (NAME_(cpl)(cpssp, cpu) <= STATE_GET(e_iopl)) {
				NAME_(set_eflags_if)(cpssp, cpu, eflags);
			}
		}
	}

	/* Load %cs/%eip. */
	cpu->eip = eip;
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_CS), cs_selector);
	NAME_(mark_accessed)(cpssp, cpu, cs_selector, NAME_(SEG_CODE));
	DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_CS),
			DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_CODE)));

	/* Release parameters from stack. */
	esp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), op_mode);
	esp += params;
	NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), esp, op_mode);
}

static void
NAME_(ljmpcallint)(
	struct cpssp *cpssp,
	struct NAME_(cpu) *cpu,
	udata_t offset,
	uint16_t selector,
	int op_mode,
	int isint,
	int iscall,
	int ishw
)
{
	int err;
	uint16_t errc;
	int cpl;
	int rpl;
	uint8_t type;
	uint8_t param;
	int int_enable;
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	uint8_t istn;
#endif

	err = NAME_(read_entry)(cpssp, cpu, NAME_(SEG_CODE), selector,
			isint, STATE_GET(cr0_pe), STATE_GET(e_vm), 1, &errc);
	if (err != NAME_(EXC_NONE)) {
		NAME_(mem_err2)(cpssp, cpu, err, errc, NAME_(EXC_GP), (selector & 0xfffc) | (isint << 1));
	}

	if (ishw) {
		/* Hardware-generated */
		cpl = 0;
		rpl = 0;
	} else if (! STATE_GET(cr0_pe)) {
		/* Real-mode */
		cpl = 0;
		rpl = 0;
	} else if (STATE_GET(e_vm)) {
		/* Virtual-8086 Mode */
		cpl = 3;
		rpl = 3;
	} else {
		/* Protected Mode */
		cpl = NAME_(cpl)(cpssp, cpu);
		rpl = selector & 3;
	}

	type = NAME_(seg_type)(cpssp, NAME_(SEG_CODE));
	if ((type & 0x14) == 0x04) {
		/*
		 * Gate
		 */
		if (NAME_(seg_dpl)(cpssp, NAME_(SEG_CODE)) < cpl
		 || NAME_(seg_dpl)(cpssp, NAME_(SEG_CODE)) < rpl) {
			/* Protection error. */
			fprintf(stderr, "WARNING: Segment protection error.\n");
			NAME_(GPe)(cpssp, cpu, (selector & 0xfffc) | (isint << 1));
		}
		if (! NAME_(seg_present)(cpssp, NAME_(SEG_CODE))) {
			/* Not present. */
			fprintf(stderr, "WARNING: Segment not present.\n");
			NAME_(NPe)(cpssp, cpu, (selector & 0xfffc) | (isint << 1));
		}

		offset = NAME_(seg_offset)(cpssp, NAME_(SEG_CODE)); // TODO 128 bit system segments desc
		selector = NAME_(seg_selector)(cpssp, NAME_(SEG_CODE));
		param = NAME_(seg_param)(cpssp, NAME_(SEG_CODE));
		op_mode = 1 + ((type >> 3) & 1);
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
		istn = 0;
		if (STATE_GET(lm_active_msr)) {
			/* Do not copy parameters when calling 64-bit call gates */
			param = 0;

			if (isint) {
				/* ISTn or legacy stack switch used */
				istn = NAME_(seg_istn)(cpssp, NAME_(SEG_CODE));
			}
		}
		op_mode += STATE_GET(lm_active_msr);
#endif
		int_enable = (~(type >> 1) & 1) | ((type >> 0) & 1);

		err = NAME_(read_entry)(cpssp, cpu, NAME_(SEG_CODE), selector, 0, STATE_GET(cr0_pe), 0, 1, &errc);
		if (err != NAME_(EXC_NONE)) {
			NAME_(mem_err2)(cpssp, cpu, err, errc, NAME_(EXC_GP), (selector & 0xfffc) | (isint << 1));
		}

		/* No protection checks! */

		switch (type) {
		case 0x05: /* Task Gate */
		case 0x0d: /* Reserved */
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
			if (STATE_GET(lm_active_msr)) {
				fprintf(stderr, "WARNING: Invalid selector in long mode\n");
				NAME_(GPe)(cpssp, cpu, (selector & 0xfffc));
			}
#endif
			if ((NAME_(seg_type)(cpssp, NAME_(SEG_CODE)) & 0x15) != 0x01) {
				/* No TSS segment. */
				fprintf(stderr, "WARNING: Task gate with non-TSS selector\n");
				NAME_(GPe)(cpssp, cpu, (selector & 0xfffc) | (isint << 1));
			}
			break;
		case 0x04: /* 16-Bit Call Gate */
		case 0x06: /* 16-Bit Interrupt Gate */
		case 0x07: /* 16-Bit Trap Gate */
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
			if (STATE_GET(lm_active_msr)) {
				fprintf(stderr, "WARNING: Invalid selector in long mode\n");
				NAME_(GPe)(cpssp, cpu, (selector & 0xfffc));
			}
#endif
		case 0x0c: /* 32-Bit Call Gate */
		case 0x0e: /* 32-Bit Interrupt Gate */
		case 0x0f: /* 32-Bit Trap Gate */
			if ((NAME_(seg_type)(cpssp, NAME_(SEG_CODE)) & 0x18) != 0x18) {
				/* No code segment. */
				fprintf(stderr, "WARNING: Call/Interrupt/Trap gate with non-code selector\n");
				NAME_(GPe)(cpssp, cpu, (selector & 0xfffc) | (isint << 1));
			}
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
			if (STATE_GET(lm_active_msr)
			 && !NAME_(seg_l)(cpssp, NAME_(SEG_CODE))) {
				fprintf(stderr, "WARNING: Gate with non-64bit code segment in long mode\n");
				NAME_(GPe)(cpssp, cpu, (selector & 0xfffc));
			}
#endif
			break;
		default:
			assert(0); /* Cannot happen. */
		}
		type = NAME_(seg_type)(cpssp, NAME_(SEG_CODE));
		cpl = NAME_(seg_dpl)(cpssp, NAME_(SEG_CODE));
		rpl = NAME_(seg_dpl)(cpssp, NAME_(SEG_CODE));

	} else {
		param = 0;
		int_enable = 1;
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
		istn = 0;
#endif
	}

	switch (type) {
	case 0x00: /* Reserved */
		fprintf(stderr, "WARNING: Calling reserved segment.\n");
		NAME_(GPe)(cpssp, cpu, (selector & 0xfffc) | (isint << 1));
		break;

	case 0x02: /* LDT */
		fprintf(stderr, "WARNING: Calling LDT segment.\n");
		NAME_(GPe)(cpssp, cpu, (selector & 0xfffc) | (isint << 1));
		break;

	case 0x03: /* 16-Bit TSS (busy) */
	case 0x0b: /* 32-Bit TSS (busy) */
		fprintf(stderr, "WARNING: Calling busy TSS segment.\n");
		NAME_(GPe)(cpssp, cpu, (selector & 0xfffc) | (isint << 1));
		break;

	case 0x04: /* 16-Bit Call Gate */
	case 0x05: /* Task Gate */
	case 0x06: /* 16-Bit Interrupt Gate */
	case 0x07: /* 16-Bit Trap Gate */
	case 0x0c: /* 32-Bit Call Gate */
	case 0x0d: /* Reserved */
	case 0x0e: /* 32-Bit Interrupt Gate */
	case 0x0f: /* 32-Bit Trap Gate */
		fprintf(stderr, "WARNING: Calling gate segment.\n");
		NAME_(GPe)(cpssp, cpu, (selector & 0xfffc) | (isint << 1));
		break;

	case 0x10: /* Data, Read-Only */
	case 0x11: /* Data, Read-Only, accessed */
	case 0x12: /* Data, Read/Write */
	case 0x13: /* Data, Read/Write, accessed */
	case 0x14: /* Data, Read-Only, expand-down */
	case 0x15: /* Data, Read-Only, expand-down, accessed */
	case 0x16: /* Data, Read/Write, expand-down */
	case 0x17: /* Data, Read/Write, expand-down, accessed */
		fprintf(stderr, "WARNING: Calling data segment.\n");
		NAME_(GPe)(cpssp, cpu, (selector & 0xfffc) | (isint << 1));
		break;

	case 0x01: /* 16-Bit TSS (available) */
	case 0x09: /* 32-Bit TSS (available) */
		/*
		 * Task State Segment
		 */
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
		if (STATE_GET(lm_active_msr)) {
			fprintf(stderr, "WARNING: Invalid selector in long mode\n");
			NAME_(GPe)(cpssp, cpu, (selector & 0xfffc));
		}
#endif
		NAME_(calltss)(cpssp, cpu, selector, offset, isint, iscall);
		break;

	case 0x18: /* Code, Execute-Only */
	case 0x19: /* Code, Execute-Only, accessed */
	case 0x1a: /* Code, Execute/Read-Only */
	case 0x1b: /* Code, Execute/Read-Only, accessed */
	case 0x1c: /* Code, Execute-Only, conforming */
	case 0x1d: /* Code, Execute-Only, conforming, accessed */
	case 0x1e: /* Code, Execute/Read-Only, conforming */
	case 0x1f: /* Code, Execute/Read-Only, conforming, accessed */
		/*
		 * Code Segment
		 */
		if (type & 0x04) {
			/* Conforming code segment. */
			if (cpl < NAME_(seg_dpl)(cpssp, NAME_(SEG_CODE))) {
				/* Protection fault. */
				NAME_(GPe)(cpssp, cpu, (selector & 0xfffc) | (isint << 1));
			}
		} else {
			/* Non-conforming code segment. */
			if (cpl < rpl
			 || cpl != NAME_(seg_dpl)(cpssp, NAME_(SEG_CODE))) {
				/* Protection fault. */
				NAME_(GPe)(cpssp, cpu, (selector & 0xfffc) | (isint << 1));
			}
		}
		if (! NAME_(seg_present)(cpssp, NAME_(SEG_CODE))) {
			/* Not present. */
			fprintf(stderr, "WARNING: Code segment not present.\n");
			NAME_(NPe)(cpssp, cpu, (selector & 0xfffc) | (isint << 1));
		}
		if (1 == op_mode) {
			/* Clear upper 16 bits */
			offset &= 0x0000ffff;
		}
		if (NAME_(seg_limit)(cpssp, NAME_(SEG_CODE)) < offset) {
			/* Dest addr outside segment limits */
			NAME_(GPe)(cpssp, cpu, 0);
		}
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
		if (STATE_GET(lm_active_msr)) {
			/* Long Mode active */
			if (NAME_(seg_l)(cpssp, NAME_(SEG_CODE)) & NAME_(seg_db)(cpssp, NAME_(SEG_CODE))) {
				/* L-Bit and D-Bit both set */
				fprintf(stderr, "WARNING: Code segment with L-Bit and D-Bit set in long mode\n");
				NAME_(GPe)(cpssp, cpu, (selector & 0xfffc));
			}
			if (!NAME_(check_eip_canonical)(cpssp, cpu, offset)) {
				/* Dest addr not canonical */
				NAME_(GPe)(cpssp, cpu, 0);
			}
		}
#endif
		NAME_(callseg)(cpssp, cpu, op_mode, selector, offset, isint, iscall,
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
				istn,
#endif
				param);
		break;

	default:
		assert(0); /* Mustn't happen. */
	}

	if (isint) {
		STATE_SET(e_tf, 0);
		STATE_SET(e_tf_delayed, 0);
		STATE_SET(e_if, int_enable & STATE_GET(e_if));
		STATE_SET(e_vm, 0);
#if 80486 <= CONFIG_CPU
		if (! STATE_GET(cr0_pe)) {
			STATE_SET(e_ac, 0);
		}
#endif
	}
}

static void
NAME_(lretiret)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int op_mode, int isint, uint8_t params)
{
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
		if (STATE_GET(lm_active_msr)) {
			assert(0); /* TODO */
		}
#endif
	if (isint
	 && STATE_GET(cr0_pe)
	 && STATE_GET(e_nt)) {
		/* Task Return */
		NAME_(rettss)(cpssp, cpu, op_mode, isint, params);

	} else {
		/* Segment Return */
		NAME_(retseg)(cpssp, cpu, op_mode, isint, params);
	}
}

static void
NAME_(ljmp)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t offset, uint16_t selector, int op_mode)
{
	NAME_(ljmpcallint)(cpssp, cpu, offset, selector, op_mode, 0, 0, 0);
}

static void
NAME_(lcall)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t offset, uint16_t selector, int op_mode)
{
	NAME_(ljmpcallint)(cpssp, cpu, offset, selector, op_mode, 0, 1, 0);
}

static void
NAME_(intn)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t int_num, int ishw)
{
	NAME_(ljmpcallint)(cpssp, cpu, 0, int_num << 3, 0, 1, 1, ishw);
}

static void
NAME_(lretn)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint32_t val, int op_mode)
{
	NAME_(lretiret)(cpssp, cpu, op_mode, 0, val);
}

static void
NAME_(iret)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int op_mode)
{
	NAME_(lretiret)(cpssp, cpu, op_mode, 1, 0);
}

static void
NAME_(smm)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	uint32_t cr0;
	uint32_t cr3;
	uint32_t base;
	uint32_t val;
	seg_descr_t descr;

	cpssp->NAME.smm = 1;

	/* Flush all callbacks. */
	DEFAULT_NAME_(cache2_unmap)(cpssp, 0x00000000, 0x80000000);
	DEFAULT_NAME_(cache2_unmap)(cpssp, 0x80000000, 0x80000000);

	/*
	 * Setup data descriptor.
	 */
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_DATA), 0x13, 0x00000000, 0xffffffff, 1, 0);

	/*
	 * Switch off paging first.
	 */
	cr0 = NAME_(get_cr)(cpssp, cpu, 0);
	STATE_SET(cr0_pe, 0);
	STATE_SET(cr0_em, 0);
	STATE_SET(cr0_ts, 0);
	STATE_SET(cr0_pg, 0);
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	/*
	 * Switching of Long Mode, because it's only active when enabled and paging is turned on.
	 * FIXME 64 bit mode smm
	 * TODO set lme = 0 ???
	 */
	STATE_SET(lm_active_msr, 0);
	fprintf(stderr, "Long Mode not active\n");
#endif

	cr3 = NAME_(get_cr)(cpssp, cpu, 3);
	NAME_(set_cr)(cpssp, cpu, 3, 0);

	/*
	 * Write current CPU state to SMBASE + 0x8000 + offset.
	 */
	base = cpssp->NAME.smbase + 0x8000;

	/* 0x7ffc: %cr0 Register */
	val = cr0;
	NAME_(smw32)(cpssp, cpu, base + 0x7ffc, NAME_(SEG_DATA), val, 0);

	/* 0x7ff8: %cr3 Register */
	val = cr3;
	NAME_(smw32)(cpssp, cpu, base + 0x7ff8, NAME_(SEG_DATA), val, 0);

	/* 0x7ff4: %eflags Register */
	val = NAME_(get_eflags)(cpssp, cpu);
	NAME_(smw32)(cpssp, cpu, base + 0x7ff4, NAME_(SEG_DATA), val, 0);

	/* 0x7ff0: %eip Register */
	val = STATE_GET(eip);
	NAME_(smw32)(cpssp, cpu, base + 0x7ff0, NAME_(SEG_DATA), val, 0);

	/* 0x7fd0-0x7fec: General Purpose Registers */
	val = STATE_GET(gp_regs[NAME_(E_DI)]);
	NAME_(smw32)(cpssp, cpu, base + 0x7fec, NAME_(SEG_DATA), val, 0);
	val = STATE_GET(gp_regs[NAME_(E_SI)]);
	NAME_(smw32)(cpssp, cpu, base + 0x7fe8, NAME_(SEG_DATA), val, 0);
	val = STATE_GET(gp_regs[NAME_(E_BP)]);
	NAME_(smw32)(cpssp, cpu, base + 0x7fe4, NAME_(SEG_DATA), val, 0);
	val = STATE_GET(gp_regs[NAME_(E_SP)]);
	NAME_(smw32)(cpssp, cpu, base + 0x7fe0, NAME_(SEG_DATA), val, 0);
	val = STATE_GET(gp_regs[NAME_(E_BX)]);
	NAME_(smw32)(cpssp, cpu, base + 0x7fdc, NAME_(SEG_DATA), val, 0);
	val = STATE_GET(gp_regs[NAME_(E_DX)]);
	NAME_(smw32)(cpssp, cpu, base + 0x7fd8, NAME_(SEG_DATA), val, 0);
	val = STATE_GET(gp_regs[NAME_(E_CX)]);
	NAME_(smw32)(cpssp, cpu, base + 0x7fd4, NAME_(SEG_DATA), val, 0);
	val = STATE_GET(gp_regs[NAME_(E_AX)]);
	NAME_(smw32)(cpssp, cpu, base + 0x7fd0, NAME_(SEG_DATA), val, 0);

	/* 0x7fcc: %dr6 */
	val = NAME_(get_dr)(cpssp, cpu, 6);
	NAME_(smw32)(cpssp, cpu, base + 0x7fcc, NAME_(SEG_DATA), val, 0);
	/* 0x7fc8: %dr7 */
	val = NAME_(get_dr)(cpssp, cpu, 7);
	NAME_(smw32)(cpssp, cpu, base + 0x7fc8, NAME_(SEG_DATA), val, 0);

	/* 0x7fc4: %tr Selector */
	val = STATE_GET(selector[NAME_(SEG_TR)]);
	NAME_(smw32)(cpssp, cpu, base + 0x7fc4, NAME_(SEG_DATA), val, 0);

	/* 0x7fc0: Reserved */

	/* 0x7fa8-0x7fbc: Selectors */
	val = STATE_GET(selector[NAME_(SEG_GS)]);
	NAME_(smw32)(cpssp, cpu, base + 0x7fbc, NAME_(SEG_DATA), val, 0);
	val = STATE_GET(selector[NAME_(SEG_FS)]);
	NAME_(smw32)(cpssp, cpu, base + 0x7fb8, NAME_(SEG_DATA), val, 0);
	val = STATE_GET(selector[NAME_(SEG_DS)]);
	NAME_(smw32)(cpssp, cpu, base + 0x7fb4, NAME_(SEG_DATA), val, 0);
	val = STATE_GET(selector[NAME_(SEG_SS)]);
	NAME_(smw32)(cpssp, cpu, base + 0x7fb0, NAME_(SEG_DATA), val, 0);
	val = STATE_GET(selector[NAME_(SEG_CS)]);
	NAME_(smw32)(cpssp, cpu, base + 0x7fac, NAME_(SEG_DATA), val, 0);
	val = STATE_GET(selector[NAME_(SEG_ES)]);
	NAME_(smw32)(cpssp, cpu, base + 0x7fa8, NAME_(SEG_DATA), val, 0);

	/* 0x7f04-0x7fa4: Reserved */

	/* 0x7f00: Auto Halt / I/O Restart */
	/* FIXME */

	/* 0x7efc: SMM Revision Identifier */
	val = (1 << 17) /* SMBASE relocation supported. */
		| (0 << 16) /* I/O Restart not supported. */
		| (1 << 0); /* SMM Release 1 */
	NAME_(smw32)(cpssp, cpu, base + 0x7efc, NAME_(SEG_DATA), val, 0);

	/* 0x7ef8: smbase */
	val = cpssp->NAME.smbase;
	NAME_(smw32)(cpssp, cpu, base + 0x7ef8, NAME_(SEG_DATA), val, 0);

	/* 0x7ef4-0x7e00: Reserved */
	/*
	 * We'll store other internals here:
	 * The following registers are saved (but not readable) and
	 * restored upon exiting SMM:
	 * - Control register CR4. (This register is cleared to all 0s
	 *     while in SMM).
	 * - The hidden segment descriptor information stored in segment
	 *     registers CS, DS, ES, FS, GS, and SS.
	 */
#if 80586 <= CONFIG_CPU
	/* %cr4 Register */
	val = NAME_(get_cr)(cpssp, cpu, 4);
	NAME_(smw32)(cpssp, cpu, base + 0x7ef4, NAME_(SEG_DATA), val, 0);
#endif

	/* Bases */
	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_CS));
	val = descr.entry[0] >> 32;
	NAME_(smw32)(cpssp, cpu, base + 0x7ef0, NAME_(SEG_DATA), val, 0);
	val = descr.entry[0] >> 0;
	NAME_(smw32)(cpssp, cpu, base + 0x7ed8, NAME_(SEG_DATA), val, 0);

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_DS));
	val = descr.entry[0] >> 32;
	NAME_(smw32)(cpssp, cpu, base + 0x7eec, NAME_(SEG_DATA), val, 0);
	val = descr.entry[0] >> 0;
	NAME_(smw32)(cpssp, cpu, base + 0x7ed4, NAME_(SEG_DATA), val, 0);

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_ES));
	val = descr.entry[0] >> 32;
	NAME_(smw32)(cpssp, cpu, base + 0x7ee8, NAME_(SEG_DATA), val, 0);
	val = descr.entry[0] >> 0;
	NAME_(smw32)(cpssp, cpu, base + 0x7ed0, NAME_(SEG_DATA), val, 0);

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_FS));
	val = descr.entry[0] >> 32;
	NAME_(smw32)(cpssp, cpu, base + 0x7ee4, NAME_(SEG_DATA), val, 0);
	val = descr.entry[0] >> 0;
	NAME_(smw32)(cpssp, cpu, base + 0x7ecc, NAME_(SEG_DATA), val, 0);

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_GS));
	val = descr.entry[0] >> 32;
	NAME_(smw32)(cpssp, cpu, base + 0x7ee0, NAME_(SEG_DATA), val, 0);
	val = descr.entry[0] >> 0;
	NAME_(smw32)(cpssp, cpu, base + 0x7ec8, NAME_(SEG_DATA), val, 0);

	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_SS));
	val = descr.entry[0] >> 32;
	NAME_(smw32)(cpssp, cpu, base + 0x7edc, NAME_(SEG_DATA), val, 0);
	val = descr.entry[0] >> 0;
	NAME_(smw32)(cpssp, cpu, base + 0x7ec4, NAME_(SEG_DATA), val, 0);

	/* %tr Register */
	descr = DEFAULT_NAME_(seg_descr_get)(cpssp, NAME_(SEG_TR));
	val = descr.entry[0] >> 32;
	NAME_(smw32)(cpssp, cpu, base + 0x7ea8, NAME_(SEG_DATA), val, 0);
	val = descr.entry[0] >> 0;
	NAME_(smw32)(cpssp, cpu, base + 0x7ea4, NAME_(SEG_DATA), val, 0);

	/*
	 * Set up SMM environment.
	 */
	/* General Purpose Registers */
	/* Undefined */

	/* %eflags */
	NAME_(set_eflags_all)(cpssp, cpu, 0x00000002, 1 + 1);

	/* Segments */
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_CS), cpssp->NAME.smbase >> 4);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_CS), 0x1b, cpssp->NAME.smbase, 0xffffffff, 0, 0);
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_SS), 0x0000);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_SS), 0x13, 0x00000000, 0xffffffff, 0, 0);
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_DS), 0x0000);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_DS), 0x13, 0x00000000, 0xffffffff, 0, 0);
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_ES), 0x0000);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_ES), 0x13, 0x00000000, 0xffffffff, 0, 0);
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_FS), 0x0000);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_FS), 0x13, 0x00000000, 0xffffffff, 0, 0);
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_GS), 0x0000);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_GS), 0x13, 0x00000000, 0xffffffff, 0, 0);

	/* %cr0 set above. */

#if 80586 <= CONFIG_CPU
	/* %cr4 */
	NAME_(set_cr)(cpssp, cpu, 4, 0x00000000);
#endif

	/* %dr6, %dr7 */
	NAME_(set_dr)(cpssp, cpu, 6, 0); /* Undefined */
	NAME_(set_dr)(cpssp, cpu, 7, 0x00000400);

	/* %eip */
	cpu->eip = 0x8000;
}

static void
NAME_(rsm)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	uint32_t e0, e1;
	seg_descr_t descr;
	uint32_t base;
	uint32_t val;

	/*
	 * Setup data descriptor.
	 */
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_DATA), 0x13, 0x00000000, 0xffffffff, 1, 0);

	base = cpssp->NAME.smbase + 0x8000;

	/* %tr Register */
	e0 = NAME_(smr32)(cpssp, cpu, base + 0x7ea4, NAME_(SEG_DATA), 0, 0);
	e1 = NAME_(smr32)(cpssp, cpu, base + 0x7ea8, NAME_(SEG_DATA), 0, 0);
	descr.entry[0] = ((uint64_t) e1 << 32) | ((uint64_t) e0 << 0);
	DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_TR), descr);

	e0 = NAME_(smr32)(cpssp, cpu, base + 0x7ec4, NAME_(SEG_DATA), 0, 0);
	e1 = NAME_(smr32)(cpssp, cpu, base + 0x7edc, NAME_(SEG_DATA), 0, 0);
	descr.entry[0] = ((uint64_t) e1 << 32) | ((uint64_t) e0 << 0);
	DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_SS), descr);

	e0 = NAME_(smr32)(cpssp, cpu, base + 0x7ec8, NAME_(SEG_DATA), 0, 0);
	e1 = NAME_(smr32)(cpssp, cpu, base + 0x7ee0, NAME_(SEG_DATA), 0, 0);
	descr.entry[0] = ((uint64_t) e1 << 32) | ((uint64_t) e0 << 0);
	DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_GS), descr);

	e0 = NAME_(smr32)(cpssp, cpu, base + 0x7ecc, NAME_(SEG_DATA), 0, 0);
	e1 = NAME_(smr32)(cpssp, cpu, base + 0x7ee4, NAME_(SEG_DATA), 0, 0);
	descr.entry[0] = ((uint64_t) e1 << 32) | ((uint64_t) e0 << 0);
	DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_FS), descr);

	e0 = NAME_(smr32)(cpssp, cpu, base + 0x7ed0, NAME_(SEG_DATA), 0, 0);
	e1 = NAME_(smr32)(cpssp, cpu, base + 0x7ee8, NAME_(SEG_DATA), 0, 0);
	descr.entry[0] = ((uint64_t) e1 << 32) | ((uint64_t) e0 << 0);
	DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_ES), descr);

	e0 = NAME_(smr32)(cpssp, cpu, base + 0x7ed4, NAME_(SEG_DATA), 0, 0);
	e1 = NAME_(smr32)(cpssp, cpu, base + 0x7eec, NAME_(SEG_DATA), 0, 0);
	descr.entry[0] = ((uint64_t) e1 << 32) | ((uint64_t) e0 << 0);
	DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_DS), descr);

	e0 = NAME_(smr32)(cpssp, cpu, base + 0x7ed8, NAME_(SEG_DATA), 0, 0);
	e1 = NAME_(smr32)(cpssp, cpu, base + 0x7ef0, NAME_(SEG_DATA), 0, 0);
	descr.entry[0] = ((uint64_t) e1 << 32) | ((uint64_t) e0 << 0);
	DEFAULT_NAME_(seg_descr_set)(cpssp, NAME_(SEG_CS), descr);

#if 80586 <= CONFIG_CPU
	/* %cr4 Register */
	val = NAME_(smr32)(cpssp, cpu, base + 0x7ef4, NAME_(SEG_DATA), 0, 0);
	NAME_(set_cr)(cpssp, cpu, 4, val);
#endif

	/* 0x7ef8: smbase */
	val = NAME_(smr32)(cpssp, cpu, base + 0x7ef8, NAME_(SEG_DATA), 0, 0);
	cpssp->NAME.smbase = val;

	/* 0x7efc: SMM Revision Identifier */
	/* Write-only */

	/* 0x7f00: Auto Halt / I/O Restart */
	/* FIXME */

	/* 0x7f04-0x7fa4: Reserved */

	/* 0x7fa8-0x7fbc: Selectors */
	val = NAME_(smr32)(cpssp, cpu, base + 0x7fa8, NAME_(SEG_DATA), 0, 0);
	STATE_SET(selector[NAME_(SEG_ES)], val);
	val = NAME_(smr32)(cpssp, cpu, base + 0x7fac, NAME_(SEG_DATA), 0, 0);
	STATE_SET(selector[NAME_(SEG_CS)], val);
	val = NAME_(smr32)(cpssp, cpu, base + 0x7fb0, NAME_(SEG_DATA), 0, 0);
	STATE_SET(selector[NAME_(SEG_SS)], val);
	val = NAME_(smr32)(cpssp, cpu, base + 0x7fb4, NAME_(SEG_DATA), 0, 0);
	STATE_SET(selector[NAME_(SEG_DS)], val);
	val = NAME_(smr32)(cpssp, cpu, base + 0x7fb8, NAME_(SEG_DATA), 0, 0);
	STATE_SET(selector[NAME_(SEG_FS)], val);
	val = NAME_(smr32)(cpssp, cpu, base + 0x7fbc, NAME_(SEG_DATA), 0, 0);
	STATE_SET(selector[NAME_(SEG_GS)], val);

	/* 0x7fc0: Reserved */

	/* 0x7fc4: %tr Selector */
	val = NAME_(smr32)(cpssp, cpu, base + 0x7fc4, NAME_(SEG_DATA), 0, 0);
	STATE_SET(selector[NAME_(SEG_TR)], val);

	/* 0x7fc8: %dr7 */
	val = NAME_(smr32)(cpssp, cpu, base + 0x7fc8, NAME_(SEG_DATA), 0, 0);
	NAME_(set_dr)(cpssp, cpu, 7, val);
	/* 0x7fcc: %dr6 */
	val = NAME_(smr32)(cpssp, cpu, base + 0x7fcc, NAME_(SEG_DATA), 0, 0);
	NAME_(set_dr)(cpssp, cpu, 6, val);

	/* 0x7fd0-0x7fec: General Purpose Registers */
	val = NAME_(smr32)(cpssp, cpu, base + 0x7fd0, NAME_(SEG_DATA), 0, 0);
	STATE_SET(gp_regs[NAME_(E_AX)], val);
	val = NAME_(smr32)(cpssp, cpu, base + 0x7fd4, NAME_(SEG_DATA), 0, 0);
	STATE_SET(gp_regs[NAME_(E_CX)], val);
	val = NAME_(smr32)(cpssp, cpu, base + 0x7fd8, NAME_(SEG_DATA), 0, 0);
	STATE_SET(gp_regs[NAME_(E_DX)], val);
	val = NAME_(smr32)(cpssp, cpu, base + 0x7fdc, NAME_(SEG_DATA), 0, 0);
	STATE_SET(gp_regs[NAME_(E_BX)], val);
	val = NAME_(smr32)(cpssp, cpu, base + 0x7fe0, NAME_(SEG_DATA), 0, 0);
	STATE_SET(gp_regs[NAME_(E_SP)], val);
	val = NAME_(smr32)(cpssp, cpu, base + 0x7fe4, NAME_(SEG_DATA), 0, 0);
	STATE_SET(gp_regs[NAME_(E_BP)], val);
	val = NAME_(smr32)(cpssp, cpu, base + 0x7fe8, NAME_(SEG_DATA), 0, 0);
	STATE_SET(gp_regs[NAME_(E_SI)], val);
	val = NAME_(smr32)(cpssp, cpu, base + 0x7fec, NAME_(SEG_DATA), 0, 0);
	STATE_SET(gp_regs[NAME_(E_DI)], val);

	/* 0x7ff0: %eip Register */
	val = NAME_(smr32)(cpssp, cpu, base + 0x7ff0, NAME_(SEG_DATA), 0, 0);
	cpu->eip = val;

	/* 0x7ff4: %eflags */
	val = NAME_(smr32)(cpssp, cpu, base + 0x7ff4, NAME_(SEG_DATA), 0, 0);
	NAME_(set_eflags_all)(cpssp, cpu, val, 1 + 1);

	/* 0x7ff8: %cr3 */
	val = NAME_(smr32)(cpssp, cpu, base + 0x7ff8, NAME_(SEG_DATA), 0, 0);
	NAME_(set_cr)(cpssp, cpu, 3, val);

	/* 0x7ffc: %cr0 */
	val = NAME_(smr32)(cpssp, cpu, base + 0x7ffc, NAME_(SEG_DATA), 0, 0);
	NAME_(set_cr)(cpssp, cpu, 0, val);

	cpssp->NAME.smm = 0;

	/* Flush all callbacks. */
	DEFAULT_NAME_(cache2_unmap)(cpssp, 0x00000000, 0x80000000);
	DEFAULT_NAME_(cache2_unmap)(cpssp, 0x80000000, 0x80000000);
}

static udata_t
NAME_(lea)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	return cpu->addr;
}

static void
NAME_(cli_sti)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int val)
{
	STATE_SET(e_if, val);
	cpssp->NAME.int_delay |= val;
	cpssp->NAME.state_event |= val;
}

static void
NAME_(cmc)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	STATE_SET(e_cf, STATE_GET(e_cf) ^ 1);
}

static void
NAME_(clc_stc)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int val)
{
	STATE_SET(e_cf, val);
}

static void
NAME_(cld_std)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int val)
{
	NAME_(change_df)(cpssp, cpu, val);
}

static void
NAME_(hlt)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	cpssp->NAME.state_event = 1;
	cpssp->NAME.state_halt = 1;
}



static uint8_t
NAME_(xlat)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t addr_mode)
{
	uint8_t ret;
	udata_t addr;

	addr = NAME_(load_reg)(cpssp, cpu, NAME_(E_BX), addr_mode);
	addr += NAME_(load_reg8)(cpssp, cpu, NAME_(AL));
	addr = NAME_(fix_size)(addr, addr_mode); // FIXME is this correct?

	ret = NAME_(smr8)(cpssp, cpu, addr, cpu->seg[0], 0, NAME_(usermode)(cpssp, cpu));

	return ret;
}

static udata_t
NAME_(alu_bt)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, udata_t op2, uint8_t operation, uint8_t op_mode)
{
	uint8_t s;
	udata_t res;
	udata_t tmp;

	s = op2 & ((0x8 << op_mode) - 1);
	tmp = op1 >> s;

	switch (operation) {
	case 0x0:	/* bt */
		res = op1;
		break;
	case 0x1:	/* bts */
		res = op1 | (0x1 << s);
		break;
	case 0x2:	/* btr */
		res = op1 & ~(0x1 << s);
		break;
	case 0x3:	/* btc */
		res = op1 ^ (0x1 << s);
		break;
	default:
		assert(0); /* Mustn't happen. */
		break;
	}

	STATE_SET(e_cf, tmp & 0x1);

	/* undefined */
	STATE_SET(e_af, 0);
	STATE_SET(e_of, 0);
	STATE_SET(e_pf, 1);
	STATE_SET(e_sf, 0);
	STATE_SET(e_zf, 1);

	return res;
}

static uint32_t
NAME_(bsf)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1)
{
	uint32_t res;

	res = 0;
	if (! op1) {
		STATE_SET(e_zf, 1);
		STATE_SET(e_pf, 1); /* Undefined */

	} else {
		while (1) {
			if (op1 & (0x1 << res)) break;
			res++;
		}
		STATE_SET(e_zf, 0);
		STATE_SET(e_pf, 0); /* Undefined */
	}
	STATE_SET(e_af, 0); /* Undefined */
	STATE_SET(e_cf, 0); /* Undefined */
	STATE_SET(e_of, 0); /* Undefined */
	STATE_SET(e_sf, 0); /* Undefined */

	return res;
}


static uint32_t
NAME_(bsr)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1)
{
	uint32_t res;

	res = 31;
	if (! op1) {
		STATE_SET(e_zf, 1);
		STATE_SET(e_pf, 1); /* Undefined */

	} else {
		while (1) {
			if (op1 & (0x1 << res)) break;
			res--;
		}
		STATE_SET(e_zf, 0);
		STATE_SET(e_pf, 0); /* Undefined */
	}
	STATE_SET(e_af, 0); /* Undefined */
	STATE_SET(e_cf, 0); /* Undefined */
	STATE_SET(e_of, 0); /* Undefined */
	STATE_SET(e_sf, 0); /* Undefined */

	return res;
}


static void
NAME_(bound)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int32_t op1, int32_t op2, int32_t op3)
{
	if (op1 < op2 || op3 < op1) {
		NAME_(BR)(cpssp, cpu);
	}
}


static uint16_t
NAME_(arpl)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t dest, uint16_t src)
{
	uint16_t res;

	if ((dest & 0x3) < (src & 0x3)) {
		res = (dest & ~0x3) | (src & 0x3);
		STATE_SET(e_zf, 1);
	} else {
		res = 0; /* Not used. */
		STATE_SET(e_zf, 0);
	}

	return res;
}


static uint32_t
NAME_(lar)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, uint8_t op_mode)
{
	int err;
	uint16_t errc;
	uint32_t res;

	err = NAME_(read_entry)(cpssp, cpu, NAME_(SEG_DATA), op1, 0, 1, 0, 0, &errc);
	if (err != NAME_(EXC_NONE)) {
		goto fail;
	}
	switch (NAME_(seg_type)(cpssp, NAME_(SEG_DATA))) {
	case 0x00: /* Reserved */
	case 0x06: /* 16-bit Interrupt Gate */
	case 0x07: /* 16-bit Trap Gate */
	case 0x08: /* Reserved */
	case 0x0a: /* Reserved */
	case 0x0d: /* Reserved */
	case 0x0e: /* 32-bit Interrupt Gate */
	case 0x0f: /* 32-bit Trap Gate */
		goto fail;
	default:
		break;
	}
#if 0 /* FIXME */
	if (! (entry.flags & ...) /* Not conforming. */
	 && (... < ... /* DPL < CPL */
	  || ... < ...) /* DPL < RPL */)
#else
	if (0)
#endif
	{
	fail:	;
		STATE_SET(e_zf, 0);
		return 0;
	}

	res = NAME_(seg_ar)(cpssp, NAME_(SEG_DATA));
	STATE_SET(e_zf, 1);

	return res;
}

static uint32_t
NAME_(lsl)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, udata_t op1, uint8_t op_mode)
{
	int err;
	uint16_t errc;
	uint32_t res;

	err = NAME_(read_entry)(cpssp, cpu, NAME_(SEG_DATA), op1, 0, 1, 0, 0, &errc);
	if (err != NAME_(EXC_NONE)) {
		goto fail;
	}
	switch (NAME_(seg_type)(cpssp, NAME_(SEG_DATA))) {
	case 0x00: /* Reserved */
	case 0x04: /* 16-bit Call Gate */
	case 0x05: /* Task Gate */
	case 0x06: /* 16-bit Interrupt Gate */
	case 0x07: /* 16-bit Trap Gate */
	case 0x08: /* Reserved */
	case 0x0a: /* Reserved */
	case 0x0c: /* 16-bit Call Gate */
	case 0x0d: /* Reserved */
	case 0x0e: /* 32-bit Interrupt Gate */
	case 0x0f: /* 32-bit Trap Gate */
		goto fail;
	default:
		break;
	}
#if 0 /* FIXME */
	if (! (entry.flags & ...) /* Not conforming. */
	 && (... < ... /* DPL < CPL */
	  || ... < ...) /* DPL < RPL */)
#else
	if (0)
#endif
	{
	fail:	;
		STATE_SET(e_zf, 0);
		return 0;
	}

	res = NAME_(seg_limit)(cpssp, NAME_(SEG_DATA));
	STATE_SET(e_zf, 1);

	return res;
}

static void
NAME_(verr)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t sel)
{
	fprintf(stderr, "WARNING: verr not implemented!\n");
	STATE_SET(e_zf, 1); /* FIXME */
}

static void
NAME_(verw)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t sel)
{
	fprintf(stderr, "WARNING: verw not implemented!\n");
	STATE_SET(e_zf, 1); /* FIXME */
}

/* ################ DEBUG ################################################## */

#if DEBUG_CONTROL_FLOW
//static inline __attribute__((__always_inline__)) void
static void
NAME_(dump)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
#if DEBUG_CONTROL_FLOW_REGS
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	if (STATE_GET(lm_active_msr)) {
		fprintf(stderr, " %016llx %016llx %016llx %016llx %016llx %016llx %016llx %016llx %016llx %016llx %016llx %016llx %016llx %016llx %016llx %016llx",
				(uint64_t) NAME_(load_reg64)(cpssp, cpu, NAME_(E_AX)),
				(uint64_t) NAME_(load_reg64)(cpssp, cpu, NAME_(E_BX)),
				(uint64_t) NAME_(load_reg64)(cpssp, cpu, NAME_(E_CX)),
				(uint64_t) NAME_(load_reg64)(cpssp, cpu, NAME_(E_DX)),
				(uint64_t) NAME_(load_reg64)(cpssp, cpu, NAME_(E_BP)),
				(uint64_t) NAME_(load_reg64)(cpssp, cpu, NAME_(E_SP)),
				(uint64_t) NAME_(load_reg64)(cpssp, cpu, NAME_(E_DI)),
				(uint64_t) NAME_(load_reg64)(cpssp, cpu, NAME_(E_SI)),
				(uint64_t) NAME_(load_reg64)(cpssp, cpu, NAME_(R8)),
				(uint64_t) NAME_(load_reg64)(cpssp, cpu, NAME_(R9)),
				(uint64_t) NAME_(load_reg64)(cpssp, cpu, NAME_(R10)),
				(uint64_t) NAME_(load_reg64)(cpssp, cpu, NAME_(R11)),
				(uint64_t) NAME_(load_reg64)(cpssp, cpu, NAME_(R12)),
				(uint64_t) NAME_(load_reg64)(cpssp, cpu, NAME_(R13)),
				(uint64_t) NAME_(load_reg64)(cpssp, cpu, NAME_(R14)),
				(uint64_t) NAME_(load_reg64)(cpssp, cpu, NAME_(R15)));
	} else
#endif
	{
		fprintf(stderr, " %08lx %08lx %08lx %08lx %08lx %08lx %08lx %08lx",
				(uint32_t) NAME_(load_reg32)(cpssp, cpu, NAME_(E_AX)),
				(uint32_t) NAME_(load_reg32)(cpssp, cpu, NAME_(E_BX)),
				(uint32_t) NAME_(load_reg32)(cpssp, cpu, NAME_(E_CX)),
				(uint32_t) NAME_(load_reg32)(cpssp, cpu, NAME_(E_DX)),
				(uint32_t) NAME_(load_reg32)(cpssp, cpu, NAME_(E_BP)),
				(uint32_t) NAME_(load_reg32)(cpssp, cpu, NAME_(E_SP)),
				(uint32_t) NAME_(load_reg32)(cpssp, cpu, NAME_(E_DI)),
				(uint32_t) NAME_(load_reg32)(cpssp, cpu, NAME_(E_SI)));
	}
#endif
#if DEBUG_CONTROL_FLOW_CREGS
		fprintf(stderr, " %08x %08x %08x %08x",
				(uint32_t) NAME_(get_cr)(cpssp, cpu, 0),
				(uint32_t) NAME_(get_cr)(cpssp, cpu, 2),
				(uint32_t) NAME_(get_cr)(cpssp, cpu, 3),
				(uint32_t) NAME_(get_cr)(cpssp, cpu, 4));
#if 0 // defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
		fprintf(stderr, " %08x",
				(uint32_t) NAME_(get_cr)(cpssp, cpu, 8));
#endif
#endif
#if DEBUG_CONTROL_FLOW_SREGS
		fprintf(stderr, " %04x %04x %04x %04x %04x %04x",
				STATE_GET(selector[NAME_(SEG_CS)]),
				STATE_GET(selector[NAME_(SEG_SS)]),
				STATE_GET(selector[NAME_(SEG_DS)]),
				STATE_GET(selector[NAME_(SEG_ES)]),
				STATE_GET(selector[NAME_(SEG_FS)]),
				STATE_GET(selector[NAME_(SEG_GS)]));
		fprintf(stderr, " %08x %08x %08x %08x %08x %08x",
				NAME_(seg_base)(cpssp, NAME_(SEG_CS)),
				NAME_(seg_base)(cpssp, NAME_(SEG_SS)),
				NAME_(seg_base)(cpssp, NAME_(SEG_DS)),
				NAME_(seg_base)(cpssp, NAME_(SEG_ES)),
				NAME_(seg_base)(cpssp, NAME_(SEG_FS)),
				NAME_(seg_base)(cpssp, NAME_(SEG_GS)));
#endif
#if DEBUG_CONTROL_FLOW_FLAGS
		fprintf(stderr, " %08lx", NAME_(get_eflags)(cpssp, cpu));
#endif
#if CONFIG_FPU
#if 0
		int i;
		for (i = 0; i < 8; i++) {
			if (cpssp->fpu.used[i]) {
				fprintf(stderr, " %Lf", cpssp->fpu.st[i]);
			} else {
				fprintf(stderr, " -");
			}
		}
		fprintf(stderr, " %d%d%d%d %d",
				cpssp->fpu.c3, cpssp->fpu.c2,
				cpssp->fpu.c1, cpssp->fpu.c0,
				cpssp->fpu.top);
#endif
#endif
		fprintf(stderr, "\n");
}
#endif


/* ################ MODRM ################################################## */

static void
NAME_(get_modrm)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	uint8_t modrm;
	uint8_t scale;
	uint8_t index;
	uint8_t base;
	uint8_t seg[2];
	uint16_t r16_1;
	uint16_t r16_2;
	uint16_t disp16;
	uint32_t disp32;
	uint32_t sib;
	uint32_t base_val;
	uint32_t index_val[2];

	modrm = NAME_(fetch8)(cpssp, cpu);

	cpu->mod = (modrm >> 6) & 0x03;
	cpu->reg_opex = (modrm >> 3) & 0x07;
	cpu->rm = modrm & 0x07;

	if (likely(0x3 == cpu->mod)) {
		return;
	}

	switch (NAME_(get_addr_mode)(cpssp, cpu)) {
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	case 0x3:
		/*
		 * 64-bit Addressing Mode
		 */
		assert(0); // TODO
		break;
#endif /* defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT */

	case 0x2:
		/*
		 * 32-bit Addressing Mode
		 */
		seg[0] = NAME_(SEG_DS);
		if (0x4 == cpu->rm) {
			/* SIB-byte */
			sib = NAME_(fetch8)(cpssp, cpu);
			scale = (sib >> 6) & 0x03;
			index = (sib >> 3) & 0x07;
			base = sib & 0x07;

			index_val[0] = NAME_(load_reg32)(cpssp, cpu, index);
			index_val[(0x4 != index)] = 0;
		} else {
			scale = 0;
			index_val[0] = 0;
			base = cpu->rm;
		}

		base_val = NAME_(load_reg32)(cpssp, cpu, base);

		switch (cpu->mod) {
		case 0x0:
			if (0x5 == base) {
				base_val = 0;
				disp32 = NAME_(fetch32)(cpssp, cpu);
			} else {
				disp32 = 0;
				seg[(0x4 != base)] = NAME_(SEG_SS);
			}
			break;
		case 0x1:
			disp32 = (int32_t) (int8_t) NAME_(fetch8)(cpssp, cpu);
			seg[((0x4 != base) & (0x5 != base))] = NAME_(SEG_SS);
			break;
		case 0x2:
			disp32 = NAME_(fetch32)(cpssp, cpu);
			seg[((0x4 != base) & (0x5 != base))] = NAME_(SEG_SS);
			break;
		default:
			assert(0); /* Cannot happen. */
		}

		cpu->addr = disp32 + base_val + (index_val[0] << scale);
		cpu->seg[0] = seg[0];
		break;

	case 0x1:
		/*
		 * 16-bit Addressing Mode
		 */
		/* Get register contents and segment. */
		switch (cpu->rm) {
		case 0x0:
			r16_1 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_BX));
			r16_2 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_SI));
			cpu->seg[0] = NAME_(SEG_DS);
			break;
		case 0x1:
			r16_1 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_BX));
			r16_2 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_DI));
			cpu->seg[0] = NAME_(SEG_DS);
			break;
		case 0x2:
			r16_1 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_BP));
			r16_2 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_SI));
			cpu->seg[0] = NAME_(SEG_SS);
			break;
		case 0x3:
			r16_1 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_BP));
			r16_2 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_DI));
			cpu->seg[0] = NAME_(SEG_SS);
			break;
		case 0x4:
			r16_1 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_SI));
			r16_2 = 0;
			cpu->seg[0] = NAME_(SEG_DS);
			break;
		case 0x5:
			r16_1 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_DI));
			r16_2 = 0;
			cpu->seg[0] = NAME_(SEG_DS);
			break;
		case 0x6:
			if (cpu->mod == 0x0) {
				r16_1 = 0;
				r16_2 = 0;
				cpu->seg[0] = NAME_(SEG_DS);
			} else {
				r16_1 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_BP));
				r16_2 = 0;
				cpu->seg[0] = NAME_(SEG_SS);
			}
			break;
		case 0x7:
			r16_1 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_BX));
			r16_2 = 0;
			cpu->seg[0] = NAME_(SEG_DS);
			break;
		default:
			assert(0); /* Cannot happen. */
		}

		/* Get displacement. */
		switch (cpu->mod) {
		case 0x0:
			if (cpu->rm == 0x6) {
				disp16 = NAME_(fetch16)(cpssp, cpu);
			} else {
				disp16 = 0;
			}
			break;
		case 0x1:
			disp16 = (int16_t) (int8_t) NAME_(fetch8)(cpssp, cpu);
			break;
		case 0x2:
			disp16 = NAME_(fetch16)(cpssp, cpu);
			break;
		default:
			assert(0); /* Cannot happen. */
		}

		cpu->addr = (uint16_t) (disp16 + r16_1 + r16_2);
		break;

	default:
		/* Should not happen */
		assert(0);
	}

	NAME_(check_seg_over)(cpssp, cpu);
}

/* ################ CPUID ################################################## */

#if 80586 <= CONFIG_CPU
static void
NAME_(cpuid)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint32_t n, uint32_t *eaxp, uint32_t *ebxp, uint32_t *ecxp, uint32_t *edxp)
{
	uint32_t val;

#if (defined CONFIG_CPU_LAHF_SUPPORT && CONFIG_CPU_LAHF_SUPPORT) \
 || (defined CONFIG_CPU_SYSCALL_SUPPORT && CONFIG_CPU_SYSCALL_SUPPORT) \
 || (defined CONFIG_CPU_NX_SUPPORT && CONFIG_CPU_NX_SUPPORT) \
 || (defined CONFIG_CPU_FFXSR_SUPPORT && CONFIG_CPU_FFXSR_SUPPORT) \
 || (defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT)
	if (n < 0x80000000) {
		if (2 < n) {
			n = 2;
		}
	} else {
		if (0x80000001 < n) {
			n = 2;
		}
	}
#else
	if (2 < n) {
		n = 2;
	}
#endif

	switch (n) {
	case 0:
		/* Brand Info */
		*eaxp = 2;
		*ebxp = ('G' << 0) | ('e' << 8) | ('n' << 16) | ('u' << 24);
		*edxp = ('i' << 0) | ('n' << 8) | ('e' << 16) | ('I' << 24);
		*ecxp = ('n' << 0) | ('t' << 8) | ('e' << 16) | ('l' << 24);
		break;
	case 1:
		/* Version Info */
		*eaxp
#if CONFIG_CPU <= 80686
			= (((CONFIG_CPU / 100) % 10) << 8) /* Family */
#else
			= (0xf << 8) /* Family */
#endif
			| (0x3 << 4) /* Model - FIXME */
			| (0x4 << 0); /* Stepping - FIXME */

		/* Reserved */
		*ebxp = 0x00000000;

		/* Feature Info */
		/*
		 * %ecx
		 */
		val = 0;
#if defined CONFIG_CPU_SSE3_SUPPORT && CONFIG_CPU_SSE3_SUPPORT
		val |= 1 << 0;
#endif
		/* Bit 1-2: reserved */
#if defined CONFIG_CPU_MONITOR_SUPPORT && CONFIG_CPU_MONITOR_SUPPORT
		val |= 1 << 3;
#endif
#if defined CONFIG_CPU_DSCPL_SUPPORT && CONFIG_CPU_DSCPL_SUPPORT
		val |= 1 << 4;
#endif
#if defined CONFIG_CPU_VMX_SUPPORT && CONFIG_CPU_VMX_SUPPORT
		val |= 1 << 5;
#endif
		/* Bit 6: reserved */
#if defined CONFIG_CPU_EST_SUPPORT && CONFIG_CPU_EST_SUPPORT
		val |= 1 << 7;
#endif
#if defined CONFIG_CPU_TM2_SUPPORT && CONFIG_CPU_TM2_SUPPORT
		val |= 1 << 8;
#endif
#if defined CONFIG_CPU_SSSE3_SUPPORT && CONFIG_CPU_SSSE3_SUPPORT
		val |= 1 << 9;
#endif
#if defined CONFIG_CPU_CNXTID_SUPPORT && CONFIG_CPU_CNXTID_SUPPORT
		val |= 1 << 10;
#endif
		/* Bit 11-12: reserved */
#if defined CONFIG_CPU_CMPXCHG16B_SUPPORT && CONFIG_CPU_CMPXCHG16B_SUPPORT
		val |= 1 << 13;
#endif
#if defined CONFIG_CPU_XTPR_SUPPORT && CONFIG_CPU_XTPR_SUPPORT
		val |= 1 << 14;
#endif
#if defined CONFIG_CPU_PDCM_SUPPORT && CONFIG_CPU_PDCM_SUPPORT
		val |= 1 << 15;
#endif
		*ecxp = val;

		/*
		 * %edx
		 */
		val = 0;
#if defined CONFIG_CPU_FPU_SUPPORT && CONFIG_CPU_FPU_SUPPORT
		val |= 1 << 0;
#endif
#if defined CONFIG_CPU_VME_SUPPORT && CONFIG_CPU_VME_SUPPORT
		val |= 1 << 1;
#endif
#if defined CONFIG_CPU_DE_SUPPORT && CONFIG_CPU_DE_SUPPORT
		val |= 1 << 2;
#endif
#if defined CONFIG_CPU_PSE_SUPPORT && CONFIG_CPU_PSE_SUPPORT
		val |= 1 << 3;
#endif
#if defined CONFIG_CPU_TSC_SUPPORT && CONFIG_CPU_TSC_SUPPORT
		val |= 1 << 4;
#endif
#if defined CONFIG_CPU_MSR_SUPPORT && CONFIG_CPU_MSR_SUPPORT
		val |= 1 << 5;
#endif
#if defined CONFIG_CPU_PAE_SUPPORT && CONFIG_CPU_PAE_SUPPORT
		val |= 1 << 6;
#endif
#if defined CONFIG_CPU_MCE_SUPPORT && CONFIG_CPU_MCE_SUPPORT
		val |= 1 << 7;
#endif
#if defined CONFIG_CPU_CX8_SUPPORT && CONFIG_CPU_CX8_SUPPORT
		val |= 1 << 8;
#endif
#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
		val |= 1 << 9;
#endif
			/* Bit 10: reserved */
#if defined CONFIG_CPU_SEP_SUPPORT && CONFIG_CPU_SEP_SUPPORT
		val |= 1 << 11;
#endif
#if defined CONFIG_CPU_MTRR_SUPPORT && CONFIG_CPU_MTRR_SUPPORT
		val |= 1 << 12;
#endif
#if defined CONFIG_CPU_PGE_SUPPORT && CONFIG_CPU_PGE_SUPPORT
		val |= 1 << 13;
#endif
#if defined CONFIG_CPU_MCA_SUPPORT && CONFIG_CPU_MCA_SUPPORT
		val |= 1 << 14;
#endif
#if defined CONFIG_CPU_CMOV_SUPPORT && CONFIG_CPU_CMOV_SUPPORT
		val |= 1 << 15;
#endif
#if defined CONFIG_CPU_PAT_SUPPORT && CONFIG_CPU_PAT_SUPPORT
		val |= 1 << 16;
#endif
#if defined CONFIG_CPU_PSE36_SUPPORT && CONFIG_CPU_PSE36_SUPPORT
		val |= 1 << 17;
#endif
#if defined CONFIG_CPU_PSN_SUPPORT && CONFIG_CPU_PSN_SUPPORT
		val |= 1 << 18;
#endif
#if defined CONFIG_CPU_CFLSH_SUPPORT && CONFIG_CPU_CFLSH_SUPPORT
		val |= 1 << 19;
#endif
			/* Bit 20: reserved */
#if defined CONFIG_CPU_DS_SUPPORT && CONFIG_CPU_DS_SUPPORT
		val |= 1 << 21;
#endif
#if defined CONFIG_CPU_ACPI_SUPPORT && CONFIG_CPU_ACPI_SUPPORT
		val |= 1 << 22;
#endif
#if defined CONFIG_CPU_MMX_SUPPORT && CONFIG_CPU_MMX_SUPPORT
		val |= 1 << 23;
#endif
#if defined CONFIG_CPU_FXSR_SUPPORT && CONFIG_CPU_FXSR_SUPPORT
		val |= 1 << 24;
#endif
#if defined CONFIG_CPU_SSE_SUPPORT && CONFIG_CPU_SSE_SUPPORT
		val |= 1 << 25;
#endif
#if defined CONFIG_CPU_SSE2_SUPPORT && CONFIG_CPU_SSE2_SUPPORT
		val |= 1 << 26;
#endif
#if defined CONFIG_CPU_SS_SUPPORT && CONFIG_CPU_SS_SUPPORT
		val |= 1 << 27;
#endif
#if defined CONFIG_CPU_HTT_SUPPORT && CONFIG_CPU_HTT_SUPPORT
		val |= 1 << 28;
#endif
#if defined CONFIG_CPU_TM_SUPPORT && CONFIG_CPU_TM_SUPPORT
		val |= 1 << 29;
#endif
			/* Bit 30: reserved */
#if defined CONFIG_CPU_PBE_SUPPORT && CONFIG_CPU_PBE_SUPPORT
		val |= 1 << 31;
#endif
		*edxp = val;

#if 80686 <= CONFIG_CPU
		cpssp->NAME.update_signature = 0;
#endif
		break;

	case 2: {
		uint8_t info[4*4];
		unsigned ninfos;

		ninfos = 0;
		info[ninfos++] = 1; /* Call cpuid once. */

		/*
		 * Instruction TLB
		 */
#if defined CONFIG_CPU_TLBI4K_SIZE
		if (CONFIG_CPU_TLBI4K_SIZE == 32
		 && CONFIG_CPU_TLBI4K_ASSOC == 4) {
			info[ninfos++] = 0x01;
		} else {
			assert(0);
		}
#endif
#if defined CONFIG_CPU_TLBI4M_SIZE
		if (CONFIG_CPU_TLBI4M_SIZE == 2
		 && CONFIG_CPU_TLBI4M_ASSOC == 2) {
			info[ninfos++] = 0x02;
		} else {
			assert(0);
		}
#endif

		/*
		 * Data TLB
		 */
#if defined CONFIG_CPU_TLBD4K_SIZE
		if (CONFIG_CPU_TLBD4K_SIZE == 64
		 && CONFIG_CPU_TLBD4K_ASSOC == 4) {
			info[ninfos++] = 0x03;
		} else {
			assert(0);
		}
#endif
#if defined CONFIG_CPU_TLBD4M_SIZE
		if (CONFIG_CPU_TLBD4M_SIZE == 8
		 && CONFIG_CPU_TLBD4M_ASSOC == 4) {
			info[ninfos++] = 0x04;
		} else {
			assert(0);
		}
#endif

		/*
		 * 1st-level Instruction Cache
		 */
		if (CONFIG_CPU_L1I_SIZE == 8*1024
		 && CONFIG_CPU_L1I_ASSOC == 4) {
			info[ninfos++] = 0x06;
		} else if (CONFIG_CPU_L1I_SIZE == 16*1024
			&& CONFIG_CPU_L1I_ASSOC == 4) {
			info[ninfos++] = 0x08;
		} else {
			assert(0);
		}

		/*
		 * 1st-level Data Cache
		 */
		if (CONFIG_CPU_L1D_SIZE == 8*1024
		 && CONFIG_CPU_L1D_ASSOC == 2) {
			info[ninfos++] = 0x0a;
		} else if (CONFIG_CPU_L1D_SIZE == 16*1024
			&& CONFIG_CPU_L1D_ASSOC == 4) {
			info[ninfos++] = 0x0c;
		} else {
			assert(0);
		}

		/*
		 * 2nd-level Cache
		 */
		if (CONFIG_CPU_L2_SIZE == 128*1024
		 && CONFIG_CPU_L2_ASSOC == 4) {
			info[ninfos++] = 0x41;
		} else if (CONFIG_CPU_L2_SIZE == 256*1024
			&& CONFIG_CPU_L2_ASSOC == 4) {
			info[ninfos++] = 0x42;
		} else if (CONFIG_CPU_L2_SIZE == 512*1024
			&& CONFIG_CPU_L2_ASSOC == 4) {
			info[ninfos++] = 0x43;
		} else if (CONFIG_CPU_L2_SIZE == 1024*1024
			&& CONFIG_CPU_L2_ASSOC == 4) {
			info[ninfos++] = 0x44;
		} else if (CONFIG_CPU_L2_SIZE == 2*1024*1024
			&& CONFIG_CPU_L2_ASSOC == 4) {
			info[ninfos++] = 0x45;
		} else {
			assert(0);
		}

		/*
		 * 3rd-level Cache
		 */
		/* FIXME */

		/*
		 * Rest of entries are NULL entries.
		 */
		for ( ; ninfos < sizeof(info) / sizeof(info[0]); ninfos++) {
			info[ninfos] = 0x00;
		}

		/* Cache Info */
		*eaxp = (info[0*4+3] << 24)
			| (info[0*4+2] << 16)
			| (info[0*4+1] << 8)
			| (info[0*4+0] << 0);
		*edxp = (info[1*4+3] << 24)
			| (info[1*4+2] << 16)
			| (info[1*4+1] << 8)
			| (info[1*4+0] << 0);
		*ebxp = (info[2*4+3] << 24)
			| (info[2*4+2] << 16)
			| (info[2*4+1] << 8)
			| (info[2*4+0] << 0);
		*ecxp = (info[3*4+3] << 24)
			| (info[3*4+2] << 16)
			| (info[3*4+1] << 8)
			| (info[3*4+0] << 0);
		break;
	    }

#if (defined CONFIG_CPU_LAHF_SUPPORT && CONFIG_CPU_LAHF_SUPPORT) \
 || (defined CONFIG_CPU_SYSCALL_SUPPORT && CONFIG_CPU_SYSCALL_SUPPORT) \
 || (defined CONFIG_CPU_NX_SUPPORT && CONFIG_CPU_NX_SUPPORT) \
 || (defined CONFIG_CPU_FFXSR_SUPPORT && CONFIG_CPU_FFXSR_SUPPORT) \
 || (defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT)
	case 0x80000000:
		/*
		 * Extended feature information.
		 */
		*eaxp = 0x80000001;
		*ebxp = 0; /* Reserved */
		*ecxp = 0; /* Reserved */
		*edxp = 0; /* Reserved */
		break;

	case 0x80000001:
		/* Extended Processor Signature and Feature Bits */
		*eaxp = 0; /* FIXME */

		*ebxp = 0; /* Reserved */

		/* Feature Info */
		/*
		 * %ecx
		 */
		val = 0;
#if defined CONFIG_CPU_LAHF_SUPPORT && CONFIG_CPU_LAHF_SUPPORT
		val |= 1 << 0;
#endif
		/* Bit 31-1: reserved */
		*ecxp = val;

		/*
		 * %edx
		 */
		val = 0;
		/* Bit 10-0: reserved */
#if defined CONFIG_CPU_SYSCALL_SUPPORT && CONFIG_CPU_SYSCALL_SUPPORT
		val |= 1 << 11;
#endif
		/* Bit 19-12: reserved */
#if defined CONFIG_CPU_NX_SUPPORT && CONFIG_CPU_NX_SUPPORT
		val |= 1 << 20;
#endif
		/* Bit 24-21: reserved */
#if defined CONFIG_CPU_FFXSR_SUPPORT && CONFIG_CPU_FFXSR_SUPPORT
		val |= 1 << 25;
#endif
		/* Bit 28-26: reserved */
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
		val |= 1 << 29;
#endif
		/* Bit 31-30: reserved */
		*edxp = val;
		break;
#endif /* ... */

	default:
		assert(0); /* Cannot happen. */
	}
}
#endif /* 80586 <= CONFIG_CPU */

/* ################ CPU_RESET ############################################## */

static void
NAME_(cpu_initialize)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int reset_flag)
{
#if defined CONFIG_CPU_MCA_SUPPORT && CONFIG_CPU_MCA_SUPPORT
	int i;
#endif

	cpssp->NAME.state_interrupt_pending = 0;
	cpssp->NAME.state_nmi_pending = 0;
	cpssp->NAME.state_smi_pending = 0;

	cpssp->NAME.smm = 0;
	if (reset_flag) {
		cpssp->NAME.smbase = 0x30000;
	}

	// TODO reset other vars and regs too
	cpssp->NAME.tsc = 0ULL;
	cpssp->NAME.int_delay = 0;
	cpssp->NAME.state_halt = 0;
#if 80686 <= CONFIG_CPU
	cpssp->NAME.update_signature = 0;
#endif

	/* Reset PE first. */
	NAME_(set_cr)(cpssp, cpu, 0, 0x60000010);
	NAME_(set_cr)(cpssp, cpu, 3, 0);
#if 80586 <= CONFIG_CPU
	NAME_(set_cr)(cpssp, cpu, 4, 0x00000000);
#endif

	NAME_(set_dr)(cpssp, cpu, 0, 0);
	NAME_(set_dr)(cpssp, cpu, 1, 0);
	NAME_(set_dr)(cpssp, cpu, 2, 0);
	NAME_(set_dr)(cpssp, cpu, 3, 0);
	NAME_(set_dr)(cpssp, cpu, 6, 0);
	NAME_(set_dr)(cpssp, cpu, 7, 0);

	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_CS), 0xf000);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_CS), 0x1b, 0xffff0000, 0x0000ffff, 0, 0);
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_SS), 0x0000);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_SS), 0x13, 0x00000000, 0x0000ffff, 0, 0);
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_DS), 0x0000);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_DS), 0x13, 0x00000000, 0x0000ffff, 0, 0);
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_ES), 0x0000);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_ES), 0x13, 0x00000000, 0x0000ffff, 0, 0);
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_FS), 0x0000);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_FS), 0x13, 0x00000000, 0x0000ffff, 0, 0);
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_GS), 0x0000);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_GS), 0x13, 0x00000000, 0x0000ffff, 0, 0);
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_LDT), 0x0000);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_LDT), 0x02, 0x00000000, 0x0000ffff, 0, 0);
	NAME_(store_sreg)(cpssp, cpu, NAME_(SEG_TR), 0x0000);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_TR), 0x03, 0x00000000, 0x0000ffff, 0, 0);

	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_GDT), 0x00, 0x00000000, 0x0000ffff, 0, 0);
	NAME_(fake_entry)(cpssp, cpu, NAME_(SEG_IDT), 0x00, 0x00000000, 0x0000ffff, 0, 0);

	cpu->eip = 0x0000fff0;
	NAME_(set_eflags_all)(cpssp, cpu, 0x00000002, 1 + 1);
	NAME_(store_reg32)(cpssp, cpu, NAME_(E_AX), 0);
	NAME_(store_reg32)(cpssp, cpu, NAME_(E_BX), 0);
	NAME_(store_reg32)(cpssp, cpu, NAME_(E_CX), 0);
	NAME_(store_reg32)(cpssp, cpu, NAME_(E_DX), 0x00000600);
	NAME_(store_reg32)(cpssp, cpu, NAME_(E_DI), 0);
	NAME_(store_reg32)(cpssp, cpu, NAME_(E_SI), 0);
	NAME_(store_reg32)(cpssp, cpu, NAME_(E_BP), 0);
	NAME_(store_reg32)(cpssp, cpu, NAME_(E_SP), 0);

#if defined CONFIG_CPU_SEP_SUPPORT && CONFIG_CPU_SEP_SUPPORT
	STATE_SET(sysenter_cs_msr, 0);
	STATE_SET(sysenter_eip_msr, 0);
	STATE_SET(sysenter_esp_msr, 0);
#endif
#if defined CONFIG_CPU_MCA_SUPPORT && CONFIG_CPU_MCA_SUPPORT
	cpssp->NAME.mca_mcg = 0;
	for (i = 0; i < 5; i++) {
		cpssp->NAME.mca_mcX_ctl[i] = 0;
	}
#endif
}

static void
NAME_(cpu_reset)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	NAME_(cpu_initialize)(cpssp, cpu, 1);
}

#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
static void
NAME_(cpu_init)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	NAME_(cpu_initialize)(cpssp, cpu, 0);
}
#endif /* defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT */

#if defined CONFIG_CPU_MMX_SUPPORT && CONFIG_CPU_MMX_SUPPORT

static void
NAME_(set_mmx)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg, uint64_t val)
{
	*(uint64_t *) &cpssp->fpu.st[reg] = val;
}

static uint64_t
NAME_(get_mmx)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t reg)
{
	return *(uint64_t *) &cpssp->fpu.st[reg];
}

static uint64_t
NAME_(load_rm_cx8)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, int wflag)
{
	uint64_t res;

	if (cpu->mod == 0x3) {
		res = NAME_(get_mmx)(cpssp, cpu, cpu->rm);
	} else {
		res = NAME_(load_mem_cx8)(cpssp, cpu, wflag);
	}

	return res;
}

static void
NAME_(store_rm_cx8)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint64_t val)
{
	if (cpu->mod == 0x3) {
		NAME_(set_mmx)(cpssp, cpu, cpu->rm, val);
	} else {
		NAME_(store_mem_cx8)(cpssp, cpu, val);
	}
}

static void
NAME_(mmx_start)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t opcode)
{
	int i;

	if (STATE_GET(cr0_em)) {
		NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
	}
	if (STATE_GET(cr0_ts)) {
		NAME_(NM)(cpssp, cpu);
	}
	if (cpssp->NAME.fpu_error) {
		NAME_(MF)(cpssp, cpu);
	}

	cpssp->fpu.top = 0;
	for (i = 0; i < 8; i++) {
		cpssp->fpu.used[i] = 1;
	}
}

static uint64_t
NAME_(mmx_unpack)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t opcode, uint64_t op1, uint64_t op2)
{
	uint64_t res;
	uint32_t d0, d1;
	uint16_t w0, w1, w2, w3;
	uint8_t b0, b1, b2, b3, b4, b5, b6, b7;

	switch (opcode) {
	case 0x60:	/* punpcklbw */
		b7 = (op2 >> 24) & 0xff;
		b6 = (op1 >> 24) & 0xff;
		b5 = (op2 >> 16) & 0xff;
		b4 = (op1 >> 16) & 0xff;
		b3 = (op2 >> 8) & 0xff;
		b2 = (op1 >> 8) & 0xff;
		b1 = (op2 >> 0) & 0xff;
		b0 = (op1 >> 0) & 0xff;
		res = ((uint64_t) b7 << 56)
			| ((uint64_t) b6 << 48)
			| ((uint64_t) b5 << 40)
			| ((uint64_t) b4 << 32)
			| ((uint64_t) b3 << 24)
			| ((uint64_t) b2 << 16)
			| ((uint64_t) b1 << 8)
			| ((uint64_t) b0 << 0);
		break;
	case 0x61:	/* punpcklwd */
		w3 = (op2 >> 16) & 0xffff;
		w2 = (op1 >> 16) & 0xffff;
		w1 = (op2 >> 0) & 0xffff;
		w0 = (op1 >> 0) & 0xffff;
		res = ((uint64_t) w3 << 48)
			| ((uint64_t) w2 << 32)
			| ((uint64_t) w1 << 16)
			| ((uint64_t) w0 << 0);
		break;
	case 0x62:	/* punpckldq */
		d1 = (op2 >> 0) & 0xffffffff;
		d0 = (op1 >> 0) & 0xffffffff;
		res = ((uint64_t) d1 << 32)
			| ((uint64_t) d0 << 0);
		break;
	case 0x68:	/* punpckhbw */
		b7 = (op2 >> 56) & 0xff;
		b6 = (op1 >> 56) & 0xff;
		b5 = (op2 >> 48) & 0xff;
		b4 = (op1 >> 48) & 0xff;
		b3 = (op2 >> 40) & 0xff;
		b2 = (op1 >> 40) & 0xff;
		b1 = (op2 >> 32) & 0xff;
		b0 = (op1 >> 32) & 0xff;
		res = ((uint64_t) b7 << 56)
			| ((uint64_t) b6 << 48)
			| ((uint64_t) b5 << 40)
			| ((uint64_t) b4 << 32)
			| ((uint64_t) b3 << 24)
			| ((uint64_t) b2 << 16)
			| ((uint64_t) b1 << 8)
			| ((uint64_t) b0 << 0);
		break;
	case 0x69:	/* punpckhwd */
		w3 = (op2 >> 48) & 0xffff;
		w2 = (op1 >> 48) & 0xffff;
		w1 = (op2 >> 32) & 0xffff;
		w0 = (op1 >> 32) & 0xffff;
		res = ((uint64_t) w3 << 48)
			| ((uint64_t) w2 << 32)
			| ((uint64_t) w1 << 16)
			| ((uint64_t) w0 << 0);
		break;
	case 0x6a:	/* punpckhdq */
		d1 = (op2 >> 32) & 0xffffffff;
		d0 = (op1 >> 32) & 0xffffffff;
		res = ((uint64_t) d1 << 32)
			| ((uint64_t) d0 << 0);
		break;
	default:
		assert(0); /* Mustn't happen. */
	}

	return res;
}

static uint64_t
NAME_(mmx_pack)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t opcode, uint64_t dst, uint64_t src)
{
	uint64_t res;
	int16_t w0, w1, w2, w3, w4, w5, w6, w7;
	int32_t d0, d1, d2, d3;

	switch (opcode) {
	case 0x63:	/* packsswb */
		w7 = (src >> 48) & 0xffff;
		w6 = (src >> 32) & 0xffff;
		w5 = (src >> 16) & 0xffff;
		w4 = (src >> 0) & 0xffff;
		w3 = (dst >> 48) & 0xffff;
		w2 = (dst >> 32) & 0xffff;
		w1 = (dst >> 16) & 0xffff;
		w0 = (dst >> 0) & 0xffff;
		if (w0 < -128) w0 = -128;
		else if (127 < w0) w0 = 127;
		if (w1 < -128) w1 = -128;
		else if (127 < w1) w1 = 127;
		if (w2 < -128) w2 = -128;
		else if (127 < w2) w2 = 127;
		if (w3 < -128) w3 = -128;
		else if (127 < w3) w3 = 127;
		if (w4 < -128) w4 = -128;
		else if (127 < w4) w4 = 127;
		if (w5 < -128) w5 = -128;
		else if (127 < w5) w5 = 127;
		if (w6 < -128) w6 = -128;
		else if (127 < w6) w6 = 127;
		if (w7 < -128) w7 = -128;
		else if (127 < w7) w7 = 127;
		res = ((uint64_t) (uint8_t) w7 << 56)
			| ((uint64_t) (uint8_t) w6 << 48)
			| ((uint64_t) (uint8_t) w5 << 40)
			| ((uint64_t) (uint8_t) w4 << 32)
			| ((uint64_t) (uint8_t) w3 << 24)
			| ((uint64_t) (uint8_t) w2 << 16)
			| ((uint64_t) (uint8_t) w1 << 8)
			| ((uint64_t) (uint8_t) w0 << 0);
		break;
	case 0x67:	/* packuswb */
		w7 = (src >> 48) & 0xffff;
		w6 = (src >> 32) & 0xffff;
		w5 = (src >> 16) & 0xffff;
		w4 = (src >> 0) & 0xffff;
		w3 = (dst >> 48) & 0xffff;
		w2 = (dst >> 32) & 0xffff;
		w1 = (dst >> 16) & 0xffff;
		w0 = (dst >> 0) & 0xffff;
		if (w0 < 0) w0 = 0;
		else if (255 < w0) w0 = 255;
		if (w1 < 0) w1 = 0;
		else if (255 < w1) w1 = 255;
		if (w2 < 0) w2 = 0;
		else if (255 < w2) w2 = 255;
		if (w3 < 0) w3 = 0;
		else if (255 < w3) w3 = 255;
		if (w4 < 0) w4 = 0;
		else if (255 < w4) w4 = 255;
		if (w5 < 0) w5 = 0;
		else if (255 < w5) w5 = 255;
		if (w6 < 0) w6 = 0;
		else if (255 < w6) w6 = 255;
		if (w7 < 0) w7 = 0;
		else if (255 < w7) w7 = 255;
		res = ((uint64_t) (uint8_t) w7 << 56)
			| ((uint64_t) (uint8_t) w6 << 48)
			| ((uint64_t) (uint8_t) w5 << 40)
			| ((uint64_t) (uint8_t) w4 << 32)
			| ((uint64_t) (uint8_t) w3 << 24)
			| ((uint64_t) (uint8_t) w2 << 16)
			| ((uint64_t) (uint8_t) w1 << 8)
			| ((uint64_t) (uint8_t) w0 << 0);
		break;
	case 0x6b:	/* packssdw */
		d3 = (src >> 32) & 0xffffffff;
		d2 = (src >> 0) & 0xffffffff;
		d1 = (dst >> 32) & 0xffffffff;
		d0 = (dst >> 0) & 0xffffffff;
		if (d0 < -32768) d0 = -32768;
		else if (32767 < d0) d0 = 32767;
		if (d1 < -32768) d1 = -32768;
		else if (32767 < d1) d1 = 32767;
		if (d2 < -32768) d2 = -32768;
		else if (32767 < d2) d2 = 32767;
		if (d3 < -32768) d3 = -32768;
		else if (32767 < d3) d3 = 32767;
		res = ((uint64_t) (uint16_t) d3 << 48)
			| ((uint64_t) (uint16_t) d2 << 32)
			| ((uint64_t) (uint16_t) d1 << 16)
			| ((uint64_t) (uint16_t) d0 << 0);
		break;
	default:
		assert(0); /* Mustn't happen. */
	}

	return res;
}

static uint64_t
NAME_(mmxb)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t opcode, uint64_t op1, uint64_t op2)
{
	uint64_t res;
	int i;

	res = 0;
	for (i = 0; i < 8; i++) {
		int8_t b1;
		int8_t b2;
		int16_t r;

		b1 = (op1 >> (8 * i)) & 0xff;
		b2 = (op2 >> (8 * i)) & 0xff;

		switch (opcode) {
		case 0x64:	/* pcmpgtb */
			if (b1 < b2) {
				r = -1;
			} else {
				r = 0;
			}
			break;
		case 0x74:	/* pcmpeqb */
			if (b1 == b2) {
				r = -1;
			} else {
				r = 0;
			}
			break;
		case 0xd8:	/* psubusb */
			r = (int16_t) (uint8_t) b2 - (int16_t) (uint8_t) b1;
			if (r < 0) r = 0;
			break;
		case 0xdc:	/* paddusb */
			r = (int16_t) (uint8_t) b2 + (int16_t) (uint8_t) b1;
			if (255 < r) r = 255;
			break;
		case 0xe8:	/* psubsb */
			r = b2 - b1;
			if (r < -128) r = -128;
			else if (127 < r) r = 127;
			break;
		case 0xec:	/* paddsb */
			r = b2 + b1;
			if (r < -128) r = -128;
			else if (127 < r) r = 127;
			break;
		case 0xf8:	/* psubb */
			r = (int8_t) (b2 - b1);
			break;
		case 0xfc:	/* paddb */
			r = (int8_t) (b2 + b1);
			break;
		default:
			assert(0); /* Mustn't happen. */
		}

		res |= (uint64_t) (uint8_t) r << (8 * i);
	}

	return res;
}

static uint64_t
NAME_(mmxw)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t opcode, uint64_t op1, uint64_t op2)
{
	uint64_t res;
	int i;

	res = 0;
	for (i = 0; i < 4; i++) {
		int16_t b1;
		int16_t b2;
		int32_t r;

		b1 = (op1 >> (16 * i)) & 0xffff;
		b2 = (op2 >> (16 * i)) & 0xffff;

		switch (opcode) {
		case 0x65:	/* pcmpgtw */
			if (b1 < b2) {
				r = -1;
			} else {
				r = 0;
			}
			break;
		case 0x75:	/* pcmpeqw */
			if (b1 == b2) {
				r = -1;
			} else {
				r = 0;
			}
			break;
		case 0xd5:	/* pmullw */
			r = (int32_t) b1 * (int32_t) b2;
			break;
		case 0xd9:	/* psubusw */
			r = (int32_t) (uint16_t) b2 - (int32_t) (uint16_t) b1;
			if (r < 0) r = 0;
			break;
		case 0xdd:	/* paddusw */
			r = (int32_t) (uint16_t) b2 + (int32_t) (uint16_t) b1;
			if (65535 < r) r = 65535;
			break;
		case 0xe4:	/* pmulhuw */
			r = ((uint32_t) (uint16_t) b1 * (uint32_t) (uint16_t) b2) >> 16;
			break;
		case 0xe5:	/* pmulhw */
			r = ((int32_t) b1 * (int32_t) b2) >> 16;
			break;
		case 0xe9:	/* psubsw */
			r = b2 - b1;
			if (r < -32768) r = -32768;
			else if (32767 < r) r = 32767;
			break;
		case 0xed:	/* paddsw */
			r = b2 + b1;
			if (r < -32768) r = -32768;
			else if (32767 < r) r = 32767;
			break;
		case 0xf9:	/* psubw */
			r = b2 - b1;
			break;
		case 0xfd:	/* paddw */
			r = b2 + b1;
			break;
		default:
			assert(0); /* Mustn't happen. */
		}

		res |= (uint64_t) (uint16_t) r << (16 * i);
	}

	return res;
}

static uint64_t
NAME_(mmxd)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t opcode, uint64_t op1, uint64_t op2)
{
	uint64_t res;
	int i;

	res = 0;
	for (i = 0; i < 2; i++) {
		int32_t b1;
		int32_t b2;
		int32_t r;

		b1 = (op1 >> (32 * i)) & 0xff;
		b2 = (op2 >> (32 * i)) & 0xff;

		switch (opcode) {
		case 0x76:	/* pcmpeqd */
			if (b1 == b2) {
				r = -1;
			} else {
				r = 0;
			}
			break;
		case 0x66:	/* pcmpgtd */
			if (b1 < b2) {
				r = -1;
			} else {
				r = 0;
			}
			break;
		case 0xfa:	/* psubd */
			r = b2 - b1;
			break;
		case 0xfe:	/* paddd */
			r = b2 + b1;
			break;
		default:
			assert(0); /* Mustn't happen. */
		}

		res |= (uint64_t) (uint32_t) r << (32 * i);
	}

	return res;
}

static uint64_t
NAME_(mmxq)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t opcode, uint64_t op1, uint64_t op2)
{
	uint64_t res;

	switch (opcode) {
	case 0xdb:	/* pand */
		res = op1 & op2;
		break;
	case 0xdf:	/* pandn */
		res = op1 & ~op2;
		break;
	case 0xeb:	/* por */
		res = op1 | op2;
		break;
	case 0xef:	/* pxor */
		res = op1 ^ op2;
		break;
	default:
		assert(0); /* Mustn't happen. */
	}

	return res;
}

static uint64_t
NAME_(mmx_shiftw)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t opcode, uint64_t op1, uint64_t op2)
{
	uint64_t res;
	int i;

	res = 0;
	for (i = 0; i < 4; i++) {
		int16_t b1;
		int32_t r;

		b1 = (op1 >> (16 * i)) & 0xffff;

		switch (opcode) {
		case 0xd1:	/* psrlw */
			if (op2 <= 15) {
				r = (uint16_t) b1 >> op2;
			} else {
				r = 0;
			}
			break;
		case 0xe1:	/* psraw */
			if (op2 <= 15) {
				r = (int16_t) b1 >> op2;
			} else {
				r = (int16_t) b1 >> 15;
			}
			break;
		case 0xf1:	/* psllw */
			if (op2 <= 15) {
				r = b1 << op2;
			} else {
				r = 0;
			}
			break;
		default:
			assert(0); /* Mustn't happen. */
		}

		res |= (uint64_t) (uint16_t) r << (16 * i);
	}

	return res;
}

static uint64_t
NAME_(mmx_shiftd)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t opcode, uint64_t op1, uint64_t op2)
{
	uint64_t res;
	int i;

	res = 0;
	for (i = 0; i < 2; i++) {
		int32_t b1;
		int32_t r;

		b1 = (op1 >> (32 * i)) & 0xffffffff;

		switch (opcode) {
		case 0xd2:	/* psrld */
			if (op2 <= 31) {
				r = (uint32_t) b1 >> op2;
			} else {
				r = 0;
			}
			break;
		case 0xe2:	/* psrad */
			if (op2 <= 31) {
				r = (int32_t) b1 >> op2;
			} else {
				r = (int32_t) b1 >> 31;
			}
			break;
		case 0xf2:	/* pslld */
			if (op2 <= 31) {
				r = b1 << op2;
			} else {
				r = 0;
			}
			break;
		default:
			assert(0); /* Mustn't happen. */
		}

		res |= (uint64_t) (uint32_t) r << (32 * i);
	}

	return res;
}

static uint64_t
NAME_(mmx_shiftq)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint8_t opcode, uint64_t op1, uint64_t op2)
{
	uint64_t res;

	switch (opcode) {
	case 0xd3:	/* psrlq */
		if (op2 <= 63) {
			res = op1 >> op2;
		} else {
			res = 0;
		}
		break;
	case 0xf3:	/* psllq */
		if (op2 <= 63) {
			res = op1 << op2;
		} else {
			res = 0;
		}
		break;
	default:
		assert(0); /* Mustn't happen. */
	}

	return res;
}

static uint64_t
NAME_(mmx_pmaddwd)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint64_t src, uint64_t dst)
{
	uint64_t res;
	int16_t s0, s1, s2, s3;
	int16_t d0, d1, d2, d3;
	int32_t r0, r1;

	s3 = (src >> 48) & 0xffff;
	s2 = (src >> 32) & 0xffff;
	s1 = (src >> 16) & 0xffff;
	s0 = (src >>  0) & 0xffff;
	d3 = (dst >> 48) & 0xffff;
	d2 = (dst >> 32) & 0xffff;
	d1 = (dst >> 16) & 0xffff;
	d0 = (dst >>  0) & 0xffff;

	r1 = (int32_t) s2 * (int32_t) d2 + (int32_t) d3 * (int32_t) d3;
	r0 = (int32_t) s0 * (int32_t) d0 + (int32_t) d1 * (int32_t) d1;

	res = ((uint64_t) (uint32_t) r1 << 32)
		| ((uint64_t) (uint32_t) r0 << 0);

	return res;
}

static void
NAME_(mmx_emms)(struct cpssp *cpssp, struct NAME_(cpu) *cpu)
{
	int i;

	for (i = 0; i < 8; i++) {
		cpssp->fpu.used[i] = 0;
	}
}

#endif /* defined CONFIG_CPU_MMX_SUPPORT && CONFIG_CPU_MMX_SUPPORT */

static void
NAME_(inv_op_64bit_mode)(struct cpssp *cpssp, struct NAME_(cpu) *cpu, uint16_t opcode)
{
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	if (NAME_(get_lm_mode)(cpssp, cpu))
		NAME_(UD)(cpssp, cpu, opcode);
#endif
}

/* ################ CPU_STEP ############################################### */

static void
NAME_(step)(struct cpssp *cpssp)
{
	int op_mode;
	int st_mode;
	uint8_t vec;
	uint16_t opcode;
	udata_t sp;
	udata_t op1;
	udata_t op2;
	udata_t op3;
	udata_t res1;
	udata_t res2;
	udata_t res3;
#if 80586 <= CONFIG_CPU
	udata_t res4;
#endif
	uint64_t descr;
#if defined CONFIG_CPU_CX8_SUPPORT && CONFIG_CPU_CX8_SUPPORT
	uint64_t op64_1;
	uint64_t op64_2;
	uint64_t op64_3;
	uint64_t res64_1;
#endif
	struct NAME_(cpu) cpu_local;
	struct NAME_(cpu) *cpu = &cpu_local;
	int i;

	cpssp->NAME.tsc++;

	cpu->pre_op = 0;
	cpu->pre_addr = 0;
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	cpu->pre_rex = 0;
#endif
	cpu->pre_repe_repne = 0;
	cpu->seg_override = 0;
	cpu->eip = STATE_GET(eip);
	cpu->esp = NAME_(load_reg32)(cpssp, cpu, NAME_(E_SP));


#if DEBUG_CONTROL_FLOW
	uint32_t base = NAME_(seg_base)(cpssp, NAME_(SEG_CS));
#endif

#if 0
	if (NAME_(seg_base)(cpssp, NAME_(SEG_CS)) + STATE_GET(eip) == 0x7c00) {
		loglevel = 1;
	}
#endif

#if ! EXPERIMENTAL_RESET
	if (unlikely(setjmp(cpu->exception_buf))) {
		return;
	}
#endif

	if (unlikely(cpssp->NAME.state_event)) {
		if (! cpssp->state_power) {
			/*
			 * Power-off.
			 */
#if DEBUG_CONTROL_FLOW
			if (2 <= DEBUG_CONTROL_FLOW
			 || loglevel) {
				fprintf(stderr, "%lld: Power-off\n",
						cpssp->process.inst_cnt);
			}
#endif

			cpssp->process.inst_cnt = cpssp->process.inst_limit;
			return;

		} else if (! cpssp->NAME.state_n_reset
#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
			|| ! cpssp->NAME.state_n_init
#endif
			) {
			if (! cpssp->NAME.state_n_reset) {
				/*
				 * Reset
				 */
#if DEBUG_CONTROL_FLOW
				if (2 <= DEBUG_CONTROL_FLOW
				 || loglevel) {
					fprintf(stderr, "%lld: Reset\n",
							cpssp->process.inst_cnt);
				}
#endif
				NAME_(cpu_reset)(cpssp, cpu);
			}
#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
			if (! cpssp->NAME.state_n_init) {
				/*
				 * Init
				 */
#if DEBUG_CONTROL_FLOW
				if (2 <= DEBUG_CONTROL_FLOW
				 || loglevel) {
					fprintf(stderr, "%lld: Init\n",
							cpssp->process.inst_cnt);
				}
#endif
				NAME_(cpu_init)(cpssp, cpu);
			}
#endif /* defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT */

			cpssp->process.inst_cnt = cpssp->process.inst_limit;
			goto EXIT_COMMIT;

		} else if (cpssp->NAME.int_delay) {
			cpssp->NAME.int_delay = 0;

		} else if (cpssp->NAME.exception_pending) {
			/*
			 * Exception pending.
			 */
#if DEBUG_CONTROL_FLOW
			if (2 <= DEBUG_CONTROL_FLOW
			 || loglevel) {
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
				if (STATE_GET(lm_active_msr)) {
					fprintf(stderr, "%lld: Exception 0x%02x at %016llx",
							cpssp->process.inst_cnt,
							cpssp->NAME.exception_num,
							STATE_GET(eip));
				} else
#endif
				{
					fprintf(stderr, "%lld: Exception 0x%02x at %08llx (%08llx:%08llx)",
							cpssp->process.inst_cnt,
							cpssp->NAME.exception_num,
							base + STATE_GET(eip),
							base,
							STATE_GET(eip));
				}
				NAME_(dump)(cpssp, cpu);
			}
#endif

			NAME_(intn)(cpssp, cpu, cpssp->NAME.exception_num, 1);

			cpssp->NAME.exception_pending = 0;
			cpssp->NAME.errc = 0;
			cpssp->process.inst_cnt++;
			goto EXIT_COMMIT;

		} else if (STATE_GET(e_tf_delayed) == 1) {
			STATE_SET(dr_bs, 1);
			vec = 1;
			NAME_(intn)(cpssp, cpu, vec, 1);
			cpssp->process.inst_cnt++;
			goto EXIT_COMMIT;

		} else if (cpssp->NAME.state_smi_pending) {
			/*
			 * SMI pending.
			 */
			cpssp->NAME.state_smi_pending = 0;

#if DEBUG_CONTROL_FLOW
			if (2 <= DEBUG_CONTROL_FLOW
			 || loglevel) {
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
				if (STATE_GET(lm_active_msr)) {
					fprintf(stderr, "%lld: SMM-Interrupt at %016llx",
							cpssp->process.inst_cnt,
							STATE_GET(eip));
				} else
#endif
				{
					fprintf(stderr, "%lld: SMM-Interrupt at %08llx (%08llx:%08llx)",
							cpssp->process.inst_cnt,
							base + STATE_GET(eip),
							base,
							STATE_GET(eip));
				}
				NAME_(dump)(cpssp, cpu);
			}
#endif
			NAME_(smm)(cpssp, cpu);
			cpssp->NAME.state_halt = 0;
			cpssp->process.inst_cnt++;
			goto EXIT_COMMIT;

		} else if (cpssp->NAME.state_nmi_pending) {
			/*
			 * NMI pending.
			 */
			vec = DEFAULT_NAME_(ack)(cpssp);
			NAME_(intn)(cpssp, cpu, vec, 1);
			cpssp->NAME.state_halt = 0;
			cpssp->process.inst_cnt++;
			goto EXIT_COMMIT;

		} else if (cpssp->NAME.state_interrupt_pending
			&& STATE_GET(e_if) == 1) {
			/*
			 * Interrupt pending.
			 */
			vec = DEFAULT_NAME_(ack)(cpssp);
#if DEBUG_CONTROL_FLOW
			if (2 <= DEBUG_CONTROL_FLOW
			 || loglevel) {
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
				if (STATE_GET(lm_active_msr)) {
					fprintf(stderr, "%lld: Interrupt 0x%02x at %016llx",
							cpssp->process.inst_cnt,
							vec,
							STATE_GET(eip));
				} else
#endif
				{
					fprintf(stderr, "%lld: Interrupt 0x%02x at %08llx (%08llx:%08llx)",
							cpssp->process.inst_cnt,
							vec,
							base + STATE_GET(eip),
							base,
							STATE_GET(eip));
				}
				NAME_(dump)(cpssp, cpu);
			}
#endif
			NAME_(intn)(cpssp, cpu, vec, 1);
			cpssp->NAME.state_halt = 0;
			cpssp->process.inst_cnt++;
			goto EXIT_COMMIT;

		} else if (cpssp->NAME.state_halt) {
			/*
			 * Halted.
			 */
#if DEBUG_CONTROL_FLOW
			if (2 <= DEBUG_CONTROL_FLOW
			 || loglevel) {
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
				if (STATE_GET(lm_active_msr)) {
					fprintf(stderr, "%lld: Halting at %016llx",
							cpssp->process.inst_cnt,
							STATE_GET(eip));
				} else
#endif
				{
					fprintf(stderr, "%lld: Halting at %08llx (%08llx:%08llx)",
							cpssp->process.inst_cnt,
							base + STATE_GET(eip),
							base,
							STATE_GET(eip));
				}
				NAME_(dump)(cpssp, cpu);
			}
#endif
			cpssp->process.inst_cnt = cpssp->process.inst_limit;
			return;

		} else {
			// TODO like this?
			cpssp->NAME.state_event = 0;
		}
	}

#if DEBUG_CONTROL_FLOW
	if (2 <= DEBUG_CONTROL_FLOW
	 || loglevel) {
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
		if (STATE_GET(lm_active_msr)) {
			fprintf(stderr, "%lld: Executing at %016lx",
					cpssp->process.inst_cnt,
					STATE_GET(eip));
		} else
#endif
		{
			fprintf(stderr, "%lld: Executing at %08lx (%08lx:%08lx)",
					cpssp->process.inst_cnt,
					base + STATE_GET(eip),
					base,
					STATE_GET(eip));
		}
		NAME_(dump)(cpssp, cpu);
	}
#endif

	STATE_SET(e_tf_delayed, STATE_GET(e_tf));
	cpssp->NAME.state_event |= STATE_GET(e_tf);

	NAME_(fetch_start)(cpssp, cpu);

FETCH_OPCODE:
	opcode = NAME_(fetch8)(cpssp, cpu);

	switch (opcode) {
/* PREFIX */
	case 0x66:	/* op */
		cpu->pre_op = 1;
		goto FETCH_OPCODE;

	case 0x67:	/* addr */
		cpu->pre_addr = 1;
		goto FETCH_OPCODE;

	case 0x2e:	/* CS */
		cpu->seg_override = 1;
		cpu->seg[1] = NAME_(SEG_CS);
		goto FETCH_OPCODE;

	case 0x3e:	/* DS */
		cpu->seg_override = 1;
		cpu->seg[1] = NAME_(SEG_DS);
		goto FETCH_OPCODE;

	case 0x26:	/* ES */
		cpu->seg_override = 1;
		cpu->seg[1] = NAME_(SEG_ES);
		goto FETCH_OPCODE;

	case 0x36:	/* SS */
		cpu->seg_override = 1;
		cpu->seg[1] = NAME_(SEG_SS);
		goto FETCH_OPCODE;

	case 0x64:	/* FS */
		cpu->seg_override = 1;
		cpu->seg[1] = NAME_(SEG_FS);
		goto FETCH_OPCODE;

	case 0x65:	/* GS */
		cpu->seg_override = 1;
		cpu->seg[1] = NAME_(SEG_GS);
		goto FETCH_OPCODE;

	case 0xf0:	/* lock */
		/* FIXME */
		goto FETCH_OPCODE;

	case 0xf2:	/* repne/repnz */
		cpu->pre_repe_repne = 0x1;
		goto FETCH_OPCODE;

	case 0xf3:	/* rep/repe/repz */
		cpu->pre_repe_repne = 0x2;
		goto FETCH_OPCODE;


/* INC/DEC/REX-Prefix */
	case 0x40:	/* inc r16/32 */
	case 0x41:
	case 0x42:
	case 0x43:
	case 0x44:
	case 0x45:
	case 0x46:
	case 0x47:
	case 0x48:	/* dec r16/32 */
	case 0x49:
	case 0x4a:
	case 0x4b:
	case 0x4c:
	case 0x4d:
	case 0x4e:
	case 0x4f:
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
		if (NAME_(get_lm_mode)(cpssp, cpu)) {
			cpu->pre_rex = opcode;
			goto FETCH_OPCODE;
		}
#endif
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		op1 = NAME_(load_reg)(cpssp, cpu, ((opcode >> 0) & 0x7), op_mode);
		res1 = NAME_(inc_dec)(cpssp, cpu, op1, op_mode, ((opcode >> 0) & 0x8));
		NAME_(store_reg)(cpssp, cpu, ((opcode >> 0) & 0x7), res1, op_mode);
		break;


/* ADD/ADC/AND/XOR/OR/SBB/SUB/CMP */
	case 0x00:	/* add/adc/and/xor/or/sbb/sub/cmp rm8, r8 */
	case 0x10:
	case 0x20:
	case 0x30:
	case 0x08:
	case 0x18:
	case 0x28:
	case 0x38:
	case 0x01:	/* add/adc/and/xor/or/sbb/sub/cmp rm16/32, r16/32 */
	case 0x11:
	case 0x21:
	case 0x31:
	case 0x09:
	case 0x19:
	case 0x29:
	case 0x39:
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		NAME_(get_modrm)(cpssp, cpu);
		op1 = NAME_(load_rm)(cpssp, cpu, op_mode, ((opcode >> 3) & 0x7) != 0x7);
		op2 = NAME_(load_reg)(cpssp, cpu, cpu->reg_opex, op_mode);
		res1 = NAME_(alu_arith)(cpssp, cpu, op1, op2, ((opcode >> 3) & 0x7), op_mode);
		if (((opcode >> 3) & 0x7) != 0x7) {
			NAME_(store_rm)(cpssp, cpu, res1, op_mode);
		}
		break;

	case 0x02:	/* add/adc/and/xor/or/sbb/sub/cmp r8, rm8 */
	case 0x12:
	case 0x22:
	case 0x32:
	case 0x0a:
	case 0x1a:
	case 0x2a:
	case 0x3a:
	case 0x03:	/* add/adc/and/xor/or/sbb/sub/cmp r16/32, rm16/32 */
	case 0x13:
	case 0x23:
	case 0x33:
	case 0x0b:
	case 0x1b:
	case 0x2b:
	case 0x3b:
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		NAME_(get_modrm)(cpssp, cpu);
		op1 = NAME_(load_reg)(cpssp, cpu, cpu->reg_opex, op_mode);
		op2 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
		res1 = NAME_(alu_arith)(cpssp, cpu, op1, op2, ((opcode >> 3) & 0x7), op_mode);
		if (((opcode >> 3) & 0x7) != 0x7) {
			NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, res1, op_mode);
		}
		break;

	case 0x04:	/* add/adc/and/xor/or/sbb/sub/cmp al, imm8 */
	case 0x14:
	case 0x24:
	case 0x34:
	case 0x0c:
	case 0x1c:
	case 0x2c:
	case 0x3c:
	case 0x05:	/* add/adc/and/xor/or/sbb/sub/cmp e/ax, imm16/32 */
	case 0x15:
	case 0x25:
	case 0x35:
	case 0x0d:
	case 0x1d:
	case 0x2d:
	case 0x3d:
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		op1 = NAME_(load_reg)(cpssp, cpu, NAME_(E_AX), op_mode);
		op2 = NAME_(fetch)(cpssp, cpu, op_mode);
		res1 = NAME_(alu_arith)(cpssp, cpu, op1, op2, ((opcode >> 3) & 0x7), op_mode);
		if (((opcode >> 3) & 0x7) != 0x7) {
			NAME_(store_reg)(cpssp, cpu, NAME_(E_AX), res1, op_mode);
		}
		break;

	case 0x80:	/* add/adc/and/xor/or/sbb/sub/cmp rm8, imm8 */
	case 0x81:	/* add/adc/and/xor/or/sbb/sub/cmp rm16/32, imm16/32 */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		NAME_(get_modrm)(cpssp, cpu);
		op1 = NAME_(load_rm)(cpssp, cpu, op_mode, cpu->reg_opex != 0x7);
		op2 = NAME_(fetch)(cpssp, cpu, op_mode);
		res1 = NAME_(alu_arith)(cpssp, cpu, op1, op2, cpu->reg_opex, op_mode);
		if (cpu->reg_opex != 0x7) {
			NAME_(store_rm)(cpssp, cpu, res1, op_mode);
		}
		break;

	case 0x82:	/* add/adc/and/xor/or/sbb/sub/cmp rm8, imm8 */ /* Same as above! Undocumented! */
	case 0x83:	/* add/adc/and/xor/or/sbb/sub/cmp rm16/32, imm8 */
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		NAME_(get_modrm)(cpssp, cpu);
		op1 = NAME_(load_rm)(cpssp, cpu, op_mode, cpu->reg_opex != 0x7);
		op2 = NAME_(fetch8)(cpssp, cpu);
		op2 = NAME_(sign_ext)(op2, 0, op_mode);
		res1 = NAME_(alu_arith)(cpssp, cpu, op1, op2, cpu->reg_opex, op_mode);
		if (cpu->reg_opex != 0x7) {
			NAME_(store_rm)(cpssp, cpu, res1, op_mode);
		}
		break;

/* IMUL */
	case 0x6b:	/* imul r16/32, rm16/32, imm8 */
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		NAME_(get_modrm)(cpssp, cpu);
		op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
		op1 = NAME_(sign_ext)(op1, op_mode, MODE_MAX);
		op2 = (data_t) (int8_t) NAME_(fetch8)(cpssp, cpu);
		res1 = NAME_(imul)(cpssp, cpu, op1, op2, &res2, op_mode);
		NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, res1, op_mode);
		break;

	case 0x69:	/* imul r16/32, rm16/32, imm16/32 */
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		NAME_(get_modrm)(cpssp, cpu);
		op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
		op1 = NAME_(sign_ext)(op1, op_mode, MODE_MAX);
		op2 = NAME_(fetch)(cpssp, cpu, op_mode);
		op2 = NAME_(sign_ext)(op2, op_mode, MODE_MAX);
		res1 = NAME_(imul)(cpssp, cpu, op1, op2, &res2, op_mode);
		NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, res1, op_mode);
		break;

/* BCD */
	case 0x27:	/* daa */
		NAME_(inv_op_64bit_mode)(cpssp, cpu, opcode);
		op1 = NAME_(load_reg8)(cpssp, cpu, NAME_(AL));
		res1 = NAME_(daa)(cpssp, cpu, op1);
		NAME_(store_reg8)(cpssp, cpu, NAME_(AL), res1);
		break;

	case 0x2f:	/* das */
		NAME_(inv_op_64bit_mode)(cpssp, cpu, opcode);
		op1 = NAME_(load_reg8)(cpssp, cpu, NAME_(AL));
		res1 = NAME_(das)(cpssp, cpu, op1);
		NAME_(store_reg8)(cpssp, cpu, NAME_(AL), res1);
		break;

	case 0x37:	/* aaa */
		NAME_(inv_op_64bit_mode)(cpssp, cpu, opcode);
		op1 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_AX));
		res1 = NAME_(aaa)(cpssp, cpu, op1);
		NAME_(store_reg16)(cpssp, cpu, NAME_(E_AX), res1);
		break;

	case 0x3f:	/* aas */
		NAME_(inv_op_64bit_mode)(cpssp, cpu, opcode);
		op1 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_AX));
		res1 = NAME_(aas)(cpssp, cpu, op1);
		NAME_(store_reg16)(cpssp, cpu, NAME_(E_AX), res1);
		break;

	case 0xd4:	/* aam imm8 */
		NAME_(inv_op_64bit_mode)(cpssp, cpu, opcode);
		op1 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_AX));
		op2 = NAME_(fetch8)(cpssp, cpu);
		res1 = NAME_(aam)(cpssp, cpu, op1, op2);
		NAME_(store_reg16)(cpssp, cpu, NAME_(E_AX), res1);
		break;

	case 0xd5:	/* aad imm8 */
		NAME_(inv_op_64bit_mode)(cpssp, cpu, opcode);
		op1 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_AX));
		op2 = NAME_(fetch8)(cpssp, cpu);
		res1 = NAME_(aad)(cpssp, cpu, op1, op2);
		NAME_(store_reg16)(cpssp, cpu, NAME_(E_AX), res1);
		break;

/* PUSH */
		// TODO in realmode trip-fault when NAME_(E_SP) == 1?
	case 0x06:	/* push ES */
	case 0x0e:	/* push CS */
	case 0x16:	/* push SS */
	case 0x1e:	/* push DS */
		NAME_(inv_op_64bit_mode)(cpssp, cpu, opcode);
		st_mode = NAME_(get_stack_mode)(cpssp, cpu);
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);
		op1 = NAME_(load_sreg)(cpssp, cpu, (opcode >> 3) & 0x7);
		NAME_(push2)(cpssp, cpu, NAME_(SEG_SS), &sp, op1, st_mode, op_mode, NAME_(usermode)(cpssp, cpu));
		NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);
		break;

	case 0x50:	/* push reg16/32 */
	case 0x51:
	case 0x52:
	case 0x53:
	case 0x54:
	case 0x55:
	case 0x56:
	case 0x57:
		st_mode = NAME_(get_stack_mode)(cpssp, cpu);
		op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
		sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);
		op1 = NAME_(load_reg)(cpssp, cpu, ((opcode >> 0) & 0x7), op_mode);
		NAME_(push2)(cpssp, cpu, NAME_(SEG_SS), &sp, op1, st_mode, op_mode, NAME_(usermode)(cpssp, cpu));
		NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);
		break;

	case 0x60:	/* pusha/pushad */
		NAME_(inv_op_64bit_mode)(cpssp, cpu, opcode);
		st_mode = NAME_(get_stack_mode)(cpssp, cpu);
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		if (0x3 == st_mode) {
			NAME_(UD)(cpssp, cpu, opcode);
		}
		NAME_(pusha)(cpssp, cpu, st_mode, op_mode, NAME_(usermode)(cpssp, cpu));
		break;

	case 0x6a:	/* push imm8 */
		st_mode = NAME_(get_stack_mode)(cpssp, cpu);
		op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
		sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);
		op1 = (data_t) (int8_t) NAME_(fetch8)(cpssp, cpu);
		NAME_(push2)(cpssp, cpu, NAME_(SEG_SS), &sp, op1, st_mode, op_mode, NAME_(usermode)(cpssp, cpu));
		NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);
		break;

	case 0x68:	/* push imm16/32 */
		st_mode = NAME_(get_stack_mode)(cpssp, cpu);
		op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
		sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);
		op1 = NAME_(fetch)(cpssp, cpu, op_mode - (op_mode == 3));
		op1 = NAME_(sign_ext)(op1, op_mode - (op_mode == 3), MODE_MAX);
		NAME_(push2)(cpssp, cpu, NAME_(SEG_SS), &sp, op1, st_mode, op_mode, NAME_(usermode)(cpssp, cpu));
		NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);
		break;

	case 0x9c:	/* pushf/pushfd */
		if (STATE_GET(e_vm) && STATE_GET(e_iopl) != 3) {
			NAME_(GPe)(cpssp, cpu, 0);
		}

		st_mode = NAME_(get_stack_mode)(cpssp, cpu);
		op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
		sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);
		op1 = NAME_(get_eflags_pushf)(cpssp, cpu);
		NAME_(push2)(cpssp, cpu, NAME_(SEG_SS), &sp, op1, st_mode, op_mode, NAME_(usermode)(cpssp, cpu));
		NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);
		break;

	case 0x07:	/* pop ES */
	/* case 0x0f:	   2-byte opcode prefix */
	case 0x17:	/* pop SS */
	case 0x1f:	/* pop DS */
		NAME_(inv_op_64bit_mode)(cpssp, cpu, opcode);
		st_mode = NAME_(get_stack_mode)(cpssp, cpu);
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);
		res1 = NAME_(pop2)(cpssp, cpu, NAME_(SEG_SS), &sp, st_mode, op_mode, NAME_(usermode)(cpssp, cpu));
		NAME_(store_segment)(cpssp, cpu, (opcode >> 3) & 0x7, res1);
		if (((opcode >> 3) & 0x7) == NAME_(SEG_SS)) {
			cpssp->NAME.int_delay = 1;
			cpssp->NAME.state_event = 1;
		}
		NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);
		break;

	case 0x58:	/* pop reg16/32 */
	case 0x59:
	case 0x5a:
	case 0x5b:
	case 0x5c:
	case 0x5d:
	case 0x5e:
	case 0x5f:
		st_mode = NAME_(get_stack_mode)(cpssp, cpu);
		op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
		sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);
		res1 = NAME_(pop2)(cpssp, cpu, NAME_(SEG_SS), &sp, st_mode, op_mode, NAME_(usermode)(cpssp, cpu));
		/* First: write back stack pointer; second: write back register contents. "pop %sp"! */
		NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);
		NAME_(store_reg)(cpssp, cpu, ((opcode >> 0) & 0x7), res1, op_mode);
		break;

	case 0x61:	/* popa/popad */
		NAME_(inv_op_64bit_mode)(cpssp, cpu, opcode);
		st_mode = NAME_(get_stack_mode)(cpssp, cpu);
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		if (0x3 == st_mode) {
			NAME_(UD)(cpssp, cpu, opcode);
		}
		NAME_(popa)(cpssp, cpu, st_mode, op_mode, NAME_(usermode)(cpssp, cpu));
		break;

	case 0x9d:	/* popf/popfd */
		if (STATE_GET(e_vm) && STATE_GET(e_iopl) != 3) {
			NAME_(GPe)(cpssp, cpu, 0);
		}

		st_mode = NAME_(get_stack_mode)(cpssp, cpu);
		op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
		sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);
		op1 = NAME_(pop2)(cpssp, cpu, NAME_(SEG_SS), &sp, st_mode, op_mode, NAME_(usermode)(cpssp, cpu));
		NAME_(set_eflags_popf)(cpssp, cpu, op1, op_mode);
		NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);
		break;

	case 0x62:	/* bound */
		NAME_(inv_op_64bit_mode)(cpssp, cpu, opcode);
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		NAME_(get_modrm)(cpssp, cpu);
		if (0x3 == cpu->mod) {
			NAME_(UD)(cpssp, cpu, opcode);
		}
		op1 = NAME_(load_reg)(cpssp, cpu, cpu->reg_opex, op_mode);
		op1 = NAME_(sign_ext)(op1, op_mode, MODE_MAX);
		op2 = NAME_(load_mem)(cpssp, cpu, op_mode, 0);
		op2 = NAME_(sign_ext)(op2, op_mode, MODE_MAX);
		cpu->addr += 1 << op_mode;
		op3 = NAME_(load_mem)(cpssp, cpu, op_mode, 0);
		op3 = NAME_(sign_ext)(op3, op_mode, MODE_MAX);
		NAME_(bound)(cpssp, cpu, op1, op2, op3);
		break;

/* ARPL / MOVSXD */
	case 0x63:	/* arpl / movsxd r32/64, rm32 */
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
		if (NAME_(get_lm_mode)(cpssp, cpu)) {
			op_mode = NAME_(get_op_mode1)(cpssp, cpu);
			NAME_(get_modrm)(cpssp, cpu);
			res1 = (data_t) (int32_t) NAME_(load_rm)(cpssp, cpu, 2, 0);
			/* FIXME manual: shouldn't be used without rex.w... correct like this?
			 * pre_op:32 -> 16, possible here, what about real hw */
			NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, res1, op_mode);
		}
#endif
		if (STATE_GET(e_vm)) {
			NAME_(UD)(cpssp, cpu, opcode);
		}
		NAME_(get_modrm)(cpssp, cpu);
		op1 = NAME_(load_rm16)(cpssp, cpu, 1);
		op2 = NAME_(load_reg16)(cpssp, cpu, cpu->reg_opex);
		res1 = NAME_(arpl)(cpssp, cpu, op1, op2);
		if (STATE_GET(e_zf)) {
			NAME_(store_rm16)(cpssp, cpu, res1);
		}
		break;

/* LAHF, SAHF */
	case 0x9e:	/* sahf */
		op1 = NAME_(load_reg8)(cpssp, cpu, NAME_(AH));
		NAME_(set_eflags8)(cpssp, cpu, op1);
		break;

	case 0x9f:	/* lahf */
		res1 = NAME_(get_eflags)(cpssp, cpu);
		NAME_(store_reg8)(cpssp, cpu, NAME_(AH), res1);
		break;

/* JMP */
	case 0xe9:	/* jmp rel16/32 */
		op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
		op1 = NAME_(fetch)(cpssp, cpu, op_mode - (op_mode == 3));
		op1 = NAME_(sign_ext)(op1, op_mode - (op_mode == 3), MODE_MAX);
		NAME_(jmp)(cpssp, cpu, op1, op_mode);
		break;

	case 0xea:	/* ljmp ptr16:16/32 */
		NAME_(inv_op_64bit_mode)(cpssp, cpu, opcode);
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		op1 = NAME_(fetch)(cpssp, cpu, op_mode);
		op2 = NAME_(fetch16)(cpssp, cpu);
		NAME_(ljmp)(cpssp, cpu, op1, op2, op_mode);
		break;

	case 0xeb:	/* jmp rel8 */
		op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
		op1 = (data_t) (int8_t) NAME_(fetch8)(cpssp, cpu);
		NAME_(jmp)(cpssp, cpu, op1, op_mode);
		break;

	case 0xe3:	/* jcxz/jecxz rel8 */
		// TODO in lm always 32bit  sign extended
		op_mode = NAME_(get_addr_mode)(cpssp, cpu);
		op1 = (data_t) (int8_t) NAME_(fetch8)(cpssp, cpu);
		op2 = NAME_(load_reg)(cpssp, cpu, NAME_(E_CX), op_mode);
		op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
		res1 = NAME_(test_ecx)(cpssp, cpu, op2);
		NAME_(jcc)(cpssp, cpu, op1, res1, op_mode);
		break;

	case 0x70:	/* jo rel8 */
	case 0x71:	/* jno rel8 */
	case 0x72:	/* jb/jc/jnae rel8 */
	case 0x73:	/* jae/jnb/jnc rel8 */
	case 0x74:	/* je/jz rel8 */
	case 0x75:	/* jne/jnz rel8 */
	case 0x76:	/* jbe/jna rel8 */
	case 0x77:	/* ja/jnbe rel8 */
	case 0x78:	/* js rel8 */
	case 0x79:	/* jns rel8 */
	case 0x7a:	/* jp/jpe rel8 */
	case 0x7b:	/* jnp/jpo rel8 */
	case 0x7c:	/* jl/jnge rel8 */
	case 0x7d:	/* jge/jnl rel8 */
	case 0x7e:	/* jle/jng rel8 */
	case 0x7f:	/* jg/jnle rel8 */
		op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
		op1 = (data_t) (int8_t) NAME_(fetch8)(cpssp, cpu);
		res1 = NAME_(test_cc)(cpssp, cpu, (opcode >> 0) & 0xf);
		NAME_(jcc)(cpssp, cpu, op1, res1, op_mode);
		break;

	case 0x84:	/* test rm8, r8 */
	case 0x85:	/* test rm16/32, r16/32 */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		NAME_(get_modrm)(cpssp, cpu);
		op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
		op2 = NAME_(load_reg)(cpssp, cpu, cpu->reg_opex, op_mode);
		NAME_(test)(cpssp, cpu, op1, op2, op_mode);
		break;

	case 0xa8:	/* test al, imm8 */
	case 0xa9:	/* test e/ax, imm16/32 */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		op1 = NAME_(load_reg)(cpssp, cpu, NAME_(E_AX), op_mode);
		op2 = NAME_(fetch)(cpssp, cpu, op_mode);
		NAME_(test)(cpssp, cpu, op1, op2, op_mode);
		break;

	case 0x8d:	/* lea r16/32, m */
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		NAME_(get_modrm)(cpssp, cpu);
		if (cpu->mod == 3) {
			NAME_(UD)(cpssp, cpu, opcode);
		}
		res1 = NAME_(lea)(cpssp, cpu);
		NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, res1, op_mode);
		break;

	case 0x86:	/* xchg rm8, r8 / xchg r8, rm8 */
	case 0x87:	/* xchg rm16/32, r16/32 / xchg r16/32, rm16/32 */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		NAME_(get_modrm)(cpssp, cpu);
		op1 = NAME_(load_reg)(cpssp, cpu, cpu->reg_opex, op_mode);
		op2 = NAME_(load_rm)(cpssp, cpu, op_mode, 1);
		NAME_(store_rm)(cpssp, cpu, op1, op_mode);
		NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, op2, op_mode);
		break;

	case 0x90:	/* xchg e/ax, r16/32 / xchg r16/32, e/ax */
#if defined CONFIG_CPU_SSE2_SUPPORT && CONFIG_CPU_SSE2_SUPPORT
		if (cpu->pre_repe_repne == 0x2) {
			assert(0);
			break;
		}
#endif
	case 0x91:	/* xchg e/ax, r16/32 / xchg r16/32, e/ax */
	case 0x92:
	case 0x93:
	case 0x94:
	case 0x95:
	case 0x96:
	case 0x97:
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		op1 = NAME_(load_reg)(cpssp, cpu, NAME_(E_AX), op_mode);
		op2 = NAME_(load_reg)(cpssp, cpu, ((opcode >> 0) & 0x7), op_mode);
		NAME_(store_reg)(cpssp, cpu, ((opcode >> 0) & 0x7), op1, op_mode);
		NAME_(store_reg)(cpssp, cpu, NAME_(E_AX), op2, op_mode);
		break;

	case 0x8f:
		op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
		st_mode = NAME_(get_stack_mode)(cpssp, cpu);
		sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);
		res1 = NAME_(pop2)(cpssp, cpu, NAME_(SEG_SS), &sp, st_mode, op_mode, NAME_(usermode)(cpssp, cpu));
		NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);
		NAME_(get_modrm)(cpssp, cpu);
		switch (cpu->reg_opex) {
		case 0x0:	/* pop rm16/32 */
			NAME_(store_rm)(cpssp, cpu, res1, op_mode);
			break;

		default:
			NAME_(UD)(cpssp, cpu, opcode);
			break;
		}
		break;

	case 0x88:	/* mov rm8, r8 */
	case 0x89:	/* mov rm16/32, r16/32 */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		NAME_(get_modrm)(cpssp, cpu);
		res1 = NAME_(load_reg)(cpssp, cpu, cpu->reg_opex, op_mode);
		NAME_(store_rm)(cpssp, cpu, res1, op_mode);
		break;

	case 0x8a:	/* mov r8, rm8 */
	case 0x8b:	/* mov r16/32, rm16/32 */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		NAME_(get_modrm)(cpssp, cpu);
		res1 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
		NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, res1, op_mode);
		break;

	case 0x8c:	/* mov rm16, Sreg */
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		NAME_(get_modrm)(cpssp, cpu);
		if (unlikely(0x5 < cpu->reg_opex)) {
			NAME_(UD)(cpssp, cpu, opcode);
		}
		res1 = NAME_(load_sreg)(cpssp, cpu, cpu->reg_opex);
#if CONFIG_CPU <= 80586
		/* Upper 16 bits in register are undefined. */
		NAME_(store_rm)(cpssp, cpu, res1, 1);
#else
		if (3 == op_mode) {
			NAME_(store_rm)(cpssp, cpu, res1, op_mode);
		} else {
			/* Upper 16 bits in register are cleared to 0. */
			NAME_(store_rm)(cpssp, cpu, res1, 1 + (op_mode == 2 && cpu->mod == 3));
		}
#endif
		break;

	case 0x8e:	/* mov Sreg, rm16 */
		NAME_(get_modrm)(cpssp, cpu);
		if (unlikely(0x5 < cpu->reg_opex || cpu->reg_opex == NAME_(SEG_CS))) {
			NAME_(UD)(cpssp, cpu, opcode);
		}
		res1 = NAME_(load_rm16)(cpssp, cpu, 0);
		NAME_(store_segment)(cpssp, cpu, cpu->reg_opex, res1);
		if (cpu->reg_opex == NAME_(SEG_SS)) {
			cpssp->NAME.int_delay = 1;
			cpssp->NAME.state_event = 1;
		}
		break;

	case 0xa0:	/* mov al, moffs8 */
	case 0xa1:	/* mov e/ax, moffs16/32 */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		cpu->seg[0] = NAME_(SEG_DS);
		NAME_(check_seg_over)(cpssp, cpu);
		cpu->addr = NAME_(fetch)(cpssp, cpu, NAME_(get_addr_mode)(cpssp, cpu));
		res1 = NAME_(load_mem)(cpssp, cpu, op_mode, 0);
		NAME_(store_reg)(cpssp, cpu, NAME_(E_AX), res1, op_mode);
		break;

	case 0xa2:	/* mov moffs8, al */
	case 0xa3:	/* mov moffs16/32, e/ax */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		cpu->seg[0] = NAME_(SEG_DS);
		NAME_(check_seg_over)(cpssp, cpu);
		cpu->addr = NAME_(fetch)(cpssp, cpu, NAME_(get_addr_mode)(cpssp, cpu));
		res1 = NAME_(load_reg)(cpssp, cpu, NAME_(E_AX), op_mode);
		NAME_(store_mem)(cpssp, cpu, res1, op_mode);
		break;

	case 0xb0:	/* mov r8, imm8 */
	case 0xb1:
	case 0xb2:
	case 0xb3:
	case 0xb4:
	case 0xb5:
	case 0xb6:
	case 0xb7:
	case 0xb8:	/* mov r16/32, imm16/32 */
	case 0xb9:
	case 0xba:
	case 0xbb:
	case 0xbc:
	case 0xbd:
	case 0xbe:
	case 0xbf:
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode >> 3);
		res1 = NAME_(fetch)(cpssp, cpu, op_mode);
		NAME_(store_reg)(cpssp, cpu, ((opcode >> 0) & 0x7), res1, op_mode);
		break;


/* CBW/CWDE */
	case 0x98:	/* cbw/cwde */
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		res1 = NAME_(load_reg)(cpssp, cpu, NAME_(E_AX), op_mode - 1);
		res1 = NAME_(sign_ext)(res1, op_mode - 1, op_mode);
		NAME_(store_reg)(cpssp, cpu, NAME_(E_AX), res1, op_mode);
		break;


/* CWD/CDQ */
	case 0x99:	/* cwd/cdq */
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		op1 = NAME_(load_reg)(cpssp, cpu, NAME_(E_AX), op_mode);
		res1 = NAME_(cwd)(cpssp, cpu, op1, op_mode);
		NAME_(store_reg)(cpssp, cpu, NAME_(E_DX), res1, op_mode);
		break;


/* CALL */
	case 0x9a:	/* call ptr16:16/32 */
		NAME_(inv_op_64bit_mode)(cpssp, cpu, opcode);
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		op1 = NAME_(fetch)(cpssp, cpu, op_mode);
		op2 = NAME_(fetch16)(cpssp, cpu);
		NAME_(lcall)(cpssp, cpu, op1, op2, op_mode);
		break;

	case 0xe8:	/* call rel16/32 */
		op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
		op1 = NAME_(fetch)(cpssp, cpu, op_mode - (op_mode == 3));
		op1 = NAME_(sign_ext)(op1, op_mode - (op_mode == 3), MODE_MAX);
		NAME_(push)(cpssp, cpu, cpu->eip, op_mode, NAME_(usermode)(cpssp, cpu));
		NAME_(jmp)(cpssp, cpu, op1, op_mode);
		break;


/* RET */
	case 0xc2:	/* ret n */
		op1 = NAME_(fetch16)(cpssp, cpu);
		NAME_(retn)(cpssp, cpu, op1);
		break;

	case 0xc3:	/* ret */
		op1 = 0;
		NAME_(retn)(cpssp, cpu, op1);
		break;

	case 0xca:	/* lret n */
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		op1 = NAME_(fetch16)(cpssp, cpu);
		NAME_(lretn)(cpssp, cpu, op1, op_mode);
		break;

	case 0xcb:	/* lret */
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		op1 = 0;
		NAME_(lretn)(cpssp, cpu, op1, op_mode);
		break;

	case 0xa4:	/* movsb / movs m8 */
	case 0xa5:	/* movsw/movsd / movs m16/32 */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		cpu->seg[0] = NAME_(SEG_DS);
		NAME_(check_seg_over)(cpssp, cpu);
		if (! cpu->pre_repe_repne
		 || NAME_(rep_test)(cpssp, cpu)) {
			NAME_(movs)(cpssp, cpu, op_mode);
		}
		if (cpu->pre_repe_repne
		 && NAME_(rep_test)(cpssp, cpu)) {
			NAME_(rep)(cpssp, cpu);
			NAME_(rep_dec)(cpssp, cpu);
		}
		break;

	case 0xa6:	/* cmpsb / cmps m8, m8 */
	case 0xa7:	/* cmpsw/cmpsd / cmps m16/32, m16/32 */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		cpu->seg[0] = NAME_(SEG_DS);
		NAME_(check_seg_over)(cpssp, cpu);
		if (! cpu->pre_repe_repne
		 || NAME_(rep_test)(cpssp, cpu)) {
			NAME_(cmps)(cpssp, cpu, op_mode);
		}
		if (cpu->pre_repe_repne
		 && NAME_(rep_test)(cpssp, cpu)) {
			NAME_(repn_z)(cpssp, cpu);
			NAME_(rep_dec)(cpssp, cpu);
		}
		break;

	case 0xaa:	/* stos m8 */
	case 0xab:	/* stos m16/32 */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		if (! cpu->pre_repe_repne
		 || NAME_(rep_test)(cpssp, cpu)) {
			NAME_(stos)(cpssp, cpu, op_mode);
		}
		if (cpu->pre_repe_repne
		 && NAME_(rep_test)(cpssp, cpu)) {
			NAME_(rep)(cpssp, cpu);
			NAME_(rep_dec)(cpssp, cpu);
		}
		break;

	case 0xac:	/* lods m8 */
	case 0xad:	/* lods m16/32 */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		cpu->seg[0] = NAME_(SEG_DS);
		NAME_(check_seg_over)(cpssp, cpu);
		if (! cpu->pre_repe_repne
		 || NAME_(rep_test)(cpssp, cpu)) {
			NAME_(lods)(cpssp, cpu, op_mode);
		}
		if (cpu->pre_repe_repne
		 && NAME_(rep_test)(cpssp, cpu)) {
			NAME_(rep)(cpssp, cpu);
			NAME_(rep_dec)(cpssp, cpu);
		}
		break;

	case 0xae:	/* scasb / scas m8 */
	case 0xaf:	/* scasw/scasd / scas m16/32 */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		cpu->seg[0] = NAME_(SEG_DS);
		NAME_(check_seg_over)(cpssp, cpu);
		if (! cpu->pre_repe_repne
		 || NAME_(rep_test)(cpssp, cpu)) {
			NAME_(scas)(cpssp, cpu, op_mode);
		}
		if (cpu->pre_repe_repne
		 && NAME_(rep_test)(cpssp, cpu)) {
			NAME_(repn_z)(cpssp, cpu);
			NAME_(rep_dec)(cpssp, cpu);
		}
		break;

	case 0x6c:	/* ins m8 */
	case 0x6d:	/* ins m16/32 */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		if (! cpu->pre_repe_repne
		 || NAME_(rep_test)(cpssp, cpu)) {
			NAME_(ins)(cpssp, cpu, op_mode);
		}
		if (cpu->pre_repe_repne
		 && NAME_(rep_test)(cpssp, cpu)) {
			NAME_(rep)(cpssp, cpu);
			NAME_(rep_dec)(cpssp, cpu);
		}
		break;

	case 0x6e:	/* outs m8 */
	case 0x6f:	/* outs m16/32 */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		cpu->seg[0] = NAME_(SEG_DS);
		NAME_(check_seg_over)(cpssp, cpu);
		if (! cpu->pre_repe_repne
		 || NAME_(rep_test)(cpssp, cpu)) {
			NAME_(outs)(cpssp, cpu, op_mode);
		}
		if (cpu->pre_repe_repne
		 && NAME_(rep_test)(cpssp, cpu)) {
			NAME_(rep)(cpssp, cpu);
			NAME_(rep_dec)(cpssp, cpu);
		}
		break;

	case 0xc0:	/* sal/shl/sar/shr rm8, imm8 */
	case 0xc1:	/* sal/shl/sar/shr rm16/32, imm8 */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		NAME_(get_modrm)(cpssp, cpu);
		op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 1);
		op2 = NAME_(fetch8)(cpssp, cpu);
		if (cpu->reg_opex == 6) {
			NAME_(UD)(cpssp, cpu, opcode);
		}
		res1 = NAME_(alu_bit)(cpssp, cpu, op1, op2, op_mode);
		NAME_(store_rm)(cpssp, cpu, res1, op_mode);
		break;

	case 0xd0:	/* sal/shl/sar/shr rm8, 1 */
	case 0xd1:	/* sal/shl/sar/shr rm16/32, 1 */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		NAME_(get_modrm)(cpssp, cpu);
		op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 1);
		op2 = 1;
		if (cpu->reg_opex == 6) {
			NAME_(UD)(cpssp, cpu, opcode);
		}
		res1 = NAME_(alu_bit)(cpssp, cpu, op1, op2, op_mode);
		NAME_(store_rm)(cpssp, cpu, res1, op_mode);
		break;

	case 0xd2:	/* sal/shl/sar/shr rm8, cl */
	case 0xd3:	/* sal/shl/sar/shr rm16/32, cl */
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		NAME_(get_modrm)(cpssp, cpu);
		op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 1);
		op2 = NAME_(load_reg8)(cpssp, cpu, NAME_(CL));
		if (cpu->reg_opex == 6) {
			NAME_(UD)(cpssp, cpu, opcode);
		}
		res1 = NAME_(alu_bit)(cpssp, cpu, op1, op2, op_mode);
		NAME_(store_rm)(cpssp, cpu, res1, op_mode);
		break;

	case 0xc6:
	case 0xc7:
		NAME_(get_modrm)(cpssp, cpu);
		switch (cpu->reg_opex) {
		case 0x0:	/* mov rm8, imm8 */
				/* mov rm16/32, imm16/32 */
			op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
			res1 = NAME_(fetch)(cpssp, cpu, op_mode);
			NAME_(store_rm)(cpssp, cpu, res1, op_mode);
			break;

		default:
			NAME_(UD)(cpssp, cpu, opcode);
			break;
		}
		break;

	case 0xc8:	/* enter imm16, imm8 */
		op1 = NAME_(fetch16)(cpssp, cpu);
		op2 = NAME_(fetch8)(cpssp, cpu);
		NAME_(enter)(cpssp, cpu, op1, op2);
		break;

	case 0xc9:	/* leave */
		NAME_(leave)(cpssp, cpu);
		break;

	case 0xcc:	/* int3 */
		cpssp->process.inst_cnt++;
		NAME_(intn)(cpssp, cpu, 3, 0);
		break;

	case 0xcd:	/* int imm8 */
		op1 = NAME_(fetch8)(cpssp, cpu);

		if (STATE_GET(e_vm) && STATE_GET(e_iopl) != 3) {
			NAME_(GPe)(cpssp, cpu, 0);
		}

		cpssp->process.inst_cnt++;

#if DEBUG_CONTROL_FLOW
		if (2 <= DEBUG_CONTROL_FLOW
		 || loglevel) {
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
			if (STATE_GET(lm_active_msr)) {
				fprintf(stderr, "%lld: Exception 0x%02x at %016llx",
						cpssp->process.inst_cnt,
						op1,
						STATE_GET(eip));
			} else
#endif
			{
				fprintf(stderr, "%lld: Exception 0x%02x at %08llx (%08llx:%08llx)",
						cpssp->process.inst_cnt,
						op1,
						base + STATE_GET(eip),
						base,
						STATE_GET(eip));
			}
			NAME_(dump)(cpssp, cpu);
		}
#endif

		NAME_(intn)(cpssp, cpu, op1, 0);
		break;

	case 0xce:	/* into */
		NAME_(inv_op_64bit_mode)(cpssp, cpu, opcode);
		if (STATE_GET(e_of)) {
			cpssp->process.inst_cnt++;
			NAME_(intn)(cpssp, cpu, 4, 0);
		}
		break;

/* IRET */
	case 0xcf:	/* iret */
		if (STATE_GET(e_vm) && STATE_GET(e_iopl) != 3) {
			NAME_(GPe)(cpssp, cpu, 0);
		}

		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
#if DEBUG_CONTROL_FLOW
		if (2 <= DEBUG_CONTROL_FLOW
		 || loglevel) {
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
			if (STATE_GET(lm_active_msr)) {
				fprintf(stderr, "0: iret at %016lx (%08lx:%08lx)",
						STATE_GET(eip));
			} else
#endif
			{
				fprintf(stderr, "0: iret at %08lx (%08lx:%08lx)",
						base + STATE_GET(eip),
						base,
						STATE_GET(eip));
			}
			NAME_(dump)(cpssp, cpu);
		}
#endif
		NAME_(iret)(cpssp, cpu, op_mode);
		break;

/* LDS/LES */
	case 0xc4:	/* les r16/32, m16:16/32 */
		NAME_(inv_op_64bit_mode)(cpssp, cpu, opcode);
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		NAME_(get_modrm)(cpssp, cpu);
		if (cpu->mod == 3) {
			NAME_(UD)(cpssp, cpu, opcode);
		}
		res1 = NAME_(load_mem_fpointer)(cpssp, cpu, &res2, op_mode);
		NAME_(store_segment)(cpssp, cpu, NAME_(SEG_ES), res2);
		NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, res1, op_mode);
		break;

	case 0xc5:	/* lds r16/32, m16:16/32 */
		NAME_(inv_op_64bit_mode)(cpssp, cpu, opcode);
		op_mode = NAME_(get_op_mode1)(cpssp, cpu);
		NAME_(get_modrm)(cpssp, cpu);
		if (cpu->mod == 3) {
			NAME_(UD)(cpssp, cpu, opcode);
		}
		res1 = NAME_(load_mem_fpointer)(cpssp, cpu, &res2, op_mode);
		NAME_(store_segment)(cpssp, cpu, NAME_(SEG_DS), res2);
		NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, res1, op_mode);
		break;

/* XLAT */
	case 0xd7:	/* xlat/xlatb */
		op_mode = NAME_(get_addr_mode)(cpssp, cpu);
		cpu->seg[0] = NAME_(SEG_DS);
		NAME_(check_seg_over)(cpssp, cpu);
		res1 = NAME_(xlat)(cpssp, cpu, op_mode);
		NAME_(store_reg8)(cpssp, cpu, NAME_(AL), res1);
		break;

/* FPU */
	case 0xd8:	/* ESC */
	case 0xd9:
	case 0xda:
	case 0xdb:
	case 0xdc:
	case 0xdd:
	case 0xde:
	case 0xdf:
		if (STATE_GET(cr0_em) || STATE_GET(cr0_ts)) {
			NAME_(NM)(cpssp, cpu);
		}
		NAME_(get_modrm)(cpssp, cpu);

		opcode = ((opcode & 0x7) << 8)
				| (cpu->mod << 6)
				| (cpu->reg_opex << 3)
				| (cpu->rm << 0);

		if (cpu->mod != 0x3) {
			uint32_t addr;

			addr = cpu->addr;

			switch ((((opcode >> 8) & 0x7) << 3) | ((opcode >> 3) & 0x7)) {
#define _(x, y) (((x & 7) << 3) | y)
			case _(0xd8, 0x0): /* fadd m32fp */
			case _(0xd8, 0x1): /* fmul m32fp */
			case _(0xd8, 0x2): /* fcom m32fp */
			case _(0xd8, 0x3): /* fcomp m32fp */
			case _(0xd8, 0x4): /* fsub m32fp */
			case _(0xd8, 0x5): /* fsubr m32fp */
			case _(0xd8, 0x6): /* fdiv m32fp */
			case _(0xd8, 0x7): /* fdivr m32fp */

			case _(0xda, 0x0): /* fiadd m32int */
			case _(0xda, 0x1): /* fimul m32int */
			case _(0xda, 0x2): /* ficom m32int */
			case _(0xda, 0x3): /* ficomp m32int */
			case _(0xda, 0x4): /* fisub m32int */
			case _(0xda, 0x5): /* fisubr m32int */
			case _(0xda, 0x6): /* fidiv m32int */
			case _(0xda, 0x7): /* fidivr m32int */

			case _(0xd9, 0x0): /* fld m32fp */
			case _(0xd9, 0x1): /* (reserved) */

			case _(0xdb, 0x0): /* fild m32int */
			case _(0xdb, 0x1): /* (reserved) */
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				op1 = NAME_(load_mem32)(cpssp, cpu, 0);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op1);
				break;

			case _(0xd9, 0x2): /* fst m32fp */
			case _(0xd9, 0x3): /* fstp m32fp */

			case _(0xdb, 0x2): /* fist m32int */
			case _(0xdb, 0x3): /* fistp m32int */
				(void) NAME_(load_mem32)(cpssp, cpu, 2);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res1);
				NAME_(store_mem32)(cpssp, cpu, res1);
				break;

			case _(0xdc, 0x0): /* fadd m64fp */
			case _(0xdc, 0x1): /* fmul m64fp */
			case _(0xdc, 0x2): /* fcom m64fp */
			case _(0xdc, 0x3): /* fcomp m64fp */
			case _(0xdc, 0x4): /* fsub m64fp */
			case _(0xdc, 0x5): /* fsubr m64fp */
			case _(0xdc, 0x6): /* fdiv m64fp */
			case _(0xdc, 0x7): /* fdivr m64fp */

			case _(0xdd, 0x0): /* fld m64fp */
			case _(0xdd, 0x1): /* (reserved) */
				op1 = NAME_(load_mem32)(cpssp, cpu, 0);
				cpu->addr += 4;
				op2 = NAME_(load_mem32)(cpssp, cpu, 0);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op1);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op2);
				break;

			case _(0xdd, 0x2): /* fst m64fp */
			case _(0xdd, 0x3): /* fstp m64fp */
				(void) NAME_(load_mem32)(cpssp, cpu, 2);
				cpu->addr += 4;
				(void) NAME_(load_mem32)(cpssp, cpu, 2);
				cpu->addr -= 4;
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res1);
				DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res2);
				NAME_(store_mem32)(cpssp, cpu, res1);
				cpu->addr += 4;
				NAME_(store_mem32)(cpssp, cpu, res2);
				break;

			case _(0xde, 0x0): /* fiadd m16int */
			case _(0xde, 0x1): /* fimul m16int */
			case _(0xde, 0x2): /* ficom m16int */
			case _(0xde, 0x3): /* ficomp m16int */
			case _(0xde, 0x4): /* fisub m16int */
			case _(0xde, 0x5): /* fisubr m16int */
			case _(0xde, 0x6): /* fidiv m16int */
			case _(0xde, 0x7): /* fidivr m16int */

			case _(0xdf, 0x0): /* fild m16int */
			case _(0xdf, 0x1): /* (reserved) */
				op1 = NAME_(load_mem16)(cpssp, cpu, 0);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op1);
				break;

			case _(0xdf, 0x2): /* fist m16int */
			case _(0xdf, 0x3): /* fistp m16int */
				(void) NAME_(load_mem16)(cpssp, cpu, 2);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res1);
				NAME_(store_mem16)(cpssp, cpu, res1);
				break;

			case _(0xd9, 0x4): /* fldenv m14/28byte */
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				op_mode = NAME_(get_op_mode1)(cpssp, cpu);
				if (op_mode == 3)
					assert(0); /* TODO 64bit mode */
				op_mode--; /* FIXME temporary fix */
				for (i = 0; i < 3; i++) {
					if (op_mode == 0) {
						op1 = NAME_(load_mem16)(cpssp, cpu, 0);
						cpu->addr += 2;
					} else {
						op1 = NAME_(load_mem32)(cpssp, cpu, 0);
						cpu->addr += 4;
					}
					DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op1);
				}
				if (op_mode == 0) {
					op1 = NAME_(load_mem16)(cpssp, cpu, 0);
					/* ... = op1; */ /* FIXME */
					cpu->addr += 2;
				} else {
					op1 = NAME_(load_mem32)(cpssp, cpu, 0);
					/* ... = op1; */ /* FIXME */
					cpu->addr += 4;
				}
				if (op_mode == 0) {
					op1 = NAME_(load_mem16)(cpssp, cpu, 0);
					/* ... = op1; */ /* FIXME */
					cpu->addr += 2;
				} else {
					op1 = NAME_(load_mem32)(cpssp, cpu, 0);
					/* ... = op1; */ /* FIXME */
					cpu->addr += 4;
				}
				if (op_mode == 0) {
					op1 = NAME_(load_mem16)(cpssp, cpu, 0);
					/* ... = op1; */ /* FIXME */
					cpu->addr += 2;
				} else {
					op1 = NAME_(load_mem32)(cpssp, cpu, 0);
					/* ... = op1; */ /* FIXME */
					cpu->addr += 4;
				}
				if (op_mode == 0) {
					op1 = NAME_(load_mem16)(cpssp, cpu, 0);
					/* ... = op1; */ /* FIXME */
					cpu->addr += 2;
				} else {
					op1 = NAME_(load_mem32)(cpssp, cpu, 0);
					/* ... = op1; */ /* FIXME */
					cpu->addr += 4;
				}
				break;

			case _(0xd9, 0x6): /* fstenv m14/28byte */
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				op_mode = NAME_(get_op_mode1)(cpssp, cpu);
				if (op_mode == 3)
					assert(0); /* TODO 64bit mode */
				op_mode--; /* FIXME temporary fix */
				for (i = 0; i < 3; i++) {
					DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res1);
					if (op_mode == 0) {
						NAME_(store_mem16)(cpssp, cpu, res1);
						cpu->addr += 2;
					} else {
						NAME_(store_mem32)(cpssp, cpu, res1);
						cpu->addr += 4;
					}
				}

				if (! STATE_GET(cr0_pe)
				 || STATE_GET(e_vm)) {
					/* Real-Address Mode Format */
					res1 = cpssp->NAME.fpu_eip & 0xffff;
				} else {
					/* Protected Mode Format */
					res1 = cpssp->NAME.fpu_eip;
				}
				if (op_mode == 0) {
					NAME_(store_mem16)(cpssp, cpu, res1);
					cpu->addr += 2;
				} else {
					NAME_(store_mem32)(cpssp, cpu, res1);
					cpu->addr += 4;
				}

				if (! STATE_GET(cr0_pe)
				 || STATE_GET(e_vm)) {
					/* Real-Address Mode Format */
					res1 = (cpssp->NAME.fpu_eip << 12)
							| (cpssp->NAME.fpu_opcode << 0);
				} else {
					/* Protected Mode Format */
					res1 = (cpssp->NAME.fpu_opcode << 16)
							| cpssp->NAME.fpu_cs;
				}
				if (op_mode == 0) {
					NAME_(store_mem16)(cpssp, cpu, res1);
					cpu->addr += 2;
				} else {
					NAME_(store_mem32)(cpssp, cpu, res1);
					cpu->addr += 4;
				}

				if (! STATE_GET(cr0_pe)
				 || STATE_GET(e_vm)) {
					/* Real-Address Mode Format */
					res1 = (0xffff << 16)
							| ((cpssp->NAME.fpu_addr & 0xffff) << 0);
				} else {
					/* Protected Mode Format */
					res1 = cpssp->NAME.fpu_addr;
				}
				if (op_mode == 0) {
					NAME_(store_mem16)(cpssp, cpu, res1);
					cpu->addr += 2;
				} else {
					NAME_(store_mem32)(cpssp, cpu, res1);
					cpu->addr += 4;
				}

				if (! STATE_GET(cr0_pe)
				 || STATE_GET(e_vm)) {
					/* Real-Address Mode Format */
					res1 = cpssp->NAME.fpu_addr << 12;
				} else {
					/* Protected Mode Format */
					res1 = (0xffff << 16)
							| cpssp->NAME.fpu_ds;
				}
				if (op_mode == 0) {
					NAME_(store_mem16)(cpssp, cpu, res1);
					cpu->addr += 2;
				} else {
					NAME_(store_mem32)(cpssp, cpu, res1);
					cpu->addr += 4;
				}
				break;

			case _(0xd9, 0x5): /* fldcw m2byte */
				op1 = NAME_(load_mem16)(cpssp, cpu, 0);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op1);
				break;

			case _(0xd9, 0x7): /* fstcw m2byte */
				(void) NAME_(load_mem16)(cpssp, cpu, 2);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res1);
				NAME_(store_mem16)(cpssp, cpu, res1);
				break;

			case _(0xdb, 0x4): /* (reserved) */
				/* FIXME */
				break;

			case _(0xdb, 0x6): /* (reserved) */
				/* FIXME */
				break;

			case _(0xdb, 0x5): /* fld m80fp */
				op1 = NAME_(load_mem32)(cpssp, cpu, 0);
				cpu->addr += 4;
				op2 = NAME_(load_mem32)(cpssp, cpu, 0);
				cpu->addr += 4;
				op3 = NAME_(load_mem16)(cpssp, cpu, 0);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op1);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op2);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op3);
				break;

			case _(0xdb, 0x7): /* fstp m80fp */
				(void) NAME_(load_mem32)(cpssp, cpu, 2);
				cpu->addr += 4;
				(void) NAME_(load_mem32)(cpssp, cpu, 2);
				cpu->addr += 4;
				(void) NAME_(load_mem16)(cpssp, cpu, 2);
				cpu->addr -= 8;
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res1);
				DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res2);
				DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res3);
				NAME_(store_mem32)(cpssp, cpu, res1);
				cpu->addr += 4;
				NAME_(store_mem32)(cpssp, cpu, res2);
				cpu->addr += 4;
				NAME_(store_mem16)(cpssp, cpu, res3);
				break;

			case _(0xdd, 0x4): /* frstor m14+80/28+80byte */
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				op_mode = NAME_(get_op_mode1)(cpssp, cpu);
				if (op_mode == 3)
					assert(0); /* TODO 64bit mode */
				op_mode--; /* FIXME temporary fix */
				for (i = 0; i < 3; i++) {
					if (op_mode == 0) {
						op1 = NAME_(load_mem16)(cpssp, cpu, 0);
						cpu->addr += 2;
					} else {
						op1 = NAME_(load_mem32)(cpssp, cpu, 0);
						cpu->addr += 4;
					}
					DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op1);
				}
				if (op_mode == 0) {
					/* FIXME */
					assert(0);
				} else {
					op1 = NAME_(load_mem32)(cpssp, cpu, 0);
					/* ... = op1; */ /* FIXME */
					cpu->addr += 4;
					op1 = NAME_(load_mem32)(cpssp, cpu, 0);
					/* ... = op1; */ /* FIXME */
					cpu->addr += 4;
					op1 = NAME_(load_mem32)(cpssp, cpu, 0);
					/* ... = op1; */ /* FIXME */
					cpu->addr += 4;
					op1 = NAME_(load_mem32)(cpssp, cpu, 0);
					/* ... = op1; */ /* FIXME */
					cpu->addr += 4;
				}
				for (i = 0; i < 8; i++) {
					op1 = NAME_(load_mem32)(cpssp, cpu, 0);
					cpu->addr += 4;
					DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op1);
					op1 = NAME_(load_mem32)(cpssp, cpu, 0);
					cpu->addr += 4;
					DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op1);
					op1 = NAME_(load_mem16)(cpssp, cpu, 0);
					cpu->addr += 2;
					DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op1);
				}
				break;

			case _(0xdd, 0x6): /* fsave m14+80/28+80byte */
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				op_mode = NAME_(get_op_mode1)(cpssp, cpu);
				if (op_mode == 3)
					assert(0); /* TODO 64bit mode */
				op_mode--; /* FIXME temporary fix */
				for (i = 0; i < 3; i++) {
					DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res1);
					if (op_mode == 0) {
						NAME_(store_mem16)(cpssp, cpu, res1);
						cpu->addr += 2;
					} else {
						NAME_(store_mem32)(cpssp, cpu, res1);
						cpu->addr += 4;
					}
				}
				if (op_mode == 0) {
					assert(0); /* FIXME */
				} else {
					res1 = cpssp->NAME.fpu_eip;
					NAME_(store_mem32)(cpssp, cpu, res1);
					cpu->addr += 4;
					res1 = (cpssp->NAME.fpu_opcode << 16)
							| cpssp->NAME.fpu_cs;
					NAME_(store_mem32)(cpssp, cpu, res1);
					cpu->addr += 4;
					res1 = cpssp->NAME.fpu_addr;
					NAME_(store_mem32)(cpssp, cpu, res1);
					cpu->addr += 4;
					res1 = (0xffff << 16)
							| cpssp->NAME.fpu_ds;
					NAME_(store_mem32)(cpssp, cpu, res1);
					cpu->addr += 4;
				}
				for (i = 0; i < 8; i++) {
					DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res1);
					NAME_(store_mem32)(cpssp, cpu, res1);
					cpu->addr += 4;
					DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res1);
					NAME_(store_mem32)(cpssp, cpu, res1);
					cpu->addr += 4;
					DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res1);
					NAME_(store_mem16)(cpssp, cpu, res1);
					cpu->addr += 2;
				}
				break;

			case _(0xdd, 0x5): /* (reserved) */
				op1 = NAME_(load_mem16)(cpssp, cpu, 0);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op1);
				break;

			case _(0xdd, 0x7): /* fsts m2byte */
				(void) NAME_(load_mem16)(cpssp, cpu, 2);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res1);
				NAME_(store_mem16)(cpssp, cpu, res1);
				break;

			case _(0xdf, 0x4): /* fbld m80bcd */
				op1 = NAME_(load_mem32)(cpssp, cpu, 0);
				cpu->addr += 4;
				op2 = NAME_(load_mem32)(cpssp, cpu, 0);
				cpu->addr += 4;
				op3 = NAME_(load_mem16)(cpssp, cpu, 0);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op1);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op2);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op3);
				break;

			case _(0xdf, 0x6): /* fbstp m80bcd */
				(void) NAME_(load_mem32)(cpssp, cpu, 2);
				cpu->addr += 4;
				(void) NAME_(load_mem32)(cpssp, cpu, 2);
				cpu->addr += 4;
				(void) NAME_(load_mem16)(cpssp, cpu, 2);
				cpu->addr -= 8;
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res1);
				DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res2);
				DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res3);
				NAME_(store_mem32)(cpssp, cpu, res1);
				cpu->addr += 4;
				NAME_(store_mem32)(cpssp, cpu, res2);
				cpu->addr += 4;
				NAME_(store_mem16)(cpssp, cpu, res3);
				break;

			case _(0xdf, 0x5): /* fild m64int */
				op1 = NAME_(load_mem32)(cpssp, cpu, 0);
				cpu->addr += 4;
				op2 = NAME_(load_mem32)(cpssp, cpu, 0);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op1);
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0004, op2);
				break;

			case _(0xdf, 0x7): /* fistp m64int */
				(void) NAME_(load_mem32)(cpssp, cpu, 2);
				cpu->addr += 4;
				(void) NAME_(load_mem32)(cpssp, cpu, 2);
				cpu->addr -= 4;
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res1);
				DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res2);
				NAME_(store_mem32)(cpssp, cpu, res1);
				cpu->addr += 4;
				NAME_(store_mem32)(cpssp, cpu, res2);
				break;

			default:
				/* Cannot happen. */
				assert(0);
#undef _
			}

			cpssp->NAME.fpu_addr = addr;
			cpssp->NAME.fpu_ds = STATE_GET(selector[cpu->seg[cpu->seg_override]]);

		} else {
			switch ((((opcode >> 8) & 0x7) << 3) | ((opcode >> 3) & 0x7)) {
#define _(x)	((((x >> 8) & 0x7) << 3) | ((x >> 3) & 0x7))
			case _(0xd8c0): /* fadd */
			case _(0xd8c8): /* fmul */
			case _(0xd8d0): /* fcom */
			case _(0xd8d8): /* fcomp */
			case _(0xd8e0): /* fsub */
			case _(0xd8e8): /* fsubr */
			case _(0xd8f0): /* fdiv */
			case _(0xd8f8): /* fdivr */

			case _(0xd9c0): /* fld */
			case _(0xd9c8): /* fxch */
			case _(0xd9d0): /* fnop(?) */
			case _(0xd9d8): /* (reserved) */
			case _(0xd9e0): /* fchs, fabs, ftst, fxam */
			case _(0xd9e8): /* fld1, fldl2t, fldl2e, fldpi, ... */
			case _(0xd9f0): /* f2xm1, fyl2x, fptan, fpatan, ... */
			case _(0xd9f8): /* fprem, fyl2xp1, fsqrt, fsincos, ... */

			case _(0xdac0): /* P6: fcmovb */
			case _(0xdac8): /* P6: fcmove */
			case _(0xdad0): /* P6: fcmovbe */
			case _(0xdad8): /* P6: fcmovu */
			case _(0xdae0): /* (reserved) */
			case _(0xdae8): /* fucompp(?) */
			case _(0xdaf0): /* (reserved) */
			case _(0xdaf8): /* (reserved) */

			case _(0xdbc0): /* P6: fcmovnb */
			case _(0xdbc8): /* P6: fcmovne */
			case _(0xdbd0): /* P6: fcmovnbe */
			case _(0xdbd8): /* P6: fcmovnu */
			case _(0xdbe0):	/* feni (287 only) */
					/* fdisi (287 only) */
					/* fnclex */
					/* fninit */
					/* fsetpm (287 only) */
					/* (reserved) */
					/* (reserved) */
					/* (reserved) */
			case _(0xdbe8): /* P6: fucomi */
			case _(0xdbf0): /* P6: fcomi */
			case _(0xdbf8): /* (reserved) */

			case _(0xdcc0): /* fadd */
			case _(0xdcc8): /* fmul */
			case _(0xdcd0): /* (reserved) */
			case _(0xdcd8): /* (reserved) */
			case _(0xdce0): /* fsubr */
			case _(0xdce8): /* fsub */
			case _(0xdcf0): /* fdivr */
			case _(0xdcf8): /* fdiv */

			case _(0xddc0): /* ffree */
			case _(0xddc8): /* ffreep (undocumented) */
			case _(0xddd0): /* fst */
			case _(0xddd8): /* fstp */
			case _(0xdde0): /* fucom */
			case _(0xdde8): /* fucomp */
			case _(0xddf0): /* (reserved) */
			case _(0xddf8): /* (reserved) */

			case _(0xdec0): /* faddp */
			case _(0xdec8): /* fmulp */
			case _(0xded0): /* (reserved) */
			case _(0xded8): /* fcompp(?) */
			case _(0xdee0): /* fsubrp */
			case _(0xdee8): /* fsubp */
			case _(0xdef0): /* fdivrp */
			case _(0xdef8): /* fdivp */

			case _(0xdfc0): /* (reserved) */
			case _(0xdfc8): /* (reserved) */
			case _(0xdfd0): /* (reserved) */
			case _(0xdfd8): /* (reserved) */
			/* case _(0xdfe0): see below */
			case _(0xdfe8): /* P6: fucomip */
			case _(0xdff0): /* P6: fcomip */
			case _(0xdff8): /* (reserved) */
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				break;

			case _(0xdfe0):	/* fsts %ax */
					/* (reserved) */
					/* (reserved) */
					/* (reserved) */
					/* (reserved) */
					/* (reserved) */
					/* (reserved) */
					/* (reserved) */
				DEFAULT_NAME_(uiowd)(cpssp, 0xffff0000, opcode);
				DEFAULT_NAME_(uiord)(cpssp, 0xffff0004, (uint32_t*) &res1);
				NAME_(store_reg16)(cpssp, cpu, NAME_(E_AX), res1);
				break;

			default:
				/* Cannot happen. */
				assert(0);
#undef _
			}

			cpssp->NAME.fpu_addr = 0;
			cpssp->NAME.fpu_ds = 0;
		}

		cpssp->NAME.fpu_opcode = opcode & 0x7ff;
		cpssp->NAME.fpu_eip = STATE_GET(eip);
		cpssp->NAME.fpu_cs = STATE_GET(selector[NAME_(SEG_CS)]);
		break;

	case 0x9b:	/* wait / fwait */
		if (STATE_GET(cr0_mp) && STATE_GET(cr0_ts)) {
			NAME_(NM)(cpssp, cpu);
		}
		if (cpssp->NAME.fpu_error) {
			NAME_(MF)(cpssp, cpu);
		}
		break;


/* LOOP/LOOPcc */
	case 0xe0:	/* loopne/loopnz rel8 */
	case 0xe1:	/* loope/loopz rel8 */
	case 0xe2:	/* loop rel8 */
		op1 = (data_t) (int8_t) NAME_(fetch8)(cpssp, cpu);
		res1 = ! NAME_(test_zf)(cpssp, cpu);
		NAME_(loopcc)(cpssp, cpu, op1, res1 | (opcode & 2), opcode & 1);
		break;

	case 0xe4:	/* in al, imm8 */
	case 0xe5:	/* in e/ax, imm8 */
		// TODO behaves the same in 64bit and non-64bit mode
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		op1 = NAME_(fetch8)(cpssp, cpu);
		res1 = NAME_(in)(cpssp, cpu, op1, op_mode);
		NAME_(store_reg)(cpssp, cpu, NAME_(E_AX), res1, op_mode);
		break;

	case 0xec:	/* in al, dx */
	case 0xed:	/* in e/ax, dx */
		// TODO behaves the same in 64bit and non-64bit mode
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		op1 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_DX));
		res1 = NAME_(in)(cpssp, cpu, op1, op_mode);
		NAME_(store_reg)(cpssp, cpu, NAME_(E_AX), res1, op_mode);
		break;

	case 0xe6:	/* out imm8, al */
	case 0xe7:	/* out imm8, e/ax */
		// TODO behaves the same in 64bit and non-64bit mode
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		op1 = NAME_(fetch8)(cpssp, cpu);
		op2 = NAME_(load_reg)(cpssp, cpu, NAME_(E_AX), op_mode);
		NAME_(out)(cpssp, cpu, op1, op2, op_mode);
		break;

	case 0xee:	/* out dx, al */
	case 0xef:	/* out dx, e/ax */
		// TODO behaves the same in 64bit and non-64bit mode
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		op1 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_DX));
		op2 = NAME_(load_reg)(cpssp, cpu, NAME_(E_AX), op_mode);
		NAME_(out)(cpssp, cpu, op1, op2, op_mode);
		break;

	case 0xf6:
	case 0xf7:
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		NAME_(get_modrm)(cpssp, cpu);
		switch (cpu->reg_opex) {
		case 0x0:	/* test rm8, imm8 */
				/* test rm16/32, imm16/32 */
			op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
			op2 = NAME_(fetch)(cpssp, cpu, op_mode);
			NAME_(test)(cpssp, cpu, op1, op2, op_mode);
			break;

		case 0x2:	/* not rm8 */
				/* not rm16/32 */
			op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 1);
			res1 = NAME_(not)(cpssp, cpu, op1);
			NAME_(store_rm)(cpssp, cpu, res1, op_mode);
			break;

		case 0x3:	/* neg rm8 */
				/* neg rm16/32 */
			op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 1);
			res1 = NAME_(neg)(cpssp, cpu, op1, op_mode);
			NAME_(store_rm)(cpssp, cpu, res1, op_mode);
			break;

		case 0x4:	/* mul rm8 (AX <- NAME_(AL) * rm8) */
				/* mul rm16 (DX:AX <- AX * rm16) */
				/* mul rm32 (EDX:EAX <- EAX * rm32) */
			op1 = NAME_(load_reg)(cpssp, cpu, NAME_(E_AX), op_mode);
			op2 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
			res1 = NAME_(mul)(cpssp, cpu, op1, op2, &res2, op_mode);
			if (op_mode == 0) {
				NAME_(store_reg)(cpssp, cpu, NAME_(E_AX), res1, 1);
			} else {
				NAME_(store_reg)(cpssp, cpu, NAME_(E_AX), res1, op_mode);
				NAME_(store_reg)(cpssp, cpu, NAME_(E_DX), res2, op_mode);
			}
			break;

		case 0x5:	/* imul rm8 (AX <- NAME_(AL) * rm8) */
				/* imul rm16 (DX:AX <- AX * rm16) */
				/* imul rm32 (EDX:EAX <- EAX * rm32) */
			op1 = NAME_(load_reg)(cpssp, cpu, NAME_(E_AX), op_mode);
			op1 = NAME_(sign_ext)(op1, op_mode, MODE_MAX);
			op2 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
			op2 = NAME_(sign_ext)(op2, op_mode, MODE_MAX);
			res1 = NAME_(imul)(cpssp, cpu, op1, op2, &res2, op_mode);
			if (op_mode == 0) {
				NAME_(store_reg)(cpssp, cpu, NAME_(E_AX), res1, 1);
			} else {
				NAME_(store_reg)(cpssp, cpu, NAME_(E_AX), res1, op_mode);
				NAME_(store_reg)(cpssp, cpu, NAME_(E_DX), res2, op_mode);
			}
			break;

		case 0x6:	/* div rm8 (AL <- AX / rm8, AH <- AX % rm8) */
				/* div rm16 (AX <- DX:AX / rm16, DX <- DX:AX % rm16) */
				/* div rm32 (EAX <- EDX:EAX / rm32, EDX <- EDX:EAX % rm32) */
			if (op_mode == 0) {
				op1 = NAME_(load_reg16)(cpssp, cpu, NAME_(E_AX));
				op2 = NAME_(load_rm8)(cpssp, cpu, 0);
				res1 = NAME_(div)(cpssp, cpu, 0, op1, op2, &res2, op_mode);
				NAME_(store_reg8)(cpssp, cpu, NAME_(AL), res1);
				NAME_(store_reg8)(cpssp, cpu, NAME_(AH), res2);
			} else {
				op1 = NAME_(load_reg)(cpssp, cpu, NAME_(E_DX), op_mode);
				op2 = NAME_(load_reg)(cpssp, cpu, NAME_(E_AX), op_mode);
				op3 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
				res1 = NAME_(div)(cpssp, cpu, op1, op2, op3, &res2, op_mode);
				NAME_(store_reg)(cpssp, cpu, NAME_(E_AX), res1, op_mode);
				NAME_(store_reg)(cpssp, cpu, NAME_(E_DX), res2, op_mode);
			}
			break;

/* IDIV */
		case 0x7:	/* idiv rm8 (AL) <- AX / rm8, (AH) <- AX % rm8) */
				/* idiv rm16 (AX <- DX:AX / rm16, DX <- DX:AX % rm16) */
				/* idiv rm32 (EAX <- EDX:EAX / rm32, EDX <- EDX:EAX % rm32) */
			if (op_mode == 0) {
				op1 = (data_t) (int16_t) NAME_(load_reg16)(cpssp, cpu, NAME_(E_AX));
				op2 = (data_t) (int8_t) NAME_(load_rm8)(cpssp, cpu, 0);
				res1 = NAME_(idiv)(cpssp, cpu, 0, op1, op2, &res2, op_mode);
				NAME_(store_reg8)(cpssp, cpu, NAME_(AL), res1);
				NAME_(store_reg8)(cpssp, cpu, NAME_(AH), res2);
			} else {
				op1 = NAME_(load_reg)(cpssp, cpu, NAME_(E_DX), op_mode);
				op1 = NAME_(sign_ext)(op1, op_mode, MODE_MAX);
				op2 = NAME_(load_reg)(cpssp, cpu, NAME_(E_AX), op_mode);
				op2 = NAME_(sign_ext)(op2, op_mode, MODE_MAX);
				op3 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
				op3 = NAME_(sign_ext)(op3, op_mode, MODE_MAX);
				res1 = NAME_(idiv)(cpssp, cpu, op1, op2, op3, &res2, op_mode);
				NAME_(store_reg)(cpssp, cpu, NAME_(E_AX), res1, op_mode);
				NAME_(store_reg)(cpssp, cpu, NAME_(E_DX), res2, op_mode);
			}
			break;

		default:
			NAME_(UD)(cpssp, cpu, opcode);
			break;
		}
		break;

/* HLT */
	case 0xf4:	/* hlt */
		if (0 < NAME_(cpl)(cpssp, cpu)) {
			NAME_(GPe)(cpssp, cpu, 0);
		}

		NAME_(hlt)(cpssp, cpu);
		break;

/* CMC */
	case 0xf5:	/* cmc */
		NAME_(cmc)(cpssp, cpu);
		break;

	case 0xf8:	/* clc */
	case 0xf9:	/* stc */
		NAME_(clc_stc)(cpssp, cpu, opcode & 1);
		break;

	case 0xfa:	/* cli */
	case 0xfb:	/* sti */
		if (STATE_GET(e_iopl) < NAME_(cpl)(cpssp, cpu)) {
			NAME_(GPe)(cpssp, cpu, 0);
		}

		NAME_(cli_sti)(cpssp, cpu, opcode & 1);
		break;

	case 0xfc:	/* cld */
	case 0xfd:	/* std */
		NAME_(cld_std)(cpssp, cpu, opcode & 1);
		break;

/* MODRM-OPCEX */
	case 0xfe:
	case 0xff:
		op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
		NAME_(get_modrm)(cpssp, cpu);
		switch (cpu->reg_opex) {
		case 0x0:	/* inc rm8 */
				/* inc rm16/32 */
		case 0x1:	/* dec rm8 */
				/* dec rm16/32 */
			op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 1);
			res1 = NAME_(inc_dec)(cpssp, cpu, op1, op_mode, cpu->reg_opex);
			NAME_(store_rm)(cpssp, cpu, res1, op_mode);
			break;

		case 0x2:	/* call rm16/32 */
			if (! op_mode) {
				NAME_(UD)(cpssp, cpu, opcode);
			}
			op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
			op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
			NAME_(push)(cpssp, cpu, cpu->eip, op_mode, NAME_(usermode)(cpssp, cpu));
			cpu->eip = op1;
			// TODO check for seg_limit
			break;

		case 0x3:	/* call m16:16/32 */
			if (! op_mode
			 || cpu->mod == 3) {
				NAME_(UD)(cpssp, cpu, opcode);
			}
			res1 = NAME_(load_mem_fpointer)(cpssp, cpu, &res2, op_mode);
			NAME_(lcall)(cpssp, cpu, res1, res2, op_mode);
			break;

		case 0x4:	/* jmp rm16/32 */
			if (! op_mode) {
				NAME_(UD)(cpssp, cpu, opcode);
			}
			op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
			op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
			cpu->eip = op1;
			// TODO check for seg_limit
			break;

		case 0x5:	/* ljmp m16:16/32 */
			if (! op_mode
			 || cpu->mod == 3) {
				NAME_(UD)(cpssp, cpu, opcode);
			}
			res1 = NAME_(load_mem_fpointer)(cpssp, cpu, &res2, op_mode);
			NAME_(ljmp)(cpssp, cpu, res1, res2, op_mode);
			break;

		case 0x6:	/* push rm16/32 */
			if (! op_mode) {
				NAME_(UD)(cpssp, cpu, opcode);
			}
			op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
			op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
			NAME_(push)(cpssp, cpu, op1, op_mode, NAME_(usermode)(cpssp, cpu));
			break;

		default:
			NAME_(UD)(cpssp, cpu, opcode);
			break;
		}
		break;

	case 0x0f:
		opcode = NAME_(fetch8)(cpssp, cpu);
		switch (opcode) {
		case 0x00:
			NAME_(get_modrm)(cpssp, cpu);

			if (unlikely(! STATE_GET(cr0_pe)
			 || STATE_GET(e_vm))) {
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
			}
			if (unlikely((cpu->reg_opex & 0x6) == 0x2
			 && NAME_(cpl)(cpssp, cpu) != 0)) {
				NAME_(GPe)(cpssp, cpu, 0);
			}

			switch (cpu->reg_opex) {
			case 0x0:	/* sldt rm16 */
				op1 = NAME_(load_sreg)(cpssp, cpu, NAME_(SEG_LDT));
				NAME_(store_rm16)(cpssp, cpu, op1);
				break;

			case 0x1:	/* str rm16 */
				op1 = NAME_(load_sreg)(cpssp, cpu, NAME_(SEG_TR));
				NAME_(store_rm16)(cpssp, cpu, op1);
				break;

			case 0x2:	/* lldt m16 */
				op2 = NAME_(load_rm16)(cpssp, cpu, 0);
				NAME_(lldt)(cpssp, cpu, op2);
				break;

			case 0x3:	/* ltr rm16 */
				op2 = NAME_(load_rm16)(cpssp, cpu, 0);
				NAME_(ltr)(cpssp, cpu, op2);
				break;

			case 0x4:	/* verr rm16 */
				op1 = NAME_(load_rm16)(cpssp, cpu, 0);
				NAME_(verr)(cpssp, cpu, op1);
				break;

			case 0x5:	/* verw rm16 */
				op1 = NAME_(load_rm16)(cpssp, cpu, 0);
				NAME_(verw)(cpssp, cpu, op1);
				break;

			default:
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
				break;
			}
			break;

		case 0x01:
			NAME_(get_modrm)(cpssp, cpu);

			if ((cpu->reg_opex & 0x2)
			 && NAME_(cpl)(cpssp, cpu) != 0) {
				NAME_(GPe)(cpssp, cpu, 0);
			}
			if (cpu->reg_opex < 4
			 && cpu->mod == 3) {
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
			}

			switch (cpu->reg_opex) {
			case 0x0:	/* sgdt m16&m32 */
				op_mode = NAME_(get_op_mode1)(cpssp, cpu);
				NAME_(sgdt)(cpssp, cpu, &descr);
				NAME_(store_limit_base)(cpssp, cpu, descr, op_mode);
				break;
			case 0x1:	/* sidt m16&m32 */
				op_mode = NAME_(get_op_mode1)(cpssp, cpu);
				NAME_(sidt)(cpssp, cpu, &descr);
				NAME_(store_limit_base)(cpssp, cpu, descr, op_mode);
				break;
			case 0x2:	/* lgdt m16&m32 */
				op_mode = NAME_(get_op_mode1)(cpssp, cpu);
				NAME_(load_limit_base)(cpssp, cpu, &descr, op_mode);
				NAME_(lgdt)(cpssp, cpu, descr);
				break;
			case 0x3:	/*lidt m16&m32 */
				op_mode = NAME_(get_op_mode1)(cpssp, cpu);
				NAME_(load_limit_base)(cpssp, cpu, &descr, op_mode);
				NAME_(lidt)(cpssp, cpu, descr);
				break;
			case 0x4:	/* smsw rm16 */
				op1 = NAME_(smsw)(cpssp, cpu);
				NAME_(store_rm16)(cpssp, cpu, op1);
				break;
			case 0x6:	/* lmsw rm16 */
				op1 = NAME_(load_rm16)(cpssp, cpu, 0);
				NAME_(lmsw)(cpssp, cpu, op1);
				break;

#if 80486 <= CONFIG_CPU
			case 0x7:	/* invlpg m */
				DEFAULT_NAME_(mmu_invlpg)(cpssp, cpu->addr);
				break;
#endif /* 80486 <= CONFIG_CPU */

			default:
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
				break;
			}
			break;

		case 0x02:	/* lar */
			if (! (STATE_GET(cr0_pe)
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
				|| STATE_GET(lm_active_msr)
#endif
				)) {
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
			}

			op_mode = NAME_(get_op_mode1)(cpssp, cpu);
			NAME_(get_modrm)(cpssp, cpu);
			op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
			res1 = NAME_(lar)(cpssp, cpu, op1, op_mode); /* FIXME op_mode? */
			if (STATE_GET(e_zf)) {
				NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, res1, op_mode);
			}
			break;

		case 0x03:	/* lsl */
			if (! (STATE_GET(cr0_pe)
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
				|| STATE_GET(lm_active_msr)
#endif
				) || STATE_GET(e_vm)) {
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
			}

			op_mode = NAME_(get_op_mode1)(cpssp, cpu);
			NAME_(get_modrm)(cpssp, cpu);
			op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
			res1 = NAME_(lsl)(cpssp, cpu, op1, op_mode); /* FIXME op_mode? */
			if (STATE_GET(e_zf)) {
				NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, res1, op_mode);
			}
			break;

		case 0x06:	/* clts */
			STATE_SET(cr0_ts, 0);
			break;

#if 80486 <= CONFIG_CPU
		case 0x08:	/* invd */
			if (STATE_GET(e_vm)) {
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
			}
			/* FIXME */
			break;

		case 0x09:	/* wbinvd */
			if (STATE_GET(e_vm)) {
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
			}
			/* FIXME */
			break;
#endif /* 80486 <= CONFIG_CPU */

		case 0x0b:	/* ud2 */
			NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
			break;

#if 1 /* FIXME */
		case 0x1f:	/* multi-byte nop */
			NAME_(get_modrm)(cpssp, cpu);
			/* Nothing to do... */
			break;
#endif /* 1 */

		case 0x20:	/* mov r32, CR0-CR7 */
			NAME_(get_modrm)(cpssp, cpu);
			if (cpu->mod != 3) {
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
			}
			if (NAME_(cpl)(cpssp, cpu) != 0) {
				NAME_(GPe)(cpssp, cpu, 0);
			}

			res1 = NAME_(get_cr)(cpssp, cpu, cpu->reg_opex);
			NAME_(store_reg32)(cpssp, cpu, cpu->rm, res1);
			break;

		case 0x22:	/* mov CR0-CR7, r32 */
			NAME_(get_modrm)(cpssp, cpu);
			if (cpu->mod != 3) {
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
			}
			if (NAME_(cpl)(cpssp, cpu) != 0) {
				NAME_(GPe)(cpssp, cpu, 0);
			}

			op1 = NAME_(load_reg32)(cpssp, cpu, cpu->rm);
			NAME_(set_cr)(cpssp, cpu, cpu->reg_opex, op1);
			break;

		case 0x21:	/* mov r32, DR0-DR7 */
			NAME_(get_modrm)(cpssp, cpu);
			if (cpu->mod != 3
#if defined CONFIG_CPU_DE_SUPPORT && CONFIG_CPU_DE_SUPPORT
			 || ((cpu->reg_opex & 0x6) == 0x4 && STATE_GET(cr4_de))
#endif
			 ) {
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
			}
			if (NAME_(cpl)(cpssp, cpu) != 0) {
				NAME_(GPe)(cpssp, cpu, 0);
			}

			if ((cpu->reg_opex & 0x6) == 0x4) {
				/* Register 4/5 are aliased to 6/7. */
				cpu->reg_opex |= 0x2;
			}
			res1 = NAME_(get_dr)(cpssp, cpu, cpu->reg_opex);
			NAME_(store_reg32)(cpssp, cpu, cpu->rm, res1);
			break;

		case 0x23:	/* mov DR0-DR7, r32 */
			NAME_(get_modrm)(cpssp, cpu);
			if (cpu->mod != 3
#if defined CONFIG_CPU_DE_SUPPORT && CONFIG_CPU_DE_SUPPORT
			 || ((cpu->reg_opex & 0x6) == 0x4 && STATE_GET(cr4_de))
#endif
			 ) {
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
			}
			if (NAME_(cpl)(cpssp, cpu) != 0) {
				NAME_(GPe)(cpssp, cpu, 0);
			}

			if ((cpu->reg_opex & 0x6) == 0x4) {
				/* Register 4/5 are aliased to 6/7. */
				cpu->reg_opex |= 0x2;
			}
			op1 = NAME_(load_reg32)(cpssp, cpu, cpu->rm);
			NAME_(set_dr)(cpssp, cpu, cpu->reg_opex, op1);
			break;

#if defined CONFIG_CPU_MSR_SUPPORT && CONFIG_CPU_MSR_SUPPORT
		case 0x30:	/* wrmsr */
			if (NAME_(cpl)(cpssp, cpu) != 0
			 || STATE_GET(e_vm)) {
				NAME_(GPe)(cpssp, cpu, 0);
			}
			op1 = NAME_(load_reg32)(cpssp, cpu, NAME_(E_CX));
			op2 = NAME_(load_reg32)(cpssp, cpu, NAME_(E_AX));
			op3 = NAME_(load_reg32)(cpssp, cpu, NAME_(E_DX));
			NAME_(wrmsr)(cpssp, cpu, op1, op2, op3);
			break;

		case 0x32:	/* rdmsr */
			if (NAME_(cpl)(cpssp, cpu) != 0
			 || STATE_GET(e_vm)) {
				NAME_(GPe)(cpssp, cpu, 0);
			}
			op1 = NAME_(load_reg32)(cpssp, cpu, NAME_(E_CX));
			NAME_(rdmsr)(cpssp, cpu, op1, (uint32_t*) &res1, (uint32_t*) &res2);
			NAME_(store_reg32)(cpssp, cpu, NAME_(E_AX), res1);
			NAME_(store_reg32)(cpssp, cpu, NAME_(E_DX), res2);
			break;
#endif /* defined CONFIG_CPU_MSR_SUPPORT && CONFIG_CPU_MSR_SUPPORT */

#if defined CONFIG_CPU_TSC_SUPPORT && CONFIG_CPU_TSC_SUPPORT
		case 0x31:	/* rdtsc */
			if (STATE_GET(cr4_tsd)
			 && STATE_GET(cr0_pe)
			 && NAME_(cpl)(cpssp, cpu) != 0) {
				NAME_(GPe)(cpssp, cpu, 0);
			}
			NAME_(rdtsc)(cpssp, cpu, (uint32_t*) &res1, (uint32_t*) &res2);
			NAME_(store_reg32)(cpssp, cpu, NAME_(E_AX), res1);
			NAME_(store_reg32)(cpssp, cpu, NAME_(E_DX), res2);
			break;
#endif /* defined CONFIG_CPU_TSC_SUPPORT && CONFIG_CPU_TSC_SUPPORT */

#if (80586 <= CONFIG_CPU && defined CONFIG_CPU_MMX_SUPPORT && CONFIG_CPU_MMX_SUPPORT) || 80686 <= CONFIG_CPU
		case 0x33:	/* rdpmc */
			assert(0); /* FIXME */
			break;
#endif /* (80586 <= CONFIG_CPU && defined CONFIG_CPU_MMX_SUPPORT && CONFIG_CPU_MMX_SUPPORT) || 80686 <= CONFIG_CPU */

#if defined CONFIG_CPU_SEP_SUPPORT && CONFIG_CPU_SEP_SUPPORT
		case 0x34:	/* sysenter */
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
			if (STATE_GET(lm_active_msr)) /* invalid in long mode */
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
#endif
			if (! STATE_GET(cr0_pe)
			 || STATE_GET(sysenter_cs_msr) == 0) {
				NAME_(GPe)(cpssp, cpu, 0);
			}

			NAME_(sysenter)(cpssp, cpu);
			break;

		case 0x35:	/* sysexit */
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
			if (STATE_GET(lm_active_msr)) /* invalid in long mode */
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
#endif
			if (! STATE_GET(cr0_pe)
			 || STATE_GET(sysenter_cs_msr) == 0
			 || NAME_(cpl)(cpssp, cpu) != 0) {
				NAME_(GPe)(cpssp, cpu, 0);
			}

			NAME_(sysexit)(cpssp, cpu);
			break;
#endif /* defined CONFIG_CPU_SEP_SUPPORT && CONFIG_CPU_SEP_SUPPORT */

#if defined CONFIG_CPU_CMOV_SUPPORT && CONFIG_CPU_CMOV_SUPPORT
		case 0x40:	/* cmovo */
		case 0x41:	/* cmovno */
		case 0x42:	/* cmovb */
		case 0x43:	/* cmovnb */
		case 0x44:	/* ... */
		case 0x45:	/* ... */
		case 0x46:	/* ... */
		case 0x47:	/* ... */
		case 0x48:	/* ... */
		case 0x49:	/* ... */
		case 0x4a:	/* ... */
		case 0x4b:	/* ... */
		case 0x4c:	/* ... */
		case 0x4d:	/* ... */
		case 0x4e:	/* ... */
		case 0x4f:	/* ... */
			op_mode = NAME_(get_op_mode1)(cpssp, cpu);
			NAME_(get_modrm)(cpssp, cpu);
			res1 = NAME_(test_cc)(cpssp, cpu, (opcode >> 0) & 0xf);
			if (res1) {
				res1 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
				NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, res1, op_mode);
			}
			break;
#endif /* defined CONFIG_CPU_CMOV_SUPPORT && CONFIG_CPU_CMOV_SUPPORT */

#if defined CONFIG_CPU_SSE_SUPPORT && CONFIG_CPU_SSE_SUPPORT
/* SSE1 */
		case 0x10:
		case 0x11:
		case 0x12:
		case 0x13:
		case 0x14:
		case 0x15:
		case 0x16:
		case 0x17:
		case 0x18:
		case 0x28:
		case 0x29:
		case 0x2a:
		case 0x2b:
		case 0x2c:
		case 0x2d:
		case 0x2e:
		case 0x2f:
		case 0x50:
		case 0x51:
		case 0x52:
		case 0x53:
		case 0x54:
		case 0x55:
		case 0x56:
		case 0x57:
		case 0x58:
		case 0x59:
		case 0x5c:
		case 0x5d:
		case 0x5e:
		case 0x5f:
		case 0x70:
		case 0xc2:
		case 0xc4:
		case 0xc5:
		case 0xc6:
		case 0xd7:
		case 0xda:
		case 0xde:
		case 0xe0:
		case 0xe3:
		case 0xe7:
		case 0xea:
		case 0xee:
		case 0xf6:
		case 0xf7:
#if defined CONFIG_CPU_SSE2_SUPPORT && CONFIG_CPU_SSE2_SUPPORT
			if (cpu->pre_op) {
				assert(0);
				break;
			}
#endif
			assert(0);
			break;
#endif

#if defined CONFIG_CPU_SSE2_SUPPORT && CONFIG_CPU_SSE2_SUPPORT
/* SSE2 */
		case 0x5a:
		case 0x5b:
		case 0x6c:
		case 0x6d:
		case 0xc3:
		case 0xd4:
		case 0xd6:
		case 0xe6:
		case 0xf4:
		case 0xfb:
			assert(0);
			break;
#endif

#if defined CONFIG_CPU_MMX_SUPPORT && CONFIG_CPU_MMX_SUPPORT
		case 0x60:	/* punpcklbw mm, mm/mm64 */
		case 0x61:	/* punpcklwd mm, mm/mm64 */
		case 0x62:	/* punpckldq mm, mm/mm64 */
		case 0x68:	/* punpckhbw mm, mm/mm64 */
		case 0x69:	/* punpckhwd mm, mm/mm64 */
		case 0x6a:	/* punpckhdq mm, mm/mm64 */
#if defined CONFIG_CPU_SSE2_SUPPORT && CONFIG_CPU_SSE2_SUPPORT
			if (cpu->pre_op) {
				assert(0);
				break;
			}
#endif
			NAME_(mmx_start)(cpssp, cpu, opcode);

			NAME_(get_modrm)(cpssp, cpu);
			op64_1 = NAME_(get_mmx)(cpssp, cpu, cpu->reg_opex);
			op64_2 = NAME_(load_rm_cx8)(cpssp, cpu, 0);

			res64_1 = NAME_(mmx_unpack)(cpssp, cpu, opcode, op64_1, op64_2);

			NAME_(set_mmx)(cpssp, cpu, cpu->reg_opex, res64_1);
			break;

		case 0x63:	/* packsswb mm, mm/mm64 */
		case 0x67:	/* packuswb mm, mm/mm64 */
		case 0x6b:	/* packssdw mm, mm/mm64 */
#if defined CONFIG_CPU_SSE2_SUPPORT && CONFIG_CPU_SSE2_SUPPORT
			if (cpu->pre_op) {
				assert(0);
				break;
			}
#endif
			NAME_(mmx_start)(cpssp, cpu, opcode);

			NAME_(get_modrm)(cpssp, cpu);
			op64_1 = NAME_(get_mmx)(cpssp, cpu, cpu->reg_opex);
			op64_2 = NAME_(load_rm_cx8)(cpssp, cpu, 0);

			res64_1 = NAME_(mmx_pack)(cpssp, cpu, opcode, op64_1, op64_2);

			NAME_(set_mmx)(cpssp, cpu, cpu->reg_opex, res64_1);
			break;

		case 0x6e:	/* movd r/m32, mm */
#if defined CONFIG_CPU_SSE2_SUPPORT && CONFIG_CPU_SSE2_SUPPORT
			if (cpu->pre_op) {
				assert(0);
				break;
			}
#endif
			NAME_(mmx_start)(cpssp, cpu, opcode);

			NAME_(get_modrm)(cpssp, cpu);
			op1 = NAME_(load_rm)(cpssp, cpu, 2, 0);
			res64_1 = (uint64_t) op1;
			NAME_(set_mmx)(cpssp, cpu, cpu->reg_opex, res64_1);
			break;

		case 0x6f:	/* movq mm/m64, mm */
#if defined CONFIG_CPU_SSE2_SUPPORT && CONFIG_CPU_SSE2_SUPPORT
			if (cpu->pre_op) {
				assert(0);
				break;
			}
#endif
			NAME_(mmx_start)(cpssp, cpu, opcode);

			NAME_(get_modrm)(cpssp, cpu);
			op64_1 = NAME_(load_rm_cx8)(cpssp, cpu, 0);
			res64_1 = op64_1;
			NAME_(set_mmx)(cpssp, cpu, cpu->reg_opex, res64_1);
			break;

		case 0x7e:	/* movd mm, r/m32 */
#if defined CONFIG_CPU_SSE2_SUPPORT && CONFIG_CPU_SSE2_SUPPORT
			if (cpu->pre_op || cpu->pre_repe_repne == 0x2) {
				assert(0);
				break;
			}
#endif
			NAME_(mmx_start)(cpssp, cpu, opcode);

			NAME_(get_modrm)(cpssp, cpu);
			op64_1 = NAME_(get_mmx)(cpssp, cpu, cpu->reg_opex);
			res1 = (uint32_t) op64_1;
			NAME_(store_rm)(cpssp, cpu, res1, 2);
			break;

		case 0x7f:	/* movq mm, mm/m64 */
#if defined CONFIG_CPU_SSE2_SUPPORT && CONFIG_CPU_SSE2_SUPPORT
			if (cpu->pre_op || cpu->pre_repe_repne == 0x2) {
				assert(0);
				break;
			}
#endif
			NAME_(mmx_start)(cpssp, cpu, opcode);

			NAME_(get_modrm)(cpssp, cpu);
			op64_1 = NAME_(get_mmx)(cpssp, cpu, cpu->reg_opex);
			res64_1 = op64_1;
			NAME_(store_rm_cx8)(cpssp, cpu, res64_1);
			break;

		case 0x71:
		case 0x72:
		case 0x73:
#if defined CONFIG_CPU_SSE2_SUPPORT && CONFIG_CPU_SSE2_SUPPORT
			if (cpu->pre_op) {
				assert(0);
				break;
			}
#endif
			NAME_(mmx_start)(cpssp, cpu, opcode);

			NAME_(get_modrm)(cpssp, cpu);
			switch (cpu->reg_opex) {
			case 2:	/* psrlw mm/m64, imm8 */
				/* psrld mm/m64, imm8 */
				/* psrlq mm/m64, imm8 */
				opcode = 0xd0 | (opcode & 3);
				break;
			case 4:	/* psraw mm/m64, imm8 */
				/* psrad mm/m64, imm8 */
				/* ? */
				opcode = 0xe0 | (opcode & 3);
				break;
			case 6:	/* psllw mm/m64, imm8 */
				/* pslld mm/m64, imm8 */
				/* psllq mm/m64, imm8 */
				opcode = 0xf0 | (opcode & 3);
				break;
			default:
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
				break;
			}

			op64_1 = NAME_(load_rm_cx8)(cpssp, cpu, 1);
			op64_2 = (uint64_t) NAME_(fetch8)(cpssp, cpu);

			switch (opcode & 0x3) {
			case 0x0: /* p...b */
				assert(0); /* Cannot happen. */
				break;
			case 0x1: /* p...w */
				res64_1 = NAME_(mmx_shiftw)(cpssp, cpu, opcode, op64_1, op64_2);
				break;
			case 0x2: /* p...d */
				res64_1 = NAME_(mmx_shiftd)(cpssp, cpu, opcode, op64_1, op64_2);
				break;
			case 0x3: /* p...q */
				res64_1 = NAME_(mmx_shiftq)(cpssp, cpu, opcode, op64_1, op64_2);
				break;
			}

			NAME_(store_rm_cx8)(cpssp, cpu, res64_1);
			break;

		case 0x77:	/* emms */
			NAME_(mmx_start)(cpssp, cpu, opcode);
			NAME_(mmx_emms)(cpssp, cpu);
			break;

		case 0x64:	/* pcmpgtb mm/m64, mm */
		case 0x65:	/* pcmpgtw mm/m64, mm */
		case 0x66:	/* pcmpgtd mm/m64, mm */
		case 0x74:	/* pcmpeqb mm/m64, mm */
		case 0x75:	/* pcmpeqw mm/m64, mm */
		case 0x76:	/* pcmpeqd mm/m64, mm */
		case 0xd5:	/* pmullw mm/m64, mm */
		case 0xd8:	/* psubusb mm/m64, mm */
		case 0xd9:	/* psubusw mm/m64, mm */
		case 0xdc:	/* paddusb mm/m64, mm */
		case 0xdb:	/* pand mm/m64, mm */
		case 0xdd:	/* paddusw mm/m64, mm */
		case 0xdf:	/* pandn mm/m64, mm */
		case 0xe4:	/* pmulhuw mm/m64, mm Special! See below! */
		case 0xe5:	/* pmulhw mm/m64, mm */
		case 0xe8:	/* psubsb mm/m64, mm */
		case 0xe9:	/* psubsw mm/m64, mm */
		case 0xeb:	/* por mm/m64, mm */
		case 0xec:	/* paddsb mm/m64, mm */
		case 0xed:	/* paddsw mm/m64, mm */
		case 0xef:	/* pxor mm/m64, mm */
		case 0xf8:	/* psubb mm/m64, mm */
		case 0xf9:	/* psubw mm/m64, mm */
		case 0xfa:	/* psubd mm/m64, mm */
		case 0xfc:	/* paddb mm/m64, mm */
		case 0xfd:	/* paddw mm/m64, mm */
		case 0xfe:	/* paddd mm/m64, mm */
#if defined CONFIG_CPU_SSE2_SUPPORT && CONFIG_CPU_SSE2_SUPPORT
			if (cpu->pre_op) {
				assert(0);
				break;
			}
#endif
			NAME_(mmx_start)(cpssp, cpu, opcode);

			NAME_(get_modrm)(cpssp, cpu);
			op64_1 = NAME_(load_rm_cx8)(cpssp, cpu, 0);
			op64_2 = NAME_(get_mmx)(cpssp, cpu, cpu->reg_opex);

			switch (opcode & 0x3) {
			case 0x0: /* p...b */
				if (opcode != 0xe4) {
					res64_1 = NAME_(mmxb)(cpssp, cpu, opcode, op64_1, op64_2);
					break;
				}
				/*FALLTHROUGH*/
			case 0x1: /* p...w */
				res64_1 = NAME_(mmxw)(cpssp, cpu, opcode, op64_1, op64_2);
				break;
			case 0x2: /* p...d */
				res64_1 = NAME_(mmxd)(cpssp, cpu, opcode, op64_1, op64_2);
				break;
			case 0x3: /* p...q */
				res64_1 = NAME_(mmxq)(cpssp, cpu, opcode, op64_1, op64_2);
				break;
			}

			NAME_(set_mmx)(cpssp, cpu, cpu->reg_opex, res64_1);
			break;

		case 0xd1:	/* psrlw mm/m64, mm */
		case 0xd2:	/* psrld mm//64, mm */
		case 0xd3:	/* psrlq mm//64, mm */
		case 0xe1:	/* psraw mm/m64, mm */
		case 0xe2:	/* psrad mm/m64, mm */
		case 0xf1:	/* psllw mm/m64, mm */
		case 0xf2:	/* pslld mm/m64, mm */
		case 0xf3:	/* psllq mm/m64, mm */
#if defined CONFIG_CPU_SSE2_SUPPORT && CONFIG_CPU_SSE2_SUPPORT
			if (cpu->pre_op) {
				assert(0);
				break;
			}
#endif
			NAME_(mmx_start)(cpssp, cpu, opcode);

			NAME_(get_modrm)(cpssp, cpu);
			op64_1 = NAME_(load_rm_cx8)(cpssp, cpu, 1);
			op64_2 = NAME_(get_mmx)(cpssp, cpu, cpu->reg_opex);

			switch (opcode & 0x3) {
			case 0x0: /* p...b */
				assert(0); /* Cannot happen. */
				break;
			case 0x1: /* p...w */
				res64_1 = NAME_(mmx_shiftw)(cpssp, cpu, opcode, op64_1, op64_2);
				break;
			case 0x2: /* p...d */
				res64_1 = NAME_(mmx_shiftd)(cpssp, cpu, opcode, op64_1, op64_2);
				break;
			case 0x3: /* p...q */
				res64_1 = NAME_(mmx_shiftq)(cpssp, cpu, opcode, op64_1, op64_2);
				break;
			}

			NAME_(store_rm_cx8)(cpssp, cpu, res64_1);
			break;

		case 0xf5:	/* pmaddwd mm/m64, mm */
#if defined CONFIG_CPU_SSE2_SUPPORT && CONFIG_CPU_SSE2_SUPPORT
			if (cpu->pre_op) {
				assert(0);
				break;
			}
#endif
			NAME_(mmx_start)(cpssp, cpu, opcode);

			NAME_(get_modrm)(cpssp, cpu);
			op64_1 = NAME_(load_rm_cx8)(cpssp, cpu, 0);
			op64_2 = NAME_(get_mmx)(cpssp, cpu, cpu->reg_opex);

			res64_1 = NAME_(mmx_pmaddwd)(cpssp, cpu, op64_1, op64_2);

			NAME_(set_mmx)(cpssp, cpu, cpu->reg_opex, res64_1);
			break;

#endif /* defined CONFIG_CPU_MMX_SUPPORT && CONFIG_CPU_MMX_SUPPORT */

		case 0x80:	/* jo rel16/32 */
		case 0x81:	/* jno rel16/32 */
		case 0x82:	/* jb/jc/jnae rel16/32 */
		case 0x83:	/* jae/jnb/jnc rel16/32 */
		case 0x84:	/* je/jz rel16/32 */
		case 0x85:	/* jne/jnz rel16/32 */
		case 0x86:	/* jbe/jna rel16/32 */
		case 0x87:	/* ja/jnbe rel16/32 */
		case 0x88:	/* js rel16/32 */
		case 0x89:	/* jns rel16/32 */
		case 0x8a:	/* jp/jpe rel16/32 */
		case 0x8b:	/* jnp/jpo rel16/32 */
		case 0x8c:	/* jl/jnge rel16/32 */
		case 0x8d:	/* jge/jnl rel16/32 */
		case 0x8e:	/* jle/jng rel16/32 */
		case 0x8f:	/* jg/jnle rel16/32 */
			op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
			op1 = NAME_(fetch)(cpssp, cpu, op_mode - (op_mode == 3));
			op1 = NAME_(sign_ext)(op1, op_mode - (op_mode == 3), MODE_MAX);
			res1 = NAME_(test_cc)(cpssp, cpu, (opcode >> 0) & 0xf);
			NAME_(jcc)(cpssp, cpu, op1, res1, op_mode);
			break;

		case 0x90:	/* seto rel16/32 */
		case 0x91:	/* setno rel16/32 */
		case 0x92:	/* setb/setc/setnae rel16/32 */
		case 0x93:	/* setae/setnb/setnc rel16/32 */
		case 0x94:	/* sete/setz rel16/32 */
		case 0x95:	/* setne/setnz rel16/32 */
		case 0x96:	/* setbe/setna rel16/32 */
		case 0x97:	/* seta/setnbe rel16/32 */
		case 0x98:	/* sets rel16/32 */
		case 0x99:	/* setns rel16/32 */
		case 0x9a:	/* setp/setpe rel16/32 */
		case 0x9b:	/* setnp/setpo rel16/32 */
		case 0x9c:	/* setl/setnge rel16/32 */
		case 0x9d:	/* setge/setnl rel16/32 */
		case 0x9e:	/* setle/setng rel16/32 */
		case 0x9f:	/* setg/setnle rel16/32 */
			NAME_(get_modrm)(cpssp, cpu);
			res1 = NAME_(test_cc)(cpssp, cpu, (opcode >> 0) & 0xf);
			NAME_(store_rm8)(cpssp, cpu, res1);
			break;

		case 0xa0:	/* push FS */
		case 0xa8:	/* push GS */
			op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
			st_mode = NAME_(get_stack_mode)(cpssp, cpu);
			op1 = NAME_(load_sreg)(cpssp, cpu, (opcode >> 3) & 0x7);
			sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);
			NAME_(push2)(cpssp, cpu, NAME_(SEG_SS), &sp, op1, st_mode, op_mode, NAME_(usermode)(cpssp, cpu));
			NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);
			break;

		case 0xa1:	/* pop FS */
		case 0xa9:	/* pop GS */
			op_mode = NAME_(get_op_mode_ip_sp)(cpssp, cpu);
			st_mode = NAME_(get_stack_mode)(cpssp, cpu);
			sp = NAME_(load_reg)(cpssp, cpu, NAME_(E_SP), st_mode);
			res1 = NAME_(pop2)(cpssp, cpu, NAME_(SEG_SS), &sp, st_mode, op_mode, NAME_(usermode)(cpssp, cpu));
			NAME_(store_segment)(cpssp, cpu, (opcode >> 3) & 0x7, res1);
			NAME_(store_reg)(cpssp, cpu, NAME_(E_SP), sp, st_mode);
			break;

#if 80586 <= CONFIG_CPU
/* CPUID */
		case 0xa2:	/* cpuid */
			op1 = NAME_(load_reg32)(cpssp, cpu, NAME_(E_AX));
			NAME_(cpuid)(cpssp, cpu, op1, (uint32_t*) &res1, (uint32_t*) &res2, (uint32_t*) &res3, (uint32_t*) &res4);
			NAME_(store_reg32)(cpssp, cpu, NAME_(E_AX), res1);
			NAME_(store_reg32)(cpssp, cpu, NAME_(E_BX), res2);
			NAME_(store_reg32)(cpssp, cpu, NAME_(E_CX), res3);
			NAME_(store_reg32)(cpssp, cpu, NAME_(E_DX), res4);
			break;
#endif /* 80586 <= CONFIG_CPU */

/* SHRD/SHLD */
		case 0xa4:	/* shld rm16/32, r16/32, imm8 */
		case 0xa5:	/* shld rm16/32, r16/32, cl */
			op_mode = NAME_(get_op_mode1)(cpssp, cpu);
			NAME_(get_modrm)(cpssp, cpu);
			op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 1);
			op2 = NAME_(load_reg)(cpssp, cpu, cpu->reg_opex, op_mode);
			if (opcode & 1) {
				op3 = NAME_(load_reg8)(cpssp, cpu, NAME_(CL));
			} else {
				op3 = NAME_(fetch8)(cpssp, cpu);
			}
			res1 = NAME_(shld)(cpssp, cpu, op1, op2, op3, op_mode);
			NAME_(store_rm)(cpssp, cpu, res1, op_mode);
			break;

		case 0xac:	/* shrd rm16/32, r16/32, imm8 */
		case 0xad:	/* shrd rm16/32, r16/32, cl */
			op_mode = NAME_(get_op_mode1)(cpssp, cpu);
			NAME_(get_modrm)(cpssp, cpu);
			op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 1);
			op2 = NAME_(load_reg)(cpssp, cpu, cpu->reg_opex, op_mode);
			if (opcode & 1) {
				op3 = NAME_(load_reg8)(cpssp, cpu, NAME_(CL));
			} else {
				op3 = NAME_(fetch8)(cpssp, cpu);
			}
			res1 = NAME_(shrd)(cpssp, cpu, op1, op2, op3, op_mode);
			NAME_(store_rm)(cpssp, cpu, res1, op_mode);
			break;

/* RSM */
		case 0xaa:	/* rsm */
			if (! cpssp->NAME.smm) {
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
			}
			NAME_(rsm)(cpssp, cpu);
			break;

/* IMUL */
		case 0xaf:	/* imul r16/32, rm16/32 */
			op_mode = NAME_(get_op_mode1)(cpssp, cpu);
			NAME_(get_modrm)(cpssp, cpu);
			op1 = NAME_(load_reg)(cpssp, cpu, cpu->reg_opex, op_mode);
			op1 = NAME_(sign_ext)(op1, op_mode, MODE_MAX);
			op2 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
			op2 = NAME_(sign_ext)(op2, op_mode, MODE_MAX);
			res1 = NAME_(imul)(cpssp, cpu, op1, op2, &res2, op_mode);
			NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, res1, op_mode);
			break;

#if 80486 <= CONFIG_CPU
/* CMPXCHG */
		case 0xb0:	/* cmpxchg r8, rm8 */
		case 0xb1:	/* cmpxchg r16/32, rm16/32 */
			op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
			NAME_(get_modrm)(cpssp, cpu);
			op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 1);
			op2 = NAME_(load_reg)(cpssp, cpu, cpu->reg_opex, op_mode);
			op3 = NAME_(load_reg)(cpssp, cpu, NAME_(E_AX), op_mode);
			NAME_(cmp)(cpssp, cpu, op3, op1, op_mode);
			if (STATE_GET(e_zf)) {
				res1 = op2;
				/* op3 = op3; */
			} else {
				res1 = op1;
				op3 = op1;
			}
			NAME_(store_reg)(cpssp, cpu, NAME_(E_AX), op3, op_mode);
			NAME_(store_rm)(cpssp, cpu, res1, op_mode);
			break;
#endif /* 80486 <= CONFIG_CPU */

/* LSS/LGS/LFS */
		case 0xb2:	/* lss r16/32, m16:16/32 */
		case 0xb4:	/* lfs r16/32, m16:16/32 */
		case 0xb5:	/* lgs r16/32, m16:16/32 */
			op_mode = NAME_(get_op_mode1)(cpssp, cpu);
			NAME_(get_modrm)(cpssp, cpu);
			if (cpu->mod == 3) {
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
			}
			res1 = NAME_(load_mem_fpointer)(cpssp, cpu, &res2, op_mode);
			NAME_(store_segment)(cpssp, cpu, (opcode >> 0) & 0x7, res2); // FIXME lm checks
			NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, res1, op_mode);
			break;

/* MOV */
		case 0xb6:	/* movzx r16/32, rm8 */
			op_mode = NAME_(get_op_mode1)(cpssp, cpu);
			NAME_(get_modrm)(cpssp, cpu);
			res1 = NAME_(load_rm8)(cpssp, cpu, 0);
			NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, res1, op_mode);
			break;

		case 0xb7:	/* movzx r32, rm16 */
			NAME_(get_modrm)(cpssp, cpu);
			res1 = NAME_(load_rm16)(cpssp, cpu, 0);
			NAME_(store_reg32)(cpssp, cpu, cpu->reg_opex, res1);
			break;

		case 0xbe:	/* movsx r16/32, rm8 */
			op_mode = NAME_(get_op_mode1)(cpssp, cpu);
			NAME_(get_modrm)(cpssp, cpu);
			res1 = (data_t) (int8_t) NAME_(load_rm8)(cpssp, cpu, 0);
			NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, res1, op_mode);
			break;

		case 0xbf:	/* movsx r32, rm16 */
			NAME_(get_modrm)(cpssp, cpu);
			res1 = (data_t) (int16_t) NAME_(load_rm)(cpssp, cpu, 1, 0);
			NAME_(store_reg32)(cpssp, cpu, cpu->reg_opex, res1);
			break;

/* BT/BTC/BTR/BTS */
		case 0xa3:	/* bt rm16/32, r16/32 */
		case 0xb3:	/* btr rm16/32, r16/32 */
		case 0xab:	/* bts rm16/32, r16/32 */
		case 0xbb:	/* btc rm16/32, r16/32 */
			op_mode = NAME_(get_op_mode1)(cpssp, cpu);
			NAME_(get_modrm)(cpssp, cpu);
			op2 = NAME_(load_reg)(cpssp, cpu, cpu->reg_opex, op_mode);
			op2 = NAME_(sign_ext)(op2, op_mode, MODE_MAX);
			cpu->addr += (((data_t) op2) >> (3 + op_mode)) << op_mode;
			op1 = NAME_(load_rm)(cpssp, cpu, op_mode, opcode != 0xa3);
			res1 = NAME_(alu_bt)(cpssp, cpu, op1, op2, (opcode >> 3) & 0x3, op_mode);
			if (opcode != 0xa3) {
				NAME_(store_rm)(cpssp, cpu, res1, op_mode);
			}
			break;

		case 0xba:
			op_mode = NAME_(get_op_mode1)(cpssp, cpu);
			NAME_(get_modrm)(cpssp, cpu);
			switch (cpu->reg_opex) {
			case 0x4:	/* bt rm16/32, imm8 */
			case 0x5:	/* bts rm16/32, imm8 */
			case 0x6:	/* btr rm16/32, imm8 */
			case 0x7:	/* btc rm16/32, imm8 */
				op1 = NAME_(load_rm)(cpssp, cpu, op_mode, cpu->reg_opex != 4);
				op2 = NAME_(fetch8)(cpssp, cpu);
				res1 = NAME_(alu_bt)(cpssp, cpu, op1, op2, cpu->reg_opex & 0x3, op_mode);
				if (cpu->reg_opex != 4) {
					NAME_(store_rm)(cpssp, cpu, res1, op_mode);
				}
				break;

			default:
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
				break;
			}
			break;

#if (defined CONFIG_CPU_FXSR_SUPPORT && CONFIG_CPU_FXSR_SUPPORT) \
	|| (defined CONFIG_CPU_SSE_SUPPORT && CONFIG_CPU_SSE_SUPPORT) \
	|| (defined CONFIG_CPU_SSE2_SUPPORT && CONFIG_CPU_SSE2_SUPPORT)
		case 0xae:	/* fxsave / fxrestore/sse/sse2 */
			assert(0);
			break;
#endif /* CONFIG_CPU_FXSR_SUPPORT */

/* BSF/BSR */
		case 0xbc:	/* bsf r16/32, rm16/32 */
			op_mode = NAME_(get_op_mode1)(cpssp, cpu);
			NAME_(get_modrm)(cpssp, cpu);
			op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
			res1 = NAME_(bsf)(cpssp, cpu, op1);
			if (! STATE_GET(e_zf)) {
				NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, res1, op_mode);
			}
			break;

		case 0xbd:	/* bsr r16/32, rm16/32 */
			op_mode = NAME_(get_op_mode1)(cpssp, cpu);
			NAME_(get_modrm)(cpssp, cpu);
			op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 0);
			res1 = NAME_(bsr)(cpssp, cpu, op1);
			if (! STATE_GET(e_zf)) {
				NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, res1, op_mode);
			}
			break;

#if 80486 <= CONFIG_CPU
		case 0xc0:	/* xadd r8, rm8 */
		case 0xc1:	/* xadd r16/32, rm16/32 */
			op_mode = NAME_(get_op_mode2)(cpssp, cpu, opcode);
			NAME_(get_modrm)(cpssp, cpu);
			op1 = NAME_(load_rm)(cpssp, cpu, op_mode, 1);
			op2 = NAME_(load_reg)(cpssp, cpu, cpu->reg_opex, op_mode);
			res1 = NAME_(add)(cpssp, cpu, op1, op2, op_mode);
			NAME_(store_rm)(cpssp, cpu, res1, op_mode);
			NAME_(store_reg)(cpssp, cpu, cpu->reg_opex, op1, op_mode);
			break;
#endif /* 80486 <= CONFIG_CPU */

		case 0xc7:
			NAME_(get_modrm)(cpssp, cpu);
			switch (cpu->reg_opex) {
#if defined CONFIG_CPU_CX8_SUPPORT && CONFIG_CPU_CX8_SUPPORT
/* CMPXCHG8B */
			case 0x1: /* cmpxchg8b */
				if (cpu->mod == 0x3) {
					NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
				}
				op64_1 = NAME_(load_mem_cx8)(cpssp, cpu, 1);
				op64_2 = NAME_(load_reg_cx8)(cpssp, cpu, NAME_(E_BX));
				op64_3 = NAME_(load_reg_cx8)(cpssp, cpu, NAME_(E_AX));
				if (op64_3 == op64_1) {
					res64_1 = op64_2;
					/* op3 = op3; */
					STATE_SET(e_zf, 1);
				} else {
					res64_1 = op64_1;
					op64_3 = op64_1;
					STATE_SET(e_zf, 0);
				}
				NAME_(store_reg_cx8)(cpssp, cpu, NAME_(E_AX), op64_3);
				NAME_(store_mem_cx8)(cpssp, cpu, res64_1);
				break;
#endif /* defined CONFIG_CPU_CX8_SUPPORT && CONFIG_CPU_CX8_SUPPORT */

			default:
				NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
			}
			break;

#if 80486 <= CONFIG_CPU
		case 0xc8:
		case 0xc9:
		case 0xca:
		case 0xcb:
		case 0xcc:
		case 0xcd:
		case 0xce:
		case 0xcf:	/* bswap r32 */
			op1 = NAME_(load_reg32)(cpssp, cpu, ((opcode >> 0) & 0x7));
			res1 = NAME_(bswap)(cpssp, cpu, op1);
			NAME_(store_reg32)(cpssp, cpu, ((opcode >> 0) & 0x7), res1);
			break;
#endif /* 80486 <= CONFIG_CPU */

		default:
			NAME_(UD)(cpssp, cpu, 0x0f00 | opcode);
			break;
		}
		break;

	default:
		NAME_(UD)(cpssp, cpu, opcode);
		break;
	}

	NAME_(fetch_end)(cpssp, cpu);
	cpssp->process.inst_cnt++;

EXIT_COMMIT:
	STATE_SET(eip, cpu->eip);
}

#if ENABLE_INSPECTION && CONFIG_INSPECTION
static void
NAME_(step_cb)(void *cpssp_)
{
	struct cpssp *cpssp = cpssp_;
	NAME_(step)(cpssp);
}
#endif

static void
NAME_(fpu_exception_update)(struct cpssp *cpssp)
{
	unsigned int val;

	val = cpssp->NAME.state_error;
#if CONFIG_FPU
	val &= cpssp->NAME.state_ignne;

	DEFAULT_NAME_(n_ferr_set)(cpssp, ~STATE_GET(cr0_ne) & val);
#endif

	cpssp->NAME.fpu_error = STATE_GET(cr0_ne) & val;
}

#if CONFIG_FPU
static void
NAME_(n_ignne_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.state_ignne = ! val;

	NAME_(fpu_exception_update)(cpssp);
}
#endif /* CONFIG_FPU */

static void
NAME_(n_error_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.state_error = ! val;

	NAME_(fpu_exception_update)(cpssp);
}

static void
NAME_(irq_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.state_event |= val;
	cpssp->NAME.state_interrupt_pending = val;
}

static void
NAME_(nmi_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.state_event |= val;
	cpssp->NAME.state_nmi_pending = val;
}

static void
NAME_(smi_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.state_event |= val;
	cpssp->NAME.state_smi_pending = val;
}

#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
static void
NAME_(startup_by_apic)(struct cpssp *cpssp, uint8_t vec)
{
	assert(0); /* FIXME */
}

static void
NAME_(init_by_apic)(struct cpssp *cpssp)
{
	assert(0); /* FIXME */
}
#endif /* defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT */

static void
NAME_(n_reset_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.state_event |= ! val;
	cpssp->NAME.state_n_reset = val;
}

#if defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT
static void
NAME_(n_init_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.state_event |= ! val;
	cpssp->NAME.state_n_init = val;
}
#endif /* defined CONFIG_CPU_APIC_SUPPORT && CONFIG_CPU_APIC_SUPPORT */

static void
NAME_(power_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.state_event |= ! val;
	cpssp->state_power = val;
}

#if ! CONFIG_INSPECTION
static void
NAME_(create)(struct cpssp *cpssp)
{
	/* general purpose registers */
	// TODO FIXME -> 64 bit and 16 regs in lm_support
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	STATE_DECL("eax", gp_regs[NAME_(E_AX)], DATA_TYPE_UINT64_T);
	STATE_DECL("ebx", gp_regs[NAME_(E_BX)], DATA_TYPE_UINT64_T);
	STATE_DECL("ecx", gp_regs[NAME_(E_CX)], DATA_TYPE_UINT64_T);
	STATE_DECL("edx", gp_regs[NAME_(E_DX)], DATA_TYPE_UINT64_T);
	STATE_DECL("esi", gp_regs[NAME_(E_SI)], DATA_TYPE_UINT64_T);
	STATE_DECL("edi", gp_regs[NAME_(E_DI)], DATA_TYPE_UINT64_T);
	/* special purpose registers */
	STATE_DECL("esp", gp_regs[NAME_(E_SP)], DATA_TYPE_UINT64_T);
	STATE_DECL("ebp", gp_regs[NAME_(E_BP)], DATA_TYPE_UINT64_T);
	/* longe mode gprs */
	STATE_DECL("r8", gp_regs[NAME_(R8)], DATA_TYPE_UINT64_T);
	STATE_DECL("r9", gp_regs[NAME_(R9)], DATA_TYPE_UINT64_T);
	STATE_DECL("r10", gp_regs[NAME_(R10)], DATA_TYPE_UINT64_T);
	STATE_DECL("r11", gp_regs[NAME_(R11)], DATA_TYPE_UINT64_T);
	STATE_DECL("r12", gp_regs[NAME_(R12)], DATA_TYPE_UINT64_T);
	STATE_DECL("r13", gp_regs[NAME_(R13)], DATA_TYPE_UINT64_T);
	STATE_DECL("r14", gp_regs[NAME_(R14)], DATA_TYPE_UINT64_T);
	STATE_DECL("r15", gp_regs[NAME_(R15)], DATA_TYPE_UINT64_T);

	/* eip */
	STATE_DECL("eip", eip, DATA_TYPE_UINT64_T);
#else
	STATE_DECL("eax", gp_regs[NAME_(E_AX)], DATA_TYPE_UINT32_T);
	STATE_DECL("ebx", gp_regs[NAME_(E_BX)], DATA_TYPE_UINT32_T);
	STATE_DECL("ecx", gp_regs[NAME_(E_CX)], DATA_TYPE_UINT32_T);
	STATE_DECL("edx", gp_regs[NAME_(E_DX)], DATA_TYPE_UINT32_T);
	STATE_DECL("esi", gp_regs[NAME_(E_SI)], DATA_TYPE_UINT32_T);
	STATE_DECL("edi", gp_regs[NAME_(E_DI)], DATA_TYPE_UINT32_T);
	/* special purpose registers */
	STATE_DECL("esp", gp_regs[NAME_(E_SP)], DATA_TYPE_UINT32_T);
	STATE_DECL("ebp", gp_regs[NAME_(E_BP)], DATA_TYPE_UINT32_T);

	/* eip */
	STATE_DECL("eip", eip, DATA_TYPE_UINT32_T);
#endif
#if 0
	/* This field is never used so inspection does not make much sense */
	STATE_DECL(cpssp->node, "tsc", tsc, DATA_TYPE_UINT64_T);
#endif

	/* sregs */
	STATE_DECL("seg[ES].selector", selector[NAME_(SEG_ES)], DATA_TYPE_UINT16_T);
	STATE_DECL("seg[CS].selector", selector[NAME_(SEG_CS)], DATA_TYPE_UINT16_T);
	STATE_DECL("seg[SS].selector", selector[NAME_(SEG_SS)], DATA_TYPE_UINT16_T);
	STATE_DECL("seg[DS].selector", selector[NAME_(SEG_DS)], DATA_TYPE_UINT16_T);
	STATE_DECL("seg[FS].selector", selector[NAME_(SEG_FS)], DATA_TYPE_UINT16_T);
	STATE_DECL("seg[GS].selector", selector[NAME_(SEG_GS)], DATA_TYPE_UINT16_T);
	STATE_DECL("seg[GDT].selector", selector[NAME_(SEG_GDT)], DATA_TYPE_UINT16_T);
	STATE_DECL("seg[LDT].selector", selector[NAME_(SEG_LDT)], DATA_TYPE_UINT16_T);
	STATE_DECL("seg[IDT].selector", selector[NAME_(SEG_IDT)], DATA_TYPE_UINT16_T);
	STATE_DECL("seg[TR].selector", selector[NAME_(SEG_TR)], DATA_TYPE_UINT16_T);

	/* cr0 */
	STATE_DECL("cr0_pe", cr0_pe, DATA_TYPE_UINT8_T);
	STATE_DECL("cr0_mp", cr0_mp, DATA_TYPE_UINT8_T);
	STATE_DECL("cr0_em", cr0_em, DATA_TYPE_UINT8_T);
	STATE_DECL("cr0_ts", cr0_ts, DATA_TYPE_UINT8_T);
	STATE_DECL("cr0_et", cr0_et, DATA_TYPE_UINT8_T);
	STATE_DECL("cr0_ne", cr0_ne, DATA_TYPE_UINT8_T);
#if 80486 <= CONFIG_CPU
	STATE_DECL("cr0_am", cr0_am, DATA_TYPE_UINT8_T);
#endif

	STATE_DECL("cr2", cr2, DATA_TYPE_UINT32_T);

	STATE_DECL("dr0.addr", dr[0].addr, DATA_TYPE_UINT32_T);
	STATE_DECL("dr0.len", dr[0].len, DATA_TYPE_UINT8_T);
	STATE_DECL("dr0.type", dr[0].type, DATA_TYPE_UINT8_T);
	STATE_DECL("dr0.detected", dr[0].detected, DATA_TYPE_UINT8_T);
	STATE_DECL("dr0.local_enabled", dr[0].local_enabled, DATA_TYPE_UINT8_T);
	STATE_DECL("dr0.global_enabled", dr[0].global_enabled, DATA_TYPE_UINT8_T);
	STATE_DECL("dr1.addr", dr[1].addr, DATA_TYPE_UINT32_T);
	STATE_DECL("dr1.len", dr[1].len, DATA_TYPE_UINT8_T);
	STATE_DECL("dr1.type", dr[1].type, DATA_TYPE_UINT8_T);
	STATE_DECL("dr1.detected", dr[1].detected, DATA_TYPE_UINT8_T);
	STATE_DECL("dr1.local_enabled", dr[1].local_enabled, DATA_TYPE_UINT8_T);
	STATE_DECL("dr1.global_enabled", dr[1].global_enabled, DATA_TYPE_UINT8_T);
	STATE_DECL("dr2.addr", dr[2].addr, DATA_TYPE_UINT32_T);
	STATE_DECL("dr2.len", dr[2].len, DATA_TYPE_UINT8_T);
	STATE_DECL("dr2.type", dr[2].type, DATA_TYPE_UINT8_T);
	STATE_DECL("dr2.detected", dr[2].detected, DATA_TYPE_UINT8_T);
	STATE_DECL("dr2.local_enabled", dr[2].local_enabled, DATA_TYPE_UINT8_T);
	STATE_DECL("dr2.global_enabled", dr[2].global_enabled, DATA_TYPE_UINT8_T);
	STATE_DECL("dr3.addr", dr[3].addr, DATA_TYPE_UINT32_T);
	STATE_DECL("dr3.len", dr[3].len, DATA_TYPE_UINT8_T);
	STATE_DECL("dr3.type", dr[3].type, DATA_TYPE_UINT8_T);
	STATE_DECL("dr3.detected", dr[3].detected, DATA_TYPE_UINT8_T);
	STATE_DECL("dr3.local_enabled", dr[3].local_enabled, DATA_TYPE_UINT8_T);
	STATE_DECL("dr3.global_enabled", dr[3].global_enabled, DATA_TYPE_UINT8_T);
	STATE_DECL("dr_bs", dr_bs, DATA_TYPE_UINT8_T);

#if 80586 <+ CONFIG_CPU
	/* cr4 */
#if defined CONFIG_CPU_TSC_SUPPORT && CONFIG_CPU_TSC_SUPPORT
	STATE_DECL("cr4_tsd", cr4_tsd, DATA_TYPE_UINT8_T);
#endif
#endif

	/* flags */
	STATE_DECL("sf", e_sf, DATA_TYPE_UINT8_T);
	STATE_DECL("zf", e_zf, DATA_TYPE_UINT8_T);
	STATE_DECL("of", e_of, DATA_TYPE_UINT8_T);
	STATE_DECL("cf", e_cf, DATA_TYPE_UINT8_T);
	STATE_DECL("pf", e_pf, DATA_TYPE_UINT8_T);
	STATE_DECL("af", e_af, DATA_TYPE_UINT8_T);
	STATE_DECL("tf", e_tf, DATA_TYPE_UINT8_T);
	STATE_DECL("if", e_if, DATA_TYPE_UINT8_T);
	STATE_DECL("df", e_df, DATA_TYPE_UINT8_T);
	STATE_DECL("iopl", e_iopl, DATA_TYPE_UINT8_T);
	STATE_DECL("nt", e_nt, DATA_TYPE_UINT8_T);
	STATE_DECL("vm", e_vm, DATA_TYPE_UINT8_T);
#if 80486 <= CONFIG_CPU
	STATE_DECL("ac", e_ac, DATA_TYPE_UINT8_T);
#endif
#if 80586 <= CONFIG_CPU
	STATE_DECL("id", e_id, DATA_TYPE_UINT8_T);
#endif

#if defined CONFIG_CPU_SEP_SUPPORT && CONFIG_CPU_SEP_SUPPORT
	STATE_DECL("sysenter_cs_msr", sysenter_cs_msr, DATA_TYPE_UINT32_T);
	STATE_DECL("sysenter_eip_msr", sysenter_eip_msr, DATA_TYPE_UINT32_T);
	STATE_DECL("sysenter_esp_msr", sysenter_esp_msr, DATA_TYPE_UINT32_T);
#endif

#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
	STATE_DECL("lm_enabled_msr", lm_enabled_msr, DATA_TYPE_UINT8_T);
	STATE_DECL("lm_active_msr", lm_active_msr, DATA_TYPE_UINT8_T);
#endif
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}
#endif /* CONFIG_INSPECTION */

#endif /* BEHAVIOR */

#undef DEBUG_CONTROL_FLOW
#undef DEBUG_CONTROL_FLOW_REGS
#undef DEBUG_CONTROL_FLOW_SREGS
#undef DEBUG_CONTROL_FLOW_FLAGS

#undef DEBUG_DISASSEMBLE
