/*
 * Copyright (C) 2016-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* Inter-integrated circuit interface (I2C) */

/*
 * For most of the comments, infos, ... see:
 * ST - RM0383 - Reference Manual - STM32F411xC/E advanced ARM-based
 * 32-bit MCUs
 */

#define DEBUG_CONTROL_FLOW	1

#ifdef INCLUDE
#endif /* INCLUDE */
#ifdef STATE

struct {
	unsigned int state_miso;
	unsigned int state_mosi;
	unsigned int state_nss;
	unsigned int state_sck;

	unsigned int counter;

	enum {
		NAME_(STATE_IDLE),
		NAME_(STATE_SEND),
	} state;
	unsigned int state_sub;

	/* SPI_CR1 */
	uint8_t bidimode;
	uint8_t bidioe;
	uint8_t crcen;
	uint8_t crcnext;
	uint8_t dff;
	uint8_t rxonly;
	uint8_t ssm;
	uint8_t ssi;
	uint8_t lsbfirst;
	uint8_t spe;
	uint8_t br;
	uint8_t mstr;
	uint8_t cpol;
	uint8_t cpha;

	/* SPI_CR2 */
	uint8_t txeie;
	uint8_t rxneie;
	uint8_t errie;
	uint8_t frf;
	uint8_t ssoe;
	uint8_t txdmaen;
	uint8_t rxdmaen;

	/* SPI_SR */
	uint8_t fre;
	uint8_t bsy;
	uint8_t ovr;
	uint8_t modf;
	uint8_t crcerr;
	uint8_t udr;
	uint8_t chside;
	uint8_t txe;
	uint8_t rxne;

	/* SPI_DR */
	uint16_t dr;

	/* SPI_CRCPR */
	uint16_t crcpoly;

	/* SPI_RXCRCR */
	uint16_t rxcrc;

	/* SPI_TXCRCR */
	uint16_t txcrc;
} NAME;

#endif /* STATE */
#ifdef EXPORT

/*forward*/ static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
NAME_(reset)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(miso_in_set)(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
NAME_(mosi_in_set)(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
NAME_(nss_in_set)(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
NAME_(sck_in_set)(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
NAME_(clk)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(create)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(destroy)(struct cpssp *cpssp);

#endif /* EXPORT */
#ifdef BEHAVIOR

static void
NAME_(reset)(struct cpssp *cpssp)
{
	cpssp->NAME.state = NAME_(STATE_IDLE);
	cpssp->NAME.state_sub = 0;

	/* SPI_CR1 */
	cpssp->NAME.bidimode = 0;
	cpssp->NAME.bidioe = 0;
	cpssp->NAME.crcen = 0;
	cpssp->NAME.crcnext = 0;
	cpssp->NAME.dff = 0;
	cpssp->NAME.rxonly = 0;
	cpssp->NAME.ssm = 0;
	cpssp->NAME.ssi = 0;
	cpssp->NAME.lsbfirst = 0;
	cpssp->NAME.spe = 0;
	cpssp->NAME.br = 0b000;
	cpssp->NAME.mstr = 0;
	cpssp->NAME.cpol = 0;
	cpssp->NAME.cpha = 0;

	/* SPI_CR2 */
	cpssp->NAME.txeie = 0;
	cpssp->NAME.rxneie = 0;
	cpssp->NAME.errie = 0;
	cpssp->NAME.frf = 0;
	cpssp->NAME.ssoe = 0;
	cpssp->NAME.txdmaen = 0;
	cpssp->NAME.rxdmaen = 0;

	/* SPI_SR */
	cpssp->NAME.fre = 0;
	cpssp->NAME.bsy = 0;
	cpssp->NAME.ovr = 0;
	cpssp->NAME.modf = 0;
	cpssp->NAME.crcerr = 0;
	cpssp->NAME.udr = 0;
	cpssp->NAME.chside = 0;
	cpssp->NAME.txe = 1;
	cpssp->NAME.rxne = 0;

	/* SPI_DR */
	cpssp->NAME.dr = 0x0000;

	/* SPI_CRCPR */
	cpssp->NAME.crcpoly = 0x0007;

	NAME_(miso_out_set)(cpssp, SIG_STD_LOGIC_Z);
	NAME_(mosi_out_set)(cpssp, SIG_STD_LOGIC_Z);
	NAME_(nss_out_set)(cpssp, SIG_STD_LOGIC_Z);
	NAME_(sck_out_set)(cpssp, SIG_STD_LOGIC_Z);
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	addr &= 0x3ff;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x00:
		/* SPI_CR1 */
		if ((bs >> 1) & 1) {
			cpssp->NAME.bidimode = (val >> 15) & 1;
			cpssp->NAME.bidioe = (val >> 14) & 1;
			cpssp->NAME.crcen = (val >> 13) & 1;
			cpssp->NAME.crcnext = (val >> 12) & 1;
			cpssp->NAME.dff = (val >> 11) & 1;
			cpssp->NAME.rxonly = (val >> 10) & 1;
			cpssp->NAME.ssm = (val >> 9) & 1;
			cpssp->NAME.ssi = (val >> 8) & 1;
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.lsbfirst = (val >> 7) & 1;
			cpssp->NAME.spe = (val >> 6) & 1;
			cpssp->NAME.br = (val >> 3) & 7;
			cpssp->NAME.mstr = (val >> 2) & 1;
			cpssp->NAME.cpol = (val >> 1) & 1;
			cpssp->NAME.cpha = (val >> 0) & 1;
		}
		break;
	case 0x04:
		/* SPI_CR2 */
		if ((bs >> 1) & 1) {
			/* Bit 15-8: Reserved */
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.txeie = (val >> 7) & 1;
			cpssp->NAME.rxneie = (val >> 6) & 1;
			cpssp->NAME.errie = (val >> 5) & 1;
			cpssp->NAME.frf = (val >> 4) & 1;
			/* Bit 3: Reserved */
			cpssp->NAME.ssoe = (val >> 2) & 1;
			cpssp->NAME.txdmaen = (val >> 1) & 1;
			cpssp->NAME.rxdmaen = (val >> 0) & 1;
		}
		break;
	case 0x08:
		/* SPI_SR */
		if ((bs >> 1) & 1) {
			/* Bit 15-9: Reserved */
			/* Bit 8: Read-only */
		}
		if ((bs >> 0) & 1) {
			/* Bit 7-5: Read-only */
			cpssp->NAME.crcerr &= (val >> 4) & 1;
			/* Bit 3-0: Read-only */
			// NAME_(irq_update)(cpssp);
		}
		break;
	case 0x0c:
		/* SPI_DR */
		if ((bs >> 1) & 1) {
			cpssp->NAME.dr &= ~(0xff << 8);
			cpssp->NAME.dr |= val & (0xff << 8);
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.dr &= ~(0xff << 0);
			cpssp->NAME.dr |= val & (0xff << 0);
		}
		cpssp->NAME.txe = 0;
		cpssp->NAME.state = NAME_(STATE_SEND);
		cpssp->NAME.state_sub = 0;
		break;
	case 0x10:
		/* SPI_CRCPR */
		if ((bs >> 1) & 1) {
			cpssp->NAME.crcpoly &= ~(0xff << 8);
			cpssp->NAME.crcpoly |= val & (0xff << 8);
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.crcpoly &= ~(0xff << 0);
			cpssp->NAME.crcpoly |= val & (0xff << 0);
		}
		break;
	case 0x14:
		/* SPI_RXCRCR */
		goto read_only;
	case 0x18:
		/* SPI_TXCRCR */
		goto read_only;
	case 0x1c:
		/* SPI_I2SCFGR */
		goto fixme;
	case 0x20:
		/* SPI_I2SPR */
		goto fixme;
	default:
	fixme:	;
		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);
		break;
	read_only:;
		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);
		break;
	}
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	addr &= 0x3ff;

	switch (addr) {
	case 0x00:
		/* SPI_CR1 */
		*valp = 0;
		*valp |= cpssp->NAME.bidimode << 15;
		*valp |= cpssp->NAME.bidioe << 14;
		*valp |= cpssp->NAME.crcen << 13;
		*valp |= cpssp->NAME.crcnext << 12;
		*valp |= cpssp->NAME.dff << 11;
		*valp |= cpssp->NAME.rxonly << 10;
		*valp |= cpssp->NAME.ssm << 9;
		*valp |= cpssp->NAME.ssi << 8;
		*valp |= cpssp->NAME.lsbfirst << 7;
		*valp |= cpssp->NAME.spe << 6;
		*valp |= cpssp->NAME.br << 3;
		*valp |= cpssp->NAME.mstr << 2;
		*valp |= cpssp->NAME.cpol << 1;
		*valp |= cpssp->NAME.cpha << 0;
		break;
	case 0x04:
		/* SPI_CR2 */
		*valp = 0;
		/* Bit 15-8: Reserved */
		*valp |= cpssp->NAME.txeie << 7;
		*valp |= cpssp->NAME.rxneie << 6;
		*valp |= cpssp->NAME.errie << 5;
		*valp |= cpssp->NAME.frf << 4;
		/* Bit 3: Reserved */
		*valp |= cpssp->NAME.ssoe << 2;
		*valp |= cpssp->NAME.txdmaen << 1;
		*valp |= cpssp->NAME.rxdmaen << 0;
		break;
	case 0x08:
		/* SPI_SR */
		*valp = 0;
		/* Bit 15-9: Reserved */
		*valp |= cpssp->NAME.fre << 8;
		*valp |= cpssp->NAME.bsy << 7;
		*valp |= cpssp->NAME.ovr << 6;
		*valp |= cpssp->NAME.modf << 5;
		*valp |= cpssp->NAME.crcerr << 4;
		*valp |= cpssp->NAME.udr << 3;
		*valp |= cpssp->NAME.chside << 2;
		*valp |= cpssp->NAME.txe << 1;
		*valp |= cpssp->NAME.rxne << 0;
		break;
	case 0x0c:
		/* SPI_DR */
		*valp = cpssp->NAME.dr << 0;
		cpssp->NAME.rxne = 0;
		cpssp->NAME.txe = 1;
		break;
	case 0x10:
		/* SPI_CRCPR */
		*valp = cpssp->NAME.crcpoly << 0;
		break;
	case 0x14:
		/* SPI_RXCRCR */
		goto fixme;
	case 0x18:
		/* SPI_TXCRCR */
		goto fixme;
	case 0x1c:
		/* SPI_I2SCFGR */
		goto fixme;
	case 0x20:
		/* SPI_I2SPR */
		goto fixme;
	default:
	fixme:	;
		*valp = 0;
		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);
		break;
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, *valp=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(miso_in_set)(struct cpssp *cpssp, unsigned int val)
{
	switch (val) {
	case SIG_STD_LOGIC_L:
	case SIG_STD_LOGIC_0:
		val = 0;
		break;
	case SIG_STD_LOGIC_H:
	case SIG_STD_LOGIC_1:
		val = 1;
		break;
	default:
		val = 0;
		break;
	}

	cpssp->NAME.state_miso = val;
}

static void
NAME_(mosi_in_set)(struct cpssp *cpssp, unsigned int val)
{
	switch (val) {
	case SIG_STD_LOGIC_L:
	case SIG_STD_LOGIC_0:
		val = 0;
		break;
	case SIG_STD_LOGIC_H:
	case SIG_STD_LOGIC_1:
		val = 1;
		break;
	default:
		val = 0;
		break;
	}

	cpssp->NAME.state_mosi = val;
}

static void
NAME_(nss_in_set)(struct cpssp *cpssp, unsigned int val)
{
	switch (val) {
	case SIG_STD_LOGIC_L:
	case SIG_STD_LOGIC_0:
		val = 0;
		break;
	case SIG_STD_LOGIC_H:
	case SIG_STD_LOGIC_1:
		val = 1;
		break;
	default:
		val = 0;
		break;
	}

	cpssp->NAME.state_nss = val;
}

static void
NAME_(sck_in_set)(struct cpssp *cpssp, unsigned int val)
{
	switch (val) {
	case SIG_STD_LOGIC_L:
	case SIG_STD_LOGIC_0:
		val = 0;
		break;
	case SIG_STD_LOGIC_H:
	case SIG_STD_LOGIC_1:
		val = 1;
		break;
	default:
		val = 0;
		break;
	}

	cpssp->NAME.state_sck = val;
}

static void
NAME_(clk2)(struct cpssp *cpssp)
{
	unsigned int bit;

again:	;
	switch (cpssp->NAME.state) {
	case NAME_(STATE_IDLE):
		if (cpssp->NAME.state_sub == 0) {
			cpssp->NAME.state_sub++;
			NAME_(mosi_out_set)(cpssp, SIG_STD_LOGIC_1);
			NAME_(sck_out_set)(cpssp, SIG_STD_LOGIC_1);
		}
		break;
	case NAME_(STATE_SEND):
		switch (cpssp->NAME.state_sub++) {
		case 0 + 0:
		case 0 + 2:
		case 0 + 4:
		case 0 + 6:
		case 0 + 8:
		case 0 + 10:
		case 0 + 12:
		case 0 + 14:
			bit = (cpssp->NAME.dr >> 7) & 1; /* FIXME */
			cpssp->NAME.dr <<= 1;
			NAME_(sck_out_set)(cpssp, SIG_STD_LOGIC_0);
			NAME_(mosi_out_set)(cpssp,
					bit ? SIG_STD_LOGIC_1 : SIG_STD_LOGIC_0);
			break;
		case 1 + 0:
		case 1 + 2:
		case 1 + 4:
		case 1 + 6:
		case 1 + 8:
		case 1 + 10:
		case 1 + 12:
		case 1 + 14:
			bit = cpssp->NAME.state_miso;
			cpssp->NAME.dr |= bit;
			NAME_(sck_out_set)(cpssp, SIG_STD_LOGIC_1);
			break;
		case 0 + 16:
			cpssp->NAME.rxne = 1;
			cpssp->NAME.state = NAME_(STATE_IDLE);
			cpssp->NAME.state_sub = 0;
			goto again;
		}
		break;
	default:
		assert(0); /* Mustn't happen. */
	}
}

static void
NAME_(clk)(struct cpssp *cpssp)
{
	cpssp->NAME.counter++;
	if (2 << cpssp->NAME.br <= cpssp->NAME.counter) {
		cpssp->NAME.counter = 0;
		NAME_(clk2)(cpssp);
	}
}

static void
NAME_(create)(struct cpssp *cpssp)
{
	cpssp->NAME.counter = 0;

	cpssp->NAME.state_miso = 0;
	cpssp->NAME.state_mosi = 0;
	cpssp->NAME.state_sck = 0;
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG_CONTROL_FLOW
