/***************************************************************************
 $RCSfile$
                             -------------------
    cvs         : $Id$
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "category.h"
#include "app.h"

#include <gwenhywfar/debug.h>
#include <stdlib.h>
#include <assert.h>




Category::Category(const std::string &id,
                   const std::string &name, const std::string &descr,
                   Category *parent)
:_parent(parent)
,_id(id)
,_name(name)
,_description(descr)
,_isIncome(false)
,_value(AB_Value_new())
,_dbRules(GWEN_DB_Group_new("rules")){
}



Category::Category(const Category &cat)
:_parent(0)
,_id(cat._id)
,_name(cat._name)
,_description(cat._description)
,_isIncome(cat._isIncome)
,_value(AB_Value_dup(cat._value))
,_dbRules(GWEN_DB_Group_dup(cat._dbRules)){
  std::list<Category*>::const_iterator it;

  for (it=cat._children.begin();
       it!=cat._children.end();
       it++) {
    _children.push_back(new Category(*(*it)));
  }
}



Category::Category(GWEN_DB_NODE *db, Category *parent)
:_parent(parent)
,_isIncome(false)
,_value(0)
,_dbRules(0) {
  GWEN_DB_NODE *dbT;
  const char *s;

  s=GWEN_DB_GetCharValue(db, "id", 0, 0);
  assert(s && *s);
  _id=s;
  s=GWEN_DB_GetCharValue(db, "name", 0, 0);
  assert(s && *s);
  _name=s;
  s=GWEN_DB_GetCharValue(db, "description", 0, 0);
  if (s)
    _description=s;
  _isIncome=GWEN_DB_GetIntValue(db, "isIncome", 0, 0);

  dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST, "rules");
  if (dbT)
    _dbRules=GWEN_DB_Group_dup(dbT);
  else
    _dbRules=GWEN_DB_Group_new("rules");

  dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST, "value");
  if (dbT)
    _value=AB_Value_fromDb(dbT);
  if (!_value)
    _value=AB_Value_new();

  /* read children */
  dbT=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST, "categories");
  if (dbT) {
    dbT=GWEN_DB_FindFirstGroup(dbT, "category");
    while(dbT) {
      _children.push_back(new Category(dbT, this));
      dbT=GWEN_DB_FindNextGroup(dbT, "category");
    }
  }
}



Category::~Category() {
  std::list<Category*>::iterator it;

  GWEN_DB_Group_free(_dbRules);
  for (it=_children.begin();
       it!=_children.end();
       it++) {
    delete (*it);
  }
  AB_Value_free(_value);
  _children.clear();
}



void Category::_setIsIncomeToBranch(bool b) {
  std::list<Category*>::iterator it;

  _isIncome=b;
  for (it=_children.begin();
       it!=_children.end();
       it++) {
    (*it)->_setIsIncomeToBranch(b);
  }
}



const AB_VALUE *Category::getValue() const {
  return _value;
}



void Category::addValue(const AB_VALUE *v) {
  AB_Value_AddValue(_value, v);
  if (_parent)
    _parent->addValue(v);
}



void Category::resetValues() {
  std::list<Category*>::iterator it;

  for (it=_children.begin();
       it!=_children.end();
       it++)
    (*it)->resetValues();
  if (_parent) {
    AB_Value_Negate(_value);
    _parent->addValue(_value);
  }
  AB_Value_free(_value);
  _value=AB_Value_new();
}



void Category::addCategory(Category *cat) {
  cat->_parent=this;
  cat->_setIsIncomeToBranch(_isIncome);
  cat->_path="";
  _children.push_back(cat);

}



void Category::unlink() {
  if (_parent) {
    std::list<Category*>::iterator it;

    for (it=_parent->_children.begin();
         it!=_parent->_children.end();
         it++) {
      if ((*it)==this) {
        _parent->_children.erase(it);
        break;
      }
    }
    _parent=0;
  }
  _path="";
}



Category *Category::getParent() {
  return _parent;
}



std::list<Category*> &Category::getChildren() {
  return _children;
}



GWEN_DB_NODE *Category::getRules() {
  return _dbRules;
}



void Category::setRules(GWEN_DB_NODE *dbRules) {
  assert(dbRules);
  GWEN_DB_Group_free(_dbRules);
  _dbRules=GWEN_DB_Group_dup(dbRules);
}



const std::string &Category::getId() const {
  return _id;
}



void Category::setId(const std::string &s) {
  _id=s;
}



const std::string &Category::getName() const {
  return _name;
}



void Category::setName(const std::string &s) {
  _name=s;
}



const std::string &Category::getDescription() const {
  return _description;
}



void Category::setDescription(const std::string &s) {
  _description=s;
}



bool Category::isIncome() const {
  return _isIncome;
}



void Category::setIsIncome(bool b) {
  _isIncome=b;
}



void Category::toDb(GWEN_DB_NODE *db) {
  GWEN_DB_NODE *dbT;

  /* checks */
  assert(!_id.empty());
  assert(!_name.empty());

  /* store data */
  GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "id", _id.c_str());
  GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "name", _name.c_str());
  if (!_description.empty())
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "description", _description.c_str());
  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                      "isIncome", _isIncome?1:0);

  /* store rules */
  dbT=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS, "rules");
  assert(dbT);
  GWEN_DB_AddGroupChildren(dbT, _dbRules);

  /* store value */
  dbT=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS, "value");
  assert(dbT);
  AB_Value_toDb(_value, dbT);

  /* store children */
  if (!_children.empty()) {
    std::list<Category*>::iterator it;
    GWEN_DB_NODE *dbSubCats;

    dbSubCats=GWEN_DB_GetGroup(db, GWEN_DB_FLAGS_OVERWRITE_GROUPS,
                               "categories");
    assert(dbSubCats);
    for (it=_children.begin();
         it!=_children.end();
         it++) {
      dbT=GWEN_DB_GetGroup(dbSubCats, GWEN_PATH_FLAGS_CREATE_GROUP,
                           "category");
      assert(dbT);
      (*it)->toDb(dbT);
    }
  }

}



const std::string &Category::getPath() {
  if (_path.empty()) {
    std::string s;
    Category *cat;

    s=_name;
    cat=_parent;
    while(cat) {
      s.insert(0, "/");
      s.insert(0, cat->_name);
      cat=cat->_parent;
    }
    _path=s;
  }

  return _path;
}







