/*
 *  Copyright (C) 2005 Geffrey Velasquez Torres <geffrey@gmail.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* $Id: header,v 1.15 2004/01/08 16:46:52 sniper Exp $ */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "php.h"
#include "php_ini.h"
#include "ext/standard/info.h"
#include "php_clamav.h"
#include <string.h>
#include <stdio.h>
#include <clamav.h>


ZEND_DECLARE_MODULE_GLOBALS(clamav)

/* True global resources - no need for thread safety here */
static int le_clamav;
static struct cl_node *root;	/* internal db representation*/
struct cl_limits limits;		/* limits configuration */ 
static int sig_num;				/* signature number */
struct cl_stat dbstat;			/* database stat */

#ifdef ZEND_ENGINE_2
static
	ZEND_BEGIN_ARG_INFO(third_and_fourth_args_force_ref, 0)
		ZEND_ARG_PASS_INFO(0)
		ZEND_ARG_PASS_INFO(0)
		ZEND_ARG_PASS_INFO(1)
		ZEND_ARG_PASS_INFO(1)
	ZEND_END_ARG_INFO()
#else /* ZEND_ENGINE_1 */
static unsigned char third_and_fourth_args_force_ref[] = 
	{ 4, BYREF_NONE, BYREF_NONE, BYREF_FORCE, BYREF_FORCE };
#endif

/* {{{ clamav_functions[]
 *
 * Every user visible function must have an entry in clamav_functions[].
 */
function_entry clamav_functions[] = {
	PHP_FE(cl_info, NULL)
	PHP_FE(cl_scanfile, NULL)
	PHP_FE(cl_scanbuff, NULL)
	PHP_FE(cl_setlimits, NULL)
	PHP_FE(cl_scanfile_ex, third_and_fourth_args_force_ref)
	PHP_FE(cl_scanbuff_ex, third_and_fourth_args_force_ref)
	PHP_FE(cl_pretcode, NULL)
	PHP_FE(clam_scan_buffer, NULL)	/* compatibility */
	PHP_FE(clam_scan_file, NULL)    /* compatibility */
	PHP_FE(clam_get_version, NULL)  /* compatibility */
	{NULL, NULL, NULL}				/* Must be the last line in clamav_functions[] */
};
/* }}} */

/* {{{ clamav_module_entry
 */
zend_module_entry clamav_module_entry = {
#if ZEND_MODULE_API_NO >= 20010901
	STANDARD_MODULE_HEADER,
#endif
	"clamav",
	clamav_functions,
	PHP_MINIT(clamav),
	PHP_MSHUTDOWN(clamav),
	PHP_RINIT(clamav),		
	PHP_RSHUTDOWN(clamav),
	PHP_MINFO(clamav),
#if ZEND_MODULE_API_NO >= 20010901
	PHP_CLAMAV_VERSION,
#endif
	STANDARD_MODULE_PROPERTIES
};
/* }}} */

#ifdef COMPILE_DL_CLAMAV
ZEND_GET_MODULE(clamav)
#endif

/* {{{ PHP_INI
 */
PHP_INI_BEGIN()

    STD_PHP_INI_ENTRY("clamav.dbpath", "/usr/local/share/clamav", PHP_INI_ALL, 
					  OnUpdateString, dbpath, zend_clamav_globals, clamav_globals)

    STD_PHP_INI_ENTRY("clamav.maxreclevel", "5", PHP_INI_ALL, OnUpdateLong, 
				 	  maxreclevel, zend_clamav_globals, clamav_globals)

    STD_PHP_INI_ENTRY("clamav.maxfiles", "1000", PHP_INI_ALL, OnUpdateLong, 
					  maxfiles, zend_clamav_globals, clamav_globals)

    STD_PHP_INI_ENTRY("clamav.archivememlim", "0", PHP_INI_ALL, OnUpdateLong, 
					  archivememlim, zend_clamav_globals, clamav_globals)

    STD_PHP_INI_ENTRY("clamav.maxfilesize", "0", PHP_INI_ALL, OnUpdateLong, 
					  maxfilesize, zend_clamav_globals, clamav_globals)

    STD_PHP_INI_ENTRY("clamav.maxratio", "200", PHP_INI_ALL, OnUpdateLong, 
					  maxratio, zend_clamav_globals, clamav_globals)

PHP_INI_END()
/* }}} */

/* {{{ php_clamav_init_globals
 */
static void php_clamav_init_globals(zend_clamav_globals *clamav_globals)
{
	clamav_globals->dbpath        = NULL;
	clamav_globals->maxreclevel   = 0;
	clamav_globals->maxfiles      = 0;
	clamav_globals->archivememlim = 0;
	clamav_globals->maxfilesize   = 0;
	clamav_globals->maxratio      = 0;
}
/* }}} */

/* {{{ PHP_MINIT_FUNCTION
 */
PHP_MINIT_FUNCTION(clamav)
{
	ZEND_INIT_MODULE_GLOBALS(clamav, php_clamav_init_globals, NULL);
	REGISTER_INI_ENTRIES();
	
	int ret;	/* return value */

	/* database loading */
	if ((ret = cl_loaddbdir(CLAMAV_G(dbpath), &root, &sig_num))) {
		php_error(E_WARNING, "cl_loaddbdir: %s\n", cl_perror(ret));
		return FAILURE;
	}

	if (( ret = cl_build(root))) {
		php_error(E_WARNING, "Database initialization error: %s\n", cl_strerror(ret));
		cl_free(root);
		return FAILURE;
	}

	/* database stat */
    memset(&dbstat, 0, sizeof(struct cl_stat));
    cl_statinidir(CLAMAV_G(dbpath), &dbstat);

	/* archive limits */
	memset(&limits, 0, sizeof(struct cl_limits));
	limits.maxfiles      = CLAMAV_G(maxfiles); 		/* max files */	
	limits.maxfilesize   = CLAMAV_G(maxfilesize);	/* maximal archived file size */
	limits.maxreclevel   = CLAMAV_G(maxreclevel);	/* maximal recursion level */
	limits.maxratio      = CLAMAV_G(maxratio);		/* maximal compression ratio */
	limits.archivememlim = CLAMAV_G(archivememlim);	/* disable memory limit for bzip2 scanner */

	/*  Register ClamAV scan options, they're also availible inside
	 *  php scripts with the same name and value.
	 */

	REGISTER_LONG_CONSTANT("CL_SCAN_RAW", CL_SCAN_RAW, 
							CONST_CS | CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("CL_SCAN_ARCHIVE", CL_SCAN_ARCHIVE, 
							CONST_CS | CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("CL_SCAN_MAIL", CL_SCAN_MAIL, 
							CONST_CS | CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("CL_SCAN_DISABLERAR", CL_SCAN_DISABLERAR, 
							CONST_CS | CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("CL_SCAN_OLE2", CL_SCAN_OLE2, 
							CONST_CS | CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("CL_SCAN_BLOCKENCRYPTED", CL_SCAN_BLOCKENCRYPTED, 
							CONST_CS | CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("CL_SCAN_HTML", CL_SCAN_HTML, 
							CONST_CS | CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("CL_SCAN_PE", CL_SCAN_PE, 
							CONST_CS | CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("CL_SCAN_BLOCKBROKEN", CL_SCAN_BLOCKBROKEN, 
							CONST_CS | CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("CL_SCAN_MAILURL", CL_SCAN_MAILURL, 
							CONST_CS | CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("CL_SCAN_BLOCKMAX", CL_SCAN_BLOCKMAX, 
							CONST_CS | CONST_PERSISTENT);
	REGISTER_LONG_CONSTANT("CL_SCAN_STDOPT", CL_SCAN_STDOPT, 
							CONST_CS | CONST_PERSISTENT);

	/*  Register ClamAV return codes, they're also available
	 *  inside php scripts with the same name and value.
	 */
    REGISTER_LONG_CONSTANT("CL_CLEAN", CL_CLEAN,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_VIRUS", CL_VIRUS,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EMAXREC", CL_EMAXREC,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EMAXSIZE", CL_EMAXSIZE,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EMAXFILES", CL_EMAXFILES,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_ERAR", CL_ERAR,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EZIP", CL_EZIP,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EMALFZIP", CL_EMALFZIP,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EGZIP", CL_EGZIP,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EBZIP", CL_EBZIP,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EOLE2", CL_EOLE2,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EMSCOMP", CL_EMSCOMP,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EMSCAB", CL_EMSCAB,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EACCES", CL_EACCES,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_ENULLARG", CL_ENULLARG,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_ETMPFILE", CL_ETMPFILE,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EFSYNC", CL_EFSYNC,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EMEM", CL_EMEM,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EOPEN", CL_EOPEN,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EMALFDB", CL_EMALFDB,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EPATSHORT", CL_EPATSHORT,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_ETMPDIR", CL_ETMPDIR,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_ECVD", CL_ECVD,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_ECVDEXTR", CL_ECVDEXTR,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EMD5", CL_EMD5,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EDSIG", CL_EDSIG,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EIO", CL_EIO,
                            CONST_CS | CONST_PERSISTENT);
    REGISTER_LONG_CONSTANT("CL_EFORMAT", CL_EFORMAT,
                            CONST_CS | CONST_PERSISTENT);

	return SUCCESS;
}
/* }}} */

/* {{{ PHP_MSHUTDOWN_FUNCTION
 */
PHP_MSHUTDOWN_FUNCTION(clamav)
{
	UNREGISTER_INI_ENTRIES();

	/* free data structures*/    
	cl_statfree(&dbstat);

	if (root) {	
		cl_free(root);
	}

	return SUCCESS;
}
/* }}} */

/* Remove if there's nothing to do at request start */
/* {{{ PHP_RINIT_FUNCTION
 */
PHP_RINIT_FUNCTION(clamav)
{
    int ret;              /* return value */

    if (cl_statchkdir(&dbstat) == 1) {
        /* database reload */
    	if (root) {
	    	cl_free(root);
		}
        root    = NULL;
        sig_num = 0;

        if ((ret = cl_loaddbdir(CLAMAV_G(dbpath), &root,
                                &sig_num)))
        {
            php_error(E_WARNING, "cl_loaddbdir: %s\n", cl_strerror(ret));
            return FAILURE;
        }

        if (( ret = cl_build(root))) {
            php_error(E_WARNING,"Database initialization error: %s\n", cl_strerror(ret));;
            cl_free(root);
            return FAILURE;
        }
        
		cl_statfree(&dbstat);
    	cl_statinidir(CLAMAV_G(dbpath), &dbstat);
    }

	return SUCCESS;
}
/* }}} */

/* Remove if there's nothing to do at request end */
/* {{{ PHP_RSHUTDOWN_FUNCTION
 */
PHP_RSHUTDOWN_FUNCTION(clamav)
{
	return SUCCESS;
}
/* }}} */

/* {{{ PHP_MINFO_FUNCTION
 */
PHP_MINFO_FUNCTION(clamav)
{
	php_info_print_table_start();
	php_info_print_table_header(2, "clamav support", "enabled");
	php_info_print_table_end();

	DISPLAY_INI_ENTRIES();
}
/* }}} */

/* {{{ php_cl_scanfile() */

static void php_cl_scanfile(INTERNAL_FUNCTION_PARAMETERS, int compat)
{
	const int NUM_ARGS = 1;
	char *filename;   	/* file to be scanned */
	int filename_len; 	/* length of the string */
	int ret;			/* clamav functions return value */
	const char *virname;

	/* argument checking */
	if (ZEND_NUM_ARGS() != NUM_ARGS) {
		WRONG_PARAM_COUNT;	
		return FAILURE;
	}

	/* argument parsing */
	if (zend_parse_parameters(NUM_ARGS TSRMLS_CC, "s", &filename, &filename_len) == FAILURE) {
    	return FAILURE;
	}
	
	ret = cl_scanfile(filename, &virname, 0, root, &limits, CL_SCAN_STDOPT);

	if (ret == CL_VIRUS) { 
        RETURN_STRINGL((char *)virname,strlen(virname),1);
	} else if (ret == CL_CLEAN) {
		if (compat == 0) RETURN_NULL() else RETURN_FALSE;
		
	} else {
	    php_error(E_WARNING,"Error: %s", cl_strerror(ret));
        if (compat == 0) RETURN_NULL() else RETURN_FALSE;
	}
}

/* }}} */

/* {{{ php_cl_scanbuff() */

static void php_cl_scanbuff(INTERNAL_FUNCTION_PARAMETERS, int compat)
{
    const int NUM_ARGS = 1;
    char *buffer;       /* buffer to be scanned */
    int buffer_len; 	/* length of the string */
    int ret;            /* clamav functions return value */
    const char *virname;

    /* argument checking */
    if (ZEND_NUM_ARGS() != NUM_ARGS) {
        WRONG_PARAM_COUNT;
        return FAILURE;
    }
    /* argument parsing */
    if (zend_parse_parameters(NUM_ARGS TSRMLS_CC, "s", &buffer, &buffer_len) == FAILURE) {
        return FAILURE;
    }

    ret = cl_scanbuff(buffer, buffer_len, &virname, root);
 
    if (ret == CL_VIRUS) {
        RETURN_STRINGL((char *)virname,strlen(virname),1);
    } else if (ret == CL_CLEAN) {
        if (compat == 0) RETURN_NULL() else RETURN_FALSE;
    } else {
        php_error(E_WARNING,"error: %s", cl_strerror(ret));
        if (compat == 0) RETURN_NULL() else RETURN_FALSE;
    }
}

/* }}} */

/* {{{ php_cl_info() */

static void php_cl_info(INTERNAL_FUNCTION_PARAMETERS, int compat)
{
	if (compat == 0) {
		php_printf("ClamAV version %s with %d virus signatures loaded", 
				   cl_retver(), sig_num);
	} else {
		/* this code has been 'cut and pasted' from the php_clam code: */
		const char *clam_v = cl_retver();
    	RETURN_STRINGL((char *)clam_v,strlen(clam_v),1);
	}
}

/* }}} */

/* {{{ proto void cl_info()
   Prints ClamAV version */

PHP_FUNCTION(cl_info)
{
	php_cl_info(INTERNAL_FUNCTION_PARAM_PASSTHRU, 0);
}

/* }}} */

/* {{{ proto string cl_scanfile(string filename)
   Scans the contents of a file, given a filename */

PHP_FUNCTION(cl_scanfile)
{
	php_cl_scanfile(INTERNAL_FUNCTION_PARAM_PASSTHRU, 0);
}

/* }}} */

/* {{{ proto string cl_scanbuff(string buffer)
   Scans a buffer, given a string (buffer). This function is not fully tested */

PHP_FUNCTION(cl_scanbuff)
{
	php_cl_scanbuff(INTERNAL_FUNCTION_PARAM_PASSTHRU, 0);
}

/* }}} */

/* {{{ proto void cl_setlimits(int maxfiles, int maxfilesize, int maxreclevel, 
							   int maxratio, int archivememlim) 
   Set the ClamAV limits for scanning */ 

PHP_FUNCTION(cl_setlimits)
{
	/* number of arguments */
	const int NUM_ARGS	= 5;
	
	int maxfiles;
	int maxfilesize;
	int maxreclevel;
	int maxratio;
	int archivememlim;

	/* argument checking */
	if (ZEND_NUM_ARGS() != NUM_ARGS) {
		WRONG_PARAM_COUNT;
		return FAILURE;	
	}

	/* argument parsing */
	if (zend_parse_parameters(NUM_ARGS TSRMLS_CC, "lllll", 
							  &maxfiles, &maxfilesize,
							  &maxreclevel, &maxratio, 
							  &archivememlim ) == FAILURE) 
		return FAILURE;
	
	/* archive limits */
	memset(&limits, 0, sizeof(struct cl_limits));
	limits.maxfiles      = maxfiles; 		/* max files */	
	limits.maxfilesize   = maxfilesize;		/* maximal archived file size */
	limits.maxreclevel   = maxreclevel;		/* maximal recursion level */
	limits.maxratio      = maxratio;		/* maximal compression ratio */
	limits.archivememlim = archivememlim;	/* disable memory limit for bzip2 scanner */

	RETURN_TRUE;	
}

/* }}} */

/* {{{ proto int cl_scanfile_ex(string filename, int options, string virusname, 
								 int retcode) 
   Scans the contents of a file, given a filename. It accepts ClamAV options 
   and returns the virus name (if it was found) and the ClamAV return code */

PHP_FUNCTION(cl_scanfile_ex) 
{
	/* number of arguments */
	const int NUM_ARGS	= 4;
	
	/* parameters */
	zval *filename;
	zval *virusname;
	zval *options;
	zval *retcode;
	
	/* used for cl_scanfile function */
	char *virname	= NULL;
	int ret;
	
	if (ZEND_NUM_ARGS() != NUM_ARGS) {
		WRONG_PARAM_COUNT;
		RETURN_FALSE;
	}
	
	/* parameters parsing */	
	if (zend_parse_parameters(NUM_ARGS TSRMLS_CC, "zzzz", &filename, &options,  &virusname, 
								&retcode) != SUCCESS) {
		WRONG_PARAM_COUNT;
		RETURN_FALSE;	
	}

	/* parameter conversion */
	convert_to_string_ex(&filename);
	convert_to_long_ex(&options);

	/* clean up old values first */
	zval_dtor(virusname);	
	zval_dtor(retcode);

	/* executing the ClamAV virus checking function */
    ret = cl_scanfile(Z_STRVAL_P(filename), &virname, 0,
					  root, &limits, Z_LVAL_P(options));

	/* copy the value of the cl_scanfile return value */	
	ZVAL_LONG(retcode, ret);
	
	/* copy the value of the cl_scanfile virusname if a virus was found */
	if (ret == CL_VIRUS) 
		ZVAL_STRING(virusname, virname, 1);
		
	RETURN_TRUE;
}

/* }}} */

/* {{{ proto int cl_scanbuff_ex(string buff, int buffsize, string virusname, 
								 int retcode) 
   Scans a buffer, given a string buffer.It returns the virus name 
   (if it was found) and the ClamAV return code */

PHP_FUNCTION(cl_scanbuff_ex) 
{
	/* number of arguments */
	const int NUM_ARGS	= 4;

	/* parameters */
	zval *buff;
	zval *buff_size;
	zval *virusname;
	zval *retcode;

	/* used for cl_scanfile function */
	char *virname	= NULL;
	int ret;

	if (ZEND_NUM_ARGS() != NUM_ARGS) {
		WRONG_PARAM_COUNT;
		RETURN_FALSE;
	}
	
	/* parameters parsing */	
	if (zend_parse_parameters(NUM_ARGS TSRMLS_CC, "zzzz", &buff, &buff_size,  &virusname, 
							   &retcode) != SUCCESS) {
		WRONG_PARAM_COUNT;
		RETURN_FALSE;
	}

	/* parameter conversion */
	convert_to_string_ex(&buff);
	convert_to_long_ex(&buff_size);
	
	/* clean up old values first */
	zval_dtor(virusname);	
	zval_dtor(retcode);
	
	/* executing the ClamAV virus checking function */
    ret = cl_scanbuff(Z_STRVAL_P(buff), Z_LVAL_P(buff_size), 
					  &virname, root);

	/* copy the value of the cl_scanfile return value */	
	ZVAL_LONG(retcode, ret);
	
	/* copy the value of the cl_scanfile virusname if a virus was found */
	if (ret == CL_VIRUS) 
		ZVAL_STRING(virusname, virname, 1);

	RETURN_TRUE;
}

/* }}} */

/* {{{ proto string cl_pretcode(int retcode)
   Translates the ClamAV return code */

PHP_FUNCTION(cl_pretcode)
{
	/* number of arguments */
	const int NUM_ARGS	= 1;
	
	/* parameters */
   	long retcode;
	
	if (ZEND_NUM_ARGS() != NUM_ARGS)
		WRONG_PARAM_COUNT;

	/* parameters parsing */
	if (zend_parse_parameters(NUM_ARGS TSRMLS_CC, "l", &retcode) != SUCCESS)
		WRONG_PARAM_COUNT;

	switch (retcode) {
		case CL_CLEAN:
			RETVAL_STRING("virus not found", 1);
			break;
		case CL_VIRUS:
			RETURN_STRING("virus found", 1);
			break;
		case CL_EMAXREC:
			RETURN_STRING("recursion level limit exceeded", 1);
			break;
		case CL_ERAR:
			RETURN_STRING("rar handler error", 1);
			break;
		case CL_EZIP:
			RETURN_STRING("zip handler error", 1);
			break;
		case CL_EMALFZIP:
			RETURN_STRING("malformed zip", 1);
			break;
		case CL_EGZIP:
			RETURN_STRING("gzip handler error", 1);
			break;
		case CL_EBZIP:
			RETURN_STRING("bzip2 handler error", 1);
			break;
		case CL_EOLE2:
			RETURN_STRING("OLE2 handler error", 1);
			break;
		case CL_EMSCOMP:
			RETURN_STRING("compress.exe handler error", 1);
			break;
		case CL_EMSCAB:
			RETURN_STRING("MS CAB module error", 1);
			break;
		case CL_EACCES:
			RETURN_STRING("access denied", 1);
			break;
		case CL_ENULLARG:
			RETURN_STRING("null argument error", 1);
			break;
		case CL_ETMPFILE:
			RETURN_STRING("tmpfile() failed", 1);
			break;
		case CL_EFSYNC:
			RETURN_STRING("fsync() failed", 1);
			break;
		case CL_EMEM:
			RETURN_STRING("memory allocation error", 1);
			break;
		case CL_EOPEN:
			RETURN_STRING("file open error", 1);
			break;
		case CL_EMALFDB:
			RETURN_STRING("malformed database", 1);
			break;
		case CL_EPATSHORT:
			RETURN_STRING("pattern too short", 1);
			break;
		case CL_ETMPDIR:
			RETURN_STRING("mkdir() failed", 1);
			break;
		case CL_ECVD:
			RETURN_STRING("not a CVD file (or broken)", 1);
			break;
		case CL_ECVDEXTR:
			RETURN_STRING("CVD extraction failure", 1);
			break;
		case CL_EMD5:
			RETURN_STRING("MD5 verification error", 1);
			break;
		case CL_EDSIG:
			RETURN_STRING("digital signature verification error", 1);
			break;
		case CL_EIO:
			RETURN_STRING("general I/O error", 1);
			break;
		case CL_EFORMAT:
			RETURN_STRING("bad format or broken file", 1);
			break;
		default :
			RETURN_STRING("unknow return code", 1);
			break;
	}	

	return SUCCESS;
}

/* }}} */

/* compatibility functions */

/* {{{ proto string clam_scan_buffer($buffer) */
PHP_FUNCTION(clam_scan_buffer)
{
	php_cl_scanbuff(INTERNAL_FUNCTION_PARAM_PASSTHRU, 1);
	return SUCCESS;
}

/* }}} */

/* {{{ proto string clam_scan_file($filename) */

PHP_FUNCTION(clam_scan_file)
{
	php_cl_scanfile(INTERNAL_FUNCTION_PARAM_PASSTHRU, 1);
	return SUCCESS;
}

/* }}} */

/* {{{ proto string clam_get_version() */

PHP_FUNCTION(clam_get_version)
{
	php_cl_info(INTERNAL_FUNCTION_PARAM_PASSTHRU, 1);
	return SUCCESS;

}

/* }}} */

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim600: noet sw=4 ts=4 fdm=marker
 * vim<600: noet sw=4 ts=4
 */
