# -*- coding: utf-8 -*-

# Session save/restore facility for Epiphany
#
# Copyright © 2007 Simon E. Ward <simon@bleah.co.uk>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of version 2 of the GNU General Public
# License as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA

import epiphany
import gtk
import os.path

_ui_str = """
<ui>
	<menubar name="menubar">
		<menu name="ToolsMenu" action="Tools">
			<separator/>
			<menuitem name="LoadSession" action="LoadSession"/>
			<menuitem name="SaveSession" action="SaveSession"/>
			<separator/>
		</menu>
	</menubar>
</ui>
"""

def _warn_on_overwrite(parent, filename):
	dirname, basename = os.path.split(filename)
	msg = 'A file named "%s" already exists in the directory "%s".'\
	      'Do you want to overwrite it?' % (basename, dirname)
	dialog = gtk.MessageDialog(
			parent=parent,
			flags=gtk.DIALOG_MODAL
			     |gtk.DIALOG_DESTROY_WITH_PARENT,
			type=gtk.MESSAGE_WARNING,
			buttons=gtk.BUTTONS_YES_NO,
			message_format=msg)
	overwrite = dialog.run()
	dialog.destroy()
	return overwrite

def _show_dialog(title, action):
	if action == gtk.FILE_CHOOSER_ACTION_SAVE:
		ok_button = gtk.STOCK_SAVE
	else:
		ok_button = gtk.STOCK_OPEN

	dialog = gtk.FileChooserDialog(
			title=title,
			action=action,
			buttons=(
				gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
				ok_button, gtk.RESPONSE_OK
				)
			)
	dialog.set_default_response(gtk.RESPONSE_OK)

	response = dialog.run()
	if response == gtk.RESPONSE_OK:
		filename = dialog.get_filename()

		if action == gtk.FILE_CHOOSER_ACTION_SAVE:
			while os.path.exists(filename):
				if _warn_on_overwrite(dialog, filename) \
					== gtk.RESPONSE_YES:
					break
				response = dialog.run()
				if response != gtk.RESPONSE_OK:
					filename = None
					break
				filename = dialog.get_filename()
	else:
		filename = None

	dialog.destroy()
	return filename

def _get_session():
	shell = epiphany.ephy_embed_shell_get_default()
	return shell.get_session()

def _save_session(action, session):
	filename = _show_dialog('Save Session',
				gtk.FILE_CHOOSER_ACTION_SAVE)
	if filename:
		session.save(filename)

def _load_session(action, session):
	filename = _show_dialog('Load Session',
				gtk.FILE_CHOOSER_ACTION_OPEN)
	if filename:
		session.load(filename, 0)

_actions = [
	('LoadSession', "gtk-open", 'Load Session…', None, None, _load_session),
	('SaveSession', "gtk-save-as", 'Save Session…', None, None, _save_session),
	]

def attach_window(window):
	session = _get_session()
	ui_manager = window.get_ui_manager()
	group = gtk.ActionGroup('SessionManager')
	group.add_actions(_actions, user_data=session)
	ui_manager.insert_action_group(group, -1)
	ui_id = ui_manager.add_ui_from_string(_ui_str)

	window._session_manager_data = (group, ui_id)

def detach_window(window):
	group, ui_id = window._session_manager_data
	del window._session_manager_data

	ui_manager = window.get_ui_manager()
	ui_manager.remove_ui(ui_id)
	ui_manager.remove_action_group(group)
	ui_manager.ensure_update()
