from shutil import rmtree

from bzrlib.errors import NoWorkingTree, NotLocalUrl, NotBranchError
from bzrlib.delta import compare_trees
from bzrlib.workingtree import WorkingTree

from errors import NotCheckout, UncommittedCheckout


def zap(path, remove_branch=False):
    try:
        wt = WorkingTree.open(path)
    except (NoWorkingTree, NotBranchError):
        raise NotCheckout(path)
    tree_base = wt.bzrdir.transport.base
    branch = wt.branch
    branch_base = branch.bzrdir.transport.base
    if tree_base == branch_base:
        raise NotCheckout(path)
    delta = compare_trees(wt.basis_tree(), wt, want_unchanged=False)
    if delta.has_changed():
        raise UncommittedCheckout()
    rmtree(path)
    if remove_branch:
        t = branch.bzrdir.transport
        while t.base != branch_base:
            t = t.clone('..')
        t = t.clone('..')
        t.delete_tree('')


def test_suite():
    import os
    from unittest import makeSuite
    
    from bzrlib.bzrdir import BzrDir, BzrDirMetaFormat1
    from bzrlib.branch import BranchReferenceFormat
    from bzrlib.tests import TestCaseInTempDir

    class TestZap(TestCaseInTempDir):

        def make_checkout(self):
            wt = BzrDir.create_standalone_workingtree('source')
            os.mkdir('checkout')
            checkout = BzrDirMetaFormat1().initialize('checkout')
            BranchReferenceFormat().initialize(checkout, wt.branch)
            return checkout.create_workingtree()

        def test_is_checkout(self):
            self.assertRaises(NotCheckout, zap, '.')
            wt = BzrDir.create_standalone_workingtree('.')
            self.assertRaises(NotCheckout, zap, '.')

        def test_zap_works(self):
            self.make_checkout()
            self.assertIs(True, os.path.exists('checkout'))
            zap('checkout')
            self.assertIs(False, os.path.exists('checkout'))
            self.assertIs(True, os.path.exists('source'))

	def test_zap_branch(self):
            self.make_checkout()
            base = WorkingTree.open('checkout').branch.base
            self.assertIs(True, os.path.exists('checkout'))
            zap('checkout', remove_branch=True)
            self.assertIs(False, os.path.exists('checkout'))
            self.assertIs(False, os.path.exists('source'))

        def test_checks_modified(self):
            checkout = self.make_checkout()
            os.mkdir('checkout/foo')
            checkout.add('foo')
            self.assertRaises(UncommittedCheckout, zap, 'checkout')
            checkout.commit('commit changes to branch')
            zap('checkout')

    return makeSuite(TestZap)
