#include <stdio.h>

#include <gc/gc.h>

#include "util.h"
#include "format.h"
#include "process.h"

const char *format_value(Tree val)
{
  int bufsz = 64;
  char *buffer = (char *)GC_malloc(bufsz);
  
  switch (TREE_CODE(val))
  {
    case INTEGER_CST:
      if (TREE_UNSIGNED(val))
        sprintf(buffer, "(unsigned long)%lu", TREE_INT_CST(val));
      else
        sprintf(buffer, "long(%ld)", (long)TREE_INT_CST(val));
      break;
    case STRING_CST:
      if (bufsz < TREE_STRING_CST_LEN(val) + 16)
        buffer = (char *)GC_realloc(buffer, TREE_STRING_CST_LEN(val) + 20);
      sprintf(buffer, "std::string(\"%s\")", TREE_STRING_CST_PTR(val));
      break;
    default:
      process_internal_error("invalid value specifier");
      break;
  }
  
  return buffer;
}

const char *format_typedescr(Tree descr)
{
  static const char *typespec_name[TYPESPEC_COUNT] = { 0 };
  int bufsz = 64, strsz = 0;
  char *buffer = GC_malloc(bufsz);
  Tree node;

  if (typespec_name[0] == 0)
  {
    typespec_name[TYPESPEC_UNSIGNED] = "unsigned";
    typespec_name[TYPESPEC_SIGNED] = "signed";
    typespec_name[TYPESPEC_LONG] = "long";
    typespec_name[TYPESPEC_SHORT] = "short";
    typespec_name[TYPESPEC_INT] = "int";
    typespec_name[TYPESPEC_CHAR] = "char";
    typespec_name[TYPESPEC_FLOAT] = "float";
    typespec_name[TYPESPEC_DOUBLE] = "double";
    typespec_name[TYPESPEC_BOOL] = "bool";
  }

  buffer[0] = '\0';

  if (TREE_TYPEDESCR_CONST(descr))
    buffer = GC_str_append(buffer, "const ", 6, &strsz, &bufsz);

  node = TREE_TYPEDESCR_SPECS(descr);

  /* in node we either have a 
   *   1) typename: TREE_LIST: [TREE_NULL] IDENTIFIER_NODE ...
   *        the optional TREE_NULL means we have a global scoped name 
   *   2) a built in type: TREE_LIST: TYPE_SPEC ...
   */
  if (TREE_VALUE(node) != NULL_TREE &&
      TREE_CODE(TREE_VALUE(node)) == TYPE_SPEC)
  {
    while (1)
    {
      buffer = GC_str_append(
              buffer, typespec_name[TREE_TYPESPEC(TREE_VALUE(node))], -1,
              &strsz, &bufsz);
      
      if ((node = TREE_CHAIN(node)) != NULL_TREE)
        buffer = GC_str_append(buffer, " ", 1, &strsz, &bufsz);
      else
        break;
    }
  }
  else
    buffer = GC_str_append(buffer, format_typename(node), -1, &strsz, &bufsz);
  
  switch (TREE_TYPEDESCR(descr))
  {
    case TYPEDESCR_NORMAL:
      break;
    case TYPEDESCR_REF:
      buffer = GC_str_append(buffer, "&", 1, &strsz, &bufsz);
      break;
    case TYPEDESCR_PTR:
      buffer = GC_str_append(buffer, " *", 2, &strsz, &bufsz);
      break;
  }
  
  return buffer;
}

const char *format_typename(Tree t)
{
  int bufsz = 64, strsz = 0;
  char *buffer = GC_malloc(bufsz);
  Tree node = t;
  
  buffer[0] = '\0';

  while (1)
  {
    Tree value = TREE_VALUE(node);
    
    if (value == NULL_TREE)
      buffer = GC_str_append(buffer, "::", 2, &strsz, &bufsz);
    else
    {
      Tree ident = TREE_TYPENAME_IDENTIFIER(value);
      Tree tspec = TREE_TYPENAME_TEMPLATE_SPEC(value);
      buffer = GC_str_append(buffer, TREE_IDENTIFIER_STR(ident),
                             TREE_IDENTIFIER_LEN(ident), &strsz, &bufsz);

      if (tspec)
      {
        Tree node1;
        buffer = GC_str_append(buffer, "<", 1, &strsz, &bufsz);
        for (node1 = tspec; node1; node1 = TREE_CHAIN(node1))
        {
          buffer = GC_str_append(buffer, format_typedescr(TREE_VALUE(node1)),
                                 -1, &strsz, &bufsz);
          if (TREE_CHAIN(node1))
            buffer = GC_str_append(buffer, ", ", 2, &strsz, &bufsz);
        }
        buffer = GC_str_append(buffer, " >", 2, &strsz, &bufsz);
      }
    }
    if ((node = TREE_CHAIN(node)) == NULL_TREE)
      break;

    if (value)
      buffer = GC_str_append(buffer, "::", 2, &strsz, &bufsz);
  }

  return buffer;
}

const char *format_typelist(Tree t)
{
  int strsz = 0, bufsz = 1;
  char *buffer = GC_malloc(bufsz);
  Tree node = t;
  
  buffer[0] = '\0';
  while (node)
  {
    Tree value = TREE_VALUE(node);
    buffer = GC_str_append(buffer, format_typedescr(value), -1,
                           &strsz, &bufsz);
    if ((node = TREE_CHAIN(node)) != NULL_TREE)
      buffer = GC_str_append(buffer, ", ", 2, &strsz, &bufsz);
  }
  return buffer;
}

const char *format_paramlist(Tree t, int fmt)
{
  int strsz = 0, bufsz = 1;
  char *buffer = GC_malloc(bufsz);
  int i = 1;
  char buf[32];
  Tree node = t;
  
  buffer[0] = '\0';
  while (node)
  {
    Tree value = TREE_VALUE(node);
    Tree type = TREE_PARAMDESCR_TYPE(value);
    Tree name = TREE_PARAMDESCR_NAME(value);
    
    if (type == NULL_TREE)
      return buffer; /* this is the empty paramlist */

    if (fmt & PARAM_TYPE)
      buffer = GC_str_append(buffer, format_typedescr(type), -1,
                             &strsz, &bufsz);
    if (fmt & PARAM_NAME)
    {
      if ((fmt & PARAM_TYPE) && buffer[strsz - 1] != '*')
        buffer = GC_str_append(buffer, " ", 1, &strsz, &bufsz);

      if (fmt & PARAM_NUMBER_NAME)
      {
        sprintf(buf, "p%i", i++);
        buffer = GC_str_append(buffer, buf, -1, &strsz, &bufsz);
      }
      else
        buffer = GC_str_append(buffer, TREE_IDENTIFIER_STR(name),
                               TREE_IDENTIFIER_LEN(name), &strsz, &bufsz);
    }
    if ((node = TREE_CHAIN(node)) != NULL_TREE)
      buffer = GC_str_append(buffer, ", ", 2, &strsz, &bufsz);
  }
  return buffer;
}

const char *format_dot_identifier_list(Tree t)
{
  int strsz = 0, bufsz = 1;
  char *buffer = GC_malloc(bufsz);
  Tree node = t;

  buffer[0] = '\0';
  while (node)
  {
    Tree value = TREE_VALUE(node);
    buffer = GC_str_append(buffer, TREE_IDENTIFIER_STR(value),
                           TREE_IDENTIFIER_LEN(value), &strsz, &bufsz);
    
    if ((node = TREE_CHAIN(node)) != NULL_TREE)
      buffer = GC_str_append(buffer, ".", 1, &strsz, &bufsz);
  }
  return buffer;
}
