/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <string.h>
#include <stdio.h>

#include <gtk/gtkbin.h>
#include <gtk/gtktext.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkscrolledwindow.h>

#include <gdk/gdkkeysyms.h>

#include "singit_debug.h"
#include "singit_marshallers.h"
#include "editor_view.h"

enum {
	ARG_0,
	ARG_CURSOR_TEXT_OFFSET
};

enum {
	MODIFIED,
	GET_TEXT,
	SET_TEXT,
	LAST_SIGNAL,
};

static GtkBinClass *parent_class = NULL;
static guint sev_signals[LAST_SIGNAL] = { 0 };

static void singit_editor_view_set_arg (GtkObject *object, GtkArg *arg, guint arg_id);
static void singit_editor_view_get_arg (GtkObject *object, GtkArg *arg, guint arg_id);
static void singit_editor_view_destroy (GtkObject *object);

static void singit_editor_view_size_request(GtkWidget *widget, GtkRequisition *requisition);
static void singit_editor_view_size_allocate(GtkWidget *widget, GtkAllocation *allocation);

static void singit_editor_view_class_init (SingitEditorViewClass *klass)
{
	GtkObjectClass *object_klass;
	GtkWidgetClass *widget_klass;

	object_klass = (GtkObjectClass*) klass;
	widget_klass = (GtkWidgetClass*) klass;

	parent_class = gtk_type_class(GTK_TYPE_BIN);

	gtk_object_add_arg_type ("SingitEditorView::cursoroffset",
			GTK_TYPE_UINT,
			GTK_ARG_READABLE,
			ARG_CURSOR_TEXT_OFFSET);

	sev_signals[MODIFIED] =
		gtk_signal_new("modified",
			GTK_RUN_FIRST,
			object_klass->type,
			GTK_SIGNAL_OFFSET (SingitEditorViewClass, modified),
			gtk_marshal_NONE__BOOL,
			GTK_TYPE_NONE, 1,
			GTK_TYPE_BOOL);

	sev_signals[GET_TEXT] =
		gtk_signal_new("get_text",
			GTK_RUN_LAST,
			object_klass->type,
			GTK_SIGNAL_OFFSET (SingitEditorViewClass, get_text),
			gtk_marshal_STRING__NONE,
			GTK_TYPE_POINTER, 0);

	sev_signals[SET_TEXT] =
		gtk_signal_new("set_text",
			GTK_RUN_FIRST,
			object_klass->type,
			GTK_SIGNAL_OFFSET (SingitEditorViewClass, set_text),
			gtk_marshal_NONE__STRING,
			GTK_TYPE_NONE, 1,
			GTK_TYPE_STRING);

	klass->modified = NULL;

	gtk_object_class_add_signals(object_klass, sev_signals, LAST_SIGNAL);

	object_klass->set_arg = singit_editor_view_set_arg;
	object_klass->get_arg = singit_editor_view_get_arg;
	object_klass->destroy = singit_editor_view_destroy;

	widget_klass->size_request = singit_editor_view_size_request;
	widget_klass->size_allocate = singit_editor_view_size_allocate;
}

static void singit_editor_view_init (SingitEditorView *sev)
{
	sev->cursor_text_offset = 0;
	sev->modified = FALSE;
}

GtkType singit_editor_view_get_type (void)
{
	static guint ev_type = 0;

	if (!ev_type) {
		GtkTypeInfo ev_info = {
			(gchar*) "SingitEditorView",
			sizeof (SingitEditorView),
			sizeof (SingitEditorViewClass),
			(GtkClassInitFunc) singit_editor_view_class_init,
			(GtkObjectInitFunc) singit_editor_view_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		ev_type = gtk_type_unique (GTK_TYPE_BIN, &ev_info);
	}

	return ev_type;
}

static void singit_editor_view_destroy (GtkObject *object)
{
	SingitEditorView *ev;

	g_return_if_fail (ev = SINGIT_EDITOR_VIEW (object));

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void singit_editor_view_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	SingitEditorView *editor;

	editor = SINGIT_EDITOR_VIEW(object);

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_editor_view.c [singit_editor_view_set_arg]\n"));
#endif

	switch (arg_id) {
	case ARG_CURSOR_TEXT_OFFSET:
		editor->cursor_text_offset = GTK_VALUE_UINT(*arg);
		break;
	default:
		break;
	}
}

static void singit_editor_view_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	SingitEditorView *editor;

	editor = SINGIT_EDITOR_VIEW(object);

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_editor_view.c [singit_editor_view_get_arg]\n"));
#endif

	switch (arg_id) {
	case ARG_CURSOR_TEXT_OFFSET:
		GTK_VALUE_UINT(*arg) = editor->cursor_text_offset;
		break;
	default:
		arg->type = GTK_TYPE_INVALID;
		break;
	}
}

gchar* singit_editor_view_get_text(SingitEditorView *sev, gint *char_offset)
{
	gchar *text = NULL;

	g_return_val_if_fail(IS_SINGIT_EDITOR_VIEW(sev), NULL);

	gtk_signal_emit(GTK_OBJECT(sev), sev_signals[GET_TEXT], &text);

	if (char_offset != NULL)
		*char_offset = sev->cursor_text_offset;

	return text;
}

void singit_editor_view_set_text(SingitEditorView *sev, gchar *text, gint char_offset)
{
	g_return_if_fail(IS_SINGIT_EDITOR_VIEW(sev));

	sev->cursor_text_offset = char_offset;

	gtk_signal_emit(GTK_OBJECT(sev), sev_signals[SET_TEXT], text);
}

static void singit_editor_view_size_request(GtkWidget *widget, GtkRequisition *requisition)
{
	GtkBin *bin;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (IS_SINGIT_EDITOR_VIEW (widget));
	g_return_if_fail (requisition != NULL);

	bin = GTK_BIN (widget);

	requisition->width = (GTK_CONTAINER (widget)->border_width +
		GTK_WIDGET (widget)->style->klass->xthickness) * 2;

	requisition->height = (GTK_CONTAINER (widget)->border_width +
		GTK_WIDGET (widget)->style->klass->ythickness) * 2;

	if (bin->child && GTK_WIDGET_VISIBLE (bin->child)) {
		GtkRequisition child_requisition;
		gtk_widget_size_request (bin->child, &child_requisition);

		requisition->width += child_requisition.width;
		requisition->height += child_requisition.height;
	}
}

static void singit_editor_view_size_allocate(GtkWidget *widget, GtkAllocation *allocation)
{
	g_return_if_fail (widget != NULL);
	g_return_if_fail (IS_SINGIT_EDITOR_VIEW (widget));
	g_return_if_fail (allocation != NULL);

	if (GTK_WIDGET_MAPPED (widget) &&
		((widget->allocation.x != allocation->x) ||
		(widget->allocation.y != allocation->y) ||
		(widget->allocation.width != allocation->width) ||
		(widget->allocation.height != allocation->height)) &&
		(widget->allocation.width != 0) &&
		(widget->allocation.height != 0))
	{
		gtk_widget_queue_clear (widget);
	}

	widget->allocation = *allocation;

	if (GTK_BIN(widget)->child && GTK_WIDGET_VISIBLE (GTK_BIN (widget)->child))
	{
		GtkAllocation child_allocation;

		gint border_width = GTK_CONTAINER (widget)->border_width;
		gint xthickness = GTK_WIDGET (widget)->style->klass->xthickness;
		gint ythickness = GTK_WIDGET (widget)->style->klass->ythickness;

		child_allocation.x = border_width + xthickness;
		child_allocation.width = MAX(1,
			((gint) allocation->width - child_allocation.x * 2));

		child_allocation.y = border_width + ythickness;
		child_allocation.height = MAX (1,
			((gint) allocation->height - child_allocation.y -
			 border_width - ythickness));
		child_allocation.x += allocation->x;
		child_allocation.y += allocation->y;

		gtk_widget_size_allocate (GTK_BIN (widget)->child, &child_allocation);
	}
}

void singit_editor_view_modify(SingitEditorView *sev, gboolean modified)
{
	g_return_if_fail(IS_SINGIT_EDITOR_VIEW(sev));

	if (modified != sev->modified) {
		sev->modified = modified;
		gtk_signal_emit(GTK_OBJECT(sev), sev_signals[MODIFIED], modified);
	}
}
