/*
 * Copyright (C) 2003 the xmms-kde team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/*
 * TODO: - still (rare) crashes when updating db and dialog is closed (???)
 *       - single inserts fail when querying at the same time (???)
 *         happens also when queried from command line so this seems to be 
 *         an sqlite bug
 */

#include <config.h>

#include <qstringlist.h>
#include <qdir.h>
#include <qlayout.h>
#include <qhbox.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qregexp.h>
#include <qhbuttongroup.h>
#include <qdatetime.h>
#include <qpixmap.h>

#include <kfilemetainfo.h>
#include <kwin.h>

#include <configdialog.h>

#include "xmms-kdedb.h"
#include "xmms-kdedb.moc"

struct eqstr {
  bool operator()(const char* s1, const char* s2) const {
    return strcmp(s1, s2) == 0;
 }
};

static int callback(void *NotUsed, int argc, char **argv, char **azColName) {

  int i;
  for(i = 0; i < argc; i++) {
    printf("%s = %s\n", azColName[i], argv[i] ? argv[i] : "NULL");
  }
  printf("\n");
  return 0;
}

/**
 * the table in the database has to be of the following format:
 * 
 * create table music
 * ( filename VARCHAR(255),
 *   title VARCHAR(255),
 *   artist VARCHAR(255),
 *   album VARCHAR(255),
 *   track INTEGER,
 *   year INTEGER,
 *   genre VARCHAR(255),
 *   comment VARCHAR(255));
 *
 */
XmmsKdeDB::XmmsKdeDB(KConfig *conf, QPixmap *ico) {

  db = NULL;
  pathListBox = NULL;
  insertThread = NULL;
  statusFrame = NULL;
  connected = false;

  icon = ico;
  config = conf;
  readConfig();

  sync = true;
  connectDB();
}

XmmsKdeDB::~XmmsKdeDB() {

  if (connected)
    disconnectDB();

}

bool XmmsKdeDB::connectDB() {
  
  char *error = 0;

  if (enable) {

    if (connected) {
      disconnectDB();
    }
    
    qDebug("xmms-kde: trying to open database");
    db = sqlite_open(name, 0, &error);

    if (error) {
      free(error);
      error = 0;
    }

    qDebug("xmms-kde: creating table");
    sqlite_exec(db, "create table music (filename string, title string, artist string, album string, track string, year string, genre string, comment string)", 0, 0, &error);

    if (error) {
      free(error);
      error = 0;
    }
    
    qDebug("xmms-kde: table created");

    querydb = sqlite_open(name, 0, &error);

    if (error) {
      free(error);
      error = 0;
    }

    if (db && querydb) {
      emit(statusChanged(i18n("connected to database")));
      connected = true;
      return true;
    } else {
      emit(statusChanged(i18n("database connection failed")));
      connected = false;
      return false;
    }
  }
  return false;
}

void XmmsKdeDB::disconnectDB() {
  
  if (enable) {
    if (connected) {
      connected = false;
      sqlite_close(db);
      sqlite_close(querydb);
      db = NULL;
      querydb = NULL;
      //emit(statusChanged("disconnected"));
    }
  }
}


bool XmmsKdeDB::isConnectedDB() {

  return connected;
}

QWidget* XmmsKdeDB::getConfigurationWidget(QWidget *parent) {

  QWidget *dbWidget = new QWidget(parent);  
  
  QVBoxLayout *vbox = new QVBoxLayout(dbWidget, 5);

  QHBoxLayout *box = new QHBoxLayout(vbox, 10);
  enableBox = new QCheckBox(i18n("enable Database support"), dbWidget);
  enableBox->setChecked(enable);

  box->addWidget(enableBox);
  box->addStretch(10);
  /*
  QPushButton *dbUpdateButton = new QPushButton(i18n("Update Database"), dbWidget);
  connect(dbUpdateButton, SIGNAL(clicked()), this, SLOT(updateDatabase()));

  box->addWidget(dbUpdateButton);
  */
  connect(enableBox, SIGNAL(clicked()), this, SLOT(configurationChanged()));

  pathListBox = new QListBox(dbWidget);
  pathListBox->insertStringList(pathList);
  vbox->addWidget(pathListBox);

  box = new QHBoxLayout(vbox, 10);

  QPushButton *addButton = new QPushButton(i18n("Add Path"), dbWidget);
  connect(addButton, SIGNAL(clicked()), this, SLOT(addPathToList()));

  QPushButton *removeButton = new QPushButton(i18n("Remove Path"), dbWidget);
  connect(removeButton, SIGNAL(clicked()), this, SLOT(removePathFromList()));

  box->addWidget(addButton);
  box->addStretch(10);
  box->addWidget(removeButton);

  return dbWidget;
}

void XmmsKdeDB::configurationChanged() {

  enable = enableBox->isChecked();  
}

void XmmsKdeDB::updateDatabase() {

  if (insertThread != NULL) {
    
    if (insertThread->running()) {
      if (statusFrame)
	statusFrame->show();
      return;
    } else {
      delete insertThread;
      insertThread = NULL;
    }
  }

  if (!enable)
    return;

  if (!connected)
    connectDB();

  if (statusFrame) {
    delete statusFrame;
    statusFrame = NULL;
  }
  
  if (pathList.empty())
    return;

  statusFrame = new QFrame(NULL, "Database status", WStyle_Title);
  statusFrame->setCaption(i18n("Database status"));
  QVBoxLayout *box = new QVBoxLayout(statusFrame, 10);
  QHBoxLayout *hbox = new QHBoxLayout(box, 20);
  
  QLabel *pixLabel = new QLabel(statusFrame);
  pixLabel->setPixmap(*icon);
  
  QVBoxLayout *vbox2 = new QVBoxLayout(hbox, 0);
  QVBoxLayout *vbox3 = new QVBoxLayout(hbox, 0);
  
  QLabel *label = new QLabel(i18n("Updating database (this will take a while)."), statusFrame);
  StatusLabel *statusLabel = new StatusLabel("", statusFrame, 45);
  
  StatusLabel *fileLabel = new StatusLabel("", statusFrame, 45);
  
  vbox2->addWidget(pixLabel);
  vbox3->addWidget(label);
  vbox3->addWidget(statusLabel);
  vbox3->addWidget(fileLabel);

  ProgressLabel *progressLabel = new ProgressLabel(statusFrame);
  box->addWidget(progressLabel);
  
  QHBoxLayout *hbox2 = new QHBoxLayout(box, 10);
  
  QPushButton *okButton = new QPushButton(i18n(i18n("OK")), statusFrame);
  okButton->setDefault(true);
  //QPushButton *cancelButton = new QPushButton(i18n("Cancel"), statusFrame);
  
  hbox2->addStretch(10);
  hbox2->addWidget(okButton, 1);
  //hbox2->addWidget(cancelButton, 1);
  
  connect(okButton, SIGNAL(clicked()), statusFrame, SLOT(hide()));
  //connect(cancelButton, SIGNAL(clicked()), this, SLOT(stopInsertThread()));
  //connect(cancelButton, SIGNAL(clicked()), statusFrame, SLOT(hide()));
  
  statusFrame->show();
  
  fileLabel->setMaximumSize(label->size());
  statusLabel->setMaximumSize(label->size());
  fileLabel->setMinimumSize(label->size());
  statusLabel->setMinimumSize(label->size());
  
  insertThread = new InsertThread(db, pathList, statusLabel, fileLabel,
				  progressLabel);
  insertThread->start();
  
  sync = true;
}

void XmmsKdeDB::readConfig() {

  config->setGroup("DB");
  enable = config->readBoolEntry("enable", false);
  QString dbfile(locateLocal("data", "xmms-kde/music.db"));
  name = config->readEntry("name", dbfile);
  qDebug("xmms-kde: using database '" + dbfile + "'");
  pathList = config->readListEntry("paths");
}


void XmmsKdeDB::writeConfig() {

  config->setGroup("DB");
  config->writeEntry("enable", enable);

  config->writeEntry("paths", pathList);  
}


void XmmsKdeDB::addPathToList() {

  QString dir = QFileDialog::getExistingDirectory();
  if (dir != NULL)
    pathListBox->insertItem(dir);

  pathList.clear();
  for (unsigned int index = 0; index < pathListBox->count(); index ++)
    pathList << pathListBox->text(index);


  sync = false;
}

void XmmsKdeDB::removePathFromList() {

  pathListBox->removeItem(pathListBox->currentItem());

  pathList.clear();
  for (unsigned int index = 0; index < pathListBox->count(); index ++)
    pathList << pathListBox->text(index);

  sync = false;
}

void XmmsKdeDB::stopInsertThread() {

 // FIXME
  /*
  if (insertThread) {
    
    if (insertThread->running())
      insertThread->terminate();
  }
  */
}

XmmsKdeDBQuery::XmmsKdeDBQuery(XmmsKdeDB *datab, PlayerInterface *p,
			       QPixmap *icon, KConfig *conf)

  : QVBox(0, "Database Query", WStyle_Title) {

  config = conf;
  readConfig();

  searchThread = NULL;
  db = datab;
  player = p;

  resultBox = new QListBox(this);
  resultBox->setSelectionMode(QListBox::Extended);

  QHBox *hbox = new QHBox(this);
  hbox->setSpacing(0);

  buttons = new QHButtonGroup(this);
  buttons->setExclusive(true);

  QRadioButton *titleButton = new QRadioButton(i18n("Title"), buttons, "title");
  QRadioButton *artistButton = new QRadioButton(i18n("Artist"), buttons, "artist");
  QRadioButton *albumButton = new QRadioButton(i18n("Album"), buttons, "album");
  QRadioButton *genreButton = new QRadioButton(i18n("Genre"), buttons, "genre");

  buttons->setButton(0);

  QHBox *box = new QHBox(this);
  query = new QLineEdit("", box);

  QPushButton *set = new QPushButton(i18n("set"), box);
  QPushButton *add = new QPushButton(i18n("add"), box);

  connect((QObject *) set, SIGNAL(clicked()),
	  this, SLOT(setPlayList()));

  connect((QObject *) add, SIGNAL(clicked()),
	  this, SLOT(addPlayList()));

  connect((QObject *) query, SIGNAL(returnPressed()),
	  (QObject *) this, SLOT(newQuery()));

  connect((QObject *) resultBox, SIGNAL(selected(int)),
	  (QObject *) this, SLOT(play(int)));

  setCaption(i18n("Database Query"));

  KWin::setIcons(winId(), *icon, *icon);
}

XmmsKdeDBQuery::~XmmsKdeDBQuery() {

}

void XmmsKdeDBQuery::setPlayer(PlayerInterface *p) {

  player = p;
}

void XmmsKdeDBQuery::newQuery() {

  if (!db)
    return;

  if (!db->isConnectedDB())
    if (!db->connectDB())
      return;

  if (searchThread) {

    if (searchThread->finished()) {
      delete searchThread;
      searchThread = NULL;
    }
  }

  if (!searchThread) {
    resultBox->clear();
    searchThread = new SearchThread(db->querydb, this, 
				    QString(buttons->selected()->name()),
				    query->text());
    searchThread->start();
  }
}

void XmmsKdeDBQuery::customEvent(QCustomEvent *e) {

  if (e->type() == 60041) {
    ResultEvent *s = (ResultEvent *) e;
    QStringList l = s->message();
    QStringList::Iterator it = l.begin();
    
    QString name(*it);
    it++;
    QString file(*it);

    resultBox->insertItem(new QueryItem(name, file));
  }
}


void XmmsKdeDBQuery::play(int index) {

  if (player) {
    player->playlistClear();
    player->playlistAdd(((QueryItem *) resultBox->item(index))->getFile());
    player->play();
  }
}

void XmmsKdeDBQuery::setPlayList() {

  if (player) {

    player->playlistClear();
    addPlayList();

    if (resultBox->count() > 0)
      player->play();
  }
}


void XmmsKdeDBQuery::addPlayList() {
  
  if (player) {
    QStringList files;
    QStringList allfiles;
    
    for (unsigned int index = 0; index < resultBox->count(); index ++) {
      allfiles += ((QueryItem *) resultBox->item(index))->getFile();

      if (resultBox->isSelected(index))
	files += ((QueryItem *) resultBox->item(index))->getFile();
    }

    if (!files.empty())
      player->playlistAdd(files);//.join(QString("\r\n")));
    else if (!allfiles.empty())
      player->playlistAdd(allfiles);//.join(QString("\r\n")));
  }

  resultBox->clearSelection();
}


void XmmsKdeDBQuery::popup() {

  if (db->isEnabled()) {
    int d = KWin::info(winId()).desktop;
    
    if (d != KWin::currentDesktop()) {
      
      KWin::setOnDesktop(winId(), KWin::currentDesktop());
    }
    
    if (!isVisible())
      show();
    
    query->setFocus();
  }
}

void XmmsKdeDBQuery::readConfig() {

  config->setGroup("DB");
  framePos = QPoint(0, 0);
  framePos = config->readPointEntry("queryframeposition", &framePos);
  frameSize = QSize(200, 320);
  frameSize = config->readSizeEntry("queryframesize", &frameSize);
  pop = config->readNumEntry("popup", 2);

  resize(frameSize);
  move(framePos);
}


void XmmsKdeDBQuery::writeConfig() {

  config->setGroup("DB");
  config->writeEntry("queryframeposition", pos());
  config->writeEntry("queryframesize", size());
  config->writeEntry("popup", pop);
}

QWidget* XmmsKdeDBQuery::getConfigurationWidget(QWidget *parent) {

  QWidget *dbWidget = new QWidget(parent);  
  
  QVBoxLayout *vbox = new QVBoxLayout(dbWidget, 10);

  queryGroup = new QVButtonGroup(i18n("Show query window"), dbWidget);
  connect(queryGroup, SIGNAL(clicked(int)), this, SLOT(popupChanged(int)));

  QRadioButton *kdeStartup = new QRadioButton(i18n("at KDE startup"), queryGroup);
  QRadioButton *playerStartup = new QRadioButton(i18n("at player startup"), queryGroup);
  QRadioButton *noStartup = new QRadioButton(i18n("when selected from menu"), queryGroup);
  queryGroup->setButton(pop);
  vbox->addWidget(queryGroup);

  return dbWidget;
}

void XmmsKdeDBQuery::popupChanged(int index) {

  pop = index;
}


QueryItem::QueryItem(QString text, QString file):

  QListBoxText(text) {
  filename = file;
}

QueryItem::~QueryItem() {

}

QString QueryItem::getFile() {

  return filename;
}

SearchThread::SearchThread(sqlite *datab, XmmsKdeDBQuery *q,
			   QString w, QString l) {

  db = datab;
  query = q;
  what = w;
  like = l;
}

void SearchThread::run() {

  char *error;
  char **result;
  int nrow;
  int ncolumn;
  
  QString text = "%%" + like + "%%";

  int rc = sqlite_get_table_printf(db, "SELECT artist, title, filename FROM music WHERE %q LIKE %Q", &result, &nrow, &ncolumn, &error,
				   what.latin1(), text.latin1());
  
  if (error) {
    qDebug("error: %s\n", error);
    free(error);
    error = 0;
  }

  if (rc != SQLITE_OK) {
    qDebug("xmms-kde: database query failed");
  } else {
    if (nrow >= 1) {
      for (int index = 1; index <= nrow; index ++) {
	// only add it when the file exists
	QString fileName(result[index * 3 + 2]);

	if (QFile::exists(fileName)) {
	  QString name = "[" + QString(result[index * 3]) + "] " +
	    QString(result[index * 3 + 1]);

	  QStringList l;
	  l += name;
	  l += fileName;

	  ResultEvent *event = new ResultEvent(l);
	  postEvent(query, event);
	}
      }
    }        
  }
  
  sqlite_free_table(result);
}


InsertThread::InsertThread(sqlite *datab, QStringList dirs,
			   StatusLabel *status, StatusLabel *file,
			   ProgressLabel *progess) {
  db = datab;
  dir = dirs;

  statusLabel = status;
  fileLabel = file;
  progressLabel = progess;
}

void InsertThread::run() {

  updateDatabase(dir);
}

void InsertThread::updateDatabase(QStringList dir) {

  StatusEvent *event = new StatusEvent(i18n("reading files from database..."));
  postEvent(statusLabel, event);


  ProgressTotalEvent *ptevent = new ProgressTotalEvent(1000);
  postEvent(progressLabel, ptevent);

  ProgressStepEvent *psevent = new ProgressStepEvent(1);
  postEvent(progressLabel, psevent);

  QTime t, r;
  t.start();
  r.start();

  // the contents of the database in a hash  
  hash_set<const char *, hash<const char *>, eqstr> database;

  // the contents of the filesystem in a hash  
  hash_set<const char *, hash<const char *>, eqstr> files;

  hash_set<const char*, hash<const char*>, eqstr>::const_iterator it, it2;

  QString q("SELECT filename from music");
  
  int nrow, ncolumn;
  char *error;
  char **result;
  
  int rc = sqlite_get_table(db, q.latin1(), &result, &nrow, &ncolumn, &error);
  
  if (error) {
    free(error);
    error = 0;
  }

  printf("xmms-kde: read files from database: %d seconds\n", t.elapsed() /1000);
  t.start();

  database.resize(nrow * 2);
    
  for (int index = 0; index < nrow; index ++) {
    database.insert(result[index]);    
  }

  printf("xmms-kde: inserted database into hash: %d seconds\n", t.elapsed() /1000);
  t.start();

  event = new StatusEvent(i18n("reading files from filesystem..."));
  postEvent(statusLabel, event);

  vector<QString *> v;
  
  for (QStringList::Iterator it = dir.begin(); it != dir.end(); ++it )
    addPathToVector(*it, &v);

  printf("xmms-kde: read files from filesystem: %d seconds\n", t.elapsed() /1000);
  t.start();

  files.resize(v.size() * 2);

  for (unsigned int index = 0; index < v.size(); index ++)
    files.insert(v[index]->latin1());

  printf("xmms-kde: inserted files into hash: %d seconds\n", t.elapsed() /1000);
  t.start();

  ptevent = new ProgressTotalEvent(v.size() + nrow + 1);
  postEvent(progressLabel, ptevent);

  event = new StatusEvent(i18n("removing old entries..."));
  postEvent(statusLabel, event);

  float thresh = 0.0;
  float size = v.size() + nrow; // number of files in filesystem + in database
  if (size == 0) size = 1;

  QString query("BEGIN TRANSACTION remove");
  rc = sqlite_exec(db, query.latin1(), NULL, 0, &error);

  if (error) {
    free(error);
    error = 0;
  }  

  int progressIndex = 0;

  it = database.begin();

  while (it != database.end()) {

    if (((float) progressIndex) / size - thresh >= 0.01) {
      thresh = ((float) progressIndex) / size;
      ProgressStepEvent *psevent = new ProgressStepEvent(progressIndex);
      postEvent(progressLabel, psevent);
    }
    
    progressIndex ++;
    
    it2 = files.find(*it);
    if (it2 == files.end()) {
      deleteFromDatabase(QString(*it));
    }
    it++;
  }

  QString query2("COMMIT TRANSACTION remove");
  rc = sqlite_exec(db, query2.latin1(), NULL, 0, &error);

  if (error) {
    free(error);
    error = 0;
  }
  
  printf("xmms-kde: removed files from database: %d seconds\n", t.elapsed() / 1000);
  t.start();

  event = new StatusEvent(i18n("adding new files..."));
  postEvent(statusLabel, event);

  QString query3("BEGIN TRANSACTION insert ON CONFLICT REPLACE");
  rc = sqlite_exec(db, query3.latin1(), NULL, 0, &error);

  if (error) {
    free(error);
    error = 0;
  }

  for (unsigned int index = 0; index < v.size(); index ++) {

    // only generate an event when progressbar should move (>=1% difference)
    if (((float) progressIndex) / size - thresh >= 0.01) {
      thresh = ((float) progressIndex) / size;
      
      ProgressStepEvent *psevent = new ProgressStepEvent(progressIndex);
      postEvent(progressLabel, psevent);
    }

    progressIndex++;

    it = database.find(v[index]->latin1());
    if (it == database.end()) {
      insertIntoDatabase(*v[index]);
    }
  }  

  psevent = new ProgressStepEvent(v.size() + nrow + 1);
  postEvent(progressLabel, psevent);

  QString query4("COMMIT TRANSACTION insert");
  rc = sqlite_exec(db, query4.latin1(), NULL, 0, &error);

  if (error) {
    free(error);
    error = 0;
  }

  printf("xmms-kde: inserted into database: %d seconds\n", t.elapsed() / 1000);

  event = new StatusEvent(i18n("updating database: done."));
  postEvent(statusLabel, event);
  event = new StatusEvent("");
  postEvent(fileLabel, event);

  sqlite_free_table(result);

  for (unsigned int index = 0; index < v.size(); index ++)
    delete v[index];

  printf("xmms-kde: updated database: %d seconds\n", r.elapsed() / 1000);
}

void InsertThread::insertIntoDatabase(QString file) {

  QFileInfo inf(file);
    
  if (inf.isFile()) {

    KFileMetaInfo m(file);
	
    if (m.isValid() && (!m.mimeType().endsWith("x-mp3") ||
			!m.mimeType().endsWith("x-ogg"))) {

      QFileInfo f(file);
      StatusEvent *event = new StatusEvent("[" + m.item("Artist").string() + 
					   "] " + m.item("Title").string());

      postEvent(fileLabel, event);
      
      QString title = m.item("Title").string();
      QString artist = m.item("Artist").string();
      QString album = m.item("Album").string();
      QString track = m.item("Tracknumber").string();
      QString date = m.item("Date").string();
      QString genre = m.item("Genre").string();
      QString comment = m.item("Comment").string();

      char *error;
      
      int rc = sqlite_exec_printf(db, 
				  "INSERT INTO music VALUES(%Q, %Q, %Q, %Q, %Q, %Q, %Q, %Q)", 0, 0, &error,
				  file.latin1(),
				  title.latin1(), artist.latin1(), 
				  album.latin1(), track.latin1(), 
				  date.latin1(), genre.latin1(), 
				  comment.latin1());

      if (error) {
	free(error);
	error = 0;
      }

      if (rc != SQLITE_OK)
	qDebug("xmms-kde: database insert failed on [%s]\n", file.latin1());
      /*
      else
	printf("xmms-kde: inserted [%s]\n", file.latin1());
      */
    } else if (file.lower().endsWith(".mp3") ||
	       file.lower().endsWith(".ogg")) {
      /* seems to be an ogg or mp3, but doesn't have any meta info 
	 This could happen if they don't have id3lib installed, or
	 if there's no id3 tag.   In this case, try to get the
	 title from the filename. */
      QString fileName = inf.baseName();
      fileName = fileName.replace(QRegExp("_"), " ");
      fileName = fileName.replace(QRegExp("-"), " ");
      fileName = fileName.stripWhiteSpace();
      
      StatusEvent *event = new StatusEvent(fileName);
      postEvent(fileLabel, event);
      
      char *error;
      
      int rc = sqlite_exec_printf(db, 
				  "INSERT INTO music VALUES(%Q, %Q, %Q, %Q, %Q, %Q, %Q, %Q)", 0, 0, &error,
				  file.latin1(), fileName.latin1(), "", "", "", "", "", "");

      if (error) {
	free(error);
	error = 0;
      }     

      if (rc != SQLITE_OK)
 	qDebug("xmms-kde: database insert failed on [%s]\n", file.latin1());
    }
  }
}


void InsertThread::deleteFromDatabase(QString file) {

  char *error;
      
  int rc = sqlite_exec_printf(db, "DELETE FROM music WHERE filename = %Q", NULL, 0, &error, file.latin1());
      
  if (error) {
    free(error);
    error = 0;
  }

  if (rc != SQLITE_OK)
    qDebug("xmms-kde: database delete failed on [%s]\n", file.latin1());
  /*
  else
    printf("xmms-kde: deleted [%s]\n", file.latin1());
  */
}

void InsertThread::addPathToVector(QString path, vector<QString *> *v) {

  QDir dir(path);
  if (!dir.exists())
    return;
      
  QStringList fileList(dir.entryList());
  
  for (QStringList::Iterator it = fileList.begin(); it != fileList.end(); ++it ) {
    QString absfile(dir.absPath() + "/" + (*it));
    QString file(*it);
    
    QFileInfo inf(absfile);
    
    if (inf.isDir()) {
      if (!file.startsWith(".")) {
	addPathToVector(absfile, v);
      }
    } else if (inf.isFile()
	       && ((file.endsWith(".mp3") || file.endsWith(".ogg") ||
		    file.endsWith(".MP3") || file.endsWith(".OGG")))) {
      
      v->push_back(new QString(absfile));
    }
  }
}


