#include "runtime.h"
#include "cfg.h"
#include "widgets.h"
#include "lcp3.h"

#include <stdio.h>
#include <math.h>

GSList * lines;
struct truntime * runtime;

const float kbaud_rates [] = {
	9.6, 14.4, 28.8, 36.6, 56., 64., 128.,
	256., 512., 1024., 10000., 100000, 0. };

gboolean runtime_init()
{
	if ( !(runtime = g_malloc(sizeof(struct truntime))) )
		return FALSE;
	lines = NULL;
	runtime->autohup_file = NULL;
	runtime->selected_line = NULL;
	runtime->terminate = FALSE;
	runtime->client_status = CLT_DISCON;
	return TRUE;
}

#define READ_NUM(b)		fscanf(runtime->autohup_file, "%d\n", &ahup->b);
#define WRITE_NUM(b)	fprintf(runtime->autohup_file, "%d\n", ((struct tline*)line->data)->ahup->b);

float get_kbaud(int bytes_per_sec)
{
	float best_kbaud = 0, best_diff = 1e8, kbit_per_sec = ((float)bytes_per_sec) * 8.0 / 1000.0;
	const float * cur = kbaud_rates;
	while ( *cur > 0.0 )
	{
		if ( fabsf(*cur - kbit_per_sec) < best_diff )
		{
			best_diff = fabsf(*cur - kbit_per_sec);
			best_kbaud = *cur;
		}
		cur++;
	}
	return(best_kbaud);
}

void runtime_save()
{
	GSList * line = lines;
	config_ahup_open_write();
	if ( !runtime->autohup_file )
	{
		fprintf(stderr, "config_ahup_open_write() failed. Can't write autohangup information.\n");
		return;
	}
	while ( line )
	{
		// write autohangup data
		WRITE_NUM(activated);
		WRITE_NUM(logic_or);
		WRITE_NUM(activity_mode);
		WRITE_NUM(btime);
		WRITE_NUM(time);
		WRITE_NUM(bup);
		WRITE_NUM(up);
		WRITE_NUM(bdown);
		WRITE_NUM(down);
		WRITE_NUM(btotal_up);
		WRITE_NUM(total_up);
		WRITE_NUM(btotal_down);
		WRITE_NUM(total_down);
		WRITE_NUM(cap_up);
		WRITE_NUM(cap_down);
		WRITE_NUM(max_up);
		WRITE_NUM(max_down);
		// next element
		line = g_slist_next(line);
	}
	fclose(runtime->autohup_file);
	runtime->autohup_file = NULL;
}

void autohup_load_default(struct tautohup * ahup)
{
	ahup->activated = FALSE;
	ahup->logic_or = TRUE;
	ahup->activity_mode = FALSE;
	ahup->btime = TRUE;
	ahup->time = 5;
	ahup->bup = FALSE;
	ahup->up = 100;
	ahup->bdown = TRUE;
	ahup->down = 1024;
	ahup->btotal_up = FALSE;
	ahup->total_up = 2;
	ahup->btotal_down = FALSE;
	ahup->total_down = 20;
	ahup->cap_up = 0;
	ahup->cap_down = 0;
	ahup->max_up = 0;
	ahup->max_down = 0;
}

void runtime_read_autohup(struct tautohup * ahup)
{
	if ( !runtime->autohup_file )
		if ( !config_ahup_open_read() )
		{
			fprintf(stderr, "config_ahup_open_read() failed. Can't load autohangup information.\n");
			return;
		}
	autohup_load_default(ahup);
	READ_NUM(activated);
	READ_NUM(logic_or);
	READ_NUM(activity_mode);
	READ_NUM(btime);
	READ_NUM(time);
	READ_NUM(bup);
	READ_NUM(up);
	READ_NUM(bdown);
	READ_NUM(down);
	READ_NUM(btotal_up);
	READ_NUM(total_up);
	READ_NUM(btotal_down);
	READ_NUM(total_down);
	READ_NUM(cap_up);
	READ_NUM(cap_down);
	READ_NUM(max_up);
	READ_NUM(max_down);
	if ( feof(runtime->autohup_file) )
	{
		#ifdef DEBUG
			fprintf(stderr, "runtime_read_autohup(): eof, closing file\n");
		#endif
		fclose(runtime->autohup_file);
		runtime->autohup_file = NULL;
	}
}

struct tline* runtime_get_line(int num)
{
	GSList * cur = lines;
	if ( !num ) num = config->default_line;
	while ( cur )
	{
		if ( ((struct tline*)cur->data)->num == num ) break;
		cur = g_slist_next(cur);
	}
	if ( cur ) return (struct tline*)cur->data;
	return NULL;
}

int runtime_get_clid_lineid(int id)
{
	GSList * cur = lines;
	int i = 1;
	if ( !id ) return 0;
	while ( cur )
	{
		if ( ((struct tline*)cur->data)->num == id ) return i;
		i++;
		cur = g_slist_next(cur);
	}
	return 0;
}

int runtime_get_clid_line(struct tline* line)
{
	GSList * cur = lines;
	int i = 1;
	if ( !line ) return 0;
	while ( cur )
	{
		if ( ((struct tline*)cur->data) == line ) return i;
		i++;
		cur = g_slist_next(cur);
	}
	return 0;
}

struct tline* runtime_get_line_clid(int num)
{
	GSList * cur = lines;
	int i = 1;
	if ( !num ) return runtime_get_line(0);

	while ( cur )
	{
		if ( i == num ) break;
		i++;
		cur = g_slist_next(cur);
	}
	if ( cur ) return (struct tline*)cur->data;
	return NULL;
}

struct tline* runtime_get_line_by_name(const char * name)
{
	GSList * cur = lines;
	if ( !name ) return NULL;
	while ( cur )
	{
		if ( !strcmp(name, ((struct tline*)cur->data)->name) )
			return (struct tline*)cur->data;
		cur = g_slist_next(cur);
	}
	return NULL ;
}

void runtime_load_line_strings(struct tline* line)
{
	if ( !line ) return;
	snprintf(line->snum, 4, "%d", line->num);
	snprintf(line->schan, 3, "%d", line->channels);
	strncpy(line->sstime, make_time(line->server_time), 20);
	strncpy(line->sctime, make_time(line->client_time), 20);
	snprintf(line->sups, 10, "%d", line->up_short);
	snprintf(line->supl, 10, "%d", line->up_long);
	snprintf(line->supt, 20, "%d", line->up_total);
	snprintf(line->sdns, 10, "%d", line->down_short);
	snprintf(line->sdnl, 10, "%d", line->down_long);
	snprintf(line->sdnt, 20, "%d", line->down_total);
/* we don't have to guess this anymore... it's coming from the server now
	if ( line->up_short > line->ahup->max_up )
	{
		line->ahup->max_up = line->up_short;
		line->ahup->cap_up = (int)(1000.0*get_kbaud(line->ahup->max_up));
	}
	if ( line->down_short > line->ahup->max_down )
	{
		line->ahup->max_down = line->down_short;
		line->ahup->cap_down = (int)(1000.0*get_kbaud(line->ahup->max_down));
	}
*/
	snprintf(line->scap_up, 10, "%.1f", ((float)line->ahup->cap_up)/1000.0);
	snprintf(line->scap_dn, 10, "%.1f", ((float)line->ahup->cap_down)/1000.0);
	snprintf(line->smax_up, 10, "%d", line->ahup->max_up);
	snprintf(line->smax_dn, 10, "%d", line->ahup->max_down);
}

void runtime_clear_lines()
{
	GSList * cur = lines;
	while ( cur )
	{
		g_free(cur->data);
		cur = g_slist_next(cur);
	}
	g_slist_free(lines);
	lines = NULL;
	runtime->selected_line = NULL;
}
