/* $Id: compile.c,v 1.6 1999/07/30 16:48:33 stano Exp $

   Compilation of the keymaps

  (C) 1999 Stanislav Meduna <stano@eunet.sk>
*/

#include <compile.h>
#include <configuration.h>
#include <utils.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>
#include <dirent.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <malloc.h>

#include <libintl.h>

#define _(s) dgettext(PACKAGE, s)

/* === Forward declarations === */

/* Unconditionally compile a map to given file */
static int do_compile(const db_record_t *map, const char *fn);

/* Merge sys and user sources into one tree to work around the xkbcomp bug */
static int merge_trees(const char *sys, const char *usr, const char *target);

/* Make a list of all absolute filenames in the directory */
static int make_file_list(const char *dirn, char ***files, int *nfiles, int *afiles);

/* Remove the merge tree */
static int remove_merged_tree(const char *dirn);

/* Recursively remove all compiled keymaps in and below the given directory */
static int remove_xkm_tree(const char *dirn, int level);

/* === Public interface === */

/* Compile a map according to the map name.
   The databases must be open (calls read_db internally)
*/

int compile_map(const char *name)
{
	int r;
	int found, denied;
	db_record_t rec;
	char *dst, *dstu, *dsts;
	int user;
	mode_t old_umask;

	r = read_db_name(name, &rec, &found);
	if (r < 0)
		return r;

	if (! found)
	{
		fprintf(stderr, _("map %s not found in the database\n"), name);
		return -1;
	}

	r = get_compiled_map_names(&rec, &dstu, &dsts);
	if (r < 0)
		return r;

	/* Decide about the destination */
	user = (rec.keymap_src >= USER);

	/* If the map is a system one and we don't have access
	   to system directory (probably because we are running
	   as user), create the map in user one.
	*/
	dst = user ? dstu : dsts;

	/* Set sane umask */
	old_umask = umask(022);

	r = make_path_for(dst, &denied);
	if (r < 0)
	{
		umask(old_umask);
		return r;
	}

	if (! user && denied)
	{
		dst = dstu;

		r = make_path_for(dst, &denied);
		if (r < 0)
		{
			umask(old_umask);
			return r;
		}
	}

	if (denied)
	{
		fprintf(stderr, _("cannot prepare path for %s - access denied\n"), dst);
		umask(old_umask);
		return -1;
	}

	if (flag_verbose)
		fprintf(stderr, _("compiling map %s into %s\n"), name, dst);

	r = do_compile(&rec, dst);

	/* Reset the umask */
	umask(old_umask);

	if (! r)
		return r;

	if(dstu != NULL)
		free(dstu);
	if(dsts != NULL)
		free(dsts);

	return 0;
}

/* Conditionally compile map, if not present
*/
int compile_map_if_needed(const char *name, char **file_name)
{
	int r;
	int found;
	char *srcu, *srcs;
	db_record_t rec;

	*file_name = 0;

	r = read_db_name(name, &rec, &found);
	if (r < 0)
		return r;

	if (! found)
	{
		fprintf(stderr, _("map %s not found in the database\n"), name);
		return -1;
	}

	r = get_compiled_map_names(&rec, &srcu, &srcs);
	if (r < 0)
		return r;

	/* If the keymap was user-specified, require
	   compiled map at user. Otherwise system one suffices.
	*/

	if (access(srcu, R_OK) < 0 &&
	    (rec.keymap_src >= USER || access(srcs, R_OK) < 0))
	{
		r = compile_map(name);
		if (r < 0)
			return r;
	}

	/* Always prefer the user one */
	if (access(srcu, R_OK) == 0)
	{
		*file_name = srcu;
		if (srcs != NULL)
			free(srcs);
	}
	else if (access(srcs, R_OK) == 0)
	{
		*file_name = srcs;
		if (srcu != NULL)
			free(srcu);
	}
	else
	{
		fprintf(stderr, _("compiled map %s not found in the %s nor %s\n"), name, srcu, srcs);

		if (srcu != NULL)
			free(srcu);
		if (srcs != NULL)
			free(srcs);

		return -1;
	}

	return 0;
}


/* Install a map (eventually compile first)
   The databases must be open (calls read_db internally)
*/
int install_map(const char *name)
{
	int r;
	int found;
	char *src;
	char command[PATH_MAX + 512];
	const char *mn;
	sel_info_t info;
	FILE *fp;

	/* If shortcut, try to translate */
	if (strchr(name, '(') && strchr(name, ')'))
		mn = name;
	else
	{
		int idx;

		r = get_cfg_map(name, &info, &idx, &found);
		if (r < 0)
			return r;

		mn = found ? info.map_name : name;
	}


	r = compile_map_if_needed(mn, &src);
	if (r < 0)
		return r;

	sprintf(command, "%s -o \'%s\' \'%s\'",
		XKBCOMP,
		display_name,
		src
		);

	free(src);

	if (flag_debug)
		fprintf(stderr, _("Running %s\n"), command);

	fp = popen(command, "w");

	r = pclose(fp);

	if (! WIFEXITED(r) || WEXITSTATUS(r) != 0)
	{
		fprintf(stderr, _("%s exited abnormally: %d\n"), XKBCOMP, r);
		return -1;
	}

	return 0;
}


/* Remove all compiled maps */
int remove_compiled_maps(int user)
{
	int r;

	r = remove_xkm_tree(user ? user_xkm_dir : sys_xkm_dir, 0);

	return r;
}



/* === Private functions === */


/* Unconditionally compile a map to given file */
static int do_compile(const db_record_t *map, const char *fn)
{
	int user;
	int merge = 0;
	int r;
	char *tmp = NULL;
	char command[3*PATH_MAX + 512];
	char *mp;
	FILE *fp;

	/* Is anything from user involved? */
	user = (map->keymap_src >= USER);

	/* xkbcomp requires that the named map exists */
	mp = 0;

	if (user)
	{
		char map_path[PATH_MAX+1];

		sprintf(map_path, "%s/%s", user_keymap_dir, map->keymap);
		if (! access(map_path, R_OK))
			mp = user_keymap_dir;
	}

	if (! mp)
	{
		char map_path[PATH_MAX+1];

		sprintf(map_path, "%s/%s", sys_keymap_dir, map->keymap);
		if (! access(map_path, R_OK))
			mp = sys_keymap_dir;
	}

	if (! mp)
	{
		char map_path[PATH_MAX+1];

		sprintf(map_path, "%s/%s", x11_keymap_dir, map->keymap);
		if (! access(map_path, R_OK))
			mp = x11_keymap_dir;
	}

	if (! mp)
	{
		fprintf(stderr, _("Cannot find map file %s\n"), map->keymap);
		return -1;
	}



	/* Do we need to merge? */
	if (user)
	{
		if (xkbcomp_has_bug < 0)
			check_xkbcomp_bug();

		merge = (xkbcomp_has_bug != 0);
	}

	if (merge)
	{
		tmp = tmpnam(NULL);

		if (tmp == NULL)
		{
			fprintf(stderr, _("cannot generate temporary name: %s\n"), strerror(errno));
			return -1;
		}

		if (flag_debug)
			fprintf(stderr, _("merging sys and user trees into: %s\n"), tmp);

		r = merge_trees(sys_dir, user_dir, tmp);
		if (r < 0)
			return r;
	}


	/* Construct the command */
	if (user && ! merge)
		sprintf(command, "%s -xkm -R\'%s\' -I\'%s\' -o \'%s\' \'%s/%s\'",
			XKBCOMP,
			user_dir, sys_dir,
			fn,
			mp,
			map->map_name
			);
	else
		sprintf(command, "%s -xkm -R\'%s\' -o \'%s\' \'%s/%s\'",
			XKBCOMP,
			user ? tmp : sys_dir,
			fn,
			mp,
			map->map_name
			);

	if (flag_debug)
		fprintf(stderr, _("Running %s\n"), command);

	fp = popen(command, "w");

	r = pclose(fp);

	if (merge)
		remove_merged_tree(tmp);

	if (! WIFEXITED(r) || WEXITSTATUS(r) != 0)
	{
		fprintf(stderr, _("%s exited abnormally: %d\n"), XKBCOMP, r);
		return -1;
	}

	return 0;
}


/* Merge sys and user sources into one tree to work around the xkbcomp bug */
static int merge_trees(const char *sys, const char *usr, const char *target)
{
	/* If it is only in one directory, link it.
           If it is (links to) a file, choose the user one.
	   If it is a directory, recurse it down. 
	   If it is dir here and file there, bark and choose the user's view.
	*/

	DIR *dir;

	struct dirent *de;
	struct stat st;

	int lenu = strlen(usr), lens = strlen(sys), lent = strlen(target);
	int maxl = lenu;
	const char *maxp = usr;

	/* Create the target */

	if (stat(target, &st) >= 0 || errno != ENOENT)
	{
		fprintf(stderr, _("Target %s exists or its parent does not exist\n"), target);
		return -1;
	}

	if (flag_debug)
		fprintf(stderr, _("creating directory %s\n"), target);

	if (mkdir(target, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH) < 0)
	{
		fprintf(stderr, _("cannot create directory %s: %s\n"), target, strerror(errno));
		return -1;
	}

	if (lens > maxl)
	{
		maxl = lens;
		maxp = sys;
	}
	if (lent > maxl)
	{
		maxl = lent;
		maxp = target;
	}

	dir = opendir(usr);

	if (dir == NULL)
	{
		fprintf(stderr, _("cannot open directory %s: %s\n"), usr, strerror(errno));
		return -1;
	}

	for (de = readdir(dir); de != NULL; de = readdir(dir))
	{
		struct stat stu, sts;
		char fullu[PATH_MAX+1], fulls[PATH_MAX+1], fullt[PATH_MAX+1];
		int ins = 0;

		if (! strcmp(de->d_name, ".") || ! strcmp(de->d_name, ".."))
			continue;

		if (strlen(de->d_name) + maxl + 2 > PATH_MAX)
		{
			fprintf(stderr, _("Total path name %s/%s too long - ignored\n"), maxp, de->d_name);
			continue;
		}

		sprintf(fullu, "%s/%s", usr, de->d_name);
		sprintf(fulls, "%s/%s", sys, de->d_name);
		sprintf(fullt, "%s/%s", target, de->d_name);

		if (stat(fullu, &stu) < 0)
		{
			fprintf(stderr, _("cannot stat %s: %s\n"), fullt, strerror(errno));
			continue;
		}

		if (stat(fulls, &sts) < 0)
		{
			if (errno != ENOENT)
			{
				fprintf(stderr, _("cannot stat %s: %s\n"), fulls, strerror(errno));
				continue;
			}
			ins = 0;
		}
		else
			ins = 1;
		
		if (! ins)
		{
			if (flag_debug)
				fprintf(stderr, _("symlinking %s as %s\n"), fullu, fullt);

			/* Only in user one - link and continue */
			if (symlink(fullu, fullt) < 0)
			{
				fprintf(stderr, _("cannot link %s as %s: %s\n"), fullu, fullt, strerror(errno));
				closedir(dir);
				return -1;
			}

			continue;
		}

		if (S_ISDIR(stu.st_mode) && S_ISDIR(sts.st_mode))
		{
			/* If both are directories, recurse */
			int r = merge_trees(fulls, fullu, fullt);

			if (r < 0)
				return r;

			continue;
		}
		else
		{
			if (S_ISREG(stu.st_mode) != S_ISREG(sts.st_mode))
				fprintf(stderr, _("warning: %s and %s not both files or both directories\n"), fulls, fullu);

			if (flag_debug)
				fprintf(stderr, _("symlinking %s as %s\n"), fullu, fullt);

			/* Choose user one */
			if (symlink(fullu, fullt) < 0)
			{
				fprintf(stderr, _("cannot link %s as %s: %s\n"), fullu, fullt, strerror(errno));
				closedir(dir);
				return -1;
			}

			continue;
		}
	}

	closedir(dir);

	/* Repeat the loop, this time starting in sys
	   and picking things only in sys
	*/

	dir = opendir(sys);

	if (dir == NULL)
	{
		fprintf(stderr, _("cannot open directory %s: %s\n"), sys, strerror(errno));
		return -1;
	}

	for (de = readdir(dir); de != NULL; de = readdir(dir))
	{
		struct stat stu, sts;
		char fullu[PATH_MAX+1], fulls[PATH_MAX+1], fullt[PATH_MAX+1];

		if (! strcmp(de->d_name, ".") || ! strcmp(de->d_name, ".."))
			continue;

		if (strlen(de->d_name) + maxl + 2 > PATH_MAX)
		{
			fprintf(stderr, _("Total path name %s/%s too long - ignored\n"), maxp, de->d_name);
			continue;
		}

		sprintf(fullu, "%s/%s", usr, de->d_name);
		sprintf(fulls, "%s/%s", sys, de->d_name);
		sprintf(fullt, "%s/%s", target, de->d_name);

		if (stat(fulls, &stu) < 0)
		{
			fprintf(stderr, _("cannot stat %s: %s\n"), fullt, strerror(errno));
			continue;
		}

		if (stat(fullu, &sts) < 0)
		{
			if (errno != ENOENT)
			{
				fprintf(stderr, _("cannot stat %s: %s\n"), fulls, strerror(errno));
				continue;
			}

			if (flag_debug)
				fprintf(stderr, _("symlinking %s as %s\n"), fulls, fullt);
			if (symlink(fulls, fullt) < 0)
			{
				fprintf(stderr, _("cannot link %s as %s: %s\n"), fulls, fullt, strerror(errno));
				closedir(dir);
				return -1;
			}

			continue;
		}
	}

	closedir(dir);

	return 1;
}

/* Remove the merge tree */
static int remove_merged_tree(const char *dirn)
{
	/* First construct a list of symlinks and directories
	   (the readdir is not happy when we simultaneously
	   mess with the directory). Then remove the links,
	   recurse to directories and finally unlink ourselves.

           There is a memleak here when something fails -
	   we know of it and ignore it :-)
	*/

	char **names=0;
	int nnames=0, allocnames=0;
	int i;
	int err;

	err = make_file_list(dirn, &names, &nnames, &allocnames);
	if (err < 0)
		return err;

	for (i=0; i < nnames; i++)
	{
		struct stat st;

		if (lstat(names[i], &st) < 0)
		{
			fprintf(stderr, _("cannot lstat %s: %s\n"), names[i], strerror(errno));
			return -1;
		}

		if (S_ISLNK(st.st_mode))
		{
			if (flag_debug)
				fprintf(stderr, _("Removing symbolic link %s\n"), names[i]);
			if (unlink(names[i]) < 0)
			{
				fprintf(stderr, _("cannot unlink %s: %s\n"), names[i], strerror(errno));
				return -1;
			}
		}
		else if (S_ISDIR(st.st_mode))
		{
			int r = remove_merged_tree(names[i]);

			if (r < 0)
				return r;

		}
		else
		{
			fprintf(stderr, _("%s not a directory nor symlink\n"), names[i]);
			return -1;
		}
	}

	/* Unlink self */
	if (flag_debug)
		fprintf(stderr, _("Removing directory %s\n"), dirn);
	if (rmdir(dirn) < 0)
	{
		fprintf(stderr, _("cannot unlink directory %s: %s\n"), dirn, strerror(errno));
		return -1;
	}

	/* cleanup */
	list_free(&names, &nnames, &allocnames);

	return 0;
}

/* Recursively remove all compiled keymaps in and below the given directory */
static int remove_xkm_tree(const char *dirn, int level)
{
	/* First construct a list of xkm files
	   (the readdir is not happy when we simultaneously
	   mess with the directory). Then remove the files,
	   recurse to directories and finally unlink ourselves.
	*/

	char **names=0;
	int nnames=0, allocnames=0;
	int i;
	int err;

	err = make_file_list(dirn, &names, &nnames, &allocnames);
	if (err < 0)
		return err;

	for (i=0; i < nnames; i++)
	{
		struct stat st;

		if (stat(names[i], &st) < 0)
		{
			fprintf(stderr, _("cannot stat %s: %s\n"), names[i], strerror(errno));
			return -1;
		}

		if (S_ISREG(st.st_mode))
		{
			int n;

			n = strlen(names[i]);

			if (n >= 4 && ! strcmp(names[i]+n-4, ".xkm"))
			{
				if (flag_debug)
					fprintf(stderr, _("Removing xkm file %s\n"), names[i]);
				if (unlink(names[i]) < 0)
				{
					fprintf(stderr, _("cannot unlink %s: %s\n"), names[i], strerror(errno));
					return -1;
				}
			}
			else
				fprintf(stderr, _("Not removing file %s - not a .xkm file\n"), names[i]);
		}
		else if (S_ISDIR(st.st_mode))
		{
			int r = remove_xkm_tree(names[i], level+1);

			if (r < 0)
				return r;

		}
		else
		{
			fprintf(stderr, _("%s not a directory nor file\n"), names[i]);
			return -1;
		}
	}

	/* Unlink self, if not top directory */
	if (level != 0)
	{
		if (flag_debug)
			fprintf(stderr, _("Removing directory %s\n"), dirn);
		if (rmdir(dirn) < 0)
		{
			fprintf(stderr, _("cannot unlink directory %s: %s\n"), dirn, strerror(errno));
			if (errno != ENOTEMPTY)
				return -1;
		}
	}

	/* cleanup */
	list_free(&names, &nnames, &allocnames);

	return 0;
}

/* Make a list of all absolute filenames in the directory */
static int make_file_list(const char *dirn, char ***files, int *nfiles, int *afiles)
{
	DIR *dir;
	int err;

	struct dirent *de;

	dir = opendir(dirn);

	if (dir == NULL)
	{
		fprintf(stderr, _("cannot open directory %s: %s\n"), dirn, strerror(errno));
		return -1;
	}

	list_start(files, nfiles, afiles);

	for (de = readdir(dir); de != NULL; de = readdir(dir))
	{
		char *s;

		if (! strcmp(de->d_name, ".") || ! strcmp(de->d_name, ".."))
			continue;


		s = (char *) malloc(strlen(dirn)+strlen(de->d_name)+2);
		if (s == NULL)
		{
			fprintf(stderr, _("Out of memory\n"));
			return -1;
		}

		sprintf(s, "%s/%s", dirn, de->d_name);

		err = list_append(files, nfiles, afiles, s);
               	if (err < 0)
			return err;
	}

	closedir(dir);

	return 0;
}
